# AUTOGENERATED! DO NOT EDIT! File to edit: ../../1-4.ipynb.

# %% auto 0
__all__ = ['identity_tensor', 'outer_product', 'zeros_state', 'normalize_tensor', 'rescale_tensor', 'tensor_contract']

# %% ../../1-4.ipynb 0
import torch

# %% ../../1-4.ipynb 5
def identity_tensor(order: int, dim: int, dtype: torch.dtype = torch.float32) -> torch.Tensor:
    """
    Create an identity tensor of given order and dimension.
    Args:
        order (int): The order of the tensor.
        dim (int): The dimension of the tensor.
        dtype (torch.dtype): The data type of the tensor. Default is torch.float32.
    Returns:
        torch.Tensor: The identity tensor of shape (dim, dim, ..., dim) with the specified order.
    """
    dims = [dim] * order
    I = torch.zeros(*dims, dtype=dtype)
    for i in range(dim):
        indices = [i] * order
        I[tuple(indices)] = 1.0

    return I

# %% ../../1-8.ipynb 3
from typing import List
import einops


def outer_product(vectors: List[torch.Tensor] | torch.Tensor) -> torch.Tensor:
    """
    Computes the outer product of a list of vectors.
    Args:
        vectors (List[torch.Tensor] | torch.Tensor): A list of 1D tensors (vectors) or a 2D tensor whose rows are vectors to be outer-producted.
    Returns:
        torch.Tensor: The outer product of the input vectors.
    """
    assert isinstance(vectors, (List, torch.Tensor)), "Input must be a list of tensors or a tensor"
    if isinstance(vectors, torch.Tensor):
        assert vectors.ndim == 2, "Input tensor must be 2D"
        vectors = [vectors[i] for i in range(vectors.shape[0])]
    else:
        assert all(isinstance(v, torch.Tensor) for v in vectors), "All elements must be tensors"
        for i, v in enumerate(vectors):
            assert v.dim() == 1, f"Expected 1D tensor, got {v.dim()}D tensor at index {i}"

    num_vectors = len(vectors)
    assert num_vectors >= 2, "At least two vectors are required for outer product"
    vec_dim_names = [f"v{i}" for i in range(num_vectors)]
    einsum_exp = "{input_string} -> {output_string}".format(
        input_string=",".join(vec_dim_names),
        output_string=" ".join(vec_dim_names),
    )
    return einops.einsum(*vectors, einsum_exp)

# %% ../../3-1.ipynb 7
def zeros_state(*, num_qubits: int, dtype: torch.dtype) -> torch.Tensor:
    """
    Returns the zero state as a tensor.
    Args:
        num_qubits (int): The number of qubits.
        dtype (torch.dtype): The data type of the tensor.
    Returns:
        torch.Tensor: The zero state as a tensor.
    """
    assert num_qubits > 0, "num_qubits must be positive"
    assert dtype in [torch.complex64, torch.complex128], "dtype must be complex64 or complex128"
    state = torch.zeros((2**num_qubits,), dtype=dtype)
    state[0] = 1.0
    shape = [2] * num_qubits
    state = state.reshape(shape)
    return state

# %% ../../3-5.ipynb 19
from typing import Tuple


def normalize_tensor(tensor: torch.Tensor, dim: int | None = None) -> torch.Tensor:
    """
    Normalize a tensor to the range [0, 1].

    Args:
        tensor (torch.Tensor): The tensor to normalize.
        dim (int | None, optional): The dimension along which to normalize.
            If None, normalizes across the entire tensor. Defaults to None.

    Returns:
        torch.Tensor: The normalized tensor with values in the range [0, 1].
    """
    if dim is None:
        min_val = torch.min(tensor)
        max_val = torch.max(tensor)
    else:
        min_val = torch.min(tensor, dim, keepdim=True).values
        max_val = torch.max(tensor, dim, keepdim=True).values
    return (tensor - min_val) / (max_val - min_val)


def rescale_tensor(
    tensor: torch.Tensor, min_val: float, max_val: float, dim: None | int | Tuple[int, ...] = None
) -> torch.Tensor:
    """
    Rescale a tensor to a specified range.

    Args:
        tensor (torch.Tensor): The tensor to rescale.
        min_val (float): The minimum value of the target range.
        max_val (float): The maximum value of the target range.
        dim (None | int | Tuple[int, ...], optional): The dimension(s) along which to rescale.
            If None, rescales across the entire tensor. Defaults to None.

    Returns:
        torch.Tensor: The rescaled tensor with values in the range [min_val, max_val].
    """

    normalized_tensor = normalize_tensor(tensor, dim)
    return normalized_tensor * (max_val - min_val) + min_val

# %% ../../tensor_product_experiments.ipynb 4
from einops import einsum
from typing import List, Set
from random import randint


def tensor_contract(*tensors, ein_expr: str, dims: List[Set[str]] | Set[str]) -> torch.Tensor:
    """
    Do tensor contraction with einsum expression.

    Args:
        tensors: List[torch.Tensor], the tensors to contract.
        ein_expr: str, the original einsum expression.
        dims: List[Set[str]] | Set[str], sets of dimensions that are aliased to each other, meaning they will be contracted together.

    Returns:
        torch.Tensor, the contracted tensor.
    """
    assert len(tensors) >= 2, "At least two tensors are needed for contraction"
    if isinstance(dims, Set):
        dims = [dims]
    elif isinstance(dims, List):
        pass
    else:
        raise ValueError(f"Invalid type for dims: {type(dims)}")

    for dim_set in dims:
        for d in dim_set:
            assert isinstance(d, str), f"Dimension name in a set must be a string, got {type(d)}"
            assert d in ein_expr, f"Dimension {d} not found in ein_expr"

    total_dims_num = sum(len(d) for d in dims)
    all_dims = set()
    for dim_set in dims:
        all_dims.update(dim_set)

    assert len(all_dims) == total_dims_num, "Sets of dimensions must be disjoint"

    prefix = randint(2 * 4, 2 ** (4 * 3))  # 2 or 3 digit hex
    prefix = f"{prefix:x}"
    postfix = randint(2 * 4, 2 ** (4 * 3))  # 2 or 3 digit hex
    postfix = f"{postfix:x}"
    mapping = {}
    for i, dim_set in enumerate(dims):
        new_name = f"a{prefix}_{i:x}_{postfix}"
        for d in dim_set:
            mapping[d] = new_name

    all_dims = list(all_dims)
    # sort all_dims by lex order, longest first
    all_dims.sort(reverse=True)

    for d in all_dims:
        ein_expr = ein_expr.replace(d, mapping[d])

    return einsum(*tensors, ein_expr)