#!/usr/bin/env python
# -- Content-Encoding: UTF-8 --
"""
Pelix remote services implementation based on Herald messaging and xmlrpclib

:author: Thomas Calmant
:copyright: Copyright 2014, isandlaTech
:license: Apache License 2.0
:version: 0.0.3
:status: Alpha

..

    Copyright 2014 isandlaTech

    Licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.
    You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.
"""

# Module version
__version_info__ = (0, 0, 3)
__version__ = ".".join(str(x) for x in __version_info__)

# Documentation strings format
__docformat__ = "restructuredtext en"

# ------------------------------------------------------------------------------

# Herald
import herald.beans as beans
import herald.remote

# iPOPO decorators
from pelix.ipopo.decorators import ComponentFactory, Requires, Validate, \
    Invalidate, Property, Provides, Instantiate

# Pelix constants
import pelix.remote
import pelix.remote.transport.commons as commons

# Standard library
import logging

# XML RPC modules
try:
    # Python 3
    # pylint: disable=F0401
    from xmlrpc.server import SimpleXMLRPCDispatcher
    import xmlrpc.client as xmlrpclib
except ImportError:
    # Python 2
    # pylint: disable=F0401
    from SimpleXMLRPCServer import SimpleXMLRPCDispatcher
    import xmlrpclib

# ------------------------------------------------------------------------------

HERALDRPC_CONFIGURATION = 'herald-xmlrpc'
""" Remote Service configuration constant """

PROP_HERALDRPC_PEER = "herald.rpc.peer"
""" UID of the peer exporting a service """

PROP_HERALDRPC_SUBJECT = 'herald.rpc.subject'
""" Subject to contact the exporter """

SUBJECT_REQUEST = 'herald/rpc/xmlrpc'
""" Subject to use for requests """

SUBJECT_REPLY = 'herald/rpc/xmlrpc/reply'
""" Subject to use for replies """

_logger = logging.getLogger(__name__)

# ------------------------------------------------------------------------------


class _XmlRpcDispatcher(SimpleXMLRPCDispatcher):
    """
    A XML-RPC dispatcher with a custom dispatch method

    Calls the dispatch method given in the constructor
    """
    def __init__(self, dispatch_method, encoding=None):
        """
        Sets up the servlet
        """
        SimpleXMLRPCDispatcher.__init__(self, allow_none=True,
                                        encoding=encoding)

        # Register the system.* functions
        self.register_introspection_functions()

        # Make a link to the dispatch method
        self._dispatch_method = dispatch_method

    def _simple_dispatch(self, name, params):
        """
        Dispatch method
        """
        try:
            # Internal method
            return self.funcs[name](*params)
        except KeyError:
            # Other method
            pass

        # Call the other method outside the except block, to avoid messy logs
        # in case of error
        return self._dispatch_method(name, params)

    def dispatch(self, data):
        """
        Handles a HTTP POST request

        :param data: The string content of the request
        :return: The XML-RPC response as a string
        """
        # Dispatch
        return self._marshaled_dispatch(data, self._simple_dispatch)


@ComponentFactory(herald.remote.FACTORY_HERALD_XMLRPC_EXPORTER)
@Requires('_directory', herald.SERVICE_DIRECTORY)
# SERVICE_EXPORT_PROVIDER is provided by the parent class
@Provides(herald.SERVICE_LISTENER)
@Property('_filters', herald.PROP_FILTERS, [SUBJECT_REQUEST])
@Property('_kinds', pelix.remote.PROP_REMOTE_CONFIGS_SUPPORTED,
          (HERALDRPC_CONFIGURATION,))
@Instantiate('herald-rpc-exporter-xmlrpc')
class HeraldRpcServiceExporter(commons.AbstractRpcServiceExporter):
    """
    Herald Remote Services exporter
    """
    def __init__(self):
        """
        Sets up the exporter
        """
        # Call parent
        super(HeraldRpcServiceExporter, self).__init__()

        # Herald directory
        self._directory = None

        # Herald filters
        self._filters = None

        # Handled configurations
        self._kinds = None

        # Dispatcher
        self._dispatcher = None

    def make_endpoint_properties(self, svc_ref, name, fw_uid):
        """
        Prepare properties for the ExportEndpoint to be created

        :param svc_ref: Service reference
        :param name: Endpoint name
        :param fw_uid: Framework UID
        :return: A dictionary of extra endpoint properties
        """
        return {PROP_HERALDRPC_PEER: self._directory.local_uid,
                PROP_HERALDRPC_SUBJECT: SUBJECT_REQUEST}

    @Validate
    def validate(self, context):
        """
        Component validated
        """
        # Call parent
        super(HeraldRpcServiceExporter, self).validate(context)

        # Setup the dispatcher
        self._dispatcher = _XmlRpcDispatcher(self.dispatch)

    @Invalidate
    def invalidate(self, context):
        """
        Component invalidated
        """
        # Call parent
        super(HeraldRpcServiceExporter, self).invalidate(context)

        # Clean up
        self._dispatcher = None

    def herald_message(self, herald_svc, message):
        """
        Received a message from Herald

        :param herald_svc: The Herald service
        :param message: A message bean
        """
        result = self._dispatcher.dispatch(message.content)
        herald_svc.reply(message, result, SUBJECT_REPLY)

# ------------------------------------------------------------------------------


class _XmlRpcEndpointProxy(object):
    """
    Proxy to use XML-RPC over Herald
    """
    def __init__(self, name, peer, subject, send_method):
        """
        Sets up the endpoint proxy

        :param name: End point name
        :param peer: UID of the peer to contact
        :param subject: Subject to use for RPC
        :param send_method: Method to use to send a request
        """
        self.__name = name
        self.__peer = peer
        self.__subject = subject
        self.__send = send_method
        self.__cache = {}

    def __getattr__(self, name):
        """
        Prefixes the requested attribute name by the endpoint name
        """
        return self.__cache.setdefault(
            name, _XmlRpcMethod("{0}.{1}".format(self.__name, name),
                                self.__peer, self.__subject, self.__send))


class _XmlRpcMethod(object):
    """
    Represents a method in a call proxy
    """
    def __init__(self, method_name, peer, subject, send_method):
        """
        Sets up the method

        :param method_name: Full method name
        :param peer: UID of the peer to contact
        :param subject: Subject to use for RPC
        :param send_method: Method to use to send a request
        """
        self.__name = method_name
        self.__peer = peer
        self.__subject = subject
        self.__send = send_method

    def __call__(self, *args):
        """
        Method is being called
        """
        # Forge the request
        request = xmlrpclib.dumps(args, self.__name, encoding='utf-8',
                                  allow_none=True)

        # Send it
        reply_message = self.__send(self.__peer, self.__subject, request)

        # Parse the reply
        parser, unmarshaller = xmlrpclib.getparser()
        parser.feed(reply_message.content)
        parser.close()
        return unmarshaller.close()


@ComponentFactory(herald.remote.FACTORY_HERALD_XMLRPC_IMPORTER)
@Requires('_herald', herald.SERVICE_HERALD)
@Provides(pelix.remote.SERVICE_IMPORT_ENDPOINT_LISTENER)
@Property('_kinds', pelix.remote.PROP_REMOTE_CONFIGS_SUPPORTED,
          (HERALDRPC_CONFIGURATION,))
@Instantiate('herald-rpc-importer-xmlrpc')
class HeraldRpcServiceImporter(commons.AbstractRpcServiceImporter):
    """
    XML-RPC Remote Services importer
    """
    def __init__(self):
        """
        Sets up the exporter
        """
        # Call parent
        super(HeraldRpcServiceImporter, self).__init__()

        # Herald service
        self._herald = None

        # Component properties
        self._kinds = None

    def __call(self, peer, subject, content):
        """
        Method called by the proxy to send a message over Herald
        """
        return self._herald.send(peer, beans.Message(subject, content))

    def make_service_proxy(self, endpoint):
        """
        Creates the proxy for the given ImportEndpoint

        :param endpoint: An ImportEndpoint bean
        :return: A service proxy
        """
        # Get Peer UID information
        peer_uid = endpoint.properties.get(PROP_HERALDRPC_PEER)
        if not peer_uid:
            _logger.warning("Herald-RPC endpoint without peer UID: %s",
                            endpoint)
            return

        # Get request subject information
        subject = endpoint.properties.get(PROP_HERALDRPC_SUBJECT)
        if not subject:
            _logger.warning("Herald-RPC endpoint without subject: %s",
                            endpoint)
            return

        # Return the proxy
        return _XmlRpcEndpointProxy(endpoint.name, peer_uid, subject,
                                    self.__call)

    def clear_service_proxy(self, endpoint):
        """
        Destroys the proxy made for the given ImportEndpoint

        :param endpoint: An ImportEndpoint bean
        """
        # Nothing to do
        return
