# pylint: skip-file
# fmt: off
r"""The unpolarized, space-like anomalous dimension :math:`\gamma_{gg}^{(3)}`."""
import numba as nb
import numpy as np

from .....harmonics import cache as c
from .....harmonics.log_functions import lm11, lm11m1, lm12m1, lm13m1


@nb.njit(cache=True)
def gamma_gg_nf3(n, cache):
    r"""Return the part proportional to :math:`nf^3` of :math:`\gamma_{gg}^{(3)}`.

    The expression is copied exact from :eqref:`3.14` of :cite:`Davies:2016jie`.

    Parameters
    ----------
    n : complex
        Mellin moment
    cache: numpy.ndarray
        Harmonic sum cache

    Returns
    -------
    complex
        |N3LO| non-singlet anomalous dimension :math:`\gamma_{gg}^{(3)}|_{nf^3}`

    """
    S1 = c.get(c.S1, cache, n)
    S2 = c.get(c.S2, cache, n)
    S3=  c.get(c.S3, cache, n)
    S21 = c.get(c.S21, cache, n)
    return 3.0 * (
        -0.0205761316872428
        + 2.599239604033225 / (-1.0 + n)
        - 1.1851851851851851 / np.power(n, 4)
        - 3.753086419753086 / np.power(n, 3)
        - 5.679012345679013 / np.power(n, 2)
        - 2.8050009209056537 / n
        - 1.1851851851851851 / np.power(1.0 + n, 4)
        + 3.753086419753086 / np.power(1.0 + n, 3)
        - 5.679012345679013 / np.power(1.0 + n, 2)
        + 2.8050009209056537 / (1.0 + n)
        - 2.599239604033225 / (2.0 + n)
        + 2.454258338353606 * S1
        - (8.674897119341564 * S1) / (-1.0 + n)
        + (2.3703703703703702 * S1) / np.power(n, 3)
        + (7.506172839506172 * S1) / np.power(n, 2)
        + (7.901234567901234 * S1) / n
        - (2.3703703703703702 * S1) / np.power(1.0 + n, 3)
        + (7.506172839506172 * S1) / np.power(1.0 + n, 2)
        - (7.901234567901234 * S1) / (1.0 + n)
        + (8.674897119341564 * S1) / (2.0 + n)
        - (2.567901234567901 * S2) / (-1.0 + n)
        + (2.3703703703703702 * S2) / np.power(n, 2)
        + (1.6296296296296295 * S2) / n
        + (2.3703703703703702 * S2) / np.power(1.0 + n, 2)
        - (1.6296296296296295 * S2) / (1.0 + n)
        + (2.567901234567901 * S2) / (2.0 + n)
        + (2.567901234567901 * (np.power(S1, 2) + S2)) / (-1.0 + n)
        - (2.3703703703703702 * (np.power(S1, 2) + S2)) / np.power(n, 2)
        - (1.6296296296296295 * (np.power(S1, 2) + S2)) / n
        - (2.3703703703703702 * (np.power(S1, 2) + S2)) / np.power(1.0 + n, 2)
        + (1.6296296296296295 * (np.power(S1, 2) + S2)) / (1.0 + n)
        - (2.567901234567901 * (np.power(S1, 2) + S2)) / (2.0 + n)
    ) + 1.3333333333333333 * (
        -0.6337448559670782
        - 54.23172286962781 / (-1.0 + n)
        + 3.5555555555555554 / np.power(n, 5)
        + 13.62962962962963 / np.power(n, 4)
        + 27.65432098765432 / np.power(n, 3)
        + 61.00190529209603 / np.power(n, 2)
        + 26.28917081074211 / n
        + 10.666666666666666 / np.power(1.0 + n, 5)
        - 31.40740740740741 / np.power(1.0 + n, 4)
        + 31.604938271604937 / np.power(1.0 + n, 3)
        - 12.479576189385448 / np.power(1.0 + n, 2)
        + 44.82194030036901 / (1.0 + n)
        - 16.879388241483305 / (2.0 + n)
        + (48.724279835390945 * S1) / (-1.0 + n)
        - (7.111111111111111 * S1) / np.power(n, 4)
        - (27.25925925925926 * S1) / np.power(n, 3)
        - (36.34567901234568 * S1) / np.power(n, 2)
        - (56.49382716049383 * S1) / n
        + (7.111111111111111 * S1) / np.power(1.0 + n, 4)
        - (17.77777777777778 * S1) / np.power(1.0 + n, 3)
        + (3.950617283950617 * S1) / np.power(1.0 + n, 2)
        + (4.345679012345679 * S1) / (1.0 + n)
        + (3.4238683127572016 * S1) / (2.0 + n)
        + (27.65432098765432 * S2) / (-1.0 + n)
        - (7.111111111111111 * S2) / np.power(n, 3)
        - (27.25925925925926 * S2) / np.power(n, 2)
        - (16.59259259259259 * S2) / n
        + (7.111111111111111 * S2) / np.power(1.0 + n, 3)
        - (27.25925925925926 * S2) / np.power(1.0 + n, 2)
        + (16.59259259259259 * S2) / (1.0 + n)
        - (27.65432098765432 * S2) / (2.0 + n)
        - (15.012345679012345 * (np.power(S1, 2) + S2)) / (-1.0 + n)
        + (7.111111111111111 * (np.power(S1, 2) + S2)) / np.power(n, 3)
        + (8.296296296296296 * (np.power(S1, 2) + S2)) / np.power(n, 2)
        + (22.51851851851852 * (np.power(S1, 2) + S2)) / n
        + (4.7407407407407405 * (np.power(S1, 2) + S2)) / np.power(1.0 + n, 2)
        - (15.407407407407407 * (np.power(S1, 2) + S2)) / (1.0 + n)
        + (7.901234567901234 * (np.power(S1, 2) + S2)) / (2.0 + n)
        - (9.481481481481481 * S21) / (-1.0 + n)
        + (14.222222222222221 * S21) / np.power(n, 2)
        + (21.333333333333332 * S21) / n
        + (14.222222222222221 * S21) / np.power(1.0 + n, 2)
        - (21.333333333333332 * S21) / (1.0 + n)
        - (28.444444444444443 * S21) / (n * (1.0 + n))
        + (9.481481481481481 * S21) / (2.0 + n)
        + (4.7407407407407405 * S3) / (-1.0 + n)
        - (7.111111111111111 * S3) / np.power(n, 2)
        - (10.666666666666666 * S3) / n
        - (7.111111111111111 * S3) / np.power(1.0 + n, 2)
        + (10.666666666666666 * S3) / (1.0 + n)
        + (14.222222222222221 * S3) / (n * (1.0 + n))
        - (4.7407407407407405 * S3) / (2.0 + n)
        - (9.481481481481481 * (S1 * S2 - 1.0 * S21 + S3)) / (-1.0 + n)
        + (14.222222222222221 * (S1 * S2 - 1.0 * S21 + S3)) / np.power(n, 2)
        + (21.333333333333332 * (S1 * S2 - 1.0 * S21 + S3)) / n
        + (14.222222222222221 * (S1 * S2 - 1.0 * S21 + S3)) / np.power(1.0 + n, 2)
        - (21.333333333333332 * (S1 * S2 - 1.0 * S21 + S3)) / (1.0 + n)
        - (28.444444444444443 * (S1 * S2 - 1.0 * S21 + S3)) / (n * (1.0 + n))
        + (9.481481481481481 * (S1 * S2 - 1.0 * S21 + S3)) / (2.0 + n)
        + (1.5802469135802468 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3))
        / (-1.0 + n)
        - (2.3703703703703702 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3))
        / np.power(n, 2)
        - (3.5555555555555554 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3)) / n
        - (2.3703703703703702 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3))
        / np.power(1.0 + n, 2)
        + (3.5555555555555554 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3))
        / (1.0 + n)
        + (4.7407407407407405 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3))
        / (n * (1.0 + n))
        - (1.5802469135802468 * (np.power(S1, 3) + 3.0 * S1 * S2 + 2.0 * S3))
        / (2.0 + n)
    )


@nb.njit(cache=True)
def gamma_gg_nf1(n, cache, variation):
    r"""Return the part proportional to :math:`nf^1` of :math:`\gamma_{gg}^{(3)}`.

    Parameters
    ----------
    n : complex
        Mellin moment
    cache: numpy.ndarray
        Harmonic sum cache
    variation : int
        |N3LO| anomalous dimension variation

    Returns
    -------
    complex
        |N3LO| non-singlet anomalous dimension :math:`\gamma_{gg}^{(3)}|_{nf^1}`

    """
    S1 = c.get(c.S1, cache, n)
    S2 = c.get(c.S2, cache, n)
    S3 = c.get(c.S3, cache, n)
    common = 18143.980574437464 + 1992.766087237516/np.power(-1. + n,3) + 20005.925925925927/np.power(n,7) - 19449.679012345678/np.power(n,6) + 80274.123066115/np.power(n,5) + 4341.13370266389/n - 11714.245609287387*S1 + 13880.514502193577*lm11(n,S1)
    if variation == 1:
        fit = -10180.036287065168/np.power(-1. + n,2) + 18513.270411873862/(-1. + n) + 1054.0462034864902/(1. + n) - 23202.37636673074/(2. + n) - 3602.060464842473*lm13m1(n,S1,S2,S3)
    elif variation == 2:
        fit = -10990.56710015112/np.power(-1. + n,2) + 20788.896977978722/(-1. + n) - 20421.35502658661/(1. + n) + 8525.968866079374*lm12m1(n,S1,S2) + 1738.6616090235632*lm13m1(n,S1,S2,S3)
    elif variation == 3:
        fit = -2634.4244909317163/np.power(-1. + n,2) + 22720.3693403071/(-1. + n) - 125998.50835459496/np.power(n,4) - 33227.36724506296/(1. + n) - 9413.257245961297*lm13m1(n,S1,S2,S3)
    elif variation == 4:
        fit = -11204.025554445116/np.power(-1. + n,2) + 29125.056079570935/(-1. + n) - 31819.903485877207/np.power(n,3) - 34636.371205966374/(1. + n) - 5216.754331416846*lm13m1(n,S1,S2,S3)
    elif variation == 5:
        fit = -10596.095479944803/np.power(-1. + n,2) + 19686.27928898392/(-1. + n) - 21569.188441280912/(1. + n) - 11201.144235965185*lm11m1(n,S1) - 1700.6621840281703*lm13m1(n,S1,S2,S3)
    elif variation == 6:
        fit = -10219.818404157402/np.power(-1. + n,2) + 18624.961714720896/(-1. + n) - 22063.56752869039/(2. + n) + 418.4678562263162*lm12m1(n,S1,S2) - 3339.9294915689893*lm13m1(n,S1,S2,S3)
    elif variation == 7:
        fit = -9948.03233312767/np.power(-1. + n,2) + 18642.625540707515/(-1. + n) - 3874.0599063901927/np.power(n,4) - 22488.97588930172/(2. + n) - 3780.736583727932*lm13m1(n,S1,S2,S3)
    elif variation == 8:
        fit = -10210.277793833904/np.power(-1. + n,2) + 18826.668606883333/(-1. + n) - 939.7381958313449/np.power(n,3) - 22517.139866390396/(2. + n) - 3649.747268844151*lm13m1(n,S1,S2,S3)
    elif variation == 9:
        fit = -10454.38560322065/np.power(-1. + n,2) + 19672.71547858848/(-1. + n) - 2028.768596859424/np.power(n,2) - 23244.272133899653/(2. + n) - 3548.6223138853798*lm13m1(n,S1,S2,S3)
    elif variation == 10:
        fit = -10199.42103109479/np.power(-1. + n,2) + 18567.922435470675/(-1. + n) - 22121.34719008958/(2. + n) - 521.8760156034384*lm11m1(n,S1) - 3513.4718226546247*lm13m1(n,S1,S2,S3)
    elif variation == 11:
        fit = -24315.85150285657/np.power(-1. + n,2) + 17708.837224620995/(-1. + n) + 200925.95790419917/np.power(n,4) + 22122.069993382116*lm12m1(n,S1,S2) + 19522.28466669723*lm13m1(n,S1,S2,S3)
    elif variation == 12:
        fit = -10683.911741932305/np.power(-1. + n,2) + 8813.133994420503/(-1. + n) + 45712.61388637733/np.power(n,3) + 20774.41339562901*lm12m1(n,S1,S2) + 11730.84286896203*lm13m1(n,S1,S2,S3)
    elif variation == 13:
        fit = -5836.51269525416/np.power(-1. + n,2) - 954.1826542477216/(-1. + n) + 37911.15300076812/np.power(n,2) + 8238.284738213413*lm12m1(n,S1,S2) + 559.8674243103725*lm13m1(n,S1,S2,S3)
    elif variation == 14:
        fit = -18008.69786800099/np.power(-1. + n,2) + 40405.809150940426/(-1. + n) + 199282.00400507107*lm11m1(n,S1) + 160213.3434703554*lm12m1(n,S1,S2) + 62928.42253561968*lm13m1(n,S1,S2,S3)
    elif variation == 15:
        fit = 5128.6565343884595/np.power(-1. + n,2) - 12028.343555706011/(-1. + n) - 119224.34855414196/np.power(n,4) + 60406.6661071792/np.power(n,2) - 10691.948251059068*lm13m1(n,S1,S2,S3)
    elif variation == 16:
        fit = -25326.250571944573/np.power(-1. + n,2) + 14072.807307962283/(-1. + n) + 233114.0751714257/np.power(n,4) - 31924.757662241216*lm11m1(n,S1) + 12568.669163696943*lm13m1(n,S1,S2,S3)
    elif variation == 17:
        fit = -2650.979531261116/np.power(-1. + n,2) - 7372.905521491985/(-1. + n) - 30040.6560602065/np.power(n,3) + 62824.974620623434/np.power(n,2) - 6781.2886081167435*lm13m1(n,S1,S2,S3)
    elif variation == 18:
        fit = -9592.62299684503/np.power(-1. + n,2) + 4106.275632673743/(-1. + n) + 52523.14189143729/np.power(n,3) - 29690.17856987198*lm11m1(n,S1) + 4103.133130507111*lm13m1(n,S1,S2,S3)
    elif variation == 19:
        fit = -5176.681219116345/np.power(-1. + n,2) - 3196.230738347968/(-1. + n) + 39966.24595996931/np.power(n,2) - 10802.706088036985*lm11m1(n,S1) - 2821.015728520721*lm13m1(n,S1,S2,S3)
    else:
        fit = -10163.154508989206/np.power(-1. + n,2) + 14038.103511363668/(-1. + n) + 9733.848224236723/np.power(n,4) + 1865.024107152609/np.power(n,3) + 10477.909004825295/np.power(n,2) - 5726.328195547913/(1. + n) - 7138.825209215919/(2. + n) + 6060.070601755382*lm11m1(n,S1) + 11594.344648415032*lm12m1(n,S1,S2) + 2899.599321273186*lm13m1(n,S1,S2,S3)
    return common + fit



@nb.njit(cache=True)
def gamma_gg_nf2(n, cache, variation):
    r"""Return the part proportional to :math:`nf^2` of :math:`\gamma_{gg}^{(3)}`.

    Parameters
    ----------
    n : complex
        Mellin moment
    cache: numpy.ndarray
        Harmonic sum cache
    variation : int
        |N3LO| anomalous dimension variation

    Returns
    -------
    complex
        |N3LO| non-singlet anomalous dimension :math:`\gamma_{gg}^{(3)}|_{nf^2}`

    """
    S1 = c.get(c.S1, cache, n)
    S2 = c.get(c.S2, cache, n)
    S3 = c.get(c.S3, cache, n)
    common = -423.811346198137 - 568.8888888888889/np.power(n,7) + 1725.6296296296296/np.power(n,6) - 2196.543209876543/np.power(n,5) + 21.333333333333336/n + 440.0487580115612*S1 - 135.11111111111114*lm11(n,S1)
    if variation == 1:
        fit = -28.872079560670887/np.power(-1. + n,2) - 600.687665018851/(-1. + n) - 782.7409584384554/(1. + n) + 1212.0129563274725/(2. + n) - 131.99661800641996*lm13m1(n,S1,S2,S3)
    elif variation == 2:
        fit = 13.46728393535039/np.power(-1. + n,2) - 719.5586305737447/(-1. + n) + 339.0607105706994/(1. + n) - 445.3675161372763*lm12m1(n,S1,S2) - 410.9777123288363*lm13m1(n,S1,S2,S3)
    elif variation == 3:
        fit = -423.0290881266415/np.power(-1. + n,2) - 820.4521550528278/(-1. + n) + 6581.732068732447/np.power(n,4) + 1008.0030845050551/(1. + n) + 171.5604693866337*lm13m1(n,S1,S2,S3)
    elif variation == 4:
        fit = 24.617625096660852/np.power(-1. + n,2) - 1155.0111323969943/(-1. + n) + 1662.163163134959/np.power(n,3) + 1081.6046427740755/(1. + n) - 47.65052104384607*lm13m1(n,S1,S2,S3)
    elif variation == 5:
        fit = -7.138567251448382/np.power(-1. + n,2) - 661.9616455143912/(-1. + n) + 399.01961082863164/(1. + n) + 585.1095476215276*lm11m1(n,S1) - 231.31917510401414*lm13m1(n,S1,S2,S3)
    elif variation == 6:
        fit = 0.6703563957663715/np.power(-1. + n,2) - 683.6302885496931/(-1. + n) + 366.326763050706/(2. + n) - 310.75671045013235*lm12m1(n,S1,S2) - 326.65663250723026*lm13m1(n,S1,S2,S3)
    elif variation == 7:
        fit = -201.15959093110155/np.power(-1. + n,2) - 696.7475504266947/(-1. + n) + 2876.8998495153933/np.power(n,4) + 682.2375331931647/(2. + n) + 0.6893269576792295*lm13m1(n,S1,S2,S3)
    elif variation == 8:
        fit = -6.414557380878645/np.power(-1. + n,2) - 833.4190222620106/(-1. + n) + 697.8551544024533/np.power(n,3) + 703.1522695219711/(2. + n) - 96.58411469486595*lm13m1(n,S1,S2,S3)
    elif variation == 9:
        fit = 174.86134878770187/np.power(-1. + n,2) - 1461.6984440261558/(-1. + n) + 1506.5755852971279/np.power(n,2) + 1243.1250013579786/(2. + n) - 171.6801056536221*lm13m1(n,S1,S2,S3)
    elif variation == 10:
        fit = -14.476853744515829/np.power(-1. + n,2) - 641.2725816829176/(-1. + n) + 409.23428207929186/(2. + n) + 387.5482225476748*lm11m1(n,S1) - 197.7830687973748*lm13m1(n,S1,S2,S3)
    elif variation == 11:
        fit = 234.71021283280834/np.power(-1. + n,2) - 668.4196514406609/(-1. + n) - 3336.0224123418016/np.power(n,4) - 671.1068807980374*lm12m1(n,S1,S2) - 706.2435192552492*lm13m1(n,S1,S2,S3)
    elif variation == 12:
        fit = 8.375810645634878/np.power(-1. + n,2) - 520.7221331157691/(-1. + n) - 758.9776156472786/np.power(n,3) - 648.7314114404783*lm12m1(n,S1,S2) - 576.880321324907*lm13m1(n,S1,S2,S3)
    elif variation == 13:
        fit = -72.10673143086429/np.power(-1. + n,2) - 358.5530058001772/(-1. + n) - 629.4480683703055/np.power(n,2) - 440.59102678646576*lm12m1(n,S1,S2) - 391.4059065714235*lm13m1(n,S1,S2,S3)
    elif variation == 14:
        fit = 129.9910112750619/np.power(-1. + n,2) - 1045.262981333448/(-1. + n) - 3308.727447025322*lm11m1(n,S1) - 2963.8697874593217*lm12m1(n,S1,S2) - 1426.926154164017*lm13m1(n,S1,S2,S3)
    elif variation == 15:
        fit = -658.5340428548301/np.power(-1. + n,2) + 233.70328023719887/(-1. + n) + 6376.227554233226/np.power(n,4) - 1832.5287499192084/np.power(n,2) + 210.35151978073395*lm13m1(n,S1,S2,S3)
    elif variation == 16:
        fit = 265.36221636680716/np.power(-1. + n,2) - 558.115114090075/(-1. + n) - 4312.498256800645/np.power(n,4) + 968.4864274161131*lm11m1(n,S1) - 495.2949411211266*lm13m1(n,S1,S2,S3)
    elif variation == 17:
        fit = -242.4719679073227/np.power(-1. + n,2) - 15.273817400740654/(-1. + n) + 1606.6018497165833/np.power(n,3) - 1961.8621081220385/np.power(n,2) + 1.2058538747811829*lm13m1(n,S1,S2,S3)
    elif variation == 18:
        fit = -25.702325492667082/np.power(-1. + n,2) - 373.7390722554988/(-1. + n) - 971.6528600084858/np.power(n,3) + 927.147789097384*lm11m1(n,S1) - 338.6865966478428*lm13m1(n,S1,S2,S3)
    elif variation == 19:
        fit = -107.39512419897143/np.power(-1. + n,2) - 238.64621708524552/(-1. + n) - 739.3563254661514/np.power(n,2) + 577.7386335438422*lm11m1(n,S1) - 210.59317344974744*lm13m1(n,S1,S2,S3)
    else:
        fit = -49.223424397058984/np.power(-1. + n,2) - 622.0772540941418/(-1. + n) + 430.8599370178221/np.power(n,4) + 117.68366797885425/np.power(n,3) - 192.45366666213553/np.power(n,2) + 107.62879422315821/(1. + n) + 242.9520423963465/(2. + n) + 7.226482800064206*lm11m1(n,S1) - 288.44333331956375*lm12m1(n,S1,S2) - 282.9932310879313*lm13m1(n,S1,S2,S3)
    return common + fit


@nb.njit(cache=True)
def gamma_gg_nf0(n, cache, variation):
    r"""Return the part proportional to :math:`nf^0` of :math:`\gamma_{gg}^{(3)}`.

    Parameters
    ----------
    n : complex
        Mellin moment
    cache: numpy.ndarray
        Harmonic sum cache
    variation : int
        |N3LO| anomalous dimension variation

    Returns
    -------
    complex
        |N3LO| non-singlet anomalous dimension :math:`\gamma_{gg}^{(3)}|_{nf^0}`

    """
    S1 = c.get(c.S1, cache, n)
    S2 = c.get(c.S2, cache, n)
    S3 = c.get(c.S3, cache, n)
    common = -68587.9129845144 - 49851.703887834694/np.power(-1. + n,4) + 213823.9810748423/np.power(-1. + n,3) - 103680./np.power(n,7) - 17280./np.power(n,6) - 627978.8224813186/np.power(n,5) - 54482.80778086425/n + 40880.33011934297*S1 - 85814.12027987762*lm11(n,S1)
    if variation == 1:
        fit = -295597.8343912357/np.power(-1. + n,2) + 224940.49334893477/(-1. + n) - 528016.0821893369/(1. + n) + 295950.8085172776/(2. + n) + 1477.798412728375*lm13m1(n,S1,S2,S3)
    elif variation == 2:
        fit = -285259.35674268805/np.power(-1. + n,2) + 195914.43535053887/(-1. + n) - 254093.176327687/(1. + n) - 108750.38571168127*lm12m1(n,S1,S2) - 66644.14697966084*lm13m1(n,S1,S2,S3)
    elif variation == 3:
        fit = -391843.5756260241/np.power(-1. + n,2) + 171278.1309382666/(-1. + n) + 1.607135399845882e6/np.power(n,4) - 90750.00793688672/(1. + n) + 75600.73990110215*lm13m1(n,S1,S2,S3)
    elif variation == 4:
        fit = -282536.6527775845/np.power(-1. + n,2) + 89585.10998095598/(-1. + n) + 405869.0374960631/np.power(n,3) - 72777.8892690519/(1. + n) + 22073.53170175139*lm13m1(n,S1,S2,S3)
    elif variation == 5:
        fit = -290290.91872975003/np.power(-1. + n,2) + 209978.53772452284/(-1. + n) - 239452.33860661855/(1. + n) + 142872.76615791937*lm11m1(n,S1) - 22774.905259686046*lm13m1(n,S1,S2,S3)
    elif variation == 6:
        fit = -275669.2984894303/np.power(-1. + n,2) + 168989.62141469866/(-1. + n) - 274526.4428921221/(2. + n) - 209628.1521964082*lm12m1(n,S1,S2) - 129834.63274025511*lm13m1(n,S1,S2,S3)
    elif variation == 7:
        fit = -411818.37425604276/np.power(-1. + n,2) + 160141.0679428944/(-1. + n) + 1.9406795709568677e6/np.power(n,4) - 61421.50009639418/(2. + n) + 90984.18244209285*lm13m1(n,S1,S2,S3)
    elif variation == 8:
        fit = -280448.59152967954/np.power(-1. + n,2) + 67946.15425201622/(-1. + n) + 470754.3927408164/np.power(n,3) - 47312.97924099773/(2. + n) + 25366.124502714512*lm13m1(n,S1,S2,S3)
    elif variation == 9:
        fit = -158164.7213188171/np.power(-1. + n,2) - 355874.3130731763/(-1. + n) + 1.0162955311005719e6/np.power(n,2) + 316938.16065307835/(2. + n) - 25291.619297074456*lm13m1(n,S1,S2,S3)
    elif variation == 10:
        fit = -285887.2006806783/np.power(-1. + n,2) + 197562.99577218326/(-1. + n) - 245582.1799793595/(2. + n) + 261429.64913634845*lm11m1(n,S1) - 42899.97897764852*lm13m1(n,S1,S2,S3)
    elif variation == 11:
        fit = -451059.5122634805/np.power(-1. + n,2) + 157590.7215229884/(-1. + n) + 2.500025820230639e6/np.power(n,4) + 60419.41309017621*lm12m1(n,S1,S2) + 154628.98908088758*lm13m1(n,S1,S2,S3)
    elif variation == 12:
        fit = -281443.7904442531/np.power(-1. + n,2) + 46905.73090962251/(-1. + n) + 568780.2423251155/np.power(n,3) + 43651.165093081385*lm12m1(n,S1,S2) + 57683.796561516865*lm13m1(n,S1,S2,S3)
    elif variation == 13:
        fit = -221129.91709477766/np.power(-1. + n,2) - 74624.32957010131/(-1. + n) + 471710.387076172/np.power(n,2) - 112329.90204316098*lm12m1(n,S1,S2) - 81311.32054580169*lm13m1(n,S1,S2,S3)
    elif variation == 14:
        fit = -372582.60911082564/np.power(-1. + n,2) + 439998.3130990782/(-1. + n) + 2.479570886293798e6*lm11m1(n,S1) + 1.7786232372348914e6*lm12m1(n,S1,S2) + 694710.8508805855*lm13m1(n,S1,S2,S3)
    elif variation == 15:
        fit = -370641.1836499627/np.power(-1. + n,2) + 76373.05020007766/(-1. + n) + 1.6256368673598904e6/np.power(n,4) + 164981.63662011502/np.power(n,2) + 72108.40125127957*lm13m1(n,S1,S2,S3)
    elif variation == 16:
        fit = -453819.096640544/np.power(-1. + n,2) + 147660.05980785593/(-1. + n) + 2.587937446690415e6/np.power(n,4) - 87192.34328327361*lm11m1(n,S1) + 135637.39527110517*lm13m1(n,S1,S2,S3)
    elif variation == 17:
        fit = -264565.00578584545/np.power(-1. + n,2) + 12895.649871143309/(-1. + n) + 409607.5894176158/np.power(n,3) + 132007.7388904974/np.power(n,2) + 18786.134714622272*lm13m1(n,S1,S2,S3)
    elif variation == 18:
        fit = -279150.776231584/np.power(-1. + n,2) + 37015.68744579348/(-1. + n) + 583090.5131692598/np.power(n,3) - 62384.957000475675*lm11m1(n,S1) + 41656.46473103294*lm13m1(n,S1,S2,S3)
    elif variation == 19:
        fit = -230126.7905259369/np.power(-1. + n,2) - 44053.76182863434/(-1. + n) + 443688.97265165334/np.power(n,2) + 147296.0640752822*lm11m1(n,S1) - 35212.61366143017*lm13m1(n,S1,S2,S3)
    else:
        fit = -309580.8003310073/np.power(-1. + n,2) + 101590.70290050836/(-1. + n) + 540074.4792149311/np.power(n,4) + 128321.14606046685/np.power(n,3) + 117299.17191257945/np.power(n,2) - 62373.13128050426/(1. + n) - 839.6912125535554/(2. + n) + 151662.74028313678*lm11m1(n,S1) + 76420.28291931044*lm12m1(n,S1,S2) + 51933.95747315065*lm13m1(n,S1,S2,S3)
    return common + fit


@nb.njit(cache=True)
def gamma_gg(n, nf, cache, variation):
    r"""Compute the |N3LO| gluon-gluon singlet anomalous dimension.

    Parameters
    ----------
    n : complex
        Mellin moment
    nf : int
        Number of active flavors
    cache: numpy.ndarray
        Harmonic sum cache
    variation : int
        |N3LO| anomalous dimension variation

    Returns
    -------
    complex
        |N3LO| gluon-gluon singlet anomalous dimension
        :math:`\gamma_{gg}^{(3)}(N)`

    """
    return (
        gamma_gg_nf0(n, cache, variation)
        + nf * gamma_gg_nf1(n, cache, variation)
        + nf**2 * gamma_gg_nf2(n, cache, variation)
        + nf**3 * gamma_gg_nf3(n, cache)
    )
