"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const sqs = require("aws-cdk-lib/aws-sqs");
const ec2 = require("aws-cdk-lib/aws-ec2");
const defaults = require("@aws-solutions-constructs/core");
const lib_1 = require("../lib");
const assertions_1 = require("aws-cdk-lib/assertions");
test('Test minimal deployment', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        }
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 2: test for an producer function
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: 'producer-function'
    });
    // Assertion 3: test for a consumer function
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: 'consumer-function'
    });
    // Assertion 4: test for a queue (queue and DLQ)
    template.resourceCountIs('AWS::SQS::Queue', 2);
    // Assertion 5: test for send-message permissions (only) on the producer function
    template.hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "sqs:SendMessage",
                        "sqs:GetQueueAttributes",
                        "sqs:GetQueueUrl"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "lambdasqslambdalambdatosqsqueue49588D68",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        }
    });
    // Assertion 6: test for consume-message permissions (only) on the consumer function
    template.hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "sqs:ReceiveMessage",
                        "sqs:ChangeMessageVisibility",
                        "sqs:GetQueueUrl",
                        "sqs:DeleteMessage",
                        "sqs:GetQueueAttributes"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "lambdasqslambdalambdatosqsqueue49588D68",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
    });
});
test('Test deployment w/ existing producer function', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Define existing resources
    const existingProducerFn = defaults.buildLambdaFunction(stack, {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'existing-producer-function'
        }
    });
    // Helper declaration
    const props = {
        existingProducerLambdaObj: existingProducerFn,
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'deployed-consumer-function'
        }
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 2: test for the existing producer function
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: 'existing-producer-function'
    });
    // Assertion 3: test for the deployed consumer function
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: 'deployed-consumer-function'
    });
});
test('Test deployment w/ existing consumer function', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Define existing resources
    const existingConsumerFn = defaults.buildLambdaFunction(stack, {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'existing-consumer-function'
        }
    });
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'deployed-producer-function'
        },
        existingConsumerLambdaObj: existingConsumerFn
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 2: test for the deployed producer function
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: 'deployed-producer-function'
    });
    // Assertion 3: test for the existing consumer function
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: 'existing-consumer-function'
    });
});
test('Test deployment w/ existing queue', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Define existing resources
    const buildQueueResponse = defaults.buildQueue(stack, 'existing-queue', {
        queueProps: {
            queueName: 'existing-queue'
        }
    });
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        },
        existingQueueObj: buildQueueResponse.queue
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 2: test for the existing queue
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::SQS::Queue', {
        QueueName: 'existing-queue'
    });
});
test('Test deployment w/ DLQ explicitly disabled', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`)
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`)
        },
        deployDeadLetterQueue: false,
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 2: test for a non-existing DLQ
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::SQS::Queue', 1);
    defaults.expectNonexistence(stack, 'AWS::SQS::Queue', {
        RedrivePolicy: {}
    });
});
test('Test deployment w/ DLQ explicitly enabled and w/ MRC override', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`)
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`)
        },
        deployDeadLetterQueue: true,
        maxReceiveCount: 6
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    const template = assertions_1.Template.fromStack(stack);
    // Assertion 2: test for an existing DLQ
    template.resourceCountIs('AWS::SQS::Queue', 2);
    template.hasResourceProperties('AWS::SQS::Queue', {
        RedrivePolicy: {}
    });
    // Assertion 3: test for the overridden max receive count
    template.hasResourceProperties('AWS::SQS::Queue', {
        RedrivePolicy: {
            maxReceiveCount: 6
        }
    });
});
test('Test overrides for producer and consumer functions', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        }
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 2: test for updated runtime on producer function
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        Runtime: "nodejs16.x"
    });
    // Assertion 3: test for updated runtime on consumer function
    template.hasResourceProperties('AWS::Lambda::Function', {
        Runtime: "nodejs16.x"
    });
});
test('Test the public pattern props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        }
    };
    const pattern = new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 1: get the producer function
    expect(pattern.producerLambdaFunction).toBeDefined();
    // Assertion 2: get the queue
    expect(pattern.sqsQueue).toBeDefined();
    // Assertion 3: get the dead letter queue
    expect(pattern.deadLetterQueue).toBeDefined();
    // Assertion 4: get the consumer function
    expect(pattern.consumerLambdaFunction).toBeDefined();
});
test('Test lambda function custom environment variable', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        },
        queueEnvironmentVariableName: 'CUSTOM_QUEUE_NAME'
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: 'producer-function',
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                CUSTOM_QUEUE_NAME: {
                    Ref: 'lambdasqslambdalambdatosqsqueue49588D68'
                }
            }
        }
    });
});
test('Pattern deployment w/ batch size', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        },
        sqsEventSourceProps: {
            batchSize: 5
        }
    };
    new lib_1.LambdaToSqsToLambda(stack, 'test-lambda-sqs-lambda', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::EventSourceMapping', {
        BatchSize: 5
    });
});
test("Test minimal deployment that deploys a VPC without vpcProps", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSqsToLambda(stack, "lambda-to-sqs-to-lambda-stack", {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        },
        deployVpc: true,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatosqstolambdastacklambdatosqsReplaceDefaultSecurityGroupsecuritygroup90A497DF",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "lambdatosqstolambdastackVpcisolatedSubnet1Subnet70F24179",
                },
                {
                    Ref: "lambdatosqstolambdastackVpcisolatedSubnet2Subnet1D6A3FAF",
                },
            ],
        },
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    template.resourceCountIs("AWS::EC2::Subnet", 2);
    template.resourceCountIs("AWS::EC2::InternetGateway", 0);
});
test("Test minimal deployment that deploys a VPC w/vpcProps", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSqsToLambda(stack, "lambda-to-sqs-to-lambda-stack", {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        },
        vpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            ipAddresses: ec2.IpAddresses.cidr("192.68.0.0/16"),
        },
        deployVpc: true,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatosqstolambdastacklambdatosqsReplaceDefaultSecurityGroupsecuritygroup90A497DF",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "lambdatosqstolambdastackVpcisolatedSubnet1Subnet70F24179",
                },
                {
                    Ref: "lambdatosqstolambdastackVpcisolatedSubnet2Subnet1D6A3FAF",
                },
            ],
        },
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: "192.68.0.0/16",
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    template.resourceCountIs("AWS::EC2::Subnet", 2);
    template.resourceCountIs("AWS::EC2::InternetGateway", 0);
});
test("Test minimal deployment with an existing VPC", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    new lib_1.LambdaToSqsToLambda(stack, "lambda-to-sqs-to-lambda-stack", {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        },
        existingVpc: testVpc,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatosqstolambdastacklambdatosqsReplaceDefaultSecurityGroupsecuritygroup90A497DF",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "testvpcPrivateSubnet1Subnet865FB50A",
                },
                {
                    Ref: "testvpcPrivateSubnet2Subnet23D3396F",
                },
            ],
        },
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
});
test("Test bad call with existingVpc and deployVpc", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToSqsToLambda(stack, "lambda-to-sqs-to-lambda-stack", {
            producerLambdaFunctionProps: {
                runtime: lambda.Runtime.NODEJS_16_X,
                handler: 'index.handler',
                code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
                functionName: 'producer-function'
            },
            consumerLambdaFunctionProps: {
                runtime: lambda.Runtime.NODEJS_16_X,
                handler: 'index.handler',
                code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
                functionName: 'consumer-function'
            },
            existingVpc: testVpc,
            deployVpc: true,
        });
    };
    // Assertion
    expect(app).toThrowError();
});
test('Confirm CheckSqsProps is being called', () => {
    // NOTE: CheckSqsProps is called from both lambda-sqs and sqs-lambda internally
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        queueProps: {},
        existingQueueObj: new sqs.Queue(stack, 'test', {})
    };
    const app = () => {
        new lib_1.LambdaToSqsToLambda(stack, 'test-eventbridge-sqs', props);
    };
    expect(app).toThrowError("Error - Either provide queueProps or existingQueueObj, but not both.\n");
});
test('Confirm CheckLambdaProps is being called', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingLambdaObj = new lambda.Function(stack, 'ExistingLambda', {
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const props = {
        existingProducerLambdaObj: existingLambdaObj,
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
    };
    const app = () => {
        new lib_1.LambdaToSqsToLambda(stack, 'test-lambda-sqs-lambda', props);
    };
    expect(app).toThrowError('Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,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