# -*- coding: utf-8 -*-
"""
Part of the astor library for Python AST manipulation.

License: 3-clause BSD

Copyright (c) 2008      Armin Ronacher
Copyright (c) 2012-2017 Patrick Maupin
Copyright (c) 2013-2017 Berker Peksag

This module converts an AST into Python source code.

Before being version-controlled as part of astor,
this code came from here (in 2012):

    https://gist.github.com/1250562

"""

import ast
import astunparse as astu
import sys

from .op_util import get_op_symbol, get_op_precedence, Precedence
from .node_util import ExplicitNodeVisitor
from .string_repr import pretty_string
from .source_repr import pretty_source
from ..laparser.laparser import LAParser 
from collections import namedtuple
import astpretty as ap
import os
import json
from jinja2 import Template

pmt_temp_functions = {\
        'pmat': 'c_pmt_create_pmat', \
        'pvec': 'c_pmt_create_pvec', \
        'pmt_gemm_nn': 'c_pmt_gemm_nn', \
        'pmt_gemm_tn': 'c_pmt_gemm_tn', \
        'pmt_gemm_nt': 'c_pmt_gemm_nt', \
        'pmt_gead': 'c_pmt_gead', \
        'pmt_getrf': 'c_pmt_getrf', \
        'pmt_getrsm': 'c_pmt_getrsm', \
        'pmt_getrsv': 'c_pmt_getrsv', \
        'pmt_potrf': 'c_pmt_potrf', \
        'pmt_potrsm': 'c_pmt_potrsm', \
        'pmt_potrsv': 'c_pmt_potrsv', \
        'pmat_fill': 'c_pmt_pmat_fill', \
        'pmat_copy': 'c_pmt_pmat_copy', \
        'pmat_tran': 'c_pmt_pmat_tran', \
        'pmat_vcat': 'c_pmt_pmat_vcat', \
        'pmat_hcat': 'c_pmt_pmat_hcat', \
        'pmat_print': 'c_pmt_pmat_print', \
        'pvec_fill': 'c_pmt_pvec_fill', \
        'pvec_copy': 'c_pmt_pvec_copy', \
        'pvec_print': 'c_pmt_pvec_print', \
        'print': 'printf'}

pmt_temp_types = {\
        'pmat': 'struct pmat *', \
        'pvec': 'struct pvec *', \
        'None': 'void', \
        'NoneType': 'void', \
        'ptr_int': 'int *', \
        'ptr_pmat': 'struct pmat **', \
        'int': 'int', \
        'float': 'double', \
        'dimv': 'dimv', \
        'dims': 'dims'}

arg_types = {\
        'pmat': ['int', 'int'], \
        'pvec': ['int'], \
        'pmt_gemm_nn': ['pmat', 'pmat', 'pmat', 'pmat'], \
        'pmt_gemm_tn': ['pmat', 'pmat', 'pmat', 'pmat'], \
        'pmt_gead':    ['float', 'pmat', 'pmat'], \
        'pmt_getrf':   ['pmat', 'pmat', 'List'], \
        'pmt_getrsm':  ['pmat', 'List', 'pmat'], \
        'pmt_getrsv':  ['pmat', 'List', 'pvec'], \
        'pmt_potrf':   ['pmat', 'pmat'], \
        'pmt_potrsm':  ['pmat', 'pmat'], \
        'pmt_potrsv':  ['pmat', 'pvec'], \
        'pmat_fill':   ['pmat', 'float'], \
        'pmat_copy':   ['pmat', 'pmat'], \
        'pmat_tran':   ['pmat', 'pmat'], \
        'pmat_vcat':   ['pmat', 'pmat', 'pmat'], \
        'pmat_hcat':   ['pmat', 'pmat', 'pmat'], \
        'pmat_print':  ['pmat'], \
        'pvec_fill':   ['pvec', 'float'], \
        'pvec_copy':   ['pvec', 'pvec', 'pmat', 'pmat'], \
        'pvec_print':  ['pmat', 'pmat', 'pmat', 'pmat'], \
}

usr_temp_types = {}

class cgenException(Exception):
    def __init__(self, message, lineno):
        super().__init__(message)
        self.message = message
        self.lineno = lineno

def to_source(node, module_name, indent_with=' ' * 4, add_line_information=False,
              pretty_string=pretty_string, pretty_source=pretty_source,
              main=False, ___c_pmt_8_heap_size=None, ___c_pmt_64_heap_size=None, 
              size_of_pointer=8, size_of_int=4, size_of_double=8):

    """This function can convert a node tree back into python sourcecode.
    This is useful for debugging purposes, especially if you're dealing with
    custom asts not generated by python itself.

    It could be that the sourcecode is evaluable when the AST itself is not
    compilable / evaluable.  The reason for this is that the AST contains some
    more data than regular sourcecode does, which is dropped during
    conversion.

    Each level of indentation is replaced with `indent_with`.  Per default this
    parameter is equal to four spaces as suggested by PEP 8, but it might be
    adjusted to match the application's styleguide.

    If `add_line_information` is set to `True` comments for the line numbers
    of the nodes are added to the output.  This can be used to spot wrong line
    number information of statement nodes.

    """
    if ___c_pmt_8_heap_size is None or ___c_pmt_64_heap_size is None :
        error('Need to pass heap_sizes! Exiting.')

    generator = SourceGenerator(indent_with, ___c_pmt_8_heap_size,
        ___c_pmt_64_heap_size, size_of_pointer, size_of_int, size_of_double,
        add_line_information, pretty_string)
    
    generator.result.source.append('#include "stdlib.h"\n')
    # generator.result.source.append('#include "pmat_blasfeo_wrapper.h"\n')
    # generator.result.source.append('#include "pvec_blasfeo_wrapper.h"\n')
    # generator.result.source.append('#include "pmt_heap.h"\n')
    generator.result.source.append('#include "%s.h"\n' %(module_name))

    generator.result.source.append('void * ___c_pmt_8_heap;\n')
    generator.result.source.append('void * ___c_pmt_64_heap;\n')
    generator.result.source.append('void * ___c_pmt_8_heap_head;\n')
    generator.result.source.append('void * ___c_pmt_64_heap_head;\n')
    generator.result.header.append('#include "prometeo.h"\n')
    generator.result.header.append('#include "pmt_aux.h"\n')
    generator.result.header.append('#ifdef __cplusplus\nextern "C" {\n#endif\n\n')

    generator.visit(node)
    
    generator.result.source.append('\n')
    if set(generator.result.source[0]) == set('\n'):
        generator.result.source[0] = ''
    

    generator.result.header.append('#ifdef __cplusplus\n}\n#endif\n\n')
    generator.result.header.append('\n')
    if set(generator.result.header[0]) == set('\n'):
        generator.result.header[0] = ''

    # dump meta-data to JSON file
    pmt_cache_dir = '__pmt_cache__'
    if not os.path.exists(pmt_cache_dir):
        os.mkdir(pmt_cache_dir)

    os.chdir(pmt_cache_dir)
    json_file = 'typed_record.json'
    with open(json_file, 'w') as f:
        json.dump(generator.typed_record, f, indent=4, sort_keys=True)
    json_file = 'var_dim_record.json'
    with open(json_file, 'w') as f:
        json.dump(generator.var_dim_record, f, indent=4, sort_keys=True)
    json_file = 'dim_record.json'
    with open(json_file, 'w') as f:
        json.dump(generator.dim_record, f, indent=4, sort_keys=True)
    json_file = 'usr_types.json'
    with open(json_file, 'w') as f:
        json.dump(usr_temp_types, f, indent=4, sort_keys=True)
    json_file = 'heap8.json'
    with open(json_file, 'w') as f:
        json.dump(generator.heap8_record, f, indent=4, sort_keys=True)
    json_file = 'heap64.json'
    with open(json_file, 'w') as f:
        json.dump(generator.heap64_record, f, indent=4, sort_keys=True)
    os.chdir('..')

    return generator.result


def precedence_setter(AST=ast.AST, get_op_precedence=get_op_precedence,
                      isinstance=isinstance, list=list):
    """ This only uses a closure for performance reasons,
        to reduce the number of attribute lookups.  (set_precedence
        is called a lot of times.)
    """

    def set_precedence(value, *nodes):
        """Set the precedence (of the parent) into the children.
        """
        if isinstance(value, AST):
            value = get_op_precedence(value)
        for node in nodes:
            if isinstance(node, AST):
                node._pp = value
            elif isinstance(node, list):
                set_precedence(value, *node)
            else:
                assert node is None, node

    return set_precedence


set_precedence = precedence_setter()

def descope(current_scope, pop):
    if current_scope.endswith(pop):
        return current_scope[:-len(pop)]
    else:
        raise Exception('Attempt to descope {}, which is \
            not the current scope'.format(pop))

def Num_or_Name(node):
    if isinstance(node, ast.Num):
        return node.n
    elif isinstance(node, ast.Name):
        return node.id
    elif isinstance(node, ast.UnaryOp):
        if isinstance(node.op, ast.USub):
            return -Num_or_Name(node.operand)
        else:
            raise cgenException('node.op is not of type ast.USub.\n', node.lineno)
    else:
        raise cgenException('node is not of type ast.Num nor ast.Name.\n', node.lineno)

class Delimit(object):
    """A context manager that can add enclosing
       delimiters around the output of a
       SourceGenerator method.  By default, the
       parentheses are added, but the enclosed code
       may set discard=True to get rid of them.
    """

    discard = False

    def __init__(self, tree, *args):
        """ use write instead of using result.source directly
            for initial data, because it may flush
            preceding data into result.source.
        """
        delimiters = '()'
        node = None
        op = None
        for arg in args:
            if isinstance(arg, ast.AST):
                if node is None:
                    node = arg
                else:
                    op = arg
            else:
                delimiters = arg
        tree.write(delimiters[0], dest = 'src')
        result = self.result = tree.result
        result.source = self.result.source = tree.result.source
        self.index = len(result.source)
        self.closing = delimiters[1]
        if node is not None:
            self.p = p = get_op_precedence(op or node)
            self.pp = pp = tree.get__pp(node)
            self.discard = p >= pp

    def __enter__(self):
        return self

    def __exit__(self, *exc_info):
        result = self.result
        result.source = self.result.source
        start = self.index - 1
        if self.discard:
            result.source[start] = ''
        else:
            result.source.append(self.closing)


class SourceGenerator(ExplicitNodeVisitor):
    """This visitor is able to transform a well formed syntax tree into C
    source code.

    For more details have a look at the docstring of the `node_to_source`
    function.
    """
    using_unicode_literals = False
    
    def __init__(self, indent_with,  ___c_pmt_8_heap_size, ___c_pmt_64_heap_size,
                size_of_pointer, size_of_int, size_of_double,
                add_line_information=False,pretty_string=pretty_string,
                 # constants
                 len=len, isinstance=isinstance, callable=callable):
        
        self.result = namedtuple('result', 'source header')
        self.result.source = [] 
        self.result.header = []
        self.indent_with = indent_with
        self.add_line_information = add_line_information
        self.indentation = 0 # Current indentation level
        self.new_lines = 0   # Number of lines to insert before next code
        self.colinfo = 0, 0  # index in result.source of string containing 
                             # linefeed, and position of last linefeed in 
                             # that string

        self.pretty_string = pretty_string
        AST = ast.AST

        visit = self.visit
        newline = self.newline
        result = self.result
        result.source = self.result.source
        result.header = self.result.header
        append_src = result.source.append
        append_hdr = result.header.append
    
        self.size_of_pointer = size_of_pointer
        self.size_of_int = size_of_int
        self.size_of_double = size_of_double
        self.heap8_size  = ___c_pmt_8_heap_size 
        self.heap64_size = ___c_pmt_64_heap_size 

        self.typed_record = {'global': dict()}
        self.heap8_record = {'global': dict()}
        self.heap64_record = {'global': dict()}
        self.scope = 'global'
        self.var_dim_record = {'global': dict()}
        self.dim_record = dict()
        self.in_main = False
        
        def write(*params, dest):
            """ self.write is a closure for performance (to reduce the number
                of attribute lookups).
            """
            for item in params:
                if isinstance(item, AST):
                    visit(item)
                elif callable(item):
                    item()
                elif item == '\n':
                    newline()
                else:
                    if dest == 'src':
                        if self.new_lines:
                            append_src('\n' * self.new_lines)
                            self.colinfo = len(result.source), 0
                            append_src(self.indent_with * self.indentation)
                            self.new_lines = 0
                        if item:
                            append_src(item)
                    if dest == 'hdr':
                        if self.new_lines:
                            append_hdr('\n' * self.new_lines)
                            self.colinfo = len(result.header), 0
                            append_hdr(self.indent_with * self.indentation)
                            self.new_lines = 0
                        if item:
                            append_hdr(item)


        self.write = write

    def __getattr__(self, name, defaults=dict(keywords=(),
                    _pp=Precedence.highest).get):
        """ Get an attribute of the node.
            like dict.get (returns None if doesn't exist)
        """
        if not name.startswith('get_'):
            raise AttributeError
        geta = getattr
        shortname = name[4:]
        default = defaults(shortname)

        def getter(node):
            return geta(node, shortname, default)

        setattr(self, name, getter)
        return getter

    def delimit(self, *args):
        return Delimit(self, *args)

    def conditional_write(self, *stuff, dest):
        if stuff[-1] is not None:
            self.write(*stuff, dest = dest)
            # Inform the caller that we wrote
            return True

    def newline(self, node=None, extra=0):
        self.new_lines = max(self.new_lines, 1 + extra)
        if node is not None and self.add_line_information:
            self.write('# line: %s' % node.lineno, dest = 'src')
            self.new_lines = 1

    def body(self, statements):
        self.indentation += 1
        self.write(*statements, dest = 'src')
        self.indentation -= 1

    def body_class(self, statements, name):
        self.indentation += 1
        self.write_class_attributes(*statements, name=name)
        self.write('};', dest = 'hdr')
        self.indentation -= 1
        self.write_class_method_prototypes(*statements, name=name)
        
        self.write('\n', dest = 'src')
        self.write_class_init(*statements, name=name)
        self.write_class_methods(*statements, name=name)


    def write_class_attributes(self, *params, name):
        """ self.write is a closure for performance (to reduce the number
            of attribute lookups).
        """
        for item in params:
            if isinstance(item, ast.AnnAssign):
                set_precedence(item, item.target, item.annotation)
                set_precedence(Precedence.Comma, item.value)
                need_parens = isinstance(item.target, ast.Name) and not item.simple
                begin = '(' if need_parens else ''
                end = ')' if need_parens else ''
                annotation = item.annotation
                # annotation = ast.parse(item.annotation.s).body[0]
                # if 'value' in annotation.value.__dict__:
                type_py = annotation.id
                if annotation.id is 'List':
                    if item.value.func.id is not 'plist': 
                        raise cgenException('Cannot create Lists without using'
                            ' plist constructor.', item.lineno)
                    else:
                        if len(item.value.args) != 2:
                            raise cgenException('Type annotations in List \
                                declaration must have the format \
                                List[<type>, <sizes>]', item.lineno)
                        # attribute is a List
                        ann = item.value.args[0].id
                        dims = Num_or_Name(item.value.args[1])
                        # ann = item.annotation.slice.value.elts[0].id
                        # dims = Num_or_Name(item.annotation.slice.value.elts[1])
                        if isinstance(dims, str):
                            self.typed_record[self.scope][item.target.id] = \
                                'List[' + ann + ', ' + dims + ']'
                        else:
                            self.typed_record[self.scope][item.target.id] = \
                                'List[' + ann + ', ' + str(dims) + ']'

                        if  ann in pmt_temp_types:
                            ann = pmt_temp_types[ann]
                        else:
                            raise cgenException ('Usage of non existing type \
                                \033[91m{}\033[0m'.format(ann), item.lineno)
                        # check is dims is not a numerical value
                        if isinstance(dims, str):
                            dim_list = self.dim_record[dims]
                        array_size = len(dim_list)
                            # array_size = str(Num_or_Name(item.value.args[1]))
                            # self.statement([], ann, ' ', item.target, '[', array_size, '];')
                        self.write('%s' %ann, ' ', '%s' %item.target.id, \
                            '[%s' %array_size, '];\n', dest = 'hdr')
                else:
                    type_c = pmt_temp_types[type_py]
                    self.write('%s' %type_c, ' ', '%s' %item.target.id, ';\n', dest = 'hdr')
                    # self.conditional_write(' = ', item.value, ';')
                    self.typed_record[self.scope][item.target.id] = type_py
                # else:
                #     type_py = annotation.id
                #     type_c = pmt_temp_types[type_py]
                #     self.write('%s' %type_c, ' ', '%s' %item.target.id, ';\n', dest = 'hdr')
                #     # self.conditional_write(' = ', item.value, ';')
                #     self.typed_record[self.scope][item.target.id] = type_py
            elif isinstance(item, ast.FunctionDef):
                # build argument mangling
                f_name_len = len(item.name)
                pre_mangl = '_Z%s' %f_name_len 
                if item.args.args[0].arg is not 'self':
                    raise cgenException('First argument in method {} \
                        must be \'self\'. You have \'{}\''.format(item.name, \
                        item.args.args[0].arg), item.lineno)
                else: 
                    # store self argument
                    self_arg = item.args.args[0]
                    # pop self from argument list
                    item.args.args.pop(0)

                post_mangl = self.build_arg_mangling(item.args)
                
                if hasattr(self.get_returns(item), 'id'):
                    ret_type = self.get_returns(item).id
                else:
                    ret_type = self.get_returns(item).value

                if ret_type is None: 
                    ret_type = 'None'

                if  ret_type in pmt_temp_types:
                    ret_type = pmt_temp_types[ret_type]
                else:

                    raise cgenException ('Usage of non existing type \
                        \033[91m{}\033[0m'.format(ann), item.lineno)
                    # raise cgenException ('Usage of non existing type {}'.format(ret_type))

                if len(item.args.args) > 0:  
                    self.write('%s (*%s%s%s' % (ret_type, pre_mangl, \
                        item.name, post_mangl) , ')', '(%s *self, ' %name, \
                        dest = 'hdr')
                else:
                    self.write('%s (*%s%s%s' % (ret_type, pre_mangl, \
                            item.name, post_mangl) , ')', '(%s *self' %name, \
                            dest = 'hdr')

                self.visit_arguments(item.args, 'hdr')
                self.write(');\n', dest = 'hdr')
                # insert back self argument 
                item.args.args.insert(0, self_arg)
        
    def write_class_method_prototypes(self, *params, name):
        """ self.write is a closure for performance (to reduce the number
            of attribute lookups).
        """
        self.write('\n\n', dest = 'hdr')
        for item in params:
            if isinstance(item, ast.FunctionDef):
                # build argument mangling
                f_name_len = len(item.name)
                pre_mangl = '_Z%s' %f_name_len 
                if item.args.args[0].arg is not 'self':
                    raise cgenException('First argument in method {} \
                        must be \'self\'. You have \'{}\''.format(item.name, \
                        item.args.args[0].arg), item.lineno)
                else: 
                    # store self argument
                    self_arg = item.args.args[0]
                    # pop self from argument list
                    item.args.args.pop(0)

                post_mangl = self.build_arg_mangling(item.args)
                if hasattr(self.get_returns(item), 'id'):
                    ret_type = self.get_returns(item).id
                else:
                    ret_type = self.get_returns(item).value

                if ret_type is None: 
                    ret_type = 'None'

                if  ret_type in pmt_temp_types:
                    ret_type = pmt_temp_types[ret_type]
                else:
                    raise cgenException ('Usage of non existing type {}'.format(ret_type), \
                        item.lineno)

                if len(item.args.args) > 0:
                    self.write('%s (%s%s%s%s' % (ret_type, pre_mangl, item.name, \
                        post_mangl, name) , '_impl)', '(%s *self, ' %name, dest = 'hdr')
                else:
                    self.write('%s (%s%s%s%s' % (ret_type, pre_mangl, item.name, \
                        post_mangl, name) , '_impl)', '(%s *self' %name, dest = 'hdr')
                self.visit_arguments(item.args, 'hdr')
                self.write(');\n', dest = 'hdr')
                # insert back self argument 
                item.args.args.insert(0, self_arg)
    
    def write_class_init(self, *params, name):
        """ self.write is a closure for performance (to reduce the number
            of attribute lookups).
        """
        self.write('void ', name, '_init(struct ', name, ' *object){', dest = 'src')
        self.indentation += 1
        for item in params:
            if isinstance(item, ast.AnnAssign):
                # set_precedence(item, item.target, item.annotation)
                set_precedence(Precedence.Comma, item.value)
                need_parens = isinstance(item.target, ast.Name) and not item.simple
                begin = '(' if need_parens else ''
                end = ')' if need_parens else ''
                # TODO(andrea): need to fix the code below!
                ann = item.annotation.id
                if ann == 'List':
                    if item.value.func.id is not 'plist':
                        raise cgenException('Invalid subscripted annotation.', 
                                ' Lists must be created using plist constructor and',  
                                ' the argument of List[] must be a valid type.\n', \
                                item.lineno)
                    else:
                        # attribute is a List
                        ann = Num_or_Name(item.value.args[0])
                        dims = Num_or_Name(item.value.args[1])
                        # ann = item.annotation.slice.value.elts[0].id
                        # dims = Num_or_Name(item.annotation.slice.value.elts[1])
                        if isinstance(dims, str):
                            dim_list = self.dim_record[dims]
                        else:
                            dim_list = dims
                        if ann == 'pmat': 
                            # build init for List of pmats
                            for i in range(len(dim_list)):
                                self.statement([], 'object->', \
                                    item.target.id, \
                                    '[', str(i),'] = c_pmt_create_pmat(', \
                                    str(dim_list[i][0]), ', ', \
                                    str(dim_list[i][1]), ');')

                        elif ann == 'pvec': 
                            # build init for List of pvecs
                            for i in range(len(dim_list)):
                                self.statement([], 'object->', \
                                    item.target.id, \
                                    '[', str(i),'] = c_pmt_create_pvec(', \
                                    str(dim_list[i][0]), ');')
                        # else: do nothing (no init required for "memoryless" objects)
                # pmat[<n>,<m>] or pvec[<n>]
                elif ann in ['pmat', 'pvec']:
                    if ann == 'pmat':
                        if item.value.func.id != 'pmat':
                            raise cgenException('pmat objects need to be declared calling',
                                'the pmat(<n>, <m>) constructor\n.', item.lineno)
                        dim1 = Num_or_Name(item.value.args[0])
                        dim2 = Num_or_Name(item.value.args[1])
                        self.var_dim_record[self.scope][item.target.id] = [dim1, dim2]
                    else:
                        # pvec
                        if item.value.func.id != 'pvec':
                            raise cgenException('pvec objects need to be declared calling', 
                                'the pvec(<n>, <m>) constructor\n.', item.lineno)
                        dim1 = Num_or_Name(item.value.args[0])
                        ann = item.annotation.value.id
                        self.var_dim_record[self.scope][item.target.id] = [dim1]
                    # add variable to typed record
                    self.typed_record[self.scope][item.target.id] = ann
                    # print('typed_record = \n', self.typed_record, '\n\n')
                    # print('var_dim_record = \n', self.var_dim_record, '\n\n')
                    if  ann in pmt_temp_types:
                        c_ann = pmt_temp_types[ann]
                        # self.statement(item, c_ann, ' ', item.target.id)
                    else:
                        raise cgenException ('Usage of non existing type {}'.format(ann), \
                            item.lineno)
                    if item.value != None:
                        if hasattr(item.value, 'value') is False:
                            self.conditional_write('\n', 'object->', \
                                item.target, ' = ', item.value, ';', dest = 'src')
                        else:
                            if item.value.value != None:
                                self.conditional_write('\n', 'object->', \
                                    item.target, ' = ', item.value, ';', dest = 'src')
                    else:
                        raise cgenException('Cannot declare attribute without' 
                            ' initialization.\n', item.lineno)
                else:
                    if item.value != None:
                        if hasattr(item.value, 'value') is False:
                            self.conditional_write('\n', 'object->', \
                                item.target, ' = ', item.value, ';', dest = 'src')
                        else:
                            if item.value.value != None:
                                self.conditional_write('\n', 'object->', \
                                    item.target, ' = ', item.value, ';', dest = 'src')
                    else:
                        raise cgenException('Cannot declare attribute without \
                            initialization.\n', item.lineno)

            elif isinstance(item, ast.FunctionDef):
                # build argument mangling
                f_name_len = len(item.name)
                pre_mangl = '_Z%s' %f_name_len 
                if item.args.args[0].arg is not 'self':
                    raise cgenException('First argument in method {} \
                        must be \'self\'. You have \'{}\'.'.format(item.name, \
                        item.args.args[0].arg), item.lineno)
                else: 
                    # store self argument
                    self_arg = item.args.args[0]
                    # pop self from argument list
                    item.args.args.pop(0)

                post_mangl = self.build_arg_mangling(item.args)
                
                self.statement(item, 'object->%s%s%s' %(pre_mangl, \
                    item.name, post_mangl), ' = &', '%s%s%s%s' %(pre_mangl, \
                    item.name, post_mangl, name), '_impl;')
                
                # build argument mangling
                arg_mangl = self.build_arg_mangling(item.args)
                # insert back self argument 
                item.args.args.insert(0, self_arg)

        self.write('\n}\n', dest = 'src')
        self.indentation -=1
    
    def write_class_methods(self, *params, name):
        """ self.write is a closure for performance (to reduce the number
            of attribute lookups).
        """
        for item in params:
            if isinstance(item, ast.FunctionDef):
                self.decorators(item, 1 if self.indentation else 2)
                # self.write()

                # build argument mangling
                f_name_len = len(item.name)
                pre_mangl = '_Z%s' %f_name_len 
                if item.args.args[0].arg is not 'self':
                    raise cgenException('First argument in method {} \
                        must be \'self\'. You have \'{}\'.'.format(item.name, \
                        item.args.args[0].arg), item.lineno)
                else: 
                    # store self argument
                    self_arg = item.args.args[0]
                    # pop self from argument list
                    item.args.args.pop(0)

                post_mangl = self.build_arg_mangling(item.args)

                if hasattr(self.get_returns(item), 'id'):
                    ret_type = self.get_returns(item).id
                else:
                    ret_type = self.get_returns(item).value

                if ret_type is None: 
                    ret_type = 'None'

                if  ret_type in pmt_temp_types:
                    ret_type = pmt_temp_types[ret_type]
                else:
                    raise cgenException ('Usage of non existing type {}.'.format(ret_type), item.lineno)

                if len(item.args.args) > 0:
                    self.statement(item, ret_type, ' %s%s%s%s' % (pre_mangl, \
                            item.name, post_mangl, name), '_impl(', name, ' *self, ')
                else:
                    self.statement(item, ret_type, ' %s%s%s%s' % (pre_mangl, \
                            item.name, post_mangl, name), '_impl(', name, ' *self')

                self.scope = self.scope + '@' + item.name
                self.typed_record[self.scope] = dict()
                self.var_dim_record[self.scope] = dict()
                self.heap8_record[self.scope] = 0
                self.heap64_record[self.scope] = 0
                self.visit_arguments(item.args, 'src')
                self.write(') {\n', dest = 'src')
                # store current pmt_heap value (and restore before returning)
                self.write('\tvoid *callee_pmt_8_heap = ___c_pmt_8_heap;\n', dest = 'src')
                self.write('\tvoid *callee_pmt_64_heap = ___c_pmt_64_heap;\n', dest = 'src')
                self.body(item.body)
                self.newline(1)
                self.write('}', dest = 'src')
                self.scope = descope(self.scope, '@' + item.name)

                if not self.indentation:
                    self.newline(extra=2)
                # insert back self argument 
                item.args.args.insert(0, self_arg)

    def else_body(self, elsewhat):
        if elsewhat:
            self.write('\n', 'else:')
            self.body(elsewhat)

    def body_or_else(self, node):
        self.body(node.body)
        self.else_body(node.orelse)

    def visit_arguments(self, node, dest_in):
        want_comma = []

        def write_comma():
            if want_comma:
                self.write(', ', dest = dest_in)
            else:
                want_comma.append(True)

        def loop_args(args, defaults):
            set_precedence(Precedence.Comma, defaults)
            padding = [None] * (len(args) - len(defaults))
            for arg, default in zip(args, padding + defaults):
                # fish C type from typed record
                if hasattr(arg.annotation, 'value'):
                    if arg.annotation.value.id == 'pmat':
                        dim1 = Num_or_Name(arg.annotation.slice.value.elts[0])
                        dim2 = Num_or_Name(arg.annotation.slice.value.elts[1])
                        arg_type_py = arg.annotation.value.id
                        self.var_dim_record[self.scope][arg.arg] = [dim1, dim2]
                    elif arg.annotation.value.id == 'pvec':
                        dim1 = Num_or_Name(arg.annotation.slice.value.elts[0])
                        arg_type_py = arg.annotation.value.id
                        self.var_dim_record[self.scope][arg.arg] = [dim1]
                    else:
                        raise cgenException('Subscripted type annotation can \
                            be used only with pmat arguments.\n', arg.lineno)
                else:
                    arg_type_py = arg.annotation.id

                arg_type_c = pmt_temp_types[arg_type_py]
                self.write(write_comma, arg_type_c,' ',arg.arg, dest = dest_in)

                # add variable to typed record
                self.typed_record[self.scope][arg.arg] = arg_type_py
                self.conditional_write('=', default, dest = 'src')

        loop_args(node.args, node.defaults)
        self.conditional_write(write_comma, '*', node.vararg, dest = 'src')

        kwonlyargs = self.get_kwonlyargs(node)
        if kwonlyargs:
            if node.vararg is None:
                self.write(write_comma, '*')
            loop_args(kwonlyargs, node.kw_defaults)
        self.conditional_write(write_comma, '**', node.kwarg, dest = 'src')

    def build_arg_mangling(self, node):
        want_comma = []

        def loop_args_mangl(args, defaults):
            set_precedence(Precedence.Comma, defaults)
            padding = [None] * (len(args) - len(defaults))
            arg_mangl = ''
            for arg, default in zip(args, padding + defaults):
                annotation = arg.annotation
                # annotation = ast.parse(arg.annotation.s).body[0]
                # if 'value' in annotation.value.__dict__:
                if 'value' in annotation.__dict__:
                    # arg_mangl = arg_mangl + annotation.value.value.id
                    arg_mangl = arg_mangl + annotation.value.id
                else:
                    # arg_mangl = arg_mangl + annotation.value.id
                    arg_mangl = arg_mangl + annotation.id
            return arg_mangl

        arg_mangl = loop_args_mangl(node.args, node.defaults)
        return arg_mangl
    
    def build_arg_mangling_mod(self, args):
        want_comma = []

        def loop_args_mangl(args):
            arg_mangl = ''
            for arg in args:
                if isinstance(arg, ast.Num):
                    if isinstance(arg.n, int):
                        arg_value = 'int'
                    elif isinstance(arg.n, float):
                        arg_value = 'double'
                    else:
                        raise cgenException('Invalid numeric argument.\n', arg.lineno)
                    arg_mangl = arg_mangl + arg_value
                else:
                    arg_value = arg.id
                    arg_mangl = arg_mangl + self.typed_record[self.scope][arg_value]
            return arg_mangl

        arg_mangl = loop_args_mangl(args)
        return arg_mangl

    def statement(self, node, *params, **kw):
        self.newline(node)
        self.write(*params, dest = 'src')

    def decorators(self, node, extra):
        self.newline(extra=extra)
        for decorator in node.decorator_list:
            self.statement(decorator, '@', decorator)

    def comma_list(self, items, trailing=False):
        set_precedence(Precedence.Comma, *items)
        for idx, item in enumerate(items):
            self.write(', ' if idx else '', item, dest = 'src')
        self.write(',' if trailing else '', dest = 'src')

    # Statements
    def visit_Assign(self, node):
        if 'targets' in node.__dict__:
            if len(node.targets) != 1:
                raise cgenException('Cannot have assignments with a number of \
                    targets other than 1.\n', node.lineno)
            # check for attributes
            if hasattr(node.targets[0], 'value'):
                if hasattr(node.targets[0].value, 'attr'):
                    if node.targets[0].value.attr not in self.typed_record[self.scope]:
                        raise cgenException('Unknown variable {}.'.format( \
                            node.targets[0].value.attr), node.lineno)
                    # TODO(andrea): need to handle attributes recursively
                    target = node.targets[0].value.attr
                    obj_name = node.targets[0].value.value.id
                    # TODO(andrea): need to compute local scope (find strings 
                    # that contain scope and have a string in common with self.scope)
                    # this assumes that the class has been defined in the global scope
                    scope = 'global@' + self.typed_record[self.scope][obj_name]
                else:
                    if node.targets[0].value.id not in self.typed_record[self.scope]:
                        raise cgenException('Unknown variable {}.'.format( \
                                node.targets[0].value.id), node.lineno)
                    target = node.targets[0].value.id
                    scope = self.scope
            else:
                if node.targets[0].id not in self.typed_record[self.scope]:
                    raise cgenException('Unknown variable {}.'.format(node.targets[0].id), node.lineno)
            if type(node.targets[0]) == ast.Subscript: 
                if target in self.typed_record[scope]: 
                    # map subscript for pmats to blasfeo el assign
                    if self.typed_record[scope][target] == 'pmat':
                        # check for ExtSlices
                        if isinstance(node.targets[0].slice, ast.ExtSlice):
                            first_index_l  = astu.unparse( \
                                node.targets[0].slice.dims[0].lower).strip('\n')

                            first_index_u  = astu.unparse( \
                                node.targets[0].slice.dims[0].upper).strip('\n')

                            second_index_l = astu.unparse( \
                                node.targets[0].slice.dims[1].lower).strip('\n')

                            second_index_u = astu.unparse( \
                                node.targets[0].slice.dims[1].upper).strip('\n')

                            # check if subscripted expression is used in the value
                            if hasattr(node.value, 'slice'):
                                first_index_value_l  = astu.unparse( \
                                    node.value.slice.dims[0].lower).strip('\n')

                                first_index_value_u  = astu.unparse( \
                                    node.value.slice.dims[0].upper).strip('\n')

                                second_index_value_l = astu.unparse( \
                                    node.value.slice.dims[1].lower).strip('\n')

                                second_index_value_u = astu.unparse( \
                                    node.value.slice.dims[1].upper).strip('\n')

                                ai = first_index_value_l
                                aj = second_index_value_l
                                value = node.value.value.id
                            else:
                                ai = '0'
                                aj = '0'
                                value = node.value.id
                            m = first_index_u + '-' + first_index_l
                            n = second_index_u + '-' + second_index_l
                            bi = second_index_l
                            bj = second_index_l
                            self.statement([], 'c_pmt_gecp(', m, ', ', n, ', ', value, \
                                ', ', ai, ', ', aj, ', ', target, ', ', bi, ', ', bj, ');')
                        else:
                            if not isinstance(node.targets[0].slice.value, ast.Tuple):
                                # ap.pprint(node)
                                raise cgenException('Subscript to a pmat object \
                                    must be of type Tuple.', node.lineno)

                            # unparse slice expression
                            first_index = astu.unparse(node.targets[0].slice.value.elts[0]).strip('\n') 
                            second_index = astu.unparse(node.targets[0].slice.value.elts[1]).strip('\n') 

                            # check if subscripted expression is used in the value
                            if isinstance(node.value, ast.Subscript):
                                # if value is a pmat
                                value = node.value.value.id
                                if value in self.typed_record[scope]:
                                    if self.typed_record[scope][value] == 'pmat':
                                        first_index_value = astu.unparse(node.value.slice.value.elts[0]).strip('\n')
                                        second_index_value = astu.unparse(node.value.slice.value.elts[1]).strip('\n')

                                        value_expr = 'c_pmt_pmat_get_el(' + value \
                                            + ', {}, {})'.format(first_index_value, \
                                            second_index_value) 

                                        self.statement([], 'c_pmt_pmat_set_el(', \
                                            target, ', {}'.format(first_index), ', \
                                            {}'.format(second_index), ', {}'.format( \
                                            value_expr), ');')

                                    elif self.typed_record[scope][value] == 'pvec':
                                        # if value is a pvec
                                        sub_type = type(node.value.slice.value)
                                        if sub_type == ast.Num: 
                                            index_value = node.value.slice.value.n
                                        elif sub_type == ast.Name: 
                                            index_value = node.value.slice.value.id
                                        else:
                                            raise cgenException('Subscripting \
                                                with value of type {} not \
                                                implemented.'.format(sub_type), \
                                                node.lineno)

                                        value_expr = 'c_pmt_pvec_get_el(' + value + \
                                            ', {})'.format(index_value) 
                                        self.statement([], 'c_pmt_pmat_set_el(', \
                                            target, ', {}'.format(first_index), ', \
                                            {}'.format(second_index), ', {}'.format(value_expr), ');')
                                else: 
                                    raise cgenException('Undefined variable {}.'.format(value), node.lineno)
                            else:
                                value = Num_or_Name(node.value)
                                self.statement([], 'c_pmt_pmat_set_el(', target, ', {}'.format(first_index), ', {}'.format(second_index), ', {}'.format(value), ');')
                        return

                    # check for pvec
                    elif self.typed_record[self.scope][target] in ('pvec'):
                        if type(node.targets[0].slice.value) not in (ast.Num, ast.Name):
                            # ap.pprint(node)
                            raise cgenException('Subscript to a pvec must \
                                object must be of type Num or Name.', node.lineno)
                        target = node.targets[0].value.id
                        if target in self.typed_record[self.scope]: 
                            # map subscript for pvec to blasfeo el assign
                            if self.typed_record[self.scope][target] in ('pvec'):
                                target = node.targets[0]
                                sub_type = type(target.slice.value)
                                if sub_type == ast.Num:
                                    index = node.targets[0].slice.value.n
                                elif sub_type == ast.Name:
                                    index = node.targets[0].slice.value.id
                                else:
                                    raise cgenException('Subscripting with \
                                        value of type {} not implemented'.format(sub_type), node.lineno)

                                # check if subscripted expression is used in the value
                                if type(node.value) == ast.Subscript:
                                    # if value is a pmat
                                    value = node.value.value.id
                                    if value in self.typed_record[self.scope]:
                                        if self.typed_record[self.scope][value] == 'pmat':
                                            first_index_value = Num_or_Name(node.value.slice.value.elts[0])
                                            second_index_value = Num_or_Name(node.value.slice.value.elts[1])
                                            value_expr = 'c_pmt_pmat_get_el(' + value + ', {}, {})'.format(first_index_value, second_index_value) 
                                        # single subscripting
                                        else:
                                            value = node.value.value.id
                                            # if value is a pvec
                                            if self.typed_record[self.scope][value] == 'pvec':
                                                sub_type = type(node.value.slice.value)
                                                if sub_type == ast.Num: 
                                                    index_value = node.value.slice.value.n
                                                elif sub_type == ast.Name: 
                                                    index_value = node.value.slice.value.id
                                                else:
                                                    raise cgenException('Subscripting \
                                                        with value of type {} not \
                                                        implemented'.format(sub_type), \
                                                        node.lineno)

                                                value_expr = 'c_pmt_pvec_get_el(' + value + ', {})'.format(index_value) 
                                                self.statement([], 'c_pmt_pvec_set_el(', target.value.id, ', {}'.format(index), ', {}'.format(value_expr), ');')
                                else:
                                    target = node.targets[0].value.id
                                    value = Num_or_Name(node.value)
                                    self.statement([], 'c_pmt_pvec_set_el(', target, ', {}'.format(index), ', {}'.format(value), ');')
                                return

            elif type(node.value) == ast.Subscript:
                target = node.targets[0].id
                if target not in self.typed_record[self.scope]:
                    raise cgenException('Undefined variable {}.'.format(target), node.lineno)
                # target must be a float
                if self.typed_record[self.scope][target] != 'float':

                    raise cgenException('Target must be a float {}.'.format(target), node.lineno)
                value = node.value.value.id
                if value in self.typed_record[self.scope]:
                    # if value is a pmat
                    if self.typed_record[self.scope][value] == 'pmat':
                        first_index_value = Num_or_Name(node.value.slice.value.elts[0])
                        second_index_value = Num_or_Name(node.value.slice.value.elts[1])
                        value_expr = 'c_pmt_pmat_get_el(' + value + ', {}, {})'.format(first_index_value, second_index_value) 
                        self.statement([], target, ' = {}'.format(value_expr), ';')
                        return
                    # single subscripting
                    else:
                        # if value is a pvec
                        if self.typed_record[self.scope][value] == 'pvec':
                            sub_type = type(node.value.slice.value)
                            if sub_type == ast.Num: 
                                index_value = node.value.slice.value.n
                            elif sub_type == ast.Name: 
                                index_value = node.value.slice.value.id
                            else:
                                raise cgenException('Subscripting with value \
                                    of type {} not implemented.'.format(sub_type), \
                                    node.lineno)

                            value_expr = 'c_pmt_pvec_get_el(' + value + ', {})'.format(index_value) 
                            self.statement([], target, ' = {}'.format(value_expr), ';')
                            return
            elif 'id' in node.targets[0].__dict__: 

                # check for Assigns targeting pmats
                target = node.targets[0].id
                # print(target)
                if target in self.typed_record[self.scope]: 
                    if self.typed_record[self.scope][target] == 'pmat':
                        if type(node.value) == ast.BinOp:
                            right_op = node.value.right.id
                            left_op = node.value.left.id
                            if right_op in self.typed_record[self.scope] and left_op in self.typed_record[self.scope]:
                                if self.typed_record[self.scope][left_op] == 'pmat' and self.typed_record[self.scope][right_op] == 'pmat':
                                    # dgemm
                                    if type(node.value.op) == ast.Mult:
                                        # set target to zero
                                        self.statement([], 'c_pmt_pmat_fill(', target, ', 0.0);')
                                        # call dgemm
                                        self.statement([], 'c_pmt_gemm_nn(', left_op, ', ', right_op, ', ', target, ', ', target, ');')
                                        return
                                    # dgead
                                    if type(node.value.op) == ast.Add:
                                        # set target to zero
                                        self.statement([], 'c_pmt_pmat_copy(', right_op, ', ', target, ');')
                                        # call dgead
                                        self.statement([], 'c_pmt_gead(1.0, ', left_op, ', ', target, ');')
                                        return
                                    # dgead (Sub)
                                    if type(node.value.op) == ast.Sub:
                                        # set target to zero
                                        self.statement([], 'c_pmt_pmat_copy(', left_op, ', ', target, ');')
                                        # call dgead
                                        self.statement([], 'c_pmt_gead(-1.0, ', right_op, ', ', target, ');')
                                        return
                                    else:
                                        raise cgenException('Unsupported operator call:{} {} {}.'\
                                            .format(self.typed_record[self.scope][left_op], \
                                            node.value.op, \
                                            self.typed_record[self.scope][right_op]), \
                                            node.lineno)

                            # elif self.typed_record[self.scope][node.value] == 'float':
                            #     import pdb; pdb.set_trace()
                            #     value = Num_or_Name(node.value)
                            #     self.statement([], 'c_pmt_pmat_set_el(', target.value.id, ', {}'.format(first_index), ', {}'.format(second_index), ', {}'.format(value_expr), ');')


                    elif self.typed_record[self.scope][target] == 'pvec':
                        if type(node.value) == ast.BinOp:
                            right_op = node.value.right.id
                            left_op = node.value.left.id
                            if right_op in self.typed_record[self.scope] and left_op in self.typed_record[self.scope]:
                                if self.typed_record[self.scope][left_op] == 'pmat' and self.typed_record[self.scope][right_op] == 'pvec':
                                    # dgemv
                                    if type(node.value.op) == ast.Mult:
                                        # set target to zero
                                        self.statement([], 'c_pmt_pvec_fill(', target, ', 0.0);')
                                        # call dgemm
                                        self.statement([], 'c_pmt_gemv_n(', left_op, ', ', right_op, ', ', target, ', ', target, ');')
                                        return
                                    # dgead
                                    else:
                                        raise cgenException('Unsupported operator call:{} {} {}.'\
                                            .format(self.typed_record[self.scope][left_op], \
                                            node.value.op, self.typed_record[self.scope][right_op]), \
                                            node.lineno)


            elif 'attr' in node.targets[0].__dict__: 
                # Assign targeting a user-defined class (C struct)
                struct_name = node.targets[0].value.id
                if struct_name in self.typed_record[self.scope]:
                    attr_value = node.value.n
                    attr_name = node.targets[0].attr 
                    self.statement([], struct_name, '->', attr_name, ' = ', str(attr_value), ';')
                else:
                    raise cgenException('Unknown variable {}.'.format(struct_name), node.lineno)
                return

            else:
                raise cgenException('Could not resolve Assign node.', node.lineno)

        set_precedence(node, node.value, *node.targets)
        self.newline(node)
        for target in node.targets:
            self.write(target, ' = ', dest = 'src')
        self.visit(node.value)
        self.write(';', dest = 'src')

    def visit_AugAssign(self, node):
        set_precedence(node, node.value, node.target)
        self.statement(node, node.target, get_op_symbol(node.op, ' %s= '),
                       node.value)

    def visit_AnnAssign(self, node):
        # ap.pprint(node)
        set_precedence(node, node.target, node.annotation)
        set_precedence(Precedence.Comma, node.value)
        need_parens = isinstance(node.target, ast.Name) and not node.simple
        begin = '(' if need_parens else ''
        end = ')' if need_parens else ''

        if 'value' not in node.__dict__:
            raise cgenException('Cannot declare variable without initialization.', node.lineno)

        ann = node.annotation.id
        # check if a CasADi function is being declared (and skip)
        if ann == 'ca':
            return
        elif ann == 'pfun':
            self.scope = self.scope + '@' + node.value.args[0].s
            self.typed_record[self.scope] = dict()
            self.var_dim_record[self.scope] = dict()
            self.heap8_record[self.scope] = 0
            self.heap64_record[self.scope] = 0
            self.scope = descope(self.scope, '@' + node.value.args[0].s)
            return
            
        # check if a List is being declared
        if ann is 'List':
            if node.value.func.id is not 'plist': 
                raise cgenException('Cannot create Lists without using \
                    plist constructor.', node.lineno)

            if len(node.value.args) != 2:
                raise cgenException('plist constructor takes 2 arguments \
                    plist(<type>, <sizes>).', node.lineno)

            # attribute is a List
            lann = node.value.args[0].id
            dims = Num_or_Name(node.value.args[1])
            if isinstance(dims, str):
                self.typed_record[self.scope][node.target.id] = 'List[' + lann + ', ' + dims + ']'
            else:
                self.typed_record[self.scope][node.target.id] = 'List[' + lann + ', ' + str(dims) + ']'
            if  lann in pmt_temp_types:
                lann = pmt_temp_types[lann]
            else:
                raise cgenException ('Usage of non existing type {}.'.format(lann), node.lineno)
            # check is dims is not a numerical value
            if isinstance(dims, str):
                dim_list = self.dim_record[dims]
                array_size = len(dim_list)
            else: 
                array_size = dims
                # array_size = str(Num_or_Name(node.value.args[1]))
                # self.statement([], lann, ' ', node.target, '[', array_size, '];')
            self.write('%s' %lann, ' ', '%s' %node.target.id, '[%s' %array_size, '];\n', dest = 'src')
            if lann == 'struct pmat *': 
                # build init for List of pmats
                for i in range(len(dim_list)):
                    self.statement([], node.target.id, \
                        '[', str(i),'] = c_pmt_create_pmat(', \
                        str(dim_list[i][0]), ', ', \
                        str(dim_list[i][1]), ');')

            elif lann == 'struct pvec *': 
                # build init for List of pvecs
                for i in range(len(dim_list)):
                    self.statement([], node.target.id, \
                        '[', str(i),'] = c_pmt_create_pvec(', \
                        str(dim_list[i][0]), ');')
            # self.conditional_write(' = ', node.value, '', dest = 'src')

        # pmat[<n>,<m>] 
        elif ann == 'pmat':
            if node.value.func.id != 'pmat':
                raise cgenException('pmat objects need to be declared calling', 
                    ' the pmat(<n>, <m>) constructor.', node.lineno)
            dim1 = Num_or_Name(node.value.args[0])
            dim2 = Num_or_Name(node.value.args[1])
            self.var_dim_record[self.scope][node.target.id] = [dim1, dim2]
            node.annotation.id = pmt_temp_types[ann]
            self.statement(node, node.annotation, ' ', node.target)
            self.conditional_write(' = ', node.value, '', dest = 'src')
            # increment scoped heap usage (3 pointers and 6 ints for pmats)
            self.heap8_record[self.scope] = self.heap8_record[self.scope] + 3*self.size_of_pointer
            self.heap8_record[self.scope] = self.heap8_record[self.scope] + 6*self.size_of_int
            # check is dims is not a numerical value
            if isinstance(dim1, str):
                if dim1 in self.dim_record:
                    dim1 = self.dim_record[dim1]
                else:
                    raise cgenException('Undefined variable {} of type dims.'.format(dim1), node.lineno)
            if isinstance(dim2, str):
                if dim2 in self.dim_record:
                    dim2 = self.dim_record[dim2]
                else:
                    raise cgenException('Undefined variable {} of type dims.'.format(dim2), node.lineno)
            self.heap64_record[self.scope] = self.heap64_record[self.scope] + int(dim1)*int(dim2)*self.size_of_double
        # or pvec[<n>]
        elif ann == 'pvec':
            if node.value.func.id != 'pvec':
                raise cgenException('pvec objects need to be declared calling the pvec(<n>, <m>) constructor.', node.lineno)
            dim1 = Num_or_Name(node.value.args[0])
            self.var_dim_record[self.scope][node.target.id] = [dim1]
            node.annotation.id = pmt_temp_types[ann]
            self.statement(node, node.annotation, ' ', node.target)
            self.conditional_write(' = ', node.value, '', dest = 'src')


        # or dims
        elif ann == 'dims':
            self.write('#define %s %s\n' %(node.target.id, node.value.n), dest='hdr')
            self.dim_record[node.target.id] = node.value.n
            # self.write('const int %s = %s;\n' %(node.target.id, node.value.n), dest='hdr')

        # or dimv
        elif ann == 'dimv':
            self.dim_record
            self.dim_record[node.target.id] = []
            for i in range(len(node.value.elts)):
                self.dim_record[node.target.id].append([])
                for j in range(len(node.value.elts[i].elts)):
                    self.dim_record[node.target.id][i].append(node.value.elts[i].elts[j].n)
                    self.write('#define %s_%s_%s %s\n' %(node.target.id, i, j, node.value.elts[i].elts[j].n), dest='hdr')

        # check if annotation corresponds to user-defined class name
        elif ann in usr_temp_types:
            class_name = node.annotation.id
            node.annotation.id = usr_temp_types[ann]
            self.statement([], 'struct ', class_name, ' ', node.target, '___;')
            self.statement(node, node.annotation, ' ', node.target, '= &', node.target, '___;')
            self.statement([], class_name, '_init(', node.target, '); //')
        else:
            if  ann in pmt_temp_types:
                c_ann = pmt_temp_types[ann]
                self.statement(node, c_ann, ' ', node.target.id)
                self.conditional_write(' = ', node.value, ';', dest = 'src')
            else:
                raise cgenException('\033[;1mUsage of non existing type\033[0;0m'
                    ' \033[1;31m{}\033[0;0m.'.format(ann), node.lineno)

        # print('typed_record = \n', self.typed_record, '\n\n')
        # print('var_dim_record = \n', self.var_dim_record, '\n\n')
        self.typed_record[self.scope][node.target.id] = ann

        # # switch to avoid double ';'
        # if type(node.value) != ast.Call:
        #     if node.value is not None:
        #         self.conditional_write(' = ', node.value, ';', dest = 'src')
        #     else:
        #         self.conditional_write(';', dest = 'src')
        # else:
        #     if node.value is not None:
        #         self.conditional_write(' = ', node.value, '', dest = 'src')
        #     else:
        #         self.conditional_write('', dest = 'src')

    def visit_ImportFrom(self, node):
        include = node.module
        include = include.replace('.','/')
        if node.level is not 0: 
            raise cgenException('Imports with level > 0 are not supported. Exiting.', node.lineno)
        if len(node.names) is not 1:
            raise cgenException('Imports with multiple names are not supported (yet). Exiting.', node.lineno)

        if node.names[0].name is not '*':
            raise cgenException('Can only transpile imports of the form: `from <...> import *`. Exiting.', node.lineno)
        self.statement(node, '#include "', 
                       include or '', '.h"\n')

    def visit_Import(self, node):
        raise cgenException('Unsupported import statement. Use instead from `<...> import *` Exiting.', node.lineno)
        # self.statement(node, 'import ')
        # self.comma_list(node.names)

    def visit_Expr(self, node):
        if type(node.value) is ast.Call:
            if 'value' in node.value.func.__dict__:
                var_name = node.value.func.value.id
                # check if we are calling a method on a pmat object
                if var_name in self.typed_record[self.scope]: 
                    if self.typed_record[self.scope][var_name] == 'pmat':
                        fun_name = node.value.func.attr 
                        # add prefix to function call
                        node.value.func.attr = 'c_pmt_pmat_' + fun_name 
        set_precedence(node, node.value)

        self.statement(node)
        self.generic_visit(node)

    def visit_FunctionDef(self, node, is_async=False):
        if node.name == 'main':
            self.in_main = True
        # ap.pprint(node)
        self.scope = self.scope + '@' + node.name
        self.typed_record[self.scope] = dict()
        self.var_dim_record[self.scope] = dict()
        self.heap8_record[self.scope] = 0
        self.heap64_record[self.scope] = 0
        prefix = 'is_async ' if is_async else ''
        self.decorators(node, 1 if self.indentation else 2)
        # self.write()
        returns = self.get_returns(node)
        if returns is None: 
            raise cgenException('Function {} does not have a \
                return type hint.', node.lineno)

        if isinstance(returns, ast.NameConstant):
            return_type_py = str(returns.value)
        elif isinstance(returns, ast.Name):
            return_type_py = returns.id
        else:
            raise cgenException('Unknown return object {}.'.format(returns), node.lineno)

        # print(return_type_py)
        return_type_c = pmt_temp_types[return_type_py]
        # function declaration
        self.write(return_type_c, ' %s' %(node.name), '(', dest = 'hdr')
        self.visit_arguments(node.args, 'hdr')
        self.write(');\n', dest = 'hdr')
        # function definition
        self.write(return_type_c, ' %s' %(node.name), '(', dest = 'src')
        self.visit_arguments(node.args, 'src')
        self.write(') {\n', dest = 'src')
        if node.name == 'main':
            self.write('    ___c_pmt_8_heap = malloc(%s); \n' %(self.heap8_size), dest = 'src')
            self.write('    ___c_pmt_8_heap_head = ___c_pmt_8_heap;\n', dest = 'src')
            self.write('    char * pmem_ptr = (char *)___c_pmt_8_heap;\n', dest = 'src')
            self.write('    align_char_to(8, &pmem_ptr);\n', dest = 'src')
            self.write('    ___c_pmt_8_heap = pmem_ptr;\n', dest = 'src')
            
            self.write('    ___c_pmt_64_heap = malloc(%s);\n' %(self.heap64_size), dest = 'src')
            self.write('    ___c_pmt_64_heap_head = ___c_pmt_64_heap;\n', dest = 'src')
            self.write('    pmem_ptr = (char *)___c_pmt_64_heap;\n', dest = 'src')
            self.write('    align_char_to(64, &pmem_ptr);\n', dest = 'src')
            self.write('    ___c_pmt_64_heap = pmem_ptr;\n', dest = 'src')

        # store current pmt_heap value (and restore before returning)
        self.write('\tvoid *callee_pmt_8_heap = ___c_pmt_8_heap;\n', dest = 'src')
        self.write('\tvoid *callee_pmt_64_heap = ___c_pmt_64_heap;\n', dest = 'src')

        # self.write(':')
        self.body(node.body)
        self.newline(1)
        if node.name == 'main':
            self.write('\tfree(___c_pmt_8_heap_head);\n', dest='src')
            self.write('\tfree(___c_pmt_64_heap_head);\n', dest='src')
            self.write('\treturn 0;\n', dest='src')
        self.write('}', dest='src')
        if not self.indentation:
            self.newline(extra=2)
        self.scope = descope(self.scope, '@' + node.name)
        self.in_main = False

    # introduced in Python 3.5
    def visit_AsyncFunctionDef(self, node):
        self.visit_FunctionDef(node, is_async=True)

    def visit_ClassDef(self, node):
        self.scope = self.scope + '@' + node.name
        self.typed_record[self.scope] = dict()
        self.var_dim_record[self.scope] = dict()
        self.heap8_record[self.scope] = 0 
        self.heap64_record[self.scope] = 0 
        have_args = []

        def paren_or_comma():
            if have_args:
                self.write(', ')
            else:
                have_args.append(True)
                self.write('(')
        # add new type to templated types
        usr_temp_types[node.name] = 'struct ' + node.name + ' *' 

        self.decorators(node, 0)
        self.write('typedef struct %s %s;\n\n' %(node.name, node.name), dest = 'hdr')
        self.write('struct %s' %node.name, dest = 'hdr')
        for base in node.bases:
            self.write(paren_or_comma, base)
        # keywords not available in early version
        for keyword in self.get_keywords(node):
            self.write(paren_or_comma, keyword.arg or '',
                       '=' if keyword.arg else '**', keyword.value)
        self.conditional_write(paren_or_comma, '*', self.get_starargs(node), dest = 'src')
        self.conditional_write(paren_or_comma, '**', self.get_kwargs(node), dest = 'src')
        self.write(have_args and ')' or '', dest = 'src')
        self.write('{\n', dest = 'hdr')
        self.body_class(node.body, node.name)
        self.scope = descope(self.scope, '@' + node.name)


    def visit_If(self, node):
        set_precedence(node, node.test)
        self.statement(node, 'if(', node.test, ') {')
        self.body(node.body)
        while True:
            else_ = node.orelse
            if len(else_) == 1 and isinstance(else_[0], ast.If):
                node = else_[0]
                set_precedence(node, node.test)
                self.write('\n', 'elif ', node.test, ':')
                self.body(node.body)
            else:
                self.else_body(else_)
                break
        self.write('\n}', dest = 'src')

    def visit_For(self, node, is_async=False):
        set_precedence(node, node.target)
        prefix = 'is_async ' if is_async else ''
        if len(node.iter.args) == 1:
            # range(<value>)
            range_value = astu.unparse(node.iter.args[0]).strip('\n') 
            self.statement(node, 'for(int ',
                    node.target, ' = 0; ', node.target, 
                    ' < {}'.format(range_value), 
                    '; ',node.target, '++) {')
        elif len(node.iter.args) == 2:
            # range(<value>, <value>)
            range_value_1 = astu.unparse(node.iter.args[0]).strip('\n') 
            range_value_2 = astu.unparse(node.iter.args[1]).strip('\n') 
            increment = 1
            self.statement(node, 'for(int ',
                    node.target, ' = {}; '.format(range_value_1), node.target, 
                    ' < {}'.format(range_value_2), 
                    '; ',node.target, '+={})'.format(increment), ' {')
        else:
            raise cgenException('Too many arguments for range().', node.lineno)

        self.body_or_else(node)
        self.write('\n    }\n', dest = 'src')

    # introduced in Python 3.5
    def visit_AsyncFor(self, node):
        self.visit_For(node, is_async=True)

    def visit_While(self, node):
        set_precedence(node, node.test)
        self.statement(node, 'while(', node.test, ') {')
        self.body_or_else(node)
        self.write('\n    }\n', dest = 'src')


    def visit_With(self, node, is_async=False):
        prefix = 'is_async ' if is_async else ''
        self.statement(node, '%swith ' % prefix)
        if hasattr(node, 'context_expr'):  # Python < 3.3
            self.visit_withitem(node)
        else:                              # Python >= 3.3
            self.comma_list(node.items)
        self.write(':')
        self.body(node.body)

    # new for Python 3.5
    def visit_AsyncWith(self, node):
        self.visit_With(node, is_async=True)

    # new for Python 3.3
    def visit_withitem(self, node):
        self.write(node.context_expr)
        self.conditional_write(' as ', node.optional_vars, dest = 'src')

    def visit_NameConstant(self, node):
        self.write(str(node.value), dest = 'src')

    def visit_Pass(self, node):
        self.statement(node, 'pass')

    def visit_Print(self, node):
        # XXX: python 2.6 only
        self.statement(node, 'print ')
        values = node.values
        if node.dest is not None:
            self.write(' >> ')
            values = [node.dest] + node.values
        self.comma_list(values, not node.nl)

    def visit_Delete(self, node):
        self.statement(node, 'del ')
        self.comma_list(node.targets)

    def visit_TryExcept(self, node):
        self.statement(node, 'try:')
        self.body(node.body)
        self.write(*node.handlers)
        self.else_body(node.orelse)

    # new for Python 3.3
    def visit_Try(self, node):
        self.statement(node, 'try:')
        self.body(node.body)
        self.write(*node.handlers)
        self.else_body(node.orelse)
        if node.finalbody:
            self.statement(node, 'finally:')
            self.body(node.finalbody)

    def visit_ExceptHandler(self, node):
        self.statement(node, 'except')
        if self.conditional_write(' ', node.type, dest = 'src'):
            self.conditional_write(' as ', node.name, dest = 'src')
        self.write(':')
        self.body(node.body)

    def visit_TryFinally(self, node):
        self.statement(node, 'try:')
        self.body(node.body)
        self.statement(node, 'finally:')
        self.body(node.finalbody)

    def visit_Exec(self, node):
        dicts = node.globals, node.locals
        dicts = dicts[::-1] if dicts[0] is None else dicts
        self.statement(node, 'exec ', node.body)
        self.conditional_write(' in ', dicts[0], dest = 'src')
        self.conditional_write(', ', dicts[1], dest = 'src')

    def visit_Assert(self, node):
        set_precedence(node, node.test, node.msg)
        self.statement(node, 'assert ', node.test)
        self.conditional_write(', ', node.msg, dest = 'src')

    def visit_Global(self, node):
        self.statement(node, 'global ', ', '.join(node.names))

    def visit_Nonlocal(self, node):
        self.statement(node, 'nonlocal ', ', '.join(node.names))

    def visit_Return(self, node):
        set_precedence(node, node.value)
        # TODO(andrea): this probably does not support 
        # stuff like `return foo()`
        # TODO(andrea): need to check type of return!!

        # restore pmt_heap values 
        self.write('\n\t___c_pmt_8_heap = callee_pmt_8_heap;\n', dest = 'src')
        self.write('\t___c_pmt_64_heap = callee_pmt_64_heap;\n', dest = 'src')
        if self.in_main is False:
            self.statement(node, 'return ')
            self.conditional_write('', node.value, ';', dest = 'src')

    def visit_Break(self, node):
        self.statement(node, 'break')

    def visit_Continue(self, node):
        self.statement(node, 'continue')

    def visit_Raise(self, node):
        # XXX: Python 2.6 / 3.0 compatibility
        self.statement(node, 'raise')
        if self.conditional_write(' ', self.get_exc(node), dest = 'src'):
            self.conditional_write(' from ', node.cause, dest = 'src')
        elif self.conditional_write(' ', self.get_type(node), dest = 'src'):
            set_precedence(node, node.inst)
            self.conditional_write(', ', node.inst, dest = 'src')
            self.conditional_write(', ', node.tback, dest = 'src')

    # Expressions

    def visit_Attribute(self, node):

        if node.value.id == 'self':
            self.write(node.value, '->', node.attr, dest = 'src')
        else:
            if  self.typed_record[self.scope][node.value.id] in usr_temp_types: 
                self.write(node.value, '->', node.attr, dest = 'src')
            else:
                raise cgenException('Accessing attribute of object {} \
                    of unknown type.'.format(node.value), node.lineno)

    def visit_Call(self, node, len=len):
        write = self.write
        want_comma = []

        def write_comma():
            if want_comma:
                # write(', ', dest = 'src')
                write(', ', dest = 'src')
            else:
                want_comma.append(True)
       
        # treat print separately
        if hasattr(node.func, 'id'):
            if node.func.id == 'print':
                if hasattr(node.args[0], 'op'):
                    if isinstance(node.args[0].op, ast.Mod):
                        # print string with arguments
                        write('printf("%s' %repr(node.args[0].left.s)[1:-1], '\\n", %s);\n' %node.args[0].right.id, dest = 'src')
                        return
                    else:
                        raise cgenException('Invalid operator in \
                            call to print().', node.lineno)

                else:
                    # print string with no arguments
                    write('printf("%s\\n");\n' %repr(node.args[0].s)[1:-1], dest = 'src')
                    return
            elif node.func.id == 'pparse':

                # update records
                os.chdir('__pmt_cache__')
                json_file = 'current_typed_record.json'
                with open(json_file, 'w') as f:
                    json.dump(self.typed_record[self.scope], f, indent=4, sort_keys=True)

                json_file = 'var_dim_record.json'
                with open(json_file, 'w') as f:
                    json.dump(self.var_dim_record[self.scope], f, indent=4, sort_keys=True)

                json_file = 'dim_record.json'
                with open(json_file, 'w') as f:
                    json.dump(self.dim_record, f, indent=4, sort_keys=True)


                os.chdir('..')

                expr = node.args[0].s
                # pass string to laparser
                try:
                    parser = LAParser( \
                        './__pmt_cache__/current_typed_record.json', 
                        './__pmt_cache__/var_dim_record.json', 
                        './__pmt_cache__/dim_record.json')
                    laparser_out = parser.parse(expr)
                except Exception as e:
                    print('\n > laparser exception: ', e)
                    raise cgenException('call to laparser failed', node.lineno)
                write(laparser_out, dest = 'src')
                return


        args = node.args
        keywords = node.keywords
        starargs = self.get_starargs(node)
        kwargs = self.get_kwargs(node)
        numargs = len(args) + len(keywords)
        numargs += starargs is not None
        numargs += kwargs is not None
        p = Precedence.Comma if numargs > 1 else Precedence.call_one_arg
        set_precedence(p, *args)

        if type(node.func) == ast.Name: 
            if  node.func.id in pmt_temp_functions:
                func_name = node.func.id
                node.func.id = pmt_temp_functions[func_name]
        elif type(node.func) == ast.Attribute: 
            # calling a method of a user-defined class
            func_name = node.func.attr
            f_name_len = len(func_name)
            pre_mangl = '_Z%s' %f_name_len 
            post_mangl = self.build_arg_mangling_mod(args)
            node.func.attr = pre_mangl + func_name + post_mangl

        self.visit(node.func)
        if type(node.func) == ast.Attribute: 
            if len(args) > 0:
                code = '(' +  node.func.value.id + ', '
            else:
                code = '(' +  node.func.value.id
            write(code, dest = 'src')
        else:
            write('(', dest = 'src')

        for arg in args:
            write(write_comma, arg, dest = 'src')

        set_precedence(Precedence.Comma, *(x.value for x in keywords))
        for keyword in keywords:
            # a keyword.arg of None indicates dictionary unpacking
            # (Python >= 3.5)
            arg = keyword.arg or ''
            write(write_comma, arg, '=' if arg else '**', keyword.value)
        # 3.5 no longer has these
        self.conditional_write(write_comma, '*', starargs, dest = 'src')
        self.conditional_write(write_comma, '**', kwargs, dest = 'src')
        # write(');\n', dest = 'src')
        write(');', dest = 'src')

    def visit_Name(self, node):
        self.write(node.id, dest = 'src')

    def visit_JoinedStr(self, node):
        self.visit_Str(node, True)

    def visit_Str(self, node, is_joined=False):

        # embedded is used to control when we might want
        # to use a triple-quoted string.  We determine
        # if we are in an assignment and/or in an expression
        precedence = self.get__pp(node)
        embedded = ((precedence > Precedence.Expr) +
                    (precedence >= Precedence.Assign))

        # Flush any pending newlines, because we're about
        # to severely abuse the result.source list.
        self.write('', dest = 'src')
        # result.source = self.result.source

        # Calculate the string representing the line
        # we are working on, up to but not including
        # the string we are adding.

        res_index, str_index = self.colinfo
        current_line = self.result.source[res_index:]
        if str_index:
            current_line[0] = current_line[0][str_index:]
        current_line = ''.join(current_line)

        if is_joined:

            # Handle new f-strings.  This is a bit complicated, because
            # the tree can contain subnodes that recurse back to JoinedStr
            # subnodes...

            def recurse(node):
                for value in node.values:
                    if isinstance(value, ast.Str):
                        self.write(value.s)
                    elif isinstance(value, ast.FormattedValue):
                        with self.delimit('{}'):
                            self.visit(value.value)
                            if value.conversion != -1:
                                self.write('!%s' % chr(value.conversion))
                            if value.format_spec is not None:
                                self.write(':')
                                recurse(value.format_spec)
                    else:
                        kind = type(value).__name__
                        assert False, 'Invalid node %s inside JoinedStr' % kind

            index = len(result.source)
            recurse(node)
            mystr = ''.join(result.source[index:])
            del result.source[index:]
            self.colinfo = res_index, str_index  # Put it back like we found it
            uni_lit = False  # No formatted byte strings

        else:
            mystr = node.s
            uni_lit = self.using_unicode_literals

        mystr = self.pretty_string(mystr, embedded, current_line, uni_lit)

        if is_joined:
            mystr = 'f' + mystr

        self.write(mystr, dest = 'src')

        lf = mystr.rfind('\n') + 1
        if lf:
            self.colinfo = len(result.source) - 1, lf

    def visit_Bytes(self, node):
        self.write(repr(node.s))

    def visit_Num(self, node,
                  # constants
                  new=sys.version_info >= (3, 0)):
        with self.delimit(node) as delimiters:
            s = repr(node.n)

            # Deal with infinities -- if detected, we can
            # generate them with 1e1000.
            signed = s.startswith('-')
            if s[signed].isalpha():
                im = s[-1] == 'j' and 'j' or ''
                assert s[signed:signed + 3] == 'inf', s
                s = '%s1e1000%s' % ('-' if signed else '', im)
            self.write(s, dest = 'src')

            # The Python 2.x compiler merges a unary minus
            # with a number.  This is a premature optimization
            # that we deal with here...
            if not new and delimiters.discard:
                if signed:
                    pow_lhs = Precedence.Pow + 1
                    delimiters.discard = delimiters.pp != pow_lhs
                else:
                    op = self.get__p_op(node)
                    delimiters.discard = not isinstance(op, ast.USub)

    def visit_Tuple(self, node):
        with self.delimit(node) as delimiters:
            # Two things are special about tuples:
            #   1) We cannot discard the enclosing parentheses if empty
            #   2) We need the trailing comma if only one item
            elts = node.elts
            delimiters.discard = delimiters.discard and elts
            self.comma_list(elts, len(elts) == 1)

    def visit_List(self, node):
        with self.delimit('[]'):
            self.comma_list(node.elts)

    def visit_Set(self, node):
        with self.delimit('{}'):
            self.comma_list(node.elts)

    def visit_Dict(self, node):
        set_precedence(Precedence.Comma, *node.values)
        with self.delimit('{}'):
            for idx, (key, value) in enumerate(zip(node.keys, node.values)):
                self.write(', ' if idx else '',
                           key if key else '',
                           ': ' if key else '**', value)

    def visit_BinOp(self, node):
        op, left, right = node.op, node.left, node.right
        with self.delimit(node, op) as delimiters:
            ispow = isinstance(op, ast.Pow)
            p = delimiters.p
            set_precedence((Precedence.Pow + 1) if ispow else p, left)
            set_precedence(Precedence.PowRHS if ispow else (p + 1), right)
            self.write(left, get_op_symbol(op, ' %s '), right, dest = 'src')

    def visit_BoolOp(self, node):
        with self.delimit(node, node.op) as delimiters:
            op = get_op_symbol(node.op, ' %s ')
            set_precedence(delimiters.p + 1, *node.values)
            for idx, value in enumerate(node.values):
                self.write(idx and op or '', value)

    def visit_Compare(self, node):
        with self.delimit(node, node.ops[0]) as delimiters:
            set_precedence(delimiters.p + 1, node.left, *node.comparators)
            self.visit(node.left)
            for op, right in zip(node.ops, node.comparators):
                self.write(get_op_symbol(op, ' %s '), right, dest = 'src')

    def visit_UnaryOp(self, node):
        with self.delimit(node, node.op) as delimiters:
            set_precedence(delimiters.p, node.operand)
            # In Python 2.x, a unary negative of a literal
            # number is merged into the number itself.  This
            # bit of ugliness means it is useful to know
            # what the parent operation was...
            node.operand._p_op = node.op
            sym = get_op_symbol(node.op)
            self.write(sym, ' ' if sym.isalpha() else '', node.operand, dest = 'src')

    def visit_Subscript(self, node):
        set_precedence(node, node.slice)
        self.write(node.value, '[', node.slice, ']', dest = 'src')

    def visit_Slice(self, node):
        set_precedence(node, node.lower, node.upper, node.step)
        self.conditional_write(node.lower, dest = 'src')
        self.write(':')
        self.conditional_write(node.upper, dest = 'src')
        if node.step is not None:
            self.write(':')
            if not (isinstance(node.step, ast.Name) and
                    node.step.id == 'None'):
                self.visit(node.step)

    def visit_Index(self, node):
        with self.delimit(node) as delimiters:
            set_precedence(delimiters.p, node.value)
            self.visit(node.value)

    def visit_ExtSlice(self, node):
        dims = node.dims
        set_precedence(node, *dims)
        self.comma_list(dims, len(dims) == 1)

    def visit_Yield(self, node):
        with self.delimit(node):
            set_precedence(get_op_precedence(node) + 1, node.value)
            self.write('yield')
            self.conditional_write(' ', node.value, dest = 'src')

    # new for Python 3.3
    def visit_YieldFrom(self, node):
        with self.delimit(node):
            self.write('yield from ', node.value)

    # new for Python 3.5
    def visit_Await(self, node):
        with self.delimit(node):
            self.write('await ', node.value)

    def visit_Lambda(self, node):
        with self.delimit(node) as delimiters:
            set_precedence(delimiters.p, node.body)
            self.write('lambda ')
            self.visit_arguments(node.args, 'src')
            self.write(': ', node.body)

    def visit_Ellipsis(self, node):
        self.write('...')

    def visit_ListComp(self, node):
        with self.delimit('[]'):
            self.write(node.elt, *node.generators)

    def visit_GeneratorExp(self, node):
        with self.delimit(node) as delimiters:
            if delimiters.pp == Precedence.call_one_arg:
                delimiters.discard = True
            set_precedence(Precedence.Comma, node.elt)
            self.write(node.elt, *node.generators)

    def visit_SetComp(self, node):
        with self.delimit('{}'):
            self.write(node.elt, *node.generators)

    def visit_DictComp(self, node):
        with self.delimit('{}'):
            self.write(node.key, ': ', node.value, *node.generators)

    def visit_IfExp(self, node):
        with self.delimit(node) as delimiters:
            set_precedence(delimiters.p + 1, node.body, node.test)
            set_precedence(delimiters.p, node.orelse)
            self.write(node.body, ' if ', node.test, ' else ', node.orelse)

    def visit_Starred(self, node):
        self.write('*', node.value)

    def visit_Repr(self, node):
        # XXX: python 2.6 only
        with self.delimit('``'):
            self.visit(node.value)

    def visit_Module(self, node):
        self.write(*node.body, dest = 'src')

    visit_Interactive = visit_Module

    def visit_Expression(self, node):
        self.visit(node.body)

    # Helper Nodes

    def visit_arg(self, node):
        self.write(node.arg)
        self.conditional_write(': ', node.annotation, dest = 'src')

    def visit_alias(self, node):
        self.write(node.name, dest = 'src')
        self.conditional_write(' as ', node.asname, dest = 'src')

    def visit_comprehension(self, node):
        set_precedence(node, node.iter, *node.ifs)
        set_precedence(Precedence.comprehension_target, node.target)
        stmt = ' is_async for ' if self.get_is_is_async(node) else ' for '
        self.write(stmt, node.target, ' in ', node.iter)
        for if_ in node.ifs:
            self.write(' if ', if_)
