# pylint: disable=line-too-long, unused-wildcard-import
"""
The GR4J model (modèle du Génie Rural à 4 parametres Journalier) is a daily
lumped four-parameter rainfall-runoff model and belongs to the family of soil moisture
accounting models.  It was initially published by :cite:t:`ref-Perrin2003` and is a
modification of GR3J.  Our implementation, |gland_gr4|, follows the one of the R
package airGR :cite:p:`ref-airGR2017` but with a few extensions.  Namely, it applies to
arbitrary simulation step sizes and offers more flexibility in simulating interception
and runoff concentration.

The following figure :cite:p:`ref-airGRManual` shows the general structure of
|gland_gr4|:

.. image:: HydPy-G-GR4.png

Integration tests
=================

.. how_to_understand_integration_tests::

The following integration tests rely on the example dataset L0123001 of airGR and allow
for comparing both model implementations.

All tests cover 50 days with a simulation step of one day:

>>> from hydpy import pub
>>> pub.timegrids = "1990-01-01", "1990-02-20", "1d"

Prepare a model instance and pass it to an element connected to a single outlet node:

>>> from hydpy.models.gland_gr4 import *
>>> parameterstep("1d")
>>> from hydpy import Element
>>> land = Element("land", outlets="outlet")
>>> land.model = model

The size of the (lumpy simulated) catchment is 360 km²:

>>> area(360.0)

|gland_gr4| requires a submodel that provides potential evaporation estimates.  We
select |evap_ret_io|, which allows using a predefined time series:

>>> with model.add_petmodel_v1("evap_ret_io"):
...     evapotranspirationfactor(1.0)

Initialise a test function object, which prepares and runs the tests and prints their
results for the selected sequences:

>>> from hydpy import IntegrationTest
>>> test = IntegrationTest(land)
>>> test.plotting_options.axis1 = inputs.p, fluxes.e, fluxes.qh
>>> test.dateformat = "%d.%m."

The mentioned precipitation and potential evapotranspiration time series stem from the
mentioned dataset:

>>> inputs.p.series = (
...     0.0, 9.3, 3.2, 7.3, 0.0, 0.0, 0.0, 0.0, 0.1, 0.2, 2.9, 0.2, 0.0, 0.0, 0.0, 3.3,
...     4.6, 0.8, 1.8, 1.1, 0.0, 5.0, 13.1, 14.6, 4.0, 0.8, 0.1, 3.3, 7.7, 10.3, 3.7,
...     15.3, 3.2, 2.7, 2.2, 8.0, 14.3, 6.3, 0.0, 5.9, 9.2, 6.1, 0.1, 0.0, 2.8, 10.6,
...     8.8, 7.2, 4.9, 1.8
... )
>>> model.petmodel.sequences.inputs.referenceevapotranspiration.series = (
...     0.3, 0.4, 0.4, 0.3, 0.1, 0.1, 0.1, 0.2, 0.2, 0.3, 0.3, 0.2, 0.2, 0.2, 0.2, 0.3,
...     0.3, 0.2, 0.2, 0.3, 0.2, 0.2, 0.3, 0.6, 0.4, 0.4, 0.4, 0.5, 0.4, 0.3, 0.3, 0.5,
...     0.5, 0.3, 0.3, 0.4, 0.4, 0.3, 0.2, 0.1, 0.1, 0.0, 0.1, 0.1, 0.0, 0.2, 0.9, 0.9,
...     0.5, 0.9
... )

.. _gland_gr4_base_example:

base example
____________

We take the following values of the first three of four original GR4J parameters |X1|,
|X2|, and |X3| directly from the airGR manual:

>>> x1(257.238)
>>> x2(1.012)
>>> x3(88.235)

The parameter |IMax| is unavailable in GR4J, which always assumes an interception
storage of zero capacity.  Hence, consistency with GR4J requires setting |IMax| to
zero:

>>> imax(0.0)

The fourth original GR4J parameter, `X4`, defines the time base of two Unit
Hydrographs.  Unlike GR4J, |gland_gr4| does not rely on hard-coded Unit Hydrographs and
has no parameter named `X4` but provides two optional submodel ports that allow adding
different types of runoff concentration submodels.  For consistency with GR4J, you can
select the |rconc_uh| submodel for both ports.  Its options `gr_uh2` and `gr_uh1`
correspond to the typical configuration of the Unit Hydrographs of the direct flow and
the routing store component, respectively.  Passing the value of the original `X4`
parameter as a function argument in both cases completes the desired consistency (note
that, for non-daily simulations, you might also want to adjust the other function
argument `beta`, which defaults to 2.5):

>>> with model.add_rconcmodel_directflow_v1("rconc_uh"):
...     uh("gr_uh2", x4=2.208)
>>> with model.add_rconcmodel_routingstore_v1("rconc_uh"):
...     uh("gr_uh1", x4=2.208)

Now that all models have been selected and parameterised, we can define their initial
conditions:

>>> test.inits = (
...     (states.i, 0.0),
...     (states.s, 0.3 * x1),
...     (states.r, 0.5 * x3),
...     (model.rconcmodel_routingstore.sequences.logs.quh, 0.0),
...     (model.rconcmodel_directflow.sequences.logs.quh, 0.0),
... )

The following simulation results agree with those of the GR4J airGR implementation:

.. integration-test::

    >>> conditions = test("gland_gr4_base_example", get_conditions="1990-01-01")
    |   date |    p |   e |  en |   pn |        ps |  ei |       es |       ae |       pr |      pr9 |      pr1 |     perc |       q9 |       q1 |       fd |       fr |       qr |       qd |       qh |        qv |   i |          s |         r |    outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |  0.0 | 0.3 | 0.3 |  0.0 |       0.0 | 0.0 | 0.152875 | 0.152875 | 0.006036 | 0.005433 | 0.000604 | 0.006036 |  0.00075 | 0.000042 | 0.089449 | 0.089449 | 0.670218 | 0.089491 | 0.759709 |  3.165453 | 0.0 |  77.012489 | 43.537481 |  3.165453 |
    | 02.01. |  9.3 | 0.4 | 0.0 |  8.9 |  8.016066 | 0.4 |      0.0 |      0.4 | 0.893833 | 0.804449 | 0.089383 | 0.009898 | 0.114538 | 0.006363 |   0.0854 |   0.0854 | 0.636315 | 0.091763 | 0.728078 |  3.033659 | 0.0 |  85.018656 | 43.101104 |  3.033659 |
    | 03.01. |  3.2 | 0.4 | 0.0 |  2.8 |  2.485106 | 0.4 |      0.0 |      0.4 | 0.326319 | 0.293687 | 0.032632 | 0.011425 | 0.558854 |  0.03125 | 0.082442 | 0.082442 | 0.636667 | 0.113691 | 0.750358 |  3.126493 | 0.0 |  87.492337 | 43.105732 |  3.126493 |
    | 04.01. |  7.3 | 0.3 | 0.0 |  7.0 |  6.131951 | 0.3 |      0.0 |      0.3 | 0.884068 | 0.795661 | 0.088407 | 0.016019 | 0.474904 | 0.056451 | 0.082473 | 0.082473 | 0.631081 | 0.138923 | 0.770004 |  3.208351 | 0.0 |   93.60827 | 43.032028 |  3.208351 |
    | 05.01. |  0.0 | 0.1 | 0.1 |  0.0 |       0.0 | 0.0 | 0.059523 | 0.059523 | 0.015954 | 0.014359 | 0.001595 | 0.015954 | 0.577812 | 0.057078 |  0.08198 |  0.08198 | 0.633099 | 0.139058 | 0.772158 |  3.217323 | 0.0 |  93.532793 | 43.058721 |  3.217323 |
    | 06.01. |  0.0 | 0.1 | 0.1 |  0.0 |       0.0 | 0.0 | 0.059485 | 0.059485 |  0.01589 | 0.014301 | 0.001589 |  0.01589 |  0.18556 | 0.045765 | 0.082158 | 0.082158 | 0.607783 | 0.127923 | 0.735706 |  3.065442 | 0.0 |  93.457417 | 42.718655 |  3.065442 |
    | 07.01. |  0.0 | 0.1 | 0.1 |  0.0 |       0.0 | 0.0 | 0.059448 | 0.059448 | 0.015826 | 0.014243 | 0.001583 | 0.015826 | 0.014306 | 0.015459 |  0.07991 |  0.07991 |  0.57353 | 0.095368 | 0.668898 |  2.787074 | 0.0 |  93.382143 | 42.239341 |  2.787074 |
    | 08.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.118792 | 0.118792 | 0.015712 | 0.014141 | 0.001571 | 0.015712 | 0.014242 | 0.002256 | 0.076815 | 0.076815 | 0.542728 | 0.079071 | 0.621799 |   2.59083 | 0.0 |  93.247639 |  41.78767 |   2.59083 |
    | 09.01. |  0.1 | 0.2 | 0.1 |  0.0 |       0.0 | 0.1 | 0.059344 | 0.159344 | 0.015649 | 0.014084 | 0.001565 | 0.015649 | 0.014156 | 0.001579 | 0.073979 | 0.073979 | 0.514894 | 0.075557 | 0.590452 |  2.460215 | 0.0 |  93.172646 |  41.36091 |  2.460215 |
    | 10.01. |  0.2 | 0.3 | 0.1 |  0.0 |       0.0 | 0.2 | 0.059307 | 0.259307 | 0.015586 | 0.014028 | 0.001559 | 0.015586 | 0.014089 |  0.00157 | 0.071368 | 0.071368 |  0.48963 | 0.072938 | 0.562568 |  2.344032 | 0.0 |  93.097752 | 40.956737 |  2.344032 |
    | 11.01. |  2.9 | 0.3 | 0.0 |  2.6 |  2.251138 | 0.3 |      0.0 |      0.3 | 0.366411 |  0.32977 | 0.036641 | 0.017549 | 0.057625 | 0.003985 | 0.068957 | 0.068957 | 0.469031 | 0.072942 | 0.541973 |  2.258219 | 0.0 |  95.331342 | 40.614288 |  2.258219 |
    | 12.01. |  0.2 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.017533 | 0.015779 | 0.001753 | 0.017533 | 0.217237 | 0.012849 |  0.06696 |  0.06696 | 0.458806 | 0.079809 | 0.538615 |  2.244229 | 0.0 |  95.313809 | 40.439679 |  2.244229 |
    | 13.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.120694 | 0.120694 | 0.017406 | 0.015665 | 0.001741 | 0.017406 | 0.084569 | 0.017242 | 0.065958 | 0.065958 | 0.442139 | 0.083199 | 0.525338 |  2.188908 | 0.0 |  95.175709 | 40.148066 |  2.188908 |
    | 14.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.120559 | 0.120559 |  0.01728 | 0.015552 | 0.001728 |  0.01728 | 0.015675 | 0.007222 | 0.064308 | 0.064308 | 0.423165 |  0.07153 | 0.494695 |  2.061229 | 0.0 |   95.03787 | 39.804884 |  2.061229 |
    | 15.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.120424 | 0.120424 | 0.017156 |  0.01544 | 0.001716 | 0.017156 | 0.015562 | 0.002006 | 0.062404 | 0.062404 | 0.405675 |  0.06441 | 0.470085 |  1.958689 | 0.0 |  94.900291 | 39.477175 |  1.958689 |
    | 16.01. |  3.3 | 0.3 | 0.0 |  3.0 |  2.580474 | 0.3 |      0.0 |      0.3 | 0.439125 | 0.395213 | 0.043913 |   0.0196 | 0.067888 | 0.004638 | 0.060625 | 0.060625 | 0.392045 | 0.065262 | 0.457308 |  1.905449 | 0.0 |  97.461166 | 39.213642 |  1.905449 |
    | 17.01. |  4.6 | 0.3 | 0.0 |  4.3 |  3.659234 | 0.3 |      0.0 |      0.3 | 0.664306 | 0.597876 | 0.066431 |  0.02354 | 0.339968 | 0.019747 |  0.05922 |  0.05922 | 0.392392 | 0.078967 | 0.471359 |  1.963996 | 0.0 | 101.096859 | 39.220437 |  1.963996 |
    | 18.01. |  0.8 | 0.2 | 0.0 |  0.6 |  0.506861 | 0.2 |      0.0 |      0.2 | 0.117247 | 0.105522 | 0.011725 | 0.024108 | 0.485502 |  0.04198 | 0.059256 | 0.059256 | 0.399845 | 0.101236 | 0.501081 |  2.087837 | 0.0 | 101.579612 |  39.36535 |  2.087837 |
    | 19.01. |  1.8 | 0.2 | 0.0 |  1.6 |  1.347178 | 0.2 |      0.0 |      0.2 |  0.27854 | 0.250686 | 0.027854 | 0.025718 | 0.233451 | 0.042152 | 0.060026 | 0.060026 | 0.394631 | 0.102177 | 0.496808 |  2.070032 | 0.0 | 102.901072 | 39.264196 |  2.070032 |
    | 20.01. |  1.1 | 0.3 | 0.0 |  0.8 |  0.671148 | 0.3 |      0.0 |      0.3 | 0.155386 | 0.139847 | 0.015539 | 0.026534 | 0.203576 | 0.026007 | 0.059488 | 0.059488 | 0.388254 | 0.085495 | 0.473749 |  1.973955 | 0.0 | 103.545686 | 39.139005 |  1.973955 |
    | 21.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.128546 | 0.128546 | 0.026336 | 0.023703 | 0.002634 | 0.026336 | 0.148103 |  0.01992 | 0.058826 | 0.058826 | 0.379599 | 0.078747 | 0.458345 |  1.909771 | 0.0 | 103.390804 | 38.966336 |  1.909771 |
    | 22.01. |  5.0 | 0.2 | 0.0 |  4.8 |  3.994165 | 0.2 |      0.0 |      0.2 | 0.837623 | 0.753861 | 0.083762 | 0.031788 | 0.149944 | 0.018003 | 0.057923 | 0.057923 |  0.37154 | 0.075926 | 0.447467 |  1.864445 | 0.0 | 107.353181 | 38.802663 |  1.864445 |
    | 23.01. | 13.1 | 0.3 | 0.0 | 12.8 | 10.347284 | 0.3 |      0.0 |      0.3 | 2.502985 | 2.252686 | 0.250298 | 0.050269 | 0.800756 | 0.048023 | 0.057076 | 0.057076 | 0.394712 | 0.105099 | 0.499811 |  2.082546 | 0.0 | 117.650196 | 39.265783 |  2.082546 |
    | 24.01. | 14.6 | 0.6 | 0.0 | 14.0 | 10.792227 | 0.6 |      0.0 |      0.6 | 3.285533 |  2.95698 | 0.328553 |  0.07776 | 2.021465 | 0.140924 | 0.059496 | 0.059496 | 0.483909 |  0.20042 | 0.684329 |   2.85137 | 0.0 | 128.364663 | 40.862836 |   2.85137 |
    | 25.01. |  4.0 | 0.4 | 0.0 |  3.6 |  2.684632 | 0.4 |      0.0 |      0.4 | 1.001334 | 0.901201 | 0.100133 | 0.085967 | 2.518868 | 0.237037 | 0.068405 | 0.068405 | 0.616263 | 0.305442 | 0.921704 |  3.840435 | 0.0 | 130.963328 | 42.833846 |  3.840435 |
    | 26.01. |  0.8 | 0.4 | 0.0 |  0.4 |  0.296087 | 0.4 |      0.0 |      0.4 |  0.19057 | 0.171513 | 0.019057 | 0.086657 | 1.250964 | 0.219605 | 0.080666 | 0.080666 | 0.667119 | 0.300271 |  0.96739 |  4.030791 | 0.0 | 131.172758 | 43.498357 |  4.030791 |
    | 27.01. |  0.1 | 0.4 | 0.3 |  0.0 |       0.0 | 0.1 | 0.227818 | 0.327818 | 0.085625 | 0.077063 | 0.008563 | 0.085625 | 0.318374 | 0.104767 | 0.085132 | 0.085132 | 0.648016 | 0.189898 | 0.837914 |   3.49131 | 0.0 | 130.859314 | 43.253847 |   3.49131 |
    | 28.01. |  3.3 | 0.5 | 0.0 |  2.8 |  2.063892 | 0.5 |      0.0 |      0.5 |  0.82839 | 0.745551 | 0.082839 | 0.092282 | 0.190038 | 0.035197 | 0.083469 | 0.083469 | 0.621605 | 0.118666 | 0.740271 |  3.084463 | 0.0 | 132.830924 | 42.905748 |  3.084463 |
    | 29.01. |  7.7 | 0.4 | 0.0 |  7.3 |  5.274803 | 0.4 |      0.0 |      0.4 | 2.136896 | 1.923206 |  0.21369 | 0.111699 | 0.761626 | 0.048947 | 0.081141 | 0.081141 | 0.637099 | 0.130088 | 0.767187 |  3.196613 | 0.0 | 137.994028 | 43.111416 |  3.196613 |
    | 30.01. | 10.3 | 0.3 | 0.0 | 10.0 |  6.973334 | 0.3 |      0.0 |      0.3 |  3.16895 | 2.852055 | 0.316895 | 0.142283 | 1.793361 |   0.1289 | 0.082511 | 0.082511 | 0.729464 | 0.211411 | 0.940874 |  3.920309 | 0.0 | 144.825078 | 44.257825 |  3.920309 |
    | 31.01. |  3.7 | 0.3 | 0.0 |  3.4 |  2.305017 | 0.3 |      0.0 |      0.3 | 1.248178 |  1.12336 | 0.124818 | 0.153195 | 2.409886 | 0.218607 | 0.090449 | 0.090449 | 0.878946 | 0.309056 | 1.188002 |  4.950008 | 0.0 |   146.9769 | 45.879214 |  4.950008 |
    | 01.02. | 15.3 | 0.5 | 0.0 | 14.8 |  9.640855 | 0.5 |      0.0 |      0.5 | 5.368374 | 4.831537 | 0.536837 | 0.209229 | 2.014047 | 0.252335 | 0.102587 | 0.102587 |  0.99654 | 0.354923 | 1.351463 |  5.631095 | 0.0 | 156.408526 | 46.999308 |  5.631095 |
    | 02.02. |  3.2 | 0.5 | 0.0 |  2.7 |  1.690954 | 0.5 |      0.0 |      0.5 | 1.228334 | 1.105501 | 0.122833 | 0.219288 | 3.504616 | 0.287945 | 0.111624 | 0.111624 | 1.284427 | 0.399569 | 1.683996 |  7.016651 | 0.0 | 157.880192 | 49.331121 |  7.016651 |
    | 03.02. |  2.7 | 0.3 | 0.0 |  2.4 |  1.487382 | 0.3 |      0.0 |      0.3 | 1.140817 | 1.026735 | 0.114082 | 0.228199 | 1.911124 | 0.308222 | 0.132239 | 0.132239 | 1.378427 | 0.440462 | 1.818889 |  7.578702 | 0.0 | 159.139375 | 49.996057 |  7.578702 |
    | 04.02. |  2.2 | 0.3 | 0.0 |  1.9 |   1.16747 | 0.3 |      0.0 |      0.3 | 0.967513 | 0.870762 | 0.096751 | 0.234983 | 1.022465 | 0.183223 | 0.138584 | 0.138584 | 1.350994 | 0.321807 | 1.672801 |  6.970004 | 0.0 | 160.071862 | 49.806111 |  6.970004 |
    | 05.02. |  8.0 | 0.4 | 0.0 |  7.6 |  4.571734 | 0.4 |      0.0 |      0.4 |  3.29669 | 2.967021 | 0.329669 | 0.268424 | 1.194307 | 0.128022 |  0.13675 |  0.13675 | 1.348499 | 0.264772 |  1.61327 |   6.72196 | 0.0 | 164.375172 | 49.788669 |   6.72196 |
    | 06.02. | 14.3 | 0.4 | 0.0 | 13.9 |  7.942363 | 0.4 |      0.0 |      0.4 | 6.294452 | 5.665006 | 0.629445 | 0.336815 |  2.88009 | 0.211302 | 0.136582 | 0.136582 | 1.569491 | 0.347884 | 1.917375 |  7.989064 | 0.0 |  171.98072 |  51.23585 |  7.989064 |
    | 07.02. |  6.3 | 0.3 | 0.0 |  6.0 |  3.266581 | 0.3 |      0.0 |      0.3 | 3.099733 | 2.789759 | 0.309973 | 0.366314 | 4.676893 | 0.386439 | 0.150989 | 0.150989 | 2.076895 | 0.537429 | 2.614323 | 10.893014 | 0.0 | 174.880987 | 53.986838 | 10.893014 |
    | 08.02. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.179455 | 0.179455 | 0.360668 | 0.324602 | 0.036067 | 0.360668 | 3.079531 | 0.434647 |  0.18132 |  0.18132 | 2.287879 | 0.615967 | 2.903846 | 12.099357 | 0.0 | 174.340864 |  54.95981 | 12.099357 |
    | 09.02. |  5.9 | 0.1 | 0.0 |  5.8 |  3.088152 | 0.1 |      0.0 |      0.1 | 3.101436 | 2.791292 | 0.310144 | 0.389587 | 1.205298 | 0.272248 | 0.193017 | 0.193017 | 2.127981 | 0.465265 | 2.593245 | 10.805188 | 0.0 | 177.039429 | 54.230144 | 10.805188 |
    | 10.02. |  9.2 | 0.1 | 0.0 |  9.1 |  4.673919 | 0.1 |      0.0 |      0.1 | 4.864791 | 4.378311 | 0.486479 |  0.43871 | 2.469831 | 0.202813 | 0.184196 | 0.184196 | 2.221524 | 0.387009 | 2.608533 | 10.868888 | 0.0 | 181.274638 | 54.662647 | 10.868888 |
    | 11.02. |  6.1 | 0.0 | 0.0 |  6.1 |  3.019732 | 0.0 |      0.0 |      0.0 | 3.550868 | 3.195781 | 0.355087 |   0.4706 | 3.867308 | 0.326898 | 0.189389 | 0.189389 | 2.571137 | 0.516287 | 3.087424 | 12.864268 | 0.0 |  183.82377 | 56.148207 | 12.864268 |
    | 12.02. |  0.1 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.464652 | 0.418187 | 0.046465 | 0.464652 | 3.071495 | 0.382728 | 0.208024 | 0.208024 | 2.716044 | 0.590752 | 3.306796 | 13.778317 | 0.0 | 183.359118 | 56.711683 | 13.778317 |
    | 13.02. |  0.0 | 0.1 | 0.1 |  0.0 |       0.0 | 0.0 | 0.091741 | 0.091741 | 0.457698 | 0.411928 |  0.04577 | 0.457698 | 1.025984 | 0.254821 | 0.215423 | 0.215423 | 2.420694 | 0.470244 | 2.890938 | 12.045576 | 0.0 | 182.809679 | 55.532396 | 12.045576 |
    | 14.02. |  2.8 | 0.0 | 0.0 |  2.8 |  1.375188 | 0.0 |      0.0 |      0.0 | 1.894022 |  1.70462 | 0.189402 | 0.469211 | 0.591742 | 0.107942 | 0.200147 | 0.200147 | 2.122062 | 0.308089 | 2.430152 | 10.125632 | 0.0 | 183.715656 | 54.202222 | 10.125632 |
    | 15.02. | 10.6 | 0.2 | 0.0 | 10.4 |   4.94975 | 0.2 |      0.0 |      0.2 | 5.979038 | 5.381134 | 0.597904 | 0.528788 | 1.928852 | 0.132535 | 0.183864 | 0.183864 |  2.12043 |   0.3164 |  2.43683 | 10.153457 | 0.0 | 188.136618 | 54.194509 | 10.153457 |
    | 16.02. |  8.8 | 0.9 | 0.0 |  7.9 |  3.592434 | 0.9 |      0.0 |      0.9 | 4.880438 | 4.392395 | 0.488044 | 0.572872 | 4.439006 | 0.317665 | 0.183773 | 0.183773 | 2.590836 | 0.501438 | 3.092274 | 12.884474 | 0.0 |  191.15618 | 56.226451 | 12.884474 |
    | 17.02. |  7.2 | 0.9 | 0.0 |  6.3 |  2.770092 | 0.9 |      0.0 |      0.9 | 4.136157 | 3.722541 | 0.413616 | 0.606248 | 4.516594 | 0.481497 |  0.20904 |  0.20904 | 3.047331 | 0.690537 | 3.737868 | 15.574451 | 0.0 | 193.320023 | 57.904754 | 15.574451 |
    | 18.02. |  4.9 | 0.5 | 0.0 |  4.4 |  1.890459 | 0.5 |      0.0 |      0.5 |    3.136 |   2.8224 |   0.3136 | 0.626459 | 3.745073 | 0.467026 | 0.231706 | 0.231706 | 3.262692 | 0.698732 | 3.961424 | 16.505933 | 0.0 | 194.584023 |  58.61884 | 16.505933 |
    | 19.02. |  1.8 | 0.9 | 0.0 |  0.9 |  0.384007 | 0.9 |      0.0 |      0.9 | 1.138596 | 1.024736 |  0.11386 | 0.622603 | 2.771502 | 0.373883 | 0.241862 | 0.241862 | 3.203911 | 0.615745 | 3.819656 | 15.915234 | 0.0 | 194.345427 | 58.428293 | 15.915234 |

There is no indication of an error in the water balance:

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _gland_gr4_groundwater_loss:

groundwater loss
________________

In the :ref:`gland_gr4_base_example`, the groundwater exchange coefficient |X2| is
positive, which results in a groundwater gain.  Here, we demonstrate the opposite case
of a net groundwater loss and start with empty production and routing stores:

>>> x2(-1.012)
>>> test.inits.s = 0.0
>>> test.inits.r = 0.0

.. integration-test::

    >>> conditions = test("gland_gr4_groundwater_loss", get_conditions="1990-01-01")
    |   date |    p |   e |  en |   pn |        ps |  ei |       es |       ae |       pr |      pr9 |      pr1 |     perc |       q9 |       q1 |        fd |        fr |       qr |       qd |       qh |       qv |   i |          s |         r |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |  0.0 | 0.3 | 0.3 |  0.0 |       0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |       0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 |        0.0 |       0.0 |      0.0 |
    | 02.01. |  9.3 | 0.4 | 0.0 |  8.9 |   8.89645 | 0.4 |      0.0 |      0.4 |  0.00355 | 0.003195 | 0.000355 |      0.0 | 0.000441 | 0.000024 |       0.0 |       0.0 |      0.0 | 0.000024 | 0.000024 | 0.000102 | 0.0 |    8.89645 |  0.000441 | 0.000102 |
    | 03.01. |  3.2 | 0.4 | 0.0 |  2.8 |  2.795488 | 0.4 |      0.0 |      0.4 | 0.004512 | 0.004061 | 0.000451 |      0.0 | 0.002614 | 0.000145 |       0.0 |       0.0 |      0.0 | 0.000145 | 0.000145 | 0.000605 | 0.0 |  11.691938 |  0.003055 | 0.000605 |
    | 04.01. |  7.3 | 0.3 | 0.0 |  7.0 |   6.97519 | 0.3 |      0.0 |      0.3 | 0.024815 | 0.022333 | 0.002481 | 0.000005 | 0.006394 | 0.000474 |       0.0 |       0.0 |      0.0 | 0.000474 | 0.000474 | 0.001976 | 0.0 |  18.667123 |  0.009449 | 0.001976 |
    | 05.01. |  0.0 | 0.1 | 0.1 |  0.0 |       0.0 | 0.0 | 0.013982 | 0.013982 | 0.000005 | 0.000005 | 0.000001 | 0.000005 | 0.015247 | 0.001054 |       0.0 |       0.0 |      0.0 | 0.001054 | 0.001054 | 0.004392 | 0.0 |  18.653136 |  0.024696 | 0.004392 |
    | 06.01. |  0.0 | 0.1 | 0.1 |  0.0 |       0.0 | 0.0 | 0.013972 | 0.013972 | 0.000005 | 0.000005 | 0.000001 | 0.000005 | 0.004898 | 0.001178 |       0.0 |       0.0 |      0.0 | 0.001178 | 0.001178 | 0.004907 | 0.0 |  18.639159 |  0.029593 | 0.004907 |
    | 07.01. |  0.0 | 0.1 | 0.1 |  0.0 |       0.0 | 0.0 | 0.013962 | 0.013962 | 0.000005 | 0.000004 |      0.0 | 0.000005 | 0.000005 | 0.000393 |       0.0 |       0.0 |      0.0 | 0.000393 | 0.000393 | 0.001639 | 0.0 |  18.625193 |  0.029598 | 0.001639 |
    | 08.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.027893 | 0.027893 | 0.000005 | 0.000004 |      0.0 | 0.000005 | 0.000004 |  0.00002 |       0.0 |       0.0 |      0.0 |  0.00002 |  0.00002 | 0.000082 | 0.0 |  18.597295 |  0.029602 | 0.000082 |
    | 09.01. |  0.1 | 0.2 | 0.1 |  0.0 |       0.0 | 0.1 | 0.013932 | 0.113932 | 0.000005 | 0.000004 |      0.0 | 0.000005 | 0.000004 |      0.0 |       0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.000002 | 0.0 |  18.583358 |  0.029607 | 0.000002 |
    | 10.01. |  0.2 | 0.3 | 0.1 |  0.0 |       0.0 | 0.2 | 0.013921 | 0.213921 | 0.000005 | 0.000004 |      0.0 | 0.000005 | 0.000004 |      0.0 |       0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.000002 | 0.0 |  18.569432 |  0.029611 | 0.000002 |
    | 11.01. |  2.9 | 0.3 | 0.0 |  2.6 |  2.584477 | 0.3 |      0.0 |      0.3 | 0.015532 | 0.013979 | 0.001553 | 0.000009 | 0.001933 | 0.000108 |       0.0 |       0.0 |      0.0 | 0.000108 | 0.000108 | 0.000449 | 0.0 |    21.1539 |  0.031545 | 0.000449 |
    | 12.01. |  0.2 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.000009 | 0.000008 | 0.000001 | 0.000009 | 0.008988 |   0.0005 |       0.0 |       0.0 |      0.0 |   0.0005 |   0.0005 | 0.002082 | 0.0 |   21.15389 |  0.040533 | 0.002082 |
    | 13.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.031519 | 0.031519 | 0.000009 | 0.000008 | 0.000001 | 0.000009 |  0.00307 | 0.000691 |       0.0 |       0.0 |      0.0 | 0.000691 | 0.000691 | 0.002881 | 0.0 |  21.122362 |  0.043603 | 0.002881 |
    | 14.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.031474 | 0.031474 | 0.000009 | 0.000008 | 0.000001 | 0.000009 | 0.000008 | 0.000245 |       0.0 |       0.0 |      0.0 | 0.000245 | 0.000245 | 0.001019 | 0.0 |  21.090879 |  0.043611 | 0.001019 |
    | 15.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.031429 | 0.031429 | 0.000009 | 0.000008 | 0.000001 | 0.000009 | 0.000008 | 0.000013 |       0.0 |       0.0 |      0.0 | 0.000013 | 0.000013 | 0.000054 | 0.0 |  21.059441 |  0.043619 | 0.000054 |
    | 16.01. |  3.3 | 0.3 | 0.0 |  3.0 |  2.976916 | 0.3 |      0.0 |      0.3 | 0.023102 | 0.020792 |  0.00231 | 0.000018 | 0.002877 |  0.00016 |       0.0 |       0.0 |      0.0 |  0.00016 |  0.00016 | 0.000668 | 0.0 |  24.036339 |  0.046497 | 0.000668 |
    | 17.01. |  4.6 | 0.3 | 0.0 |  4.3 |  4.255413 | 0.3 |      0.0 |      0.3 | 0.044627 | 0.040164 | 0.004463 |  0.00004 | 0.018912 | 0.001051 |       0.0 |       0.0 |      0.0 | 0.001051 | 0.001051 |  0.00438 | 0.0 |  28.291712 |  0.065408 |  0.00438 |
    | 18.01. |  0.8 | 0.2 | 0.0 |  0.6 |  0.592589 | 0.2 |      0.0 |      0.2 | 0.007456 |  0.00671 | 0.000746 | 0.000045 | 0.031301 | 0.002514 |       0.0 |       0.0 |      0.0 | 0.002514 | 0.002514 | 0.010474 | 0.0 |  28.884256 |  0.096709 | 0.010474 |
    | 19.01. |  1.8 | 0.2 | 0.0 |  1.6 |  1.578704 | 0.2 |      0.0 |      0.2 | 0.021354 | 0.019219 | 0.002135 | 0.000058 | 0.015768 | 0.002735 |       0.0 |       0.0 |      0.0 | 0.002735 | 0.002735 | 0.011396 | 0.0 |  30.462902 |  0.112477 | 0.011396 |
    | 20.01. |  1.1 | 0.3 | 0.0 |  0.8 |  0.788488 | 0.3 |      0.0 |      0.3 | 0.011579 | 0.010421 | 0.001158 | 0.000066 | 0.015263 | 0.001816 |       0.0 |       0.0 |      0.0 | 0.001816 | 0.001816 | 0.007568 | 0.0 |  31.251323 |   0.12774 | 0.007568 |
    | 21.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.045612 | 0.045612 | 0.000066 | 0.000059 | 0.000007 | 0.000066 | 0.010918 | 0.001474 |       0.0 |       0.0 |      0.0 | 0.001474 | 0.001474 | 0.006142 | 0.0 |  31.205645 |  0.138659 | 0.006142 |
    | 22.01. |  5.0 | 0.2 | 0.0 |  4.8 |  4.718135 | 0.2 |      0.0 |      0.2 | 0.081999 | 0.073799 |   0.0082 | 0.000133 | 0.012509 | 0.001424 |       0.0 |       0.0 |      0.0 | 0.001424 | 0.001424 | 0.005934 | 0.0 |  35.923646 |  0.151168 | 0.005934 |
    | 23.01. | 13.1 | 0.3 | 0.0 | 12.8 | 12.453552 | 0.3 |      0.0 |      0.3 | 0.347038 | 0.312335 | 0.034704 |  0.00059 | 0.090567 | 0.005232 |       0.0 |       0.0 |      0.0 | 0.005232 | 0.005232 |   0.0218 | 0.0 |  48.376608 |  0.241735 |   0.0218 |
    | 24.01. | 14.6 | 0.6 | 0.0 | 14.0 | 13.354986 | 0.6 |      0.0 |      0.6 | 0.647011 |  0.58231 | 0.064701 | 0.001997 | 0.297331 | 0.019278 |       0.0 |       0.0 |      0.0 | 0.019278 | 0.019278 | 0.080325 | 0.0 |  61.729597 |  0.539066 | 0.080325 |
    | 25.01. |  4.0 | 0.4 | 0.0 |  3.6 |  3.381115 | 0.4 |      0.0 |      0.4 | 0.221492 | 0.199343 | 0.022149 | 0.002607 | 0.470285 | 0.039051 |       0.0 |       0.0 |      0.0 | 0.039051 | 0.039051 | 0.162713 | 0.0 |  65.108105 |  1.009351 | 0.162713 |
    | 26.01. |  0.8 | 0.4 | 0.0 |  0.4 |  0.374228 | 0.4 |      0.0 |      0.4 | 0.028454 | 0.025609 | 0.002845 | 0.002682 | 0.259281 | 0.041611 |       0.0 |       0.0 |      0.0 | 0.041611 | 0.041611 | 0.173379 | 0.0 |  65.479651 |  1.268632 | 0.173379 |
    | 27.01. |  0.1 | 0.4 | 0.3 |  0.0 |       0.0 | 0.1 | 0.133175 | 0.233175 | 0.002654 | 0.002389 | 0.000265 | 0.002654 | 0.060474 |  0.02121 |       0.0 |       0.0 |      0.0 |  0.02121 |  0.02121 | 0.088375 | 0.0 |  65.343822 |  1.329106 | 0.088375 |
    | 28.01. |  3.3 | 0.5 | 0.0 |  2.8 |     2.612 | 0.5 |      0.0 |      0.5 | 0.191228 | 0.172105 | 0.019123 | 0.003228 | 0.030905 | 0.006646 |       0.0 |       0.0 |      0.0 | 0.006646 | 0.006646 | 0.027691 | 0.0 |  67.952594 |   1.36001 | 0.027691 |
    | 29.01. |  7.7 | 0.4 | 0.0 |  7.3 |  6.738271 | 0.4 |      0.0 |      0.4 | 0.566907 | 0.510216 | 0.056691 | 0.005178 | 0.181587 | 0.010794 |       0.0 |       0.0 |      0.0 | 0.010794 | 0.010794 | 0.044975 | 0.0 |  74.685687 |  1.541597 | 0.044975 |
    | 30.01. | 10.3 | 0.3 | 0.0 | 10.0 |  9.050338 | 0.3 |      0.0 |      0.3 | 0.958831 | 0.862948 | 0.095883 | 0.009169 | 0.484816 |  0.03341 | -0.000001 | -0.000001 |      0.0 | 0.033409 |  0.03341 | 0.139206 | 0.0 |  83.726856 |  2.026411 | 0.139206 |
    | 31.01. |  3.7 | 0.3 | 0.0 |  3.4 |  3.026608 | 0.3 |      0.0 |      0.3 | 0.384336 | 0.345902 | 0.038434 | 0.010944 |  0.71428 | 0.061696 | -0.000002 | -0.000002 | 0.000001 | 0.061694 | 0.061695 | 0.257061 | 0.0 |   86.74252 |  2.740689 | 0.257061 |
    | 01.02. | 15.3 | 0.5 | 0.0 | 14.8 |  12.85356 | 0.5 |      0.0 |      0.5 |  1.96826 | 1.771434 | 0.196826 | 0.021819 | 0.655983 | 0.077642 | -0.000005 | -0.000005 | 0.000002 | 0.077636 | 0.077638 | 0.323492 | 0.0 |   99.57426 |  3.396665 | 0.323492 |
    | 02.02. |  3.2 | 0.5 | 0.0 |  2.7 |  2.286063 | 0.5 |      0.0 |      0.5 |  0.43835 | 0.394515 | 0.043835 | 0.024414 | 1.268985 | 0.098875 | -0.000011 | -0.000011 | 0.000009 | 0.098863 | 0.098872 | 0.411968 | 0.0 |  101.83591 |  4.665629 | 0.411968 |
    | 03.02. |  2.7 | 0.3 | 0.0 |  2.4 |   2.01636 | 0.3 |      0.0 |      0.3 | 0.410535 | 0.369482 | 0.041054 | 0.026895 | 0.692787 |  0.11126 | -0.000034 | -0.000034 | 0.000018 | 0.111226 | 0.111244 | 0.463517 | 0.0 | 103.825375 |  5.358364 | 0.463517 |
    | 04.02. |  2.2 | 0.3 | 0.0 |  1.9 |  1.585723 | 0.3 |      0.0 |      0.3 | 0.343251 | 0.308926 | 0.034325 | 0.028974 | 0.366608 | 0.066258 | -0.000056 | -0.000056 | 0.000025 | 0.066203 | 0.066228 | 0.275949 | 0.0 | 105.382124 |  5.724891 | 0.275949 |
    | 05.02. |  8.0 | 0.4 | 0.0 |  7.6 |   6.24708 | 0.4 |      0.0 |      0.4 | 1.391501 | 1.252351 |  0.13915 | 0.038582 | 0.452425 | 0.047298 |  -0.00007 |  -0.00007 | 0.000037 | 0.047228 | 0.047265 | 0.196937 | 0.0 | 111.590623 |  6.177209 | 0.196937 |
    | 06.02. | 14.3 | 0.4 | 0.0 | 13.9 | 11.015301 | 0.4 |      0.0 |      0.4 | 2.946341 | 2.651707 | 0.294634 | 0.061643 | 1.238782 | 0.087113 | -0.000092 | -0.000092 | 0.000093 | 0.087021 | 0.087114 | 0.362974 | 0.0 | 122.544282 |  7.415806 | 0.362974 |
    | 07.02. |  6.3 | 0.3 | 0.0 |  6.0 |   4.58655 | 0.3 |      0.0 |      0.3 | 1.487325 | 1.338592 | 0.148732 | 0.073874 | 2.163801 | 0.172576 | -0.000174 | -0.000174 | 0.000333 | 0.172402 | 0.172734 | 0.719727 | 0.0 | 127.056957 |  9.579101 | 0.719727 |
    | 08.02. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 |  0.14872 |  0.14872 |  0.07323 | 0.065907 | 0.007323 |  0.07323 | 1.450658 | 0.201497 | -0.000427 | -0.000427 | 0.000673 |  0.20107 | 0.201744 | 0.840598 | 0.0 | 126.835007 | 11.028659 | 0.840598 |
    | 09.02. |  5.9 | 0.1 | 0.0 |  5.8 |   4.34095 | 0.1 |      0.0 |      0.1 | 1.545432 | 1.390889 | 0.154543 | 0.086382 | 0.527693 | 0.126511 | -0.000699 | -0.000699 |  0.00085 | 0.125813 | 0.126662 | 0.527759 | 0.0 | 131.089575 | 11.554804 | 0.527759 |
    | 10.02. |  9.2 | 0.1 | 0.0 |  9.1 |  6.614754 | 0.1 |      0.0 |      0.1 | 2.595333 |   2.3358 | 0.259533 | 0.110088 | 1.230977 |  0.09646 | -0.000822 | -0.000822 | 0.001408 | 0.095637 | 0.097046 | 0.404358 | 0.0 | 137.594241 |  12.78355 | 0.404358 |
    | 11.02. |  6.1 | 0.0 | 0.0 |  6.1 |    4.2994 | 0.0 |      0.0 |      0.0 | 1.928451 | 1.735606 | 0.192845 | 0.127851 | 2.045889 | 0.167778 | -0.001171 | -0.001171 | 0.002955 | 0.166606 | 0.169562 | 0.706508 | 0.0 |  141.76579 | 14.825313 | 0.706508 |
    | 12.02. |  0.1 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.127277 |  0.11455 | 0.012728 | 0.127277 | 1.643359 | 0.202641 | -0.001968 | -0.001968 |  0.00499 | 0.200673 | 0.205663 | 0.856929 | 0.0 | 141.638513 | 16.461714 | 0.856929 |
    | 13.02. |  0.0 | 0.1 | 0.1 |  0.0 |       0.0 | 0.0 | 0.079791 | 0.079791 | 0.126352 | 0.113717 | 0.012635 | 0.126352 | 0.469661 | 0.132688 | -0.002839 | -0.002839 | 0.005729 | 0.129849 | 0.135579 | 0.564911 | 0.0 |  141.43237 | 16.922807 | 0.564911 |
    | 14.02. |  2.8 | 0.0 | 0.0 |  2.8 |  1.941883 | 0.0 |      0.0 |      0.0 | 0.992767 |  0.89349 | 0.099277 |  0.13465 | 0.221539 | 0.048846 | -0.003127 | -0.003127 | 0.006098 |  0.04572 | 0.051818 | 0.215907 | 0.0 | 143.239603 | 17.135121 | 0.215907 |
    | 15.02. | 10.6 | 0.2 | 0.0 | 10.4 |  7.013585 | 0.2 |      0.0 |      0.2 | 3.556536 | 3.200882 | 0.355654 | 0.170121 | 1.041127 | 0.065561 | -0.003266 | -0.003266 | 0.008166 | 0.062295 | 0.070461 | 0.293587 | 0.0 | 150.083067 | 18.164815 | 0.293587 |
    | 16.02. |  8.8 | 0.9 | 0.0 |  7.9 |  5.117514 | 0.9 |      0.0 |      0.9 | 2.982443 | 2.684199 | 0.298244 | 0.199957 | 2.623935 | 0.181145 | -0.004006 | -0.004006 | 0.015969 | 0.177139 | 0.193108 | 0.804615 | 0.0 | 155.000624 | 20.768775 | 0.804615 |
    | 17.02. |  7.2 | 0.9 | 0.0 |  6.3 |  3.953493 | 0.9 |      0.0 |      0.9 | 2.571769 | 2.314592 | 0.257177 | 0.225262 | 2.746401 | 0.287518 | -0.006403 | -0.006403 | 0.029523 | 0.281115 | 0.310638 | 1.294326 | 0.0 | 158.728855 |  23.47925 | 1.294326 |
    | 18.02. |  4.9 | 0.5 | 0.0 |  4.4 |  2.695976 | 0.5 |      0.0 |      0.5 | 1.947291 | 1.752562 | 0.194729 | 0.243267 | 2.318003 | 0.285379 | -0.009836 | -0.009836 | 0.046821 | 0.275543 | 0.322364 | 1.343182 | 0.0 | 161.181564 | 25.740595 | 1.343182 |
    | 19.02. |  1.8 | 0.9 | 0.0 |  0.9 |  0.545454 | 0.9 |      0.0 |      0.9 | 0.600092 | 0.540083 | 0.060009 | 0.245546 | 1.708351 | 0.230702 | -0.013571 | -0.013571 | 0.063739 | 0.217131 |  0.28087 | 1.170292 | 0.0 | 161.481472 | 27.371637 | 1.170292 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _gland_gr4_interception:

interception
____________

This integration test repeats the :ref:`gland_gr4_base_example` with a non-zero
interception capacity:

>>> imax(10.0)
>>> x2(1.012)
>>> test.inits.s = 0.3 * x1
>>> test.inits.r = 0.5 * x3

.. integration-test::

    >>> conditions = test("gland_gr4_interception", get_conditions="1990-01-01")
    |   date |    p |   e |  en |   pn |        ps |  ei |       es |       ae |       pr |      pr9 |      pr1 |     perc |       q9 |       q1 |       fd |       fr |       qr |       qd |       qh |        qv |    i |          s |         r |    outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |  0.0 | 0.3 | 0.3 |  0.0 |       0.0 | 0.0 | 0.152875 | 0.152875 | 0.006036 | 0.005433 | 0.000604 | 0.006036 |  0.00075 | 0.000042 | 0.089449 | 0.089449 | 0.670218 | 0.089491 | 0.759709 |  3.165453 |  0.0 |  77.012489 | 43.537481 |  3.165453 |
    | 02.01. |  9.3 | 0.4 | 0.0 |  0.0 |       0.0 | 0.4 |      0.0 |      0.4 | 0.006034 |  0.00543 | 0.000603 | 0.006034 | 0.004242 | 0.000236 |   0.0854 |   0.0854 | 0.628558 | 0.085636 | 0.714194 |  2.975809 |  8.9 |  77.006455 | 42.998565 |  2.975809 |
    | 03.01. |  3.2 | 0.4 | 0.0 |  1.7 |  1.544575 | 0.4 |      0.0 |      0.4 | 0.162086 | 0.145877 | 0.016209 | 0.006661 | 0.024818 | 0.001581 | 0.081757 | 0.081757 | 0.592835 | 0.083338 | 0.676173 |  2.817388 | 10.0 |  78.544369 | 42.512305 |  2.817388 |
    | 04.01. |  7.3 | 0.3 | 0.0 |  7.0 |  6.293538 | 0.3 |      0.0 |      0.3 |  0.71625 | 0.644625 | 0.071625 | 0.009788 | 0.183947 | 0.010516 | 0.078567 | 0.078567 | 0.571052 | 0.089083 | 0.660135 |  2.750564 | 10.0 |  84.828119 | 42.203767 |  2.750564 |
    | 05.01. |  0.0 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.009782 | 0.008804 | 0.000978 | 0.009782 | 0.447565 | 0.030399 | 0.076589 | 0.076589 | 0.568011 | 0.106988 | 0.674999 |  2.812495 |  9.9 |  84.818337 |  42.15991 |  2.812495 |
    | 06.01. |  0.0 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.009777 | 0.008799 | 0.000978 | 0.009777 | 0.148132 | 0.034796 | 0.076311 | 0.076311 | 0.546108 | 0.111107 | 0.657215 |  2.738395 |  9.8 |   84.80856 | 41.838246 |  2.738395 |
    | 07.01. |  0.0 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.009771 | 0.008794 | 0.000977 | 0.009771 | 0.008799 | 0.012185 | 0.074292 | 0.074292 | 0.517631 | 0.086477 | 0.604108 |  2.517119 |  9.7 |  84.798789 | 41.403706 |  2.517119 |
    | 08.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.009765 | 0.008789 | 0.000977 | 0.009765 | 0.008794 | 0.001522 | 0.071627 | 0.071627 | 0.491812 | 0.073148 |  0.56496 |  2.354001 |  9.5 |  84.789024 | 40.992316 |  2.354001 |
    | 09.01. |  0.1 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 |  0.00976 | 0.008784 | 0.000976 |  0.00976 | 0.008789 | 0.000977 | 0.069167 | 0.069167 | 0.468304 | 0.070143 | 0.538447 |  2.243529 |  9.4 |  84.779264 | 40.601968 |  2.243529 |
    | 10.01. |  0.2 | 0.3 | 0.0 |  0.0 |       0.0 | 0.3 |      0.0 |      0.3 | 0.009754 | 0.008779 | 0.000975 | 0.009754 | 0.008784 | 0.000976 | 0.066889 | 0.066889 | 0.446817 | 0.067865 | 0.514682 |   2.14451 |  9.3 |  84.769509 | 40.230824 |   2.14451 |
    | 11.01. |  2.9 | 0.3 | 0.0 |  1.9 |  1.689527 | 0.3 |      0.0 |      0.3 | 0.221233 | 0.199109 | 0.022123 | 0.010759 | 0.035053 | 0.002435 | 0.064773 | 0.064773 | 0.428474 | 0.067208 | 0.495683 |  2.065345 | 10.0 |  86.448277 | 39.902175 |  2.065345 |
    | 12.01. |  0.2 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.010753 | 0.009677 | 0.001075 | 0.010753 | 0.131253 |  0.00778 |  0.06294 |  0.06294 | 0.416425 | 0.070719 | 0.487145 |  2.029769 | 10.0 |  86.437524 | 39.679942 |  2.029769 |
    | 13.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.010746 | 0.009671 | 0.001075 | 0.010746 | 0.051187 | 0.010423 | 0.061721 | 0.061721 |  0.40121 | 0.072144 | 0.473354 |  1.972308 |  9.8 |  86.426778 | 39.391641 |  1.972308 |
    | 14.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.010739 | 0.009665 | 0.001074 | 0.010739 | 0.009672 | 0.004378 | 0.060166 | 0.060166 | 0.385097 | 0.064544 | 0.449641 |  1.873503 |  9.6 |  86.416039 | 39.076383 |  1.873503 |
    | 15.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.010733 | 0.009659 | 0.001073 | 0.010733 | 0.009666 | 0.001237 | 0.058498 | 0.058498 | 0.370162 | 0.059734 | 0.429897 |  1.791236 |  9.4 |  86.405306 | 38.774384 |  1.791236 |
    | 16.01. |  3.3 | 0.3 | 0.0 |  2.4 |  2.122503 | 0.3 |      0.0 |      0.3 | 0.289606 | 0.260645 | 0.028961 | 0.012109 | 0.044307 | 0.002999 |  0.05693 |  0.05693 | 0.357849 | 0.059929 | 0.417778 |  1.740741 | 10.0 |    88.5157 | 38.517772 |  1.740741 |
    | 17.01. |  4.6 | 0.3 | 0.0 |  4.3 |  3.768829 | 0.3 |      0.0 |      0.3 | 0.546076 | 0.491469 | 0.054608 | 0.014905 | 0.237509 | 0.013732 | 0.055623 | 0.055623 | 0.354934 | 0.069354 | 0.424288 |  1.767867 | 10.0 |  92.269624 |  38.45597 |  1.767867 |
    | 18.01. |  0.8 | 0.2 | 0.0 |  0.6 |  0.522365 | 0.2 |      0.0 |      0.2 | 0.092954 | 0.083659 | 0.009295 | 0.015319 | 0.384594 | 0.031254 | 0.055311 | 0.055311 | 0.358765 | 0.086565 |  0.44533 |  1.855541 | 10.0 |   92.77667 |  38.53711 |  1.855541 |
    | 19.01. |  1.8 | 0.2 | 0.0 |  1.6 |   1.38874 | 0.2 |      0.0 |      0.2 | 0.227747 | 0.204972 | 0.022775 | 0.016487 | 0.189769 | 0.033408 |  0.05572 |  0.05572 | 0.353665 | 0.089129 | 0.442793 |  1.844973 | 10.0 |  94.148923 | 38.428934 |  1.844973 |
    | 20.01. |  1.1 | 0.3 | 0.0 |  0.8 |  0.692046 | 0.3 |      0.0 |      0.3 | 0.125041 | 0.112537 | 0.012504 | 0.017087 | 0.165629 | 0.021114 | 0.055175 | 0.055175 | 0.347755 | 0.076288 | 0.424044 |  1.766849 | 10.0 |  94.823883 | 38.301983 |  1.766849 |
    | 21.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.017071 | 0.015364 | 0.001707 | 0.017071 | 0.119379 | 0.016149 |  0.05454 |  0.05454 |  0.34014 | 0.070688 | 0.410829 |  1.711786 |  9.8 |  94.806811 | 38.135761 |  1.711786 |
    | 22.01. |  5.0 | 0.2 | 0.0 |  4.6 |  3.948718 | 0.2 |      0.0 |      0.2 | 0.672197 | 0.604977 |  0.06722 | 0.020914 | 0.118047 | 0.014398 | 0.053716 | 0.053716 | 0.332889 | 0.068113 | 0.401003 |  1.670845 | 10.0 |  98.734615 | 37.974634 |  1.670845 |
    | 23.01. | 13.1 | 0.3 | 0.0 | 12.8 | 10.701065 | 0.3 |      0.0 |      0.3 | 2.133875 | 1.920487 | 0.213387 |  0.03494 | 0.657366 | 0.039231 | 0.052926 | 0.052926 | 0.349313 | 0.092156 | 0.441469 |  1.839456 | 10.0 |  109.40074 | 38.335613 |  1.839456 |
    | 24.01. | 14.6 | 0.6 | 0.0 | 14.0 | 11.197565 | 0.6 |      0.0 |      0.6 | 2.859198 | 2.573278 |  0.28592 | 0.056763 | 1.722327 |  0.11859 | 0.054707 | 0.054707 | 0.417254 | 0.173297 | 0.590551 |  2.460628 | 10.0 | 120.541541 | 39.695393 |  2.460628 |
    | 25.01. |  4.0 | 0.4 | 0.0 |  3.6 |  2.791007 | 0.4 |      0.0 |      0.4 | 0.872482 | 0.785233 | 0.087248 | 0.063489 |  2.18341 | 0.203419 | 0.061806 | 0.061806 | 0.518793 | 0.265225 | 0.784018 |  3.266741 | 10.0 |  123.26906 | 41.421816 |  3.266741 |
    | 26.01. |  0.8 | 0.4 | 0.0 |  0.4 |  0.307916 | 0.4 |      0.0 |      0.4 | 0.156204 | 0.140583 |  0.01562 |  0.06412 | 1.088065 | 0.190338 | 0.071736 | 0.071736 | 0.558602 | 0.262074 | 0.820676 |  3.419484 | 10.0 | 123.512856 | 42.023016 |  3.419484 |
    | 27.01. |  0.1 | 0.4 | 0.0 |  0.0 |       0.0 | 0.4 |      0.0 |      0.4 | 0.063954 | 0.057559 | 0.006395 | 0.063954 | 0.270386 | 0.090803 | 0.075447 | 0.075447 | 0.545135 |  0.16625 | 0.711385 |  2.964106 |  9.7 | 123.448902 | 41.823714 |  2.964106 |
    | 28.01. |  3.3 | 0.5 | 0.0 |  2.5 |  1.915243 | 0.5 |      0.0 |      0.5 | 0.653644 |  0.58828 | 0.065364 | 0.068887 | 0.149012 | 0.029415 | 0.074202 | 0.074202 | 0.525239 | 0.103617 | 0.628856 |  2.620231 | 10.0 | 125.295258 |  41.52169 |  2.620231 |
    | 29.01. |  7.7 | 0.4 | 0.0 |  7.3 |  5.490734 | 0.4 |      0.0 |      0.4 | 1.894374 | 1.704936 | 0.189437 | 0.085108 | 0.626124 | 0.040053 | 0.072344 | 0.072344 | 0.535874 | 0.112397 | 0.648271 |  2.701128 | 10.0 | 130.700885 | 41.684283 |  2.701128 |
    | 30.01. | 10.3 | 0.3 | 0.0 | 10.0 |  7.271132 | 0.3 |      0.0 |      0.3 | 2.840028 | 2.556025 | 0.284003 |  0.11116 | 1.577724 | 0.110694 |  0.07334 |  0.07334 | 0.608391 | 0.184034 | 0.792425 |   3.30177 | 10.0 | 137.860856 | 42.726956 |   3.30177 |
    | 31.01. |  3.7 | 0.3 | 0.0 |  3.4 |  2.406274 | 0.3 |      0.0 |      0.3 | 1.114428 | 1.002985 | 0.111443 | 0.120702 | 2.155144 | 0.193563 | 0.079964 | 0.079964 | 0.727486 | 0.273527 | 1.001013 |  4.170886 | 10.0 | 140.146428 | 44.234578 |  4.170886 |
    | 01.02. | 15.3 | 0.5 | 0.0 | 14.8 | 10.079977 | 0.5 |      0.0 |      0.5 | 4.889993 | 4.400993 | 0.488999 |  0.16997 | 1.812365 |  0.22616 | 0.090283 | 0.090283 | 0.824095 | 0.316442 | 1.140537 |  4.752239 | 10.0 | 150.056435 | 45.313131 |  4.752239 |
    | 02.02. |  3.2 | 0.5 | 0.0 |  2.7 |  1.770334 | 0.5 |      0.0 |      0.5 | 1.108862 | 0.997976 | 0.110886 | 0.179196 |  3.18663 | 0.260444 | 0.098225 | 0.098225 | 1.057891 | 0.358669 |  1.41656 |  5.902333 | 10.0 | 151.647573 | 47.540095 |  5.902333 |
    | 03.02. |  2.7 | 0.3 | 0.0 |  2.4 |  1.557301 | 0.3 |      0.0 |      0.3 | 1.030156 | 0.927141 | 0.103016 | 0.187457 |  1.73391 | 0.279983 | 0.116184 | 0.116184 | 1.142909 | 0.396167 | 1.539076 |  6.412816 | 10.0 | 153.017417 |  48.24728 |  6.412816 |
    | 04.02. |  2.2 | 0.3 | 0.0 |  1.9 |  1.222304 | 0.3 |      0.0 |      0.3 | 0.871554 | 0.784399 | 0.087155 | 0.193858 | 0.922959 | 0.166076 | 0.122347 | 0.122347 | 1.132165 | 0.288423 | 1.420588 |  5.919115 | 10.0 | 154.045863 | 48.160421 |  5.919115 |
    | 05.02. |  8.0 | 0.4 | 0.0 |  7.6 |  4.788405 | 0.4 |      0.0 |      0.4 | 3.036012 | 2.732411 | 0.303601 | 0.224417 |  1.08458 | 0.115971 | 0.121578 | 0.121578 | 1.140303 | 0.237549 | 1.377851 |  5.741048 | 10.0 | 158.609851 | 48.226276 |  5.741048 |
    | 06.02. | 14.3 | 0.4 | 0.0 | 13.9 |  8.329851 | 0.4 |      0.0 |      0.4 | 5.857753 | 5.271977 | 0.585775 | 0.287604 | 2.656098 | 0.193811 |  0.12216 |  0.12216 | 1.331985 | 0.315972 | 1.647957 |  6.866487 | 10.0 | 166.652099 | 49.672549 |  6.866487 |
    | 07.02. |  6.3 | 0.3 | 0.0 |  6.0 |  3.429287 | 0.3 |      0.0 |      0.3 | 2.886312 | 2.597681 | 0.288631 | 0.315599 | 4.346318 | 0.357741 | 0.135471 | 0.135471 | 1.766963 | 0.493212 | 2.260175 |  9.417397 | 10.0 | 169.765786 | 52.387374 |  9.417397 |
    | 08.02. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.312693 | 0.281423 | 0.031269 | 0.312693 | 2.863972 | 0.403862 | 0.163204 | 0.163204 |   1.9674 | 0.567066 | 2.534467 | 10.560278 |  9.8 | 169.453093 |  53.44715 | 10.560278 |
    | 09.02. |  5.9 | 0.1 | 0.0 |  5.6 |  3.124638 | 0.1 |      0.0 |      0.1 | 2.814717 | 2.533246 | 0.281472 | 0.339356 | 1.099831 | 0.252176 | 0.175055 | 0.175055 | 1.855305 | 0.427231 | 2.282535 |  9.510565 | 10.0 | 172.238376 |  52.86673 |  9.510565 |
    | 10.02. |  9.2 | 0.1 | 0.0 |  9.1 |  4.902115 | 0.1 |      0.0 |      0.1 | 4.584328 | 4.125895 | 0.458433 | 0.386444 | 2.259646 | 0.185841 | 0.168491 | 0.168491 | 1.947683 | 0.354332 | 2.302015 |  9.591728 | 10.0 | 176.754048 | 53.347184 |  9.591728 |
    | 11.02. |  6.1 | 0.0 | 0.0 |  6.1 |  3.167752 | 0.0 |      0.0 |      0.0 | 3.349851 | 3.014866 | 0.334985 | 0.417603 | 3.623528 | 0.302823 | 0.173911 | 0.173911 | 2.268922 | 0.476735 | 2.745657 | 11.440238 | 10.0 | 179.504197 | 54.875701 | 11.440238 |
    | 12.02. |  0.1 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.412802 | 0.371521 |  0.04128 | 0.412802 | 2.893444 | 0.358895 | 0.191985 | 0.191985 | 2.422239 |  0.55088 | 2.973119 | 12.387996 | 10.0 | 179.091395 | 55.538891 | 12.387996 |
    | 13.02. |  0.0 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.408098 | 0.367288 |  0.04081 | 0.408098 |  0.95018 | 0.239246 | 0.200229 | 0.200229 | 2.186531 | 0.439475 | 2.626007 | 10.941695 |  9.9 | 178.683297 | 54.502769 | 10.941695 |
    | 14.02. |  2.8 | 0.0 | 0.0 |  2.7 |  1.387085 | 0.0 |      0.0 |      0.0 | 1.732237 | 1.559014 | 0.173224 | 0.419322 | 0.532721 | 0.099553 | 0.187457 | 0.187457 | 1.935904 |  0.28701 | 2.222914 |  9.262143 | 10.0 | 179.651059 | 53.287043 |  9.262143 |
    | 15.02. | 10.6 | 0.2 | 0.0 | 10.4 |  5.178456 | 0.2 |      0.0 |      0.2 | 5.698981 | 5.129083 | 0.569898 | 0.477437 | 1.790677 | 0.122227 | 0.173226 | 0.173226 | 1.940484 | 0.295453 | 2.235937 |  9.316402 | 10.0 | 184.352079 | 53.310462 |  9.316402 |
    | 16.02. |  8.8 | 0.9 | 0.0 |  7.9 |  3.758646 | 0.9 |      0.0 |      0.9 | 4.662457 | 4.196211 | 0.466246 | 0.521103 | 4.217991 | 0.299143 | 0.173493 | 0.173493 |  2.37279 | 0.472636 | 2.845425 | 11.855939 | 10.0 | 187.589622 | 55.329157 | 11.855939 |
    | 17.02. |  7.2 | 0.9 | 0.0 |  6.3 |  2.897342 | 0.9 |      0.0 |      0.9 | 3.957319 | 3.561587 | 0.395732 | 0.554661 | 4.313031 | 0.458219 | 0.197595 | 0.197595 | 2.802951 | 0.655814 | 3.458765 | 14.411521 | 10.0 | 189.932303 | 57.036832 | 14.411521 |
    | 18.02. |  4.9 | 0.5 | 0.0 |  4.4 |  1.976127 | 0.5 |      0.0 |      0.5 | 2.999414 | 2.699472 | 0.299941 | 0.575541 | 3.581648 | 0.446116 | 0.219777 | 0.219777 | 3.021659 | 0.665893 | 3.687552 | 15.364801 | 10.0 |  191.33289 | 57.816597 | 15.364801 |
    | 19.02. |  1.8 | 0.9 | 0.0 |  0.9 |  0.401044 | 0.9 |      0.0 |      0.9 | 1.071901 | 0.964711 |  0.10719 | 0.572945 | 2.648925 | 0.357439 | 0.230474 | 0.230474 | 2.989789 | 0.587913 | 3.577701 | 14.907089 | 10.0 | 191.160989 | 57.706206 | 14.907089 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _gland_gr4_no_rconc_submodels:

no rconc submodels
__________________

As mentioned above, the two runoff concentration ports are optional, meaning that
|gland_gr4| can work without the corresponding submodels.  In this case, it neglects
any related runoff concentration, resulting in an earlier outflow peak then in the last
example:

>>> model.rconcmodel_directflow = None
>>> model.rconcmodel_routingstore = None

.. integration-test::

    >>> conditions = test("gland_gr4_no_rconc_submodels", get_conditions="1990-01-01")
    |   date |    p |   e |  en |   pn |        ps |  ei |       es |       ae |       pr |      pr9 |      pr1 |     perc |       q9 |       q1 |       fd |       fr |       qr |       qd |       qh |        qv |    i |          s |         r |    outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |  0.0 | 0.3 | 0.3 |  0.0 |       0.0 | 0.0 | 0.152875 | 0.152875 | 0.006036 | 0.005433 | 0.000604 | 0.006036 | 0.005433 | 0.000604 | 0.089449 | 0.089449 | 0.670562 | 0.090053 | 0.760615 |   3.16923 |  0.0 |  77.012489 | 43.541819 |   3.16923 |
    | 02.01. |  9.3 | 0.4 | 0.0 |  0.0 |       0.0 | 0.4 |      0.0 |      0.4 | 0.006034 |  0.00543 | 0.000603 | 0.006034 |  0.00543 | 0.000603 |  0.08543 |  0.08543 | 0.628947 | 0.086033 | 0.714981 |  2.979086 |  8.9 |  77.006455 | 43.003732 |  2.979086 |
    | 03.01. |  3.2 | 0.4 | 0.0 |  1.7 |  1.544575 | 0.4 |      0.0 |      0.4 | 0.162086 | 0.145877 | 0.016209 | 0.006661 | 0.145877 | 0.016209 | 0.081792 | 0.081792 | 0.601328 |    0.098 | 0.699329 |  2.913869 | 10.0 |  78.544369 | 42.630073 |  2.913869 |
    | 04.01. |  7.3 | 0.3 | 0.0 |  7.0 |  6.293538 | 0.3 |      0.0 |      0.3 |  0.71625 | 0.644625 | 0.071625 | 0.009788 | 0.644625 | 0.071625 | 0.079331 | 0.079331 | 0.609667 | 0.150956 | 0.760623 |  3.169263 | 10.0 |  84.828119 | 42.744362 |  3.169263 |
    | 05.01. |  0.0 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.009782 | 0.008804 | 0.000978 | 0.009782 | 0.008804 | 0.000978 | 0.080078 | 0.080078 | 0.574859 | 0.081056 | 0.655916 |  2.732981 |  9.9 |  84.818337 | 42.258385 |  2.732981 |
    | 06.01. |  0.0 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.009777 | 0.008799 | 0.000978 | 0.009777 | 0.008799 | 0.000978 | 0.076937 | 0.076937 | 0.543586 | 0.077914 |   0.6215 |  2.589585 |  9.8 |   84.80856 | 41.800534 |  2.589585 |
    | 07.01. |  0.0 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.009771 | 0.008794 | 0.000977 | 0.009771 | 0.008794 | 0.000977 | 0.074058 | 0.074058 | 0.515349 | 0.075035 | 0.590385 |  2.459936 |  9.7 |  84.798789 | 41.368037 |  2.459936 |
    | 08.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.009765 | 0.008789 | 0.000977 | 0.009765 | 0.008789 | 0.000977 | 0.071411 | 0.071411 | 0.489737 | 0.072387 | 0.562125 |  2.342187 |  9.5 |  84.789024 | 40.958499 |  2.342187 |
    | 09.01. |  0.1 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 |  0.00976 | 0.008784 | 0.000976 |  0.00976 | 0.008784 | 0.000976 | 0.068967 | 0.068967 | 0.466411 | 0.069943 | 0.536354 |  2.234807 |  9.4 |  84.779264 |  40.56984 |  2.234807 |
    | 10.01. |  0.2 | 0.3 | 0.0 |  0.0 |       0.0 | 0.3 |      0.0 |      0.3 | 0.009754 | 0.008779 | 0.000975 | 0.009754 | 0.008779 | 0.000975 | 0.066704 | 0.066704 | 0.445084 | 0.067679 | 0.512763 |   2.13651 |  9.3 |  84.769509 | 40.200238 |   2.13651 |
    | 11.01. |  2.9 | 0.3 | 0.0 |  1.9 |  1.689527 | 0.3 |      0.0 |      0.3 | 0.221233 | 0.199109 | 0.022123 | 0.010759 | 0.199109 | 0.022123 | 0.064601 | 0.064601 |  0.43545 | 0.086724 | 0.522174 |  2.175726 | 10.0 |  86.448277 | 40.028498 |  2.175726 |
    | 12.01. |  0.2 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.010753 | 0.009677 | 0.001075 | 0.010753 | 0.009677 | 0.001075 |  0.06364 |  0.06364 | 0.416702 | 0.064715 | 0.481418 |  2.005907 | 10.0 |  86.437524 | 39.685113 |  2.005907 |
    | 13.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.010746 | 0.009671 | 0.001075 | 0.010746 | 0.009671 | 0.001075 |  0.06175 |  0.06175 | 0.399419 | 0.062824 | 0.462243 |  1.926011 |  9.8 |  86.426778 | 39.357116 |  1.926011 |
    | 14.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.010739 | 0.009665 | 0.001074 | 0.010739 | 0.009665 | 0.001074 | 0.059982 | 0.059982 | 0.383438 | 0.061056 | 0.444494 |  1.852057 |  9.6 |  86.416039 | 39.043325 |  1.852057 |
    | 15.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.010733 | 0.009659 | 0.001073 | 0.010733 | 0.009659 | 0.001073 | 0.058325 | 0.058325 | 0.368623 | 0.059398 | 0.428021 |  1.783419 |  9.4 |  86.405306 | 38.742686 |  1.783419 |
    | 16.01. |  3.3 | 0.3 | 0.0 |  2.4 |  2.122503 | 0.3 |      0.0 |      0.3 | 0.289606 | 0.260645 | 0.028961 | 0.012109 | 0.260645 | 0.028961 | 0.056768 | 0.056768 | 0.366261 | 0.085728 | 0.451989 |  1.883287 | 10.0 |    88.5157 | 38.693838 |  1.883287 |
    | 17.01. |  4.6 | 0.3 | 0.0 |  4.3 |  3.768829 | 0.3 |      0.0 |      0.3 | 0.546076 | 0.491469 | 0.054608 | 0.014905 | 0.491469 | 0.054608 | 0.056518 | 0.056518 | 0.374697 | 0.111125 | 0.485823 |  2.024261 | 10.0 |  92.269624 | 38.867127 |  2.024261 |
    | 18.01. |  0.8 | 0.2 | 0.0 |  0.6 |  0.522365 | 0.2 |      0.0 |      0.2 | 0.092954 | 0.083659 | 0.009295 | 0.015319 | 0.083659 | 0.009295 | 0.057408 | 0.057408 | 0.363878 | 0.066704 | 0.430582 |  1.794093 | 10.0 |   92.77667 | 38.644316 |  1.794093 |
    | 19.01. |  1.8 | 0.2 | 0.0 |  1.6 |   1.38874 | 0.2 |      0.0 |      0.2 | 0.227747 | 0.204972 | 0.022775 | 0.016487 | 0.204972 | 0.022775 | 0.056265 | 0.056265 | 0.359203 | 0.079039 | 0.438243 |  1.826011 | 10.0 |  94.148923 | 38.546349 |  1.826011 |
    | 20.01. |  1.1 | 0.3 | 0.0 |  0.8 |  0.692046 | 0.3 |      0.0 |      0.3 | 0.125041 | 0.112537 | 0.012504 | 0.017087 | 0.112537 | 0.012504 | 0.055767 | 0.055767 | 0.350633 | 0.068271 | 0.418904 |  1.745434 | 10.0 |  94.823883 |  38.36402 |  1.745434 |
    | 21.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.017071 | 0.015364 | 0.001707 | 0.017071 | 0.015364 | 0.001707 | 0.054849 | 0.054849 | 0.338335 | 0.056557 | 0.394891 |  1.645379 |  9.8 |  94.806811 | 38.095899 |  1.645379 |
    | 22.01. |  5.0 | 0.2 | 0.0 |  4.6 |  3.948718 | 0.2 |      0.0 |      0.2 | 0.672197 | 0.604977 |  0.06722 | 0.020914 | 0.604977 |  0.06722 | 0.053519 | 0.053519 | 0.352404 | 0.120739 | 0.473143 |  1.971429 | 10.0 |  98.734615 | 38.401992 |  1.971429 |
    | 23.01. | 13.1 | 0.3 | 0.0 | 12.8 | 10.701065 | 0.3 |      0.0 |      0.3 | 2.133875 | 1.920487 | 0.213387 |  0.03494 | 1.920487 | 0.213387 |  0.05504 |  0.05504 | 0.430917 | 0.268427 | 0.699344 |  2.913935 | 10.0 |  109.40074 | 39.946602 |  2.913935 |
    | 24.01. | 14.6 | 0.6 | 0.0 | 14.0 | 11.197565 | 0.6 |      0.0 |      0.6 | 2.859198 | 2.573278 |  0.28592 | 0.056763 | 2.573278 |  0.28592 | 0.063185 | 0.063185 | 0.558695 | 0.349105 |   0.9078 |  3.782499 | 10.0 | 120.541541 | 42.024371 |  3.782499 |
    | 25.01. |  4.0 | 0.4 | 0.0 |  3.6 |  2.791007 | 0.4 |      0.0 |      0.4 | 0.872482 | 0.785233 | 0.087248 | 0.063489 | 0.785233 | 0.087248 | 0.075456 | 0.075456 | 0.578252 | 0.162704 | 0.740956 |  3.087316 | 10.0 |  123.26906 | 42.306808 |  3.087316 |
    | 26.01. |  0.8 | 0.4 | 0.0 |  0.4 |  0.307916 | 0.4 |      0.0 |      0.4 | 0.156204 | 0.140583 |  0.01562 |  0.06412 | 0.140583 |  0.01562 | 0.077246 | 0.077246 | 0.554971 | 0.092866 | 0.647836 |  2.699318 | 10.0 | 123.512856 | 41.969666 |  2.699318 |
    | 27.01. |  0.1 | 0.4 | 0.0 |  0.0 |       0.0 | 0.4 |      0.0 |      0.4 | 0.063954 | 0.057559 | 0.006395 | 0.063954 | 0.057559 | 0.006395 | 0.075112 | 0.075112 | 0.528623 | 0.081508 |  0.61013 |   2.54221 |  9.7 | 123.448902 | 41.573715 |   2.54221 |
    | 28.01. |  3.3 | 0.5 | 0.0 |  2.5 |  1.915243 | 0.5 |      0.0 |      0.5 | 0.653644 |  0.58828 | 0.065364 | 0.068887 |  0.58828 | 0.065364 | 0.072661 | 0.072661 | 0.536772 | 0.138026 | 0.674797 |  2.811655 | 10.0 | 125.295258 | 41.697884 |  2.811655 |
    | 29.01. |  7.7 | 0.4 | 0.0 |  7.3 |  5.490734 | 0.4 |      0.0 |      0.4 | 1.894374 | 1.704936 | 0.189437 | 0.085108 | 1.704936 | 0.189437 | 0.073424 | 0.073424 | 0.618066 | 0.262861 | 0.880927 |  3.670531 | 10.0 | 130.700885 | 42.858178 |  3.670531 |
    | 30.01. | 10.3 | 0.3 | 0.0 | 10.0 |  7.271132 | 0.3 |      0.0 |      0.3 | 2.840028 | 2.556025 | 0.284003 |  0.11116 | 2.556025 | 0.284003 | 0.080827 | 0.080827 | 0.770158 |  0.36483 | 1.134988 |  4.729116 | 10.0 | 137.860856 | 44.724872 |  4.729116 |
    | 31.01. |  3.7 | 0.3 | 0.0 |  3.4 |  2.406274 | 0.3 |      0.0 |      0.3 | 1.114428 | 1.002985 | 0.111443 | 0.120702 | 1.002985 | 0.111443 | 0.093834 | 0.093834 | 0.797245 | 0.205277 | 1.002521 |  4.177172 | 10.0 | 140.146428 | 45.024446 |  4.177172 |
    | 01.02. | 15.3 | 0.5 | 0.0 | 14.8 | 10.079977 | 0.5 |      0.0 |      0.5 | 4.889993 | 4.400993 | 0.488999 |  0.16997 | 4.400993 | 0.488999 | 0.096052 | 0.096052 | 1.157483 | 0.585051 | 1.742534 |   7.26056 | 10.0 | 150.056435 | 48.364009 |   7.26056 |
    | 02.02. |  3.2 | 0.5 | 0.0 |  2.7 |  1.770334 | 0.5 |      0.0 |      0.5 | 1.108862 | 0.997976 | 0.110886 | 0.179196 | 0.997976 | 0.110886 | 0.123386 | 0.123386 |  1.15346 | 0.234272 | 1.387732 |  5.782216 | 10.0 | 151.647573 | 48.331911 |  5.782216 |
    | 03.02. |  2.7 | 0.3 | 0.0 |  2.4 |  1.557301 | 0.3 |      0.0 |      0.3 | 1.030156 | 0.927141 | 0.103016 | 0.187457 | 0.927141 | 0.103016 | 0.123099 | 0.123099 | 1.142021 | 0.226115 | 1.368136 |  5.700566 | 10.0 | 153.017417 |  48.24013 |  5.700566 |
    | 04.02. |  2.2 | 0.3 | 0.0 |  1.9 |  1.222304 | 0.3 |      0.0 |      0.3 | 0.871554 | 0.784399 | 0.087155 | 0.193858 | 0.784399 | 0.087155 | 0.122283 | 0.122283 |  1.11626 | 0.209439 | 1.325699 |  5.523746 | 10.0 | 154.045863 | 48.030551 |  5.523746 |
    | 05.02. |  8.0 | 0.4 | 0.0 |  7.6 |  4.788405 | 0.4 |      0.0 |      0.4 | 3.036012 | 2.732411 | 0.303601 | 0.224417 | 2.732411 | 0.303601 | 0.120434 | 0.120434 | 1.317036 | 0.424035 | 1.741071 |  7.254462 | 10.0 | 158.609851 |  49.56636 |  7.254462 |
    | 06.02. | 14.3 | 0.4 | 0.0 | 13.9 |  8.329851 | 0.4 |      0.0 |      0.4 | 5.857753 | 5.271977 | 0.585775 | 0.287604 | 5.271977 | 0.585775 |  0.13446 |  0.13446 |  1.89534 | 0.720235 | 2.615575 |  10.89823 | 10.0 | 166.652099 | 53.077457 |  10.89823 |
    | 07.02. |  6.3 | 0.3 | 0.0 |  6.0 |  3.429287 | 0.3 |      0.0 |      0.3 | 2.886312 | 2.597681 | 0.288631 | 0.315599 | 2.597681 | 0.288631 | 0.170853 | 0.170853 | 2.039688 | 0.459484 | 2.499172 | 10.413216 | 10.0 | 169.765786 | 53.806304 | 10.413216 |
    | 08.02. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.312693 | 0.281423 | 0.031269 | 0.312693 | 0.281423 | 0.031269 | 0.179206 | 0.179206 | 1.784234 | 0.210476 | 1.994709 |  8.311289 |  9.8 | 169.453093 |   52.4827 |  8.311289 |
    | 09.02. |  5.9 | 0.1 | 0.0 |  5.6 |  3.124638 | 0.1 |      0.0 |      0.1 | 2.814717 | 2.533246 | 0.281472 | 0.339356 | 2.533246 | 0.281472 | 0.164246 | 0.164246 | 1.928926 | 0.445717 | 2.374644 |  9.894349 | 10.0 | 172.238376 | 53.251264 |  9.894349 |
    | 10.02. |  9.2 | 0.1 | 0.0 |  9.1 |  4.902115 | 0.1 |      0.0 |      0.1 | 4.584328 | 4.125895 | 0.458433 | 0.386444 | 4.125895 | 0.458433 | 0.172819 | 0.172819 | 2.344136 | 0.631252 | 2.975388 | 12.397449 | 10.0 | 176.754048 | 55.205844 | 12.397449 |
    | 11.02. |  6.1 | 0.0 | 0.0 |  6.1 |  3.167752 | 0.0 |      0.0 |      0.0 | 3.349851 | 3.014866 | 0.334985 | 0.417603 | 3.014866 | 0.334985 | 0.196058 | 0.196058 | 2.510955 | 0.531043 | 3.041998 | 12.674992 | 10.0 | 179.504197 | 55.905813 | 12.674992 |
    | 12.02. |  0.1 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.412802 | 0.371521 |  0.04128 | 0.412802 | 0.371521 |  0.04128 | 0.204898 | 0.204898 | 2.149788 | 0.246178 | 2.395965 |  9.983189 | 10.0 | 179.091395 | 54.332445 |  9.983189 |
    | 13.02. |  0.0 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.408098 | 0.367288 |  0.04081 | 0.408098 | 0.367288 |  0.04081 | 0.185415 | 0.185415 | 1.881276 | 0.226225 |   2.1075 |  8.781251 |  9.9 | 178.683297 | 53.003872 |  8.781251 |
    | 14.02. |  2.8 | 0.0 | 0.0 |  2.7 |  1.387085 | 0.0 |      0.0 |      0.0 | 1.732237 | 1.559014 | 0.173224 | 0.419322 | 1.559014 | 0.173224 | 0.170025 | 0.170025 | 1.857028 | 0.343249 | 2.200278 |  9.167824 | 10.0 | 179.651059 | 52.875883 |  9.167824 |
    | 15.02. | 10.6 | 0.2 | 0.0 | 10.4 |  5.178456 | 0.2 |      0.0 |      0.2 | 5.698981 | 5.129083 | 0.569898 | 0.477437 | 5.129083 | 0.569898 | 0.168593 | 0.168593 | 2.463316 | 0.738491 | 3.201807 | 13.340862 | 10.0 | 184.352079 | 55.710242 | 13.340862 |
    | 16.02. |  8.8 | 0.9 | 0.0 |  7.9 |  3.758646 | 0.9 |      0.0 |      0.9 | 4.662457 | 4.196211 | 0.466246 | 0.521103 | 4.196211 | 0.466246 |   0.2024 |   0.2024 | 2.860751 | 0.668645 | 3.529397 | 14.705819 | 10.0 | 187.589622 | 57.248102 | 14.705819 |
    | 17.02. |  7.2 | 0.9 | 0.0 |  6.3 |  2.897342 | 0.9 |      0.0 |      0.9 | 3.957319 | 3.561587 | 0.395732 | 0.554661 | 3.561587 | 0.395732 | 0.222639 | 0.222639 |  3.06552 | 0.618371 | 3.683891 | 15.349546 | 10.0 | 189.932303 | 57.966808 | 15.349546 |
    | 18.02. |  4.9 | 0.5 | 0.0 |  4.4 |  1.976127 | 0.5 |      0.0 |      0.5 | 2.999414 | 2.699472 | 0.299941 | 0.575541 | 2.699472 | 0.299941 | 0.232576 | 0.232576 | 3.035308 | 0.532518 | 3.567825 | 14.865938 | 10.0 |  191.33289 | 57.863549 | 14.865938 |
    | 19.02. |  1.8 | 0.9 | 0.0 |  0.9 |  0.401044 | 0.9 |      0.0 |      0.9 | 1.071901 | 0.964711 |  0.10719 | 0.572945 | 0.964711 |  0.10719 | 0.231129 | 0.231129 | 2.639991 | 0.338319 |  2.97831 | 12.409627 | 10.0 | 191.160989 | 56.419398 | 12.409627 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0
"""

# import...
# ...from HydPy
from hydpy.exe.modelimports import *
from hydpy.core import modeltools
from hydpy.core.typingtools import *
from hydpy.interfaces import petinterfaces
from hydpy.interfaces import rconcinterfaces

# ...from  gland
from hydpy.models.gland import gland_model


class Model(gland_model.Main_PETModel_V1, gland_model.Main_RConcModel_V2):
    """|gland_gr4.DOCNAME.complete|."""

    DOCNAME = modeltools.DocName(
        short="G-GR4", description="Génie Rural model with 4 parameters"
    )
    __HYDPY_ROOTMODEL__ = True

    INLET_METHODS = ()
    RECEIVER_METHODS = ()
    RUN_METHODS = (
        gland_model.Calc_E_V1,
        gland_model.Calc_EI_V1,
        gland_model.Calc_PN_V1,
        gland_model.Calc_EN_V1,
        gland_model.Update_I_V1,
        gland_model.Calc_PS_V1,
        gland_model.Calc_ES_V1,
        gland_model.Update_S_V1,
        gland_model.Calc_Perc_V1,
        gland_model.Update_S_V2,
        gland_model.Calc_AE_V1,
        gland_model.Calc_Pr_V1,
        gland_model.Calc_PR1_PR9_V1,
        gland_model.Calc_Q9_V1,
        gland_model.Calc_Q1_V1,
        gland_model.Calc_FR_V1,
        gland_model.Update_R_V1,
        gland_model.Calc_QR_V1,
        gland_model.Update_R_V3,
        gland_model.Calc_FD_V1,
        gland_model.Calc_QD_V1,
        gland_model.Calc_QH_V1,
        gland_model.Calc_QV_V1,
    )
    INTERFACE_METHODS = ()
    ADD_METHODS = (gland_model.Calc_E_PETModel_V1, gland_model.Calc_Q_RConcModel_V1)
    OUTLET_METHODS = (gland_model.Pass_Q_V1,)
    SENDER_METHODS = ()
    SUBMODELINTERFACES = (petinterfaces.PETModel_V1, rconcinterfaces.RConcModel_V1)
    SUBMODELS = ()

    petmodel = modeltools.SubmodelProperty(petinterfaces.PETModel_V1)

    rconcmodel_routingstore = modeltools.SubmodelProperty(
        rconcinterfaces.RConcModel_V1, optional=True
    )
    rconcmodel_directflow = modeltools.SubmodelProperty(
        rconcinterfaces.RConcModel_V1, optional=True
    )

    def check_waterbalance(self, initial_conditions: ConditionsModel) -> float:
        r"""Determine the water balance error of the previous simulation run in mm.

        Method |Model.check_waterbalance| calculates the balance error as follows:

          .. math::
            Error = \Sigma InOut - \Delta Vol - \Delta Rconc
            \\ \\
            \Sigma InOut = \sum_{t=t0}^{t1} P_t - AE_t + FR_t + FD_t - QH_t
            \\
            \Delta Vol  =  \left( I_{t1} - I_{t0} \right)
            + \left( S_{t1} - S_{t0} \right) + \left( R_{t1} - R_{t0} \right)
            \\
            \Delta Rconc  = get\_waterbalance_{direct \ flow}(*)
            + get\_waterbalance_{routing \ store}(*)
        """
        fluxes = self.sequences.fluxes
        inputs = self.sequences.inputs
        last = self.sequences.states
        first = initial_conditions["model"]["states"]
        return (
            numpy.sum(inputs.p.evalseries)
            - numpy.sum(fluxes.ae.evalseries)
            + numpy.sum(fluxes.fr.evalseries)
            + numpy.sum(fluxes.fd.evalseries)
            - numpy.sum(fluxes.qh.evalseries)
            - (last.i - first["i"])
            - (last.s - first["s"])
            - (last.r - first["r"])
            - self._get_rconcmodel_waterbalance_directflow(initial_conditions)
            - self._get_rconcmodel_waterbalance_routingstore(initial_conditions)
        )


tester = Tester()
cythonizer = Cythonizer()
