"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProwlerAudit = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// import { Stack, Duration, RemovalPolicy, CustomResource } from 'aws-cdk-lib';
// eslint-disable-next-line no-duplicate-imports
// import { aws_iam as iam, aws_logs as logs, aws_s3 as s3, aws_codebuild as codebuild, aws_lambda as lambda, custom_resources as cr } from 'aws-cdk-lib';
const aws_cdk_lib_1 = require("aws-cdk-lib");
const codebuild = require("aws-cdk-lib/aws-codebuild");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
// eslint-disable-next-line import/no-extraneous-dependencies
const s3 = require("aws-cdk-lib/aws-s3");
const cr = require("aws-cdk-lib/custom-resources");
const statement = require("cdk-iam-floyd");
const constructs_1 = require("constructs");
/**
 * Creates a CodeBuild project to audit an AWS account with Prowler and stores the html report in a S3 bucket. This will run onece at the beginning and on a schedule afterwards. Partial contribution from https://github.com/stevecjones
 */
class ProwlerAudit extends constructs_1.Construct {
    constructor(parent, id, props) {
        super(parent, id);
        // defaults
        this.serviceName = props?.serviceName ? props.serviceName : 'prowler';
        this.logsRetentionInDays = props?.logsRetentionInDays ? props.logsRetentionInDays : logs.RetentionDays.THREE_DAYS;
        this.enableScheduler = props?.enableScheduler ? props.enableScheduler : false;
        this.prowlerScheduler = props?.prowlerScheduler ? props.prowlerScheduler : 'cron(0 22 * * ? *)';
        this.prowlerOptions = props?.prowlerOptions ? props.prowlerOptions : '-M text,junit-xml,html,csv,json';
        this.prowlerVersion = props?.prowlerVersion ? props.prowlerVersion : '2.7.0';
        const reportBucket = props?.reportBucket ?? new s3.Bucket(this, 'ReportBucket', {
            //bucketName: `${'123456'}-prowler-reports`,
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        const reportGroup = new codebuild.ReportGroup(this, 'reportGroup', { /**reportGroupName: 'testReportGroup', */ removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY });
        const preBuildCommands = [];
        if (!!props?.allowlist) {
            const prowlerFilename = 'allowlist.txt';
            if (props.allowlist.isZipArchive) {
                preBuildCommands.push(`aws s3 cp ${props.allowlist.s3ObjectUrl} .`);
                preBuildCommands.push(`unzip ${props.allowlist.s3ObjectKey} -d prowler`);
            }
            else {
                preBuildCommands.push(`aws s3 cp ${props.allowlist.s3ObjectUrl} prowler/${prowlerFilename}`);
            }
            this.prowlerOptions = this.prowlerOptions + ` -w ${prowlerFilename}`;
        }
        const prowlerBuild = this.codebuildProject = new codebuild.Project(this, 'prowlerBuild', {
            description: 'Run Prowler assessment',
            timeout: aws_cdk_lib_1.Duration.hours(5),
            environment: {
                environmentVariables: {
                    BUCKET_REPORT: { value: reportBucket.bucketName || '' },
                    BUCKET_PREFIX: { value: props?.reportBucketPrefix ?? '' },
                    ADDITIONAL_S3_ARGS: { value: props?.additionalS3CopyArgs ?? '' },
                    PROWLER_OPTIONS: { value: this.prowlerOptions || '' },
                },
                buildImage: codebuild.LinuxBuildImage.fromCodeBuildImageId('aws/codebuild/amazonlinux2-x86_64-standard:3.0'),
            },
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        'runtime-versions': {
                            python: 3.8,
                        },
                        'commands': [
                            'echo "Installing Prowler and dependencies..."',
                            'pip3 install detect-secrets',
                            'yum -y install jq',
                            'curl "https://awscli.amazonaws.com/awscli-exe-linux-x86_64.zip" -o "awscliv2.zip"',
                            'unzip awscliv2.zip',
                            './aws/install',
                            `git clone -b ${this.prowlerVersion} https://github.com/prowler-cloud/prowler`,
                        ],
                    },
                    pre_build: {
                        commands: preBuildCommands,
                    },
                    build: {
                        commands: [
                            `echo "Running Prowler as ./prowler ${this.prowlerOptions} && echo OK || echo FAILED"`,
                            'cd prowler',
                            `./prowler ${this.prowlerOptions} && echo OK || echo FAILED`,
                        ],
                    },
                    post_build: {
                        commands: [
                            'echo "Uploading reports to S3..." ',
                            'aws s3 cp --sse AES256 output/ s3://$BUCKET_REPORT/$BUCKET_PREFIX --recursive $ADDITIONAL_S3_ARGS',
                            'echo "Done!"',
                        ],
                    },
                },
                reports: {
                    [reportGroup.reportGroupName]: {
                        'files': ['**/*'],
                        'base-directory': 'prowler/junit-reports',
                        'file-format': 'JunitXml',
                    },
                },
            }),
        });
        if (!!props?.allowlist) {
            props.allowlist.bucket.grantRead(prowlerBuild);
        }
        prowlerBuild.role?.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('SecurityAudit'));
        prowlerBuild.role?.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('job-function/ViewOnlyAccess'));
        // prowlerBuild.addToRolePolicy(new statement.Dax().allow().to());
        prowlerBuild.addToRolePolicy(new statement.Ds().allow().toListAuthorizedApplications());
        prowlerBuild.addToRolePolicy(new statement.Ec2().allow().toGetEbsEncryptionByDefault());
        prowlerBuild.addToRolePolicy(new statement.Ecr().allow().toDescribeImageScanFindings().toDescribeImages().toDescribeRegistry());
        prowlerBuild.addToRolePolicy(new statement.Tag().allow().toGetTagKeys());
        prowlerBuild.addToRolePolicy(new statement.Lambda().allow().toGetFunction());
        prowlerBuild.addToRolePolicy(new statement.Glue().allow().toSearchTables().toGetConnections());
        prowlerBuild.addToRolePolicy(new statement.Apigateway().allow().toGET());
        prowlerBuild.addToRolePolicy(new iam.PolicyStatement({ actions: ['support:Describe*'], resources: ['*'] }));
        reportBucket.grantPut(prowlerBuild);
        const myRole = new iam.Role(this, 'MyRole', { assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com') });
        const prowlerStartBuildLambda = new lambda.Function(this, 'prowlerStartBuildLambda', {
            runtime: lambda.Runtime.PYTHON_3_6,
            timeout: aws_cdk_lib_1.Duration.seconds(120),
            handler: 'index.lambda_handler',
            code: lambda.Code.fromInline(`import boto3
import cfnresponse
from botocore.exceptions import ClientError
def lambda_handler(event,context):
  props = event['ResourceProperties']
  codebuild_client = boto3.client('codebuild')
  if (event['RequestType'] == 'Create' or event['RequestType'] == 'Update'):
    try:
        response = codebuild_client.start_build(projectName=props['Build'])
        print(response)
        print("Respond: SUCCESS")
        cfnresponse.send(event, context, cfnresponse.SUCCESS, {})
    except Exception as ex:
        print(ex.response['Error']['Message'])
        cfnresponse.send(event, context, cfnresponse.FAILED, ex.response)
      `),
        });
        prowlerStartBuildLambda.addToRolePolicy(new statement.Codebuild().allow().toStartBuild()); // onResource project ...
        const myProvider = new cr.Provider(this, 'MyProvider', {
            onEventHandler: prowlerStartBuildLambda,
            logRetention: this.logsRetentionInDays,
            role: myRole,
        });
        new aws_cdk_lib_1.CustomResource(this, 'Resource1', {
            serviceToken: myProvider.serviceToken,
            properties: {
                Build: prowlerBuild.projectName,
                RERUN_PROWLER: Boolean(this.node.tryGetContext('reRunProwler')) ? Date.now().toString() : '',
            },
        });
        if (this.enableScheduler) {
            const prowlerSchedulerLambda = new lambda.Function(this, 'ScheduleLambda', {
                runtime: lambda.Runtime.PYTHON_3_6,
                timeout: aws_cdk_lib_1.Duration.seconds(120),
                handler: 'index.lambda_handler',
                environment: {
                    buildName: prowlerBuild.projectName,
                },
                code: lambda.Code.fromInline(`import boto3
        import os
        def lambda_handler(event,context):
          codebuild_client = boto3.client('codebuild')
          print("Running Prowler scheduled!: " + os.environ['buildName'])
          project_name = os.environ['buildName']
          response = codebuild_client.start_build(projectName=project_name)
          print(response)
          print("Respond: SUCCESS")
        `),
            });
            new events.Rule(this, 'Scheduler', {
                description: 'A schedule for the Lambda function that triggers Prowler in CodeBuild.',
                targets: [new targets.LambdaFunction(prowlerSchedulerLambda)],
                schedule: events.Schedule.expression(this.prowlerScheduler || ''),
            });
        }
    }
}
exports.ProwlerAudit = ProwlerAudit;
_a = JSII_RTTI_SYMBOL_1;
ProwlerAudit[_a] = { fqn: "cdk-prowler.ProwlerAudit", version: "2.29.2" };
//# sourceMappingURL=data:application/json;base64,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