from __future__ import annotations
from typing import Any, Dict, Optional
from node_graph import NodeGraph

from ..core.base import BaseEngine
from ..core.execution import (
    compute_graph_outputs,
    execute_node_job,
    iterate_node_order,
    mark_process_failure,
    mark_process_success,
    prepare_graph_run,
)
from ..core.task import EngineNodeExecutor
from ..core.utils import _collect_literals, update_nested_dict_with_special_keys


def _node_job(
    parent_pid: Optional[str],
    _ng_meta,
    _ng_callable=None,
    _ng_engine_name: str = "",
    _ng_node_inputs=None,
    _ng_node_outputs=None,
    **kwargs: Any,
) -> Dict[str, Any]:
    return execute_node_job(
        parent_pid=parent_pid,
        meta=_ng_meta,
        callable_payload=_ng_callable,
        runtime_inputs=kwargs,
        engine_name=_ng_engine_name,
        node_inputs=_ng_node_inputs,
        node_outputs=_ng_node_outputs,
        build_sub_engine=lambda name: DirectEngine(name=name),
    )


class DirectEngine(BaseEngine):
    """
    Sync, dependency-free runner with provenance:

    - @node: calls the underlying python function, normalizes to {result: ...}
    - @node.graph: builds & runs a sub-NodeGraph, resolves returned socket-handles to values
    - Provenance: records runtime *flattened* inputs & outputs around each node run
    - Link semantics from utils: _wait, _outputs, and multi-fan-in bundling
    """

    engine_kind = "direct"

    def __init__(
        self,
        name: str = "direct-flow",
    ):
        super().__init__(name)

    def run(
        self,
        ng: NodeGraph,
        parent_pid: Optional[str] = None,
    ) -> Dict[str, Any]:
        """Execute ``ng`` and return the graph outputs as plain values."""
        context = prepare_graph_run(
            ng,
            parent_pid=parent_pid,
        )
        self._graph_pid = context.process_node.uuid
        values = context.values

        try:
            for name in iterate_node_order(context.order):
                print(f"Running node: {name}")

                node = ng.nodes[name]

                kw = dict(_collect_literals(node))
                link_kwargs = self._build_link_kwargs(
                    target_name=name,
                    links=context.incoming.get(name, []),
                    source_map=values,
                )
                kw.update(link_kwargs)
                kw = update_nested_dict_with_special_keys(kw)

                label_kind = "return" if self._is_graph_node(node) else "create"
                executor = self._build_node_executor(
                    node,
                    label_kind=label_kind,
                )
                results = executor.invoke(
                    parent_pid=context.process_node.uuid,
                    **kw,
                )
                values[name] = results

            graph_outputs = compute_graph_outputs(
                incoming=context.incoming,
                values=values,
                link_builder=self._build_link_kwargs,
            )
            mark_process_success(context.process_node, graph_outputs)
            return graph_outputs
        except Exception as e:
            mark_process_failure(context.process_node, e)
            raise
        finally:
            context.process_node.seal()

    def _build_node_executor(
        self,
        node,
        label_kind: str,
    ) -> EngineNodeExecutor:
        executor = node.spec.executor.to_dict()
        meta = self._build_node_task_meta(node, label_kind)

        static_kwargs = {
            "_ng_engine_name": self.name,
            "_ng_node_inputs": node.spec.inputs.to_dict(),
            "_ng_node_outputs": node.spec.outputs.to_dict(),
        }

        return EngineNodeExecutor(
            runner=_node_job,
            meta=meta,
            callable=executor,
            static_kwargs=static_kwargs,
        )
