"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStore = exports.ValidationSettingsMode = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const iam = require("aws-cdk-lib/aws-iam");
const aws_verifiedpermissions_1 = require("aws-cdk-lib/aws-verifiedpermissions");
const core_1 = require("aws-cdk-lib/core");
const cedar_helpers_1 = require("./cedar-helpers");
const policy_1 = require("./policy");
const permissions_1 = require("./private/permissions");
/**
 * Validation Settings mode, according to the Cloudformation PolicyStore resource
 */
var ValidationSettingsMode;
(function (ValidationSettingsMode) {
    ValidationSettingsMode["OFF"] = "OFF";
    ValidationSettingsMode["STRICT"] = "STRICT";
})(ValidationSettingsMode || (exports.ValidationSettingsMode = ValidationSettingsMode = {}));
class PolicyStoreBase extends core_1.Resource {
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.policyStoreArn],
            scope: this,
        });
    }
    grantAuth(grantee) {
        return this.grant(grantee, ...permissions_1.AUTH_ACTIONS);
    }
    grantRead(grantee) {
        return this.grant(grantee, ...permissions_1.READ_ACTIONS);
    }
    grantWrite(grantee) {
        return this.grant(grantee, ...permissions_1.WRITE_ACTIONS.concat(permissions_1.READ_ACTIONS));
    }
}
class PolicyStore extends PolicyStoreBase {
    /**
     * Create a PolicyStore construct that represents an external policy store via policy store id.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param policyStoreId The PolicyStore's id.
     */
    static fromPolicyStoreId(scope, id, policyStoreId) {
        return PolicyStore.fromPolicyStoreAttributes(scope, id, { policyStoreId });
    }
    /**
     * Create a PolicyStore construct that represents an external PolicyStore via policy store arn.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param policyStoreArn The PolicyStore's ARN.
     */
    static fromPolicyStoreArn(scope, id, policyStoreArn) {
        return PolicyStore.fromPolicyStoreAttributes(scope, id, { policyStoreArn });
    }
    /**
     * Creates a PolicyStore construct that represents an external Policy Store.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `PolicyStoreAttributes` object.
     */
    static fromPolicyStoreAttributes(scope, id, attrs) {
        class Import extends PolicyStoreBase {
            constructor(policyStoreArn, policyStoreId) {
                super(scope, id);
                this.policyStoreArn = policyStoreArn;
                this.policyStoreId = policyStoreId;
            }
        }
        let policyStoreId;
        let policyStoreArn;
        const stack = core_1.Stack.of(scope);
        if (!attrs.policyStoreId) {
            if (!attrs.policyStoreArn) {
                throw new Error('One of policyStoreId or policyStoreArn is required!');
            }
            policyStoreArn = attrs.policyStoreArn;
            const maybeId = stack.splitArn(attrs.policyStoreArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
            if (!maybeId) {
                throw new Error(`ARN for PolicyStore must be in the form: ${core_1.ArnFormat.SLASH_RESOURCE_NAME}`);
            }
            policyStoreId = maybeId;
        }
        else {
            if (attrs.policyStoreArn) {
                throw new Error('Only one of policyStoreArn or policyStoreId can be provided');
            }
            policyStoreId = attrs.policyStoreId;
            policyStoreArn = stack.formatArn({
                resource: 'policy-store',
                resourceName: attrs.policyStoreId,
                service: 'verifiedpermissions',
            });
        }
        return new Import(policyStoreArn, policyStoreId);
    }
    /**
     * This method generates a schema based on an swagger file. It makes the same assumptions and decisions
     * made in the Amazon Verified Permissions console. This feature is built for swagger files generated from an Amazon API Gateway
     * export. It's possible that some swagger files generated by other tools will not work. In that case, please
     * file an issue.
     * @param swaggerFilePath absolute path to a swagger file in the local directory structure, in json format
     * @param groupEntityTypeName optional parameter to specify the group entity type name. If passed, the schema's User type will have a parent of this type.
     */
    static schemaFromOpenApiSpec(swaggerFilePath, groupEntityTypeName) {
        const RELEVANT_HTTP_METHODS = ['get', 'post', 'put', 'patch', 'delete', 'head'];
        const openApiSpecString = fs.readFileSync(swaggerFilePath, 'utf-8');
        const openApiSpec = JSON.parse(openApiSpecString);
        if (!openApiSpec.paths) {
            throw new Error('Invalid OpenAPI spec - missing paths object');
        }
        const namespace = (0, cedar_helpers_1.cleanUpApiNameForNamespace)(openApiSpec.info.title);
        const pathUrls = Object.keys(openApiSpec.paths);
        const actionNames = [];
        for (const pathUrl of pathUrls) {
            const pathDef = openApiSpec.paths[pathUrl];
            if (!pathDef) {
                continue;
            }
            let pathVerbs = Object.keys(pathDef);
            if (pathVerbs.includes('x-amazon-apigateway-any-method')) {
                pathVerbs = RELEVANT_HTTP_METHODS;
            }
            for (const httpVerb of pathVerbs) {
                if (!RELEVANT_HTTP_METHODS.includes(httpVerb)) {
                    continue;
                }
                const actionName = `${httpVerb} ${pathUrl}`;
                actionNames.push(actionName);
            }
        }
        return (0, cedar_helpers_1.buildSchema)(namespace, actionNames, groupEntityTypeName);
    }
    constructor(scope, id, props = {
        validationSettings: {
            mode: ValidationSettingsMode.OFF,
        },
    }) {
        super(scope, id);
        if (props.schema) {
            (0, cedar_helpers_1.checkParseSchema)(props.schema.cedarJson);
        }
        this.policyStore = new aws_verifiedpermissions_1.CfnPolicyStore(this, id, {
            schema: props.schema
                ? {
                    cedarJson: props.schema.cedarJson,
                }
                : undefined,
            validationSettings: {
                mode: props.validationSettings.mode,
            },
            description: props.description,
        });
        this.policyStoreArn = this.getResourceArnAttribute(this.policyStore.attrArn, {
            resource: 'policy-store',
            resourceName: this.physicalName,
            service: 'verifiedpermissions',
        });
        this.policyStoreName = this.getResourceNameAttribute(this.policyStore.ref);
        this.policyStoreId = this.policyStore.attrPolicyStoreId;
        this.schema = props.schema;
        this.validationSettings = props.validationSettings;
        this.description = props.description;
    }
    /**
     * Add multiple policies to the policy store
     *
     * @param policyDefinitions An array of policy options for the policy stores policies.
     * @returns An array of created policy constructs.
     */
    addPolicies(policyDefinitions) {
        let policies = policyDefinitions.map((policyOption) => {
            return new policy_1.Policy(this, policyOption.policyId, {
                policyStore: this,
                definition: policyOption.policyConfiguration,
            });
        });
        return policies;
    }
    /**
     * Takes in an absolute path to a directory containing .cedar files and adds the contents of each
     * .cedar file as policies to this policy store. Parses the policies with cedar-wasm and, if the policy store has a schema,
     * performs semantic validation of the policies as well.
     * @param absolutePath a string representing an absolute path to the directory containing your policies
     * @returns An array of created Policy constructs.
     */
    addPoliciesFromPath(absolutePath) {
        if (!fs.statSync(absolutePath).isDirectory()) {
            throw new Error(`The path ${absolutePath} does not appear to be a directory`);
        }
        const policyFileNames = fs
            .readdirSync(absolutePath)
            .map((f) => path.join(absolutePath, f))
            .filter((f) => !fs.statSync(f).isDirectory() && f.endsWith('.cedar'));
        if (this.validationSettings.mode === ValidationSettingsMode.STRICT) {
            if (!this.schema) {
                throw new Error('A schema must exist when adding policies to a policy store with strict validation mode.');
            }
            for (const policyFile of policyFileNames) {
                const policyStatement = fs.readFileSync(policyFile, 'utf-8');
                (0, cedar_helpers_1.validatePolicy)(policyStatement, this.schema.cedarJson);
            }
        }
        const policies = policyFileNames.map((cedarFile) => policy_1.Policy.fromFile(this, cedarFile, {
            path: cedarFile,
            policyStore: this,
        }));
        return policies;
    }
}
exports.PolicyStore = PolicyStore;
_a = JSII_RTTI_SYMBOL_1;
PolicyStore[_a] = { fqn: "@cdklabs/cdk-verified-permissions.PolicyStore", version: "0.1.2" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicG9saWN5LXN0b3JlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL3BvbGljeS1zdG9yZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLHlCQUF5QjtBQUN6Qiw2QkFBNkI7QUFDN0IsMkNBQTJDO0FBQzNDLGlGQUFxRTtBQUNyRSwyQ0FBeUU7QUFFekUsbURBQTRHO0FBQzVHLHFDQUE0RDtBQUM1RCx1REFJK0I7QUFVL0I7O0dBRUc7QUFDSCxJQUFZLHNCQUdYO0FBSEQsV0FBWSxzQkFBc0I7SUFDaEMscUNBQVcsQ0FBQTtJQUNYLDJDQUFpQixDQUFBO0FBQ25CLENBQUMsRUFIVyxzQkFBc0Isc0NBQXRCLHNCQUFzQixRQUdqQztBQTJHRCxNQUFlLGVBQWdCLFNBQVEsZUFBUTtJQUl0QyxLQUFLLENBQUMsT0FBdUIsRUFBRSxHQUFHLE9BQWlCO1FBQ3hELE9BQU8sR0FBRyxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUM7WUFDOUIsT0FBTztZQUNQLE9BQU87WUFDUCxZQUFZLEVBQUUsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDO1lBQ25DLEtBQUssRUFBRSxJQUFJO1NBQ1osQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVNLFNBQVMsQ0FBQyxPQUF1QjtRQUN0QyxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLEdBQUcsMEJBQVksQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFTSxTQUFTLENBQUMsT0FBdUI7UUFDdEMsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxHQUFHLDBCQUFZLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRU0sVUFBVSxDQUFDLE9BQXVCO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsR0FBRywyQkFBYSxDQUFDLE1BQU0sQ0FBQywwQkFBWSxDQUFDLENBQUMsQ0FBQztJQUNwRSxDQUFDO0NBQ0Y7QUFFRCxNQUFhLFdBQVksU0FBUSxlQUFlO0lBQzlDOzs7Ozs7T0FNRztJQUNJLE1BQU0sQ0FBQyxpQkFBaUIsQ0FDN0IsS0FBZ0IsRUFDaEIsRUFBVSxFQUNWLGFBQXFCO1FBRXJCLE9BQU8sV0FBVyxDQUFDLHlCQUF5QixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsRUFBRSxhQUFhLEVBQUUsQ0FBQyxDQUFDO0lBQzdFLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsa0JBQWtCLENBQzlCLEtBQWdCLEVBQ2hCLEVBQVUsRUFDVixjQUFzQjtRQUV0QixPQUFPLFdBQVcsQ0FBQyx5QkFBeUIsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEVBQUUsY0FBYyxFQUFFLENBQUMsQ0FBQztJQUM5RSxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLHlCQUF5QixDQUNyQyxLQUFnQixFQUNoQixFQUFVLEVBQ1YsS0FBNEI7UUFFNUIsTUFBTSxNQUFPLFNBQVEsZUFBZTtZQUlsQyxZQUFZLGNBQXNCLEVBQUUsYUFBcUI7Z0JBQ3ZELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7Z0JBRWpCLElBQUksQ0FBQyxjQUFjLEdBQUcsY0FBYyxDQUFDO2dCQUNyQyxJQUFJLENBQUMsYUFBYSxHQUFHLGFBQWEsQ0FBQztZQUNyQyxDQUFDO1NBQ0Y7UUFFRCxJQUFJLGFBQXFCLENBQUM7UUFDMUIsSUFBSSxjQUFzQixDQUFDO1FBQzNCLE1BQU0sS0FBSyxHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFOUIsSUFBSSxDQUFDLEtBQUssQ0FBQyxhQUFhLEVBQUUsQ0FBQztZQUN6QixJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsRUFBRSxDQUFDO2dCQUMxQixNQUFNLElBQUksS0FBSyxDQUFDLHFEQUFxRCxDQUFDLENBQUM7WUFDekUsQ0FBQztZQUVELGNBQWMsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDO1lBQ3RDLE1BQU0sT0FBTyxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQzVCLEtBQUssQ0FBQyxjQUFjLEVBQ3BCLGdCQUFTLENBQUMsbUJBQW1CLENBQzlCLENBQUMsWUFBWSxDQUFDO1lBRWYsSUFBSSxDQUFDLE9BQU8sRUFBRSxDQUFDO2dCQUNiLE1BQU0sSUFBSSxLQUFLLENBQ2IsNENBQTRDLGdCQUFTLENBQUMsbUJBQW1CLEVBQUUsQ0FDNUUsQ0FBQztZQUNKLENBQUM7WUFFRCxhQUFhLEdBQUcsT0FBTyxDQUFDO1FBQzFCLENBQUM7YUFBTSxDQUFDO1lBQ04sSUFBSSxLQUFLLENBQUMsY0FBYyxFQUFFLENBQUM7Z0JBQ3pCLE1BQU0sSUFBSSxLQUFLLENBQ2IsNkRBQTZELENBQzlELENBQUM7WUFDSixDQUFDO1lBRUQsYUFBYSxHQUFHLEtBQUssQ0FBQyxhQUFhLENBQUM7WUFDcEMsY0FBYyxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUM7Z0JBQy9CLFFBQVEsRUFBRSxjQUFjO2dCQUN4QixZQUFZLEVBQUUsS0FBSyxDQUFDLGFBQWE7Z0JBQ2pDLE9BQU8sRUFBRSxxQkFBcUI7YUFDL0IsQ0FBQyxDQUFDO1FBQ0wsQ0FBQztRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsY0FBYyxFQUFFLGFBQWEsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7Ozs7OztPQU9HO0lBQ0ksTUFBTSxDQUFDLHFCQUFxQixDQUFDLGVBQXVCLEVBQUUsbUJBQTRCO1FBQ3ZGLE1BQU0scUJBQXFCLEdBQUcsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLEtBQUssRUFBRSxPQUFPLEVBQUUsUUFBUSxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ2hGLE1BQU0saUJBQWlCLEdBQUcsRUFBRSxDQUFDLFlBQVksQ0FBQyxlQUFlLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDcEUsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBUSxDQUFDO1FBQ3pELElBQUksQ0FBQyxXQUFXLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDdkIsTUFBTSxJQUFJLEtBQUssQ0FBQyw2Q0FBNkMsQ0FBQyxDQUFDO1FBQ2pFLENBQUM7UUFDRCxNQUFNLFNBQVMsR0FBRyxJQUFBLDBDQUEwQixFQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFckUsTUFBTSxRQUFRLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDaEQsTUFBTSxXQUFXLEdBQUcsRUFBRSxDQUFDO1FBQ3ZCLEtBQUssTUFBTSxPQUFPLElBQUksUUFBUSxFQUFFLENBQUM7WUFDL0IsTUFBTSxPQUFPLEdBQUcsV0FBVyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztZQUMzQyxJQUFJLENBQUMsT0FBTyxFQUFFLENBQUM7Z0JBQ2IsU0FBUztZQUNYLENBQUM7WUFDRCxJQUFJLFNBQVMsR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1lBQ3JDLElBQUksU0FBUyxDQUFDLFFBQVEsQ0FBQyxnQ0FBZ0MsQ0FBQyxFQUFFLENBQUM7Z0JBQ3pELFNBQVMsR0FBRyxxQkFBcUIsQ0FBQztZQUNwQyxDQUFDO1lBQ0QsS0FBSyxNQUFNLFFBQVEsSUFBSSxTQUFTLEVBQUUsQ0FBQztnQkFDakMsSUFBSSxDQUFDLHFCQUFxQixDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDO29CQUM5QyxTQUFTO2dCQUNYLENBQUM7Z0JBQ0QsTUFBTSxVQUFVLEdBQUcsR0FBRyxRQUFRLElBQUksT0FBTyxFQUFFLENBQUM7Z0JBQzVDLFdBQVcsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7WUFDL0IsQ0FBQztRQUNILENBQUM7UUFDRCxPQUFPLElBQUEsMkJBQVcsRUFBQyxTQUFTLEVBQUUsV0FBVyxFQUFFLG1CQUFtQixDQUFDLENBQUM7SUFDbEUsQ0FBQztJQW9DRCxZQUNFLEtBQWdCLEVBQ2hCLEVBQVUsRUFDVixRQUEwQjtRQUN4QixrQkFBa0IsRUFBRTtZQUNsQixJQUFJLEVBQUUsc0JBQXNCLENBQUMsR0FBRztTQUNqQztLQUNGO1FBRUQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNqQixJQUFJLEtBQUssQ0FBQyxNQUFNLEVBQUUsQ0FBQztZQUNqQixJQUFBLGdDQUFnQixFQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDM0MsQ0FBQztRQUNELElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSx3Q0FBYyxDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDOUMsTUFBTSxFQUFFLEtBQUssQ0FBQyxNQUFNO2dCQUNsQixDQUFDLENBQUM7b0JBQ0EsU0FBUyxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUztpQkFDbEM7Z0JBQ0QsQ0FBQyxDQUFDLFNBQVM7WUFDYixrQkFBa0IsRUFBRTtnQkFDbEIsSUFBSSxFQUFFLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJO2FBQ3BDO1lBQ0QsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1NBQy9CLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDLHVCQUF1QixDQUNoRCxJQUFJLENBQUMsV0FBVyxDQUFDLE9BQU8sRUFDeEI7WUFDRSxRQUFRLEVBQUUsY0FBYztZQUN4QixZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDL0IsT0FBTyxFQUFFLHFCQUFxQjtTQUMvQixDQUNGLENBQUM7UUFDRixJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQzNFLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxpQkFBaUIsQ0FBQztRQUN4RCxJQUFJLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUM7UUFDM0IsSUFBSSxDQUFDLGtCQUFrQixHQUFHLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQztRQUNuRCxJQUFJLENBQUMsV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksV0FBVyxDQUFDLGlCQUFxQztRQUN0RCxJQUFJLFFBQVEsR0FBRyxpQkFBaUIsQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLEVBQUUsRUFBRTtZQUNwRCxPQUFPLElBQUksZUFBTSxDQUFDLElBQUksRUFBRSxZQUFZLENBQUMsUUFBUSxFQUFFO2dCQUM3QyxXQUFXLEVBQUUsSUFBSTtnQkFDakIsVUFBVSxFQUFFLFlBQVksQ0FBQyxtQkFBbUI7YUFDN0MsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFDSCxPQUFPLFFBQVEsQ0FBQztJQUNsQixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CLENBQUMsWUFBb0I7UUFDN0MsSUFBSSxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDLENBQUMsV0FBVyxFQUFFLEVBQUUsQ0FBQztZQUM3QyxNQUFNLElBQUksS0FBSyxDQUNiLFlBQVksWUFBWSxvQ0FBb0MsQ0FDN0QsQ0FBQztRQUNKLENBQUM7UUFDRCxNQUFNLGVBQWUsR0FBRyxFQUFFO2FBQ3ZCLFdBQVcsQ0FBQyxZQUFZLENBQUM7YUFDekIsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksRUFBRSxDQUFDLENBQUMsQ0FBQzthQUN0QyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7UUFFeEUsSUFBSSxJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxLQUFLLHNCQUFzQixDQUFDLE1BQU0sRUFBRSxDQUFDO1lBQ25FLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLENBQUM7Z0JBQ2pCLE1BQU0sSUFBSSxLQUFLLENBQ2IseUZBQXlGLENBQzFGLENBQUM7WUFDSixDQUFDO1lBQ0QsS0FBSyxNQUFNLFVBQVUsSUFBSSxlQUFlLEVBQUUsQ0FBQztnQkFDekMsTUFBTSxlQUFlLEdBQUcsRUFBRSxDQUFDLFlBQVksQ0FBQyxVQUFVLEVBQUUsT0FBTyxDQUFDLENBQUM7Z0JBQzdELElBQUEsOEJBQWMsRUFBQyxlQUFlLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQztZQUN6RCxDQUFDO1FBQ0gsQ0FBQztRQUVELE1BQU0sUUFBUSxHQUFHLGVBQWUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxTQUFTLEVBQUUsRUFBRSxDQUNqRCxlQUFNLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDL0IsSUFBSSxFQUFFLFNBQVM7WUFDZixXQUFXLEVBQUUsSUFBSTtTQUNsQixDQUFDLENBQ0gsQ0FBQztRQUVGLE9BQU8sUUFBUSxDQUFDO0lBQ2xCLENBQUM7O0FBclFILGtDQXNRQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XG5pbXBvcnQgeyBDZm5Qb2xpY3lTdG9yZSB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy12ZXJpZmllZHBlcm1pc3Npb25zJztcbmltcG9ydCB7IEFybkZvcm1hdCwgSVJlc291cmNlLCBSZXNvdXJjZSwgU3RhY2sgfSBmcm9tICdhd3MtY2RrLWxpYi9jb3JlJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgYnVpbGRTY2hlbWEsIGNoZWNrUGFyc2VTY2hlbWEsIGNsZWFuVXBBcGlOYW1lRm9yTmFtZXNwYWNlLCB2YWxpZGF0ZVBvbGljeSB9IGZyb20gJy4vY2VkYXItaGVscGVycyc7XG5pbXBvcnQgeyBQb2xpY3ksIFBvbGljeURlZmluaXRpb25Qcm9wZXJ0eSB9IGZyb20gJy4vcG9saWN5JztcbmltcG9ydCB7XG4gIEFVVEhfQUNUSU9OUyxcbiAgUkVBRF9BQ1RJT05TLFxuICBXUklURV9BQ1RJT05TLFxufSBmcm9tICcuL3ByaXZhdGUvcGVybWlzc2lvbnMnO1xuXG5leHBvcnQgaW50ZXJmYWNlIFNjaGVtYSB7XG4gIHJlYWRvbmx5IGNlZGFySnNvbjogc3RyaW5nO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFZhbGlkYXRpb25TZXR0aW5ncyB7XG4gIHJlYWRvbmx5IG1vZGU6IFZhbGlkYXRpb25TZXR0aW5nc01vZGU7XG59XG5cbi8qKlxuICogVmFsaWRhdGlvbiBTZXR0aW5ncyBtb2RlLCBhY2NvcmRpbmcgdG8gdGhlIENsb3VkZm9ybWF0aW9uIFBvbGljeVN0b3JlIHJlc291cmNlXG4gKi9cbmV4cG9ydCBlbnVtIFZhbGlkYXRpb25TZXR0aW5nc01vZGUge1xuICBPRkYgPSAnT0ZGJyxcbiAgU1RSSUNUID0gJ1NUUklDVCcsXG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgSVBvbGljeVN0b3JlIGV4dGVuZHMgSVJlc291cmNlIHtcbiAgLyoqXG4gICAqIEFSTiBvZiB0aGUgUG9saWN5IFN0b3JlLlxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSBwb2xpY3lTdG9yZUFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBJRCBvZiB0aGUgUG9saWN5IFN0b3JlLlxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSBwb2xpY3lTdG9yZUlkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqXG4gICAqIEFkZHMgYW4gSUFNIHBvbGljeSBzdGF0ZW1lbnQgYXNzb2NpYXRlZCB3aXRoIHRoaXMgcG9saWN5IHN0b3JlIHRvIGFuIElBTVxuICAgKiBwcmluY2lwYWwncyBwb2xpY3kuXG4gICAqXG4gICAqIEBwYXJhbSBncmFudGVlIFRoZSBwcmluY2lwYWwgKG5vLW9wIGlmIHVuZGVmaW5lZClcbiAgICogQHBhcmFtIGFjdGlvbnMgVGhlIHNldCBvZiBhY3Rpb25zIHRvIGFsbG93IChpLmUuIFwidmVyaWZpZWRwZXJtaXNzaW9uczpJc0F1dGhvcml6ZWRcIiwgXCJ2ZXJpZmllZHBlcm1pc3Npb25zOkxpc3RQb2xpY2llc1wiLCAuLi4pXG4gICAqL1xuICBncmFudChncmFudGVlOiBpYW0uSUdyYW50YWJsZSwgLi4uYWN0aW9uczogc3RyaW5nW10pOiBpYW0uR3JhbnQ7XG5cbiAgLyoqXG4gICAqIFBlcm1pdHMgYW4gSUFNIHByaW5jaXBhbCBhbGwgcmVhZCBvcGVyYXRpb25zIG9uIHRoZSBwb2xpY3kgc3RvcmU6IEdldElkZW50aXR5U291cmNlLFxuICAgKiBHZXRQb2xpY3ksIEdldFBvbGljeVN0b3JlLCBHZXRQb2xpY3lUZW1wbGF0ZSxcbiAgICogR2V0U2NoZW1hLCBMaXN0SWRlbnRpdHlTb3VyY2VzLCBMaXN0UG9saWNpZXMsIExpc3RQb2xpY3lUZW1wbGF0ZXMuXG4gICAqXG4gICAqIEBwYXJhbSBncmFudGVlXG4gICAqL1xuICBncmFudFJlYWQoZ3JhbnRlZTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQ7XG5cbiAgLyoqXG4gICAqIFBlcm1pdHMgYW4gSUFNIHByaW5jaXBhbCBhbGwgd3JpdGUgJiByZWFkIG9wZXJhdGlvbnMgb24gdGhlIHBvbGljeSBzdG9yZTogQ3JlYXRlSWRlbnRpdHlTb3VyY2UsXG4gICAqIENyZWF0ZVBvbGljeSxDcmVhdGVQb2xpY3lUZW1wbGF0ZSwgRGVsZXRlSWRlbnRpdHlTb3VyY2UsIERlbGV0ZVBvbGljeSxcbiAgICogRGVsZXRlUG9saWN5VGVtcGxhdGUsIFB1dFNjaGVtYSwgVXBkYXRlSWRlbnRpdHlTb3VyY2UsIFVwZGF0ZVBvbGljeSwgVXBkYXRlUG9saWN5VGVtcGxhdGUuXG4gICAqXG4gICAqIEBwYXJhbSBncmFudGVlXG4gICAqL1xuICBncmFudFdyaXRlKGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50O1xuXG4gIC8qKlxuICAgKiBQZXJtaXRzIGFuIElBTSBwcmluY2lwYWwgYWxsIGF1dGggb3BlcmF0aW9ucyBvbiB0aGUgcG9saWN5IHN0b3JlOlxuICAgKiBJc0F1dGhvcml6ZWQsIElzQXV0aG9yaXplZFdpdGhUb2tlblxuICAgKiBAcGFyYW0gZ3JhbnRlZVxuICAgKi9cbiAgZ3JhbnRBdXRoKGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50O1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFBvbGljeVN0b3JlUHJvcHMge1xuICAvKipcbiAgICogVGhpcyBhdHRyaWJ1dGUgaXMgbm90IHJlcXVpcmVkIGZyb20gYW4gQVBJIHBvaW50IG9mIHZpZXcuXG4gICAqIEl0IHJlcHJlc2VudHMgdGhlIHNjaGVtYSAoaW4gQ2VkYXIpIHRvIGJlIGFwcGxpZWQgdG8gdGhlIFBvbGljeVN0b3JlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIHNjaGVtYS5cbiAgICovXG4gIHJlYWRvbmx5IHNjaGVtYT86IFNjaGVtYTtcblxuICAvKipcbiAgICogVGhlIHBvbGljeSBzdG9yZSdzIHZhbGlkYXRpb24gc2V0dGluZ3MuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gSWYgbm90IHByb3ZpZGVkLCB0aGUgUG9saWN5IHN0b3JlIHdpbGwgYmUgY3JlYXRlZCB3aXRoIFZhbGlkYXRpb25TZXR0aW5nc01vZGUgPSBcIk9GRlwiXG4gICAqL1xuICByZWFkb25seSB2YWxpZGF0aW9uU2V0dGluZ3M6IFZhbGlkYXRpb25TZXR0aW5ncztcblxuICAvKipcbiAgICogVGhlIHBvbGljeSBzdG9yZSdzIGRlc2NyaXB0aW9uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gZGVzY3JpcHRpb24uXG4gICAqL1xuICByZWFkb25seSBkZXNjcmlwdGlvbj86IHN0cmluZztcbn1cblxuZXhwb3J0IGludGVyZmFjZSBBZGRQb2xpY3lPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBpZCBvZiB0aGUgUG9saWN5LlxuICAgKi9cbiAgcmVhZG9ubHkgcG9saWN5SWQ6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGNvbmZpZ3VyYXRpb24gb2YgdGhlIFBvbGljeS5cbiAgICovXG4gIHJlYWRvbmx5IHBvbGljeUNvbmZpZ3VyYXRpb246IFBvbGljeURlZmluaXRpb25Qcm9wZXJ0eTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBQb2xpY3lTdG9yZUF0dHJpYnV0ZXMge1xuICAvKipcbiAgICogVGhlIEFSTiBvZiB0aGUgQW1hem9uIFZlcmlmaWVkIFBlcm1pc3Npb25zIFBvbGljeSBTdG9yZS5cbiAgICogT25lIG9mIHRoaXMsIG9yIGBwb2xpY3lTdG9yZUlkYCwgaXMgcmVxdWlyZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gUG9saWN5U3RvcmUgYXJuXG4gICAqL1xuICByZWFkb25seSBwb2xpY3lTdG9yZUFybj86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGlkIG9mIHRoZSBBbWF6b24gVmVyaWZpZWQgUGVybWlzc2lvbnMgUG9saWN5U3RvcmUuXG4gICAqIE9uZSBvZiB0aGlzLCBvciBgcG9saWN5U3RvcmVBcm5gLCBpcyByZXF1aXJlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBQb2xpY3lTdG9yZSBpZFxuICAgKi9cbiAgcmVhZG9ubHkgcG9saWN5U3RvcmVJZD86IHN0cmluZztcbn1cblxuYWJzdHJhY3QgY2xhc3MgUG9saWN5U3RvcmVCYXNlIGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJUG9saWN5U3RvcmUge1xuICBhYnN0cmFjdCByZWFkb25seSBwb2xpY3lTdG9yZUFybjogc3RyaW5nO1xuICBhYnN0cmFjdCByZWFkb25seSBwb2xpY3lTdG9yZUlkOiBzdHJpbmc7XG5cbiAgcHVibGljIGdyYW50KGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlLCAuLi5hY3Rpb25zOiBzdHJpbmdbXSk6IGlhbS5HcmFudCB7XG4gICAgcmV0dXJuIGlhbS5HcmFudC5hZGRUb1ByaW5jaXBhbCh7XG4gICAgICBncmFudGVlLFxuICAgICAgYWN0aW9ucyxcbiAgICAgIHJlc291cmNlQXJuczogW3RoaXMucG9saWN5U3RvcmVBcm5dLFxuICAgICAgc2NvcGU6IHRoaXMsXG4gICAgfSk7XG4gIH1cblxuICBwdWJsaWMgZ3JhbnRBdXRoKGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICByZXR1cm4gdGhpcy5ncmFudChncmFudGVlLCAuLi5BVVRIX0FDVElPTlMpO1xuICB9XG5cbiAgcHVibGljIGdyYW50UmVhZChncmFudGVlOiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudCB7XG4gICAgcmV0dXJuIHRoaXMuZ3JhbnQoZ3JhbnRlZSwgLi4uUkVBRF9BQ1RJT05TKTtcbiAgfVxuXG4gIHB1YmxpYyBncmFudFdyaXRlKGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICByZXR1cm4gdGhpcy5ncmFudChncmFudGVlLCAuLi5XUklURV9BQ1RJT05TLmNvbmNhdChSRUFEX0FDVElPTlMpKTtcbiAgfVxufVxuXG5leHBvcnQgY2xhc3MgUG9saWN5U3RvcmUgZXh0ZW5kcyBQb2xpY3lTdG9yZUJhc2Uge1xuICAvKipcbiAgICogQ3JlYXRlIGEgUG9saWN5U3RvcmUgY29uc3RydWN0IHRoYXQgcmVwcmVzZW50cyBhbiBleHRlcm5hbCBwb2xpY3kgc3RvcmUgdmlhIHBvbGljeSBzdG9yZSBpZC5cbiAgICpcbiAgICogQHBhcmFtIHNjb3BlIFRoZSBwYXJlbnQgY3JlYXRpbmcgY29uc3RydWN0ICh1c3VhbGx5IGB0aGlzYCkuXG4gICAqIEBwYXJhbSBpZCBUaGUgY29uc3RydWN0J3MgbmFtZS5cbiAgICogQHBhcmFtIHBvbGljeVN0b3JlSWQgVGhlIFBvbGljeVN0b3JlJ3MgaWQuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21Qb2xpY3lTdG9yZUlkKFxuICAgIHNjb3BlOiBDb25zdHJ1Y3QsXG4gICAgaWQ6IHN0cmluZyxcbiAgICBwb2xpY3lTdG9yZUlkOiBzdHJpbmcsXG4gICk6IElQb2xpY3lTdG9yZSB7XG4gICAgcmV0dXJuIFBvbGljeVN0b3JlLmZyb21Qb2xpY3lTdG9yZUF0dHJpYnV0ZXMoc2NvcGUsIGlkLCB7IHBvbGljeVN0b3JlSWQgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGEgUG9saWN5U3RvcmUgY29uc3RydWN0IHRoYXQgcmVwcmVzZW50cyBhbiBleHRlcm5hbCBQb2xpY3lTdG9yZSB2aWEgcG9saWN5IHN0b3JlIGFybi5cbiAgICpcbiAgICogQHBhcmFtIHNjb3BlIFRoZSBwYXJlbnQgY3JlYXRpbmcgY29uc3RydWN0ICh1c3VhbGx5IGB0aGlzYCkuXG4gICAqIEBwYXJhbSBpZCBUaGUgY29uc3RydWN0J3MgbmFtZS5cbiAgICogQHBhcmFtIHBvbGljeVN0b3JlQXJuIFRoZSBQb2xpY3lTdG9yZSdzIEFSTi5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVBvbGljeVN0b3JlQXJuKFxuICAgIHNjb3BlOiBDb25zdHJ1Y3QsXG4gICAgaWQ6IHN0cmluZyxcbiAgICBwb2xpY3lTdG9yZUFybjogc3RyaW5nLFxuICApOiBJUG9saWN5U3RvcmUge1xuICAgIHJldHVybiBQb2xpY3lTdG9yZS5mcm9tUG9saWN5U3RvcmVBdHRyaWJ1dGVzKHNjb3BlLCBpZCwgeyBwb2xpY3lTdG9yZUFybiB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgUG9saWN5U3RvcmUgY29uc3RydWN0IHRoYXQgcmVwcmVzZW50cyBhbiBleHRlcm5hbCBQb2xpY3kgU3RvcmUuXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZSBUaGUgcGFyZW50IGNyZWF0aW5nIGNvbnN0cnVjdCAodXN1YWxseSBgdGhpc2ApLlxuICAgKiBAcGFyYW0gaWQgVGhlIGNvbnN0cnVjdCdzIG5hbWUuXG4gICAqIEBwYXJhbSBhdHRycyBBIGBQb2xpY3lTdG9yZUF0dHJpYnV0ZXNgIG9iamVjdC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVBvbGljeVN0b3JlQXR0cmlidXRlcyhcbiAgICBzY29wZTogQ29uc3RydWN0LFxuICAgIGlkOiBzdHJpbmcsXG4gICAgYXR0cnM6IFBvbGljeVN0b3JlQXR0cmlidXRlcyxcbiAgKTogSVBvbGljeVN0b3JlIHtcbiAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBQb2xpY3lTdG9yZUJhc2Uge1xuICAgICAgcmVhZG9ubHkgcG9saWN5U3RvcmVBcm46IHN0cmluZztcbiAgICAgIHJlYWRvbmx5IHBvbGljeVN0b3JlSWQ6IHN0cmluZztcblxuICAgICAgY29uc3RydWN0b3IocG9saWN5U3RvcmVBcm46IHN0cmluZywgcG9saWN5U3RvcmVJZDogc3RyaW5nKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICAgICAgdGhpcy5wb2xpY3lTdG9yZUFybiA9IHBvbGljeVN0b3JlQXJuO1xuICAgICAgICB0aGlzLnBvbGljeVN0b3JlSWQgPSBwb2xpY3lTdG9yZUlkO1xuICAgICAgfVxuICAgIH1cblxuICAgIGxldCBwb2xpY3lTdG9yZUlkOiBzdHJpbmc7XG4gICAgbGV0IHBvbGljeVN0b3JlQXJuOiBzdHJpbmc7XG4gICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZihzY29wZSk7XG5cbiAgICBpZiAoIWF0dHJzLnBvbGljeVN0b3JlSWQpIHtcbiAgICAgIGlmICghYXR0cnMucG9saWN5U3RvcmVBcm4pIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdPbmUgb2YgcG9saWN5U3RvcmVJZCBvciBwb2xpY3lTdG9yZUFybiBpcyByZXF1aXJlZCEnKTtcbiAgICAgIH1cblxuICAgICAgcG9saWN5U3RvcmVBcm4gPSBhdHRycy5wb2xpY3lTdG9yZUFybjtcbiAgICAgIGNvbnN0IG1heWJlSWQgPSBzdGFjay5zcGxpdEFybihcbiAgICAgICAgYXR0cnMucG9saWN5U3RvcmVBcm4sXG4gICAgICAgIEFybkZvcm1hdC5TTEFTSF9SRVNPVVJDRV9OQU1FLFxuICAgICAgKS5yZXNvdXJjZU5hbWU7XG5cbiAgICAgIGlmICghbWF5YmVJZCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgYEFSTiBmb3IgUG9saWN5U3RvcmUgbXVzdCBiZSBpbiB0aGUgZm9ybTogJHtBcm5Gb3JtYXQuU0xBU0hfUkVTT1VSQ0VfTkFNRX1gLFxuICAgICAgICApO1xuICAgICAgfVxuXG4gICAgICBwb2xpY3lTdG9yZUlkID0gbWF5YmVJZDtcbiAgICB9IGVsc2Uge1xuICAgICAgaWYgKGF0dHJzLnBvbGljeVN0b3JlQXJuKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICAnT25seSBvbmUgb2YgcG9saWN5U3RvcmVBcm4gb3IgcG9saWN5U3RvcmVJZCBjYW4gYmUgcHJvdmlkZWQnLFxuICAgICAgICApO1xuICAgICAgfVxuXG4gICAgICBwb2xpY3lTdG9yZUlkID0gYXR0cnMucG9saWN5U3RvcmVJZDtcbiAgICAgIHBvbGljeVN0b3JlQXJuID0gc3RhY2suZm9ybWF0QXJuKHtcbiAgICAgICAgcmVzb3VyY2U6ICdwb2xpY3ktc3RvcmUnLFxuICAgICAgICByZXNvdXJjZU5hbWU6IGF0dHJzLnBvbGljeVN0b3JlSWQsXG4gICAgICAgIHNlcnZpY2U6ICd2ZXJpZmllZHBlcm1pc3Npb25zJyxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHJldHVybiBuZXcgSW1wb3J0KHBvbGljeVN0b3JlQXJuLCBwb2xpY3lTdG9yZUlkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGlzIG1ldGhvZCBnZW5lcmF0ZXMgYSBzY2hlbWEgYmFzZWQgb24gYW4gc3dhZ2dlciBmaWxlLiBJdCBtYWtlcyB0aGUgc2FtZSBhc3N1bXB0aW9ucyBhbmQgZGVjaXNpb25zXG4gICAqIG1hZGUgaW4gdGhlIEFtYXpvbiBWZXJpZmllZCBQZXJtaXNzaW9ucyBjb25zb2xlLiBUaGlzIGZlYXR1cmUgaXMgYnVpbHQgZm9yIHN3YWdnZXIgZmlsZXMgZ2VuZXJhdGVkIGZyb20gYW4gQW1hem9uIEFQSSBHYXRld2F5XG4gICAqIGV4cG9ydC4gSXQncyBwb3NzaWJsZSB0aGF0IHNvbWUgc3dhZ2dlciBmaWxlcyBnZW5lcmF0ZWQgYnkgb3RoZXIgdG9vbHMgd2lsbCBub3Qgd29yay4gSW4gdGhhdCBjYXNlLCBwbGVhc2VcbiAgICogZmlsZSBhbiBpc3N1ZS5cbiAgICogQHBhcmFtIHN3YWdnZXJGaWxlUGF0aCBhYnNvbHV0ZSBwYXRoIHRvIGEgc3dhZ2dlciBmaWxlIGluIHRoZSBsb2NhbCBkaXJlY3Rvcnkgc3RydWN0dXJlLCBpbiBqc29uIGZvcm1hdFxuICAgKiBAcGFyYW0gZ3JvdXBFbnRpdHlUeXBlTmFtZSBvcHRpb25hbCBwYXJhbWV0ZXIgdG8gc3BlY2lmeSB0aGUgZ3JvdXAgZW50aXR5IHR5cGUgbmFtZS4gSWYgcGFzc2VkLCB0aGUgc2NoZW1hJ3MgVXNlciB0eXBlIHdpbGwgaGF2ZSBhIHBhcmVudCBvZiB0aGlzIHR5cGUuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHNjaGVtYUZyb21PcGVuQXBpU3BlYyhzd2FnZ2VyRmlsZVBhdGg6IHN0cmluZywgZ3JvdXBFbnRpdHlUeXBlTmFtZT86IHN0cmluZykge1xuICAgIGNvbnN0IFJFTEVWQU5UX0hUVFBfTUVUSE9EUyA9IFsnZ2V0JywgJ3Bvc3QnLCAncHV0JywgJ3BhdGNoJywgJ2RlbGV0ZScsICdoZWFkJ107XG4gICAgY29uc3Qgb3BlbkFwaVNwZWNTdHJpbmcgPSBmcy5yZWFkRmlsZVN5bmMoc3dhZ2dlckZpbGVQYXRoLCAndXRmLTgnKTtcbiAgICBjb25zdCBvcGVuQXBpU3BlYyA9IEpTT04ucGFyc2Uob3BlbkFwaVNwZWNTdHJpbmcpIGFzIGFueTtcbiAgICBpZiAoIW9wZW5BcGlTcGVjLnBhdGhzKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0ludmFsaWQgT3BlbkFQSSBzcGVjIC0gbWlzc2luZyBwYXRocyBvYmplY3QnKTtcbiAgICB9XG4gICAgY29uc3QgbmFtZXNwYWNlID0gY2xlYW5VcEFwaU5hbWVGb3JOYW1lc3BhY2Uob3BlbkFwaVNwZWMuaW5mby50aXRsZSk7XG5cbiAgICBjb25zdCBwYXRoVXJscyA9IE9iamVjdC5rZXlzKG9wZW5BcGlTcGVjLnBhdGhzKTtcbiAgICBjb25zdCBhY3Rpb25OYW1lcyA9IFtdO1xuICAgIGZvciAoY29uc3QgcGF0aFVybCBvZiBwYXRoVXJscykge1xuICAgICAgY29uc3QgcGF0aERlZiA9IG9wZW5BcGlTcGVjLnBhdGhzW3BhdGhVcmxdO1xuICAgICAgaWYgKCFwYXRoRGVmKSB7XG4gICAgICAgIGNvbnRpbnVlO1xuICAgICAgfVxuICAgICAgbGV0IHBhdGhWZXJicyA9IE9iamVjdC5rZXlzKHBhdGhEZWYpO1xuICAgICAgaWYgKHBhdGhWZXJicy5pbmNsdWRlcygneC1hbWF6b24tYXBpZ2F0ZXdheS1hbnktbWV0aG9kJykpIHtcbiAgICAgICAgcGF0aFZlcmJzID0gUkVMRVZBTlRfSFRUUF9NRVRIT0RTO1xuICAgICAgfVxuICAgICAgZm9yIChjb25zdCBodHRwVmVyYiBvZiBwYXRoVmVyYnMpIHtcbiAgICAgICAgaWYgKCFSRUxFVkFOVF9IVFRQX01FVEhPRFMuaW5jbHVkZXMoaHR0cFZlcmIpKSB7XG4gICAgICAgICAgY29udGludWU7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgYWN0aW9uTmFtZSA9IGAke2h0dHBWZXJifSAke3BhdGhVcmx9YDtcbiAgICAgICAgYWN0aW9uTmFtZXMucHVzaChhY3Rpb25OYW1lKTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIGJ1aWxkU2NoZW1hKG5hbWVzcGFjZSwgYWN0aW9uTmFtZXMsIGdyb3VwRW50aXR5VHlwZU5hbWUpO1xuICB9XG5cbiAgcHJpdmF0ZSByZWFkb25seSBwb2xpY3lTdG9yZTogQ2ZuUG9saWN5U3RvcmU7XG4gIC8qKlxuICAgKiBBUk4gb2YgdGhlIFBvbGljeSBTdG9yZS5cbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgcG9saWN5U3RvcmVBcm46IHN0cmluZztcbiAgLyoqXG4gICAqIElEIG9mIHRoZSBQb2xpY3kgU3RvcmUuXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IHBvbGljeVN0b3JlSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogTmFtZSBvZiB0aGUgUG9saWN5IFN0b3JlLlxuICAgKi9cbiAgcmVhZG9ubHkgcG9saWN5U3RvcmVOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFNjaGVtYSBkZWZpbml0aW9uIG9mIHRoZSBQb2xpY3kgU3RvcmUuXG4gICAqL1xuICByZWFkb25seSBzY2hlbWE/OiBTY2hlbWE7XG5cbiAgLyoqXG4gICAqIFZhbGlkYXRpb24gU2V0dGluZ3Mgb2YgdGhlIFBvbGljeSBTdG9yZS5cbiAgICovXG4gIHJlYWRvbmx5IHZhbGlkYXRpb25TZXR0aW5nczogVmFsaWRhdGlvblNldHRpbmdzO1xuXG4gIC8qKlxuICAgKiBEZXNjcmlwdGlvbiBvZiB0aGUgUG9saWN5IFN0b3JlXG4gICAqL1xuICByZWFkb25seSBkZXNjcmlwdGlvbj86IHN0cmluZztcblxuICBjb25zdHJ1Y3RvcihcbiAgICBzY29wZTogQ29uc3RydWN0LFxuICAgIGlkOiBzdHJpbmcsXG4gICAgcHJvcHM6IFBvbGljeVN0b3JlUHJvcHMgPSB7XG4gICAgICB2YWxpZGF0aW9uU2V0dGluZ3M6IHtcbiAgICAgICAgbW9kZTogVmFsaWRhdGlvblNldHRpbmdzTW9kZS5PRkYsXG4gICAgICB9LFxuICAgIH0sXG4gICkge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gICAgaWYgKHByb3BzLnNjaGVtYSkge1xuICAgICAgY2hlY2tQYXJzZVNjaGVtYShwcm9wcy5zY2hlbWEuY2VkYXJKc29uKTtcbiAgICB9XG4gICAgdGhpcy5wb2xpY3lTdG9yZSA9IG5ldyBDZm5Qb2xpY3lTdG9yZSh0aGlzLCBpZCwge1xuICAgICAgc2NoZW1hOiBwcm9wcy5zY2hlbWFcbiAgICAgICAgPyB7XG4gICAgICAgICAgY2VkYXJKc29uOiBwcm9wcy5zY2hlbWEuY2VkYXJKc29uLFxuICAgICAgICB9XG4gICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgdmFsaWRhdGlvblNldHRpbmdzOiB7XG4gICAgICAgIG1vZGU6IHByb3BzLnZhbGlkYXRpb25TZXR0aW5ncy5tb2RlLFxuICAgICAgfSxcbiAgICAgIGRlc2NyaXB0aW9uOiBwcm9wcy5kZXNjcmlwdGlvbixcbiAgICB9KTtcbiAgICB0aGlzLnBvbGljeVN0b3JlQXJuID0gdGhpcy5nZXRSZXNvdXJjZUFybkF0dHJpYnV0ZShcbiAgICAgIHRoaXMucG9saWN5U3RvcmUuYXR0ckFybixcbiAgICAgIHtcbiAgICAgICAgcmVzb3VyY2U6ICdwb2xpY3ktc3RvcmUnLFxuICAgICAgICByZXNvdXJjZU5hbWU6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgICAgICBzZXJ2aWNlOiAndmVyaWZpZWRwZXJtaXNzaW9ucycsXG4gICAgICB9LFxuICAgICk7XG4gICAgdGhpcy5wb2xpY3lTdG9yZU5hbWUgPSB0aGlzLmdldFJlc291cmNlTmFtZUF0dHJpYnV0ZSh0aGlzLnBvbGljeVN0b3JlLnJlZik7XG4gICAgdGhpcy5wb2xpY3lTdG9yZUlkID0gdGhpcy5wb2xpY3lTdG9yZS5hdHRyUG9saWN5U3RvcmVJZDtcbiAgICB0aGlzLnNjaGVtYSA9IHByb3BzLnNjaGVtYTtcbiAgICB0aGlzLnZhbGlkYXRpb25TZXR0aW5ncyA9IHByb3BzLnZhbGlkYXRpb25TZXR0aW5ncztcbiAgICB0aGlzLmRlc2NyaXB0aW9uID0gcHJvcHMuZGVzY3JpcHRpb247XG4gIH1cblxuICAvKipcbiAgICogQWRkIG11bHRpcGxlIHBvbGljaWVzIHRvIHRoZSBwb2xpY3kgc3RvcmVcbiAgICpcbiAgICogQHBhcmFtIHBvbGljeURlZmluaXRpb25zIEFuIGFycmF5IG9mIHBvbGljeSBvcHRpb25zIGZvciB0aGUgcG9saWN5IHN0b3JlcyBwb2xpY2llcy5cbiAgICogQHJldHVybnMgQW4gYXJyYXkgb2YgY3JlYXRlZCBwb2xpY3kgY29uc3RydWN0cy5cbiAgICovXG4gIHB1YmxpYyBhZGRQb2xpY2llcyhwb2xpY3lEZWZpbml0aW9uczogQWRkUG9saWN5T3B0aW9uc1tdKTogUG9saWN5W10ge1xuICAgIGxldCBwb2xpY2llcyA9IHBvbGljeURlZmluaXRpb25zLm1hcCgocG9saWN5T3B0aW9uKSA9PiB7XG4gICAgICByZXR1cm4gbmV3IFBvbGljeSh0aGlzLCBwb2xpY3lPcHRpb24ucG9saWN5SWQsIHtcbiAgICAgICAgcG9saWN5U3RvcmU6IHRoaXMsXG4gICAgICAgIGRlZmluaXRpb246IHBvbGljeU9wdGlvbi5wb2xpY3lDb25maWd1cmF0aW9uLFxuICAgICAgfSk7XG4gICAgfSk7XG4gICAgcmV0dXJuIHBvbGljaWVzO1xuICB9XG5cbiAgLyoqXG4gICAqIFRha2VzIGluIGFuIGFic29sdXRlIHBhdGggdG8gYSBkaXJlY3RvcnkgY29udGFpbmluZyAuY2VkYXIgZmlsZXMgYW5kIGFkZHMgdGhlIGNvbnRlbnRzIG9mIGVhY2hcbiAgICogLmNlZGFyIGZpbGUgYXMgcG9saWNpZXMgdG8gdGhpcyBwb2xpY3kgc3RvcmUuIFBhcnNlcyB0aGUgcG9saWNpZXMgd2l0aCBjZWRhci13YXNtIGFuZCwgaWYgdGhlIHBvbGljeSBzdG9yZSBoYXMgYSBzY2hlbWEsXG4gICAqIHBlcmZvcm1zIHNlbWFudGljIHZhbGlkYXRpb24gb2YgdGhlIHBvbGljaWVzIGFzIHdlbGwuXG4gICAqIEBwYXJhbSBhYnNvbHV0ZVBhdGggYSBzdHJpbmcgcmVwcmVzZW50aW5nIGFuIGFic29sdXRlIHBhdGggdG8gdGhlIGRpcmVjdG9yeSBjb250YWluaW5nIHlvdXIgcG9saWNpZXNcbiAgICogQHJldHVybnMgQW4gYXJyYXkgb2YgY3JlYXRlZCBQb2xpY3kgY29uc3RydWN0cy5cbiAgICovXG4gIHB1YmxpYyBhZGRQb2xpY2llc0Zyb21QYXRoKGFic29sdXRlUGF0aDogc3RyaW5nKTogUG9saWN5W10ge1xuICAgIGlmICghZnMuc3RhdFN5bmMoYWJzb2x1dGVQYXRoKS5pc0RpcmVjdG9yeSgpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIGBUaGUgcGF0aCAke2Fic29sdXRlUGF0aH0gZG9lcyBub3QgYXBwZWFyIHRvIGJlIGEgZGlyZWN0b3J5YCxcbiAgICAgICk7XG4gICAgfVxuICAgIGNvbnN0IHBvbGljeUZpbGVOYW1lcyA9IGZzXG4gICAgICAucmVhZGRpclN5bmMoYWJzb2x1dGVQYXRoKVxuICAgICAgLm1hcCgoZikgPT4gcGF0aC5qb2luKGFic29sdXRlUGF0aCwgZikpXG4gICAgICAuZmlsdGVyKChmKSA9PiAhZnMuc3RhdFN5bmMoZikuaXNEaXJlY3RvcnkoKSAmJiBmLmVuZHNXaXRoKCcuY2VkYXInKSk7XG5cbiAgICBpZiAodGhpcy52YWxpZGF0aW9uU2V0dGluZ3MubW9kZSA9PT0gVmFsaWRhdGlvblNldHRpbmdzTW9kZS5TVFJJQ1QpIHtcbiAgICAgIGlmICghdGhpcy5zY2hlbWEpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICdBIHNjaGVtYSBtdXN0IGV4aXN0IHdoZW4gYWRkaW5nIHBvbGljaWVzIHRvIGEgcG9saWN5IHN0b3JlIHdpdGggc3RyaWN0IHZhbGlkYXRpb24gbW9kZS4nLFxuICAgICAgICApO1xuICAgICAgfVxuICAgICAgZm9yIChjb25zdCBwb2xpY3lGaWxlIG9mIHBvbGljeUZpbGVOYW1lcykge1xuICAgICAgICBjb25zdCBwb2xpY3lTdGF0ZW1lbnQgPSBmcy5yZWFkRmlsZVN5bmMocG9saWN5RmlsZSwgJ3V0Zi04Jyk7XG4gICAgICAgIHZhbGlkYXRlUG9saWN5KHBvbGljeVN0YXRlbWVudCwgdGhpcy5zY2hlbWEuY2VkYXJKc29uKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBjb25zdCBwb2xpY2llcyA9IHBvbGljeUZpbGVOYW1lcy5tYXAoKGNlZGFyRmlsZSkgPT5cbiAgICAgIFBvbGljeS5mcm9tRmlsZSh0aGlzLCBjZWRhckZpbGUsIHtcbiAgICAgICAgcGF0aDogY2VkYXJGaWxlLFxuICAgICAgICBwb2xpY3lTdG9yZTogdGhpcyxcbiAgICAgIH0pLFxuICAgICk7XG5cbiAgICByZXR1cm4gcG9saWNpZXM7XG4gIH1cbn1cbiJdfQ==