# Copyright (c) 2012, Braiden Kindt.
# All rights reserved.
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 
#   1. Redistributions of source code must retain the above copyright
#      notice, this list of conditions and the following disclaimer.
# 
#   2. Redistributions in binary form must reproduce the above
#      copyright notice, this list of conditions and the following
#      disclaimer in the documentation and/or other materials
#      provided with the distribution.
# 
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS
# ''AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
# FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
# COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
# BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY
# WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.


import logging
import dbm
import os

_log = logging.getLogger("antd.plugin")
_plugins = []

class Plugin(object):
    """
    A plugin receives notifications when new data
    is availible, it can consume the data or transform it.
    TCX file generation, and garmin connect upload are
    both implementations of plugin. You can implement
    your own to produce new file formats or upload somewhere.
    """

    def data_availible(self, device_sn, format, files):
        """
        Notification that data is availible, this could
        be raw packet data from device, or higher level
        data generated by other plugins, e.g. TCX.
        Return: files which were sucessfullly processed.
        """
        pass


class PluginQueue(object):
    """
    File based queue representing unprocessed
    files which were not handled the a plugin.
    """

    def __init__(self, plugin):
        try: self.queue_file_name = plugin.cache
        except AttributeError: self.queue_file_name = None
        self.queue = []

    def load_queue(self):
        if self.queue_file_name and os.path.isfile(self.queue_file_name):
            with open(self.queue_file_name, "r") as file:
                lines = file.read().splitlines()
            self.queue = []
            for line in lines:
                device_sn, format, file = line.split(",")
                if os.path.isfile(file):
                    self.queue.append((int(device_sn), format, file))
                else:
                    _log.warning("File pending processing, but disappeared. %s", file)
    
    def save_queue(self):
        if self.queue_file_name and self.queue: 
            with open(self.queue_file_name, "w") as file:
                file.writelines("%d,%s,%s\n" % e for e in self.queue)
        elif self.queue_file_name and os.path.isfile(self.queue_file_name):
            os.unlink(self.queue_file_name)
    
    def add_to_queue(self, device_sn, format, files):
        for file in files:
            self.queue.append((device_sn, format, file))


def register_plugins(*plugins):
    _plugins.extend(p for p in plugins if p is not None)
    for plugin in plugins:
        try: plugin and recover_and_publish_data(plugin) 
        except Exception: _log.warning("Plugin failed. %s", plugin, exc_info=True)

def recover_and_publish_data(plugin):
    q = PluginQueue(plugin)
    q.load_queue()
    if q.queue:
        try:
            _log.debug("Attempting to reprocess failed files.")
            for device_sn, format, file in list(q.queue):
                if plugin.data_availible(device_sn, format, [file]):
                    q.queue.remove((device_sn, format, file))
        except Exception:
            _log.warning("Plugin failed. %s", plugin, exc_info=True)
        finally:
            q.save_queue()
    
def publish_data(device_sn, format, files):
    for plugin in _plugins:
        try:
            processed = plugin.data_availible(device_sn, format, files)
            not_processed = [f for f in files if f not in processed]
        except Exception: 
            processed = []
            not_processed = files
            _log.warning("Plugin failed. %s", plugin, exc_info=True)
        finally:
            q = PluginQueue(plugin)
            q.load_queue()
            q.add_to_queue(device_sn, format, not_processed)
            q.save_queue()


# vim: ts=4 sts=4 et
