"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.rootPathTo = exports.Stack = exports.STACK_RESOURCE_LIMIT_CONTEXT = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const constructs_1 = require("constructs");
const annotations_1 = require("./annotations");
const app_1 = require("./app");
const arn_1 = require("./arn");
const cfn_element_1 = require("./cfn-element");
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cfn_resource_1 = require("./cfn-resource");
const context_provider_1 = require("./context-provider");
const feature_flags_1 = require("./feature-flags");
const cloudformation_lang_1 = require("./private/cloudformation-lang");
const logical_id_1 = require("./private/logical-id");
const resolve_1 = require("./private/resolve");
const uniqueid_1 = require("./private/uniqueid");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const construct_compat_1 = require("./construct-compat");
const STACK_SYMBOL = Symbol.for('@aws-cdk/core.Stack');
const MY_STACK_CACHE = Symbol.for('@aws-cdk/core.Stack.myStack');
exports.STACK_RESOURCE_LIMIT_CONTEXT = '@aws-cdk/core:stackResourceLimit';
const VALID_STACK_NAME_REGEX = /^[A-Za-z][A-Za-z0-9-]*$/;
const MAX_RESOURCES = 500;
/**
 * A root construct which represents a single CloudFormation stack.
 *
 * @stability stable
 */
class Stack extends construct_compat_1.Construct {
    /**
     * Creates a new stack.
     *
     * @param scope Parent of this stack, usually an `App` or a `Stage`, but could be any construct.
     * @param id The construct ID of this stack.
     * @param props Stack properties.
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b, _c, _d;
        // For unit test scope and id are optional for stacks, but we still want an App
        // as the parent because apps implement much of the synthesis logic.
        scope = scope !== null && scope !== void 0 ? scope : new app_1.App({
            autoSynth: false,
            outdir: fs_1.FileSystem.mkdtemp('cdk-test-app-'),
        });
        // "Default" is a "hidden id" from a `node.uniqueId` perspective
        id = id !== null && id !== void 0 ? id : 'Default';
        super(scope, id);
        this._missingContext = new Array();
        this._stackDependencies = {};
        this.templateOptions = {};
        Object.defineProperty(this, STACK_SYMBOL, { value: true });
        this._logicalIds = new logical_id_1.LogicalIDs();
        const { account, region, environment } = this.parseEnvironment(props.env);
        this.account = account;
        this.region = region;
        this.environment = environment;
        this.terminationProtection = props.terminationProtection;
        if (props.description !== undefined) {
            // Max length 1024 bytes
            // Typically 2 bytes per character, may be more for more exotic characters
            if (props.description.length > 512) {
                throw new Error(`Stack description must be <= 1024 bytes. Received description: '${props.description}'`);
            }
            this.templateOptions.description = props.description;
        }
        this._stackName = (_b = props.stackName) !== null && _b !== void 0 ? _b : this.generateStackName();
        this.tags = new tag_manager_1.TagManager(cfn_resource_1.TagType.KEY_VALUE, 'aws:cdk:stack', props.tags);
        if (!VALID_STACK_NAME_REGEX.test(this.stackName)) {
            throw new Error(`Stack name must match the regular expression: ${VALID_STACK_NAME_REGEX.toString()}, got '${this.stackName}'`);
        }
        // the preferred behavior is to generate a unique id for this stack and use
        // it as the artifact ID in the assembly. this allows multiple stacks to use
        // the same name. however, this behavior is breaking for 1.x so it's only
        // applied under a feature flag which is applied automatically for new
        // projects created using `cdk init`.
        //
        // Also use the new behavior if we are using the new CI/CD-ready synthesizer; that way
        // people only have to flip one flag.
        const featureFlags = feature_flags_1.FeatureFlags.of(this);
        const stackNameDupeContext = featureFlags.isEnabled(cxapi.ENABLE_STACK_NAME_DUPLICATES_CONTEXT);
        const newStyleSynthesisContext = featureFlags.isEnabled(cxapi.NEW_STYLE_STACK_SYNTHESIS_CONTEXT);
        this.artifactId = (stackNameDupeContext || newStyleSynthesisContext)
            ? this.generateStackArtifactId()
            : this.stackName;
        this.templateFile = `${this.artifactId}.template.json`;
        // Not for nested stacks
        this._versionReportingEnabled = ((_c = props.analyticsReporting) !== null && _c !== void 0 ? _c : this.node.tryGetContext(cxapi.ANALYTICS_REPORTING_ENABLED_CONTEXT))
            && !this.nestedStackParent;
        this.synthesizer = (_d = props.synthesizer) !== null && _d !== void 0 ? _d : (newStyleSynthesisContext
            ? new stack_synthesizers_1.DefaultStackSynthesizer()
            : new stack_synthesizers_1.LegacyStackSynthesizer());
        this.synthesizer.bind(this);
    }
    /**
     * Return whether the given object is a Stack.
     *
     * We do attribute detection since we can't reliably use 'instanceof'.
     *
     * @stability stable
     */
    static isStack(x) {
        return x !== null && typeof (x) === 'object' && STACK_SYMBOL in x;
    }
    /**
     * Looks up the first stack scope in which `construct` is defined.
     *
     * Fails if there is no stack up the tree.
     *
     * @param construct The construct to start the search from.
     * @stability stable
     */
    static of(construct) {
        // we want this to be as cheap as possible. cache this result by mutating
        // the object. anecdotally, at the time of this writing, @aws-cdk/core unit
        // tests hit this cache 1,112 times, @aws-cdk/aws-cloudformation unit tests
        // hit this 2,435 times).
        const cache = construct[MY_STACK_CACHE];
        if (cache) {
            return cache;
        }
        else {
            const value = _lookup(construct);
            Object.defineProperty(construct, MY_STACK_CACHE, {
                enumerable: false,
                writable: false,
                configurable: false,
                value,
            });
            return value;
        }
        function _lookup(c) {
            var _b, _c;
            if (Stack.isStack(c)) {
                return c;
            }
            const _scope = constructs_1.Node.of(c).scope;
            if (stage_1.Stage.isStage(c) || !_scope) {
                throw new Error(`${(_c = (_b = construct.constructor) === null || _b === void 0 ? void 0 : _b.name) !== null && _c !== void 0 ? _c : 'Construct'} at '${constructs_1.Node.of(construct).path}' should be created in the scope of a Stack, but no Stack found`);
            }
            return _lookup(_scope);
        }
    }
    /**
     * Resolve a tokenized value in the context of the current stack.
     *
     * @stability stable
     */
    resolve(obj) {
        return resolve_1.resolve(obj, {
            scope: this,
            prefix: [],
            resolver: cloudformation_lang_1.CLOUDFORMATION_TOKEN_RESOLVER,
            preparing: false,
        });
    }
    /**
     * Convert an object, potentially containing tokens, to a JSON string.
     *
     * @stability stable
     */
    toJsonString(obj, space) {
        return cloudformation_lang_1.CloudFormationLang.toJSON(obj, space).toString();
    }
    /**
     * (deprecated) DEPRECATED.
     *
     * @deprecated use `reportMissingContextKey()`
     */
    reportMissingContext(report) {
        if (!Object.values(cxschema.ContextProvider).includes(report.provider)) {
            throw new Error(`Unknown context provider requested in: ${JSON.stringify(report)}`);
        }
        this.reportMissingContextKey(report);
    }
    /**
     * Indicate that a context key was expected.
     *
     * Contains instructions which will be emitted into the cloud assembly on how
     * the key should be supplied.
     *
     * @param report The set of parameters needed to obtain the context.
     * @stability stable
     */
    reportMissingContextKey(report) {
        this._missingContext.push(report);
    }
    /**
     * Rename a generated logical identities.
     *
     * To modify the naming scheme strategy, extend the `Stack` class and
     * override the `allocateLogicalId` method.
     *
     * @stability stable
     */
    renameLogicalId(oldId, newId) {
        this._logicalIds.addRename(oldId, newId);
    }
    /**
     * Allocates a stack-unique CloudFormation-compatible logical identity for a specific resource.
     *
     * This method is called when a `CfnElement` is created and used to render the
     * initial logical identity of resources. Logical ID renames are applied at
     * this stage.
     *
     * This method uses the protected method `allocateLogicalId` to render the
     * logical ID for an element. To modify the naming scheme, extend the `Stack`
     * class and override this method.
     *
     * @param element The CloudFormation element for which a logical identity is needed.
     * @stability stable
     */
    getLogicalId(element) {
        const logicalId = this.allocateLogicalId(element);
        return this._logicalIds.applyRename(logicalId);
    }
    /**
     * Add a dependency between this stack and another stack.
     *
     * This can be used to define dependencies between any two stacks within an
     * app, and also supports nested stacks.
     *
     * @stability stable
     */
    addDependency(target, reason) {
        deps_1.addDependency(this, target, reason);
    }
    /**
     * Return the stacks this stack depends on.
     *
     * @stability stable
     */
    get dependencies() {
        return Object.values(this._stackDependencies).map(x => x.stack);
    }
    /**
     * The concrete CloudFormation physical stack name.
     *
     * This is either the name defined explicitly in the `stackName` prop or
     * allocated based on the stack's location in the construct tree. Stacks that
     * are directly defined under the app use their construct `id` as their stack
     * name. Stacks that are defined deeper within the tree will use a hashed naming
     * scheme based on the construct path to ensure uniqueness.
     *
     * If you wish to obtain the deploy-time AWS::StackName intrinsic,
     * you can use `Aws.stackName` directly.
     *
     * @stability stable
     */
    get stackName() {
        return this._stackName;
    }
    /**
     * The partition in which this stack is defined.
     *
     * @stability stable
     */
    get partition() {
        // Always return a non-scoped partition intrinsic. These will usually
        // be used to construct an ARN, but there are no cross-partition
        // calls anyway.
        return cfn_pseudo_1.Aws.PARTITION;
    }
    /**
     * The Amazon domain suffix for the region in which this stack is defined.
     *
     * @stability stable
     */
    get urlSuffix() {
        // Since URL Suffix always follows partition, it is unscoped like partition is.
        return cfn_pseudo_1.Aws.URL_SUFFIX;
    }
    /**
     * The ID of the stack.
     *
     * @stability stable
     * @example
     *
     * // After resolving, looks like
     * 'arn:aws:cloudformation:us-west-2:123456789012:stack/teststack/51af3dc0-da77-11e4-872e-1234567db123'
     */
    get stackId() {
        return new cfn_pseudo_1.ScopedAws(this).stackId;
    }
    /**
     * Returns the list of notification Amazon Resource Names (ARNs) for the current stack.
     *
     * @stability stable
     */
    get notificationArns() {
        return new cfn_pseudo_1.ScopedAws(this).notificationArns;
    }
    /**
     * Indicates if this is a nested stack, in which case `parentStack` will include a reference to it's parent.
     *
     * @stability stable
     */
    get nested() {
        return this.nestedStackResource !== undefined;
    }
    /**
     * Creates an ARN from components.
     *
     * If `partition`, `region` or `account` are not specified, the stack's
     * partition, region and account will be used.
     *
     * If any component is the empty string, an empty string will be inserted
     * into the generated ARN at the location that component corresponds to.
     *
     * The ARN will be formatted as follows:
     *
     *    arn:{partition}:{service}:{region}:{account}:{resource}{sep}}{resource-name}
     *
     * The required ARN pieces that are omitted will be taken from the stack that
     * the 'scope' is attached to. If all ARN pieces are supplied, the supplied scope
     * can be 'undefined'.
     *
     * @stability stable
     */
    formatArn(components) {
        return arn_1.Arn.format(components, this);
    }
    /**
     * Given an ARN, parses it and returns components.
     *
     * IF THE ARN IS A CONCRETE STRING...
     *
     * ...it will be parsed and validated. The separator (`sep`) will be set to '/'
     * if the 6th component includes a '/', in which case, `resource` will be set
     * to the value before the '/' and `resourceName` will be the rest. In case
     * there is no '/', `resource` will be set to the 6th components and
     * `resourceName` will be set to the rest of the string.
     *
     * IF THE ARN IS A TOKEN...
     *
     * ...it cannot be validated, since we don't have the actual value yet at the
     * time of this function call. You will have to supply `sepIfToken` and
     * whether or not ARNs of the expected format usually have resource names
     * in order to parse it properly. The resulting `ArnComponents` object will
     * contain tokens for the subexpressions of the ARN, not string literals.
     *
     * If the resource name could possibly contain the separator char, the actual
     * resource name cannot be properly parsed. This only occurs if the separator
     * char is '/', and happens for example for S3 object ARNs, IAM Role ARNs,
     * IAM OIDC Provider ARNs, etc. To properly extract the resource name from a
     * Tokenized ARN, you must know the resource type and call
     * `Arn.extractResourceName`.
     *
     * @param arn The ARN string to parse.
     * @param sepIfToken The separator used to separate resource from resourceName.
     * @param hasName Whether there is a name component in the ARN at all.
     * @returns an ArnComponents object which allows access to the various
     * components of the ARN.
     * @stability stable
     */
    parseArn(arn, sepIfToken = '/', hasName = true) {
        return arn_1.Arn.parse(arn, sepIfToken, hasName);
    }
    /**
     * Returns the list of AZs that are available in the AWS environment (account/region) associated with this stack.
     *
     * If the stack is environment-agnostic (either account and/or region are
     * tokens), this property will return an array with 2 tokens that will resolve
     * at deploy-time to the first two availability zones returned from CloudFormation's
     * `Fn::GetAZs` intrinsic function.
     *
     * If they are not available in the context, returns a set of dummy values and
     * reports them as missing, and let the CLI resolve them by calling EC2
     * `DescribeAvailabilityZones` on the target environment.
     *
     * To specify a different strategy for selecting availability zones override this method.
     *
     * @stability stable
     */
    get availabilityZones() {
        // if account/region are tokens, we can't obtain AZs through the context
        // provider, so we fallback to use Fn::GetAZs. the current lowest common
        // denominator is 2 AZs across all AWS regions.
        const agnostic = token_1.Token.isUnresolved(this.account) || token_1.Token.isUnresolved(this.region);
        if (agnostic) {
            return this.node.tryGetContext(cxapi.AVAILABILITY_ZONE_FALLBACK_CONTEXT_KEY) || [
                cfn_fn_1.Fn.select(0, cfn_fn_1.Fn.getAzs()),
                cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.getAzs()),
            ];
        }
        const value = context_provider_1.ContextProvider.getValue(this, {
            provider: cxschema.ContextProvider.AVAILABILITY_ZONE_PROVIDER,
            dummyValue: ['dummy1a', 'dummy1b', 'dummy1c'],
        }).value;
        if (!Array.isArray(value)) {
            throw new Error(`Provider ${cxschema.ContextProvider.AVAILABILITY_ZONE_PROVIDER} expects a list`);
        }
        return value;
    }
    /**
     * (deprecated) Register a file asset on this Stack.
     *
     * @deprecated Use `stack.synthesizer.addFileAsset()` if you are calling,
     * and a different IStackSynthesizer class if you are implementing.
     */
    addFileAsset(asset) {
        return this.synthesizer.addFileAsset(asset);
    }
    /**
     * (deprecated) Register a docker image asset on this Stack.
     *
     * @deprecated Use `stack.synthesizer.addDockerImageAsset()` if you are calling,
     * and a different `IStackSynthesizer` class if you are implementing.
     */
    addDockerImageAsset(asset) {
        return this.synthesizer.addDockerImageAsset(asset);
    }
    /**
     * If this is a nested stack, returns it's parent stack.
     *
     * @stability stable
     */
    get nestedStackParent() {
        return this.nestedStackResource && Stack.of(this.nestedStackResource);
    }
    /**
     * (deprecated) Returns the parent of a nested stack.
     *
     * @deprecated use `nestedStackParent`
     */
    get parentStack() {
        return this.nestedStackParent;
    }
    /**
     * Add a Transform to this stack. A Transform is a macro that AWS CloudFormation uses to process your template.
     *
     * Duplicate values are removed when stack is synthesized.
     *
     * @param transform The transform to add.
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/transform-section-structure.html
     * @stability stable
     * @example
     *
     * stack.addTransform('AWS::Serverless-2016-10-31')
     */
    addTransform(transform) {
        if (!this.templateOptions.transforms) {
            this.templateOptions.transforms = [];
        }
        this.templateOptions.transforms.push(transform);
    }
    /**
     * Called implicitly by the `addDependency` helper function in order to
     * realize a dependency between two top-level stacks at the assembly level.
     *
     * Use `stack.addDependency` to define the dependency between any two stacks,
     * and take into account nested stack relationships.
     *
     * @internal
     */
    _addAssemblyDependency(target, reason) {
        // defensive: we should never get here for nested stacks
        if (this.nested || target.nested) {
            throw new Error('Cannot add assembly-level dependencies for nested stacks');
        }
        reason = reason || 'dependency added using stack.addDependency()';
        const cycle = target.stackDependencyReasons(this);
        if (cycle !== undefined) {
            // eslint-disable-next-line max-len
            throw new Error(`'${target.node.path}' depends on '${this.node.path}' (${cycle.join(', ')}). Adding this dependency (${reason}) would create a cyclic reference.`);
        }
        let dep = this._stackDependencies[names_1.Names.uniqueId(target)];
        if (!dep) {
            dep = this._stackDependencies[names_1.Names.uniqueId(target)] = {
                stack: target,
                reasons: [],
            };
        }
        dep.reasons.push(reason);
        if (process.env.CDK_DEBUG_DEPS) {
            // eslint-disable-next-line no-console
            console.error(`[CDK_DEBUG_DEPS] stack "${this.node.path}" depends on "${target.node.path}" because: ${reason}`);
        }
    }
    /**
     * Synthesizes the cloudformation template into a cloud assembly.
     * @internal
     */
    _synthesizeTemplate(session) {
        // In principle, stack synthesis is delegated to the
        // StackSynthesis object.
        //
        // However, some parts of synthesis currently use some private
        // methods on Stack, and I don't really see the value in refactoring
        // this right now, so some parts still happen here.
        const builder = session.assembly;
        const template = this._toCloudFormation();
        // write the CloudFormation template as a JSON file
        const outPath = path.join(builder.outdir, this.templateFile);
        if (this.maxResources > 0) {
            const resources = template.Resources || {};
            const numberOfResources = Object.keys(resources).length;
            if (numberOfResources > this.maxResources) {
                throw new Error(`Number of resources: ${numberOfResources} is greater than allowed maximum of ${this.maxResources}`);
            }
            else if (numberOfResources >= (this.maxResources * 0.8)) {
                annotations_1.Annotations.of(this).addInfo(`Number of resources: ${numberOfResources} is approaching allowed maximum of ${this.maxResources}`);
            }
        }
        fs.writeFileSync(outPath, JSON.stringify(template, undefined, 2));
        for (const ctx of this._missingContext) {
            builder.addMissing(ctx);
        }
    }
    /**
     * Create a CloudFormation Export for a value.
     *
     * Returns a string representing the corresponding `Fn.importValue()`
     * expression for this Export. You can control the name for the export by
     * passing the `name` option.
     *
     * If you don't supply a value for `name`, the value you're exporting must be
     * a Resource attribute (for example: `bucket.bucketName`) and it will be
     * given the same name as the automatic cross-stack reference that would be created
     * if you used the attribute in another Stack.
     *
     * One of the uses for this method is to *remove* the relationship between
     * two Stacks established by automatic cross-stack references. It will
     * temporarily ensure that the CloudFormation Export still exists while you
     * remove the reference from the consuming stack. After that, you can remove
     * the resource and the manual export.
     *
     * ## Example
     *
     * Here is how the process works. Let's say there are two stacks,
     * `producerStack` and `consumerStack`, and `producerStack` has a bucket
     * called `bucket`, which is referenced by `consumerStack` (perhaps because
     * an AWS Lambda Function writes into it, or something like that).
     *
     * It is not safe to remove `producerStack.bucket` because as the bucket is being
     * deleted, `consumerStack` might still be using it.
     *
     * Instead, the process takes two deployments:
     *
     * ### Deployment 1: break the relationship
     *
     * - Make sure `consumerStack` no longer references `bucket.bucketName` (maybe the consumer
     *    stack now uses its own bucket, or it writes to an AWS DynamoDB table, or maybe you just
     *    remove the Lambda Function altogether).
     * - In the `ProducerStack` class, call `this.exportValue(this.bucket.bucketName)`. This
     *    will make sure the CloudFormation Export continues to exist while the relationship
     *    between the two stacks is being broken.
     * - Deploy (this will effectively only change the `consumerStack`, but it's safe to deploy both).
     *
     * ### Deployment 2: remove the bucket resource
     *
     * - You are now free to remove the `bucket` resource from `producerStack`.
     * - Don't forget to remove the `exportValue()` call as well.
     * - Deploy again (this time only the `producerStack` will be changed -- the bucket will be deleted).
     *
     * @stability stable
     */
    exportValue(exportedValue, options = {}) {
        if (options.name) {
            new cfn_output_1.CfnOutput(this, `Export${options.name}`, {
                value: exportedValue,
                exportName: options.name,
            });
            return cfn_fn_1.Fn.importValue(options.name);
        }
        const resolvable = token_1.Tokenization.reverse(exportedValue);
        if (!resolvable || !reference_1.Reference.isReference(resolvable)) {
            throw new Error('exportValue: either supply \'name\' or make sure to export a resource attribute (like \'bucket.bucketName\')');
        }
        // "teleport" the value here, in case it comes from a nested stack. This will also
        // ensure the value is from our own scope.
        const exportable = refs_1.referenceNestedStackValueInParent(resolvable, this);
        // Ensure a singleton "Exports" scoping Construct
        // This mostly exists to trigger LogicalID munging, which would be
        // disabled if we parented constructs directly under Stack.
        // Also it nicely prevents likely construct name clashes
        const exportsScope = getCreateExportsScope(this);
        // Ensure a singleton CfnOutput for this value
        const resolved = this.resolve(exportable);
        const id = 'Output' + JSON.stringify(resolved);
        const exportName = generateExportName(exportsScope, id);
        if (token_1.Token.isUnresolved(exportName)) {
            throw new Error(`unresolved token in generated export name: ${JSON.stringify(this.resolve(exportName))}`);
        }
        const output = exportsScope.node.tryFindChild(id);
        if (!output) {
            new cfn_output_1.CfnOutput(exportsScope, id, { value: token_1.Token.asString(exportable), exportName });
        }
        return cfn_fn_1.Fn.importValue(exportName);
    }
    /**
     * Returns the naming scheme used to allocate logical IDs.
     *
     * By default, uses
     * the `HashedAddressingScheme` but this method can be overridden to customize
     * this behavior.
     *
     * In order to make sure logical IDs are unique and stable, we hash the resource
     * construct tree path (i.e. toplevel/secondlevel/.../myresource) and add it as
     * a suffix to the path components joined without a separator (CloudFormation
     * IDs only allow alphanumeric characters).
     *
     * The result will be:
     *
     *    <path.join('')><md5(path.join('/')>
     *      "human"      "hash"
     *
     * If the "human" part of the ID exceeds 240 characters, we simply trim it so
     * the total ID doesn't exceed CloudFormation's 255 character limit.
     *
     * We only take 8 characters from the md5 hash (0.000005 chance of collision).
     *
     * Special cases:
     *
     * - If the path only contains a single component (i.e. it's a top-level
     *    resource), we won't add the hash to it. The hash is not needed for
     *    disamiguation and also, it allows for a more straightforward migration an
     *    existing CloudFormation template to a CDK stack without logical ID changes
     *    (or renames).
     * - For aesthetic reasons, if the last components of the path are the same
     *    (i.e. `L1/L2/Pipeline/Pipeline`), they will be de-duplicated to make the
     *    resulting human portion of the ID more pleasing: `L1L2Pipeline<HASH>`
     *    instead of `L1L2PipelinePipeline<HASH>`
     * - If a component is named "Default" it will be omitted from the path. This
     *    allows refactoring higher level abstractions around constructs without affecting
     *    the IDs of already deployed resources.
     * - If a component is named "Resource" it will be omitted from the user-visible
     *    path, but included in the hash. This reduces visual noise in the human readable
     *    part of the identifier.
     *
     * @param cfnElement The element for which the logical ID is allocated.
     * @stability stable
     */
    allocateLogicalId(cfnElement) {
        const scopes = cfnElement.node.scopes;
        const stackIndex = scopes.indexOf(cfnElement.stack);
        const pathComponents = scopes.slice(stackIndex + 1).map(x => x.node.id);
        return uniqueid_1.makeUniqueId(pathComponents);
    }
    /**
     * Validate stack name
     *
     * CloudFormation stack names can include dashes in addition to the regular identifier
     * character classes, and we don't allow one of the magic markers.
     *
     * @internal
     */
    _validateId(name) {
        if (name && !VALID_STACK_NAME_REGEX.test(name)) {
            throw new Error(`Stack name must match the regular expression: ${VALID_STACK_NAME_REGEX.toString()}, got '${name}'`);
        }
    }
    /**
     * Returns the CloudFormation template for this stack by traversing
     * the tree and invoking _toCloudFormation() on all Entity objects.
     *
     * @internal
     */
    _toCloudFormation() {
        let transform;
        if (this.templateOptions.transform) {
            // eslint-disable-next-line max-len
            annotations_1.Annotations.of(this).addWarning('This stack is using the deprecated `templateOptions.transform` property. Consider switching to `addTransform()`.');
            this.addTransform(this.templateOptions.transform);
        }
        if (this.templateOptions.transforms) {
            if (this.templateOptions.transforms.length === 1) { // Extract single value
                transform = this.templateOptions.transforms[0];
            }
            else { // Remove duplicate values
                transform = Array.from(new Set(this.templateOptions.transforms));
            }
        }
        const template = {
            Description: this.templateOptions.description,
            Transform: transform,
            AWSTemplateFormatVersion: this.templateOptions.templateFormatVersion,
            Metadata: this.templateOptions.metadata,
        };
        const elements = cfnElements(this);
        const fragments = elements.map(e => this.resolve(e._toCloudFormation()));
        // merge in all CloudFormation fragments collected from the tree
        for (const fragment of fragments) {
            merge(template, fragment);
        }
        // resolve all tokens and remove all empties
        const ret = this.resolve(template) || {};
        this._logicalIds.assertAllRenamesApplied();
        return ret;
    }
    /**
     * (deprecated) Deprecated.
     *
     * @returns reference itself without any change
     * @see https://github.com/aws/aws-cdk/pull/7187
     * @deprecated cross reference handling has been moved to `App.prepare()`.
     */
    prepareCrossReference(_sourceStack, reference) {
        return reference;
    }
    /**
     * Determine the various stack environment attributes.
     *
     */
    parseEnvironment(env = {}) {
        var _b, _c, _d, _e;
        // if an environment property is explicitly specified when the stack is
        // created, it will be used. if not, use tokens for account and region.
        //
        // (They do not need to be anchored to any construct like resource attributes
        // are, because we'll never Export/Fn::ImportValue them -- the only situation
        // in which Export/Fn::ImportValue would work is if the value are the same
        // between producer and consumer anyway, so we can just assume that they are).
        const containingAssembly = stage_1.Stage.of(this);
        const account = (_c = (_b = env.account) !== null && _b !== void 0 ? _b : containingAssembly === null || containingAssembly === void 0 ? void 0 : containingAssembly.account) !== null && _c !== void 0 ? _c : cfn_pseudo_1.Aws.ACCOUNT_ID;
        const region = (_e = (_d = env.region) !== null && _d !== void 0 ? _d : containingAssembly === null || containingAssembly === void 0 ? void 0 : containingAssembly.region) !== null && _e !== void 0 ? _e : cfn_pseudo_1.Aws.REGION;
        // this is the "aws://" env specification that will be written to the cloud assembly
        // manifest. it will use "unknown-account" and "unknown-region" to indicate
        // environment-agnosticness.
        const envAccount = !token_1.Token.isUnresolved(account) ? account : cxapi.UNKNOWN_ACCOUNT;
        const envRegion = !token_1.Token.isUnresolved(region) ? region : cxapi.UNKNOWN_REGION;
        return {
            account,
            region,
            environment: cxapi.EnvironmentUtils.format(envAccount, envRegion),
        };
    }
    /**
     * Maximum number of resources in the stack
     *
     * Set to 0 to mean "unlimited".
     */
    get maxResources() {
        const contextLimit = this.node.tryGetContext(exports.STACK_RESOURCE_LIMIT_CONTEXT);
        return contextLimit !== undefined ? parseInt(contextLimit, 10) : MAX_RESOURCES;
    }
    /**
     * Check whether this stack has a (transitive) dependency on another stack
     *
     * Returns the list of reasons on the dependency path, or undefined
     * if there is no dependency.
     */
    stackDependencyReasons(other) {
        if (this === other) {
            return [];
        }
        for (const dep of Object.values(this._stackDependencies)) {
            const ret = dep.stack.stackDependencyReasons(other);
            if (ret !== undefined) {
                return [...dep.reasons, ...ret];
            }
        }
        return undefined;
    }
    /**
     * Calculate the stack name based on the construct path
     *
     * The stack name is the name under which we'll deploy the stack,
     * and incorporates containing Stage names by default.
     *
     * Generally this looks a lot like how logical IDs are calculated.
     * The stack name is calculated based on the construct root path,
     * as follows:
     *
     * - Path is calculated with respect to containing App or Stage (if any)
     * - If the path is one component long just use that component, otherwise
     *   combine them with a hash.
     *
     * Since the hash is quite ugly and we'd like to avoid it if possible -- but
     * we can't anymore in the general case since it has been written into legacy
     * stacks. The introduction of Stages makes it possible to make this nicer however.
     * When a Stack is nested inside a Stage, we use the path components below the
     * Stage, and prefix the path components of the Stage before it.
     */
    generateStackName() {
        const assembly = stage_1.Stage.of(this);
        const prefix = (assembly && assembly.stageName) ? `${assembly.stageName}-` : '';
        return `${prefix}${this.generateStackId(assembly)}`;
    }
    /**
     * The artifact ID for this stack
     *
     * Stack artifact ID is unique within the App's Cloud Assembly.
     */
    generateStackArtifactId() {
        return this.generateStackId(this.node.root);
    }
    /**
     * Generate an ID with respect to the given container construct.
     */
    generateStackId(container) {
        const rootPath = rootPathTo(this, container);
        const ids = rootPath.map(c => constructs_1.Node.of(c).id);
        // In unit tests our Stack (which is the only component) may not have an
        // id, so in that case just pretend it's "Stack".
        if (ids.length === 1 && !ids[0]) {
            throw new Error('unexpected: stack id must always be defined');
        }
        return makeStackName(ids);
    }
}
exports.Stack = Stack;
_a = JSII_RTTI_SYMBOL_1;
Stack[_a] = { fqn: "@aws-cdk/core.Stack", version: "1.99.0" };
function merge(template, fragment) {
    for (const section of Object.keys(fragment)) {
        const src = fragment[section];
        // create top-level section if it doesn't exist
        const dest = template[section];
        if (!dest) {
            template[section] = src;
        }
        else {
            template[section] = mergeSection(section, dest, src);
        }
    }
}
function mergeSection(section, val1, val2) {
    switch (section) {
        case 'Description':
            return `${val1}\n${val2}`;
        case 'AWSTemplateFormatVersion':
            if (val1 != null && val2 != null && val1 !== val2) {
                throw new Error(`Conflicting CloudFormation template versions provided: '${val1}' and '${val2}`);
            }
            return val1 !== null && val1 !== void 0 ? val1 : val2;
        case 'Transform':
            return mergeSets(val1, val2);
        default:
            return mergeObjectsWithoutDuplicates(section, val1, val2);
    }
}
function mergeSets(val1, val2) {
    const array1 = val1 == null ? [] : (Array.isArray(val1) ? val1 : [val1]);
    const array2 = val2 == null ? [] : (Array.isArray(val2) ? val2 : [val2]);
    for (const value of array2) {
        if (!array1.includes(value)) {
            array1.push(value);
        }
    }
    return array1.length === 1 ? array1[0] : array1;
}
function mergeObjectsWithoutDuplicates(section, dest, src) {
    if (typeof dest !== 'object') {
        throw new Error(`Expecting ${JSON.stringify(dest)} to be an object`);
    }
    if (typeof src !== 'object') {
        throw new Error(`Expecting ${JSON.stringify(src)} to be an object`);
    }
    // add all entities from source section to destination section
    for (const id of Object.keys(src)) {
        if (id in dest) {
            throw new Error(`section '${section}' already contains '${id}'`);
        }
        dest[id] = src[id];
    }
    return dest;
}
/**
 * Collect all CfnElements from a Stack.
 *
 * @param node Root node to collect all CfnElements from
 * @param into Array to append CfnElements to
 * @returns The same array as is being collected into
 */
function cfnElements(node, into = []) {
    if (cfn_element_1.CfnElement.isCfnElement(node)) {
        into.push(node);
    }
    for (const child of constructs_1.Node.of(node).children) {
        // Don't recurse into a substack
        if (Stack.isStack(child)) {
            continue;
        }
        cfnElements(child, into);
    }
    return into;
}
/**
 * Return the construct root path of the given construct relative to the given ancestor
 *
 * If no ancestor is given or the ancestor is not found, return the entire root path.
 */
function rootPathTo(construct, ancestor) {
    const scopes = constructs_1.Node.of(construct).scopes;
    for (let i = scopes.length - 2; i >= 0; i--) {
        if (scopes[i] === ancestor) {
            return scopes.slice(i + 1);
        }
    }
    return scopes;
}
exports.rootPathTo = rootPathTo;
/**
 * makeUniqueId, specialized for Stack names
 *
 * Stack names may contain '-', so we allow that character if the stack name
 * has only one component. Otherwise we fall back to the regular "makeUniqueId"
 * behavior.
 */
function makeStackName(components) {
    if (components.length === 1) {
        return components[0];
    }
    return uniqueid_1.makeUniqueId(components);
}
function getCreateExportsScope(stack) {
    const exportsName = 'Exports';
    let stackExports = stack.node.tryFindChild(exportsName);
    if (stackExports === undefined) {
        stackExports = new construct_compat_1.Construct(stack, exportsName);
    }
    return stackExports;
}
function generateExportName(stackExports, id) {
    const stackRelativeExports = feature_flags_1.FeatureFlags.of(stackExports).isEnabled(cxapi.STACK_RELATIVE_EXPORTS_CONTEXT);
    const stack = Stack.of(stackExports);
    const components = [
        ...stackExports.node.scopes
            .slice(stackRelativeExports ? stack.node.scopes.length : 2)
            .map(c => c.node.id),
        id,
    ];
    const prefix = stack.stackName ? stack.stackName + ':' : '';
    const localPart = uniqueid_1.makeUniqueId(components);
    const maxLength = 255;
    return prefix + localPart.slice(Math.max(0, localPart.length - maxLength + prefix.length));
}
// These imports have to be at the end to prevent circular imports
const cfn_output_1 = require("./cfn-output");
const deps_1 = require("./deps");
const fs_1 = require("./fs");
const names_1 = require("./names");
const reference_1 = require("./reference");
const stack_synthesizers_1 = require("./stack-synthesizers");
const stage_1 = require("./stage");
const tag_manager_1 = require("./tag-manager");
const token_1 = require("./token");
const refs_1 = require("./private/refs");
//# sourceMappingURL=data:application/json;base64,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