from typing import Dict

# Values extracted from RDKit/The Blue Obelisk
# Units: Angstroms (Å)
VDW_RADII: Dict[str, float] = {
    "H": 1.2,
    "He": 1.4,
    "Li": 1.82,
    "Be": 1.7,
    "B": 2.08,
    "C": 1.95,
    "N": 1.85,
    "O": 1.7,
    "F": 1.73,
    "Ne": 1.54,
    "Na": 2.27,
    "Mg": 1.73,
    "Al": 2.05,
    "Si": 2.1,
    "P": 2.08,
    "S": 2.0,
    "Cl": 1.97,
    "Ar": 1.88,
    "K": 2.75,
    "Ca": 1.973,
    "Sc": 1.7,
    "Ti": 1.7,
    "V": 1.7,
    "Cr": 1.7,
    "Mn": 1.7,
    "Fe": 1.7,
    "Co": 1.7,
    "Ni": 1.63,
    "Cu": 1.4,
    "Zn": 1.39,
    "Ga": 1.87,
    "Ge": 1.7,
    "As": 1.85,
    "Se": 1.9,
    "Br": 2.1,
    "Kr": 2.02,
    "Rb": 1.7,
    "Sr": 1.7,
    "Y": 1.7,
    "Zr": 1.7,
    "Nb": 1.7,
    "Mo": 1.7,
    "Tc": 1.7,
    "Ru": 1.7,
    "Rh": 1.7,
    "Pd": 1.63,
    "Ag": 1.72,
    "Cd": 1.58,
    "In": 1.93,
    "Sn": 2.17,
    "Sb": 2.2,
    "Te": 2.06,
    "I": 2.15,
    "Xe": 2.16,
    "Cs": 1.7,
    "Ba": 1.7,
    "La": 1.7,
    "Ce": 1.7,
    "Pr": 1.7,
    "Nd": 1.7,
    "Pm": 1.7,
    "Sm": 1.7,
    "Eu": 1.7,
    "Gd": 1.7,
    "Tb": 1.7,
    "Dy": 1.7,
    "Ho": 1.7,
    "Er": 1.7,
    "Tm": 1.7,
    "Yb": 1.7,
    "Lu": 1.7,
    "Hf": 1.7,
    "Ta": 1.7,
    "W": 1.7,
    "Re": 1.7,
    "Os": 1.7,
    "Ir": 1.7,
    "Pt": 1.72,
    "Au": 1.66,
    "Hg": 1.55,
    "Tl": 1.96,
    "Pb": 2.02,
    "Bi": 1.7,
    "Po": 1.7,
    "At": 1.7,
    "Rn": 1.7,
    "Fr": 1.7,
    "Ra": 1.7,
    "Ac": 1.7,
    "Th": 1.7,
    "Pa": 1.7,
    "U": 1.86,
    "Np": 1.7,
    "Pu": 1.7,
    "Am": 1.7,
    "Cm": 1.7,
    "Bk": 1.7,
    "Cf": 1.7,
    "Es": 1.7,
    "Fm": 1.7,
    "Md": 1.7,
    "No": 1.7,
    "Lr": 1.7,
    "Rf": 1.7,
    "Db": 1.7,
    "Sg": 1.7,
    "Bh": 1.7,
    "Hs": 1.7,
    "Mt": 1.7,
    "Ds": 1.7,
    "Rg": 1.7,
    "Cn": 1.7,
    "Nh": 1.7,
    "Uut": 1.7,
    "Fl": 1.7,
    "Mc": 1.7,
    "Uup": 1.7,
    "Lv": 1.7,
    "Ts": 1.7,
    "Og": 1.7,
}


def get_vdw_radius(element: str, default: float = 1.9) -> float:
    """
    Get the van der Waals radius for a given chemical element.

    Args:
        element (str): The chemical element symbol
        default (float, optional): Default radius if element not found. Defaults to 1.9 Å.

    Returns:
        float: Van der Waals radius in Angstroms

    Example:
        >>> get_vdw_radius('C')
        1.95
        >>> get_vdw_radius('Unknown', 2.0)
        2.0
    """
    return VDW_RADII.get(element, default)
