import torch
from torch.nn import Embedding, Sequential

from pororo.models.tts.tacotron.layers import (
    ConvBlockGenerated,
    HighwayConvBlockGenerated,
)


class GeneratedConvolutionalEncoder(torch.nn.Module):
    """Convolutional encoder (possibly multi-lingual) with weights generated by another network.

    Arguments:
        see ConvolutionalEncoder
        embedding_dim -- size of the generator embedding (should be language embedding)
        bottleneck_dim -- size of the generating layer
    Keyword arguments:
        see ConvolutionalEncoder
    """

    def __init__(
        self,
        input_dim,
        output_dim,
        dropout,
        embedding_dim,
        bottleneck_dim,
        groups=1,
    ):
        super(GeneratedConvolutionalEncoder, self).__init__()

        self._groups = groups
        self._input_dim = input_dim
        self._output_dim = output_dim

        input_dim *= groups
        output_dim *= groups

        layers = ([
            ConvBlockGenerated(
                embedding_dim,
                bottleneck_dim,
                input_dim,
                output_dim,
                1,
                dropout=dropout,
                activation="relu",
                groups=groups,
            ),
            ConvBlockGenerated(
                embedding_dim,
                bottleneck_dim,
                output_dim,
                output_dim,
                1,
                dropout=dropout,
                groups=groups,
            ),
        ] + [
            HighwayConvBlockGenerated(
                embedding_dim,
                bottleneck_dim,
                output_dim,
                output_dim,
                3,
                dropout=dropout,
                dilation=3**i,
                groups=groups,
            ) for i in range(4)
        ] + [
            HighwayConvBlockGenerated(
                embedding_dim,
                bottleneck_dim,
                output_dim,
                output_dim,
                3,
                dropout=dropout,
                dilation=3**i,
                groups=groups,
            ) for i in range(4)
        ] + [
            HighwayConvBlockGenerated(
                embedding_dim,
                bottleneck_dim,
                output_dim,
                output_dim,
                3,
                dropout=dropout,
                dilation=1,
                groups=groups,
            ) for _ in range(2)
        ] + [
            HighwayConvBlockGenerated(
                embedding_dim,
                bottleneck_dim,
                output_dim,
                output_dim,
                1,
                dropout=dropout,
                dilation=1,
                groups=groups,
            ) for _ in range(2)
        ])

        self._layers = Sequential(*layers)
        self._embedding = Embedding(groups, embedding_dim)

    def forward(self, x, x_lenghts=None, x_langs=None):

        # x_langs is specified during inference with batch size 1, so we need to
        # expand the single language to create complete groups (all langs. in parallel)
        if x_langs is not None and x_langs.shape[0] == 1:
            x = x.expand((self._groups, -1, -1))

        # create generator embeddings for all groups
        e = self._embedding(torch.arange(self._groups, device=x.device))

        bs = x.shape[0]
        x = x.transpose(1, 2)
        x = x.reshape(bs // self._groups, self._groups * self._input_dim, -1)
        _, x = self._layers((e, x))
        x = x.reshape(bs, self._output_dim, -1)
        x = x.transpose(1, 2)

        if x_langs is not None and x_langs.shape[0] == 1:
            xr = torch.zeros(1, x.shape[1], x.shape[2], device=x.device)
            x_langs_normed = x_langs / x_langs.sum(2, keepdim=True)[0]
            for l in range(self._groups):
                w = x_langs_normed[0, :, l].reshape(-1, 1)
                xr[0] += w * x[l]
            x = xr

        return x
