import os
import tempfile
import subprocess
import time
import traceback

from pytubefix import YouTube
from pydub import AudioSegment
from InquirerPy import inquirer
from InquirerPy.base.control import Choice
from InquirerPy.validator import PathValidator, EmptyInputValidator
from InquirerPy.separator import Separator


BASE_PATH = os.path.abspath(os.getcwd())


def _progress_callback(stream, data, bytes_remaining):
    progress = stream.filesize - bytes_remaining
    progress = (progress / stream.filesize) * 100
    progress = round(progress)
    proglen = 82
    progchars = (progress / 100) * proglen
    progchars = round(progchars)
    empty = proglen - progchars
    os.system('clear')
    print(f'[{"#"*progchars}{"-"*empty}]')


def chapter_export(yt_obj, filepath, audio_format="mp3"):
    try:
        audio = AudioSegment.from_file(filepath, format=audio_format)
    except Exception as e:
        traceback.print_exc()
        print(f"The above error ocurred while attempting to load {filepath} (using format {audio_format})")
        return False
    framerate = audio.frame_rate
    track_data = {}
    chapters = list(yt_obj.chapters)
    output_dir = os.path.dirname(filepath)
    for chapter in chapters:
        title = chapter.title
        start = chapter.start_seconds * framerate
        end = start + chapter.duration * framerate
        track_data[title] = (start, end)

    auto_choices = [
        Choice(name="Automatically trigger export", value=True),
        Choice(name="Export on confirmation", value=False)
    ]
    auto = inquirer.select(
        choices=auto_choices,
        message="Choose whether to allow the tool to export every chapter or one by one >> ",
        default=auto_choices[0]
    ).execute()

    while len(chapters) > 0:
        header = "Chapter Export Queue"
        print(f"{header}\n{'-'*len(header)}")
        for idx, chapter in enumerate(chapters):
            print(f"[{idx+1}] {chapter.title}")
        
        current = chapters.pop(0)
        title = current.title
        start, end = track_data[title]
        if not auto:
            filename = inquirer.text(
                message="Confirm export filename (do not include file extension) >> ",
                default=title
            ).execute()
            if not inquirer.confirm(
                message="Confirm when ready to begin export of chapter to audio file >> ",
                default=True
            ).execute():
                continue
        else:
            filename = title
        filename = f'{filename}.{audio_format}'

        operation_start = time.time()
        
        print(f"Now slicing audio for chapter '{title}'")
        sample = audio.get_sample_slice(start, end)
        print("Audio data sliced")

        with open(os.path.join(output_dir, filename), 'wb') as f:
            sample.export(out_f=f, format=audio_format)
        operation_end = time.time()
        print(f"{title} successfully exported in {operation_end-operation_start}s")
        print('-'*82)
        print('\n')
    print(f"Finished exporting {len(track_data)} chapters from {yt_obj.title}")
    if inquirer.confirm(
        message="Would you like to delete the original audio track? >> ",
        default=True
    ).execute():
        os.remove(filepath)
        print("Original file purged")
    return main()


def main():
    os.system('clear')
    mix_path = inquirer.filepath(
        message="Enter the output director to save YouTube audio exports to >> ",
        validate=PathValidator(is_dir=True, message="That is not a directory."),
        only_directories=True,
        default=BASE_PATH
    ).execute()
    audio_formats = [
        Choice(name="MP3", value="mp3"),
        Choice(name="WAV", value="wav"),
        Choice(name="FLAC", value="flac"),
    ]
    audio_format = inquirer.select(
        message="Select the audio format to save exported audio >> ",
        choices=audio_formats,
        default=audio_formats[0]
    ).execute()
    if not audio_format:
        audio_format = "mp3"
    
    target_url = inquirer.text(
        message="Enter the URL pointing to a YouTube music mix >> ",
        validate=EmptyInputValidator(message="Empty value is invalid."),
    ).execute()
    if not target_url.startswith('https://') or 'youtube.com' not in target_url.lower():
        raise ValueError(f"{target_url} is not a valid YouTube URL.")
    
    yt = YouTube(target_url, on_progress_callback=_progress_callback)
    print(f'Target URL points to video: "{yt.title}"')

    default_fname = yt.title

    filename = inquirer.text(
        message="Confirm name of file to export audio stream to >> ",
        default=default_fname,
        validate=EmptyInputValidator(message="Filename cannot be empty.")
    ).execute()

    filename = f"{filename}.{audio_format}"

    if not inquirer.confirm(
        message="Confirm when ready to extract audio stream from YouTube video >> ",
        default=True
    ).execute():
        raise SystemExit('Aborting...')

    operation_start = time.time()

    print("\n\nBeginning operation. This may take a while depending on the filesize.")
    try:
        with tempfile.NamedTemporaryFile() as fp:
            stream = yt.streams.filter(only_audio=True).order_by('abr').desc().first()
            print("Streaming video data to buffer.")
            stream.stream_to_buffer(fp)
            print("Video data successfully cached.")
            fp.seek(0)

            print(f"Exporting raw audio data to {audio_format} with FFMPEG")
            command = [
                "ffmpeg",
                "-y",
                "-i",
                fp.name,
                os.path.join(mix_path, filename)
            ]
            subprocess.call(command, stdout=subprocess.DEVNULL, stderr=subprocess.STDOUT)

        if filename not in os.listdir(mix_path):
            raise FileNotFoundError("FFMPEG export file not found in mix path.")
    except Exception as e:
        traceback.print_exc()
        print(f"\nOperation failed with above exceptions.")
        return False
    
    operation_end = time.time()
    print(f"Operation completed in {operation_end-operation_start}s")
    op_choices = [
        Choice(name="Execute again", value="new"),
        Choice(name="Export chapters as audio", value="export"),
        Separator(),
        Choice(name="Exit", value=None),
    ]

    callback = inquirer.select(
        message="Select another operation, or exit the script >> ",
        choices=op_choices,
        default=op_choices[0],
        validate=EmptyInputValidator(message='Invalid input'),
    ).execute()
    if callback == "new":
        return main()
    elif callback == "export":
        return chapter_export(
            yt,
            os.path.join(mix_path, filename),
            audio_format=audio_format
        )
    elif callback is None:
        return
    

if __name__ == "__main__":
    main()

