"""hammad.genai.agents.types.agent_response"""

from typing import List, Any, TypeVar, Literal

from ....cache import cached
from ...models.language.types import (
    LanguageModelResponse,
)


__all__ = [
    "AgentResponse",
    "_create_agent_response_from_language_model_response",
]


T = TypeVar("T")


def _create_agent_response_from_language_model_response(
    response: LanguageModelResponse[T],
    steps: List[LanguageModelResponse[str]] | None = None,
    context: Any = None,
) -> "AgentResponse[T]":
    """Create a AgentResponse from a LanguageModelResponse."""
    try:
        return AgentResponse(
            type="agent",
            model=response.model,
            output=response.output,
            content=response.content,
            completion=response.completion,
            refusal=response.refusal,
            steps=steps or [],
            context=context,
        )
    except Exception as e:
        raise ValueError(
            "Failed to create AgentResponse from LanguageModelResponse."
        ) from e


class AgentResponse(LanguageModelResponse[T]):
    """A response generated by an agent, that includes the steps and final
    output during the agent's execution."""

    type: Literal["agent"] = "agent"
    """The type of the response. Always `agent`."""

    steps: List[LanguageModelResponse[str]]
    """
    A list of steps taken by the agent **BEFORE** its final output.

    NOTE: If the final output was also the first step, this will be
    empty.
    """

    context: Any = None
    """
    The final context object after agent execution.
    
    This is always the final version of the context after any updates
    have been applied during the agent's execution.
    """

    @cached
    def __str__(self) -> str:
        """Pretty prints the response object."""
        output = "AgentResponse:"

        if self.output or self.content:
            output += f"\n{self.output if self.output else self.content}"
        else:
            output += f"\n{self.completion}"

        output += f"\n\n>>> Model: {self.model}"
        # NOTE:
        # added +1 to include final step in the output
        output += f"\n>>> Steps: {len(self.steps) + 1}"
        output += f"\n>>> Tool Calls: {len(self.tool_calls) if self.tool_calls else 0}"

        # Calculate total tool calls across all steps
        total_tool_calls = 0
        for step in self.steps:
            if step.has_tool_calls():
                total_tool_calls += len(step.tool_calls)

        output += f"\n>>> Total Tool Calls: {total_tool_calls}"

        return output
