from typing import Optional

from guarani.webtools import FullDict


class OAuth2Error(Exception):
    """
    Representation of the errors that can occur during the authorization process.

    This is a base class that provides the main attributes defined by
    `<https://tools.ietf.org/html/draft-parecki-oauth-v2-1-03#section-4.1.2.1>`_ and
    `<https://tools.ietf.org/html/draft-parecki-oauth-v2-1-03#section-5.2>`_,
    which are::

        * "error": Denotes the code of the error.
        * "error_description": Human readable description with the details of the error.
        * "error_uri": URI containing more information about the error.
        * "state": String representing the state of the Client before the request.

    :param description: Contains the description of the error.
    :type description: str

    :param uri: Contains the URI that describes the error.
    :type uri: str

    :param state: State of the Client provided in the Request.
    :type state: str, optional
    """

    error: str = None

    def __init__(
        self,
        description: str = None,
        uri: str = None,
        state: Optional[str] = None,
        headers: Optional[dict] = None,
    ):
        self.description = description
        self.uri = uri
        self.state = state
        self.headers = headers or {}

    def dump(self) -> dict:
        """
        Returns a dictionary representation of the error.

        :return: Error as a dictionary.
        :rtype: dict
        """

        return FullDict(
            error=self.error,
            error_description=self.description,
            error_uri=self.uri,
            state=self.state,
        )


class FatalError(OAuth2Error):
    """
    This error is used when the Provider **WILL NOT** redirect to the Client's
    provided Redirect URI due to either the Client or the Redirect URI being invalid.
    """

    error: str = "fatal_error"


class InvalidRequestError(OAuth2Error):
    error: str = "invalid_request"


class InvalidClientError(OAuth2Error):
    error: str = "invalid_client"


class InvalidGrantError(OAuth2Error):
    error: str = "invalid_grant"


class UnauthorizedClientError(OAuth2Error):
    error: str = "unauthorized_client"


class AccessDeniedError(OAuth2Error):
    error: str = "access_denied"


class UnsupportedResponseTypeError(OAuth2Error):
    error: str = "unsupported_response_type"


class UnsupportedGrantTypeError(OAuth2Error):
    error: str = "unsupported_grant_type"


class UnsupportedTokenTypeError(OAuth2Error):
    error: str = "unsupported_token_type"


class InvalidScopeError(OAuth2Error):
    error: str = "invalid_scope"


class ServerError(OAuth2Error):
    error: str = "server_error"


class TemporarilyUnavailableError(OAuth2Error):
    error: str = "temporarily_unavailable"
