import customize_motif_vec
import extract_substructure
import motif_class
import __init__
import json_utility
import glycan_profile
import plot_glycan_utilities
import matplotlib.pyplot as plt
from glypy.io import glycoct

# stats
import numpy as np
from scipy.stats import hypergeom
from scipy.stats import chi2_contingency
from scipy.stats.mstats import mquantiles
from scipy.special import comb
import pandas as pd

import statsmodels.api as sm
from statsmodels.formula.api import ols


##################################
# support functions
##################################
def get_relative(m, mh, dir='child'):
    if dir == 'child':
        return mh[mh[:, 0] == m, 1]
    elif dir == 'parent':
        return mh[mh[:, 1] == m, 0]
    else:
        assert dir in ['child', 'parent']


# def check_assumptions(d1,d2):
#     tab = table( data$go_i , data$occurance )
#     eo = apply(tab,1,function(x) apply(tab,2,function(y) (sum(x)*sum(y))/sum(tab)))
#     if( any(eo<10) ){next} # cochran(1952,1954)
# #        if( any(eo<1) | sum(eo<5)<(.2*prod(dim(tab))) ){next} # Yates, Moore & McCabe 1999 (tables larger and 2x2)
#     prop = sum(data$go_i)/nrow(data)
#     data$weights = ifelse( data$go_i==1 , 1 , prop)

def hypergeometric(N, K, n, k):
    return hypergeom.cdf(k, M=K, n=n, N=N)


def multi_hypergeometric(N, K, n, k):
    return sum([comb(Ki, ki) for Ki, ki in zip(K, k)]) / comb(N, n)


def chi_squared(M):
    chi2, p, dof, ex = chi2_contingency(M)
    if (ex < 10).any():  # cochran(1952,1954)
        Warning('there are expectated values less than 10')
    return p


#
# def relative():
#     pass
# ###################################
### Representative Motif Extraction - HIERARCHICAL - 1 glycoprofile
###################################

# Occurance based

def hier_enrich_glycoprofile_occurence(glycoprofile_list, scoredMotifs_occurence_vector, motif_hierarchy, motif_vector,
                                       method='chi_squared', relative='child'):
    """Perform Hierarchical Enrichment: Motifs occuring in a significant number of glycans using hypergeometric enrichment.
    Keyword arguments:
    glycoprofile -- list of glypy.structures.glycans
    scoredMotifs_occurence_vector -- integer vector, number of occurances of each motif in the glycoprofile (same dimesion of motif_vector)
    motif_hierarchy -- edgelist of motif membership relations (generated by motif_lib.motif_dependence_tree()), 2D table
    motif_vector -- list of motifs
    Return
    [min,Q1,median,Q3,max] p-values
    """
    # checks
    assert type(motif_hierarchy) == np.ndarray and motif_hierarchy.shape[1] == 2
    # run
    rep_motif_p = []
    # Number of marbles in the urn is the number of glycans in the glycoprofile
    urn_size = len(glycoprofile_list)
    for motif in range(len(motif_vector)):
        # get the child of this motif
        children = get_relative(motif, motif_hierarchy, dir=relative)
        child_p = []
        #         print(children)
        #         break
        for child in children:
            whiteMarbles_Urn = scoredMotifs_occurence_vector[child]  #### assumes that the index in the hierarchy matches the motif index in the occurance vector
            whiteMarbles_Select = scoredMotifs_occurence_vector[motif]
            select_size = whiteMarbles_Urn  # changed from 'urn_size' 8/7/2018
            if method == 'hypergeometric':
                child_p.append(hypergeometric(N=urn_size, K=whiteMarbles_Urn, n=select_size, k=whiteMarbles_Select))
            elif method == 'chi_squared':
                child_p.append(chi_squared([[urn_size, whiteMarbles_Urn], [select_size, whiteMarbles_Select]]))
            else:
                assert method in ['hypergeometric', 'chi_squared']
            #         print(child_p)
            # fisher_squared([[urn_size,whiteMarbles_Urn],[select_size,whiteMarbles_Select]])
        rep_motif_p.append(mquantiles(np.array(child_p), prob=[0, 0.25, 0.5, 0.75, 1]))
    return rep_motif_p


###################################
### Representative Motif Extraction - multiple glycoprofiles, group compare
###################################

# Abundance or discrete, heirarncical (anova) or flat (wald,LRT)
def enrich_glycoprofile(glycoprofile_list, condition, scoredMotifs_abundance_matrix, motif_hierarchy, motif_vector,
                        method='anova', relative='child'):
    """
    Perform Hierarchical Enrichment: Compare integer motif scores
    Keyword arguments:
    glycoprofile -- matrix of glycoprofiles
    scoredMotifs_abundance_matrix -- double matrix, motif abundance for each glycoprofile (same dimesion of motif_vector)
    motif_hierarchy -- edgelist of motif membership relations (generated by motif_lib.motif_dependence_tree())
    motif_vector -- list of motifs
    condition -- list of conditions (string or double) for each glycoprofile
    methods -- 'anova' does not consider the hierachy, direct comparison of motif association with condition. 'wald' wald
       test on motif of interest while accounting for variance better described by a parent or child (as specificed by the
       'relative' parameter),'LRT' likelihood ratio test on motif of interest while removing variance better described by a
       parent or child (as specificed by the 'relative' parameter)
    returns
        vector of p-value quartiles
        vector of effects size quartiles
        vector of coefficients: no response for anova
    """
    # checks
    assert type(motif_hierarchy) == np.ndarray and motif_hierarchy.shape[1] == 2
    # run
    rep_motif_p, rep_motif_e, rep_motif_c = [], [], []
    # rep_motif_e = []
    # Traverse the `motif hierarchy`
    for motif in motif_vector:
        children = get_relative(motif, motif_hierarchy, dir=relative)
        child_p, child_c, child_e = [], [], []
        for child in children:
            c = scoredMotifs_abundance_matrix[child,]
            m = scoredMotifs_abundance_matrix[motif,]
            df = pd.DataFrame(data={'condition': condition, 'child': c, 'motif': m})
            mod0 = ols('condition ~ child', data=df).fit()
            mod1 = ols('condition ~ motif+child', data=df).fit()
            mod = ols('condition ~ motif', data=df).fit()

            if method == 'anova':
                aov_table = sm.stats.anova_lm(mod, typ=2)
                child_p.append(aov_table['PR(>F)'][0])
                child_e.append(aov_table['sum_sq'][0] / (aov_table['sum_sq'][0] + aov_table['sum_sq'][1]))
                # next
            elif method == 'wald':
                s = mod1.summary2()
                child_p.append(s.tables[1]['P>|t|'][1])
                child_c.append(s.tables[1]['Coef.'][1])
                child_e.append(float(s.tables[0][1][6]))
            elif method == 'LRT':
                lrt = mod1.compare_lr_test(mod0)
                s = mod1.summary2()
                child_p.append(lrt[2])
                child_c.append(s.tables[1]['Coef.'][1])
                child_e.append(float(s.tables[0][1][6]))
            else:
                assert method in ['anova', 'wald', 'LRT']
                # fisher_squared([[urn_size,whiteMarbles_Urn],[select_size,whiteMarbles_Select]])
        rep_motif_p.append(mquantiles(np.array(child_p), prob=[0, 0.25, 0.5, 0.75, 1]))
        rep_motif_c.append(mquantiles(np.array(child_c), prob=[0, 0.25, 0.5, 0.75, 1]))
        rep_motif_e.append(mquantiles(np.array(child_e), prob=[0, 0.25, 0.5, 0.75, 1]))
    return np.array(rep_motif_p), np.array(rep_motif_e), np.array(rep_motif_c)

    # # ###################################
    # # ### Unique Motif Extraction         - multiple glycoprofile
    # # ###################################
    #
    # # # Occurance based
    # # def hier_enrich_glycoprofile_occurence(glycoprofile,scoredMotifs_occurence_matrix,motif_hierarchy,motif_vector,method='chi_squared',relative='child'):
    # #     """Perform Hierarchical Enrichment: Motifs occuring in a significant number of glycans using hypergeometric enrichment.
    #
    # #     Keyword arguments:
    # #     glycoprofile -- list of glycans
    # #     scoredMotifs_abundance_matrix -- integer matrix, motif abundance for each glycoprofile (same dimesion of motif_vector)
    # #     motif_hierarchy -- edgelist of motif membership relations (generated by motif_lib.motif_dependence_tree())
    # #     motif_vector -- list of motifs
    # #     """
    # #     # checks
    # #     assert type(motif_hierarchy)==np.ndarray and motif_hierarchy.shape[1]==2
    # #     # run
    # #     rep_motif_p = []
    # #     # Number of marbles in the urn is the number of glycans in the glycoprofile
    # #     urn_size = len(glycoprofile)
    # #     # Traverse the `motif hierarchy`
    # #     for motif in motif_vector:
    # #         children = relative(motif,motif_hierarchy,dir=relative)
    # #         child_p = []
    # #         for child in children:
    # #             whiteMarbles_Urn = scoredMotifs_occurence[child,]
    # #             whiteMarbles_Select = scoredMotifs_occurence[motif,]
    # #             select_size = urn_size # revisit
    # #             chi_squared([[urn_size,whiteMarbles_Urn],[select_size,whiteMarbles_Select]...2x2x2?])
    #
    #
    #
    #
    # # ################################################################################
    # # ###################################
    # # ### Representative Motif Comparison - 2 glycoprofiles
    # # ###################################
    #
    # # # Occurance based
    # # def hier_enrich_compareGlycoprofiles_occurence(scored_motif_vec1 , scored_motif_vec2 , motif_hierarchy):
    # #     """Perform Hierarchical Enrichment: Compare integer motif scores
    #
    # #     """
    #
    # # # Abundance-based
    # # def hier_enrich_compareGlycoprofiles_abundance(scored_motif_vec1 , scored_motif_vec2 , motif_hierarchy):
    #
    #
    #
    #
    # #def test():
    #     # load glycoprofiles
    # abundance_data_table = json_utility.load_json("../generated_json_file/NBT_dict_name_abundance_cross_profile.json")
    # merged_glycan_profile, _ = glycan_profile.load_glycan_profile_dic()
    # glycan_profiles_obj = glycan_profile.combine_profile_mz_with_motif_existance(merged_glycan_profile,abundance_data_table)
    # motif_abd_table = glycan_profiles_obj.table_against_wt_relative_abd()
    # motif_abd_table.head()
    #
    #     # load motif vector and return edge_list
    # # motif_vector = json_utility.load_json("../generated_json_file/Unicarbkb_motif_vec_12259.json")
    # # motif_lib = gc_glycan_motif.GlycanMotifLib(motif_dict)
    # motif_lib = motif_class.GlycanMotifLib(json_utility.load_json(__init__.output_motif_dic_degree_list_addr)) # unicarbkb_motifs_12259.json
    # tree_type_dp, edge_list = motif_lib.motif_dependence_tree()
    #
    # # representative motif extraction from 1 glycoprofile
    # hier_enrich_glycoprofile_occurence(glycoprofile,scoredMotifs_occurence_vector,
    #     np.array(edge_list),motif_vector)
