#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
This class is used to construct the list used to execute the MKVMerge
command line

strCommand = Command obtained from mkvtoolnix-gui with the modifications
    needed for Multiplexing a series in a directory

path for executable and target options are parsed from the command line

MC018
"""

import os
import re
import shlex
import platform
import logging

from pathlib import Path


MODULELOG = logging.getLogger(__name__)
MODULELOG.addHandler(logging.NullHandler())


class MKVCommand(object):
    """
    Class to work with mkvmerge command

    :param strCommand: command line as generated by mkvtoolnix-gui
    :type strCommand: str
    :param bRemoveTitle: remove title from command
    :type bRemoveTitle: bool
    """

    log = False

    def __init__(self, strCommand="", bRemoveTitle=True):

        self.strShellcommand = ""
        self.objDestinationFile = None
        self.lstObjFiles = []
        self.lstCommandTemplate = []
        self.lstProcessCommand = []
        self._bInconsistent = False
        self.strError = ""
        self.bRaiseError = False

        self._initHelper(strCommand, bRemoveTitle)

    def _initHelper(self, strCommand, bRemoveTitle=True):

        if strCommand == "":

            self._reset()

        else:

            self.strShellcommand = strCommand
            self._bInconsistent = False
            self.strError = ""

            if MKVCommand.bLooksOk(strCommand):

                #Substitute file names for shlex help with apostrophe
                lstFileNames = []
                dictFileNames = {}
                dictFileTokens = {}
                lstParsed = []

                strCommand = _strStripEscapeChars(strCommand)

                _parseSourceFiles(strCommand, lstFileNames,
                                  dictFileNames, dictFileTokens, MKVCommand.log)

                if lstFileNames:

                    for strFile in lstFileNames:
                        strCommand = strCommand.replace(
                            strFile, dictFileTokens[strFile])

                    lstParsed = shlex.split(strCommand)

                    for t in enumerate(lstParsed):
                        if lstParsed[t[0]] in dictFileNames:
                            lstParsed[t[0]] = dictFileNames[t[1]]

                    self.objDestinationFile = MKVSourceFile(
                        lstParsed.index("--output") + 1,
                        lstParsed[lstParsed.index("--output") + 1]
                    )

                else:
                    self._bInconsistent = True
                    self.strError = "No output file found."

                # Windows present inconsistent use of
                # forward and backward slash fix for
                # windows
                #ft = Path(lstParsed[0])
                #lstParsed[0] = str(ft)

                if bRemoveTitle and lstParsed:
                    #Remove title if found since this is for batch processing
                    #the title will propagate to all the files maybe erroneously.
                    #This parameters are preserved from the source files.

                    while "--title" in lstParsed:

                        i = lstParsed.index("--title")
                        del lstParsed[i:i+2]

                # Get the index of files surrounded by parenthesis these
                # are source files there have to be at least one
                lstIndices = [i for i, x in enumerate(lstParsed) if x == "("]

                # Store source files in list
                j = 0
                for i in lstIndices:
                    self.lstObjFiles.append(
                        MKVSourceFile(i - (j * 2),
                                      lstParsed[i + 1])
                    )
                    j += 1

                # Remove parenthesis it does not work for the subprocess
                # execution
                lstParsed = [
                    i for i in lstParsed if (i != "(") and (i != ")")
                ]

                self.lstCommandTemplate.extend(lstParsed)

                self.lstCommandTemplate[self.objDestinationFile.index] = ""

                for objFile in self.lstObjFiles:

                    self.lstCommandTemplate[objFile.index] = ""

            else:

                self._bInconsistent = True
                self.strError = "Error parsing command line."

            if self._bInconsistent:
                self.lstCommandTemplate = []
                self.lstProcessCommand = []
                if MKVCommand.log:
                    MODULELOG.error("MC001: ".join(self.strError))

    def _reset(self):
        """Reset variable properties"""
        self.strShellcommand = ""
        self.objDestinationFile = None
        self.lstObjFiles = []
        self.lstCommandTemplate = []
        self.lstProcessCommand = []
        self._bInconsistent = False
        self.bRaiseError = False
        self.strError = ""

    def __len__(self):
        return len(self.lstProcessCommand)

    def __bool__(self):
        """
        if self._bInconsistent:
            return False
        if self.strShellcommand == "":
            return False
        return True
        """
        return not (self._bInconsistent or (self.strShellcommand == "") or self.bRaiseError)

    @property
    def command(self):
        """
        property command produced by mkvtoolnix-gui can be set
        """
        return self.strShellcommand

    @command.setter
    def command(self, value):
        """Update command through property"""
        if isinstance(value, str):
            self._reset()
            self._initHelper(value)

    @staticmethod
    def bLooksOk(strCommand):
        """
        Rudimentary sanity check any failure results in no action whatsoever
        and since the original are not modified the resulting command should be safe

        :param strCommand: command line as generated by mkvtoolnix-gui
        :type strCommand: str
        :rtype: bool
        """
        # For Windows and linux
        rg = r"^'?(.*?)'?\s.*?\-\-output.'(.*?)'\s.*?\s'\('\s'(.*?)'\s'\)\'.*?\-\-track-order\s(.*)"  # pylint: disable=C0301
        #     r"^'?(.*?)'?.*?\-\-output.'(.*?)'\s.*?\s'\('\s'(.*?)'\s'\)\'.*?\-\-track-order\s(.*)")
        regEx = re.compile(rg)

        bOk = False

        outputIndex = strCommand.find(r"--output")
        strTmp = _strStripEscapeChars(strCommand)  # Comvert line to bash style

        matchOptions = regEx.match(strTmp)

        # To look Ok must match the 5 group in the command line that
        # are expected
        # 1: mkvmerge name with fullpath
        # 2: output file
        # 3: first or only source file
        # 4: track order
        if matchOptions and \
            (len(matchOptions.groups()) == 4) and \
                (outputIndex > 0):

            bOk = True

            if MKVCommand.log:

                MODULELOG.debug("MC002: Command to match [ %s ]", strCommand)
                MODULELOG.debug(
                    "MC003: Total matche(s) %d of 4 %s",
                    len(matchOptions.groups()),
                    matchOptions.groups()
                )
                for i in range(len(matchOptions.groups()) + 1):
                    MODULELOG.debug("MC004: Match (%d) %s", i,
                                    matchOptions.group(i))

        return bOk

    def setFiles(self, lstFiles, strPrefix="new-"):
        """
        Substitute file names on command template

        :param lstFiles: list of files in command line to process
        :type lstFiles: list
        :param strPrefix: prefix for output files
        :type strPrefix: str
        """

        if not self._bInconsistent:

            self.lstProcessCommand = self.lstCommandTemplate

            _, filename = os.path.split(lstFiles[0])

            strTmp = os.path.join(
                self.objDestinationFile.directory,
                os.path.splitext(filename)[0] +
                self.objDestinationFile.extension
            )

            # Check if destination file exist and add prefix if it does
            if os.path.isfile(strTmp):
                strSuffix = ""
                n = 1
                while True:
                    strDestinationFile = os.path.join(
                        self.objDestinationFile.directory,
                        strPrefix
                        + os.path.splitext(filename)[0]
                        + strSuffix
                        + self.objDestinationFile.extension
                    )

                    if os.path.isfile(strDestinationFile):
                        strSuffix = " (%d)" % n
                        n += 1
                    else:
                        break
            else:
                strDestinationFile = strTmp

            self.lstProcessCommand[self.objDestinationFile.index] = strDestinationFile

            for objFile, strFile in zip(self.lstObjFiles, lstFiles):
                self.lstProcessCommand[objFile.index] = strFile

        else:

            self.lstProcessCommand = []

    def strCmdSourceFile(self):
        """First source file fullpath name"""

        return self.lstObjFiles[0].fullPathName if self.lstObjFiles else None

    def strCmdSourceDirectory(self):
        """First source file directory"""
        return self.lstObjFiles[0].directory if self.lstObjFiles else None

    def strCmdSourceExtension(self):
        """First source file extension"""
        return self.lstObjFiles[0].extension if self.lstObjFiles else None


class MKVSourceFile(object):
    """
    Source file properties

    :param index: index of file in the command line
    :type index: int
    :param fullPathName: filename with full path
    :type fullPathName: str
    """


    def __init__(self, index, fullPathName):
        """Use path to convert to right path OS independent"""
        fileName = Path(fullPathName)

        self.index = index
        self.fullPathName = str(fileName)
        self.directory = str(fileName.parent)
        self.extension = str(fileName.suffix)

    def __str__(self):
        return "Index: " + str(self.index) \
            + "\nName: " + self.fullPathName \
            + "\nDirectory: " + self.directory \
            + "\nExtension: " + self.extension \
            + "\n"


def _strStripEscapeChars(strCommand):
    """
    Strip escape chars for the command line in the end they won't be used in a shell
    the resulting command string should work for Windows and linux
    """

    strTmp = strCommand

    if strTmp.find(r'^"^(^"') > 0:
        # This is for cmd in Windows
        strTmp = strTmp.replace('^', '').replace('/', '\\').replace('"', "'")

    return strTmp

def _parseSourceFiles(strCommand, lstFileNames, dictFileNames, dictFileTokens, log=False):
    """
    In order to work with apostrophe in file names
    substitute the file names with tokens for shlex
    here parse files and create token dictionaries
    """

    regExOutput = re.compile(r".*?--output\s'(.*?)'\s--.*")
    regExSource = re.compile(r"'\('\s'(.*?)'\s'\)'")
    regExAttachFile = re.compile(r"--attach-file\s'(.*?)'\s--")
    regExTitle = re.compile(r"--title\s'(.*?)'\s--")

    fileMatch = regExOutput.match(strCommand)

    # Required Files
    if fileMatch:
        lstFileNames.append(fileMatch.group(1))
    else:
        lstFileNames = []
        return

    fileMatch = regExSource.finditer(strCommand)

    if fileMatch:

        for match in fileMatch:
            lstFileNames.append(match.group(1))
    else:
        lstFileNames = []
        return

    # Optional Files
    fileMatch = regExAttachFile.finditer(strCommand)

    for match in fileMatch:
        lstFileNames.append(match.group(1))

    fileMatch = regExTitle.finditer(strCommand)

    for match in fileMatch:
        lstFileNames.append(match.group(1))

    if lstFileNames:
        i = 0
        for strFile in lstFileNames:
            fileToken = "TOKEN!!!!!FILE{}".format(i)
            dictFileNames[fileToken] = strFile
            dictFileTokens[strFile] = fileToken
            i += 1
            if log:
                MODULELOG.debug(
                    "MC005: Token  %s - %s",
                    fileToken,
                    strFile
                )
