/*
 * Copyright 2018-2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import '../style/index.css';
import { ILayoutRestorer } from '@jupyterlab/application';
import { WidgetTracker, ICommandPalette } from '@jupyterlab/apputils';
import { CodeEditor, IEditorServices } from '@jupyterlab/codeeditor';
import { IFileBrowserFactory } from '@jupyterlab/filebrowser';
import { IEditorTracker } from '@jupyterlab/fileeditor';
import { ILauncher } from '@jupyterlab/launcher';
import { IMainMenu } from '@jupyterlab/mainmenu';
import { ISettingRegistry } from '@jupyterlab/settingregistry';
import { pythonIcon } from '@jupyterlab/ui-components';
import { PythonFileEditorFactory } from './PythonFileEditor';
const PYTHON_FACTORY = 'PyEditor';
const PYTHON = 'python';
const PYTHON_EDITOR_NAMESPACE = 'elyra-python-runner-extension';
const commandIDs = {
    createNewPython: 'pyeditor:create-new-python-file',
    openPyEditor: 'pyeditor:open',
    openDocManager: 'docmanager:open',
    newDocManager: 'docmanager:new-untitled'
};
/**
 * Initialization data for the python-editor-extension extension.
 */
const extension = {
    id: PYTHON_EDITOR_NAMESPACE,
    autoStart: true,
    requires: [
        IEditorServices,
        IEditorTracker,
        ICommandPalette,
        ISettingRegistry,
        IFileBrowserFactory
    ],
    optional: [ILayoutRestorer, IMainMenu, ILauncher],
    activate: (app, editorServices, editorTracker, palette, settingRegistry, browserFactory, restorer, menu, launcher) => {
        console.log('Elyra - python-runner extension is activated!');
        const factory = new PythonFileEditorFactory({
            editorServices,
            factoryOptions: {
                name: PYTHON_FACTORY,
                fileTypes: [PYTHON],
                defaultFor: [PYTHON]
            }
        });
        const { restored } = app;
        /**
         * Track PythonFileEditor widget on page refresh
         */
        const tracker = new WidgetTracker({
            namespace: PYTHON_EDITOR_NAMESPACE
        });
        let config = Object.assign({}, CodeEditor.defaultConfig);
        if (restorer) {
            // Handle state restoration
            void restorer.restore(tracker, {
                command: commandIDs.openDocManager,
                args: widget => ({
                    path: widget.context.path,
                    factory: PYTHON_FACTORY
                }),
                name: widget => widget.context.path
            });
        }
        /**
         * Update the setting values. Adapted from fileeditor-extension.
         */
        const updateSettings = (settings) => {
            config = Object.assign(Object.assign({}, CodeEditor.defaultConfig), settings.get('editorConfig').composite);
            // Trigger a refresh of the rendered commands
            app.commands.notifyCommandChanged();
        };
        /**
         * Update the settings of the current tracker instances. Adapted from fileeditor-extension.
         */
        const updateTracker = () => {
            tracker.forEach(widget => {
                updateWidget(widget);
            });
        };
        /**
         * Update the settings of a widget. Adapted from fileeditor-extension.
         */
        const updateWidget = (widget) => {
            if (!editorTracker.has(widget)) {
                editorTracker.add(widget);
            }
            const editor = widget.content.editor;
            Object.keys(config).forEach((keyStr) => {
                const key = keyStr;
                editor.setOption(key, config[key]);
            });
        };
        // Fetch the initial state of the settings. Adapted from fileeditor-extension.
        Promise.all([
            settingRegistry.load('@jupyterlab/fileeditor-extension:plugin'),
            restored
        ])
            .then(([settings]) => {
            updateSettings(settings);
            updateTracker();
            settings.changed.connect(() => {
                updateSettings(settings);
                updateTracker();
            });
        })
            .catch((reason) => {
            console.error(reason.message);
            updateTracker();
        });
        app.docRegistry.addWidgetFactory(factory);
        factory.widgetCreated.connect((sender, widget) => {
            void tracker.add(widget);
            // Notify the widget tracker if restore data needs to update
            widget.context.pathChanged.connect(() => {
                void tracker.save(widget);
            });
            updateWidget(widget);
        });
        // Handle the settings of new widgets. Adapted from fileeditor-extension.
        tracker.widgetAdded.connect((sender, widget) => {
            updateWidget(widget);
        });
        /**
         * Create new python file from launcher and file menu
         */
        // Add a python launcher
        if (launcher) {
            launcher.add({
                command: commandIDs.createNewPython,
                category: 'Other',
                rank: 3
            });
        }
        if (menu) {
            // Add new python file creation to the file menu
            menu.fileMenu.newMenu.addGroup([{ command: commandIDs.createNewPython }], 30);
        }
        // Function to create a new untitled python file, given the current working directory
        const createNew = (cwd) => {
            return app.commands
                .execute(commandIDs.newDocManager, {
                path: cwd,
                type: 'file',
                ext: '.py'
            })
                .then(model => {
                return app.commands.execute(commandIDs.openDocManager, {
                    path: model.path,
                    factory: PYTHON_FACTORY
                });
            });
        };
        // Add a command to create new Python file
        app.commands.addCommand(commandIDs.createNewPython, {
            label: args => (args['isPalette'] ? 'New Python File' : 'Python File'),
            caption: 'Create a new python file',
            icon: args => (args['isPalette'] ? undefined : pythonIcon),
            execute: args => {
                const cwd = args['cwd'] || browserFactory.defaultBrowser.model.path;
                return createNew(cwd);
            }
        });
        palette.addItem({
            command: commandIDs.createNewPython,
            args: { isPalette: true },
            category: 'Python Editor'
        });
    }
};
export default extension;
//# sourceMappingURL=index.js.map