"use strict";
/*
 * Copyright 2018-2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const path = __importStar(require("path"));
const application_1 = require("@elyra/application");
const canvas_1 = require("@elyra/canvas");
const ui_components_1 = require("@elyra/ui-components");
const apputils_1 = require("@jupyterlab/apputils");
const docregistry_1 = require("@jupyterlab/docregistry");
const ui_components_2 = require("@jupyterlab/ui-components");
const algorithm_1 = require("@lumino/algorithm");
require("@elyra/canvas/dist/common-canvas.min.css");
require("@elyra/canvas/dist/common-canvas.min.css");
require("carbon-components/css/carbon-components.min.css");
const React = __importStar(require("react"));
const react_intl_1 = require("react-intl");
const i18nData = __importStar(require("./en.json"));
const palette = __importStar(require("./palette.json"));
const PipelineExportDialog_1 = require("./PipelineExportDialog");
const PipelineService_1 = require("./PipelineService");
const PipelineSubmissionDialog_1 = require("./PipelineSubmissionDialog");
const properties = __importStar(require("./properties.json"));
const utils_1 = __importDefault(require("./utils"));
const PIPELINE_CLASS = 'elyra-PipelineEditor';
const NODE_TOOLTIP_CLASS = 'elyra-PipelineNodeTooltip';
const TIP_TYPE_NODE = 'tipTypeNode';
const PIPELINE_CURRENT_VERSION = 1;
const NodeProperties = (properties) => {
    return (React.createElement("dl", { className: NODE_TOOLTIP_CLASS }, Object.keys(properties).map((key, idx) => {
        let value = properties[key];
        if (Array.isArray(value)) {
            value = value.join('\n');
        }
        else if (typeof value === 'boolean') {
            value = value ? 'Yes' : 'No';
        }
        return (React.createElement(React.Fragment, { key: idx },
            React.createElement("dd", null, key),
            React.createElement("dt", null, value)));
    })));
};
exports.commandIDs = {
    openPipelineEditor: 'pipeline-editor:open',
    openDocManager: 'docmanager:open',
    newDocManager: 'docmanager:new-untitled',
    submitNotebook: 'notebook:submit'
};
/**
 * Wrapper Class for Common Canvas React Component
 */
class PipelineEditorWidget extends apputils_1.ReactWidget {
    constructor(props) {
        super(props);
        this.app = props.app;
        this.browserFactory = props.browserFactory;
        this.context = props.context;
    }
    render() {
        return (React.createElement(PipelineEditor, { app: this.app, browserFactory: this.browserFactory, widgetContext: this.context }));
    }
}
exports.PipelineEditorWidget = PipelineEditorWidget;
/**
 * Class for Common Canvas React Component
 */
class PipelineEditor extends React.Component {
    constructor(props) {
        super(props);
        this.position = 10;
        this.app = props.app;
        this.browserFactory = props.browserFactory;
        this.canvasController = new canvas_1.CanvasController();
        this.canvasController.setPipelineFlowPalette(palette);
        this.widgetContext = props.widgetContext;
        this.toolbarMenuActionHandler = this.toolbarMenuActionHandler.bind(this);
        this.contextMenuHandler = this.contextMenuHandler.bind(this);
        this.contextMenuActionHandler = this.contextMenuActionHandler.bind(this);
        this.clickActionHandler = this.clickActionHandler.bind(this);
        this.editActionHandler = this.editActionHandler.bind(this);
        this.tipHandler = this.tipHandler.bind(this);
        this.state = { showPropertiesDialog: false, propertiesInfo: {} };
        this.initPropertiesInfo();
        this.applyPropertyChanges = this.applyPropertyChanges.bind(this);
        this.closePropertiesDialog = this.closePropertiesDialog.bind(this);
        this.openPropertiesDialog = this.openPropertiesDialog.bind(this);
        this.node = React.createRef();
        this.handleEvent = this.handleEvent.bind(this);
    }
    render() {
        const style = { height: '100%' };
        const emptyCanvasContent = (React.createElement("div", null,
            React.createElement(ui_components_1.dragDropIcon.react, { tag: "div", elementPosition: "center", height: "120px" }),
            React.createElement("h1", null,
                ' ',
                "Start your new pipeline by dragging files from the file browser pane.",
                ' ')));
        const canvasConfig = {
            enableInternalObjectModel: true,
            emptyCanvasContent: emptyCanvasContent,
            enablePaletteLayout: 'Modal',
            paletteInitialState: false
        };
        const toolbarConfig = [
            { action: 'run', label: 'Run Pipeline', enable: true },
            {
                action: 'save',
                label: 'Save Pipeline',
                enable: true,
                iconEnabled: ui_components_1.IconUtil.encode(ui_components_1.savePipelineIcon),
                iconDisabled: ui_components_1.IconUtil.encode(ui_components_1.savePipelineIcon)
            },
            {
                action: 'export',
                label: 'Export Pipeline',
                enable: true,
                iconEnabled: ui_components_1.IconUtil.encode(ui_components_1.exportPipelineIcon),
                iconDisabled: ui_components_1.IconUtil.encode(ui_components_1.exportPipelineIcon)
            },
            {
                action: 'clear',
                label: 'Clear Pipeline',
                enable: true,
                iconEnabled: ui_components_1.IconUtil.encode(ui_components_1.clearPipelineIcon),
                iconDisabled: ui_components_1.IconUtil.encode(ui_components_1.clearPipelineIcon)
            },
            { divider: true },
            { action: 'undo', label: 'Undo', enable: true },
            { action: 'redo', label: 'Redo', enable: true },
            { action: 'cut', label: 'Cut', enable: false },
            { action: 'copy', label: 'Copy', enable: false },
            { action: 'paste', label: 'Paste', enable: false },
            { action: 'addComment', label: 'Add Comment', enable: true },
            { action: 'delete', label: 'Delete', enable: true },
            {
                action: 'arrangeHorizontally',
                label: 'Arrange Horizontally',
                enable: true
            },
            { action: 'arrangeVertically', label: 'Arrange Vertically', enable: true }
        ];
        const propertiesCallbacks = {
            applyPropertyChanges: this.applyPropertyChanges,
            closePropertiesDialog: this.closePropertiesDialog
        };
        const commProps = this.state.showPropertiesDialog ? (React.createElement(react_intl_1.IntlProvider, { key: "IntlProvider2", locale: 'en', messages: i18nData.messages },
            React.createElement(canvas_1.CommonProperties, { propertiesInfo: this.propertiesInfo, propertiesConfig: {}, callbacks: propertiesCallbacks }))) : null;
        return (React.createElement("div", { style: style, ref: this.node },
            React.createElement(canvas_1.CommonCanvas, { canvasController: this.canvasController, toolbarMenuActionHandler: this.toolbarMenuActionHandler, contextMenuHandler: this.contextMenuHandler, contextMenuActionHandler: this.contextMenuActionHandler, clickActionHandler: this.clickActionHandler, editActionHandler: this.editActionHandler, tipHandler: this.tipHandler, toolbarConfig: toolbarConfig, config: canvasConfig }),
            commProps));
    }
    updateModel() {
        this.widgetContext.model.fromString(JSON.stringify(this.canvasController.getPipelineFlow(), null, 2));
    }
    initPropertiesInfo() {
        return __awaiter(this, void 0, void 0, function* () {
            const runtimeImages = yield PipelineService_1.PipelineService.getRuntimeImages();
            const imageEnum = [];
            for (const runtimeImage in runtimeImages) {
                imageEnum.push(runtimeImage);
                properties.resources['runtime_image.' + runtimeImage + '.label'] = runtimeImages[runtimeImage];
            }
            properties.parameters[1].enum = imageEnum;
            this.propertiesInfo = {
                parameterDef: properties,
                appData: { id: '' }
            };
        });
    }
    openPropertiesDialog(source) {
        console.log('Opening properties dialog');
        const node_id = source.targetObject.id;
        const app_data = this.canvasController.getNode(node_id).app_data;
        const node_props = this.propertiesInfo;
        node_props.appData.id = node_id;
        node_props.parameterDef.current_parameters.filename = app_data.filename;
        node_props.parameterDef.current_parameters.runtime_image =
            app_data.runtime_image;
        node_props.parameterDef.current_parameters.outputs = app_data.outputs;
        node_props.parameterDef.current_parameters.env_vars = app_data.env_vars;
        node_props.parameterDef.current_parameters.dependencies =
            app_data.dependencies;
        node_props.parameterDef.current_parameters.include_subdirectories =
            app_data.include_subdirectories;
        this.setState({ showPropertiesDialog: true, propertiesInfo: node_props });
    }
    applyPropertyChanges(propertySet, appData) {
        console.log('Applying changes to properties');
        const app_data = this.canvasController.getNode(appData.id).app_data;
        app_data.runtime_image = propertySet.runtime_image;
        app_data.outputs = propertySet.outputs;
        app_data.env_vars = propertySet.env_vars;
        app_data.dependencies = propertySet.dependencies;
        app_data.include_subdirectories = propertySet.include_subdirectories;
    }
    closePropertiesDialog() {
        console.log('Closing properties dialog');
        this.setState({ showPropertiesDialog: false, propertiesInfo: {} });
    }
    /*
     * Add options to the node context menu
     * Pipeline specific context menu items are:
     *  - Enable opening selected notebook(s)
     *  - Enable node properties for single node
     */
    contextMenuHandler(source, defaultMenu) {
        let customMenu = defaultMenu;
        // Remove option to create super node
        customMenu.splice(4, 2);
        if (source.type === 'node') {
            if (source.selectedObjectIds.length > 1) {
                // multiple nodes selected
                customMenu = customMenu.concat({
                    action: 'openNotebook',
                    label: 'Open Notebooks'
                });
            }
            else {
                // single node selected
                customMenu = customMenu.concat({
                    action: 'openNotebook',
                    label: 'Open Notebook'
                }, {
                    action: 'properties',
                    label: 'Properties'
                });
            }
        }
        return customMenu;
    }
    /*
     * Handles context menu actions
     * Pipeline specific actions are:
     *  - Open the associated Notebook
     *  - Open node properties dialog
     */
    contextMenuActionHandler(action, source) {
        if (action === 'openNotebook' && source.type === 'node') {
            this.handleOpenNotebook(source.selectedObjectIds);
        }
        else if (action === 'properties' && source.type === 'node') {
            if (this.state.showPropertiesDialog) {
                this.closePropertiesDialog();
            }
            else {
                this.openPropertiesDialog(source);
            }
        }
    }
    /*
     * Handles mouse click actions
     */
    clickActionHandler(source) {
        // opens the Jupyter Notebook associated with a given node
        if (source.clickType === 'DOUBLE_CLICK' && source.objectType === 'node') {
            this.handleOpenNotebook(source.selectedObjectIds);
        }
    }
    /*
     * Handles creating new nodes in the canvas
     */
    editActionHandler(data) {
        this.updateModel();
    }
    /*
     * Handles displaying node properties
     */
    tipHandler(tipType, data) {
        if (tipType === TIP_TYPE_NODE) {
            const appData = this.canvasController.getNode(data.node.id).app_data;
            const propsInfo = this.propertiesInfo.parameterDef.uihints.parameter_info;
            const tooltipProps = {};
            propsInfo.forEach((info) => {
                if (Object.prototype.hasOwnProperty.call(appData, info.parameter_ref)) {
                    tooltipProps[info.label.default] = appData[info.parameter_ref];
                }
            });
            return React.createElement(NodeProperties, Object.assign({}, tooltipProps));
        }
    }
    handleAddFileToPipelineCanvas(x, y) {
        let failedAdd = 0;
        let position = 0;
        const missingXY = !(x && y);
        // if either x or y is undefined use the default coordinates
        if (missingXY) {
            position = this.position;
            x = 75;
            y = 85;
        }
        const fileBrowser = this.browserFactory.defaultBrowser;
        algorithm_1.toArray(fileBrowser.selectedItems()).map(item => {
            // if the selected item is a notebook file
            if (item.type == 'notebook') {
                //add each selected notebook
                console.log('Adding ==> ' + item.path);
                const nodeTemplate = this.canvasController.getPaletteNode('execute-notebook-node');
                if (nodeTemplate) {
                    const data = {
                        editType: 'createNode',
                        offsetX: x + position,
                        offsetY: y + position,
                        nodeTemplate: this.canvasController.convertNodeTemplate(nodeTemplate)
                    };
                    // create a notebook widget to get a string with the node content then dispose of it
                    const notebookWidget = fileBrowser.model.manager.open(item.path);
                    const notebookStr = notebookWidget.content.model.toString();
                    notebookWidget.dispose();
                    const env_vars = application_1.NotebookParser.getEnvVars(notebookStr).map(str => str + '=');
                    data.nodeTemplate.label = item.path.replace(/^.*[\\/]/, '');
                    data.nodeTemplate.label = data.nodeTemplate.label.replace(/\.[^/.]+$/, '');
                    data.nodeTemplate.image = ui_components_1.IconUtil.encode(ui_components_2.notebookIcon);
                    data.nodeTemplate.app_data['filename'] = item.path;
                    data.nodeTemplate.app_data['runtime_image'] = this.propertiesInfo.parameterDef.current_parameters.runtime_image;
                    data.nodeTemplate.app_data['env_vars'] = env_vars;
                    data.nodeTemplate.app_data['include_subdirectories'] = this.propertiesInfo.parameterDef.current_parameters.include_subdirectories;
                    this.canvasController.editActionHandler(data);
                    position += 20;
                }
            }
            else {
                failedAdd++;
            }
        });
        // update position if the default coordinates were used
        if (missingXY) {
            this.position = position;
        }
        if (failedAdd) {
            return apputils_1.showDialog({
                title: 'Unsupported File(s)',
                body: 'Currently, only selected notebook files can be added to a pipeline',
                buttons: [apputils_1.Dialog.okButton()]
            });
        }
    }
    /*
     * Open node associated notebook
     */
    handleOpenNotebook(selectedNodes) {
        for (let i = 0; i < selectedNodes.length; i++) {
            const path = this.canvasController.getNode(selectedNodes[i]).app_data
                .filename;
            this.app.commands.execute(exports.commandIDs.openDocManager, { path });
        }
    }
    handleExportPipeline() {
        return __awaiter(this, void 0, void 0, function* () {
            const runtimes = yield PipelineService_1.PipelineService.getRuntimes();
            apputils_1.showDialog({
                title: 'Export pipeline',
                body: new PipelineExportDialog_1.PipelineExportDialog({ runtimes }),
                buttons: [apputils_1.Dialog.cancelButton(), apputils_1.Dialog.okButton()],
                focusNodeSelector: '#runtime_config'
            }).then(result => {
                if (result.value == null) {
                    // When Cancel is clicked on the dialog, just return
                    return;
                }
                // prepare pipeline submission details
                const pipelineFlow = this.canvasController.getPipelineFlow();
                const pipeline_path = this.widgetContext.path;
                const pipeline_dir = path.dirname(pipeline_path);
                const pipeline_name = path.basename(pipeline_path, path.extname(pipeline_path));
                const pipeline_export_format = result.value.pipeline_filetype;
                const pipeline_export_path = pipeline_dir + '/' + pipeline_name + '.' + pipeline_export_format;
                const overwrite = result.value.overwrite;
                pipelineFlow.pipelines[0]['app_data']['name'] = pipeline_name;
                pipelineFlow.pipelines[0]['app_data']['runtime'] = 'kfp';
                pipelineFlow.pipelines[0]['app_data']['runtime-config'] =
                    result.value.runtime_config;
                PipelineService_1.PipelineService.exportPipeline(pipelineFlow, pipeline_export_format, pipeline_export_path, overwrite);
            });
        });
    }
    handleOpenPipeline() {
        return __awaiter(this, void 0, void 0, function* () {
            this.widgetContext.ready.then(() => {
                let pipelineJson = this.widgetContext.model.toJSON();
                const pipelineVersion = +utils_1.default.getPipelineVersion(pipelineJson);
                if (pipelineVersion !== PIPELINE_CURRENT_VERSION) {
                    // pipeline version and current version are divergent
                    if (pipelineVersion > PIPELINE_CURRENT_VERSION) {
                        // in this case, pipeline was last edited in a "more recent release" and
                        // the user should update his version of Elyra to consume the pipeline
                        apputils_1.showDialog({
                            title: 'Load pipeline failed!',
                            body: (React.createElement("p", null, "This pipeline corresponds to a more recent version of Elyra and cannot be used until Elyra has been upgraded.")),
                            buttons: [apputils_1.Dialog.okButton()]
                        });
                        this.handleClosePipeline();
                        return;
                    }
                    else {
                        // in this case, pipeline was last edited in a "old" version of Elyra and
                        // it needs to be updated/migrated.
                        apputils_1.showDialog({
                            title: 'Migrate pipeline?',
                            body: (React.createElement("p", null,
                                "This pipeline corresponds to an older version of Elyra and needs to be migrated.",
                                React.createElement("br", null),
                                "Although the pipeline can be further edited and/or submitted after its update,",
                                React.createElement("br", null),
                                "the migration will not be completed until the pipeline has been saved within the editor.",
                                React.createElement("br", null),
                                React.createElement("br", null),
                                "Proceed with migration?")),
                            buttons: [apputils_1.Dialog.cancelButton(), apputils_1.Dialog.okButton()]
                        }).then(result => {
                            if (result.button.accept) {
                                // proceed with migration
                                pipelineJson = PipelineService_1.PipelineService.convertPipeline(pipelineJson);
                                this.canvasController.setPipelineFlow(pipelineJson);
                            }
                            else {
                                this.handleClosePipeline();
                            }
                        });
                    }
                }
                else {
                    // in this case, pipeline version is current
                    this.canvasController.setPipelineFlow(pipelineJson);
                }
            });
        });
    }
    handleRunPipeline() {
        return __awaiter(this, void 0, void 0, function* () {
            const runtimes = yield PipelineService_1.PipelineService.getRuntimes();
            apputils_1.showDialog({
                title: 'Run pipeline',
                body: new PipelineSubmissionDialog_1.PipelineSubmissionDialog({ runtimes }),
                buttons: [apputils_1.Dialog.cancelButton(), apputils_1.Dialog.okButton()],
                focusNodeSelector: '#pipeline_name'
            }).then(result => {
                if (result.value == null) {
                    // When Cancel is clicked on the dialog, just return
                    return;
                }
                // prepare pipeline submission details
                const pipelineFlow = this.canvasController.getPipelineFlow();
                pipelineFlow.pipelines[0]['app_data']['name'] =
                    result.value.pipeline_name;
                // TODO: Be more flexible and remove hardcoded runtime type
                pipelineFlow.pipelines[0]['app_data']['runtime'] = 'kfp';
                pipelineFlow.pipelines[0]['app_data']['runtime-config'] =
                    result.value.runtime_config;
                PipelineService_1.PipelineService.submitPipeline(pipelineFlow, result.value.runtime_config);
            });
        });
    }
    handleSavePipeline() {
        this.updateModel();
        this.widgetContext.save();
    }
    handleClearPipeline() {
        return apputils_1.showDialog({
            title: 'Clear Pipeline?',
            body: 'Are you sure you want to clear? You can not undo this.',
            buttons: [apputils_1.Dialog.cancelButton(), apputils_1.Dialog.okButton({ label: 'Clear' })]
        }).then(result => {
            if (result.button.accept) {
                this.canvasController.clearPipelineFlow();
                this.updateModel();
                this.position = 10;
            }
        });
    }
    handleClosePipeline() {
        if (this.app.shell.currentWidget) {
            this.app.shell.currentWidget.close();
        }
    }
    /**
     * Handles submitting pipeline runs
     */
    toolbarMenuActionHandler(action, source) {
        console.log('Handling action: ' + action);
        if (action == 'run') {
            // When executing the pipeline
            this.handleRunPipeline();
        }
        else if (action == 'export') {
            this.handleExportPipeline();
        }
        else if (action == 'save') {
            this.handleSavePipeline();
        }
        else if (action == 'clear') {
            this.handleClearPipeline();
        }
    }
    componentDidMount() {
        const node = this.node.current;
        node.addEventListener('dragenter', this.handleEvent);
        node.addEventListener('dragover', this.handleEvent);
        node.addEventListener('lm-dragenter', this.handleEvent);
        node.addEventListener('lm-dragover', this.handleEvent);
        node.addEventListener('lm-drop', this.handleEvent);
        this.handleOpenPipeline();
    }
    componentWillUnmount() {
        const node = this.node.current;
        node.removeEventListener('lm-drop', this.handleEvent);
        node.removeEventListener('lm-dragover', this.handleEvent);
        node.removeEventListener('lm-dragenter', this.handleEvent);
        node.removeEventListener('dragover', this.handleEvent);
        node.removeEventListener('dragenter', this.handleEvent);
    }
    /**
     * Handle the DOM events.
     *
     * @param event - The DOM event.
     */
    handleEvent(event) {
        switch (event.type) {
            case 'dragenter':
                event.preventDefault();
                break;
            case 'dragover':
                event.preventDefault();
                break;
            case 'lm-dragenter':
                event.preventDefault();
                break;
            case 'lm-dragover':
                event.preventDefault();
                event.stopPropagation();
                event.dropAction = event.proposedAction;
                break;
            case 'lm-drop':
                event.preventDefault();
                event.stopPropagation();
                this.handleAddFileToPipelineCanvas(event.offsetX, event.offsetY);
                break;
            default:
                break;
        }
    }
}
exports.PipelineEditor = PipelineEditor;
class PipelineEditorFactory extends docregistry_1.ABCWidgetFactory {
    constructor(options) {
        super(options);
        this.app = options.app;
        this.browserFactory = options.browserFactory;
    }
    createNewWidget(context) {
        // Creates a blank widget with a DocumentWidget wrapper
        const props = {
            app: this.app,
            browserFactory: this.browserFactory,
            context: context
        };
        const content = new PipelineEditorWidget(props);
        const widget = new docregistry_1.DocumentWidget({
            content,
            context,
            node: document.createElement('div')
        });
        widget.addClass(PIPELINE_CLASS);
        widget.title.icon = ui_components_1.pipelineIcon;
        return widget;
    }
}
exports.PipelineEditorFactory = PipelineEditorFactory;
//# sourceMappingURL=PipelineEditorWidget.js.map