"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
/*
 * Copyright 2018-2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
const v4_1 = __importDefault(require("uuid/v4"));
const pipeline_template_json_1 = __importDefault(require("./pipeline-template.json"));
/**
 * A utilities class for static functions.
 */
class Utils {
    static getUUID() {
        return v4_1.default();
    }
    /**
     * Utility to create a one node pipeline to submit a single Notebook as a pipeline
     */
    static generateNotebookPipeline(filename, options) {
        const template = pipeline_template_json_1.default;
        const generated_uuid = Utils.getUUID();
        const artifactFileName = filename.replace(/^.*[\\/]/, '');
        const artifactName = artifactFileName.replace(/\.[^/.]+$/, '');
        template.id = generated_uuid;
        template.primary_pipeline = generated_uuid;
        template.pipelines[0].id = generated_uuid;
        template.pipelines[0].nodes[0].id = generated_uuid;
        template.pipelines[0].nodes[0].app_data.filename = filename;
        template.pipelines[0].nodes[0].app_data.runtime_image = options.framework;
        template.pipelines[0].nodes[0].app_data.env_vars = options.env;
        template.pipelines[0].nodes[0].app_data.dependencies = options.dependencies;
        template.pipelines[0].app_data.name = artifactName;
        template.pipelines[0].app_data.runtime = 'kfp';
        template.pipelines[0].app_data['runtime-config'] = options.runtime_config;
        return template;
    }
    /**
     * Read the version of a Pipeline. If no version is found return 0
     */
    static getPipelineVersion(pipelineDefinition) {
        const version = +this.getPipelineAppdataField(pipelineDefinition.pipelines[0], 'version') || 0;
        return version;
    }
    /**
     * Read an application specific field from the pipeline definition
     * (e.g. pipelines[0][app_data][fieldName])
     */
    static getPipelineAppdataField(node, fieldName) {
        if (this.hasPipelineAppdataField(node, fieldName)) {
            return node['app_data'][fieldName];
        }
        else {
            return null;
        }
    }
    /**
     * Check if an application specific field from the pipeline defintion exists
     * (e.g. pipelines[0][app_data][fieldName])
     */
    static hasPipelineAppdataField(node, fieldName) {
        return (Object.prototype.hasOwnProperty.call(node, 'app_data') &&
            Object.prototype.hasOwnProperty.call(node['app_data'], fieldName));
    }
    /**
     * Delete an application specific field from the pipeline definition
     * (e.g. pipelines[0][app_data][fieldName])
     */
    static deletePipelineAppdataField(node, fieldName) {
        if (this.hasPipelineAppdataField(node, fieldName)) {
            delete node['app_data'][fieldName];
        }
    }
    /**
     * Rename an application specific field from the pepileine definition if it exists by
     * by copying the field value to the new field name and then deleting the previously
     * existing field
     */
    static renamePipelineAppdataField(node, currentFieldName, newFieldName) {
        if (this.hasPipelineAppdataField(node, currentFieldName)) {
            node['app_data'][newFieldName] = node['app_data'][currentFieldName];
            this.deletePipelineAppdataField(node, currentFieldName);
        }
    }
}
exports.default = Utils;
//# sourceMappingURL=utils.js.map