/*
 * Copyright 2018-2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import '../style/index.css';
import { ExpandableComponent } from '@elyra/ui-components';
import { ReactWidget, UseSignal, Clipboard, Dialog, showDialog } from '@jupyterlab/apputils';
import { CodeCell, MarkdownCell } from '@jupyterlab/cells';
import { PathExt } from '@jupyterlab/coreutils';
import { DocumentWidget } from '@jupyterlab/docregistry';
import { FileEditor } from '@jupyterlab/fileeditor';
import { NotebookPanel } from '@jupyterlab/notebook';
import { copyIcon, addIcon } from '@jupyterlab/ui-components';
import { Signal } from '@lumino/signaling';
import React from 'react';
import { CodeSnippetService } from './CodeSnippetService';
/**
 * The CSS class added to code snippet widget.
 */
const CODE_SNIPPETS_CLASS = 'elyra-CodeSnippets';
const CODE_SNIPPETS_HEADER_CLASS = 'elyra-codeSnippetsHeader';
const CODE_SNIPPET_ITEM = 'elyra-codeSnippet-item';
/**
 * A React Component for code-snippets display list.
 */
class CodeSnippetDisplay extends React.Component {
    constructor() {
        // TODO: Use code mirror to display code
        super(...arguments);
        // Handle code snippet insert into an editor
        this.insertCodeSnippet = async (snippet) => {
            var _a, _b, _c;
            const widget = this.props.getCurrentWidget();
            const snippetStr = snippet.code.join('\n');
            if (widget instanceof DocumentWidget &&
                widget.content instanceof FileEditor) {
                const documentWidget = widget;
                const fileEditor = documentWidget.content.editor;
                const markdownRegex = /^\.(md|mkdn?|mdown|markdown)$/;
                if (PathExt.extname(widget.context.path).match(markdownRegex) !== null) {
                    // Wrap snippet into a code block when inserting it into a markdown file
                    fileEditor.replaceSelection('```' + snippet.language + '\n' + snippetStr + '\n```');
                }
                else if (widget.constructor.name == 'PythonFileEditor') {
                    this.verifyLanguageAndInsert(snippet, 'python', fileEditor);
                }
                else {
                    fileEditor.replaceSelection(snippetStr);
                }
            }
            else if (widget instanceof NotebookPanel) {
                const notebookWidget = widget;
                const notebookCell = notebookWidget.content.activeCell;
                const notebookCellEditor = notebookCell.editor;
                if (notebookCell instanceof CodeCell) {
                    const kernelInfo = await ((_b = (_a = notebookWidget.sessionContext.session) === null || _a === void 0 ? void 0 : _a.kernel) === null || _b === void 0 ? void 0 : _b.info);
                    const kernelLanguage = ((_c = kernelInfo) === null || _c === void 0 ? void 0 : _c.language_info.name) || '';
                    this.verifyLanguageAndInsert(snippet, kernelLanguage, notebookCellEditor);
                }
                else if (notebookCell instanceof MarkdownCell) {
                    // Wrap snippet into a code block when inserting it into a markdown cell
                    notebookCellEditor.replaceSelection('```' + snippet.language + '\n' + snippetStr + '\n```');
                }
                else {
                    notebookCellEditor.replaceSelection(snippetStr);
                }
            }
            else {
                this.showErrDialog('Code snippet insert failed: Unsupported widget');
            }
        };
        // Handle language compatibility between code snippet and editor
        this.verifyLanguageAndInsert = async (snippet, editorLanguage, editor) => {
            const snippetStr = snippet.code.join('\n');
            if (editorLanguage &&
                snippet.language.toLowerCase() !== editorLanguage.toLowerCase()) {
                const result = await this.showWarnDialog(editorLanguage, snippet.displayName);
                if (result.button.accept) {
                    editor.replaceSelection(snippetStr);
                }
            }
            else {
                // Language match or editorLanguage is unavailable
                editor.replaceSelection(snippetStr);
            }
        };
        // Display warning dialog when inserting a code snippet incompatible with editor's language
        this.showWarnDialog = async (editorLanguage, snippetName) => {
            return showDialog({
                title: 'Warning',
                body: 'Code snippet "' +
                    snippetName +
                    '" is incompatible with ' +
                    editorLanguage +
                    '. Continue?',
                buttons: [Dialog.cancelButton(), Dialog.okButton()]
            });
        };
        // Display error dialog when inserting a code snippet into unsupported widget (i.e. not an editor)
        this.showErrDialog = (errMsg) => {
            return showDialog({
                title: 'Error',
                body: errMsg,
                buttons: [Dialog.okButton()]
            });
        };
        // Render display of code snippet list
        this.renderCodeSnippet = (codeSnippet) => {
            const displayName = '[' + codeSnippet.language + '] ' + codeSnippet.displayName;
            const actionButtons = [
                {
                    title: 'Copy',
                    icon: copyIcon,
                    onClick: () => {
                        Clipboard.copyToSystem(codeSnippet.code.join('\n'));
                    }
                },
                {
                    title: 'Insert',
                    icon: addIcon,
                    onClick: () => {
                        this.insertCodeSnippet(codeSnippet);
                    }
                }
            ];
            return (React.createElement("div", { key: codeSnippet.name, className: CODE_SNIPPET_ITEM },
                React.createElement(ExpandableComponent, { displayName: displayName, tooltip: codeSnippet.description, actionButtons: actionButtons },
                    React.createElement("textarea", { defaultValue: codeSnippet.code.join('\n') }))));
        };
    }
    render() {
        return (React.createElement("div", null,
            React.createElement("div", { id: "codeSnippets" },
                React.createElement("div", null, this.props.codeSnippets.map(this.renderCodeSnippet)))));
    }
}
/**
 * A widget for Code Snippets.
 */
export class CodeSnippetWidget extends ReactWidget {
    constructor(getCurrentWidget) {
        super();
        this.getCurrentWidget = getCurrentWidget;
        this.codeSnippetManager = new CodeSnippetService();
        this.renderCodeSnippetsSignal = new Signal(this);
    }
    // Request code snippets from server
    async fetchData() {
        return await this.codeSnippetManager.findAll();
    }
    // Triggered when the widget button on side palette is clicked
    onAfterShow(msg) {
        this.fetchData().then((codeSnippets) => {
            this.renderCodeSnippetsSignal.emit(codeSnippets);
        });
    }
    render() {
        return (React.createElement("div", { className: CODE_SNIPPETS_CLASS },
            React.createElement("header", { className: CODE_SNIPPETS_HEADER_CLASS }, '</> Code Snippets'),
            React.createElement(UseSignal, { signal: this.renderCodeSnippetsSignal, initialArgs: [] }, (_, codeSnippets) => (React.createElement(CodeSnippetDisplay, { codeSnippets: codeSnippets, getCurrentWidget: this.getCurrentWidget })))));
    }
}
//# sourceMappingURL=CodeSnippetWidget.js.map