import { Dialog, ReactWidget, showDialog, showErrorMessage, ToolbarButtonComponent, UseSignal } from '@jupyterlab/apputils';
import { Widget } from '@lumino/widgets';
import * as React from 'react';
import { AUTH_ERROR_MESSAGES } from '../git';
import { cloneButtonStyle } from '../style/CloneButton';
import { GitCredentialsForm } from './CredentialsBox';
export function addCloneButton(model, filebrowser) {
    filebrowser.toolbar.addItem('gitClone', ReactWidget.create(React.createElement(UseSignal, { signal: model.repositoryChanged, initialArgs: {
            name: 'pathRepository',
            oldValue: null,
            newValue: model.pathRepository
        } }, (_, change) => (React.createElement(ToolbarButtonComponent, { enabled: change.newValue === null, iconClass: `${cloneButtonStyle} jp-Icon jp-Icon-16`, onClick: async () => {
            await doGitClone(model, filebrowser.model.path);
            filebrowser.model.refresh();
        }, tooltip: 'Git Clone' })))));
}
/**
 * Makes the API call to the server.
 *
 * @param cloneUrl
 */
async function makeApiCall(model, path, cloneUrl) {
    try {
        let response = await model.clone(path, cloneUrl);
        let retry = false;
        while (response.code !== 0) {
            if (response.code === 128 &&
                AUTH_ERROR_MESSAGES.map(message => response.message.indexOf(message) > -1).indexOf(true) > -1) {
                // request user credentials and try to clone again
                const result = await showDialog({
                    title: 'Git credentials required',
                    body: new GitCredentialsForm('Enter credentials for remote repository', retry ? 'Incorrect username or password.' : '')
                });
                retry = true;
                if (result.button.accept) {
                    // user accepted attempt to login
                    // try to clone again
                    response = await model.clone(path, cloneUrl, result.value);
                }
                else {
                    showErrorMessage('Clone failed', response.message, [
                        Dialog.warnButton({ label: 'DISMISS' })
                    ]);
                    break;
                }
            }
            else {
                showErrorMessage('Clone failed', response.message, [
                    Dialog.warnButton({ label: 'DISMISS' })
                ]);
                break;
            }
        }
    }
    catch (error) {
        showErrorMessage('Clone failed', error, [
            Dialog.warnButton({ label: 'DISMISS' })
        ]);
    }
}
/**
 * Callback method on Git Clone button in the File Browser toolbar.
 * 1. Invokes a new dialog box with form fields.
 * 2. Invokes the server API with the form input.
 */
export async function doGitClone(model, path) {
    const result = await showDialog({
        title: 'Clone a repo',
        body: new GitCloneForm(),
        focusNodeSelector: 'input',
        buttons: [Dialog.cancelButton(), Dialog.okButton({ label: 'CLONE' })]
    });
    if (result.button.accept) {
        if (typeof result.value !== 'undefined' && result.value) {
            const cloneUrl = result.value;
            await makeApiCall(model, path, cloneUrl);
        }
    }
}
/**
 * The UI for the form fields shown within the Clone modal.
 */
class GitCloneForm extends Widget {
    /**
     * Create a redirect form.
     */
    constructor() {
        super({ node: GitCloneForm.createFormNode() });
    }
    static createFormNode() {
        const node = document.createElement('div');
        const label = document.createElement('label');
        const input = document.createElement('input');
        const text = document.createElement('span');
        const warning = document.createElement('div');
        node.className = 'jp-RedirectForm';
        warning.className = 'jp-RedirectForm-warning';
        text.textContent = 'Enter the Clone URI of the repository';
        input.placeholder = 'https://host.com/org/repo.git';
        label.appendChild(text);
        label.appendChild(input);
        node.appendChild(label);
        node.appendChild(warning);
        return node;
    }
    /**
     * Returns the input value.
     */
    getValue() {
        return encodeURIComponent(this.node.querySelector('input').value);
    }
}
//# sourceMappingURL=gitClone.js.map