import numpy as np
import pandas as pd

from .. import tools
from ..algo import Algo


class OLMAR(Algo):
    """On-Line Portfolio Selection with Moving Average Reversion

    Reference:
        B. Li and S. C. H. Hoi.
        On-line portfolio selection with moving average reversion, 2012.
        http://icml.cc/2012/papers/168.pdf
    """

    PRICE_TYPE = "raw"
    REPLACE_MISSING = True

    def __init__(self, window=5, eps=10):
        """
        :param window: Lookback window.
        :param eps: Constraint on return for new weights on last price (average of prices).
            x * w >= eps for new weights w.
        """

        super().__init__(min_history=window)

        # input check
        if window < 2:
            raise ValueError("window parameter must be >=3")
        if eps < 1:
            raise ValueError("epsilon parameter must be >=1")

        self.window = window
        self.eps = eps

    def init_weights(self, columns):
        m = len(columns)
        return np.ones(m) / m

    def step(self, x, last_b, history):
        # calculate return prediction
        x_pred = self.predict(x, history.iloc[-self.window :])
        b = self.update(last_b, x_pred, self.eps)
        return b

    def predict(self, x, history):
        """Predict returns on next day."""
        return (history / x).mean()

    def update(self, b, x_pred, eps):
        """Update portfolio weights to satisfy constraint b * x >= eps
        and minimize distance to previous weights."""
        x_pred_mean = np.mean(x_pred)
        lam = max(
            0.0, (eps - np.dot(b, x_pred)) / np.linalg.norm(x_pred - x_pred_mean) ** 2
        )

        # limit lambda to avoid numerical problems
        lam = min(100000, lam)

        # update portfolio
        b = b + lam * (x_pred - x_pred_mean)

        # project it onto simplex
        return tools.simplex_proj(b)


if __name__ == "__main__":
    tools.quickrun(OLMAR())
