"use strict";
const fs = require("fs");
const os = require("os");
const path = require("path");
const libfs = require("../../lib/fs");
module.exports = {
    files: {
        'does not change with the file name'(test) {
            // GIVEN
            const workdir = fs.mkdtempSync(path.join(os.tmpdir(), 'hash-tests'));
            const content = 'Hello, world!';
            const input1 = path.join(workdir, 'input1.txt');
            const input2 = path.join(workdir, 'input2.txt');
            const input3 = path.join(workdir, 'input3.txt');
            fs.writeFileSync(input1, content);
            fs.writeFileSync(input2, content);
            fs.writeFileSync(input3, content + '.'); // add one character, hash should be different
            // WHEN
            const hash1 = libfs.fingerprint(input1);
            const hash2 = libfs.fingerprint(input2);
            const hash3 = libfs.fingerprint(input3);
            // THEN
            test.deepEqual(hash1, hash2);
            test.notDeepEqual(hash3, hash1);
            test.done();
        },
        'works on empty files'(test) {
            // GIVEN
            const workdir = fs.mkdtempSync(path.join(os.tmpdir(), 'hash-tests'));
            const input1 = path.join(workdir, 'empty');
            const input2 = path.join(workdir, 'empty');
            fs.writeFileSync(input1, '');
            fs.writeFileSync(input2, '');
            // WHEN
            const hash1 = libfs.fingerprint(input1);
            const hash2 = libfs.fingerprint(input2);
            // THEN
            test.deepEqual(hash1, hash2);
            test.done();
        },
    },
    directories: {
        'works on directories'(test) {
            // GIVEN
            const srcdir = path.join(__dirname, 'fixtures', 'symlinks');
            const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
            libfs.copyDirectory(srcdir, outdir);
            // WHEN
            const hashSrc = libfs.fingerprint(srcdir);
            const hashCopy = libfs.fingerprint(outdir);
            // THEN
            test.deepEqual(hashSrc, hashCopy);
            test.done();
        },
        'ignores requested files'(test) {
            // GIVEN
            const srcdir = path.join(__dirname, 'fixtures', 'symlinks');
            const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
            libfs.copyDirectory(srcdir, outdir);
            // WHEN
            const hashSrc = libfs.fingerprint(srcdir, { exclude: ['*.ignoreme'] });
            fs.writeFileSync(path.join(outdir, `${hashSrc}.ignoreme`), 'Ignore me!');
            const hashCopy = libfs.fingerprint(outdir, { exclude: ['*.ignoreme'] });
            // THEN
            test.deepEqual(hashSrc, hashCopy);
            test.done();
        },
        'changes with file names'(test) {
            // GIVEN
            const srcdir = path.join(__dirname, 'fixtures', 'symlinks');
            const cpydir = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            libfs.copyDirectory(srcdir, cpydir);
            // be careful not to break a symlink
            fs.renameSync(path.join(cpydir, 'normal-dir', 'file-in-subdir.txt'), path.join(cpydir, 'move-me.txt'));
            // WHEN
            const hashSrc = libfs.fingerprint(srcdir);
            const hashCopy = libfs.fingerprint(cpydir);
            // THEN
            test.notDeepEqual(hashSrc, hashCopy);
            test.done();
        },
    },
    symlinks: {
        'changes with the contents of followed symlink referent'(test) {
            // GIVEN
            const dir1 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const dir2 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const target = path.join(dir1, 'boom.txt');
            const content = 'boom';
            fs.writeFileSync(target, content);
            fs.symlinkSync(target, path.join(dir2, 'link-to-boom.txt'));
            // now dir2 contains a symlink to a file in dir1
            // WHEN
            const original = libfs.fingerprint(dir2);
            // now change the contents of the target
            fs.writeFileSync(target, 'changning you!');
            const afterChange = libfs.fingerprint(dir2);
            // revert the content to original and expect hash to be reverted
            fs.writeFileSync(target, content);
            const afterRevert = libfs.fingerprint(dir2);
            // THEN
            test.notDeepEqual(original, afterChange);
            test.deepEqual(afterRevert, original);
            test.done();
        },
        'does not change with the contents of un-followed symlink referent'(test) {
            // GIVEN
            const dir1 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const dir2 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const target = path.join(dir1, 'boom.txt');
            const content = 'boom';
            fs.writeFileSync(target, content);
            fs.symlinkSync(target, path.join(dir2, 'link-to-boom.txt'));
            // now dir2 contains a symlink to a file in dir1
            // WHEN
            const original = libfs.fingerprint(dir2, { follow: libfs.FollowMode.NEVER });
            // now change the contents of the target
            fs.writeFileSync(target, 'changning you!');
            const afterChange = libfs.fingerprint(dir2, { follow: libfs.FollowMode.NEVER });
            // revert the content to original and expect hash to be reverted
            fs.writeFileSync(target, content);
            const afterRevert = libfs.fingerprint(dir2, { follow: libfs.FollowMode.NEVER });
            // THEN
            test.deepEqual(original, afterChange);
            test.deepEqual(afterRevert, original);
            test.done();
        },
    },
    exclude: {
        'encodes exclude patterns'(test) {
            // GIVEN
            const dir = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const options1 = { path: dir, exclude: ['**', '!file.py'], sourcePath: dir };
            const options2 = { path: dir, exclude: ['**', '!otherfile.py'], sourcePath: dir };
            // WHEN
            const f1 = libfs.fingerprint(dir, options1);
            const f2 = libfs.fingerprint(dir, options2);
            // THEN
            test.notDeepEqual(f1, f2);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5mcy1maW5nZXJwcmludC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3QuZnMtZmluZ2VycHJpbnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLHlCQUF5QjtBQUV6Qix5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLHNDQUFzQztBQUV0QyxpQkFBUztJQUNQLEtBQUssRUFBRTtRQUNMLG9DQUFvQyxDQUFDLElBQVU7WUFDN0MsUUFBUTtZQUNSLE1BQU0sT0FBTyxHQUFHLEVBQUUsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsTUFBTSxFQUFFLEVBQUUsWUFBWSxDQUFDLENBQUMsQ0FBQztZQUNyRSxNQUFNLE9BQU8sR0FBRyxlQUFlLENBQUM7WUFDaEMsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsWUFBWSxDQUFDLENBQUM7WUFDaEQsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsWUFBWSxDQUFDLENBQUM7WUFDaEQsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsWUFBWSxDQUFDLENBQUM7WUFDaEQsRUFBRSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDbEMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDbEMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsT0FBTyxHQUFHLEdBQUcsQ0FBQyxDQUFDLENBQUMsOENBQThDO1lBRXZGLE9BQU87WUFDUCxNQUFNLEtBQUssR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQ3hDLE1BQU0sS0FBSyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDeEMsTUFBTSxLQUFLLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUV4QyxPQUFPO1lBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFDN0IsSUFBSSxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFDaEMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELHNCQUFzQixDQUFDLElBQVU7WUFDL0IsUUFBUTtZQUNSLE1BQU0sT0FBTyxHQUFHLEVBQUUsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsTUFBTSxFQUFFLEVBQUUsWUFBWSxDQUFDLENBQUMsQ0FBQztZQUNyRSxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxPQUFPLENBQUMsQ0FBQztZQUMzQyxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxPQUFPLENBQUMsQ0FBQztZQUMzQyxFQUFFLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsQ0FBQztZQUM3QixFQUFFLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsQ0FBQztZQUU3QixPQUFPO1lBQ1AsTUFBTSxLQUFLLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUN4QyxNQUFNLEtBQUssR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBRXhDLE9BQU87WUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztZQUM3QixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBQ0Y7SUFFRCxXQUFXLEVBQUU7UUFDWCxzQkFBc0IsQ0FBQyxJQUFVO1lBQy9CLFFBQVE7WUFDUixNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxVQUFVLEVBQUUsVUFBVSxDQUFDLENBQUM7WUFDNUQsTUFBTSxNQUFNLEdBQUcsRUFBRSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsRUFBRSxZQUFZLENBQUMsQ0FBQyxDQUFDO1lBQ3BFLEtBQUssQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLE1BQU0sQ0FBQyxDQUFDO1lBRXBDLE9BQU87WUFDUCxNQUFNLE9BQU8sR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQzFDLE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUMsTUFBTSxDQUFDLENBQUM7WUFFM0MsT0FBTztZQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1lBQ2xDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCx5QkFBeUIsQ0FBQyxJQUFVO1lBQ2xDLFFBQVE7WUFDUixNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxVQUFVLEVBQUUsVUFBVSxDQUFDLENBQUM7WUFDNUQsTUFBTSxNQUFNLEdBQUcsRUFBRSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsRUFBRSxZQUFZLENBQUMsQ0FBQyxDQUFDO1lBQ3BFLEtBQUssQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLE1BQU0sQ0FBQyxDQUFDO1lBRXBDLE9BQU87WUFDUCxNQUFNLE9BQU8sR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDLE1BQU0sRUFBRSxFQUFFLE9BQU8sRUFBRSxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUV2RSxFQUFFLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLEdBQUcsT0FBTyxXQUFXLENBQUMsRUFBRSxZQUFZLENBQUMsQ0FBQztZQUN6RSxNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDLE1BQU0sRUFBRSxFQUFFLE9BQU8sRUFBRSxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUV4RSxPQUFPO1lBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFDbEMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELHlCQUF5QixDQUFDLElBQVU7WUFDbEMsUUFBUTtZQUNSLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFVBQVUsRUFBRSxVQUFVLENBQUMsQ0FBQztZQUM1RCxNQUFNLE1BQU0sR0FBRyxFQUFFLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxFQUFFLG1CQUFtQixDQUFDLENBQUMsQ0FBQztZQUMzRSxLQUFLLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxNQUFNLENBQUMsQ0FBQztZQUVwQyxvQ0FBb0M7WUFDcEMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxZQUFZLEVBQUUsb0JBQW9CLENBQUMsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxhQUFhLENBQUMsQ0FBQyxDQUFDO1lBRXZHLE9BQU87WUFDUCxNQUFNLE9BQU8sR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQzFDLE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUMsTUFBTSxDQUFDLENBQUM7WUFFM0MsT0FBTztZQUNQLElBQUksQ0FBQyxZQUFZLENBQUMsT0FBTyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1lBQ3JDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7S0FDRjtJQUVELFFBQVEsRUFBRTtRQUNSLHdEQUF3RCxDQUFDLElBQVU7WUFDakUsUUFBUTtZQUNSLE1BQU0sSUFBSSxHQUFHLEVBQUUsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsTUFBTSxFQUFFLEVBQUUsbUJBQW1CLENBQUMsQ0FBQyxDQUFDO1lBQ3pFLE1BQU0sSUFBSSxHQUFHLEVBQUUsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsTUFBTSxFQUFFLEVBQUUsbUJBQW1CLENBQUMsQ0FBQyxDQUFDO1lBQ3pFLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1lBQzNDLE1BQU0sT0FBTyxHQUFHLE1BQU0sQ0FBQztZQUN2QixFQUFFLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNsQyxFQUFFLENBQUMsV0FBVyxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7WUFFNUQsZ0RBQWdEO1lBRWhELE9BQU87WUFDUCxNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxDQUFDO1lBRXpDLHdDQUF3QztZQUN4QyxFQUFFLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO1lBQzNDLE1BQU0sV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7WUFFNUMsZ0VBQWdFO1lBQ2hFLEVBQUUsQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ2xDLE1BQU0sV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7WUFFNUMsT0FBTztZQUNQLElBQUksQ0FBQyxZQUFZLENBQUMsUUFBUSxFQUFFLFdBQVcsQ0FBQyxDQUFDO1lBQ3pDLElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1lBQ3RDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxtRUFBbUUsQ0FBQyxJQUFVO1lBQzVFLFFBQVE7WUFDUixNQUFNLElBQUksR0FBRyxFQUFFLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxFQUFFLG1CQUFtQixDQUFDLENBQUMsQ0FBQztZQUN6RSxNQUFNLElBQUksR0FBRyxFQUFFLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxFQUFFLG1CQUFtQixDQUFDLENBQUMsQ0FBQztZQUN6RSxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxVQUFVLENBQUMsQ0FBQztZQUMzQyxNQUFNLE9BQU8sR0FBRyxNQUFNLENBQUM7WUFDdkIsRUFBRSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDbEMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO1lBRTVELGdEQUFnRDtZQUVoRCxPQUFPO1lBQ1AsTUFBTSxRQUFRLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsRUFBRSxNQUFNLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsQ0FBQyxDQUFDO1lBRTdFLHdDQUF3QztZQUN4QyxFQUFFLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO1lBQzNDLE1BQU0sV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUMsSUFBSSxFQUFFLEVBQUUsTUFBTSxFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLENBQUMsQ0FBQztZQUVoRixnRUFBZ0U7WUFDaEUsRUFBRSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDbEMsTUFBTSxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsRUFBRSxNQUFNLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsQ0FBQyxDQUFDO1lBRWhGLE9BQU87WUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsRUFBRSxXQUFXLENBQUMsQ0FBQztZQUN0QyxJQUFJLENBQUMsU0FBUyxDQUFDLFdBQVcsRUFBRSxRQUFRLENBQUMsQ0FBQztZQUN0QyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBQ0Y7SUFDRCxPQUFPLEVBQUU7UUFDUCwwQkFBMEIsQ0FBQyxJQUFVO1lBQ25DLFFBQVE7WUFDUixNQUFNLEdBQUcsR0FBRyxFQUFFLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxFQUFFLG1CQUFtQixDQUFDLENBQUMsQ0FBQztZQUN4RSxNQUFNLFFBQVEsR0FBRyxFQUFDLElBQUksRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFFLENBQUMsSUFBSSxFQUFFLFVBQVUsQ0FBQyxFQUFFLFVBQVUsRUFBRSxHQUFHLEVBQUMsQ0FBQztZQUMzRSxNQUFNLFFBQVEsR0FBRyxFQUFDLElBQUksRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFFLENBQUMsSUFBSSxFQUFFLGVBQWUsQ0FBQyxFQUFFLFVBQVUsRUFBRSxHQUFHLEVBQUMsQ0FBQztZQUVoRixPQUFPO1lBQ1AsTUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFDNUMsTUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFFNUMsT0FBTztZQUNQLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBQzFCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7S0FDRjtDQUNGLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBmcyBmcm9tICdmcyc7XG5pbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0ICogYXMgb3MgZnJvbSAnb3MnO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCAqIGFzIGxpYmZzIGZyb20gJy4uLy4uL2xpYi9mcyc7XG5cbmV4cG9ydCA9IHtcbiAgZmlsZXM6IHtcbiAgICAnZG9lcyBub3QgY2hhbmdlIHdpdGggdGhlIGZpbGUgbmFtZScodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHdvcmtkaXIgPSBmcy5ta2R0ZW1wU3luYyhwYXRoLmpvaW4ob3MudG1wZGlyKCksICdoYXNoLXRlc3RzJykpO1xuICAgICAgY29uc3QgY29udGVudCA9ICdIZWxsbywgd29ybGQhJztcbiAgICAgIGNvbnN0IGlucHV0MSA9IHBhdGguam9pbih3b3JrZGlyLCAnaW5wdXQxLnR4dCcpO1xuICAgICAgY29uc3QgaW5wdXQyID0gcGF0aC5qb2luKHdvcmtkaXIsICdpbnB1dDIudHh0Jyk7XG4gICAgICBjb25zdCBpbnB1dDMgPSBwYXRoLmpvaW4od29ya2RpciwgJ2lucHV0My50eHQnKTtcbiAgICAgIGZzLndyaXRlRmlsZVN5bmMoaW5wdXQxLCBjb250ZW50KTtcbiAgICAgIGZzLndyaXRlRmlsZVN5bmMoaW5wdXQyLCBjb250ZW50KTtcbiAgICAgIGZzLndyaXRlRmlsZVN5bmMoaW5wdXQzLCBjb250ZW50ICsgJy4nKTsgLy8gYWRkIG9uZSBjaGFyYWN0ZXIsIGhhc2ggc2hvdWxkIGJlIGRpZmZlcmVudFxuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBoYXNoMSA9IGxpYmZzLmZpbmdlcnByaW50KGlucHV0MSk7XG4gICAgICBjb25zdCBoYXNoMiA9IGxpYmZzLmZpbmdlcnByaW50KGlucHV0Mik7XG4gICAgICBjb25zdCBoYXNoMyA9IGxpYmZzLmZpbmdlcnByaW50KGlucHV0Myk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QuZGVlcEVxdWFsKGhhc2gxLCBoYXNoMik7XG4gICAgICB0ZXN0Lm5vdERlZXBFcXVhbChoYXNoMywgaGFzaDEpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICd3b3JrcyBvbiBlbXB0eSBmaWxlcycodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHdvcmtkaXIgPSBmcy5ta2R0ZW1wU3luYyhwYXRoLmpvaW4ob3MudG1wZGlyKCksICdoYXNoLXRlc3RzJykpO1xuICAgICAgY29uc3QgaW5wdXQxID0gcGF0aC5qb2luKHdvcmtkaXIsICdlbXB0eScpO1xuICAgICAgY29uc3QgaW5wdXQyID0gcGF0aC5qb2luKHdvcmtkaXIsICdlbXB0eScpO1xuICAgICAgZnMud3JpdGVGaWxlU3luYyhpbnB1dDEsICcnKTtcbiAgICAgIGZzLndyaXRlRmlsZVN5bmMoaW5wdXQyLCAnJyk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IGhhc2gxID0gbGliZnMuZmluZ2VycHJpbnQoaW5wdXQxKTtcbiAgICAgIGNvbnN0IGhhc2gyID0gbGliZnMuZmluZ2VycHJpbnQoaW5wdXQyKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgdGVzdC5kZWVwRXF1YWwoaGFzaDEsIGhhc2gyKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gIH0sXG5cbiAgZGlyZWN0b3JpZXM6IHtcbiAgICAnd29ya3Mgb24gZGlyZWN0b3JpZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzcmNkaXIgPSBwYXRoLmpvaW4oX19kaXJuYW1lLCAnZml4dHVyZXMnLCAnc3ltbGlua3MnKTtcbiAgICAgIGNvbnN0IG91dGRpciA9IGZzLm1rZHRlbXBTeW5jKHBhdGguam9pbihvcy50bXBkaXIoKSwgJ2NvcHktdGVzdHMnKSk7XG4gICAgICBsaWJmcy5jb3B5RGlyZWN0b3J5KHNyY2Rpciwgb3V0ZGlyKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgaGFzaFNyYyA9IGxpYmZzLmZpbmdlcnByaW50KHNyY2Rpcik7XG4gICAgICBjb25zdCBoYXNoQ29weSA9IGxpYmZzLmZpbmdlcnByaW50KG91dGRpcik7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QuZGVlcEVxdWFsKGhhc2hTcmMsIGhhc2hDb3B5KTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnaWdub3JlcyByZXF1ZXN0ZWQgZmlsZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzcmNkaXIgPSBwYXRoLmpvaW4oX19kaXJuYW1lLCAnZml4dHVyZXMnLCAnc3ltbGlua3MnKTtcbiAgICAgIGNvbnN0IG91dGRpciA9IGZzLm1rZHRlbXBTeW5jKHBhdGguam9pbihvcy50bXBkaXIoKSwgJ2NvcHktdGVzdHMnKSk7XG4gICAgICBsaWJmcy5jb3B5RGlyZWN0b3J5KHNyY2Rpciwgb3V0ZGlyKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgaGFzaFNyYyA9IGxpYmZzLmZpbmdlcnByaW50KHNyY2RpciwgeyBleGNsdWRlOiBbJyouaWdub3JlbWUnXSB9KTtcblxuICAgICAgZnMud3JpdGVGaWxlU3luYyhwYXRoLmpvaW4ob3V0ZGlyLCBgJHtoYXNoU3JjfS5pZ25vcmVtZWApLCAnSWdub3JlIG1lIScpO1xuICAgICAgY29uc3QgaGFzaENvcHkgPSBsaWJmcy5maW5nZXJwcmludChvdXRkaXIsIHsgZXhjbHVkZTogWycqLmlnbm9yZW1lJ10gfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QuZGVlcEVxdWFsKGhhc2hTcmMsIGhhc2hDb3B5KTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnY2hhbmdlcyB3aXRoIGZpbGUgbmFtZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzcmNkaXIgPSBwYXRoLmpvaW4oX19kaXJuYW1lLCAnZml4dHVyZXMnLCAnc3ltbGlua3MnKTtcbiAgICAgIGNvbnN0IGNweWRpciA9IGZzLm1rZHRlbXBTeW5jKHBhdGguam9pbihvcy50bXBkaXIoKSwgJ2ZpbmdlcnByaW50LXRlc3RzJykpO1xuICAgICAgbGliZnMuY29weURpcmVjdG9yeShzcmNkaXIsIGNweWRpcik7XG5cbiAgICAgIC8vIGJlIGNhcmVmdWwgbm90IHRvIGJyZWFrIGEgc3ltbGlua1xuICAgICAgZnMucmVuYW1lU3luYyhwYXRoLmpvaW4oY3B5ZGlyLCAnbm9ybWFsLWRpcicsICdmaWxlLWluLXN1YmRpci50eHQnKSwgcGF0aC5qb2luKGNweWRpciwgJ21vdmUtbWUudHh0JykpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBoYXNoU3JjID0gbGliZnMuZmluZ2VycHJpbnQoc3JjZGlyKTtcbiAgICAgIGNvbnN0IGhhc2hDb3B5ID0gbGliZnMuZmluZ2VycHJpbnQoY3B5ZGlyKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgdGVzdC5ub3REZWVwRXF1YWwoaGFzaFNyYywgaGFzaENvcHkpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgfSxcblxuICBzeW1saW5rczoge1xuICAgICdjaGFuZ2VzIHdpdGggdGhlIGNvbnRlbnRzIG9mIGZvbGxvd2VkIHN5bWxpbmsgcmVmZXJlbnQnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBkaXIxID0gZnMubWtkdGVtcFN5bmMocGF0aC5qb2luKG9zLnRtcGRpcigpLCAnZmluZ2VycHJpbnQtdGVzdHMnKSk7XG4gICAgICBjb25zdCBkaXIyID0gZnMubWtkdGVtcFN5bmMocGF0aC5qb2luKG9zLnRtcGRpcigpLCAnZmluZ2VycHJpbnQtdGVzdHMnKSk7XG4gICAgICBjb25zdCB0YXJnZXQgPSBwYXRoLmpvaW4oZGlyMSwgJ2Jvb20udHh0Jyk7XG4gICAgICBjb25zdCBjb250ZW50ID0gJ2Jvb20nO1xuICAgICAgZnMud3JpdGVGaWxlU3luYyh0YXJnZXQsIGNvbnRlbnQpO1xuICAgICAgZnMuc3ltbGlua1N5bmModGFyZ2V0LCBwYXRoLmpvaW4oZGlyMiwgJ2xpbmstdG8tYm9vbS50eHQnKSk7XG5cbiAgICAgIC8vIG5vdyBkaXIyIGNvbnRhaW5zIGEgc3ltbGluayB0byBhIGZpbGUgaW4gZGlyMVxuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBvcmlnaW5hbCA9IGxpYmZzLmZpbmdlcnByaW50KGRpcjIpO1xuXG4gICAgICAvLyBub3cgY2hhbmdlIHRoZSBjb250ZW50cyBvZiB0aGUgdGFyZ2V0XG4gICAgICBmcy53cml0ZUZpbGVTeW5jKHRhcmdldCwgJ2NoYW5nbmluZyB5b3UhJyk7XG4gICAgICBjb25zdCBhZnRlckNoYW5nZSA9IGxpYmZzLmZpbmdlcnByaW50KGRpcjIpO1xuXG4gICAgICAvLyByZXZlcnQgdGhlIGNvbnRlbnQgdG8gb3JpZ2luYWwgYW5kIGV4cGVjdCBoYXNoIHRvIGJlIHJldmVydGVkXG4gICAgICBmcy53cml0ZUZpbGVTeW5jKHRhcmdldCwgY29udGVudCk7XG4gICAgICBjb25zdCBhZnRlclJldmVydCA9IGxpYmZzLmZpbmdlcnByaW50KGRpcjIpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICB0ZXN0Lm5vdERlZXBFcXVhbChvcmlnaW5hbCwgYWZ0ZXJDaGFuZ2UpO1xuICAgICAgdGVzdC5kZWVwRXF1YWwoYWZ0ZXJSZXZlcnQsIG9yaWdpbmFsKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnZG9lcyBub3QgY2hhbmdlIHdpdGggdGhlIGNvbnRlbnRzIG9mIHVuLWZvbGxvd2VkIHN5bWxpbmsgcmVmZXJlbnQnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBkaXIxID0gZnMubWtkdGVtcFN5bmMocGF0aC5qb2luKG9zLnRtcGRpcigpLCAnZmluZ2VycHJpbnQtdGVzdHMnKSk7XG4gICAgICBjb25zdCBkaXIyID0gZnMubWtkdGVtcFN5bmMocGF0aC5qb2luKG9zLnRtcGRpcigpLCAnZmluZ2VycHJpbnQtdGVzdHMnKSk7XG4gICAgICBjb25zdCB0YXJnZXQgPSBwYXRoLmpvaW4oZGlyMSwgJ2Jvb20udHh0Jyk7XG4gICAgICBjb25zdCBjb250ZW50ID0gJ2Jvb20nO1xuICAgICAgZnMud3JpdGVGaWxlU3luYyh0YXJnZXQsIGNvbnRlbnQpO1xuICAgICAgZnMuc3ltbGlua1N5bmModGFyZ2V0LCBwYXRoLmpvaW4oZGlyMiwgJ2xpbmstdG8tYm9vbS50eHQnKSk7XG5cbiAgICAgIC8vIG5vdyBkaXIyIGNvbnRhaW5zIGEgc3ltbGluayB0byBhIGZpbGUgaW4gZGlyMVxuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBvcmlnaW5hbCA9IGxpYmZzLmZpbmdlcnByaW50KGRpcjIsIHsgZm9sbG93OiBsaWJmcy5Gb2xsb3dNb2RlLk5FVkVSIH0pO1xuXG4gICAgICAvLyBub3cgY2hhbmdlIHRoZSBjb250ZW50cyBvZiB0aGUgdGFyZ2V0XG4gICAgICBmcy53cml0ZUZpbGVTeW5jKHRhcmdldCwgJ2NoYW5nbmluZyB5b3UhJyk7XG4gICAgICBjb25zdCBhZnRlckNoYW5nZSA9IGxpYmZzLmZpbmdlcnByaW50KGRpcjIsIHsgZm9sbG93OiBsaWJmcy5Gb2xsb3dNb2RlLk5FVkVSIH0pO1xuXG4gICAgICAvLyByZXZlcnQgdGhlIGNvbnRlbnQgdG8gb3JpZ2luYWwgYW5kIGV4cGVjdCBoYXNoIHRvIGJlIHJldmVydGVkXG4gICAgICBmcy53cml0ZUZpbGVTeW5jKHRhcmdldCwgY29udGVudCk7XG4gICAgICBjb25zdCBhZnRlclJldmVydCA9IGxpYmZzLmZpbmdlcnByaW50KGRpcjIsIHsgZm9sbG93OiBsaWJmcy5Gb2xsb3dNb2RlLk5FVkVSIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICB0ZXN0LmRlZXBFcXVhbChvcmlnaW5hbCwgYWZ0ZXJDaGFuZ2UpO1xuICAgICAgdGVzdC5kZWVwRXF1YWwoYWZ0ZXJSZXZlcnQsIG9yaWdpbmFsKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gIH0sXG4gIGV4Y2x1ZGU6IHtcbiAgICAnZW5jb2RlcyBleGNsdWRlIHBhdHRlcm5zJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgZGlyID0gZnMubWtkdGVtcFN5bmMocGF0aC5qb2luKG9zLnRtcGRpcigpLCAnZmluZ2VycHJpbnQtdGVzdHMnKSk7XG4gICAgICBjb25zdCBvcHRpb25zMSA9IHtwYXRoOiBkaXIsIGV4Y2x1ZGU6IFsnKionLCAnIWZpbGUucHknXSwgc291cmNlUGF0aDogZGlyfTtcbiAgICAgIGNvbnN0IG9wdGlvbnMyID0ge3BhdGg6IGRpciwgZXhjbHVkZTogWycqKicsICchb3RoZXJmaWxlLnB5J10sIHNvdXJjZVBhdGg6IGRpcn07XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IGYxID0gbGliZnMuZmluZ2VycHJpbnQoZGlyLCBvcHRpb25zMSk7XG4gICAgICBjb25zdCBmMiA9IGxpYmZzLmZpbmdlcnByaW50KGRpciwgb3B0aW9uczIpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICB0ZXN0Lm5vdERlZXBFcXVhbChmMSwgZjIpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgfSxcbn07XG4iXX0=