"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const constructs_1 = require("constructs");
const stack_1 = require("../stack");
const refs_1 = require("./refs");
/**
 * Prepares the app for synthesis. This function is called by the root `prepare`
 * (normally this the App, but if a Stack is a root, it is called by the stack),
 * which means it's the last 'prepare' that executes.
 *
 * It takes care of reifying cross-references between stacks (or nested stacks),
 * and of creating assets for nested stack templates.
 *
 * @param root The root of the construct tree.
 */
function prepareApp(root) {
    if (root.node.scope) {
        throw new Error('prepareApp must be called on the root node');
    }
    // depth-first (children first) queue of nested stacks. We will pop a stack
    // from the head of this queue to prepare it's template asset.
    const queue = findAllNestedStacks(root);
    while (true) {
        refs_1.resolveReferences(root);
        const nested = queue.shift();
        if (!nested) {
            break;
        }
        defineNestedStackAsset(nested);
    }
}
exports.prepareApp = prepareApp;
/**
 * Prepares the assets for nested stacks in this app.
 * @returns `true` if assets were added to the parent of a nested stack, which
 * implies that another round of reference resolution is in order. If this
 * function returns `false`, we know we are done.
 */
function defineNestedStackAsset(nestedStack) {
    // this is needed temporarily until we move NestedStack to '@aws-cdk/core'.
    const nested = nestedStack;
    nested._prepareTemplateAsset();
}
function findAllNestedStacks(root) {
    const result = new Array();
    // create a list of all nested stacks in depth-first post order this means
    // that we first prepare the leaves and then work our way up.
    for (const stack of root.node.findAll(constructs_1.ConstructOrder.POSTORDER /* <== important */)) {
        if (stack_1.Stack.isStack(stack) && stack.nested) {
            result.push(stack);
        }
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,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