"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/cdk");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
/**
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class LambdaDeploymentGroup extends cdk.Resource {
    /**
     * Import an Lambda Deployment Group defined either outside the CDK,
     * or in a different CDK Stack and exported using the {@link #export} method.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param attrs the properties of the referenced Deployment Group
     * @returns a Construct representing a reference to an existing Deployment Group
     */
    static fromLambdaDeploymentGroupAttributes(scope, id, attrs) {
        return new ImportedLambdaDeploymentGroup(scope, id, attrs);
    }
    constructor(scope, id, props) {
        super(scope, id);
        this.application = props.application || new application_1.LambdaApplication(this, 'Application');
        this.alarms = props.alarms || [];
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('codedeploy.amazonaws.com')
        });
        this.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSCodeDeployRoleForLambda'));
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            serviceRoleArn: this.role.roleArn,
            deploymentGroupName: props.deploymentGroupName,
            deploymentConfigName: (props.deploymentConfig || deployment_config_1.LambdaDeploymentConfig.AllAtOnce).deploymentConfigName,
            deploymentStyle: {
                deploymentType: 'BLUE_GREEN',
                deploymentOption: 'WITH_TRAFFIC_CONTROL'
            },
            alarmConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAlarmConfiguration(this.alarms, props.ignorePollAlarmsFailure) }),
            autoRollbackConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAutoRollbackConfiguration(this.alarms, props.autoRollback) }),
        });
        this.deploymentGroupName = resource.refAsString;
        this.deploymentGroupArn = utils_1.arnForDeploymentGroup(this.application.applicationName, this.deploymentGroupName);
        if (props.preHook) {
            this.addPreHook(props.preHook);
        }
        if (props.postHook) {
            this.addPostHook(props.postHook);
        }
        props.alias.node.findChild('Resource').options.updatePolicy = {
            codeDeployLambdaAliasUpdate: {
                applicationName: this.application.applicationName,
                deploymentGroupName: resource.refAsString,
                beforeAllowTrafficHook: cdk.Lazy.stringValue({ produce: () => this.preHook && this.preHook.functionName }),
                afterAllowTrafficHook: cdk.Lazy.stringValue({ produce: () => this.postHook && this.postHook.functionName }),
            }
        };
    }
    /**
     * Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group
     */
    addAlarm(alarm) {
        this.alarms.push(alarm);
    }
    /**
     * Associate a function to run before deployment begins.
     * @param preHook function to run before deployment beings
     * @throws an error if a pre-hook function is already configured
     */
    addPreHook(preHook) {
        if (this.preHook !== undefined) {
            throw new Error('A pre-hook function is already defined for this deployment group');
        }
        this.preHook = preHook;
        this.grantPutLifecycleEventHookExecutionStatus(this.preHook);
        this.preHook.grantInvoke(this.role);
    }
    /**
     * Associate a function to run after deployment completes.
     * @param postHook function to run after deployment completes
     * @throws an error if a post-hook function is already configured
     */
    addPostHook(postHook) {
        if (this.postHook !== undefined) {
            throw new Error('A post-hook function is already defined for this deployment group');
        }
        this.postHook = postHook;
        this.grantPutLifecycleEventHookExecutionStatus(this.postHook);
        this.postHook.grantInvoke(this.role);
    }
    /**
     * Grant a principal permission to codedeploy:PutLifecycleEventHookExecutionStatus
     * on this deployment group resource.
     * @param grantee to grant permission to
     */
    grantPutLifecycleEventHookExecutionStatus(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [this.deploymentGroupArn],
            actions: ['codedeploy:PutLifecycleEventHookExecutionStatus'],
        });
    }
}
exports.LambdaDeploymentGroup = LambdaDeploymentGroup;
class ImportedLambdaDeploymentGroup extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.application = props.application;
        this.deploymentGroupName = props.deploymentGroupName;
        this.deploymentGroupArn = utils_1.arnForDeploymentGroup(props.application.applicationName, props.deploymentGroupName);
    }
}
//# sourceMappingURL=data:application/json;base64,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