"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/cdk");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
/**
 * Minimum number of healthy hosts for a server deployment.
 */
class MinimumHealthyHosts {
    constructor(json) {
        this.json = json;
    }
    /**
     * The minimum healhty hosts threshold expressed as an absolute number.
     */
    static count(value) {
        return new MinimumHealthyHosts({
            type: 'HOST_COUNT',
            value
        });
    }
    /**
     * The minmum healhty hosts threshold expressed as a percentage of the fleet.
     */
    static percentage(value) {
        return new MinimumHealthyHosts({
            type: 'FLEET_PERCENT',
            value
        });
    }
    /**
     * @internal
     */
    get _json() {
        return this.json;
    }
}
exports.MinimumHealthyHosts = MinimumHealthyHosts;
/**
 * A custom Deployment Configuration for an EC2/on-premise Deployment Group.
 *
 * @resource AWS::CodeDeploy::DeploymentConfig
 */
class ServerDeploymentConfig extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const resource = new codedeploy_generated_1.CfnDeploymentConfig(this, 'Resource', {
            deploymentConfigName: props.deploymentConfigName,
            minimumHealthyHosts: props.minimumHealthyHosts._json,
        });
        this.deploymentConfigName = resource.refAsString;
        this.deploymentConfigArn = utils_1.arnForDeploymentConfig(this.deploymentConfigName);
    }
    /**
     * Import a custom Deployment Configuration for an EC2/on-premise Deployment Group defined either outside the CDK,
     * or in a different CDK Stack and exported using the {@link #export} method.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param serverDeploymentConfigName the properties of the referenced custom Deployment Configuration
     * @returns a Construct representing a reference to an existing custom Deployment Configuration
     */
    static fromServerDeploymentConfigName(scope, id, serverDeploymentConfigName) {
        ignore(scope);
        ignore(id);
        return deploymentConfig(serverDeploymentConfigName);
    }
}
ServerDeploymentConfig.OneAtATime = deploymentConfig('CodeDeployDefault.OneAtATime');
ServerDeploymentConfig.HalfAtATime = deploymentConfig('CodeDeployDefault.HalfAtATime');
ServerDeploymentConfig.AllAtOnce = deploymentConfig('CodeDeployDefault.AllAtOnce');
exports.ServerDeploymentConfig = ServerDeploymentConfig;
function deploymentConfig(name) {
    return {
        deploymentConfigName: name,
        deploymentConfigArn: utils_1.arnForDeploymentConfig(name),
    };
}
function ignore(_x) { return; }
//# sourceMappingURL=data:application/json;base64,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