"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/cdk");
const cdk_1 = require("@aws-cdk/cdk");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
const load_balancer_1 = require("./load-balancer");
/**
 * Represents a reference to a CodeDeploy EC2/on-premise Deployment Group.
 *
 * If you're managing the Deployment Group alongside the rest of your CDK resources,
 * use the {@link ServerDeploymentGroup} class.
 *
 * If you want to reference an already existing Deployment Group,
 * or one defined in a different CDK Stack,
 * use the {@link #import} method.
 */
class ServerDeploymentGroupBase extends cdk.Resource {
    constructor(scope, id, deploymentConfig) {
        super(scope, id);
        this.deploymentConfig = deploymentConfig || deployment_config_1.ServerDeploymentConfig.OneAtATime;
    }
}
class ImportedServerDeploymentGroup extends ServerDeploymentGroupBase {
    constructor(scope, id, props) {
        super(scope, id, props.deploymentConfig);
        this.role = undefined;
        this.autoScalingGroups = undefined;
        this.application = props.application;
        this.deploymentGroupName = props.deploymentGroupName;
        this.deploymentGroupArn = utils_1.arnForDeploymentGroup(props.application.applicationName, props.deploymentGroupName);
    }
}
/**
 * Represents a set of instance tag groups.
 * An instance will match a set if it matches all of the groups in the set -
 * in other words, sets follow 'and' semantics.
 * You can have a maximum of 3 tag groups inside a set.
 */
class InstanceTagSet {
    constructor(...instanceTagGroups) {
        if (instanceTagGroups.length > 3) {
            throw new Error('An instance tag set can have a maximum of 3 instance tag groups, ' +
                `but ${instanceTagGroups.length} were provided`);
        }
        this._instanceTagGroups = instanceTagGroups;
    }
    get instanceTagGroups() {
        return this._instanceTagGroups.slice();
    }
}
exports.InstanceTagSet = InstanceTagSet;
/**
 * A CodeDeploy Deployment Group that deploys to EC2/on-premise instances.
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class ServerDeploymentGroup extends ServerDeploymentGroupBase {
    /**
     * Import an EC2/on-premise Deployment Group defined either outside the CDK,
     * or in a different CDK Stack and exported using the {@link #export} method.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param attrs the properties of the referenced Deployment Group
     * @returns a Construct representing a reference to an existing Deployment Group
     */
    static fromServerDeploymentGroupAttributes(scope, id, attrs) {
        return new ImportedServerDeploymentGroup(scope, id, attrs);
    }
    constructor(scope, id, props = {}) {
        super(scope, id, props.deploymentConfig);
        this.application = props.application || new application_1.ServerApplication(this, 'Application');
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('codedeploy.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSCodeDeployRole')],
        });
        this._autoScalingGroups = props.autoScalingGroups || [];
        this.installAgent = props.installAgent === undefined ? true : props.installAgent;
        this.codeDeployBucket = s3.Bucket.fromBucketName(this, 'Bucket', `aws-codedeploy-${cdk_1.Stack.of(this).region}`);
        for (const asg of this._autoScalingGroups) {
            this.addCodeDeployAgentInstallUserData(asg);
        }
        this.alarms = props.alarms || [];
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            deploymentGroupName: props.deploymentGroupName,
            serviceRoleArn: this.role.roleArn,
            deploymentConfigName: props.deploymentConfig &&
                props.deploymentConfig.deploymentConfigName,
            autoScalingGroups: cdk.Lazy.listValue({ produce: () => this._autoScalingGroups.map(asg => asg.autoScalingGroupName) }, { omitEmpty: true }),
            loadBalancerInfo: this.loadBalancerInfo(props.loadBalancer),
            deploymentStyle: props.loadBalancer === undefined
                ? undefined
                : {
                    deploymentOption: 'WITH_TRAFFIC_CONTROL',
                },
            ec2TagSet: this.ec2TagSet(props.ec2InstanceTags),
            onPremisesTagSet: this.onPremiseTagSet(props.onPremiseInstanceTags),
            alarmConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAlarmConfiguration(this.alarms, props.ignorePollAlarmsFailure) }),
            autoRollbackConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAutoRollbackConfiguration(this.alarms, props.autoRollback) }),
        });
        this.deploymentGroupName = resource.refAsString;
        this.deploymentGroupArn = utils_1.arnForDeploymentGroup(this.application.applicationName, this.deploymentGroupName);
    }
    /**
     * Adds an additional auto-scaling group to this Deployment Group.
     *
     * @param asg the auto-scaling group to add to this Deployment Group.
     * [disable-awslint:ref-via-interface] is needed in order to install the code
     * deploy agent by updating the ASGs user data.
     */
    addAutoScalingGroup(asg) {
        this._autoScalingGroups.push(asg);
        this.addCodeDeployAgentInstallUserData(asg);
    }
    /**
     * Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group
     */
    addAlarm(alarm) {
        this.alarms.push(alarm);
    }
    get autoScalingGroups() {
        return this._autoScalingGroups.slice();
    }
    addCodeDeployAgentInstallUserData(asg) {
        if (!this.installAgent) {
            return;
        }
        this.codeDeployBucket.grantRead(asg.role, 'latest/*');
        switch (asg.osType) {
            case ec2.OperatingSystemType.Linux:
                asg.addUserData('PKG_CMD=`which yum 2>/dev/null`', 'if [ -z "$PKG_CMD" ]; then', 'PKG_CMD=apt-get', 'else', 'PKG=CMD=yum', 'fi', '$PKG_CMD update -y', '$PKG_CMD install -y ruby2.0', 'if [ $? -ne 0 ]; then', '$PKG_CMD install -y ruby', 'fi', '$PKG_CMD install -y awscli', 'TMP_DIR=`mktemp -d`', 'cd $TMP_DIR', `aws s3 cp s3://aws-codedeploy-${cdk_1.Stack.of(this).region}/latest/install . --region ${cdk_1.Stack.of(this).region}`, 'chmod +x ./install', './install auto', 'rm -fr $TMP_DIR');
                break;
            case ec2.OperatingSystemType.Windows:
                asg.addUserData('Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName', `aws s3 cp s3://aws-codedeploy-${cdk_1.Stack.of(this).region}/latest/codedeploy-agent.msi $TEMPDIR\\codedeploy-agent.msi`, '$TEMPDIR\\codedeploy-agent.msi /quiet /l c:\\temp\\host-agent-install-log.txt');
                break;
        }
    }
    loadBalancerInfo(loadBalancer) {
        if (!loadBalancer) {
            return undefined;
        }
        switch (loadBalancer.generation) {
            case load_balancer_1.LoadBalancerGeneration.FIRST:
                return {
                    elbInfoList: [
                        { name: loadBalancer.name },
                    ],
                };
            case load_balancer_1.LoadBalancerGeneration.SECOND:
                return {
                    targetGroupInfoList: [
                        { name: loadBalancer.name },
                    ]
                };
        }
    }
    ec2TagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            ec2TagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    ec2TagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    onPremiseTagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            onPremisesTagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    onPremisesTagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    tagGroup2TagsArray(tagGroup) {
        const tagsInGroup = [];
        for (const tagKey in tagGroup) {
            if (tagGroup.hasOwnProperty(tagKey)) {
                const tagValues = tagGroup[tagKey];
                if (tagKey.length > 0) {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                key: tagKey,
                                value: tagValue,
                                type: 'KEY_AND_VALUE',
                            });
                        }
                    }
                    else {
                        tagsInGroup.push({
                            key: tagKey,
                            type: 'KEY_ONLY',
                        });
                    }
                }
                else {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                value: tagValue,
                                type: 'VALUE_ONLY',
                            });
                        }
                    }
                    else {
                        throw new Error('Cannot specify both an empty key and no values for an instance tag filter');
                    }
                }
            }
        }
        return tagsInGroup;
    }
}
exports.ServerDeploymentGroup = ServerDeploymentGroup;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGVwbG95bWVudC1ncm91cC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImRlcGxveW1lbnQtZ3JvdXAudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFFQSx3Q0FBeUM7QUFDekMsd0NBQXlDO0FBQ3pDLHNDQUF1QztBQUN2QyxvQ0FBcUM7QUFDckMsc0NBQXFDO0FBQ3JDLGtFQUE2RDtBQUU3RCxvQ0FBNEc7QUFDNUcsK0NBQXNFO0FBQ3RFLDJEQUFzRjtBQUN0RixtREFBdUU7QUE2Q3ZFOzs7Ozs7Ozs7R0FTRztBQUNILE1BQWUseUJBQTBCLFNBQVEsR0FBRyxDQUFDLFFBQVE7SUFRM0QsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxnQkFBMEM7UUFDdEYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNqQixJQUFJLENBQUMsZ0JBQWdCLEdBQUcsZ0JBQWdCLElBQUksMENBQXNCLENBQUMsVUFBVSxDQUFDO0lBQ2hGLENBQUM7Q0FDRjtBQUVELE1BQU0sNkJBQThCLFNBQVEseUJBQXlCO0lBT25FLFlBQVksS0FBb0IsRUFBRSxFQUFVLEVBQUUsS0FBc0M7UUFDbEYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFOM0IsU0FBSSxHQUFjLFNBQVMsQ0FBQztRQUc1QixzQkFBaUIsR0FBb0MsU0FBUyxDQUFDO1FBSzdFLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztRQUNyQyxJQUFJLENBQUMsbUJBQW1CLEdBQUcsS0FBSyxDQUFDLG1CQUFtQixDQUFDO1FBQ3JELElBQUksQ0FBQyxrQkFBa0IsR0FBRyw2QkFBcUIsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLGVBQWUsRUFBRSxLQUFLLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUNoSCxDQUFDO0NBQ0Y7QUFlRDs7Ozs7R0FLRztBQUNILE1BQWEsY0FBYztJQUd6QixZQUFZLEdBQUcsaUJBQXFDO1FBQ2xELElBQUksaUJBQWlCLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNoQyxNQUFNLElBQUksS0FBSyxDQUFDLG1FQUFtRTtnQkFDakYsT0FBTyxpQkFBaUIsQ0FBQyxNQUFNLGdCQUFnQixDQUFDLENBQUM7U0FDcEQ7UUFDRCxJQUFJLENBQUMsa0JBQWtCLEdBQUcsaUJBQWlCLENBQUM7SUFDOUMsQ0FBQztJQUVELElBQVcsaUJBQWlCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxDQUFDO0lBQ3pDLENBQUM7Q0FDRjtBQWRELHdDQWNDO0FBMEdEOzs7R0FHRztBQUNILE1BQWEscUJBQXNCLFNBQVEseUJBQXlCO0lBQ2xFOzs7Ozs7OztPQVFHO0lBQ0ksTUFBTSxDQUFDLG1DQUFtQyxDQUM3QyxLQUFvQixFQUNwQixFQUFVLEVBQ1YsS0FBc0M7UUFDeEMsT0FBTyxJQUFJLDZCQUE2QixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDN0QsQ0FBQztJQVlELFlBQVksS0FBb0IsRUFBRSxFQUFVLEVBQUUsUUFBb0MsRUFBRTtRQUNsRixLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUV6QyxJQUFJLENBQUMsV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLElBQUksSUFBSSwrQkFBaUIsQ0FBQyxJQUFJLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFFbkYsSUFBSSxDQUFDLElBQUksR0FBRyxLQUFLLENBQUMsSUFBSSxJQUFJLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsTUFBTSxFQUFFO1lBQ25ELFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQywwQkFBMEIsQ0FBQztZQUMvRCxlQUFlLEVBQUUsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLGdDQUFnQyxDQUFDLENBQUM7U0FDaEcsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGtCQUFrQixHQUFHLEtBQUssQ0FBQyxpQkFBaUIsSUFBSSxFQUFFLENBQUM7UUFDeEQsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDO1FBQ2pGLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxFQUFFLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsUUFBUSxFQUFFLGtCQUFrQixXQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUM7UUFDNUcsS0FBSyxNQUFNLEdBQUcsSUFBSSxJQUFJLENBQUMsa0JBQWtCLEVBQUU7WUFDekMsSUFBSSxDQUFDLGlDQUFpQyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQzdDO1FBRUQsSUFBSSxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUMsTUFBTSxJQUFJLEVBQUUsQ0FBQztRQUVqQyxNQUFNLFFBQVEsR0FBRyxJQUFJLHlDQUFrQixDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDeEQsZUFBZSxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsZUFBZTtZQUNqRCxtQkFBbUIsRUFBRSxLQUFLLENBQUMsbUJBQW1CO1lBQzlDLGNBQWMsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU87WUFDakMsb0JBQW9CLEVBQUUsS0FBSyxDQUFDLGdCQUFnQjtnQkFDMUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLG9CQUFvQjtZQUM3QyxpQkFBaUIsRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLEVBQUUsRUFBRSxFQUFFLFNBQVMsRUFBRSxJQUFJLEVBQUUsQ0FBQztZQUMzSSxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQztZQUMzRCxlQUFlLEVBQUUsS0FBSyxDQUFDLFlBQVksS0FBSyxTQUFTO2dCQUMvQyxDQUFDLENBQUMsU0FBUztnQkFDWCxDQUFDLENBQUM7b0JBQ0EsZ0JBQWdCLEVBQUUsc0JBQXNCO2lCQUN6QztZQUNILFNBQVMsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxlQUFlLENBQUM7WUFDaEQsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLGVBQWUsQ0FBQyxLQUFLLENBQUMscUJBQXFCLENBQUM7WUFDbkUsa0JBQWtCLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsZ0NBQXdCLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUMsdUJBQXVCLENBQUMsRUFBRSxDQUFDO1lBQzlILHlCQUF5QixFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLHVDQUErQixDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUM7U0FDbEksQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLG1CQUFtQixHQUFHLFFBQVEsQ0FBQyxXQUFXLENBQUM7UUFDaEQsSUFBSSxDQUFDLGtCQUFrQixHQUFHLDZCQUFxQixDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsZUFBZSxFQUFFLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQzlHLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUIsQ0FBQyxHQUFpQztRQUMxRCxJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ2xDLElBQUksQ0FBQyxpQ0FBaUMsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLFFBQVEsQ0FBQyxLQUF3QjtRQUN0QyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMxQixDQUFDO0lBRUQsSUFBVyxpQkFBaUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDekMsQ0FBQztJQUVPLGlDQUFpQyxDQUFDLEdBQWlDO1FBQ3pFLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxFQUFFO1lBQ3RCLE9BQU87U0FDUjtRQUVELElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLElBQUksRUFBRSxVQUFVLENBQUMsQ0FBQztRQUV0RCxRQUFRLEdBQUcsQ0FBQyxNQUFNLEVBQUU7WUFDbEIsS0FBSyxHQUFHLENBQUMsbUJBQW1CLENBQUMsS0FBSztnQkFDaEMsR0FBRyxDQUFDLFdBQVcsQ0FDYixpQ0FBaUMsRUFDakMsNEJBQTRCLEVBQzFCLGlCQUFpQixFQUNuQixNQUFNLEVBQ0osYUFBYSxFQUNmLElBQUksRUFDSixvQkFBb0IsRUFDcEIsNkJBQTZCLEVBQzdCLHVCQUF1QixFQUNyQiwwQkFBMEIsRUFDNUIsSUFBSSxFQUNKLDRCQUE0QixFQUM1QixxQkFBcUIsRUFDckIsYUFBYSxFQUNiLGlDQUFpQyxXQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sOEJBQThCLFdBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQzNHLG9CQUFvQixFQUNwQixnQkFBZ0IsRUFDaEIsaUJBQWlCLENBQ2xCLENBQUM7Z0JBQ0YsTUFBTTtZQUNSLEtBQUssR0FBRyxDQUFDLG1CQUFtQixDQUFDLE9BQU87Z0JBQ2xDLEdBQUcsQ0FBQyxXQUFXLENBQ2IscUVBQXFFLEVBQ3JFLGlDQUFpQyxXQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sNkRBQTZELEVBQ25ILCtFQUErRSxDQUNoRixDQUFDO2dCQUNGLE1BQU07U0FDVDtJQUNILENBQUM7SUFFTyxnQkFBZ0IsQ0FBQyxZQUEyQjtRQUVsRCxJQUFJLENBQUMsWUFBWSxFQUFFO1lBQ2pCLE9BQU8sU0FBUyxDQUFDO1NBQ2xCO1FBRUQsUUFBUSxZQUFZLENBQUMsVUFBVSxFQUFFO1lBQy9CLEtBQUssc0NBQXNCLENBQUMsS0FBSztnQkFDL0IsT0FBTztvQkFDTCxXQUFXLEVBQUU7d0JBQ1gsRUFBRSxJQUFJLEVBQUUsWUFBWSxDQUFDLElBQUksRUFBRTtxQkFDNUI7aUJBQ0YsQ0FBQztZQUNKLEtBQUssc0NBQXNCLENBQUMsTUFBTTtnQkFDaEMsT0FBTztvQkFDTCxtQkFBbUIsRUFBRTt3QkFDbkIsRUFBRSxJQUFJLEVBQUUsWUFBWSxDQUFDLElBQUksRUFBRTtxQkFDNUI7aUJBQ0YsQ0FBQztTQUNMO0lBQ0gsQ0FBQztJQUVPLFNBQVMsQ0FBQyxNQUF1QjtRQUV2QyxJQUFJLENBQUMsTUFBTSxJQUFJLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3BELE9BQU8sU0FBUyxDQUFDO1NBQ2xCO1FBRUQsT0FBTztZQUNMLGFBQWEsRUFBRSxNQUFNLENBQUMsaUJBQWlCLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxFQUFFO2dCQUNyRCxPQUFPO29CQUNMLFdBQVcsRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsUUFBUSxDQUNGO2lCQUM1QyxDQUFDO1lBQ0osQ0FBQyxDQUFDO1NBQ0gsQ0FBQztJQUNKLENBQUM7SUFFTyxlQUFlLENBQUMsTUFBdUI7UUFFN0MsSUFBSSxDQUFDLE1BQU0sSUFBSSxNQUFNLENBQUMsaUJBQWlCLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNwRCxPQUFPLFNBQVMsQ0FBQztTQUNsQjtRQUVELE9BQU87WUFDTCxvQkFBb0IsRUFBRSxNQUFNLENBQUMsaUJBQWlCLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxFQUFFO2dCQUM1RCxPQUFPO29CQUNMLGtCQUFrQixFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRLENBQ1o7aUJBQ3pDLENBQUM7WUFDSixDQUFDLENBQUM7U0FDSCxDQUFDO0lBQ0osQ0FBQztJQUVPLGtCQUFrQixDQUFDLFFBQTBCO1FBQ25ELE1BQU0sV0FBVyxHQUFHLEVBQUUsQ0FBQztRQUN2QixLQUFLLE1BQU0sTUFBTSxJQUFJLFFBQVEsRUFBRTtZQUM3QixJQUFJLFFBQVEsQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLEVBQUU7Z0JBQ25DLE1BQU0sU0FBUyxHQUFHLFFBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQztnQkFDbkMsSUFBSSxNQUFNLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtvQkFDckIsSUFBSSxTQUFTLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTt3QkFDeEIsS0FBSyxNQUFNLFFBQVEsSUFBSSxTQUFTLEVBQUU7NEJBQ2hDLFdBQVcsQ0FBQyxJQUFJLENBQUM7Z0NBQ2YsR0FBRyxFQUFFLE1BQU07Z0NBQ1gsS0FBSyxFQUFFLFFBQVE7Z0NBQ2YsSUFBSSxFQUFFLGVBQWU7NkJBQ3RCLENBQUMsQ0FBQzt5QkFDSjtxQkFDRjt5QkFBTTt3QkFDTCxXQUFXLENBQUMsSUFBSSxDQUFDOzRCQUNmLEdBQUcsRUFBRSxNQUFNOzRCQUNYLElBQUksRUFBRSxVQUFVO3lCQUNqQixDQUFDLENBQUM7cUJBQ0o7aUJBQ0Y7cUJBQU07b0JBQ0wsSUFBSSxTQUFTLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTt3QkFDeEIsS0FBSyxNQUFNLFFBQVEsSUFBSSxTQUFTLEVBQUU7NEJBQ2hDLFdBQVcsQ0FBQyxJQUFJLENBQUM7Z0NBQ2YsS0FBSyxFQUFFLFFBQVE7Z0NBQ2YsSUFBSSxFQUFFLFlBQVk7NkJBQ25CLENBQUMsQ0FBQzt5QkFDSjtxQkFDRjt5QkFBTTt3QkFDTCxNQUFNLElBQUksS0FBSyxDQUFDLDJFQUEyRSxDQUFDLENBQUM7cUJBQzlGO2lCQUNGO2FBQ0Y7U0FDRjtRQUNELE9BQU8sV0FBVyxDQUFDO0lBQ3JCLENBQUM7Q0FDRjtBQWhPRCxzREFnT0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgYXV0b3NjYWxpbmcgPSByZXF1aXJlKCdAYXdzLWNkay9hd3MtYXV0b3NjYWxpbmcnKTtcbmltcG9ydCBjbG91ZHdhdGNoID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWNsb3Vkd2F0Y2gnKTtcbmltcG9ydCBlYzIgPSByZXF1aXJlKCdAYXdzLWNkay9hd3MtZWMyJyk7XG5pbXBvcnQgaWFtID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWlhbScpO1xuaW1wb3J0IHMzID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLXMzJyk7XG5pbXBvcnQgY2RrID0gcmVxdWlyZSgnQGF3cy1jZGsvY2RrJyk7XG5pbXBvcnQgeyBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2Nkayc7XG5pbXBvcnQgeyBDZm5EZXBsb3ltZW50R3JvdXAgfSBmcm9tICcuLi9jb2RlZGVwbG95LmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBBdXRvUm9sbGJhY2tDb25maWcgfSBmcm9tICcuLi9yb2xsYmFjay1jb25maWcnO1xuaW1wb3J0IHsgYXJuRm9yRGVwbG95bWVudEdyb3VwLCByZW5kZXJBbGFybUNvbmZpZ3VyYXRpb24sIHJlbmRlckF1dG9Sb2xsYmFja0NvbmZpZ3VyYXRpb24gfSBmcm9tICcuLi91dGlscyc7XG5pbXBvcnQgeyBJU2VydmVyQXBwbGljYXRpb24sIFNlcnZlckFwcGxpY2F0aW9uIH0gZnJvbSAnLi9hcHBsaWNhdGlvbic7XG5pbXBvcnQgeyBJU2VydmVyRGVwbG95bWVudENvbmZpZywgU2VydmVyRGVwbG95bWVudENvbmZpZyB9IGZyb20gJy4vZGVwbG95bWVudC1jb25maWcnO1xuaW1wb3J0IHsgTG9hZEJhbGFuY2VyLCBMb2FkQmFsYW5jZXJHZW5lcmF0aW9uIH0gZnJvbSAnLi9sb2FkLWJhbGFuY2VyJztcblxuZXhwb3J0IGludGVyZmFjZSBJU2VydmVyRGVwbG95bWVudEdyb3VwIGV4dGVuZHMgY2RrLklSZXNvdXJjZSB7XG4gIHJlYWRvbmx5IGFwcGxpY2F0aW9uOiBJU2VydmVyQXBwbGljYXRpb247XG4gIHJlYWRvbmx5IHJvbGU/OiBpYW0uSVJvbGU7XG4gIC8qKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSBkZXBsb3ltZW50R3JvdXBOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGRlcGxveW1lbnRHcm91cEFybjogc3RyaW5nO1xuICByZWFkb25seSBkZXBsb3ltZW50Q29uZmlnOiBJU2VydmVyRGVwbG95bWVudENvbmZpZztcbiAgcmVhZG9ubHkgYXV0b1NjYWxpbmdHcm91cHM/OiBhdXRvc2NhbGluZy5BdXRvU2NhbGluZ0dyb3VwW107XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBvZiBhIHJlZmVyZW5jZSB0byBhIENvZGVEZXBsb3kgRUMyL29uLXByZW1pc2UgRGVwbG95bWVudCBHcm91cC5cbiAqXG4gKiBAc2VlIFNlcnZlckRlcGxveW1lbnRHcm91cCNpbXBvcnRcbiAqIEBzZWUgSVNlcnZlckRlcGxveW1lbnRHcm91cCNleHBvcnRcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTZXJ2ZXJEZXBsb3ltZW50R3JvdXBBdHRyaWJ1dGVzIHtcbiAgLyoqXG4gICAqIFRoZSByZWZlcmVuY2UgdG8gdGhlIENvZGVEZXBsb3kgRUMyL29uLXByZW1pc2UgQXBwbGljYXRpb25cbiAgICogdGhhdCB0aGlzIERlcGxveW1lbnQgR3JvdXAgYmVsb25ncyB0by5cbiAgICovXG4gIHJlYWRvbmx5IGFwcGxpY2F0aW9uOiBJU2VydmVyQXBwbGljYXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSBwaHlzaWNhbCwgaHVtYW4tcmVhZGFibGUgbmFtZSBvZiB0aGUgQ29kZURlcGxveSBFQzIvb24tcHJlbWlzZSBEZXBsb3ltZW50IEdyb3VwXG4gICAqIHRoYXQgd2UgYXJlIHJlZmVyZW5jaW5nLlxuICAgKi9cbiAgcmVhZG9ubHkgZGVwbG95bWVudEdyb3VwTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgRGVwbG95bWVudCBDb25maWd1cmF0aW9uIHRoaXMgRGVwbG95bWVudCBHcm91cCB1c2VzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBTZXJ2ZXJEZXBsb3ltZW50Q29uZmlnI09uZUF0QVRpbWVcbiAgICovXG4gIHJlYWRvbmx5IGRlcGxveW1lbnRDb25maWc/OiBJU2VydmVyRGVwbG95bWVudENvbmZpZztcbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIGEgcmVmZXJlbmNlIHRvIGEgQ29kZURlcGxveSBFQzIvb24tcHJlbWlzZSBEZXBsb3ltZW50IEdyb3VwLlxuICpcbiAqIElmIHlvdSdyZSBtYW5hZ2luZyB0aGUgRGVwbG95bWVudCBHcm91cCBhbG9uZ3NpZGUgdGhlIHJlc3Qgb2YgeW91ciBDREsgcmVzb3VyY2VzLFxuICogdXNlIHRoZSB7QGxpbmsgU2VydmVyRGVwbG95bWVudEdyb3VwfSBjbGFzcy5cbiAqXG4gKiBJZiB5b3Ugd2FudCB0byByZWZlcmVuY2UgYW4gYWxyZWFkeSBleGlzdGluZyBEZXBsb3ltZW50IEdyb3VwLFxuICogb3Igb25lIGRlZmluZWQgaW4gYSBkaWZmZXJlbnQgQ0RLIFN0YWNrLFxuICogdXNlIHRoZSB7QGxpbmsgI2ltcG9ydH0gbWV0aG9kLlxuICovXG5hYnN0cmFjdCBjbGFzcyBTZXJ2ZXJEZXBsb3ltZW50R3JvdXBCYXNlIGV4dGVuZHMgY2RrLlJlc291cmNlIGltcGxlbWVudHMgSVNlcnZlckRlcGxveW1lbnRHcm91cCB7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBhcHBsaWNhdGlvbjogSVNlcnZlckFwcGxpY2F0aW9uO1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgcm9sZT86IGlhbS5JUm9sZTtcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGRlcGxveW1lbnRHcm91cE5hbWU6IHN0cmluZztcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGRlcGxveW1lbnRHcm91cEFybjogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgZGVwbG95bWVudENvbmZpZzogSVNlcnZlckRlcGxveW1lbnRDb25maWc7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBhdXRvU2NhbGluZ0dyb3Vwcz86IGF1dG9zY2FsaW5nLkF1dG9TY2FsaW5nR3JvdXBbXTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgZGVwbG95bWVudENvbmZpZz86IElTZXJ2ZXJEZXBsb3ltZW50Q29uZmlnKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICB0aGlzLmRlcGxveW1lbnRDb25maWcgPSBkZXBsb3ltZW50Q29uZmlnIHx8IFNlcnZlckRlcGxveW1lbnRDb25maWcuT25lQXRBVGltZTtcbiAgfVxufVxuXG5jbGFzcyBJbXBvcnRlZFNlcnZlckRlcGxveW1lbnRHcm91cCBleHRlbmRzIFNlcnZlckRlcGxveW1lbnRHcm91cEJhc2Uge1xuICBwdWJsaWMgcmVhZG9ubHkgYXBwbGljYXRpb246IElTZXJ2ZXJBcHBsaWNhdGlvbjtcbiAgcHVibGljIHJlYWRvbmx5IHJvbGU/OiBpYW0uUm9sZSA9IHVuZGVmaW5lZDtcbiAgcHVibGljIHJlYWRvbmx5IGRlcGxveW1lbnRHcm91cE5hbWU6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGRlcGxveW1lbnRHcm91cEFybjogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgYXV0b1NjYWxpbmdHcm91cHM/OiBhdXRvc2NhbGluZy5BdXRvU2NhbGluZ0dyb3VwW10gPSB1bmRlZmluZWQ7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBTZXJ2ZXJEZXBsb3ltZW50R3JvdXBBdHRyaWJ1dGVzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcy5kZXBsb3ltZW50Q29uZmlnKTtcblxuICAgIHRoaXMuYXBwbGljYXRpb24gPSBwcm9wcy5hcHBsaWNhdGlvbjtcbiAgICB0aGlzLmRlcGxveW1lbnRHcm91cE5hbWUgPSBwcm9wcy5kZXBsb3ltZW50R3JvdXBOYW1lO1xuICAgIHRoaXMuZGVwbG95bWVudEdyb3VwQXJuID0gYXJuRm9yRGVwbG95bWVudEdyb3VwKHByb3BzLmFwcGxpY2F0aW9uLmFwcGxpY2F0aW9uTmFtZSwgcHJvcHMuZGVwbG95bWVudEdyb3VwTmFtZSk7XG4gIH1cbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIGEgZ3JvdXAgb2YgaW5zdGFuY2UgdGFncy5cbiAqIEFuIGluc3RhbmNlIHdpbGwgbWF0Y2ggYSBncm91cCBpZiBpdCBoYXMgYSB0YWcgbWF0Y2hpbmdcbiAqIGFueSBvZiB0aGUgZ3JvdXAncyB0YWdzIGJ5IGtleSBhbmQgYW55IG9mIHRoZSBwcm92aWRlZCB2YWx1ZXMgLVxuICogaW4gb3RoZXIgd29yZHMsIHRhZyBncm91cHMgZm9sbG93ICdvcicgc2VtYW50aWNzLlxuICogSWYgdGhlIHZhbHVlIGZvciBhIGdpdmVuIGtleSBpcyBhbiBlbXB0eSBhcnJheSxcbiAqIGFuIGluc3RhbmNlIHdpbGwgbWF0Y2ggd2hlbiBpdCBoYXMgYSB0YWcgd2l0aCB0aGUgZ2l2ZW4ga2V5LFxuICogcmVnYXJkbGVzcyBvZiB0aGUgdmFsdWUuXG4gKiBJZiB0aGUga2V5IGlzIGFuIGVtcHR5IHN0cmluZywgYW55IHRhZyxcbiAqIHJlZ2FyZGxlc3Mgb2YgaXRzIGtleSwgd2l0aCBhbnkgb2YgdGhlIGdpdmVuIHZhbHVlcywgd2lsbCBtYXRjaC5cbiAqL1xuZXhwb3J0IHR5cGUgSW5zdGFuY2VUYWdHcm91cCA9IHtba2V5OiBzdHJpbmddOiBzdHJpbmdbXX07XG5cbi8qKlxuICogUmVwcmVzZW50cyBhIHNldCBvZiBpbnN0YW5jZSB0YWcgZ3JvdXBzLlxuICogQW4gaW5zdGFuY2Ugd2lsbCBtYXRjaCBhIHNldCBpZiBpdCBtYXRjaGVzIGFsbCBvZiB0aGUgZ3JvdXBzIGluIHRoZSBzZXQgLVxuICogaW4gb3RoZXIgd29yZHMsIHNldHMgZm9sbG93ICdhbmQnIHNlbWFudGljcy5cbiAqIFlvdSBjYW4gaGF2ZSBhIG1heGltdW0gb2YgMyB0YWcgZ3JvdXBzIGluc2lkZSBhIHNldC5cbiAqL1xuZXhwb3J0IGNsYXNzIEluc3RhbmNlVGFnU2V0IHtcbiAgcHJpdmF0ZSByZWFkb25seSBfaW5zdGFuY2VUYWdHcm91cHM6IEluc3RhbmNlVGFnR3JvdXBbXTtcblxuICBjb25zdHJ1Y3RvciguLi5pbnN0YW5jZVRhZ0dyb3VwczogSW5zdGFuY2VUYWdHcm91cFtdKSB7XG4gICAgaWYgKGluc3RhbmNlVGFnR3JvdXBzLmxlbmd0aCA+IDMpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQW4gaW5zdGFuY2UgdGFnIHNldCBjYW4gaGF2ZSBhIG1heGltdW0gb2YgMyBpbnN0YW5jZSB0YWcgZ3JvdXBzLCAnICtcbiAgICAgICAgYGJ1dCAke2luc3RhbmNlVGFnR3JvdXBzLmxlbmd0aH0gd2VyZSBwcm92aWRlZGApO1xuICAgIH1cbiAgICB0aGlzLl9pbnN0YW5jZVRhZ0dyb3VwcyA9IGluc3RhbmNlVGFnR3JvdXBzO1xuICB9XG5cbiAgcHVibGljIGdldCBpbnN0YW5jZVRhZ0dyb3VwcygpOiBJbnN0YW5jZVRhZ0dyb3VwW10ge1xuICAgIHJldHVybiB0aGlzLl9pbnN0YW5jZVRhZ0dyb3Vwcy5zbGljZSgpO1xuICB9XG59XG5cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgZm9yIHtAbGluayBTZXJ2ZXJEZXBsb3ltZW50R3JvdXB9LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFNlcnZlckRlcGxveW1lbnRHcm91cFByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBDb2RlRGVwbG95IEVDMi9vbi1wcmVtaXNlIEFwcGxpY2F0aW9uIHRoaXMgRGVwbG95bWVudCBHcm91cCBiZWxvbmdzIHRvLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEEgbmV3IEFwcGxpY2F0aW9uIHdpbGwgYmUgY3JlYXRlZC5cbiAgICovXG4gIHJlYWRvbmx5IGFwcGxpY2F0aW9uPzogSVNlcnZlckFwcGxpY2F0aW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgc2VydmljZSBSb2xlIG9mIHRoaXMgRGVwbG95bWVudCBHcm91cC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBIG5ldyBSb2xlIHdpbGwgYmUgY3JlYXRlZC5cbiAgICovXG4gIHJlYWRvbmx5IHJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgLyoqXG4gICAqIFRoZSBwaHlzaWNhbCwgaHVtYW4tcmVhZGFibGUgbmFtZSBvZiB0aGUgQ29kZURlcGxveSBEZXBsb3ltZW50IEdyb3VwLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEFuIGF1dG8tZ2VuZXJhdGVkIG5hbWUgd2lsbCBiZSB1c2VkLlxuICAgKi9cbiAgcmVhZG9ubHkgZGVwbG95bWVudEdyb3VwTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEVDMi9vbi1wcmVtaXNlIERlcGxveW1lbnQgQ29uZmlndXJhdGlvbiB0byB1c2UgZm9yIHRoaXMgRGVwbG95bWVudCBHcm91cC5cbiAgICpcbiAgICogQGRlZmF1bHQgU2VydmVyRGVwbG95bWVudENvbmZpZyNPbmVBdEFUaW1lXG4gICAqL1xuICByZWFkb25seSBkZXBsb3ltZW50Q29uZmlnPzogSVNlcnZlckRlcGxveW1lbnRDb25maWc7XG5cbiAgLyoqXG4gICAqIFRoZSBhdXRvLXNjYWxpbmcgZ3JvdXBzIGJlbG9uZ2luZyB0byB0aGlzIERlcGxveW1lbnQgR3JvdXAuXG4gICAqXG4gICAqIEF1dG8tc2NhbGluZyBncm91cHMgY2FuIGFsc28gYmUgYWRkZWQgYWZ0ZXIgdGhlIERlcGxveW1lbnQgR3JvdXAgaXMgY3JlYXRlZFxuICAgKiB1c2luZyB0aGUge0BsaW5rICNhZGRBdXRvU2NhbGluZ0dyb3VwfSBtZXRob2QuXG4gICAqXG4gICAqIFtkaXNhYmxlLWF3c2xpbnQ6cmVmLXZpYS1pbnRlcmZhY2VdIGlzIG5lZWRlZCBiZWNhdXNlIHdlIHVwZGF0ZSB1c2VyZGF0YVxuICAgKiBmb3IgQVNHcyB0byBpbnN0YWxsIHRoZSBjb2RlZGVwbG95IGFnZW50LlxuICAgKlxuICAgKiBAZGVmYXVsdCBbXVxuICAgKi9cbiAgcmVhZG9ubHkgYXV0b1NjYWxpbmdHcm91cHM/OiBhdXRvc2NhbGluZy5BdXRvU2NhbGluZ0dyb3VwW107XG5cbiAgLyoqXG4gICAqIElmIHlvdSd2ZSBwcm92aWRlZCBhbnkgYXV0by1zY2FsaW5nIGdyb3VwcyB3aXRoIHRoZSB7QGxpbmsgI2F1dG9TY2FsaW5nR3JvdXBzfSBwcm9wZXJ0eSxcbiAgICogeW91IGNhbiBzZXQgdGhpcyBwcm9wZXJ0eSB0byBhZGQgVXNlciBEYXRhIHRoYXQgaW5zdGFsbHMgdGhlIENvZGVEZXBsb3kgYWdlbnQgb24gdGhlIGluc3RhbmNlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlZGVwbG95L2xhdGVzdC91c2VyZ3VpZGUvY29kZWRlcGxveS1hZ2VudC1vcGVyYXRpb25zLWluc3RhbGwuaHRtbFxuICAgKi9cbiAgcmVhZG9ubHkgaW5zdGFsbEFnZW50PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIGxvYWQgYmFsYW5jZXIgdG8gcGxhY2UgaW4gZnJvbnQgb2YgdGhpcyBEZXBsb3ltZW50IEdyb3VwLlxuICAgKiBDYW4gYmUgY3JlYXRlZCBmcm9tIGVpdGhlciBhIGNsYXNzaWMgRWxhc3RpYyBMb2FkIEJhbGFuY2VyLFxuICAgKiBvciBhbiBBcHBsaWNhdGlvbiBMb2FkIEJhbGFuY2VyIC8gTmV0d29yayBMb2FkIEJhbGFuY2VyIFRhcmdldCBHcm91cC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBEZXBsb3ltZW50IEdyb3VwIHdpbGwgbm90IGhhdmUgYSBsb2FkIGJhbGFuY2VyIGRlZmluZWQuXG4gICAqL1xuICByZWFkb25seSBsb2FkQmFsYW5jZXI/OiBMb2FkQmFsYW5jZXI7XG5cbiAgLyoqXG4gICAqIEFsbCBFQzIgaW5zdGFuY2VzIG1hdGNoaW5nIHRoZSBnaXZlbiBzZXQgb2YgdGFncyB3aGVuIGEgZGVwbG95bWVudCBvY2N1cnMgd2lsbCBiZSBhZGRlZCB0byB0aGlzIERlcGxveW1lbnQgR3JvdXAuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gYWRkaXRpb25hbCBFQzIgaW5zdGFuY2VzIHdpbGwgYmUgYWRkZWQgdG8gdGhlIERlcGxveW1lbnQgR3JvdXAuXG4gICAqL1xuICByZWFkb25seSBlYzJJbnN0YW5jZVRhZ3M/OiBJbnN0YW5jZVRhZ1NldDtcblxuICAvKipcbiAgICogQWxsIG9uLXByZW1pc2UgaW5zdGFuY2VzIG1hdGNoaW5nIHRoZSBnaXZlbiBzZXQgb2YgdGFncyB3aGVuIGEgZGVwbG95bWVudCBvY2N1cnMgd2lsbCBiZSBhZGRlZCB0byB0aGlzIERlcGxveW1lbnQgR3JvdXAuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gYWRkaXRpb25hbCBvbi1wcmVtaXNlIGluc3RhbmNlcyB3aWxsIGJlIGFkZGVkIHRvIHRoZSBEZXBsb3ltZW50IEdyb3VwLlxuICAgKi9cbiAgcmVhZG9ubHkgb25QcmVtaXNlSW5zdGFuY2VUYWdzPzogSW5zdGFuY2VUYWdTZXQ7XG5cbiAgLyoqXG4gICAqIFRoZSBDbG91ZFdhdGNoIGFsYXJtcyBhc3NvY2lhdGVkIHdpdGggdGhpcyBEZXBsb3ltZW50IEdyb3VwLlxuICAgKiBDb2RlRGVwbG95IHdpbGwgc3RvcCAoYW5kIG9wdGlvbmFsbHkgcm9sbCBiYWNrKVxuICAgKiBhIGRlcGxveW1lbnQgaWYgZHVyaW5nIGl0IGFueSBvZiB0aGUgYWxhcm1zIHRyaWdnZXIuXG4gICAqXG4gICAqIEFsYXJtcyBjYW4gYWxzbyBiZSBhZGRlZCBhZnRlciB0aGUgRGVwbG95bWVudCBHcm91cCBpcyBjcmVhdGVkIHVzaW5nIHRoZSB7QGxpbmsgI2FkZEFsYXJtfSBtZXRob2QuXG4gICAqXG4gICAqIEBkZWZhdWx0IFtdXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVkZXBsb3kvbGF0ZXN0L3VzZXJndWlkZS9tb25pdG9yaW5nLWNyZWF0ZS1hbGFybXMuaHRtbFxuICAgKi9cbiAgcmVhZG9ubHkgYWxhcm1zPzogY2xvdWR3YXRjaC5JQWxhcm1bXTtcblxuICAvKipcbiAgICogV2hldGhlciB0byBjb250aW51ZSBhIGRlcGxveW1lbnQgZXZlbiBpZiBmZXRjaGluZyB0aGUgYWxhcm0gc3RhdHVzIGZyb20gQ2xvdWRXYXRjaCBmYWlsZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBpZ25vcmVQb2xsQWxhcm1zRmFpbHVyZT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBhdXRvLXJvbGxiYWNrIGNvbmZpZ3VyYXRpb24gZm9yIHRoaXMgRGVwbG95bWVudCBHcm91cC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBkZWZhdWx0IEF1dG9Sb2xsYmFja0NvbmZpZy5cbiAgICovXG4gIHJlYWRvbmx5IGF1dG9Sb2xsYmFjaz86IEF1dG9Sb2xsYmFja0NvbmZpZztcbn1cblxuLyoqXG4gKiBBIENvZGVEZXBsb3kgRGVwbG95bWVudCBHcm91cCB0aGF0IGRlcGxveXMgdG8gRUMyL29uLXByZW1pc2UgaW5zdGFuY2VzLlxuICogQHJlc291cmNlIEFXUzo6Q29kZURlcGxveTo6RGVwbG95bWVudEdyb3VwXG4gKi9cbmV4cG9ydCBjbGFzcyBTZXJ2ZXJEZXBsb3ltZW50R3JvdXAgZXh0ZW5kcyBTZXJ2ZXJEZXBsb3ltZW50R3JvdXBCYXNlIHtcbiAgLyoqXG4gICAqIEltcG9ydCBhbiBFQzIvb24tcHJlbWlzZSBEZXBsb3ltZW50IEdyb3VwIGRlZmluZWQgZWl0aGVyIG91dHNpZGUgdGhlIENESyxcbiAgICogb3IgaW4gYSBkaWZmZXJlbnQgQ0RLIFN0YWNrIGFuZCBleHBvcnRlZCB1c2luZyB0aGUge0BsaW5rICNleHBvcnR9IG1ldGhvZC5cbiAgICpcbiAgICogQHBhcmFtIHNjb3BlIHRoZSBwYXJlbnQgQ29uc3RydWN0IGZvciB0aGlzIG5ldyBDb25zdHJ1Y3RcbiAgICogQHBhcmFtIGlkIHRoZSBsb2dpY2FsIElEIG9mIHRoaXMgbmV3IENvbnN0cnVjdFxuICAgKiBAcGFyYW0gYXR0cnMgdGhlIHByb3BlcnRpZXMgb2YgdGhlIHJlZmVyZW5jZWQgRGVwbG95bWVudCBHcm91cFxuICAgKiBAcmV0dXJucyBhIENvbnN0cnVjdCByZXByZXNlbnRpbmcgYSByZWZlcmVuY2UgdG8gYW4gZXhpc3RpbmcgRGVwbG95bWVudCBHcm91cFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tU2VydmVyRGVwbG95bWVudEdyb3VwQXR0cmlidXRlcyhcbiAgICAgIHNjb3BlOiBjZGsuQ29uc3RydWN0LFxuICAgICAgaWQ6IHN0cmluZyxcbiAgICAgIGF0dHJzOiBTZXJ2ZXJEZXBsb3ltZW50R3JvdXBBdHRyaWJ1dGVzKTogSVNlcnZlckRlcGxveW1lbnRHcm91cCB7XG4gICAgcmV0dXJuIG5ldyBJbXBvcnRlZFNlcnZlckRlcGxveW1lbnRHcm91cChzY29wZSwgaWQsIGF0dHJzKTtcbiAgfVxuXG4gIHB1YmxpYyByZWFkb25seSBhcHBsaWNhdGlvbjogSVNlcnZlckFwcGxpY2F0aW9uO1xuICBwdWJsaWMgcmVhZG9ubHkgcm9sZT86IGlhbS5JUm9sZTtcbiAgcHVibGljIHJlYWRvbmx5IGRlcGxveW1lbnRHcm91cEFybjogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgZGVwbG95bWVudEdyb3VwTmFtZTogc3RyaW5nO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgX2F1dG9TY2FsaW5nR3JvdXBzOiBhdXRvc2NhbGluZy5BdXRvU2NhbGluZ0dyb3VwW107XG4gIHByaXZhdGUgcmVhZG9ubHkgaW5zdGFsbEFnZW50OiBib29sZWFuO1xuICBwcml2YXRlIHJlYWRvbmx5IGNvZGVEZXBsb3lCdWNrZXQ6IHMzLklCdWNrZXQ7XG4gIHByaXZhdGUgcmVhZG9ubHkgYWxhcm1zOiBjbG91ZHdhdGNoLklBbGFybVtdO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogU2VydmVyRGVwbG95bWVudEdyb3VwUHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMuZGVwbG95bWVudENvbmZpZyk7XG5cbiAgICB0aGlzLmFwcGxpY2F0aW9uID0gcHJvcHMuYXBwbGljYXRpb24gfHwgbmV3IFNlcnZlckFwcGxpY2F0aW9uKHRoaXMsICdBcHBsaWNhdGlvbicpO1xuXG4gICAgdGhpcy5yb2xlID0gcHJvcHMucm9sZSB8fCBuZXcgaWFtLlJvbGUodGhpcywgJ1JvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnY29kZWRlcGxveS5hbWF6b25hd3MuY29tJyksXG4gICAgICBtYW5hZ2VkUG9saWNpZXM6IFtpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ3NlcnZpY2Utcm9sZS9BV1NDb2RlRGVwbG95Um9sZScpXSxcbiAgICB9KTtcblxuICAgIHRoaXMuX2F1dG9TY2FsaW5nR3JvdXBzID0gcHJvcHMuYXV0b1NjYWxpbmdHcm91cHMgfHwgW107XG4gICAgdGhpcy5pbnN0YWxsQWdlbnQgPSBwcm9wcy5pbnN0YWxsQWdlbnQgPT09IHVuZGVmaW5lZCA/IHRydWUgOiBwcm9wcy5pbnN0YWxsQWdlbnQ7XG4gICAgdGhpcy5jb2RlRGVwbG95QnVja2V0ID0gczMuQnVja2V0LmZyb21CdWNrZXROYW1lKHRoaXMsICdCdWNrZXQnLCBgYXdzLWNvZGVkZXBsb3ktJHtTdGFjay5vZih0aGlzKS5yZWdpb259YCk7XG4gICAgZm9yIChjb25zdCBhc2cgb2YgdGhpcy5fYXV0b1NjYWxpbmdHcm91cHMpIHtcbiAgICAgIHRoaXMuYWRkQ29kZURlcGxveUFnZW50SW5zdGFsbFVzZXJEYXRhKGFzZyk7XG4gICAgfVxuXG4gICAgdGhpcy5hbGFybXMgPSBwcm9wcy5hbGFybXMgfHwgW107XG5cbiAgICBjb25zdCByZXNvdXJjZSA9IG5ldyBDZm5EZXBsb3ltZW50R3JvdXAodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgYXBwbGljYXRpb25OYW1lOiB0aGlzLmFwcGxpY2F0aW9uLmFwcGxpY2F0aW9uTmFtZSxcbiAgICAgIGRlcGxveW1lbnRHcm91cE5hbWU6IHByb3BzLmRlcGxveW1lbnRHcm91cE5hbWUsXG4gICAgICBzZXJ2aWNlUm9sZUFybjogdGhpcy5yb2xlLnJvbGVBcm4sXG4gICAgICBkZXBsb3ltZW50Q29uZmlnTmFtZTogcHJvcHMuZGVwbG95bWVudENvbmZpZyAmJlxuICAgICAgICBwcm9wcy5kZXBsb3ltZW50Q29uZmlnLmRlcGxveW1lbnRDb25maWdOYW1lLFxuICAgICAgYXV0b1NjYWxpbmdHcm91cHM6IGNkay5MYXp5Lmxpc3RWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMuX2F1dG9TY2FsaW5nR3JvdXBzLm1hcChhc2cgPT4gYXNnLmF1dG9TY2FsaW5nR3JvdXBOYW1lKSB9LCB7IG9taXRFbXB0eTogdHJ1ZSB9KSxcbiAgICAgIGxvYWRCYWxhbmNlckluZm86IHRoaXMubG9hZEJhbGFuY2VySW5mbyhwcm9wcy5sb2FkQmFsYW5jZXIpLFxuICAgICAgZGVwbG95bWVudFN0eWxlOiBwcm9wcy5sb2FkQmFsYW5jZXIgPT09IHVuZGVmaW5lZFxuICAgICAgICA/IHVuZGVmaW5lZFxuICAgICAgICA6IHtcbiAgICAgICAgICBkZXBsb3ltZW50T3B0aW9uOiAnV0lUSF9UUkFGRklDX0NPTlRST0wnLFxuICAgICAgICB9LFxuICAgICAgZWMyVGFnU2V0OiB0aGlzLmVjMlRhZ1NldChwcm9wcy5lYzJJbnN0YW5jZVRhZ3MpLFxuICAgICAgb25QcmVtaXNlc1RhZ1NldDogdGhpcy5vblByZW1pc2VUYWdTZXQocHJvcHMub25QcmVtaXNlSW5zdGFuY2VUYWdzKSxcbiAgICAgIGFsYXJtQ29uZmlndXJhdGlvbjogY2RrLkxhenkuYW55VmFsdWUoeyBwcm9kdWNlOiAoKSA9PiByZW5kZXJBbGFybUNvbmZpZ3VyYXRpb24odGhpcy5hbGFybXMsIHByb3BzLmlnbm9yZVBvbGxBbGFybXNGYWlsdXJlKSB9KSxcbiAgICAgIGF1dG9Sb2xsYmFja0NvbmZpZ3VyYXRpb246IGNkay5MYXp5LmFueVZhbHVlKHsgcHJvZHVjZTogKCkgPT4gcmVuZGVyQXV0b1JvbGxiYWNrQ29uZmlndXJhdGlvbih0aGlzLmFsYXJtcywgcHJvcHMuYXV0b1JvbGxiYWNrKSB9KSxcbiAgICB9KTtcblxuICAgIHRoaXMuZGVwbG95bWVudEdyb3VwTmFtZSA9IHJlc291cmNlLnJlZkFzU3RyaW5nO1xuICAgIHRoaXMuZGVwbG95bWVudEdyb3VwQXJuID0gYXJuRm9yRGVwbG95bWVudEdyb3VwKHRoaXMuYXBwbGljYXRpb24uYXBwbGljYXRpb25OYW1lLCB0aGlzLmRlcGxveW1lbnRHcm91cE5hbWUpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYW4gYWRkaXRpb25hbCBhdXRvLXNjYWxpbmcgZ3JvdXAgdG8gdGhpcyBEZXBsb3ltZW50IEdyb3VwLlxuICAgKlxuICAgKiBAcGFyYW0gYXNnIHRoZSBhdXRvLXNjYWxpbmcgZ3JvdXAgdG8gYWRkIHRvIHRoaXMgRGVwbG95bWVudCBHcm91cC5cbiAgICogW2Rpc2FibGUtYXdzbGludDpyZWYtdmlhLWludGVyZmFjZV0gaXMgbmVlZGVkIGluIG9yZGVyIHRvIGluc3RhbGwgdGhlIGNvZGVcbiAgICogZGVwbG95IGFnZW50IGJ5IHVwZGF0aW5nIHRoZSBBU0dzIHVzZXIgZGF0YS5cbiAgICovXG4gIHB1YmxpYyBhZGRBdXRvU2NhbGluZ0dyb3VwKGFzZzogYXV0b3NjYWxpbmcuQXV0b1NjYWxpbmdHcm91cCk6IHZvaWQge1xuICAgIHRoaXMuX2F1dG9TY2FsaW5nR3JvdXBzLnB1c2goYXNnKTtcbiAgICB0aGlzLmFkZENvZGVEZXBsb3lBZ2VudEluc3RhbGxVc2VyRGF0YShhc2cpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFzc29jaWF0ZXMgYW4gYWRkaXRpb25hbCBhbGFybSB3aXRoIHRoaXMgRGVwbG95bWVudCBHcm91cC5cbiAgICpcbiAgICogQHBhcmFtIGFsYXJtIHRoZSBhbGFybSB0byBhc3NvY2lhdGUgd2l0aCB0aGlzIERlcGxveW1lbnQgR3JvdXBcbiAgICovXG4gIHB1YmxpYyBhZGRBbGFybShhbGFybTogY2xvdWR3YXRjaC5JQWxhcm0pOiB2b2lkIHtcbiAgICB0aGlzLmFsYXJtcy5wdXNoKGFsYXJtKTtcbiAgfVxuXG4gIHB1YmxpYyBnZXQgYXV0b1NjYWxpbmdHcm91cHMoKTogYXV0b3NjYWxpbmcuQXV0b1NjYWxpbmdHcm91cFtdIHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gdGhpcy5fYXV0b1NjYWxpbmdHcm91cHMuc2xpY2UoKTtcbiAgfVxuXG4gIHByaXZhdGUgYWRkQ29kZURlcGxveUFnZW50SW5zdGFsbFVzZXJEYXRhKGFzZzogYXV0b3NjYWxpbmcuQXV0b1NjYWxpbmdHcm91cCk6IHZvaWQge1xuICAgIGlmICghdGhpcy5pbnN0YWxsQWdlbnQpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICB0aGlzLmNvZGVEZXBsb3lCdWNrZXQuZ3JhbnRSZWFkKGFzZy5yb2xlLCAnbGF0ZXN0LyonKTtcblxuICAgIHN3aXRjaCAoYXNnLm9zVHlwZSkge1xuICAgICAgY2FzZSBlYzIuT3BlcmF0aW5nU3lzdGVtVHlwZS5MaW51eDpcbiAgICAgICAgYXNnLmFkZFVzZXJEYXRhKFxuICAgICAgICAgICdQS0dfQ01EPWB3aGljaCB5dW0gMj4vZGV2L251bGxgJyxcbiAgICAgICAgICAnaWYgWyAteiBcIiRQS0dfQ01EXCIgXTsgdGhlbicsXG4gICAgICAgICAgICAnUEtHX0NNRD1hcHQtZ2V0JyxcbiAgICAgICAgICAnZWxzZScsXG4gICAgICAgICAgICAnUEtHPUNNRD15dW0nLFxuICAgICAgICAgICdmaScsXG4gICAgICAgICAgJyRQS0dfQ01EIHVwZGF0ZSAteScsXG4gICAgICAgICAgJyRQS0dfQ01EIGluc3RhbGwgLXkgcnVieTIuMCcsXG4gICAgICAgICAgJ2lmIFsgJD8gLW5lIDAgXTsgdGhlbicsXG4gICAgICAgICAgICAnJFBLR19DTUQgaW5zdGFsbCAteSBydWJ5JyxcbiAgICAgICAgICAnZmknLFxuICAgICAgICAgICckUEtHX0NNRCBpbnN0YWxsIC15IGF3c2NsaScsXG4gICAgICAgICAgJ1RNUF9ESVI9YG1rdGVtcCAtZGAnLFxuICAgICAgICAgICdjZCAkVE1QX0RJUicsXG4gICAgICAgICAgYGF3cyBzMyBjcCBzMzovL2F3cy1jb2RlZGVwbG95LSR7U3RhY2sub2YodGhpcykucmVnaW9ufS9sYXRlc3QvaW5zdGFsbCAuIC0tcmVnaW9uICR7U3RhY2sub2YodGhpcykucmVnaW9ufWAsXG4gICAgICAgICAgJ2NobW9kICt4IC4vaW5zdGFsbCcsXG4gICAgICAgICAgJy4vaW5zdGFsbCBhdXRvJyxcbiAgICAgICAgICAncm0gLWZyICRUTVBfRElSJyxcbiAgICAgICAgKTtcbiAgICAgICAgYnJlYWs7XG4gICAgICBjYXNlIGVjMi5PcGVyYXRpbmdTeXN0ZW1UeXBlLldpbmRvd3M6XG4gICAgICAgIGFzZy5hZGRVc2VyRGF0YShcbiAgICAgICAgICAnU2V0LVZhcmlhYmxlIC1OYW1lIFRFTVBESVIgLVZhbHVlIChOZXctVGVtcG9yYXJ5RmlsZSkuRGlyZWN0b3J5TmFtZScsXG4gICAgICAgICAgYGF3cyBzMyBjcCBzMzovL2F3cy1jb2RlZGVwbG95LSR7U3RhY2sub2YodGhpcykucmVnaW9ufS9sYXRlc3QvY29kZWRlcGxveS1hZ2VudC5tc2kgJFRFTVBESVJcXFxcY29kZWRlcGxveS1hZ2VudC5tc2lgLFxuICAgICAgICAgICckVEVNUERJUlxcXFxjb2RlZGVwbG95LWFnZW50Lm1zaSAvcXVpZXQgL2wgYzpcXFxcdGVtcFxcXFxob3N0LWFnZW50LWluc3RhbGwtbG9nLnR4dCcsXG4gICAgICAgICk7XG4gICAgICAgIGJyZWFrO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgbG9hZEJhbGFuY2VySW5mbyhsb2FkQmFsYW5jZXI/OiBMb2FkQmFsYW5jZXIpOlxuICAgICAgQ2ZuRGVwbG95bWVudEdyb3VwLkxvYWRCYWxhbmNlckluZm9Qcm9wZXJ0eSB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKCFsb2FkQmFsYW5jZXIpIHtcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuXG4gICAgc3dpdGNoIChsb2FkQmFsYW5jZXIuZ2VuZXJhdGlvbikge1xuICAgICAgY2FzZSBMb2FkQmFsYW5jZXJHZW5lcmF0aW9uLkZJUlNUOlxuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIGVsYkluZm9MaXN0OiBbXG4gICAgICAgICAgICB7IG5hbWU6IGxvYWRCYWxhbmNlci5uYW1lIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfTtcbiAgICAgIGNhc2UgTG9hZEJhbGFuY2VyR2VuZXJhdGlvbi5TRUNPTkQ6XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgdGFyZ2V0R3JvdXBJbmZvTGlzdDogW1xuICAgICAgICAgICAgeyBuYW1lOiBsb2FkQmFsYW5jZXIubmFtZSB9LFxuICAgICAgICAgIF1cbiAgICAgICAgfTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGVjMlRhZ1NldCh0YWdTZXQ/OiBJbnN0YW5jZVRhZ1NldCk6XG4gICAgICBDZm5EZXBsb3ltZW50R3JvdXAuRUMyVGFnU2V0UHJvcGVydHkgfCB1bmRlZmluZWQge1xuICAgIGlmICghdGFnU2V0IHx8IHRhZ1NldC5pbnN0YW5jZVRhZ0dyb3Vwcy5sZW5ndGggPT09IDApIHtcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuXG4gICAgcmV0dXJuIHtcbiAgICAgIGVjMlRhZ1NldExpc3Q6IHRhZ1NldC5pbnN0YW5jZVRhZ0dyb3Vwcy5tYXAodGFnR3JvdXAgPT4ge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIGVjMlRhZ0dyb3VwOiB0aGlzLnRhZ0dyb3VwMlRhZ3NBcnJheSh0YWdHcm91cCkgYXNcbiAgICAgICAgICAgIENmbkRlcGxveW1lbnRHcm91cC5FQzJUYWdGaWx0ZXJQcm9wZXJ0eVtdLFxuICAgICAgICB9O1xuICAgICAgfSksXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgb25QcmVtaXNlVGFnU2V0KHRhZ1NldD86IEluc3RhbmNlVGFnU2V0KTpcbiAgICAgIENmbkRlcGxveW1lbnRHcm91cC5PblByZW1pc2VzVGFnU2V0UHJvcGVydHkgfCB1bmRlZmluZWQge1xuICAgIGlmICghdGFnU2V0IHx8IHRhZ1NldC5pbnN0YW5jZVRhZ0dyb3Vwcy5sZW5ndGggPT09IDApIHtcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuXG4gICAgcmV0dXJuIHtcbiAgICAgIG9uUHJlbWlzZXNUYWdTZXRMaXN0OiB0YWdTZXQuaW5zdGFuY2VUYWdHcm91cHMubWFwKHRhZ0dyb3VwID0+IHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICBvblByZW1pc2VzVGFnR3JvdXA6IHRoaXMudGFnR3JvdXAyVGFnc0FycmF5KHRhZ0dyb3VwKSBhc1xuICAgICAgICAgICAgQ2ZuRGVwbG95bWVudEdyb3VwLlRhZ0ZpbHRlclByb3BlcnR5W10sXG4gICAgICAgIH07XG4gICAgICB9KSxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSB0YWdHcm91cDJUYWdzQXJyYXkodGFnR3JvdXA6IEluc3RhbmNlVGFnR3JvdXApOiBhbnlbXSB7XG4gICAgY29uc3QgdGFnc0luR3JvdXAgPSBbXTtcbiAgICBmb3IgKGNvbnN0IHRhZ0tleSBpbiB0YWdHcm91cCkge1xuICAgICAgaWYgKHRhZ0dyb3VwLmhhc093blByb3BlcnR5KHRhZ0tleSkpIHtcbiAgICAgICAgY29uc3QgdGFnVmFsdWVzID0gdGFnR3JvdXBbdGFnS2V5XTtcbiAgICAgICAgaWYgKHRhZ0tleS5sZW5ndGggPiAwKSB7XG4gICAgICAgICAgaWYgKHRhZ1ZhbHVlcy5sZW5ndGggPiAwKSB7XG4gICAgICAgICAgICBmb3IgKGNvbnN0IHRhZ1ZhbHVlIG9mIHRhZ1ZhbHVlcykge1xuICAgICAgICAgICAgICB0YWdzSW5Hcm91cC5wdXNoKHtcbiAgICAgICAgICAgICAgICBrZXk6IHRhZ0tleSxcbiAgICAgICAgICAgICAgICB2YWx1ZTogdGFnVmFsdWUsXG4gICAgICAgICAgICAgICAgdHlwZTogJ0tFWV9BTkRfVkFMVUUnLFxuICAgICAgICAgICAgICB9KTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgdGFnc0luR3JvdXAucHVzaCh7XG4gICAgICAgICAgICAgIGtleTogdGFnS2V5LFxuICAgICAgICAgICAgICB0eXBlOiAnS0VZX09OTFknLFxuICAgICAgICAgICAgfSk7XG4gICAgICAgICAgfVxuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIGlmICh0YWdWYWx1ZXMubGVuZ3RoID4gMCkge1xuICAgICAgICAgICAgZm9yIChjb25zdCB0YWdWYWx1ZSBvZiB0YWdWYWx1ZXMpIHtcbiAgICAgICAgICAgICAgdGFnc0luR3JvdXAucHVzaCh7XG4gICAgICAgICAgICAgICAgdmFsdWU6IHRhZ1ZhbHVlLFxuICAgICAgICAgICAgICAgIHR5cGU6ICdWQUxVRV9PTkxZJyxcbiAgICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IHNwZWNpZnkgYm90aCBhbiBlbXB0eSBrZXkgYW5kIG5vIHZhbHVlcyBmb3IgYW4gaW5zdGFuY2UgdGFnIGZpbHRlcicpO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gdGFnc0luR3JvdXA7XG4gIH1cbn1cbiJdfQ==