"use strict";
const assert_1 = require("@aws-cdk/assert");
const autoscaling = require("@aws-cdk/aws-autoscaling");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const lbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const cdk = require("@aws-cdk/cdk");
const codedeploy = require("../../lib");
module.exports = {
    'CodeDeploy Server Deployment Group': {
        "can be created by explicitly passing an Application"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.ServerApplication(stack, 'MyApp');
            new codedeploy.ServerDeploymentGroup(stack, 'MyDG', {
                application,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "ApplicationName": {
                    "Ref": "MyApp3CE31C26"
                },
            }));
            test.done();
        },
        'can be imported'(test) {
            const stack = new cdk.Stack();
            const application = codedeploy.ServerApplication.fromServerApplicationName(stack, 'MyApp', 'MyApp');
            const deploymentGroup = codedeploy.ServerDeploymentGroup.fromServerDeploymentGroupAttributes(stack, 'MyDG', {
                application,
                deploymentGroupName: 'MyDG',
            });
            test.notEqual(deploymentGroup, undefined);
            test.done();
        },
        "created with ASGs contains the ASG names"(test) {
            const stack = new cdk.Stack();
            const asg = new autoscaling.AutoScalingGroup(stack, 'ASG', {
                instanceType: new ec2.InstanceTypePair(ec2.InstanceClass.Standard3, ec2.InstanceSize.Small),
                machineImage: new ec2.AmazonLinuxImage(),
                vpc: new ec2.Vpc(stack, 'VPC'),
            });
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                autoScalingGroups: [asg],
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "AutoScalingGroups": [
                    {
                        "Ref": "ASG46ED3070",
                    },
                ]
            }));
            test.done();
        },
        "created without ASGs but adding them later contains the ASG names"(test) {
            const stack = new cdk.Stack();
            const asg = new autoscaling.AutoScalingGroup(stack, 'ASG', {
                instanceType: new ec2.InstanceTypePair(ec2.InstanceClass.Standard3, ec2.InstanceSize.Small),
                machineImage: new ec2.AmazonLinuxImage(),
                vpc: new ec2.Vpc(stack, 'VPC'),
            });
            const deploymentGroup = new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup');
            deploymentGroup.addAutoScalingGroup(asg);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "AutoScalingGroups": [
                    {
                        "Ref": "ASG46ED3070",
                    },
                ]
            }));
            test.done();
        },
        'can be created with an ALB Target Group as the load balancer'(test) {
            const stack = new cdk.Stack();
            const alb = new lbv2.ApplicationLoadBalancer(stack, 'ALB', {
                vpc: new ec2.Vpc(stack, 'VPC'),
            });
            const listener = alb.addListener('Listener', { protocol: lbv2.ApplicationProtocol.Http });
            const targetGroup = listener.addTargets('Fleet', { protocol: lbv2.ApplicationProtocol.Http });
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                loadBalancer: codedeploy.LoadBalancer.application(targetGroup),
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "LoadBalancerInfo": {
                    "TargetGroupInfoList": [
                        {
                            "Name": {
                                "Fn::GetAtt": [
                                    "ALBListenerFleetGroup008CEEE4",
                                    "TargetGroupName",
                                ],
                            },
                        },
                    ],
                },
                "DeploymentStyle": {
                    "DeploymentOption": "WITH_TRAFFIC_CONTROL",
                },
            }));
            test.done();
        },
        'can be created with an NLB Target Group as the load balancer'(test) {
            const stack = new cdk.Stack();
            const nlb = new lbv2.NetworkLoadBalancer(stack, 'NLB', {
                vpc: new ec2.Vpc(stack, 'VPC'),
            });
            const listener = nlb.addListener('Listener', { port: 80 });
            const targetGroup = listener.addTargets('Fleet', { port: 80 });
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                loadBalancer: codedeploy.LoadBalancer.network(targetGroup),
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "LoadBalancerInfo": {
                    "TargetGroupInfoList": [
                        {
                            "Name": {
                                "Fn::GetAtt": [
                                    "NLBListenerFleetGroupB882EC86",
                                    "TargetGroupName",
                                ],
                            },
                        },
                    ],
                },
                "DeploymentStyle": {
                    "DeploymentOption": "WITH_TRAFFIC_CONTROL",
                },
            }));
            test.done();
        },
        'can be created with a single EC2 instance tag set with a single or no value'(test) {
            const stack = new cdk.Stack();
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                ec2InstanceTags: new codedeploy.InstanceTagSet({
                    'some-key': ['some-value'],
                    'other-key': [],
                }),
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "Ec2TagSet": {
                    "Ec2TagSetList": [
                        {
                            "Ec2TagGroup": [
                                {
                                    "Key": "some-key",
                                    "Value": "some-value",
                                    "Type": "KEY_AND_VALUE",
                                },
                                {
                                    "Key": "other-key",
                                    "Type": "KEY_ONLY",
                                },
                            ],
                        },
                    ],
                },
            }));
            test.done();
        },
        'can be created with two on-premise instance tag sets with multiple values or without a key'(test) {
            const stack = new cdk.Stack();
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                onPremiseInstanceTags: new codedeploy.InstanceTagSet({
                    'some-key': ['some-value', 'another-value'],
                }, {
                    '': ['keyless-value'],
                }),
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "OnPremisesTagSet": {
                    "OnPremisesTagSetList": [
                        {
                            "OnPremisesTagGroup": [
                                {
                                    "Key": "some-key",
                                    "Value": "some-value",
                                    "Type": "KEY_AND_VALUE",
                                },
                                {
                                    "Key": "some-key",
                                    "Value": "another-value",
                                    "Type": "KEY_AND_VALUE",
                                },
                            ],
                        },
                        {
                            "OnPremisesTagGroup": [
                                {
                                    "Value": "keyless-value",
                                    "Type": "VALUE_ONLY",
                                },
                            ],
                        },
                    ],
                },
            }));
            test.done();
        },
        'cannot be created with an instance tag set containing a keyless, valueless filter'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                    onPremiseInstanceTags: new codedeploy.InstanceTagSet({
                        '': [],
                    }),
                });
            });
            test.done();
        },
        'cannot be created with an instance tag set containing 4 instance tag groups'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                    onPremiseInstanceTags: new codedeploy.InstanceTagSet({}, {}, {}, {}),
                });
            }, /3/);
            test.done();
        },
        'can have alarms added to it after being created'(test) {
            const stack = new cdk.Stack();
            const alarm = new cloudwatch.Alarm(stack, 'Alarm1', {
                metric: new cloudwatch.Metric({
                    metricName: 'Errors',
                    namespace: 'my.namespace',
                }),
                threshold: 1,
                evaluationPeriods: 1,
            });
            const deploymentGroup = new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup');
            deploymentGroup.addAlarm(alarm);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "AlarmConfiguration": {
                    "Alarms": [
                        {
                            "Name": {
                                "Ref": "Alarm1F9009D71",
                            },
                        },
                    ],
                    "Enabled": true,
                },
            }));
            test.done();
        },
        'only automatically rolls back failed deployments by default'(test) {
            const stack = new cdk.Stack();
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup');
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "AutoRollbackConfiguration": {
                    "Enabled": true,
                    "Events": [
                        "DEPLOYMENT_FAILURE",
                    ],
                },
            }));
            test.done();
        },
        'rolls back alarmed deployments if at least one alarm has been added'(test) {
            const stack = new cdk.Stack();
            const alarm = new cloudwatch.Alarm(stack, 'Alarm1', {
                metric: new cloudwatch.Metric({
                    metricName: 'Errors',
                    namespace: 'my.namespace',
                }),
                threshold: 1,
                evaluationPeriods: 1,
            });
            const deploymentGroup = new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                autoRollback: {
                    failedDeployment: false,
                },
            });
            deploymentGroup.addAlarm(alarm);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "AutoRollbackConfiguration": {
                    "Enabled": true,
                    "Events": [
                        "DEPLOYMENT_STOP_ON_ALARM",
                    ],
                },
            }));
            test.done();
        },
        'setting to roll back on alarms without providing any results in an exception'(test) {
            const stack = new cdk.Stack();
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                autoRollback: {
                    deploymentInAlarm: true,
                },
            });
            test.throws(() => assert_1.SynthUtils.toCloudFormation(stack), /deploymentInAlarm/);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5kZXBsb3ltZW50LWdyb3VwLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVzdC5kZXBsb3ltZW50LWdyb3VwLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSw0Q0FBbUU7QUFDbkUsd0RBQXlEO0FBQ3pELHNEQUF1RDtBQUN2RCx3Q0FBeUM7QUFDekMsNERBQTZEO0FBQzdELG9DQUFxQztBQUVyQyx3Q0FBeUM7QUFJekMsaUJBQVM7SUFDUCxvQ0FBb0MsRUFBRTtRQUNwQyxxREFBcUQsQ0FBQyxJQUFVO1lBQzlELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBRTlCLE1BQU0sV0FBVyxHQUFHLElBQUksVUFBVSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNyRSxJQUFJLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNsRCxXQUFXO2FBQ1osQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtDQUFrQyxFQUFFO2dCQUNoRSxpQkFBaUIsRUFBRTtvQkFDakIsS0FBSyxFQUFFLGVBQWU7aUJBQ3ZCO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsaUJBQWlCLENBQUMsSUFBVTtZQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUU5QixNQUFNLFdBQVcsR0FBRyxVQUFVLENBQUMsaUJBQWlCLENBQUMseUJBQXlCLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNwRyxNQUFNLGVBQWUsR0FBRyxVQUFVLENBQUMscUJBQXFCLENBQUMsbUNBQW1DLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDMUcsV0FBVztnQkFDWCxtQkFBbUIsRUFBRSxNQUFNO2FBQzVCLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxRQUFRLENBQUMsZUFBZSxFQUFFLFNBQVMsQ0FBQyxDQUFDO1lBRTFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCwwQ0FBMEMsQ0FBQyxJQUFVO1lBQ25ELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBRTlCLE1BQU0sR0FBRyxHQUFHLElBQUksV0FBVyxDQUFDLGdCQUFnQixDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7Z0JBQ3pELFlBQVksRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLFNBQVMsRUFBRSxHQUFHLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQztnQkFDM0YsWUFBWSxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixFQUFFO2dCQUN4QyxHQUFHLEVBQUUsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUM7YUFDL0IsQ0FBQyxDQUFDO1lBRUgsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO2dCQUM3RCxpQkFBaUIsRUFBRSxDQUFDLEdBQUcsQ0FBQzthQUN6QixDQUFDLENBQUM7WUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0NBQWtDLEVBQUU7Z0JBQ2hFLG1CQUFtQixFQUFFO29CQUNuQjt3QkFDQSxLQUFLLEVBQUUsYUFBYTtxQkFDbkI7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxtRUFBbUUsQ0FBQyxJQUFVO1lBQzVFLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBRTlCLE1BQU0sR0FBRyxHQUFHLElBQUksV0FBVyxDQUFDLGdCQUFnQixDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7Z0JBQ3pELFlBQVksRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLFNBQVMsRUFBRSxHQUFHLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQztnQkFDM0YsWUFBWSxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixFQUFFO2dCQUN4QyxHQUFHLEVBQUUsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUM7YUFDL0IsQ0FBQyxDQUFDO1lBRUgsTUFBTSxlQUFlLEdBQUcsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLGlCQUFpQixDQUFDLENBQUM7WUFDdkYsZUFBZSxDQUFDLG1CQUFtQixDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBRXpDLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxrQ0FBa0MsRUFBRTtnQkFDaEUsbUJBQW1CLEVBQUU7b0JBQ25CO3dCQUNBLEtBQUssRUFBRSxhQUFhO3FCQUNuQjtpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELDhEQUE4RCxDQUFDLElBQVU7WUFDdkUsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFFOUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxJQUFJLENBQUMsdUJBQXVCLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtnQkFDekQsR0FBRyxFQUFFLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDO2FBQy9CLENBQUMsQ0FBQztZQUNILE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxXQUFXLENBQUMsVUFBVSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1lBQzFGLE1BQU0sV0FBVyxHQUFHLFFBQVEsQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1lBRTlGLElBQUksVUFBVSxDQUFDLHFCQUFxQixDQUFDLEtBQUssRUFBRSxpQkFBaUIsRUFBRTtnQkFDN0QsWUFBWSxFQUFFLFVBQVUsQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFDLFdBQVcsQ0FBQzthQUMvRCxDQUFDLENBQUM7WUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0NBQWtDLEVBQUU7Z0JBQ2hFLGtCQUFrQixFQUFFO29CQUNsQixxQkFBcUIsRUFBRTt3QkFDckI7NEJBQ0UsTUFBTSxFQUFFO2dDQUNOLFlBQVksRUFBRTtvQ0FDWiwrQkFBK0I7b0NBQy9CLGlCQUFpQjtpQ0FDbEI7NkJBQ0Y7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7Z0JBQ0QsaUJBQWlCLEVBQUU7b0JBQ2pCLGtCQUFrQixFQUFFLHNCQUFzQjtpQkFDM0M7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCw4REFBOEQsQ0FBQyxJQUFVO1lBQ3ZFLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBRTlCLE1BQU0sR0FBRyxHQUFHLElBQUksSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7Z0JBQ3JELEdBQUcsRUFBRSxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQzthQUMvQixDQUFDLENBQUM7WUFDSCxNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsV0FBVyxDQUFDLFVBQVUsRUFBRSxFQUFFLElBQUksRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBQzNELE1BQU0sV0FBVyxHQUFHLFFBQVEsQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLEVBQUUsSUFBSSxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUM7WUFFL0QsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO2dCQUM3RCxZQUFZLEVBQUUsVUFBVSxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDO2FBQzNELENBQUMsQ0FBQztZQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxrQ0FBa0MsRUFBRTtnQkFDaEUsa0JBQWtCLEVBQUU7b0JBQ2xCLHFCQUFxQixFQUFFO3dCQUNyQjs0QkFDRSxNQUFNLEVBQUU7Z0NBQ04sWUFBWSxFQUFFO29DQUNaLCtCQUErQjtvQ0FDL0IsaUJBQWlCO2lDQUNsQjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjtnQkFDRCxpQkFBaUIsRUFBRTtvQkFDakIsa0JBQWtCLEVBQUUsc0JBQXNCO2lCQUMzQzthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELDZFQUE2RSxDQUFDLElBQVU7WUFDdEYsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFFOUIsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO2dCQUM3RCxlQUFlLEVBQUUsSUFBSSxVQUFVLENBQUMsY0FBYyxDQUM1QztvQkFDRSxVQUFVLEVBQUUsQ0FBQyxZQUFZLENBQUM7b0JBQzFCLFdBQVcsRUFBRSxFQUFFO2lCQUNoQixDQUNGO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtDQUFrQyxFQUFFO2dCQUNoRSxXQUFXLEVBQUU7b0JBQ1gsZUFBZSxFQUFFO3dCQUNmOzRCQUNFLGFBQWEsRUFBRTtnQ0FDYjtvQ0FDRSxLQUFLLEVBQUUsVUFBVTtvQ0FDakIsT0FBTyxFQUFFLFlBQVk7b0NBQ3JCLE1BQU0sRUFBRSxlQUFlO2lDQUN4QjtnQ0FDRDtvQ0FDRSxLQUFLLEVBQUUsV0FBVztvQ0FDbEIsTUFBTSxFQUFFLFVBQVU7aUNBQ25COzZCQUNGO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsNEZBQTRGLENBQUMsSUFBVTtZQUNyRyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUU5QixJQUFJLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUU7Z0JBQzdELHFCQUFxQixFQUFFLElBQUksVUFBVSxDQUFDLGNBQWMsQ0FDbEQ7b0JBQ0UsVUFBVSxFQUFFLENBQUMsWUFBWSxFQUFFLGVBQWUsQ0FBQztpQkFDNUMsRUFDRDtvQkFDRSxFQUFFLEVBQUUsQ0FBQyxlQUFlLENBQUM7aUJBQ3RCLENBQ0Y7YUFDRixDQUFDLENBQUM7WUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0NBQWtDLEVBQUU7Z0JBQ2hFLGtCQUFrQixFQUFFO29CQUNsQixzQkFBc0IsRUFBRTt3QkFDdEI7NEJBQ0Usb0JBQW9CLEVBQUU7Z0NBQ3BCO29DQUNFLEtBQUssRUFBRSxVQUFVO29DQUNqQixPQUFPLEVBQUUsWUFBWTtvQ0FDckIsTUFBTSxFQUFFLGVBQWU7aUNBQ3hCO2dDQUNEO29DQUNFLEtBQUssRUFBRSxVQUFVO29DQUNqQixPQUFPLEVBQUUsZUFBZTtvQ0FDeEIsTUFBTSxFQUFFLGVBQWU7aUNBQ3hCOzZCQUNGO3lCQUNGO3dCQUNEOzRCQUNFLG9CQUFvQixFQUFFO2dDQUNwQjtvQ0FDRSxPQUFPLEVBQUUsZUFBZTtvQ0FDeEIsTUFBTSxFQUFFLFlBQVk7aUNBQ3JCOzZCQUNGO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsbUZBQW1GLENBQUMsSUFBVTtZQUM1RixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUU5QixJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtnQkFDZixJQUFJLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUU7b0JBQzdELHFCQUFxQixFQUFFLElBQUksVUFBVSxDQUFDLGNBQWMsQ0FBQzt3QkFDbkQsRUFBRSxFQUFFLEVBQUU7cUJBQ1AsQ0FBQztpQkFDSCxDQUFDLENBQUM7WUFDTCxDQUFDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCw2RUFBNkUsQ0FBQyxJQUFVO1lBQ3RGLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBRTlCLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO2dCQUNmLElBQUksVUFBVSxDQUFDLHFCQUFxQixDQUFDLEtBQUssRUFBRSxpQkFBaUIsRUFBRTtvQkFDN0QscUJBQXFCLEVBQUUsSUFBSSxVQUFVLENBQUMsY0FBYyxDQUFDLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsQ0FBQztpQkFDckUsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDO1lBRVIsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELGlEQUFpRCxDQUFDLElBQVU7WUFDMUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFFOUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxVQUFVLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7Z0JBQ2xELE1BQU0sRUFBRSxJQUFJLFVBQVUsQ0FBQyxNQUFNLENBQUM7b0JBQzVCLFVBQVUsRUFBRSxRQUFRO29CQUNwQixTQUFTLEVBQUUsY0FBYztpQkFDMUIsQ0FBQztnQkFDRixTQUFTLEVBQUUsQ0FBQztnQkFDWixpQkFBaUIsRUFBRSxDQUFDO2FBQ3JCLENBQUMsQ0FBQztZQUVILE1BQU0sZUFBZSxHQUFHLElBQUksVUFBVSxDQUFDLHFCQUFxQixDQUFDLEtBQUssRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO1lBQ3ZGLGVBQWUsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7WUFFaEMsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtDQUFrQyxFQUFFO2dCQUNoRSxvQkFBb0IsRUFBRTtvQkFDcEIsUUFBUSxFQUFFO3dCQUNSOzRCQUNFLE1BQU0sRUFBRTtnQ0FDTixLQUFLLEVBQUUsZ0JBQWdCOzZCQUN4Qjt5QkFDRjtxQkFDRjtvQkFDRCxTQUFTLEVBQUUsSUFBSTtpQkFDaEI7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCw2REFBNkQsQ0FBQyxJQUFVO1lBQ3RFLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBRTlCLElBQUksVUFBVSxDQUFDLHFCQUFxQixDQUFDLEtBQUssRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO1lBRS9ELGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxrQ0FBa0MsRUFBRTtnQkFDaEUsMkJBQTJCLEVBQUU7b0JBQzNCLFNBQVMsRUFBRSxJQUFJO29CQUNmLFFBQVEsRUFBRTt3QkFDUixvQkFBb0I7cUJBQ3JCO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQscUVBQXFFLENBQUMsSUFBVTtZQUM5RSxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUU5QixNQUFNLEtBQUssR0FBRyxJQUFJLFVBQVUsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtnQkFDbEQsTUFBTSxFQUFFLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQztvQkFDNUIsVUFBVSxFQUFFLFFBQVE7b0JBQ3BCLFNBQVMsRUFBRSxjQUFjO2lCQUMxQixDQUFDO2dCQUNGLFNBQVMsRUFBRSxDQUFDO2dCQUNaLGlCQUFpQixFQUFFLENBQUM7YUFDckIsQ0FBQyxDQUFDO1lBRUgsTUFBTSxlQUFlLEdBQUcsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO2dCQUNyRixZQUFZLEVBQUU7b0JBQ1osZ0JBQWdCLEVBQUUsS0FBSztpQkFDeEI7YUFDRixDQUFDLENBQUM7WUFDSCxlQUFlLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBRWhDLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxrQ0FBa0MsRUFBRTtnQkFDaEUsMkJBQTJCLEVBQUU7b0JBQzNCLFNBQVMsRUFBRSxJQUFJO29CQUNmLFFBQVEsRUFBRTt3QkFDUiwwQkFBMEI7cUJBQzNCO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsOEVBQThFLENBQUMsSUFBVTtZQUN2RixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUU5QixJQUFJLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUU7Z0JBQzdELFlBQVksRUFBRTtvQkFDWixpQkFBaUIsRUFBRSxJQUFJO2lCQUN4QjthQUNGLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsbUJBQVUsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsRUFBRSxtQkFBbUIsQ0FBQyxDQUFDO1lBQzNFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7S0FDRjtDQUNGLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBleHBlY3QsIGhhdmVSZXNvdXJjZSwgU3ludGhVdGlscyB9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgYXV0b3NjYWxpbmcgPSByZXF1aXJlKCdAYXdzLWNkay9hd3MtYXV0b3NjYWxpbmcnKTtcbmltcG9ydCBjbG91ZHdhdGNoID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWNsb3Vkd2F0Y2gnKTtcbmltcG9ydCBlYzIgPSByZXF1aXJlKCdAYXdzLWNkay9hd3MtZWMyJyk7XG5pbXBvcnQgbGJ2MiA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1lbGFzdGljbG9hZGJhbGFuY2luZ3YyJyk7XG5pbXBvcnQgY2RrID0gcmVxdWlyZSgnQGF3cy1jZGsvY2RrJyk7XG5pbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0IGNvZGVkZXBsb3kgPSByZXF1aXJlKCcuLi8uLi9saWInKTtcblxuLy8gdHNsaW50OmRpc2FibGU6b2JqZWN0LWxpdGVyYWwta2V5LXF1b3Rlc1xuXG5leHBvcnQgPSB7XG4gICdDb2RlRGVwbG95IFNlcnZlciBEZXBsb3ltZW50IEdyb3VwJzoge1xuICAgIFwiY2FuIGJlIGNyZWF0ZWQgYnkgZXhwbGljaXRseSBwYXNzaW5nIGFuIEFwcGxpY2F0aW9uXCIodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAgIGNvbnN0IGFwcGxpY2F0aW9uID0gbmV3IGNvZGVkZXBsb3kuU2VydmVyQXBwbGljYXRpb24oc3RhY2ssICdNeUFwcCcpO1xuICAgICAgbmV3IGNvZGVkZXBsb3kuU2VydmVyRGVwbG95bWVudEdyb3VwKHN0YWNrLCAnTXlERycsIHtcbiAgICAgICAgYXBwbGljYXRpb24sXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6Q29kZURlcGxveTo6RGVwbG95bWVudEdyb3VwJywge1xuICAgICAgICBcIkFwcGxpY2F0aW9uTmFtZVwiOiB7XG4gICAgICAgICAgXCJSZWZcIjogXCJNeUFwcDNDRTMxQzI2XCJcbiAgICAgICAgfSxcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdjYW4gYmUgaW1wb3J0ZWQnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgICBjb25zdCBhcHBsaWNhdGlvbiA9IGNvZGVkZXBsb3kuU2VydmVyQXBwbGljYXRpb24uZnJvbVNlcnZlckFwcGxpY2F0aW9uTmFtZShzdGFjaywgJ015QXBwJywgJ015QXBwJyk7XG4gICAgICBjb25zdCBkZXBsb3ltZW50R3JvdXAgPSBjb2RlZGVwbG95LlNlcnZlckRlcGxveW1lbnRHcm91cC5mcm9tU2VydmVyRGVwbG95bWVudEdyb3VwQXR0cmlidXRlcyhzdGFjaywgJ015REcnLCB7XG4gICAgICAgIGFwcGxpY2F0aW9uLFxuICAgICAgICBkZXBsb3ltZW50R3JvdXBOYW1lOiAnTXlERycsXG4gICAgICB9KTtcblxuICAgICAgdGVzdC5ub3RFcXVhbChkZXBsb3ltZW50R3JvdXAsIHVuZGVmaW5lZCk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICBcImNyZWF0ZWQgd2l0aCBBU0dzIGNvbnRhaW5zIHRoZSBBU0cgbmFtZXNcIih0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgICAgY29uc3QgYXNnID0gbmV3IGF1dG9zY2FsaW5nLkF1dG9TY2FsaW5nR3JvdXAoc3RhY2ssICdBU0cnLCB7XG4gICAgICAgIGluc3RhbmNlVHlwZTogbmV3IGVjMi5JbnN0YW5jZVR5cGVQYWlyKGVjMi5JbnN0YW5jZUNsYXNzLlN0YW5kYXJkMywgZWMyLkluc3RhbmNlU2l6ZS5TbWFsbCksXG4gICAgICAgIG1hY2hpbmVJbWFnZTogbmV3IGVjMi5BbWF6b25MaW51eEltYWdlKCksXG4gICAgICAgIHZwYzogbmV3IGVjMi5WcGMoc3RhY2ssICdWUEMnKSxcbiAgICAgIH0pO1xuXG4gICAgICBuZXcgY29kZWRlcGxveS5TZXJ2ZXJEZXBsb3ltZW50R3JvdXAoc3RhY2ssICdEZXBsb3ltZW50R3JvdXAnLCB7XG4gICAgICAgIGF1dG9TY2FsaW5nR3JvdXBzOiBbYXNnXSxcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpDb2RlRGVwbG95OjpEZXBsb3ltZW50R3JvdXAnLCB7XG4gICAgICAgIFwiQXV0b1NjYWxpbmdHcm91cHNcIjogW1xuICAgICAgICAgIHtcbiAgICAgICAgICBcIlJlZlwiOiBcIkFTRzQ2RUQzMDcwXCIsXG4gICAgICAgICAgfSxcbiAgICAgICAgXVxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgXCJjcmVhdGVkIHdpdGhvdXQgQVNHcyBidXQgYWRkaW5nIHRoZW0gbGF0ZXIgY29udGFpbnMgdGhlIEFTRyBuYW1lc1wiKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgICBjb25zdCBhc2cgPSBuZXcgYXV0b3NjYWxpbmcuQXV0b1NjYWxpbmdHcm91cChzdGFjaywgJ0FTRycsIHtcbiAgICAgICAgaW5zdGFuY2VUeXBlOiBuZXcgZWMyLkluc3RhbmNlVHlwZVBhaXIoZWMyLkluc3RhbmNlQ2xhc3MuU3RhbmRhcmQzLCBlYzIuSW5zdGFuY2VTaXplLlNtYWxsKSxcbiAgICAgICAgbWFjaGluZUltYWdlOiBuZXcgZWMyLkFtYXpvbkxpbnV4SW1hZ2UoKSxcbiAgICAgICAgdnBjOiBuZXcgZWMyLlZwYyhzdGFjaywgJ1ZQQycpLFxuICAgICAgfSk7XG5cbiAgICAgIGNvbnN0IGRlcGxveW1lbnRHcm91cCA9IG5ldyBjb2RlZGVwbG95LlNlcnZlckRlcGxveW1lbnRHcm91cChzdGFjaywgJ0RlcGxveW1lbnRHcm91cCcpO1xuICAgICAgZGVwbG95bWVudEdyb3VwLmFkZEF1dG9TY2FsaW5nR3JvdXAoYXNnKTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6Q29kZURlcGxveTo6RGVwbG95bWVudEdyb3VwJywge1xuICAgICAgICBcIkF1dG9TY2FsaW5nR3JvdXBzXCI6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgXCJSZWZcIjogXCJBU0c0NkVEMzA3MFwiLFxuICAgICAgICAgIH0sXG4gICAgICAgIF1cbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdjYW4gYmUgY3JlYXRlZCB3aXRoIGFuIEFMQiBUYXJnZXQgR3JvdXAgYXMgdGhlIGxvYWQgYmFsYW5jZXInKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgICBjb25zdCBhbGIgPSBuZXcgbGJ2Mi5BcHBsaWNhdGlvbkxvYWRCYWxhbmNlcihzdGFjaywgJ0FMQicsIHtcbiAgICAgICAgdnBjOiBuZXcgZWMyLlZwYyhzdGFjaywgJ1ZQQycpLFxuICAgICAgfSk7XG4gICAgICBjb25zdCBsaXN0ZW5lciA9IGFsYi5hZGRMaXN0ZW5lcignTGlzdGVuZXInLCB7IHByb3RvY29sOiBsYnYyLkFwcGxpY2F0aW9uUHJvdG9jb2wuSHR0cCB9KTtcbiAgICAgIGNvbnN0IHRhcmdldEdyb3VwID0gbGlzdGVuZXIuYWRkVGFyZ2V0cygnRmxlZXQnLCB7IHByb3RvY29sOiBsYnYyLkFwcGxpY2F0aW9uUHJvdG9jb2wuSHR0cCB9KTtcblxuICAgICAgbmV3IGNvZGVkZXBsb3kuU2VydmVyRGVwbG95bWVudEdyb3VwKHN0YWNrLCAnRGVwbG95bWVudEdyb3VwJywge1xuICAgICAgICBsb2FkQmFsYW5jZXI6IGNvZGVkZXBsb3kuTG9hZEJhbGFuY2VyLmFwcGxpY2F0aW9uKHRhcmdldEdyb3VwKSxcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpDb2RlRGVwbG95OjpEZXBsb3ltZW50R3JvdXAnLCB7XG4gICAgICAgIFwiTG9hZEJhbGFuY2VySW5mb1wiOiB7XG4gICAgICAgICAgXCJUYXJnZXRHcm91cEluZm9MaXN0XCI6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgXCJOYW1lXCI6IHtcbiAgICAgICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICAgICAgXCJBTEJMaXN0ZW5lckZsZWV0R3JvdXAwMDhDRUVFNFwiLFxuICAgICAgICAgICAgICAgICAgXCJUYXJnZXRHcm91cE5hbWVcIixcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgICBcIkRlcGxveW1lbnRTdHlsZVwiOiB7XG4gICAgICAgICAgXCJEZXBsb3ltZW50T3B0aW9uXCI6IFwiV0lUSF9UUkFGRklDX0NPTlRST0xcIixcbiAgICAgICAgfSxcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdjYW4gYmUgY3JlYXRlZCB3aXRoIGFuIE5MQiBUYXJnZXQgR3JvdXAgYXMgdGhlIGxvYWQgYmFsYW5jZXInKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgICBjb25zdCBubGIgPSBuZXcgbGJ2Mi5OZXR3b3JrTG9hZEJhbGFuY2VyKHN0YWNrLCAnTkxCJywge1xuICAgICAgICB2cGM6IG5ldyBlYzIuVnBjKHN0YWNrLCAnVlBDJyksXG4gICAgICB9KTtcbiAgICAgIGNvbnN0IGxpc3RlbmVyID0gbmxiLmFkZExpc3RlbmVyKCdMaXN0ZW5lcicsIHsgcG9ydDogODAgfSk7XG4gICAgICBjb25zdCB0YXJnZXRHcm91cCA9IGxpc3RlbmVyLmFkZFRhcmdldHMoJ0ZsZWV0JywgeyBwb3J0OiA4MCB9KTtcblxuICAgICAgbmV3IGNvZGVkZXBsb3kuU2VydmVyRGVwbG95bWVudEdyb3VwKHN0YWNrLCAnRGVwbG95bWVudEdyb3VwJywge1xuICAgICAgICBsb2FkQmFsYW5jZXI6IGNvZGVkZXBsb3kuTG9hZEJhbGFuY2VyLm5ldHdvcmsodGFyZ2V0R3JvdXApLFxuICAgICAgfSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkNvZGVEZXBsb3k6OkRlcGxveW1lbnRHcm91cCcsIHtcbiAgICAgICAgXCJMb2FkQmFsYW5jZXJJbmZvXCI6IHtcbiAgICAgICAgICBcIlRhcmdldEdyb3VwSW5mb0xpc3RcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcIk5hbWVcIjoge1xuICAgICAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgICAgICBcIk5MQkxpc3RlbmVyRmxlZXRHcm91cEI4ODJFQzg2XCIsXG4gICAgICAgICAgICAgICAgICBcIlRhcmdldEdyb3VwTmFtZVwiLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICAgIFwiRGVwbG95bWVudFN0eWxlXCI6IHtcbiAgICAgICAgICBcIkRlcGxveW1lbnRPcHRpb25cIjogXCJXSVRIX1RSQUZGSUNfQ09OVFJPTFwiLFxuICAgICAgICB9LFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2NhbiBiZSBjcmVhdGVkIHdpdGggYSBzaW5nbGUgRUMyIGluc3RhbmNlIHRhZyBzZXQgd2l0aCBhIHNpbmdsZSBvciBubyB2YWx1ZScodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAgIG5ldyBjb2RlZGVwbG95LlNlcnZlckRlcGxveW1lbnRHcm91cChzdGFjaywgJ0RlcGxveW1lbnRHcm91cCcsIHtcbiAgICAgICAgZWMySW5zdGFuY2VUYWdzOiBuZXcgY29kZWRlcGxveS5JbnN0YW5jZVRhZ1NldChcbiAgICAgICAgICB7XG4gICAgICAgICAgICAnc29tZS1rZXknOiBbJ3NvbWUtdmFsdWUnXSxcbiAgICAgICAgICAgICdvdGhlci1rZXknOiBbXSxcbiAgICAgICAgICB9LFxuICAgICAgICApLFxuICAgICAgfSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkNvZGVEZXBsb3k6OkRlcGxveW1lbnRHcm91cCcsIHtcbiAgICAgICAgXCJFYzJUYWdTZXRcIjoge1xuICAgICAgICAgIFwiRWMyVGFnU2V0TGlzdFwiOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFwiRWMyVGFnR3JvdXBcIjogW1xuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIFwiS2V5XCI6IFwic29tZS1rZXlcIixcbiAgICAgICAgICAgICAgICAgIFwiVmFsdWVcIjogXCJzb21lLXZhbHVlXCIsXG4gICAgICAgICAgICAgICAgICBcIlR5cGVcIjogXCJLRVlfQU5EX1ZBTFVFXCIsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICBcIktleVwiOiBcIm90aGVyLWtleVwiLFxuICAgICAgICAgICAgICAgICAgXCJUeXBlXCI6IFwiS0VZX09OTFlcIixcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2NhbiBiZSBjcmVhdGVkIHdpdGggdHdvIG9uLXByZW1pc2UgaW5zdGFuY2UgdGFnIHNldHMgd2l0aCBtdWx0aXBsZSB2YWx1ZXMgb3Igd2l0aG91dCBhIGtleScodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAgIG5ldyBjb2RlZGVwbG95LlNlcnZlckRlcGxveW1lbnRHcm91cChzdGFjaywgJ0RlcGxveW1lbnRHcm91cCcsIHtcbiAgICAgICAgb25QcmVtaXNlSW5zdGFuY2VUYWdzOiBuZXcgY29kZWRlcGxveS5JbnN0YW5jZVRhZ1NldChcbiAgICAgICAgICB7XG4gICAgICAgICAgICAnc29tZS1rZXknOiBbJ3NvbWUtdmFsdWUnLCAnYW5vdGhlci12YWx1ZSddLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgJyc6IFsna2V5bGVzcy12YWx1ZSddLFxuICAgICAgICAgIH0sXG4gICAgICAgICksXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6Q29kZURlcGxveTo6RGVwbG95bWVudEdyb3VwJywge1xuICAgICAgICBcIk9uUHJlbWlzZXNUYWdTZXRcIjoge1xuICAgICAgICAgIFwiT25QcmVtaXNlc1RhZ1NldExpc3RcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcIk9uUHJlbWlzZXNUYWdHcm91cFwiOiBbXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgXCJLZXlcIjogXCJzb21lLWtleVwiLFxuICAgICAgICAgICAgICAgICAgXCJWYWx1ZVwiOiBcInNvbWUtdmFsdWVcIixcbiAgICAgICAgICAgICAgICAgIFwiVHlwZVwiOiBcIktFWV9BTkRfVkFMVUVcIixcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIFwiS2V5XCI6IFwic29tZS1rZXlcIixcbiAgICAgICAgICAgICAgICAgIFwiVmFsdWVcIjogXCJhbm90aGVyLXZhbHVlXCIsXG4gICAgICAgICAgICAgICAgICBcIlR5cGVcIjogXCJLRVlfQU5EX1ZBTFVFXCIsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFwiT25QcmVtaXNlc1RhZ0dyb3VwXCI6IFtcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICBcIlZhbHVlXCI6IFwia2V5bGVzcy12YWx1ZVwiLFxuICAgICAgICAgICAgICAgICAgXCJUeXBlXCI6IFwiVkFMVUVfT05MWVwiLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnY2Fubm90IGJlIGNyZWF0ZWQgd2l0aCBhbiBpbnN0YW5jZSB0YWcgc2V0IGNvbnRhaW5pbmcgYSBrZXlsZXNzLCB2YWx1ZWxlc3MgZmlsdGVyJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgICBuZXcgY29kZWRlcGxveS5TZXJ2ZXJEZXBsb3ltZW50R3JvdXAoc3RhY2ssICdEZXBsb3ltZW50R3JvdXAnLCB7XG4gICAgICAgICAgb25QcmVtaXNlSW5zdGFuY2VUYWdzOiBuZXcgY29kZWRlcGxveS5JbnN0YW5jZVRhZ1NldCh7XG4gICAgICAgICAgICAnJzogW10sXG4gICAgICAgICAgfSksXG4gICAgICAgIH0pO1xuICAgICAgfSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnY2Fubm90IGJlIGNyZWF0ZWQgd2l0aCBhbiBpbnN0YW5jZSB0YWcgc2V0IGNvbnRhaW5pbmcgNCBpbnN0YW5jZSB0YWcgZ3JvdXBzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgICBuZXcgY29kZWRlcGxveS5TZXJ2ZXJEZXBsb3ltZW50R3JvdXAoc3RhY2ssICdEZXBsb3ltZW50R3JvdXAnLCB7XG4gICAgICAgICAgb25QcmVtaXNlSW5zdGFuY2VUYWdzOiBuZXcgY29kZWRlcGxveS5JbnN0YW5jZVRhZ1NldCh7fSwge30sIHt9LCB7fSksXG4gICAgICAgIH0pO1xuICAgICAgfSwgLzMvKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdjYW4gaGF2ZSBhbGFybXMgYWRkZWQgdG8gaXQgYWZ0ZXIgYmVpbmcgY3JlYXRlZCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAgIGNvbnN0IGFsYXJtID0gbmV3IGNsb3Vkd2F0Y2guQWxhcm0oc3RhY2ssICdBbGFybTEnLCB7XG4gICAgICAgIG1ldHJpYzogbmV3IGNsb3Vkd2F0Y2guTWV0cmljKHtcbiAgICAgICAgICBtZXRyaWNOYW1lOiAnRXJyb3JzJyxcbiAgICAgICAgICBuYW1lc3BhY2U6ICdteS5uYW1lc3BhY2UnLFxuICAgICAgICB9KSxcbiAgICAgICAgdGhyZXNob2xkOiAxLFxuICAgICAgICBldmFsdWF0aW9uUGVyaW9kczogMSxcbiAgICAgIH0pO1xuXG4gICAgICBjb25zdCBkZXBsb3ltZW50R3JvdXAgPSBuZXcgY29kZWRlcGxveS5TZXJ2ZXJEZXBsb3ltZW50R3JvdXAoc3RhY2ssICdEZXBsb3ltZW50R3JvdXAnKTtcbiAgICAgIGRlcGxveW1lbnRHcm91cC5hZGRBbGFybShhbGFybSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkNvZGVEZXBsb3k6OkRlcGxveW1lbnRHcm91cCcsIHtcbiAgICAgICAgXCJBbGFybUNvbmZpZ3VyYXRpb25cIjoge1xuICAgICAgICAgIFwiQWxhcm1zXCI6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgXCJOYW1lXCI6IHtcbiAgICAgICAgICAgICAgICBcIlJlZlwiOiBcIkFsYXJtMUY5MDA5RDcxXCIsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgICAgXCJFbmFibGVkXCI6IHRydWUsXG4gICAgICAgIH0sXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnb25seSBhdXRvbWF0aWNhbGx5IHJvbGxzIGJhY2sgZmFpbGVkIGRlcGxveW1lbnRzIGJ5IGRlZmF1bHQnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgICBuZXcgY29kZWRlcGxveS5TZXJ2ZXJEZXBsb3ltZW50R3JvdXAoc3RhY2ssICdEZXBsb3ltZW50R3JvdXAnKTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6Q29kZURlcGxveTo6RGVwbG95bWVudEdyb3VwJywge1xuICAgICAgICBcIkF1dG9Sb2xsYmFja0NvbmZpZ3VyYXRpb25cIjoge1xuICAgICAgICAgIFwiRW5hYmxlZFwiOiB0cnVlLFxuICAgICAgICAgIFwiRXZlbnRzXCI6IFtcbiAgICAgICAgICAgIFwiREVQTE9ZTUVOVF9GQUlMVVJFXCIsXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdyb2xscyBiYWNrIGFsYXJtZWQgZGVwbG95bWVudHMgaWYgYXQgbGVhc3Qgb25lIGFsYXJtIGhhcyBiZWVuIGFkZGVkJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgICAgY29uc3QgYWxhcm0gPSBuZXcgY2xvdWR3YXRjaC5BbGFybShzdGFjaywgJ0FsYXJtMScsIHtcbiAgICAgICAgbWV0cmljOiBuZXcgY2xvdWR3YXRjaC5NZXRyaWMoe1xuICAgICAgICAgIG1ldHJpY05hbWU6ICdFcnJvcnMnLFxuICAgICAgICAgIG5hbWVzcGFjZTogJ215Lm5hbWVzcGFjZScsXG4gICAgICAgIH0pLFxuICAgICAgICB0aHJlc2hvbGQ6IDEsXG4gICAgICAgIGV2YWx1YXRpb25QZXJpb2RzOiAxLFxuICAgICAgfSk7XG5cbiAgICAgIGNvbnN0IGRlcGxveW1lbnRHcm91cCA9IG5ldyBjb2RlZGVwbG95LlNlcnZlckRlcGxveW1lbnRHcm91cChzdGFjaywgJ0RlcGxveW1lbnRHcm91cCcsIHtcbiAgICAgICAgYXV0b1JvbGxiYWNrOiB7XG4gICAgICAgICAgZmFpbGVkRGVwbG95bWVudDogZmFsc2UsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICAgIGRlcGxveW1lbnRHcm91cC5hZGRBbGFybShhbGFybSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkNvZGVEZXBsb3k6OkRlcGxveW1lbnRHcm91cCcsIHtcbiAgICAgICAgXCJBdXRvUm9sbGJhY2tDb25maWd1cmF0aW9uXCI6IHtcbiAgICAgICAgICBcIkVuYWJsZWRcIjogdHJ1ZSxcbiAgICAgICAgICBcIkV2ZW50c1wiOiBbXG4gICAgICAgICAgICBcIkRFUExPWU1FTlRfU1RPUF9PTl9BTEFSTVwiLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnc2V0dGluZyB0byByb2xsIGJhY2sgb24gYWxhcm1zIHdpdGhvdXQgcHJvdmlkaW5nIGFueSByZXN1bHRzIGluIGFuIGV4Y2VwdGlvbicodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAgIG5ldyBjb2RlZGVwbG95LlNlcnZlckRlcGxveW1lbnRHcm91cChzdGFjaywgJ0RlcGxveW1lbnRHcm91cCcsIHtcbiAgICAgICAgYXV0b1JvbGxiYWNrOiB7XG4gICAgICAgICAgZGVwbG95bWVudEluQWxhcm06IHRydWUsXG4gICAgICAgIH0sXG4gICAgICB9KTtcblxuICAgICAgdGVzdC50aHJvd3MoKCkgPT4gU3ludGhVdGlscy50b0Nsb3VkRm9ybWF0aW9uKHN0YWNrKSwgL2RlcGxveW1lbnRJbkFsYXJtLyk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICB9LFxufTtcbiJdfQ==