import { IResource, Resource } from 'aws-cdk-lib/core';
import { Construct } from 'constructs';
import { IPolicyStore } from './policy-store';
import { IPolicyTemplate } from './policy-template';
export interface EntityIdentifierProperty {
    /**
     * The identifier of an entity.
     */
    readonly entityId: string;
    /**
     * The type of an entity.
     */
    readonly entityType: string;
}
export interface TemplateLinkedPolicyDefinitionProperty {
    /**
     * The unique identifier of the policy template used to create this policy.
     */
    readonly policyTemplate: IPolicyTemplate;
    /**
     * The principal associated with this template-linked policy.
     *
     * @default - No Principal. It is set to unspecified.
     */
    readonly principal?: EntityIdentifierProperty;
    /**
     * The resource associated with this template-linked policy.
     *
     * @default - No Resource. It is set to unspecified.
     */
    readonly resource?: EntityIdentifierProperty;
}
export interface StaticPolicyDefinitionProperty {
    /**
     * The policy content of the static policy, written in the Cedar policy language.
     */
    readonly statement: string;
    /**
     * The description of the static policy.
     *
     * @default - Empty description.
     */
    readonly description?: string;
}
export interface PolicyDefinitionProperty {
    /**
     * A structure that describes a static policy.
     *
     * @default - Static must be set for policies created from a static definition. Otherwise, use template linked definitions.
     */
    readonly static?: StaticPolicyDefinitionProperty;
    /**
     * A structure that describes a policy that was instantiated from a template.
     *
     * @default - Template linked must be set for policies created from a static definition. Otherwise, use static definitions.
     */
    readonly templateLinked?: TemplateLinkedPolicyDefinitionProperty;
}
export interface IPolicy extends IResource {
    /**
     * The unique ID of the new or updated policy.
     *
     * @attribute
     */
    readonly policyId: string;
    /**
     * The type of the policy. This is one of the following values: Static or TemplateLinked.
     *
     * @attribute
     */
    readonly policyType: PolicyType;
}
export interface PolicyAttributes {
    /**
     * The unique ID of the new or updated policy.
     */
    readonly policyId: string;
    /**
     * The type of the policy. This is one of the following values: Static or TemplateLinked
     *
     * @default - Static
     */
    readonly policyType?: PolicyType;
}
export interface PolicyProps {
    /**
     * Specifies the policy type and content to use for the new or updated policy.
     * The definition structure must include either a Static or a TemplateLinked element.
     */
    readonly definition: PolicyDefinitionProperty;
    /**
     * The policy store that contains the policy.
     */
    readonly policyStore: IPolicyStore;
}
declare abstract class PolicyBase extends Resource implements IPolicy {
    abstract readonly policyId: string;
    abstract readonly policyType: PolicyType;
}
export declare class Policy extends PolicyBase {
    /**
     *  Import a policy into the CDK using its id.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct id.
     * @param policyId The policy id.
     */
    static fromPolicyId(scope: Construct, id: string, policyId: string): IPolicy;
    /**
     * Import a Policy construct from attributes.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct id.
     * @param attrs A `PolicyAttributes` object.
     * @returns
     */
    static fromPolicyAttributes(scope: Construct, id: string, attrs: PolicyAttributes): IPolicy;
    readonly policyId: string;
    readonly policyType: PolicyType;
    readonly definition: PolicyDefinitionProperty;
    readonly policyStoreId: string;
    private readonly policy;
    constructor(scope: Construct, id: string, props: PolicyProps);
}
/**
 * PolicyType options
 */
export declare enum PolicyType {
    STATIC = "Static",
    TEMPLATELINKED = "TemplateLinked"
}
export {};
