"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Protocol = exports.Service = exports.ServiceType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base = require("./base");
const k8s = require("./imports/k8s");
const ingress = require("./ingress");
/**
 * For some parts of your application (for example, frontends) you may want to expose a Service onto an
 * external IP address, that's outside of your cluster.
 * Kubernetes ServiceTypes allow you to specify what kind of Service you want.
 * The default is ClusterIP.
 */
var ServiceType;
(function (ServiceType) {
    /**
     * Exposes the Service on a cluster-internal IP.
     * Choosing this value makes the Service only reachable from within the cluster.
     * This is the default ServiceType
     */
    ServiceType["CLUSTER_IP"] = "ClusterIP";
    /**
     * Exposes the Service on each Node's IP at a static port (the NodePort).
     * A ClusterIP Service, to which the NodePort Service routes, is automatically created.
     * You'll be able to contact the NodePort Service, from outside the cluster,
     * by requesting <NodeIP>:<NodePort>.
     */
    ServiceType["NODE_PORT"] = "NodePort";
    /**
     * Exposes the Service externally using a cloud provider's load balancer.
     * NodePort and ClusterIP Services, to which the external load balancer routes,
     * are automatically created.
     */
    ServiceType["LOAD_BALANCER"] = "LoadBalancer";
    /**
     * Maps the Service to the contents of the externalName field (e.g. foo.bar.example.com), by returning a CNAME record with its value.
     * No proxying of any kind is set up.
     *
     * > Note: You need either kube-dns version 1.7 or CoreDNS version 0.0.8 or higher to use the ExternalName type.
     */
    ServiceType["EXTERNAL_NAME"] = "ExternalName";
})(ServiceType = exports.ServiceType || (exports.ServiceType = {}));
/**
 * An abstract way to expose an application running on a set of Pods as a network service.
 * With Kubernetes you don't need to modify your application to use an unfamiliar service discovery mechanism.
 * Kubernetes gives Pods their own IP addresses and a single DNS name for a set of Pods, and can load-balance across them.
 *
 * For example, consider a stateless image-processing backend which is running with 3 replicas. Those replicas are fungible—frontends do not care which backend they use.
 * While the actual Pods that compose the backend set may change, the frontend clients should not need to be aware of that,
 * nor should they need to keep track of the set of backends themselves.
 * The Service abstraction enables this decoupling.
 *
 * If you're able to use Kubernetes APIs for service discovery in your application, you can query the API server for Endpoints,
 * that get updated whenever the set of Pods in a Service changes. For non-native applications, Kubernetes offers ways to place a network port
 * or load balancer in between your application and the backend Pods.
 */
class Service extends base.Resource {
    constructor(scope, id, props = {}) {
        var _b, _c, _d;
        super(scope, id);
        this.resourceType = 'services';
        this.apiObject = new k8s.KubeService(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.clusterIP = props.clusterIP;
        this.externalName = props.externalName;
        if (props.externalName !== undefined) {
            this.type = ServiceType.EXTERNAL_NAME;
        }
        else {
            this.type = (_b = props.type) !== null && _b !== void 0 ? _b : ServiceType.CLUSTER_IP;
        }
        this._externalIPs = (_c = props.externalIPs) !== null && _c !== void 0 ? _c : [];
        this._ports = [];
        this._selector = {};
        this._loadBalancerSourceRanges = props.loadBalancerSourceRanges;
        if (props.selector) {
            this.select(props.selector);
        }
        for (const port of (_d = props.ports) !== null && _d !== void 0 ? _d : []) {
            this.bind(port.port, port);
        }
    }
    /**
     * Expose a service via an ingress using the specified path.
     *
     * @param path The path to expose the service under.
     * @param options Additional options.
     *
     * @returns The `Ingress` resource that was used.
     */
    exposeViaIngress(path, options = {}) {
        var _b;
        const ingr = (_b = options.ingress) !== null && _b !== void 0 ? _b : new ingress.Ingress(this, 'Ingress');
        ingr.addRule(path, ingress.IngressBackend.fromService(this), options.pathType);
        return ingr;
    }
    /**
     * Ports for this service.
     *
     * Use `bind()` to bind additional service ports.
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Configure a port the service will bind to.
     * This method can be called multiple times.
     *
     * @param port The port definition.
     */
    bind(port, options = {}) {
        this._ports.push({ ...options, port });
    }
    /**
     * Require this service to select pods matching the selector.
     *
     * Note that invoking this method multiple times acts as an AND operator
     * on the resulting labels.
     */
    select(selector) {
        var _b;
        const labels = (_b = selector.toPodSelectorConfig().labelSelector._toKube().matchLabels) !== null && _b !== void 0 ? _b : {};
        for (const [key, value] of Object.entries(labels)) {
            this._selector[key] = value;
        }
    }
    /**
     * Require this service to select pods with this label.
     *
     * Note that invoking this method multiple times acts as an AND operator
     * on the resulting labels.
     */
    selectLabel(key, value) {
        this._selector[key] = value;
    }
    /**
     * @internal
     */
    _toKube() {
        if (this._ports.length === 0 && this.type !== ServiceType.EXTERNAL_NAME) {
            throw new Error('A service must be configured with a port');
        }
        if (this.type === ServiceType.EXTERNAL_NAME && this.externalName === undefined) {
            throw new Error('A service with type EXTERNAL_NAME requires an externalName prop');
        }
        const ports = [];
        for (const port of this._ports) {
            ports.push({
                name: port.name,
                port: port.port,
                targetPort: port.targetPort ? k8s.IntOrString.fromNumber(port.targetPort) : undefined,
                nodePort: port.nodePort,
                protocol: port.protocol ? this._portProtocolToKube(port.protocol) : undefined,
            });
        }
        const serviceType = this._serviceTypeToKube(this.type);
        return this.type !== ServiceType.EXTERNAL_NAME ? {
            clusterIp: this.clusterIP,
            externalIPs: this._externalIPs,
            externalName: this.externalName,
            type: serviceType,
            selector: this._selector,
            ports: ports,
            loadBalancerSourceRanges: this._loadBalancerSourceRanges,
        } : {
            type: serviceType,
            externalName: this.externalName,
        };
    }
    _portProtocolToKube(protocol) {
        switch (protocol) {
            case Protocol.SCTP:
                return k8s.IoK8SApiCoreV1ServicePortProtocol.SCTP;
            case Protocol.TCP:
                return k8s.IoK8SApiCoreV1ServicePortProtocol.TCP;
            case Protocol.UDP:
                return k8s.IoK8SApiCoreV1ServicePortProtocol.UDP;
            default:
                throw new Error(`Unsupported port protocol: ${protocol}`);
        }
    }
    _serviceTypeToKube(serviceType) {
        switch (serviceType) {
            case ServiceType.CLUSTER_IP:
                return k8s.IoK8SApiCoreV1ServiceSpecType.CLUSTER_IP;
            case ServiceType.EXTERNAL_NAME:
                return k8s.IoK8SApiCoreV1ServiceSpecType.EXTERNAL_NAME;
            case ServiceType.LOAD_BALANCER:
                return k8s.IoK8SApiCoreV1ServiceSpecType.LOAD_BALANCER;
            case ServiceType.NODE_PORT:
                return k8s.IoK8SApiCoreV1ServiceSpecType.NODE_PORT;
            default:
                throw new Error(`Unsupported service type: ${serviceType}`);
        }
    }
}
exports.Service = Service;
_a = JSII_RTTI_SYMBOL_1;
Service[_a] = { fqn: "cdk8s-plus-23.Service", version: "2.0.0-rc.6" };
var Protocol;
(function (Protocol) {
    Protocol["TCP"] = "TCP";
    Protocol["UDP"] = "UDP";
    Protocol["SCTP"] = "SCTP";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2VydmljZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9zZXJ2aWNlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsaUNBQXdDO0FBRXhDLCtCQUErQjtBQUMvQixxQ0FBcUM7QUFDckMscUNBQXFDO0FBaUhyQzs7Ozs7R0FLRztBQUNILElBQVksV0ErQlg7QUEvQkQsV0FBWSxXQUFXO0lBRXJCOzs7O09BSUc7SUFDSCx1Q0FBd0IsQ0FBQTtJQUV4Qjs7Ozs7T0FLRztJQUNILHFDQUFzQixDQUFBO0lBRXRCOzs7O09BSUc7SUFDSCw2Q0FBOEIsQ0FBQTtJQUU5Qjs7Ozs7T0FLRztJQUNILDZDQUE4QixDQUFBO0FBQ2hDLENBQUMsRUEvQlcsV0FBVyxHQUFYLG1CQUFXLEtBQVgsbUJBQVcsUUErQnRCO0FBY0Q7Ozs7Ozs7Ozs7Ozs7R0FhRztBQUNILE1BQWEsT0FBUSxTQUFRLElBQUksQ0FBQyxRQUFRO0lBOEJ4QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQXNCLEVBQUU7O1FBQ2hFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFSSCxpQkFBWSxHQUFHLFVBQVUsQ0FBQztRQVV4QyxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksR0FBRyxDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ3JELFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTtZQUN4QixJQUFJLEVBQUUsWUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQztTQUNsRCxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUM7UUFDakMsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDO1FBRXZDLElBQUksS0FBSyxDQUFDLFlBQVksS0FBSyxTQUFTLEVBQUU7WUFDcEMsSUFBSSxDQUFDLElBQUksR0FBRyxXQUFXLENBQUMsYUFBYSxDQUFDO1NBQ3ZDO2FBQU07WUFDTCxJQUFJLENBQUMsSUFBSSxTQUFHLEtBQUssQ0FBQyxJQUFJLG1DQUFJLFdBQVcsQ0FBQyxVQUFVLENBQUM7U0FDbEQ7UUFFRCxJQUFJLENBQUMsWUFBWSxTQUFHLEtBQUssQ0FBQyxXQUFXLG1DQUFJLEVBQUUsQ0FBQztRQUM1QyxJQUFJLENBQUMsTUFBTSxHQUFHLEVBQUUsQ0FBQztRQUNqQixJQUFJLENBQUMsU0FBUyxHQUFHLEVBQUcsQ0FBQztRQUNyQixJQUFJLENBQUMseUJBQXlCLEdBQUcsS0FBSyxDQUFDLHdCQUF3QixDQUFDO1FBRWhFLElBQUksS0FBSyxDQUFDLFFBQVEsRUFBRTtZQUNsQixJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQztTQUM3QjtRQUVELEtBQUssTUFBTSxJQUFJLFVBQUksS0FBSyxDQUFDLEtBQUssbUNBQUksRUFBRSxFQUFFO1lBQ3BDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztTQUM1QjtJQUNILENBQUM7SUFFRDs7Ozs7OztPQU9HO0lBQ0ksZ0JBQWdCLENBQUMsSUFBWSxFQUFFLFVBQTBDLEVBQUU7O1FBQ2hGLE1BQU0sSUFBSSxTQUFHLE9BQU8sQ0FBQyxPQUFPLG1DQUFJLElBQUksT0FBTyxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDckUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLEVBQUUsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQy9FLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxJQUFXLEtBQUs7UUFDZCxPQUFPLENBQUMsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDMUIsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksSUFBSSxDQUFDLElBQVksRUFBRSxVQUE4QixFQUFHO1FBQ3pELElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsR0FBRyxPQUFPLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxNQUFNLENBQUMsUUFBMEI7O1FBQ3RDLE1BQU0sTUFBTSxTQUFHLFFBQVEsQ0FBQyxtQkFBbUIsRUFBRSxDQUFDLGFBQWEsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxXQUFXLG1DQUFJLEVBQUUsQ0FBQztRQUN4RixLQUFLLE1BQU0sQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRTtZQUNqRCxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQztTQUM3QjtJQUNILENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLFdBQVcsQ0FBQyxHQUFXLEVBQUUsS0FBYTtRQUMzQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQztJQUM5QixDQUFDO0lBRUQ7O09BRUc7SUFDSSxPQUFPO1FBQ1osSUFBSSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sS0FBSyxDQUFDLElBQUksSUFBSSxDQUFDLElBQUksS0FBSyxXQUFXLENBQUMsYUFBYSxFQUFFO1lBQ3ZFLE1BQU0sSUFBSSxLQUFLLENBQUMsMENBQTBDLENBQUMsQ0FBQztTQUM3RDtRQUVELElBQUksSUFBSSxDQUFDLElBQUksS0FBSyxXQUFXLENBQUMsYUFBYSxJQUFJLElBQUksQ0FBQyxZQUFZLEtBQUssU0FBUyxFQUFFO1lBQzlFLE1BQU0sSUFBSSxLQUFLLENBQUMsaUVBQWlFLENBQUMsQ0FBQztTQUNwRjtRQUVELE1BQU0sS0FBSyxHQUFzQixFQUFFLENBQUM7UUFFcEMsS0FBSyxNQUFNLElBQUksSUFBSSxJQUFJLENBQUMsTUFBTSxFQUFFO1lBQzlCLEtBQUssQ0FBQyxJQUFJLENBQUM7Z0JBQ1QsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJO2dCQUNmLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSTtnQkFDZixVQUFVLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUNyRixRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7Z0JBQ3ZCLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO2FBQzlFLENBQUMsQ0FBQztTQUNKO1FBRUQsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2RCxPQUFPLElBQUksQ0FBQyxJQUFJLEtBQUssV0FBVyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUM7WUFDL0MsU0FBUyxFQUFFLElBQUksQ0FBQyxTQUFTO1lBQ3pCLFdBQVcsRUFBRSxJQUFJLENBQUMsWUFBWTtZQUM5QixZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDL0IsSUFBSSxFQUFFLFdBQVc7WUFDakIsUUFBUSxFQUFFLElBQUksQ0FBQyxTQUFTO1lBQ3hCLEtBQUssRUFBRSxLQUFLO1lBQ1osd0JBQXdCLEVBQUUsSUFBSSxDQUFDLHlCQUF5QjtTQUN6RCxDQUFDLENBQUMsQ0FBQztZQUNGLElBQUksRUFBRSxXQUFXO1lBQ2pCLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTtTQUNoQyxDQUFDO0lBQ0osQ0FBQztJQUVPLG1CQUFtQixDQUFDLFFBQWtCO1FBQzVDLFFBQVEsUUFBUSxFQUFFO1lBQ2hCLEtBQUssUUFBUSxDQUFDLElBQUk7Z0JBQ2hCLE9BQU8sR0FBRyxDQUFDLGlDQUFpQyxDQUFDLElBQUksQ0FBQztZQUNwRCxLQUFLLFFBQVEsQ0FBQyxHQUFHO2dCQUNmLE9BQU8sR0FBRyxDQUFDLGlDQUFpQyxDQUFDLEdBQUcsQ0FBQztZQUNuRCxLQUFLLFFBQVEsQ0FBQyxHQUFHO2dCQUNmLE9BQU8sR0FBRyxDQUFDLGlDQUFpQyxDQUFDLEdBQUcsQ0FBQztZQUNuRDtnQkFDRSxNQUFNLElBQUksS0FBSyxDQUFDLDhCQUE4QixRQUFRLEVBQUUsQ0FBQyxDQUFDO1NBQzdEO0lBQ0gsQ0FBQztJQUVPLGtCQUFrQixDQUFDLFdBQXdCO1FBQ2pELFFBQVEsV0FBVyxFQUFFO1lBQ25CLEtBQUssV0FBVyxDQUFDLFVBQVU7Z0JBQ3pCLE9BQU8sR0FBRyxDQUFDLDZCQUE2QixDQUFDLFVBQVUsQ0FBQztZQUN0RCxLQUFLLFdBQVcsQ0FBQyxhQUFhO2dCQUM1QixPQUFPLEdBQUcsQ0FBQyw2QkFBNkIsQ0FBQyxhQUFhLENBQUM7WUFDekQsS0FBSyxXQUFXLENBQUMsYUFBYTtnQkFDNUIsT0FBTyxHQUFHLENBQUMsNkJBQTZCLENBQUMsYUFBYSxDQUFDO1lBQ3pELEtBQUssV0FBVyxDQUFDLFNBQVM7Z0JBQ3hCLE9BQU8sR0FBRyxDQUFDLDZCQUE2QixDQUFDLFNBQVMsQ0FBQztZQUNyRDtnQkFDRSxNQUFNLElBQUksS0FBSyxDQUFDLDZCQUE2QixXQUFXLEVBQUUsQ0FBQyxDQUFDO1NBQy9EO0lBQ0gsQ0FBQzs7QUF0TEgsMEJBdUxDOzs7QUFFRCxJQUFZLFFBSVg7QUFKRCxXQUFZLFFBQVE7SUFDbEIsdUJBQVcsQ0FBQTtJQUNYLHVCQUFXLENBQUE7SUFDWCx5QkFBYSxDQUFBO0FBQ2YsQ0FBQyxFQUpXLFFBQVEsR0FBUixnQkFBUSxLQUFSLGdCQUFRLFFBSW5CIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQXBpT2JqZWN0LCBMYXp5IH0gZnJvbSAnY2RrOHMnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgKiBhcyBiYXNlIGZyb20gJy4vYmFzZSc7XG5pbXBvcnQgKiBhcyBrOHMgZnJvbSAnLi9pbXBvcnRzL2s4cyc7XG5pbXBvcnQgKiBhcyBpbmdyZXNzIGZyb20gJy4vaW5ncmVzcyc7XG5pbXBvcnQgKiBhcyBwb2QgZnJvbSAnLi9wb2QnO1xuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGBTZXJ2aWNlYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTZXJ2aWNlUHJvcHMgZXh0ZW5kcyBiYXNlLlJlc291cmNlUHJvcHMge1xuICAvKipcbiAgICogV2hpY2ggcG9kcyBzaG91bGQgdGhlIHNlcnZpY2Ugc2VsZWN0IGFuZCByb3V0ZSB0by5cbiAgICpcbiAgICogWW91IGNhbiBwYXNzIG9uZSBvZiB0aGUgZm9sbG93aW5nOlxuICAgKlxuICAgKiAtIEFuIGluc3RhbmNlIG9mIGBQb2RgIG9yIGFueSB3b3JrbG9hZCByZXNvdXJjZSAoZS5nIGBEZXBsb3ltZW50YCwgYFN0YXRlZnVsU2V0YCwgLi4uKVxuICAgKiAtIFBvZHMgc2VsZWN0ZWQgYnkgdGhlIGBQb2RzLnNlbGVjdGAgZnVuY3Rpb24uIE5vdGUgdGhhdCBpbiB0aGlzIGNhc2Ugb25seSBsYWJlbHMgY2FuIGJlIHNwZWNpZmllZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB1bnNldCwgdGhlIHNlcnZpY2UgaXMgYXNzdW1lZCB0byBoYXZlIGFuIGV4dGVybmFsIHByb2Nlc3MgbWFuYWdpbmdcbiAgICogaXRzIGVuZHBvaW50cywgd2hpY2ggS3ViZXJuZXRlcyB3aWxsIG5vdCBtb2RpZnkuXG4gICAqXG4gICAqIEBleGFtcGxlXG4gICAqXG4gICAqIC8vIHNlbGVjdCB0aGUgcG9kcyBvZiBhIHNwZWNpZmljIGRlcGxveW1lbnRcbiAgICogY29uc3QgYmFja2VuZCA9IG5ldyBrcGx1cy5EZXBsb3ltZW50KHRoaXMsICdCYWNrZW5kJywgLi4uKTtcbiAgICogbmV3IGtwbHVzLlNlcnZpY2UodGhpcywgJ1NlcnZpY2UnLCB7IHNlbGVjdG9yOiBiYWNrZW5kIH0pO1xuICAgKlxuICAgKiAvLyBzZWxlY3QgYWxsIHBvZHMgbGFiZWxlZCB3aXRoIHRoZSBgdGllcj1iYWNrZW5kYCBsYWJlbFxuICAgKiBjb25zdCBiYWNrZW5kID0ga3BsdXMuUG9kLmxhYmVsZWQoeyB0aWVyOiAnYmFja2VuZCcgfSk7XG4gICAqIG5ldyBrcGx1cy5TZXJ2aWNlKHRoaXMsICdTZXJ2aWNlJywgeyBzZWxlY3RvcjogYmFja2VuZCB9KTtcbiAgICovXG4gIHJlYWRvbmx5IHNlbGVjdG9yPzogcG9kLklQb2RTZWxlY3RvcjtcblxuICAvKipcbiAgICogVGhlIElQIGFkZHJlc3Mgb2YgdGhlIHNlcnZpY2UgYW5kIGlzIHVzdWFsbHkgYXNzaWduZWQgcmFuZG9tbHkgYnkgdGhlXG4gICAqIG1hc3Rlci4gSWYgYW4gYWRkcmVzcyBpcyBzcGVjaWZpZWQgbWFudWFsbHkgYW5kIGlzIG5vdCBpbiB1c2UgYnkgb3RoZXJzLCBpdFxuICAgKiB3aWxsIGJlIGFsbG9jYXRlZCB0byB0aGUgc2VydmljZTsgb3RoZXJ3aXNlLCBjcmVhdGlvbiBvZiB0aGUgc2VydmljZSB3aWxsXG4gICAqIGZhaWwuIFRoaXMgZmllbGQgY2FuIG5vdCBiZSBjaGFuZ2VkIHRocm91Z2ggdXBkYXRlcy4gVmFsaWQgdmFsdWVzIGFyZVxuICAgKiBcIk5vbmVcIiwgZW1wdHkgc3RyaW5nIChcIlwiKSwgb3IgYSB2YWxpZCBJUCBhZGRyZXNzLiBcIk5vbmVcIiBjYW4gYmUgc3BlY2lmaWVkXG4gICAqIGZvciBoZWFkbGVzcyBzZXJ2aWNlcyB3aGVuIHByb3h5aW5nIGlzIG5vdCByZXF1aXJlZC4gT25seSBhcHBsaWVzIHRvIHR5cGVzXG4gICAqIENsdXN0ZXJJUCwgTm9kZVBvcnQsIGFuZCBMb2FkQmFsYW5jZXIuIElnbm9yZWQgaWYgdHlwZSBpcyBFeHRlcm5hbE5hbWUuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvc2VydmljZXMtbmV0d29ya2luZy9zZXJ2aWNlLyN2aXJ0dWFsLWlwcy1hbmQtc2VydmljZS1wcm94aWVzXG4gICAqIEBkZWZhdWx0IC0gQXV0b21hdGljYWxseSBhc3NpZ25lZC5cbiAgICpcbiAgICovXG4gIHJlYWRvbmx5IGNsdXN0ZXJJUD86IHN0cmluZztcblxuICAvKipcbiAgICogQSBsaXN0IG9mIElQIGFkZHJlc3NlcyBmb3Igd2hpY2ggbm9kZXMgaW4gdGhlIGNsdXN0ZXIgd2lsbCBhbHNvIGFjY2VwdFxuICAgKiB0cmFmZmljIGZvciB0aGlzIHNlcnZpY2UuIFRoZXNlIElQcyBhcmUgbm90IG1hbmFnZWQgYnkgS3ViZXJuZXRlcy4gVGhlIHVzZXJcbiAgICogaXMgcmVzcG9uc2libGUgZm9yIGVuc3VyaW5nIHRoYXQgdHJhZmZpYyBhcnJpdmVzIGF0IGEgbm9kZSB3aXRoIHRoaXMgSVAuIEFcbiAgICogY29tbW9uIGV4YW1wbGUgaXMgZXh0ZXJuYWwgbG9hZC1iYWxhbmNlcnMgdGhhdCBhcmUgbm90IHBhcnQgb2YgdGhlXG4gICAqIEt1YmVybmV0ZXMgc3lzdGVtLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGV4dGVybmFsIElQcy5cbiAgICovXG4gIHJlYWRvbmx5IGV4dGVybmFsSVBzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIERldGVybWluZXMgaG93IHRoZSBTZXJ2aWNlIGlzIGV4cG9zZWQuXG4gICAqXG4gICAqIE1vcmUgaW5mbzogaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvc2VydmljZXMtbmV0d29ya2luZy9zZXJ2aWNlLyNwdWJsaXNoaW5nLXNlcnZpY2VzLXNlcnZpY2UtdHlwZXNcbiAgICpcbiAgICogQGRlZmF1bHQgU2VydmljZVR5cGUuQ2x1c3RlcklQXG4gICAqL1xuICByZWFkb25seSB0eXBlPzogU2VydmljZVR5cGU7XG5cbiAgLyoqXG4gICAqIFRoZSBwb3J0cyB0aGlzIHNlcnZpY2UgYmluZHMgdG8uXG4gICAqXG4gICAqIElmIHRoZSBzZWxlY3RvciBvZiB0aGUgc2VydmljZSBpcyBhIG1hbmFnZWQgcG9kIC8gd29ya2xvYWQsXG4gICAqIGl0cyBwb3J0cyB3aWxsIGFyZSBhdXRvbWF0aWNhbGx5IGV4dHJhY3RlZCBhbmQgdXNlZCBhcyB0aGUgZGVmYXVsdCB2YWx1ZS5cbiAgICogT3RoZXJ3aXNlLCBubyBwb3J0cyBhcmUgYm91bmQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZWl0aGVyIHRoZSBzZWxlY3RvciBwb3J0cywgb3Igbm9uZS5cbiAgICovXG4gIHJlYWRvbmx5IHBvcnRzPzogU2VydmljZVBvcnRbXTtcblxuICAvKipcbiAgICogVGhlIGV4dGVybmFsTmFtZSB0byBiZSB1c2VkIHdoZW4gU2VydmljZVR5cGUuRVhURVJOQUxfTkFNRSBpcyBzZXRcbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBleHRlcm5hbCBuYW1lLlxuICAgKi9cbiAgcmVhZG9ubHkgZXh0ZXJuYWxOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBIGxpc3Qgb2YgQ0lEUiBJUCBhZGRyZXNzZXMsIGlmIHNwZWNpZmllZCBhbmQgc3VwcG9ydGVkIGJ5IHRoZSBwbGF0Zm9ybSxcbiAgICogd2lsbCByZXN0cmljdCB0cmFmZmljIHRocm91Z2ggdGhlIGNsb3VkLXByb3ZpZGVyIGxvYWQtYmFsYW5jZXIgdG8gdGhlIHNwZWNpZmllZCBjbGllbnQgSVBzLlxuICAgKlxuICAgKiBNb3JlIGluZm86IGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL3Rhc2tzL2FjY2Vzcy1hcHBsaWNhdGlvbi1jbHVzdGVyL2NvbmZpZ3VyZS1jbG91ZC1wcm92aWRlci1maXJld2FsbC9cbiAgICovXG4gIHJlYWRvbmx5IGxvYWRCYWxhbmNlclNvdXJjZVJhbmdlcz86IHN0cmluZ1tdO1xuXG59XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgZXhwb3NpbmcgYSBzZXJ2aWNlIHVzaW5nIGFuIGluZ3Jlc3MuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRXhwb3NlU2VydmljZVZpYUluZ3Jlc3NPcHRpb25zIHtcblxuICAvKipcbiAgICogVGhlIHR5cGUgb2YgdGhlIHBhdGhcbiAgICpcbiAgICogQGRlZmF1bHQgSHR0cEluZ3Jlc3NQYXRoVHlwZS5QUkVGSVhcbiAgICovXG4gIHJlYWRvbmx5IHBhdGhUeXBlPzogaW5ncmVzcy5IdHRwSW5ncmVzc1BhdGhUeXBlO1xuXG4gIC8qKlxuICAgKiBUaGUgaW5ncmVzcyB0byBhZGQgcnVsZXMgdG8uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQW4gaW5ncmVzcyB3aWxsIGJlIGF1dG9tYXRpY2FsbHkgY3JlYXRlZC5cbiAgICovXG4gIHJlYWRvbmx5IGluZ3Jlc3M/OiBpbmdyZXNzLkluZ3Jlc3M7XG59XG5cbi8qKlxuICogRm9yIHNvbWUgcGFydHMgb2YgeW91ciBhcHBsaWNhdGlvbiAoZm9yIGV4YW1wbGUsIGZyb250ZW5kcykgeW91IG1heSB3YW50IHRvIGV4cG9zZSBhIFNlcnZpY2Ugb250byBhblxuICogZXh0ZXJuYWwgSVAgYWRkcmVzcywgdGhhdCdzIG91dHNpZGUgb2YgeW91ciBjbHVzdGVyLlxuICogS3ViZXJuZXRlcyBTZXJ2aWNlVHlwZXMgYWxsb3cgeW91IHRvIHNwZWNpZnkgd2hhdCBraW5kIG9mIFNlcnZpY2UgeW91IHdhbnQuXG4gKiBUaGUgZGVmYXVsdCBpcyBDbHVzdGVySVAuXG4gKi9cbmV4cG9ydCBlbnVtIFNlcnZpY2VUeXBlIHtcblxuICAvKipcbiAgICogRXhwb3NlcyB0aGUgU2VydmljZSBvbiBhIGNsdXN0ZXItaW50ZXJuYWwgSVAuXG4gICAqIENob29zaW5nIHRoaXMgdmFsdWUgbWFrZXMgdGhlIFNlcnZpY2Ugb25seSByZWFjaGFibGUgZnJvbSB3aXRoaW4gdGhlIGNsdXN0ZXIuXG4gICAqIFRoaXMgaXMgdGhlIGRlZmF1bHQgU2VydmljZVR5cGVcbiAgICovXG4gIENMVVNURVJfSVAgPSAnQ2x1c3RlcklQJyxcblxuICAvKipcbiAgICogRXhwb3NlcyB0aGUgU2VydmljZSBvbiBlYWNoIE5vZGUncyBJUCBhdCBhIHN0YXRpYyBwb3J0ICh0aGUgTm9kZVBvcnQpLlxuICAgKiBBIENsdXN0ZXJJUCBTZXJ2aWNlLCB0byB3aGljaCB0aGUgTm9kZVBvcnQgU2VydmljZSByb3V0ZXMsIGlzIGF1dG9tYXRpY2FsbHkgY3JlYXRlZC5cbiAgICogWW91J2xsIGJlIGFibGUgdG8gY29udGFjdCB0aGUgTm9kZVBvcnQgU2VydmljZSwgZnJvbSBvdXRzaWRlIHRoZSBjbHVzdGVyLFxuICAgKiBieSByZXF1ZXN0aW5nIDxOb2RlSVA+OjxOb2RlUG9ydD4uXG4gICAqL1xuICBOT0RFX1BPUlQgPSAnTm9kZVBvcnQnLFxuXG4gIC8qKlxuICAgKiBFeHBvc2VzIHRoZSBTZXJ2aWNlIGV4dGVybmFsbHkgdXNpbmcgYSBjbG91ZCBwcm92aWRlcidzIGxvYWQgYmFsYW5jZXIuXG4gICAqIE5vZGVQb3J0IGFuZCBDbHVzdGVySVAgU2VydmljZXMsIHRvIHdoaWNoIHRoZSBleHRlcm5hbCBsb2FkIGJhbGFuY2VyIHJvdXRlcyxcbiAgICogYXJlIGF1dG9tYXRpY2FsbHkgY3JlYXRlZC5cbiAgICovXG4gIExPQURfQkFMQU5DRVIgPSAnTG9hZEJhbGFuY2VyJyxcblxuICAvKipcbiAgICogTWFwcyB0aGUgU2VydmljZSB0byB0aGUgY29udGVudHMgb2YgdGhlIGV4dGVybmFsTmFtZSBmaWVsZCAoZS5nLiBmb28uYmFyLmV4YW1wbGUuY29tKSwgYnkgcmV0dXJuaW5nIGEgQ05BTUUgcmVjb3JkIHdpdGggaXRzIHZhbHVlLlxuICAgKiBObyBwcm94eWluZyBvZiBhbnkga2luZCBpcyBzZXQgdXAuXG4gICAqXG4gICAqID4gTm90ZTogWW91IG5lZWQgZWl0aGVyIGt1YmUtZG5zIHZlcnNpb24gMS43IG9yIENvcmVETlMgdmVyc2lvbiAwLjAuOCBvciBoaWdoZXIgdG8gdXNlIHRoZSBFeHRlcm5hbE5hbWUgdHlwZS5cbiAgICovXG4gIEVYVEVSTkFMX05BTUUgPSAnRXh0ZXJuYWxOYW1lJ1xufVxuXG4vKipcbiAqIE9wdGlvbnMgdG8gYWRkIGEgZGVwbG95bWVudCB0byBhIHNlcnZpY2UuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQWRkRGVwbG95bWVudE9wdGlvbnMgZXh0ZW5kcyBTZXJ2aWNlQmluZE9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIHBvcnQgbnVtYmVyIHRoZSBzZXJ2aWNlIHdpbGwgYmluZCB0by5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBDb3BpZWQgZnJvbSB0aGUgZmlyc3QgY29udGFpbmVyIG9mIHRoZSBkZXBsb3ltZW50LlxuICAgKi9cbiAgcmVhZG9ubHkgcG9ydD86IG51bWJlcjtcbn1cblxuLyoqXG4gKiBBbiBhYnN0cmFjdCB3YXkgdG8gZXhwb3NlIGFuIGFwcGxpY2F0aW9uIHJ1bm5pbmcgb24gYSBzZXQgb2YgUG9kcyBhcyBhIG5ldHdvcmsgc2VydmljZS5cbiAqIFdpdGggS3ViZXJuZXRlcyB5b3UgZG9uJ3QgbmVlZCB0byBtb2RpZnkgeW91ciBhcHBsaWNhdGlvbiB0byB1c2UgYW4gdW5mYW1pbGlhciBzZXJ2aWNlIGRpc2NvdmVyeSBtZWNoYW5pc20uXG4gKiBLdWJlcm5ldGVzIGdpdmVzIFBvZHMgdGhlaXIgb3duIElQIGFkZHJlc3NlcyBhbmQgYSBzaW5nbGUgRE5TIG5hbWUgZm9yIGEgc2V0IG9mIFBvZHMsIGFuZCBjYW4gbG9hZC1iYWxhbmNlIGFjcm9zcyB0aGVtLlxuICpcbiAqIEZvciBleGFtcGxlLCBjb25zaWRlciBhIHN0YXRlbGVzcyBpbWFnZS1wcm9jZXNzaW5nIGJhY2tlbmQgd2hpY2ggaXMgcnVubmluZyB3aXRoIDMgcmVwbGljYXMuIFRob3NlIHJlcGxpY2FzIGFyZSBmdW5naWJsZeKAlGZyb250ZW5kcyBkbyBub3QgY2FyZSB3aGljaCBiYWNrZW5kIHRoZXkgdXNlLlxuICogV2hpbGUgdGhlIGFjdHVhbCBQb2RzIHRoYXQgY29tcG9zZSB0aGUgYmFja2VuZCBzZXQgbWF5IGNoYW5nZSwgdGhlIGZyb250ZW5kIGNsaWVudHMgc2hvdWxkIG5vdCBuZWVkIHRvIGJlIGF3YXJlIG9mIHRoYXQsXG4gKiBub3Igc2hvdWxkIHRoZXkgbmVlZCB0byBrZWVwIHRyYWNrIG9mIHRoZSBzZXQgb2YgYmFja2VuZHMgdGhlbXNlbHZlcy5cbiAqIFRoZSBTZXJ2aWNlIGFic3RyYWN0aW9uIGVuYWJsZXMgdGhpcyBkZWNvdXBsaW5nLlxuICpcbiAqIElmIHlvdSdyZSBhYmxlIHRvIHVzZSBLdWJlcm5ldGVzIEFQSXMgZm9yIHNlcnZpY2UgZGlzY292ZXJ5IGluIHlvdXIgYXBwbGljYXRpb24sIHlvdSBjYW4gcXVlcnkgdGhlIEFQSSBzZXJ2ZXIgZm9yIEVuZHBvaW50cyxcbiAqIHRoYXQgZ2V0IHVwZGF0ZWQgd2hlbmV2ZXIgdGhlIHNldCBvZiBQb2RzIGluIGEgU2VydmljZSBjaGFuZ2VzLiBGb3Igbm9uLW5hdGl2ZSBhcHBsaWNhdGlvbnMsIEt1YmVybmV0ZXMgb2ZmZXJzIHdheXMgdG8gcGxhY2UgYSBuZXR3b3JrIHBvcnRcbiAqIG9yIGxvYWQgYmFsYW5jZXIgaW4gYmV0d2VlbiB5b3VyIGFwcGxpY2F0aW9uIGFuZCB0aGUgYmFja2VuZCBQb2RzLlxuICovXG5leHBvcnQgY2xhc3MgU2VydmljZSBleHRlbmRzIGJhc2UuUmVzb3VyY2Uge1xuXG4gIC8qKlxuICAgKiBUaGUgSVAgYWRkcmVzcyBvZiB0aGUgc2VydmljZSBhbmQgaXMgdXN1YWxseSBhc3NpZ25lZCByYW5kb21seSBieSB0aGVcbiAgICogbWFzdGVyLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGNsdXN0ZXJJUD86IHN0cmluZztcblxuICAvKipcbiAgICogRGV0ZXJtaW5lcyBob3cgdGhlIFNlcnZpY2UgaXMgZXhwb3NlZC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB0eXBlOiBTZXJ2aWNlVHlwZTtcblxuICAvKipcbiAgICogVGhlIGV4dGVybmFsTmFtZSB0byBiZSB1c2VkIGZvciBFWFRFUk5BTF9OQU1FIHR5cGVzXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZXh0ZXJuYWxOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBAc2VlIGJhc2UuUmVzb3VyY2UuYXBpT2JqZWN0XG4gICAqL1xuICBwcm90ZWN0ZWQgcmVhZG9ubHkgYXBpT2JqZWN0OiBBcGlPYmplY3Q7XG5cbiAgcHVibGljIHJlYWRvbmx5IHJlc291cmNlVHlwZSA9ICdzZXJ2aWNlcyc7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBfZXh0ZXJuYWxJUHM6IHN0cmluZ1tdO1xuICBwcml2YXRlIHJlYWRvbmx5IF9zZWxlY3RvcjogUmVjb3JkPHN0cmluZywgc3RyaW5nPjtcbiAgcHJpdmF0ZSByZWFkb25seSBfcG9ydHM6IFNlcnZpY2VQb3J0W107XG4gIHByaXZhdGUgcmVhZG9ubHkgX2xvYWRCYWxhbmNlclNvdXJjZVJhbmdlcz86IHN0cmluZ1tdO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBTZXJ2aWNlUHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLmFwaU9iamVjdCA9IG5ldyBrOHMuS3ViZVNlcnZpY2UodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgbWV0YWRhdGE6IHByb3BzLm1ldGFkYXRhLFxuICAgICAgc3BlYzogTGF6eS5hbnkoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLl90b0t1YmUoKSB9KSxcbiAgICB9KTtcblxuICAgIHRoaXMuY2x1c3RlcklQID0gcHJvcHMuY2x1c3RlcklQO1xuICAgIHRoaXMuZXh0ZXJuYWxOYW1lID0gcHJvcHMuZXh0ZXJuYWxOYW1lO1xuXG4gICAgaWYgKHByb3BzLmV4dGVybmFsTmFtZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aGlzLnR5cGUgPSBTZXJ2aWNlVHlwZS5FWFRFUk5BTF9OQU1FO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLnR5cGUgPSBwcm9wcy50eXBlID8/IFNlcnZpY2VUeXBlLkNMVVNURVJfSVA7XG4gICAgfVxuXG4gICAgdGhpcy5fZXh0ZXJuYWxJUHMgPSBwcm9wcy5leHRlcm5hbElQcyA/PyBbXTtcbiAgICB0aGlzLl9wb3J0cyA9IFtdO1xuICAgIHRoaXMuX3NlbGVjdG9yID0geyB9O1xuICAgIHRoaXMuX2xvYWRCYWxhbmNlclNvdXJjZVJhbmdlcyA9IHByb3BzLmxvYWRCYWxhbmNlclNvdXJjZVJhbmdlcztcblxuICAgIGlmIChwcm9wcy5zZWxlY3Rvcikge1xuICAgICAgdGhpcy5zZWxlY3QocHJvcHMuc2VsZWN0b3IpO1xuICAgIH1cblxuICAgIGZvciAoY29uc3QgcG9ydCBvZiBwcm9wcy5wb3J0cyA/PyBbXSkge1xuICAgICAgdGhpcy5iaW5kKHBvcnQucG9ydCwgcG9ydCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEV4cG9zZSBhIHNlcnZpY2UgdmlhIGFuIGluZ3Jlc3MgdXNpbmcgdGhlIHNwZWNpZmllZCBwYXRoLlxuICAgKlxuICAgKiBAcGFyYW0gcGF0aCBUaGUgcGF0aCB0byBleHBvc2UgdGhlIHNlcnZpY2UgdW5kZXIuXG4gICAqIEBwYXJhbSBvcHRpb25zIEFkZGl0aW9uYWwgb3B0aW9ucy5cbiAgICpcbiAgICogQHJldHVybnMgVGhlIGBJbmdyZXNzYCByZXNvdXJjZSB0aGF0IHdhcyB1c2VkLlxuICAgKi9cbiAgcHVibGljIGV4cG9zZVZpYUluZ3Jlc3MocGF0aDogc3RyaW5nLCBvcHRpb25zOiBFeHBvc2VTZXJ2aWNlVmlhSW5ncmVzc09wdGlvbnMgPSB7fSk6IGluZ3Jlc3MuSW5ncmVzcyB7XG4gICAgY29uc3QgaW5nciA9IG9wdGlvbnMuaW5ncmVzcyA/PyBuZXcgaW5ncmVzcy5JbmdyZXNzKHRoaXMsICdJbmdyZXNzJyk7XG4gICAgaW5nci5hZGRSdWxlKHBhdGgsIGluZ3Jlc3MuSW5ncmVzc0JhY2tlbmQuZnJvbVNlcnZpY2UodGhpcyksIG9wdGlvbnMucGF0aFR5cGUpO1xuICAgIHJldHVybiBpbmdyO1xuICB9XG5cbiAgLyoqXG4gICAqIFBvcnRzIGZvciB0aGlzIHNlcnZpY2UuXG4gICAqXG4gICAqIFVzZSBgYmluZCgpYCB0byBiaW5kIGFkZGl0aW9uYWwgc2VydmljZSBwb3J0cy5cbiAgICovXG4gIHB1YmxpYyBnZXQgcG9ydHMoKSB7XG4gICAgcmV0dXJuIFsuLi50aGlzLl9wb3J0c107XG4gIH1cblxuICAvKipcbiAgICogQ29uZmlndXJlIGEgcG9ydCB0aGUgc2VydmljZSB3aWxsIGJpbmQgdG8uXG4gICAqIFRoaXMgbWV0aG9kIGNhbiBiZSBjYWxsZWQgbXVsdGlwbGUgdGltZXMuXG4gICAqXG4gICAqIEBwYXJhbSBwb3J0IFRoZSBwb3J0IGRlZmluaXRpb24uXG4gICAqL1xuICBwdWJsaWMgYmluZChwb3J0OiBudW1iZXIsIG9wdGlvbnM6IFNlcnZpY2VCaW5kT3B0aW9ucyA9IHsgfSkge1xuICAgIHRoaXMuX3BvcnRzLnB1c2goeyAuLi5vcHRpb25zLCBwb3J0IH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlcXVpcmUgdGhpcyBzZXJ2aWNlIHRvIHNlbGVjdCBwb2RzIG1hdGNoaW5nIHRoZSBzZWxlY3Rvci5cbiAgICpcbiAgICogTm90ZSB0aGF0IGludm9raW5nIHRoaXMgbWV0aG9kIG11bHRpcGxlIHRpbWVzIGFjdHMgYXMgYW4gQU5EIG9wZXJhdG9yXG4gICAqIG9uIHRoZSByZXN1bHRpbmcgbGFiZWxzLlxuICAgKi9cbiAgcHVibGljIHNlbGVjdChzZWxlY3RvcjogcG9kLklQb2RTZWxlY3Rvcikge1xuICAgIGNvbnN0IGxhYmVscyA9IHNlbGVjdG9yLnRvUG9kU2VsZWN0b3JDb25maWcoKS5sYWJlbFNlbGVjdG9yLl90b0t1YmUoKS5tYXRjaExhYmVscyA/PyB7fTtcbiAgICBmb3IgKGNvbnN0IFtrZXksIHZhbHVlXSBvZiBPYmplY3QuZW50cmllcyhsYWJlbHMpKSB7XG4gICAgICB0aGlzLl9zZWxlY3RvcltrZXldID0gdmFsdWU7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFJlcXVpcmUgdGhpcyBzZXJ2aWNlIHRvIHNlbGVjdCBwb2RzIHdpdGggdGhpcyBsYWJlbC5cbiAgICpcbiAgICogTm90ZSB0aGF0IGludm9raW5nIHRoaXMgbWV0aG9kIG11bHRpcGxlIHRpbWVzIGFjdHMgYXMgYW4gQU5EIG9wZXJhdG9yXG4gICAqIG9uIHRoZSByZXN1bHRpbmcgbGFiZWxzLlxuICAgKi9cbiAgcHVibGljIHNlbGVjdExhYmVsKGtleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nKSB7XG4gICAgdGhpcy5fc2VsZWN0b3Jba2V5XSA9IHZhbHVlO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIF90b0t1YmUoKTogazhzLlNlcnZpY2VTcGVjIHtcbiAgICBpZiAodGhpcy5fcG9ydHMubGVuZ3RoID09PSAwICYmIHRoaXMudHlwZSAhPT0gU2VydmljZVR5cGUuRVhURVJOQUxfTkFNRSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdBIHNlcnZpY2UgbXVzdCBiZSBjb25maWd1cmVkIHdpdGggYSBwb3J0Jyk7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMudHlwZSA9PT0gU2VydmljZVR5cGUuRVhURVJOQUxfTkFNRSAmJiB0aGlzLmV4dGVybmFsTmFtZSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Egc2VydmljZSB3aXRoIHR5cGUgRVhURVJOQUxfTkFNRSByZXF1aXJlcyBhbiBleHRlcm5hbE5hbWUgcHJvcCcpO1xuICAgIH1cblxuICAgIGNvbnN0IHBvcnRzOiBrOHMuU2VydmljZVBvcnRbXSA9IFtdO1xuXG4gICAgZm9yIChjb25zdCBwb3J0IG9mIHRoaXMuX3BvcnRzKSB7XG4gICAgICBwb3J0cy5wdXNoKHtcbiAgICAgICAgbmFtZTogcG9ydC5uYW1lLFxuICAgICAgICBwb3J0OiBwb3J0LnBvcnQsXG4gICAgICAgIHRhcmdldFBvcnQ6IHBvcnQudGFyZ2V0UG9ydCA/IGs4cy5JbnRPclN0cmluZy5mcm9tTnVtYmVyKHBvcnQudGFyZ2V0UG9ydCkgOiB1bmRlZmluZWQsXG4gICAgICAgIG5vZGVQb3J0OiBwb3J0Lm5vZGVQb3J0LFxuICAgICAgICBwcm90b2NvbDogcG9ydC5wcm90b2NvbCA/IHRoaXMuX3BvcnRQcm90b2NvbFRvS3ViZShwb3J0LnByb3RvY29sKSA6IHVuZGVmaW5lZCxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIGNvbnN0IHNlcnZpY2VUeXBlID0gdGhpcy5fc2VydmljZVR5cGVUb0t1YmUodGhpcy50eXBlKTtcbiAgICByZXR1cm4gdGhpcy50eXBlICE9PSBTZXJ2aWNlVHlwZS5FWFRFUk5BTF9OQU1FID8ge1xuICAgICAgY2x1c3RlcklwOiB0aGlzLmNsdXN0ZXJJUCxcbiAgICAgIGV4dGVybmFsSVBzOiB0aGlzLl9leHRlcm5hbElQcyxcbiAgICAgIGV4dGVybmFsTmFtZTogdGhpcy5leHRlcm5hbE5hbWUsXG4gICAgICB0eXBlOiBzZXJ2aWNlVHlwZSxcbiAgICAgIHNlbGVjdG9yOiB0aGlzLl9zZWxlY3RvcixcbiAgICAgIHBvcnRzOiBwb3J0cyxcbiAgICAgIGxvYWRCYWxhbmNlclNvdXJjZVJhbmdlczogdGhpcy5fbG9hZEJhbGFuY2VyU291cmNlUmFuZ2VzLFxuICAgIH0gOiB7XG4gICAgICB0eXBlOiBzZXJ2aWNlVHlwZSxcbiAgICAgIGV4dGVybmFsTmFtZTogdGhpcy5leHRlcm5hbE5hbWUsXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgX3BvcnRQcm90b2NvbFRvS3ViZShwcm90b2NvbDogUHJvdG9jb2wpOiBrOHMuSW9LOFNBcGlDb3JlVjFTZXJ2aWNlUG9ydFByb3RvY29sIHtcbiAgICBzd2l0Y2ggKHByb3RvY29sKSB7XG4gICAgICBjYXNlIFByb3RvY29sLlNDVFA6XG4gICAgICAgIHJldHVybiBrOHMuSW9LOFNBcGlDb3JlVjFTZXJ2aWNlUG9ydFByb3RvY29sLlNDVFA7XG4gICAgICBjYXNlIFByb3RvY29sLlRDUDpcbiAgICAgICAgcmV0dXJuIGs4cy5Jb0s4U0FwaUNvcmVWMVNlcnZpY2VQb3J0UHJvdG9jb2wuVENQO1xuICAgICAgY2FzZSBQcm90b2NvbC5VRFA6XG4gICAgICAgIHJldHVybiBrOHMuSW9LOFNBcGlDb3JlVjFTZXJ2aWNlUG9ydFByb3RvY29sLlVEUDtcbiAgICAgIGRlZmF1bHQ6XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgVW5zdXBwb3J0ZWQgcG9ydCBwcm90b2NvbDogJHtwcm90b2NvbH1gKTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIF9zZXJ2aWNlVHlwZVRvS3ViZShzZXJ2aWNlVHlwZTogU2VydmljZVR5cGUpOiBrOHMuSW9LOFNBcGlDb3JlVjFTZXJ2aWNlU3BlY1R5cGUge1xuICAgIHN3aXRjaCAoc2VydmljZVR5cGUpIHtcbiAgICAgIGNhc2UgU2VydmljZVR5cGUuQ0xVU1RFUl9JUDpcbiAgICAgICAgcmV0dXJuIGs4cy5Jb0s4U0FwaUNvcmVWMVNlcnZpY2VTcGVjVHlwZS5DTFVTVEVSX0lQO1xuICAgICAgY2FzZSBTZXJ2aWNlVHlwZS5FWFRFUk5BTF9OQU1FOlxuICAgICAgICByZXR1cm4gazhzLklvSzhTQXBpQ29yZVYxU2VydmljZVNwZWNUeXBlLkVYVEVSTkFMX05BTUU7XG4gICAgICBjYXNlIFNlcnZpY2VUeXBlLkxPQURfQkFMQU5DRVI6XG4gICAgICAgIHJldHVybiBrOHMuSW9LOFNBcGlDb3JlVjFTZXJ2aWNlU3BlY1R5cGUuTE9BRF9CQUxBTkNFUjtcbiAgICAgIGNhc2UgU2VydmljZVR5cGUuTk9ERV9QT1JUOlxuICAgICAgICByZXR1cm4gazhzLklvSzhTQXBpQ29yZVYxU2VydmljZVNwZWNUeXBlLk5PREVfUE9SVDtcbiAgICAgIGRlZmF1bHQ6XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgVW5zdXBwb3J0ZWQgc2VydmljZSB0eXBlOiAke3NlcnZpY2VUeXBlfWApO1xuICAgIH1cbiAgfVxufVxuXG5leHBvcnQgZW51bSBQcm90b2NvbCB7XG4gIFRDUCA9ICdUQ1AnLFxuICBVRFAgPSAnVURQJyxcbiAgU0NUUCA9ICdTQ1RQJ1xufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGBTZXJ2aWNlLmJpbmRgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFNlcnZpY2VCaW5kT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGlzIHBvcnQgd2l0aGluIHRoZSBzZXJ2aWNlLiBUaGlzIG11c3QgYmUgYSBETlNfTEFCRUwuIEFsbFxuICAgKiBwb3J0cyB3aXRoaW4gYSBTZXJ2aWNlU3BlYyBtdXN0IGhhdmUgdW5pcXVlIG5hbWVzLiBUaGlzIG1hcHMgdG8gdGhlICdOYW1lJ1xuICAgKiBmaWVsZCBpbiBFbmRwb2ludFBvcnQgb2JqZWN0cy4gT3B0aW9uYWwgaWYgb25seSBvbmUgU2VydmljZVBvcnQgaXMgZGVmaW5lZFxuICAgKiBvbiB0aGlzIHNlcnZpY2UuXG4gICAqL1xuICByZWFkb25seSBuYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcG9ydCBvbiBlYWNoIG5vZGUgb24gd2hpY2ggdGhpcyBzZXJ2aWNlIGlzIGV4cG9zZWQgd2hlbiB0eXBlPU5vZGVQb3J0XG4gICAqIG9yIExvYWRCYWxhbmNlci4gVXN1YWxseSBhc3NpZ25lZCBieSB0aGUgc3lzdGVtLiBJZiBzcGVjaWZpZWQsIGl0IHdpbGwgYmVcbiAgICogYWxsb2NhdGVkIHRvIHRoZSBzZXJ2aWNlIGlmIHVudXNlZCBvciBlbHNlIGNyZWF0aW9uIG9mIHRoZSBzZXJ2aWNlIHdpbGxcbiAgICogZmFpbC4gRGVmYXVsdCBpcyB0byBhdXRvLWFsbG9jYXRlIGEgcG9ydCBpZiB0aGUgU2VydmljZVR5cGUgb2YgdGhpcyBTZXJ2aWNlXG4gICAqIHJlcXVpcmVzIG9uZS5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy9zZXJ2aWNlcy1uZXR3b3JraW5nL3NlcnZpY2UvI3R5cGUtbm9kZXBvcnRcbiAgICpcbiAgICogQGRlZmF1bHQgLSBhdXRvLWFsbG9jYXRlIGEgcG9ydCBpZiB0aGUgU2VydmljZVR5cGUgb2YgdGhpcyBTZXJ2aWNlIHJlcXVpcmVzIG9uZS5cbiAgICovXG4gIHJlYWRvbmx5IG5vZGVQb3J0PzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgSVAgcHJvdG9jb2wgZm9yIHRoaXMgcG9ydC4gU3VwcG9ydHMgXCJUQ1BcIiwgXCJVRFBcIiwgYW5kIFwiU0NUUFwiLiBEZWZhdWx0IGlzIFRDUC5cbiAgICpcbiAgICogQGRlZmF1bHQgUHJvdG9jb2wuVENQXG4gICAqL1xuICByZWFkb25seSBwcm90b2NvbD86IFByb3RvY29sO1xuXG4gIC8qKlxuICAgKiBUaGUgcG9ydCBudW1iZXIgdGhlIHNlcnZpY2Ugd2lsbCByZWRpcmVjdCB0by5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBUaGUgdmFsdWUgb2YgYHBvcnRgIHdpbGwgYmUgdXNlZC5cbiAgICovXG4gIHJlYWRvbmx5IHRhcmdldFBvcnQ/OiBudW1iZXI7XG59XG5cbi8qKlxuICogRGVmaW5pdGlvbiBvZiBhIHNlcnZpY2UgcG9ydC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTZXJ2aWNlUG9ydCBleHRlbmRzIFNlcnZpY2VCaW5kT3B0aW9ucyB7XG5cbiAgLyoqXG4gICAqIFRoZSBwb3J0IG51bWJlciB0aGUgc2VydmljZSB3aWxsIGJpbmQgdG8uXG4gICAqL1xuICByZWFkb25seSBwb3J0OiBudW1iZXI7XG59XG4iXX0=