"""MCP server command with streamable HTTP transport."""

import os
import typer
from typing import Optional

from basic_memory.cli.app import app
from basic_memory.config import ConfigManager, init_mcp_logging

# Import mcp instance (has lifespan that handles initialization and file sync)
from basic_memory.mcp.server import mcp as mcp_server  # pragma: no cover

# Import mcp tools to register them
import basic_memory.mcp.tools  # noqa: F401  # pragma: no cover

# Import prompts to register them
import basic_memory.mcp.prompts  # noqa: F401  # pragma: no cover
from loguru import logger

config = ConfigManager().config

if not config.cloud_mode_enabled:

    @app.command()
    def mcp(
        transport: str = typer.Option(
            "stdio", help="Transport type: stdio, streamable-http, or sse"
        ),
        host: str = typer.Option(
            "0.0.0.0", help="Host for HTTP transports (use 0.0.0.0 to allow external connections)"
        ),
        port: int = typer.Option(8000, help="Port for HTTP transports"),
        path: str = typer.Option("/mcp", help="Path prefix for streamable-http transport"),
        project: Optional[str] = typer.Option(None, help="Restrict MCP server to single project"),
    ):  # pragma: no cover
        """Run the MCP server with configurable transport options.

        This command starts an MCP server using one of three transport options:

        - stdio: Standard I/O (good for local usage)
        - streamable-http: Recommended for web deployments (default)
        - sse: Server-Sent Events (for compatibility with existing clients)

        Initialization, file sync, and cleanup are handled by the MCP server's lifespan.
        """
        # Initialize logging for MCP (file only, stdout breaks protocol)
        init_mcp_logging()

        # Validate and set project constraint if specified
        if project:
            config_manager = ConfigManager()
            project_name, _ = config_manager.get_project(project)
            if not project_name:
                typer.echo(f"No project found named: {project}", err=True)
                raise typer.Exit(1)

            # Set env var with validated project name
            os.environ["BASIC_MEMORY_MCP_PROJECT"] = project_name
            logger.info(f"MCP server constrained to project: {project_name}")

        # Run the MCP server (blocks)
        # Lifespan handles: initialization, migrations, file sync, cleanup
        logger.info(f"Starting MCP server with {transport.upper()} transport")

        if transport == "stdio":
            mcp_server.run(
                transport=transport,
            )
        elif transport == "streamable-http" or transport == "sse":
            mcp_server.run(
                transport=transport,
                host=host,
                port=port,
                path=path,
                log_level="INFO",
            )
