#
# Copyright (c) 2018-2019, FusionAuth, All Rights Reserved
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
# either express or implied. See the License for the specific
# language governing permissions and limitations under the License.
#

from fusionauth.rest_client import RESTClient, JSONBodyHandler


class FusionAuthClient:
    """The FusionAuthClient provides easy access to the FusionAuth API."""

    def __init__(self, api_key, base_url):
        """Constructs a new FusionAuthClient.

        Attributes:
            api_key: A string representing the API used to authenticate the API call to FusionAuth
            base_url: A string representing the URL use to access FusionAuth
        """
        self.api_key = api_key
        self.base_url = base_url

    def action_user(self, actionee_user_id, request):
        """
        Takes an action on a user. The user being actioned is called the "actionee" and the user taking the action is called the
        "actioner". Both user ids are required. You pass the actionee's user id into the method and the actioner's is put into the
        request object.

        Attributes:
            actionee_user_id: The actionee's user id.
            request: The action request that includes all of the information about the action being taken including
                    the id of the action, any options and the duration (if applicable).
        """
        return self.start().uri('/api/user/action') \
            .url_segment(actionee_user_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def add_user_to_family(self, family_id, request):
        """
        Adds a user to an existing family. The family id must be specified.

        Attributes:
            family_id: The id of the family.
            request: The request object that contains all of the information used to determine which user to add to the family.
        """
        return self.start().uri('/api/user/family') \
            .url_segment(family_id) \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def cancel_action(self, action_id, request):
        """
        Cancels the user action.

        Attributes:
            action_id: The action id of the action to cancel.
            request: The action request that contains the information about the cancellation.
        """
        return self.start().uri('/api/user/action') \
            .url_segment(action_id) \
            .body_handler(JSONBodyHandler(request)) \
            .delete() \
            .go()

    def change_password(self, change_password_id, request):
        """
        Changes a user's password using the change password Id. This usually occurs after an email has been sent to the user
        and they clicked on a link to reset their password.

        Attributes:
            change_password_id: The change password Id used to find the user. This value is generated by FusionAuth once the change password workflow has been initiated.
            request: The change password request that contains all of the information used to change the password.
        """
        return self.start().uri('/api/user/change-password') \
            .url_segment(change_password_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def change_password_by_identity(self, request):
        """
        Changes a user's password using their identity (login id and password). Using a loginId instead of the changePasswordId
        bypasses the email verification and allows a password to be changed directly without first calling the #forgotPassword
        method.

        Attributes:
            request: The change password request that contains all of the information used to change the password.
        """
        return self.start().uri('/api/user/change-password') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def comment_on_user(self, request):
        """
        Adds a comment to the user's account.

        Attributes:
            request: The request object that contains all of the information used to create the user comment.
        """
        return self.start().uri('/api/user/comment') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def create_application(self, application_id, request):
        """
        Creates an application. You can optionally specify an Id for the application, if not provided one will be generated.

        Attributes:
            application_id: (Optional) The Id to use for the application. If not provided a secure random UUID will be generated.
            request: The request object that contains all of the information used to create the application.
        """
        return self.start().uri('/api/application') \
            .url_segment(application_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def create_application_role(self, application_id, role_id, request):
        """
        Creates a new role for an application. You must specify the id of the application you are creating the role for.
        You can optionally specify an Id for the role inside the ApplicationRole object itself, if not provided one will be generated.

        Attributes:
            application_id: The Id of the application to create the role on.
            role_id: (Optional) The Id of the role. If not provided a secure random UUID will be generated.
            request: The request object that contains all of the information used to create the application role.
        """
        return self.start().uri('/api/application') \
            .url_segment(application_id) \
            .url_segment("role") \
            .url_segment(role_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def create_audit_log(self, request):
        """
        Creates an audit log with the message and user name (usually an email). Audit logs should be written anytime you
        make changes to the FusionAuth database. When using the FusionAuth App web interface, any changes are automatically
        written to the audit log. However, if you are accessing the API, you must write the audit logs yourself.

        Attributes:
            request: The request object that contains all of the information used to create the audit log entry.
        """
        return self.start().uri('/api/system/audit-log') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def create_consent(self, consent_id, request):
        """
        Creates a user consent type. You can optionally specify an Id for the consent type, if not provided one will be generated.

        Attributes:
            consent_id: (Optional) The Id for the consent. If not provided a secure random UUID will be generated.
            request: The request object that contains all of the information used to create the consent.
        """
        return self.start().uri('/api/consent') \
            .url_segment(consent_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def create_email_template(self, email_template_id, request):
        """
        Creates an email template. You can optionally specify an Id for the template, if not provided one will be generated.

        Attributes:
            email_template_id: (Optional) The Id for the template. If not provided a secure random UUID will be generated.
            request: The request object that contains all of the information used to create the email template.
        """
        return self.start().uri('/api/email/template') \
            .url_segment(email_template_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def create_family(self, family_id, request):
        """
        Creates a family with the user id in the request as the owner and sole member of the family. You can optionally specify an id for the
        family, if not provided one will be generated.

        Attributes:
            family_id: (Optional) The id for the family. If not provided a secure random UUID will be generated.
            request: The request object that contains all of the information used to create the family.
        """
        return self.start().uri('/api/user/family') \
            .url_segment(family_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def create_group(self, group_id, request):
        """
        Creates a group. You can optionally specify an Id for the group, if not provided one will be generated.

        Attributes:
            group_id: (Optional) The Id for the group. If not provided a secure random UUID will be generated.
            request: The request object that contains all of the information used to create the group.
        """
        return self.start().uri('/api/group') \
            .url_segment(group_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def create_group_members(self, request):
        """
        Creates a member in a group.

        Attributes:
            request: The request object that contains all of the information used to create the group member(s).
        """
        return self.start().uri('/api/group/member') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def create_identity_provider(self, identity_provider_id, request):
        """
        Creates an identity provider. You can optionally specify an Id for the identity provider, if not provided one will be generated.

        Attributes:
            identity_provider_id: (Optional) The Id of the identity provider. If not provided a secure random UUID will be generated.
            request: The request object that contains all of the information used to create the identity provider.
        """
        return self.start().uri('/api/identity-provider') \
            .url_segment(identity_provider_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def create_lambda(self, lambda_id, request):
        """
        Creates a Lambda. You can optionally specify an Id for the lambda, if not provided one will be generated.

        Attributes:
            lambda_id: (Optional) The Id for the lambda. If not provided a secure random UUID will be generated.
            request: The request object that contains all of the information used to create the lambda.
        """
        return self.start().uri('/api/lambda') \
            .url_segment(lambda_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def create_tenant(self, tenant_id, request):
        """
        Creates a tenant. You can optionally specify an Id for the tenant, if not provided one will be generated.

        Attributes:
            tenant_id: (Optional) The Id for the tenant. If not provided a secure random UUID will be generated.
            request: The request object that contains all of the information used to create the tenant.
        """
        return self.start().uri('/api/tenant') \
            .url_segment(tenant_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def create_theme(self, theme_id, request):
        """
        Creates a Theme. You can optionally specify an Id for the theme, if not provided one will be generated.

        Attributes:
            theme_id: (Optional) The Id for the theme. If not provided a secure random UUID will be generated.
            request: The request object that contains all of the information used to create the theme.
        """
        return self.start().uri('/api/theme') \
            .url_segment(theme_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def create_user(self, user_id, request):
        """
        Creates a user. You can optionally specify an Id for the user, if not provided one will be generated.

        Attributes:
            user_id: (Optional) The Id for the user. If not provided a secure random UUID will be generated.
            request: The request object that contains all of the information used to create the user.
        """
        return self.start().uri('/api/user') \
            .url_segment(user_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def create_user_action(self, user_action_id, request):
        """
        Creates a user action. This action cannot be taken on a user until this call successfully returns. Anytime after
        that the user action can be applied to any user.

        Attributes:
            user_action_id: (Optional) The Id for the user action. If not provided a secure random UUID will be generated.
            request: The request object that contains all of the information used to create the user action.
        """
        return self.start().uri('/api/user-action') \
            .url_segment(user_action_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def create_user_action_reason(self, user_action_reason_id, request):
        """
        Creates a user reason. This user action reason cannot be used when actioning a user until this call completes
        successfully. Anytime after that the user action reason can be used.

        Attributes:
            user_action_reason_id: (Optional) The Id for the user action reason. If not provided a secure random UUID will be generated.
            request: The request object that contains all of the information used to create the user action reason.
        """
        return self.start().uri('/api/user-action-reason') \
            .url_segment(user_action_reason_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def create_user_consent(self, user_consent_id, request):
        """
        Creates a single User consent.

        Attributes:
            user_consent_id: (Optional) The Id for the User consent. If not provided a secure random UUID will be generated.
            request: The request that contains the user consent information.
        """
        return self.start().uri('/api/user/consent') \
            .url_segment(user_consent_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def create_webhook(self, webhook_id, request):
        """
        Creates a webhook. You can optionally specify an Id for the webhook, if not provided one will be generated.

        Attributes:
            webhook_id: (Optional) The Id for the webhook. If not provided a secure random UUID will be generated.
            request: The request object that contains all of the information used to create the webhook.
        """
        return self.start().uri('/api/webhook') \
            .url_segment(webhook_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def deactivate_application(self, application_id):
        """
        Deactivates the application with the given Id.

        Attributes:
            application_id: The Id of the application to deactivate.
        """
        return self.start().uri('/api/application') \
            .url_segment(application_id) \
            .delete() \
            .go()

    def deactivate_user(self, user_id):
        """
        Deactivates the user with the given Id.

        Attributes:
            user_id: The Id of the user to deactivate.
        """
        return self.start().uri('/api/user') \
            .url_segment(user_id) \
            .delete() \
            .go()

    def deactivate_user_action(self, user_action_id):
        """
        Deactivates the user action with the given Id.

        Attributes:
            user_action_id: The Id of the user action to deactivate.
        """
        return self.start().uri('/api/user-action') \
            .url_segment(user_action_id) \
            .delete() \
            .go()

    def deactivate_users(self, user_ids):
        """
        Deactivates the users with the given ids.

        Attributes:
            user_ids: The ids of the users to deactivate.
        """
        return self.start().uri('/api/user/bulk') \
            .url_parameter('userId', user_ids) \
            .delete() \
            .go()

    def delete_application(self, application_id):
        """
        Hard deletes an application. This is a dangerous operation and should not be used in most circumstances. This will
        delete the application, any registrations for that application, metrics and reports for the application, all the
        roles for the application, and any other data associated with the application. This operation could take a very
        long time, depending on the amount of data in your database.

        Attributes:
            application_id: The Id of the application to delete.
        """
        return self.start().uri('/api/application') \
            .url_segment(application_id) \
            .url_parameter('hardDelete', "true") \
            .delete() \
            .go()

    def delete_application_role(self, application_id, role_id):
        """
        Hard deletes an application role. This is a dangerous operation and should not be used in most circumstances. This
        permanently removes the given role from all users that had it.

        Attributes:
            application_id: The Id of the application to deactivate.
            role_id: The Id of the role to delete.
        """
        return self.start().uri('/api/application') \
            .url_segment(application_id) \
            .url_segment("role") \
            .url_segment(role_id) \
            .delete() \
            .go()

    def delete_consent(self, consent_id):
        """
        Deletes the consent for the given Id.

        Attributes:
            consent_id: The Id of the consent to delete.
        """
        return self.start().uri('/api/consent') \
            .url_segment(consent_id) \
            .delete() \
            .go()

    def delete_email_template(self, email_template_id):
        """
        Deletes the email template for the given Id.

        Attributes:
            email_template_id: The Id of the email template to delete.
        """
        return self.start().uri('/api/email/template') \
            .url_segment(email_template_id) \
            .delete() \
            .go()

    def delete_group(self, group_id):
        """
        Deletes the group for the given Id.

        Attributes:
            group_id: The Id of the group to delete.
        """
        return self.start().uri('/api/group') \
            .url_segment(group_id) \
            .delete() \
            .go()

    def delete_group_members(self, request):
        """
        Removes users as members of a group.

        Attributes:
            request: The member request that contains all of the information used to remove members to the group.
        """
        return self.start().uri('/api/group/member') \
            .body_handler(JSONBodyHandler(request)) \
            .delete() \
            .go()

    def delete_identity_provider(self, identity_provider_id):
        """
        Deletes the identity provider for the given Id.

        Attributes:
            identity_provider_id: The Id of the identity provider to delete.
        """
        return self.start().uri('/api/identity-provider') \
            .url_segment(identity_provider_id) \
            .delete() \
            .go()

    def delete_key(self, key_od):
        """
        Deletes the key for the given Id.

        Attributes:
            key_od: The Id of the key to delete.
        """
        return self.start().uri('/api/key') \
            .url_segment(key_od) \
            .delete() \
            .go()

    def delete_lambda(self, lambda_id):
        """
        Deletes the lambda for the given Id.

        Attributes:
            lambda_id: The Id of the lambda to delete.
        """
        return self.start().uri('/api/lambda') \
            .url_segment(lambda_id) \
            .delete() \
            .go()

    def delete_registration(self, user_id, application_id):
        """
        Deletes the user registration for the given user and application.

        Attributes:
            user_id: The Id of the user whose registration is being deleted.
            application_id: The Id of the application to remove the registration for.
        """
        return self.start().uri('/api/user/registration') \
            .url_segment(user_id) \
            .url_segment(application_id) \
            .delete() \
            .go()

    def delete_tenant(self, tenant_id):
        """
        Deletes the tenant for the given Id.

        Attributes:
            tenant_id: The Id of the tenant to delete.
        """
        return self.start().uri('/api/tenant') \
            .url_segment(tenant_id) \
            .delete() \
            .go()

    def delete_theme(self, theme_id):
        """
        Deletes the theme for the given Id.

        Attributes:
            theme_id: The Id of the theme to delete.
        """
        return self.start().uri('/api/theme') \
            .url_segment(theme_id) \
            .delete() \
            .go()

    def delete_user(self, user_id):
        """
        Deletes the user for the given Id. This permanently deletes all information, metrics, reports and data associated
        with the user.

        Attributes:
            user_id: The Id of the user to delete.
        """
        return self.start().uri('/api/user') \
            .url_segment(user_id) \
            .url_parameter('hardDelete', "true") \
            .delete() \
            .go()

    def delete_user_action(self, user_action_id):
        """
        Deletes the user action for the given Id. This permanently deletes the user action and also any history and logs of
        the action being applied to any users.

        Attributes:
            user_action_id: The Id of the user action to delete.
        """
        return self.start().uri('/api/user-action') \
            .url_segment(user_action_id) \
            .url_parameter('hardDelete', "true") \
            .delete() \
            .go()

    def delete_user_action_reason(self, user_action_reason_id):
        """
        Deletes the user action reason for the given Id.

        Attributes:
            user_action_reason_id: The Id of the user action reason to delete.
        """
        return self.start().uri('/api/user-action-reason') \
            .url_segment(user_action_reason_id) \
            .delete() \
            .go()

    def delete_users(self, request):
        """
        Deletes the users with the given ids.

        Attributes:
            request: The ids of the users to delete.
        """
        return self.start().uri('/api/user/bulk') \
            .body_handler(JSONBodyHandler(request)) \
            .delete() \
            .go()

    def delete_webhook(self, webhook_id):
        """
        Deletes the webhook for the given Id.

        Attributes:
            webhook_id: The Id of the webhook to delete.
        """
        return self.start().uri('/api/webhook') \
            .url_segment(webhook_id) \
            .delete() \
            .go()

    def disable_two_factor(self, user_id, code):
        """
        Disable Two Factor authentication for a user.

        Attributes:
            user_id: The Id of the User for which you're disabling Two Factor authentication.
            code: The Two Factor code used verify the the caller knows the Two Factor secret.
        """
        return self.start().uri('/api/user/two-factor') \
            .url_parameter('userId', user_id) \
            .url_parameter('code', code) \
            .delete() \
            .go()

    def enable_two_factor(self, user_id, request):
        """
        Enable Two Factor authentication for a user.

        Attributes:
            user_id: The Id of the user to enable Two Factor authentication.
            request: The two factor enable request information.
        """
        return self.start().uri('/api/user/two-factor') \
            .url_segment(user_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def exchange_refresh_token_for_jwt(self, request):
        """
        Exchange a refresh token for a new JWT.

        Attributes:
            request: The refresh request.
        """
        return self.start().uri('/api/jwt/refresh') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def forgot_password(self, request):
        """
        Begins the forgot password sequence, which kicks off an email to the user so that they can reset their password.

        Attributes:
            request: The request that contains the information about the user so that they can be emailed.
        """
        return self.start().uri('/api/user/forgot-password') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def generate_email_verification_id(self, email):
        """
        Generate a new Email Verification Id to be used with the Verify Email API. This API will not attempt to send an
        email to the User. This API may be used to collect the verificationId for use with a third party system.

        Attributes:
            email: The email address of the user that needs a new verification email.
        """
        return self.start().uri('/api/user/verify-email') \
            .url_parameter('email', email) \
            .url_parameter('sendVerifyEmail', "false") \
            .put() \
            .go()

    def generate_key(self, key_id, request):
        """
        Generate a new RSA or EC key pair or an HMAC secret.

        Attributes:
            key_id: (Optional) The Id for the key. If not provided a secure random UUID will be generated.
            request: The request object that contains all of the information used to create the key.
        """
        return self.start().uri('/api/key/generate') \
            .url_segment(key_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def generate_registration_verification_id(self, email, application_id):
        """
        Generate a new Application Registration Verification Id to be used with the Verify Registration API. This API will not attempt to send an
        email to the User. This API may be used to collect the verificationId for use with a third party system.

        Attributes:
            email: The email address of the user that needs a new verification email.
            application_id: The Id of the application to be verified.
        """
        return self.start().uri('/api/user/verify-registration') \
            .url_parameter('email', email) \
            .url_parameter('sendVerifyPasswordEmail', "false") \
            .url_parameter('applicationId', application_id) \
            .put() \
            .go()

    def generate_two_factor_secret(self):
        """
        Generate a Two Factor secret that can be used to enable Two Factor authentication for a User. The response will contain
        both the secret and a Base32 encoded form of the secret which can be shown to a User when using a 2 Step Authentication
        application such as Google Authenticator.

        Attributes:
        """
        return self.start().uri('/api/two-factor/secret') \
            .get() \
            .go()

    def generate_two_factor_secret_using_jwt(self, encoded_jwt):
        """
        Generate a Two Factor secret that can be used to enable Two Factor authentication for a User. The response will contain
        both the secret and a Base32 encoded form of the secret which can be shown to a User when using a 2 Step Authentication
        application such as Google Authenticator.

        Attributes:
            encoded_jwt: The encoded JWT (access token).
        """
        return self.start().uri('/api/two-factor/secret') \
            .authorization("JWT " + encoded_jwt) \
            .get() \
            .go()

    def identity_provider_login(self, request):
        """
        Handles login via third-parties including Social login, external OAuth and OpenID Connect, and other
        login systems.

        Attributes:
            request: The third-party login request that contains information from the third-party login
                    providers that FusionAuth uses to reconcile the user's account.
        """
        return self.start().uri('/api/identity-provider/login') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def import_key(self, key_id, request):
        """
        Import an existing RSA or EC key pair or an HMAC secret.

        Attributes:
            key_id: (Optional) The Id for the key. If not provided a secure random UUID will be generated.
            request: The request object that contains all of the information used to create the key.
        """
        return self.start().uri('/api/key/import') \
            .url_segment(key_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def import_users(self, request):
        """
        Bulk imports multiple users. This does some validation, but then tries to run batch inserts of users. This reduces
        latency when inserting lots of users. Therefore, the error response might contain some information about failures,
        but it will likely be pretty generic.

        Attributes:
            request: The request that contains all of the information about all of the users to import.
        """
        return self.start().uri('/api/user/import') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def issue_jwt(self, application_id, encoded_jwt):
        """
        Issue a new access token (JWT) for the requested Application after ensuring the provided JWT is valid. A valid
        access token is properly signed and not expired.
        <p>
        This API may be used in an SSO configuration to issue new tokens for another application after the user has
        obtained a valid token from authentication.

        Attributes:
            application_id: The Application Id for which you are requesting a new access token be issued.
            encoded_jwt: The encoded JWT (access token).
        """
        return self.start().uri('/api/jwt/issue') \
            .authorization("JWT " + encoded_jwt) \
            .url_parameter('applicationId', application_id) \
            .get() \
            .go()

    def login(self, request):
        """
        Authenticates a user to FusionAuth. 
        
        This API optionally requires an API key. See <code>Application.loginConfiguration.requireAuthentication</code>.

        Attributes:
            request: The login request that contains the user credentials used to log them in.
        """
        return self.start().uri('/api/login') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def login_ping(self, user_id, application_id, caller_ip_address):
        """
        Sends a ping to FusionAuth indicating that the user was automatically logged into an application. When using
        FusionAuth's SSO or your own, you should call this if the user is already logged in centrally, but accesses an
        application where they no longer have a session. This helps correctly track login counts, times and helps with
        reporting.

        Attributes:
            user_id: The Id of the user that was logged in.
            application_id: The Id of the application that they logged into.
            caller_ip_address: (Optional) The IP address of the end-user that is logging in. If a null value is provided
                    the IP address will be that of the client or last proxy that sent the request.
        """
        return self.start().uri('/api/login') \
            .url_segment(user_id) \
            .url_segment(application_id) \
            .url_parameter('ipAddress', caller_ip_address) \
            .put() \
            .go()

    def logout(self, _global, refresh_token):
        """
        The Logout API is intended to be used to remove the refresh token and access token cookies if they exist on the
        client and revoke the refresh token stored. This API does nothing if the request does not contain an access
        token or refresh token cookies.

        Attributes:
            _global: When this value is set to true all of the refresh tokens issued to the owner of the
                    provided token will be revoked.
            refresh_token: (Optional) The refresh_token as a request parameter instead of coming in via a cookie.
                    If provided this takes precedence over the cookie.
        """
        return self.start().uri('/api/logout') \
            .url_parameter('global', _global) \
            .url_parameter('refreshToken', refresh_token) \
            .post() \
            .go()

    def lookup_identity_provider(self, domain):
        """
        Retrieves the identity provider for the given domain. A 200 response code indicates the domain is managed
        by a registered identity provider. A 404 indicates the domain is not managed.

        Attributes:
            domain: The domain or email address to lookup.
        """
        return self.start().uri('/api/identity-provider/lookup') \
            .url_parameter('domain', domain) \
            .get() \
            .go()

    def modify_action(self, action_id, request):
        """
        Modifies a temporal user action by changing the expiration of the action and optionally adding a comment to the
        action.

        Attributes:
            action_id: The Id of the action to modify. This is technically the user action log id.
            request: The request that contains all of the information about the modification.
        """
        return self.start().uri('/api/user/action') \
            .url_segment(action_id) \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def passwordless_login(self, request):
        """
        Complete a login request using a passwordless code

        Attributes:
            request: The passwordless login request that contains all of the information used to complete login.
        """
        return self.start().uri('/api/passwordless/login') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def reactivate_application(self, application_id):
        """
        Reactivates the application with the given Id.

        Attributes:
            application_id: The Id of the application to reactivate.
        """
        return self.start().uri('/api/application') \
            .url_segment(application_id) \
            .url_parameter('reactivate', "true") \
            .put() \
            .go()

    def reactivate_user(self, user_id):
        """
        Reactivates the user with the given Id.

        Attributes:
            user_id: The Id of the user to reactivate.
        """
        return self.start().uri('/api/user') \
            .url_segment(user_id) \
            .url_parameter('reactivate', "true") \
            .put() \
            .go()

    def reactivate_user_action(self, user_action_id):
        """
        Reactivates the user action with the given Id.

        Attributes:
            user_action_id: The Id of the user action to reactivate.
        """
        return self.start().uri('/api/user-action') \
            .url_segment(user_action_id) \
            .url_parameter('reactivate', "true") \
            .put() \
            .go()

    def reconcile_jwt(self, request):
        """
        Reconcile a User to FusionAuth using JWT issued from another Identity Provider.

        Attributes:
            request: The reconcile request that contains the data to reconcile the User.
        """
        return self.start().uri('/api/jwt/reconcile') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def refresh_user_search_index(self):
        """
        Request a refresh of the User search index. This API is not generally necessary and the search index will become consistent in a
        reasonable amount of time. There may be scenarios where you may wish to manually request an index refresh. One example may be 
        if you are using the Search API or Delete Tenant API immediately following a User Create etc, you may wish to request a refresh to
         ensure the index immediately current before making a query request to the search index.

        Attributes:
        """
        return self.start().uri('/api/user/search') \
            .put() \
            .go()

    def register(self, user_id, request):
        """
        Registers a user for an application. If you provide the User and the UserRegistration object on this request, it
        will create the user as well as register them for the application. This is called a Full Registration. However, if
        you only provide the UserRegistration object, then the user must already exist and they will be registered for the
        application. The user id can also be provided and it will either be used to look up an existing user or it will be
        used for the newly created User.

        Attributes:
            user_id: (Optional) The Id of the user being registered for the application and optionally created.
            request: The request that optionally contains the User and must contain the UserRegistration.
        """
        return self.start().uri('/api/user/registration') \
            .url_segment(user_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def remove_user_from_family(self, family_id, user_id):
        """
        Removes a user from the family with the given id.

        Attributes:
            family_id: The id of the family to remove the user from.
            user_id: The id of the user to remove from the family.
        """
        return self.start().uri('/api/user/family') \
            .url_segment(family_id) \
            .url_segment(user_id) \
            .delete() \
            .go()

    def resend_email_verification(self, email):
        """
        Re-sends the verification email to the user.

        Attributes:
            email: The email address of the user that needs a new verification email.
        """
        return self.start().uri('/api/user/verify-email') \
            .url_parameter('email', email) \
            .put() \
            .go()

    def resend_registration_verification(self, email, application_id):
        """
        Re-sends the application registration verification email to the user.

        Attributes:
            email: The email address of the user that needs a new verification email.
            application_id: The Id of the application to be verified.
        """
        return self.start().uri('/api/user/verify-registration') \
            .url_parameter('email', email) \
            .url_parameter('applicationId', application_id) \
            .put() \
            .go()

    def retrieve_action(self, action_id):
        """
        Retrieves a single action log (the log of a user action that was taken on a user previously) for the given Id.

        Attributes:
            action_id: The Id of the action to retrieve.
        """
        return self.start().uri('/api/user/action') \
            .url_segment(action_id) \
            .get() \
            .go()

    def retrieve_actions(self, user_id):
        """
        Retrieves all of the actions for the user with the given Id. This will return all time based actions that are active,
        and inactive as well as non-time based actions.

        Attributes:
            user_id: The Id of the user to fetch the actions for.
        """
        return self.start().uri('/api/user/action') \
            .url_parameter('userId', user_id) \
            .get() \
            .go()

    def retrieve_actions_preventing_login(self, user_id):
        """
        Retrieves all of the actions for the user with the given Id that are currently preventing the User from logging in.

        Attributes:
            user_id: The Id of the user to fetch the actions for.
        """
        return self.start().uri('/api/user/action') \
            .url_parameter('userId', user_id) \
            .url_parameter('preventingLogin', "true") \
            .get() \
            .go()

    def retrieve_active_actions(self, user_id):
        """
        Retrieves all of the actions for the user with the given Id that are currently active.
        An active action means one that is time based and has not been canceled, and has not ended.

        Attributes:
            user_id: The Id of the user to fetch the actions for.
        """
        return self.start().uri('/api/user/action') \
            .url_parameter('userId', user_id) \
            .url_parameter('active', "true") \
            .get() \
            .go()

    def retrieve_application(self, application_id):
        """
        Retrieves the application for the given id or all of the applications if the id is null.

        Attributes:
            application_id: (Optional) The application id.
        """
        return self.start().uri('/api/application') \
            .url_segment(application_id) \
            .get() \
            .go()

    def retrieve_applications(self):
        """
        Retrieves all of the applications.

        Attributes:
        """
        return self.start().uri('/api/application') \
            .get() \
            .go()

    def retrieve_audit_log(self, audit_log_id):
        """
        Retrieves a single audit log for the given Id.

        Attributes:
            audit_log_id: The Id of the audit log to retrieve.
        """
        return self.start().uri('/api/system/audit-log') \
            .url_segment(audit_log_id) \
            .get() \
            .go()

    def retrieve_consent(self, consent_id):
        """
        Retrieves the Consent for the given Id.

        Attributes:
            consent_id: The Id of the consent.
        """
        return self.start().uri('/api/consent') \
            .url_segment(consent_id) \
            .get() \
            .go()

    def retrieve_consents(self):
        """
        Retrieves all of the consent.

        Attributes:
        """
        return self.start().uri('/api/consent') \
            .get() \
            .go()

    def retrieve_daily_active_report(self, application_id, start, end):
        """
        Retrieves the daily active user report between the two instants. If you specify an application id, it will only
        return the daily active counts for that application.

        Attributes:
            application_id: (Optional) The application id.
            start: The start instant as UTC milliseconds since Epoch.
            end: The end instant as UTC milliseconds since Epoch.
        """
        return self.start().uri('/api/report/daily-active-user') \
            .url_parameter('applicationId', application_id) \
            .url_parameter('start', start) \
            .url_parameter('end', end) \
            .get() \
            .go()

    def retrieve_email_template(self, email_template_id):
        """
        Retrieves the email template for the given Id. If you don't specify the id, this will return all of the email templates.

        Attributes:
            email_template_id: (Optional) The Id of the email template.
        """
        return self.start().uri('/api/email/template') \
            .url_segment(email_template_id) \
            .get() \
            .go()

    def retrieve_email_template_preview(self, request):
        """
        Creates a preview of the email template provided in the request. This allows you to preview an email template that
        hasn't been saved to the database yet. The entire email template does not need to be provided on the request. This
        will create the preview based on whatever is given.

        Attributes:
            request: The request that contains the email template and optionally a locale to render it in.
        """
        return self.start().uri('/api/email/template/preview') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def retrieve_email_templates(self):
        """
        Retrieves all of the email templates.

        Attributes:
        """
        return self.start().uri('/api/email/template') \
            .get() \
            .go()

    def retrieve_event_log(self, event_log_id):
        """
        Retrieves a single event log for the given Id.

        Attributes:
            event_log_id: The Id of the event log to retrieve.
        """
        return self.start().uri('/api/system/event-log') \
            .url_segment(event_log_id) \
            .get() \
            .go()

    def retrieve_families(self, user_id):
        """
        Retrieves all of the families that a user belongs to.

        Attributes:
            user_id: The User's id
        """
        return self.start().uri('/api/user/family') \
            .url_parameter('userId', user_id) \
            .get() \
            .go()

    def retrieve_family_members_by_family_id(self, family_id):
        """
        Retrieves all of the members of a family by the unique Family Id.

        Attributes:
            family_id: The unique Id of the Family.
        """
        return self.start().uri('/api/user/family') \
            .url_segment(family_id) \
            .get() \
            .go()

    def retrieve_group(self, group_id):
        """
        Retrieves the group for the given Id.

        Attributes:
            group_id: The Id of the group.
        """
        return self.start().uri('/api/group') \
            .url_segment(group_id) \
            .get() \
            .go()

    def retrieve_groups(self):
        """
        Retrieves all of the groups.

        Attributes:
        """
        return self.start().uri('/api/group') \
            .get() \
            .go()

    def retrieve_identity_provider(self, identity_provider_id):
        """
        Retrieves the identity provider for the given id or all of the identity providers if the id is null.

        Attributes:
            identity_provider_id: (Optional) The identity provider id.
        """
        return self.start().uri('/api/identity-provider') \
            .url_segment(identity_provider_id) \
            .get() \
            .go()

    def retrieve_identity_providers(self):
        """
        Retrieves all of the identity providers.

        Attributes:
        """
        return self.start().uri('/api/identity-provider') \
            .get() \
            .go()

    def retrieve_inactive_actions(self, user_id):
        """
        Retrieves all of the actions for the user with the given Id that are currently inactive.
        An inactive action means one that is time based and has been canceled or has expired, or is not time based.

        Attributes:
            user_id: The Id of the user to fetch the actions for.
        """
        return self.start().uri('/api/user/action') \
            .url_parameter('userId', user_id) \
            .url_parameter('active', "false") \
            .get() \
            .go()

    def retrieve_inactive_applications(self):
        """
        Retrieves all of the applications that are currently inactive.

        Attributes:
        """
        return self.start().uri('/api/application') \
            .url_parameter('inactive', "true") \
            .get() \
            .go()

    def retrieve_inactive_user_actions(self):
        """
        Retrieves all of the user actions that are currently inactive.

        Attributes:
        """
        return self.start().uri('/api/user-action') \
            .url_parameter('inactive', "true") \
            .get() \
            .go()

    def retrieve_integration(self):
        """
        Retrieves the available integrations.

        Attributes:
        """
        return self.start().uri('/api/integration') \
            .get() \
            .go()

    def retrieve_jwt_public_key(self, key_id):
        """
        Retrieves the Public Key configured for verifying JSON Web Tokens (JWT) by the key Id (kid).

        Attributes:
            key_id: The Id of the public key (kid).
        """
        return self.start().uri('/api/jwt/public-key') \
            .url_parameter('kid', key_id) \
            .get() \
            .go()

    def retrieve_jwt_public_key_by_application_id(self, application_id):
        """
        Retrieves the Public Key configured for verifying the JSON Web Tokens (JWT) issued by the Login API by the Application Id.

        Attributes:
            application_id: The Id of the Application for which this key is used.
        """
        return self.start().uri('/api/jwt/public-key') \
            .url_parameter('applicationId', application_id) \
            .get() \
            .go()

    def retrieve_jwt_public_keys(self):
        """
        Retrieves all Public Keys configured for verifying JSON Web Tokens (JWT).

        Attributes:
        """
        return self.start().uri('/api/jwt/public-key') \
            .get() \
            .go()

    def retrieve_key(self, key_id):
        """
        Retrieves the key for the given Id.

        Attributes:
            key_id: The Id of the key.
        """
        return self.start().uri('/api/key') \
            .url_segment(key_id) \
            .get() \
            .go()

    def retrieve_keys(self):
        """
        Retrieves all of the keys.

        Attributes:
        """
        return self.start().uri('/api/key') \
            .get() \
            .go()

    def retrieve_lambda(self, lambda_id):
        """
        Retrieves the lambda for the given Id.

        Attributes:
            lambda_id: The Id of the lambda.
        """
        return self.start().uri('/api/lambda') \
            .url_segment(lambda_id) \
            .get() \
            .go()

    def retrieve_lambdas(self):
        """
        Retrieves all of the lambdas.

        Attributes:
        """
        return self.start().uri('/api/lambda') \
            .get() \
            .go()

    def retrieve_lambdas_by_type(self, type):
        """
        Retrieves all of the lambdas for the provided type.

        Attributes:
            type: The type of the lambda to return.
        """
        return self.start().uri('/api/lambda') \
            .url_parameter('type', type) \
            .get() \
            .go()

    def retrieve_login_report(self, application_id, start, end):
        """
        Retrieves the login report between the two instants. If you specify an application id, it will only return the
        login counts for that application.

        Attributes:
            application_id: (Optional) The application id.
            start: The start instant as UTC milliseconds since Epoch.
            end: The end instant as UTC milliseconds since Epoch.
        """
        return self.start().uri('/api/report/login') \
            .url_parameter('applicationId', application_id) \
            .url_parameter('start', start) \
            .url_parameter('end', end) \
            .get() \
            .go()

    def retrieve_monthly_active_report(self, application_id, start, end):
        """
        Retrieves the monthly active user report between the two instants. If you specify an application id, it will only
        return the monthly active counts for that application.

        Attributes:
            application_id: (Optional) The application id.
            start: The start instant as UTC milliseconds since Epoch.
            end: The end instant as UTC milliseconds since Epoch.
        """
        return self.start().uri('/api/report/monthly-active-user') \
            .url_parameter('applicationId', application_id) \
            .url_parameter('start', start) \
            .url_parameter('end', end) \
            .get() \
            .go()

    def retrieve_oauth_configuration(self, application_id):
        """
        Retrieves the Oauth2 configuration for the application for the given Application Id.

        Attributes:
            application_id: The Id of the Application to retrieve OAuth configuration.
        """
        return self.start().uri('/api/application') \
            .url_segment(application_id) \
            .url_segment("oauth-configuration") \
            .get() \
            .go()

    def retrieve_password_validation_rules(self):
        """
        Retrieves the password validation rules for a specific tenant. This method requires a tenantId to be provided 
        through the use of a Tenant scoped API key or an HTTP header X-FusionAuth-TenantId to specify the Tenant Id.
        
        This API does not require an API key.

        Attributes:
        """
        return self.start().uri('/api/tenant/password-validation-rules') \
            .get() \
            .go()

    def retrieve_password_validation_rules_with_tenant_id(self, tenant_id):
        """
        Retrieves the password validation rules for a specific tenant.
        
        This API does not require an API key.

        Attributes:
            tenant_id: The Id of the tenant.
        """
        return self.start().uri('/api/tenant/password-validation-rules') \
            .url_segment(tenant_id) \
            .get() \
            .go()

    def retrieve_pending_children(self, parent_email):
        """
        Retrieves all of the children for the given parent email address.

        Attributes:
            parent_email: The email of the parent.
        """
        return self.start().uri('/api/user/family/pending') \
            .url_parameter('parentEmail', parent_email) \
            .get() \
            .go()

    def retrieve_recent_logins(self, offset, limit):
        """
        Retrieves the last number of login records.

        Attributes:
            offset: The initial record. e.g. 0 is the last login, 100 will be the 100th most recent login.
            limit: (Optional, defaults to 10) The number of records to retrieve.
        """
        return self.start().uri('/api/user/recent-login') \
            .url_parameter('offset', offset) \
            .url_parameter('limit', limit) \
            .get() \
            .go()

    def retrieve_refresh_tokens(self, user_id):
        """
        Retrieves the refresh tokens that belong to the user with the given Id.

        Attributes:
            user_id: The Id of the user.
        """
        return self.start().uri('/api/jwt/refresh') \
            .url_parameter('userId', user_id) \
            .get() \
            .go()

    def retrieve_registration(self, user_id, application_id):
        """
        Retrieves the user registration for the user with the given id and the given application id.

        Attributes:
            user_id: The Id of the user.
            application_id: The Id of the application.
        """
        return self.start().uri('/api/user/registration') \
            .url_segment(user_id) \
            .url_segment(application_id) \
            .get() \
            .go()

    def retrieve_registration_report(self, application_id, start, end):
        """
        Retrieves the registration report between the two instants. If you specify an application id, it will only return
        the registration counts for that application.

        Attributes:
            application_id: (Optional) The application id.
            start: The start instant as UTC milliseconds since Epoch.
            end: The end instant as UTC milliseconds since Epoch.
        """
        return self.start().uri('/api/report/registration') \
            .url_parameter('applicationId', application_id) \
            .url_parameter('start', start) \
            .url_parameter('end', end) \
            .get() \
            .go()

    def retrieve_system_configuration(self):
        """
        Retrieves the system configuration.

        Attributes:
        """
        return self.start().uri('/api/system-configuration') \
            .get() \
            .go()

    def retrieve_tenant(self, tenant_id):
        """
        Retrieves the tenant for the given Id.

        Attributes:
            tenant_id: The Id of the tenant.
        """
        return self.start().uri('/api/tenant') \
            .url_segment(tenant_id) \
            .get() \
            .go()

    def retrieve_tenants(self):
        """
        Retrieves all of the tenants.

        Attributes:
        """
        return self.start().uri('/api/tenant') \
            .get() \
            .go()

    def retrieve_theme(self, theme_id):
        """
        Retrieves the theme for the given Id.

        Attributes:
            theme_id: The Id of the theme.
        """
        return self.start().uri('/api/theme') \
            .url_segment(theme_id) \
            .get() \
            .go()

    def retrieve_themes(self):
        """
        Retrieves all of the themes.

        Attributes:
        """
        return self.start().uri('/api/theme') \
            .get() \
            .go()

    def retrieve_total_report(self):
        """
        Retrieves the totals report. This contains all of the total counts for each application and the global registration
        count.

        Attributes:
        """
        return self.start().uri('/api/report/totals') \
            .get() \
            .go()

    def retrieve_user(self, user_id):
        """
        Retrieves the user for the given Id.

        Attributes:
            user_id: The Id of the user.
        """
        return self.start().uri('/api/user') \
            .url_segment(user_id) \
            .get() \
            .go()

    def retrieve_user_action(self, user_action_id):
        """
        Retrieves the user action for the given Id. If you pass in null for the id, this will return all of the user
        actions.

        Attributes:
            user_action_id: (Optional) The Id of the user action.
        """
        return self.start().uri('/api/user-action') \
            .url_segment(user_action_id) \
            .get() \
            .go()

    def retrieve_user_action_reason(self, user_action_reason_id):
        """
        Retrieves the user action reason for the given Id. If you pass in null for the id, this will return all of the user
        action reasons.

        Attributes:
            user_action_reason_id: (Optional) The Id of the user action reason.
        """
        return self.start().uri('/api/user-action-reason') \
            .url_segment(user_action_reason_id) \
            .get() \
            .go()

    def retrieve_user_action_reasons(self):
        """
        Retrieves all the user action reasons.

        Attributes:
        """
        return self.start().uri('/api/user-action-reason') \
            .get() \
            .go()

    def retrieve_user_actions(self):
        """
        Retrieves all of the user actions.

        Attributes:
        """
        return self.start().uri('/api/user-action') \
            .get() \
            .go()

    def retrieve_user_by_change_password_id(self, change_password_id):
        """
        Retrieves the user by a change password Id. The intended use of this API is to retrieve a user after the forgot
        password workflow has been initiated and you may not know the user's email or username.

        Attributes:
            change_password_id: The unique change password Id that was sent via email or returned by the Forgot Password API.
        """
        return self.start().uri('/api/user') \
            .url_parameter('changePasswordId', change_password_id) \
            .get() \
            .go()

    def retrieve_user_by_email(self, email):
        """
        Retrieves the user for the given email.

        Attributes:
            email: The email of the user.
        """
        return self.start().uri('/api/user') \
            .url_parameter('email', email) \
            .get() \
            .go()

    def retrieve_user_by_login_id(self, login_id):
        """
        Retrieves the user for the loginId. The loginId can be either the username or the email.

        Attributes:
            login_id: The email or username of the user.
        """
        return self.start().uri('/api/user') \
            .url_parameter('loginId', login_id) \
            .get() \
            .go()

    def retrieve_user_by_username(self, username):
        """
        Retrieves the user for the given username.

        Attributes:
            username: The username of the user.
        """
        return self.start().uri('/api/user') \
            .url_parameter('username', username) \
            .get() \
            .go()

    def retrieve_user_by_verification_id(self, verification_id):
        """
        Retrieves the user by a verificationId. The intended use of this API is to retrieve a user after the forgot
        password workflow has been initiated and you may not know the user's email or username.

        Attributes:
            verification_id: The unique verification Id that has been set on the user object.
        """
        return self.start().uri('/api/user') \
            .url_parameter('verificationId', verification_id) \
            .get() \
            .go()

    def retrieve_user_comments(self, user_id):
        """
        Retrieves all of the comments for the user with the given Id.

        Attributes:
            user_id: The Id of the user.
        """
        return self.start().uri('/api/user/comment') \
            .url_segment(user_id) \
            .get() \
            .go()

    def retrieve_user_consent(self, user_consent_id):
        """
        Retrieve a single User consent by Id.

        Attributes:
            user_consent_id: The User consent Id
        """
        return self.start().uri('/api/user/consent') \
            .url_segment(user_consent_id) \
            .get() \
            .go()

    def retrieve_user_consents(self, user_id):
        """
        Retrieves all of the consents for a User.

        Attributes:
            user_id: The User's Id
        """
        return self.start().uri('/api/user/consent') \
            .url_parameter('userId', user_id) \
            .get() \
            .go()

    def retrieve_user_login_report(self, application_id, user_id, start, end):
        """
        Retrieves the login report between the two instants for a particular user by Id. If you specify an application id, it will only return the
        login counts for that application.

        Attributes:
            application_id: (Optional) The application id.
            user_id: The userId id.
            start: The start instant as UTC milliseconds since Epoch.
            end: The end instant as UTC milliseconds since Epoch.
        """
        return self.start().uri('/api/report/login') \
            .url_parameter('applicationId', application_id) \
            .url_parameter('userId', user_id) \
            .url_parameter('start', start) \
            .url_parameter('end', end) \
            .get() \
            .go()

    def retrieve_user_login_report_by_login_id(self, application_id, login_id, start, end):
        """
        Retrieves the login report between the two instants for a particular user by login Id. If you specify an application id, it will only return the
        login counts for that application.

        Attributes:
            application_id: (Optional) The application id.
            login_id: The userId id.
            start: The start instant as UTC milliseconds since Epoch.
            end: The end instant as UTC milliseconds since Epoch.
        """
        return self.start().uri('/api/report/login') \
            .url_parameter('applicationId', application_id) \
            .url_parameter('loginId', login_id) \
            .url_parameter('start', start) \
            .url_parameter('end', end) \
            .get() \
            .go()

    def retrieve_user_recent_logins(self, user_id, offset, limit):
        """
        Retrieves the last number of login records for a user.

        Attributes:
            user_id: The Id of the user.
            offset: The initial record. e.g. 0 is the last login, 100 will be the 100th most recent login.
            limit: (Optional, defaults to 10) The number of records to retrieve.
        """
        return self.start().uri('/api/user/recent-login') \
            .url_parameter('userId', user_id) \
            .url_parameter('offset', offset) \
            .url_parameter('limit', limit) \
            .get() \
            .go()

    def retrieve_user_using_jwt(self, encoded_jwt):
        """
        Retrieves the user for the given Id. This method does not use an API key, instead it uses a JSON Web Token (JWT) for authentication.

        Attributes:
            encoded_jwt: The encoded JWT (access token).
        """
        return self.start().uri('/api/user') \
            .authorization("JWT " + encoded_jwt) \
            .get() \
            .go()

    def retrieve_webhook(self, webhook_id):
        """
        Retrieves the webhook for the given Id. If you pass in null for the id, this will return all the webhooks.

        Attributes:
            webhook_id: (Optional) The Id of the webhook.
        """
        return self.start().uri('/api/webhook') \
            .url_segment(webhook_id) \
            .get() \
            .go()

    def retrieve_webhooks(self):
        """
        Retrieves all the webhooks.

        Attributes:
        """
        return self.start().uri('/api/webhook') \
            .get() \
            .go()

    def revoke_refresh_token(self, token, user_id, application_id):
        """
        Revokes a single refresh token, all tokens for a user or all tokens for an application. If you provide a user id
        and an application id, this will delete all the refresh tokens for that user for that application.

        Attributes:
            token: (Optional) The refresh token to delete.
            user_id: (Optional) The user id whose tokens to delete.
            application_id: (Optional) The application id of the tokens to delete.
        """
        return self.start().uri('/api/jwt/refresh') \
            .url_parameter('token', token) \
            .url_parameter('userId', user_id) \
            .url_parameter('applicationId', application_id) \
            .delete() \
            .go()

    def revoke_user_consent(self, user_consent_id):
        """
        Revokes a single User consent by Id.

        Attributes:
            user_consent_id: The User Consent Id
        """
        return self.start().uri('/api/user/consent') \
            .url_segment(user_consent_id) \
            .delete() \
            .go()

    def search_audit_logs(self, request):
        """
        Searches the audit logs with the specified criteria and pagination.

        Attributes:
            request: The search criteria and pagination information.
        """
        return self.start().uri('/api/system/audit-log/search') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def search_event_logs(self, request):
        """
        Searches the event logs with the specified criteria and pagination.

        Attributes:
            request: The search criteria and pagination information.
        """
        return self.start().uri('/api/system/event-log/search') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def search_login_records(self, request):
        """
        Searches the login records with the specified criteria and pagination.

        Attributes:
            request: The search criteria and pagination information.
        """
        return self.start().uri('/api/system/login-record/search') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def search_users(self, ids):
        """
        Retrieves the users for the given ids. If any id is invalid, it is ignored.

        Attributes:
            ids: The user ids to search for.
        """
        return self.start().uri('/api/user/search') \
            .url_parameter('ids', ids) \
            .get() \
            .go()

    def search_users_by_query_string(self, request):
        """
        Retrieves the users for the given search criteria and pagination.

        Attributes:
            request: The search criteria and pagination constraints. Fields used: queryString, numberOfResults, startRow,
                    and sort fields.
        """
        return self.start().uri('/api/user/search') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def send_email(self, email_template_id, request):
        """
        Send an email using an email template id. You can optionally provide <code>requestData</code> to access key value
        pairs in the email template.

        Attributes:
            email_template_id: The id for the template.
            request: The send email request that contains all of the information used to send the email.
        """
        return self.start().uri('/api/email/send') \
            .url_segment(email_template_id) \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def send_family_request_email(self, request):
        """
        Sends out an email to a parent that they need to register and create a family or need to log in and add a child to their existing family.

        Attributes:
            request: The request object that contains the parent email.
        """
        return self.start().uri('/api/user/family/request') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def send_passwordless_code(self, request):
        """
        Send a passwordless authentication code in an email to complete login.

        Attributes:
            request: The passwordless send request that contains all of the information used to send an email containing a code.
        """
        return self.start().uri('/api/passwordless/send') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def send_two_factor_code(self, request):
        """
        Send a Two Factor authentication code to assist in setting up Two Factor authentication or disabling.

        Attributes:
            request: The request object that contains all of the information used to send the code.
        """
        return self.start().uri('/api/two-factor/send') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def send_two_factor_code_for_login(self, two_factor_id):
        """
        Send a Two Factor authentication code to allow the completion of Two Factor authentication.

        Attributes:
            two_factor_id: The Id returned by the Login API necessary to complete Two Factor authentication.
        """
        return self.start().uri('/api/two-factor/send') \
            .url_segment(two_factor_id) \
            .post() \
            .go()

    def start_passwordless_login(self, request):
        """
        Start a passwordless login request by generating a passwordless code. This code can be sent to the User using the Send
        Passwordless Code API or using a mechanism outside of FusionAuth. The passwordless login is completed by using the Passwordless Login API with this code.

        Attributes:
            request: The passwordless start request that contains all of the information used to begin the passwordless login request.
        """
        return self.start().uri('/api/passwordless/start') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def two_factor_login(self, request):
        """
        Complete login using a 2FA challenge

        Attributes:
            request: The login request that contains the user credentials used to log them in.
        """
        return self.start().uri('/api/two-factor/login') \
            .body_handler(JSONBodyHandler(request)) \
            .post() \
            .go()

    def update_application(self, application_id, request):
        """
        Updates the application with the given Id.

        Attributes:
            application_id: The Id of the application to update.
            request: The request that contains all of the new application information.
        """
        return self.start().uri('/api/application') \
            .url_segment(application_id) \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def update_application_role(self, application_id, role_id, request):
        """
        Updates the application role with the given id for the application.

        Attributes:
            application_id: The Id of the application that the role belongs to.
            role_id: The Id of the role to update.
            request: The request that contains all of the new role information.
        """
        return self.start().uri('/api/application') \
            .url_segment(application_id) \
            .url_segment("role") \
            .url_segment(role_id) \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def update_consent(self, consent_id, request):
        """
        Updates the consent with the given Id.

        Attributes:
            consent_id: The Id of the consent to update.
            request: The request that contains all of the new consent information.
        """
        return self.start().uri('/api/consent') \
            .url_segment(consent_id) \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def update_email_template(self, email_template_id, request):
        """
        Updates the email template with the given Id.

        Attributes:
            email_template_id: The Id of the email template to update.
            request: The request that contains all of the new email template information.
        """
        return self.start().uri('/api/email/template') \
            .url_segment(email_template_id) \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def update_group(self, group_id, request):
        """
        Updates the group with the given Id.

        Attributes:
            group_id: The Id of the group to update.
            request: The request that contains all of the new group information.
        """
        return self.start().uri('/api/group') \
            .url_segment(group_id) \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def update_identity_provider(self, identity_provider_id, request):
        """
        Updates the identity provider with the given Id.

        Attributes:
            identity_provider_id: The Id of the identity provider to update.
            request: The request object that contains the updated identity provider.
        """
        return self.start().uri('/api/identity-provider') \
            .url_segment(identity_provider_id) \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def update_integrations(self, request):
        """
        Updates the available integrations.

        Attributes:
            request: The request that contains all of the new integration information.
        """
        return self.start().uri('/api/integration') \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def update_key(self, key_id, request):
        """
        Updates the key with the given Id.

        Attributes:
            key_id: The Id of the key to update.
            request: The request that contains all of the new key information.
        """
        return self.start().uri('/api/key') \
            .url_segment(key_id) \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def update_lambda(self, lambda_id, request):
        """
        Updates the lambda with the given Id.

        Attributes:
            lambda_id: The Id of the lambda to update.
            request: The request that contains all of the new lambda information.
        """
        return self.start().uri('/api/lambda') \
            .url_segment(lambda_id) \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def update_registration(self, user_id, request):
        """
        Updates the registration for the user with the given id and the application defined in the request.

        Attributes:
            user_id: The Id of the user whose registration is going to be updated.
            request: The request that contains all of the new registration information.
        """
        return self.start().uri('/api/user/registration') \
            .url_segment(user_id) \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def update_system_configuration(self, request):
        """
        Updates the system configuration.

        Attributes:
            request: The request that contains all of the new system configuration information.
        """
        return self.start().uri('/api/system-configuration') \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def update_tenant(self, tenant_id, request):
        """
        Updates the tenant with the given Id.

        Attributes:
            tenant_id: The Id of the tenant to update.
            request: The request that contains all of the new tenant information.
        """
        return self.start().uri('/api/tenant') \
            .url_segment(tenant_id) \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def update_theme(self, theme_id, request):
        """
        Updates the theme with the given Id.

        Attributes:
            theme_id: The Id of the theme to update.
            request: The request that contains all of the new theme information.
        """
        return self.start().uri('/api/theme') \
            .url_segment(theme_id) \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def update_user(self, user_id, request):
        """
        Updates the user with the given Id.

        Attributes:
            user_id: The Id of the user to update.
            request: The request that contains all of the new user information.
        """
        return self.start().uri('/api/user') \
            .url_segment(user_id) \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def update_user_action(self, user_action_id, request):
        """
        Updates the user action with the given Id.

        Attributes:
            user_action_id: The Id of the user action to update.
            request: The request that contains all of the new user action information.
        """
        return self.start().uri('/api/user-action') \
            .url_segment(user_action_id) \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def update_user_action_reason(self, user_action_reason_id, request):
        """
        Updates the user action reason with the given Id.

        Attributes:
            user_action_reason_id: The Id of the user action reason to update.
            request: The request that contains all of the new user action reason information.
        """
        return self.start().uri('/api/user-action-reason') \
            .url_segment(user_action_reason_id) \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def update_user_consent(self, user_consent_id, request):
        """
        Updates a single User consent by Id.

        Attributes:
            user_consent_id: The User Consent Id
            request: The request that contains the user consent information.
        """
        return self.start().uri('/api/user/consent') \
            .url_segment(user_consent_id) \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def update_webhook(self, webhook_id, request):
        """
        Updates the webhook with the given Id.

        Attributes:
            webhook_id: The Id of the webhook to update.
            request: The request that contains all of the new webhook information.
        """
        return self.start().uri('/api/webhook') \
            .url_segment(webhook_id) \
            .body_handler(JSONBodyHandler(request)) \
            .put() \
            .go()

    def validate_jwt(self, encoded_jwt):
        """
        Validates the provided JWT (encoded JWT string) to ensure the token is valid. A valid access token is properly
        signed and not expired.
        <p>
        This API may be used to verify the JWT as well as decode the encoded JWT into human readable identity claims.

        Attributes:
            encoded_jwt: The encoded JWT (access token).
        """
        return self.start().uri('/api/jwt/validate') \
            .authorization("JWT " + encoded_jwt) \
            .get() \
            .go()

    def verify_email(self, verification_id):
        """
        Confirms a email verification. The Id given is usually from an email sent to the user.

        Attributes:
            verification_id: The email verification id sent to the user.
        """
        return self.start().uri('/api/user/verify-email') \
            .url_segment(verification_id) \
            .post() \
            .go()

    def verify_registration(self, verification_id):
        """
        Confirms an application registration. The Id given is usually from an email sent to the user.

        Attributes:
            verification_id: The registration verification Id sent to the user.
        """
        return self.start().uri('/api/user/verify-registration') \
            .url_segment(verification_id) \
            .post() \
            .go()

    def start(self):
        return RESTClient().authorization(self.api_key).url(self.base_url)
