"use strict";
// Copyright (c) TileDB
// Distributed under the terms of the Modified BSD License.
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const base_1 = require("@jupyter-widgets/base");
const d3 = __importStar(require("d3"));
const debounce_1 = __importDefault(require("./debounce"));
const version_1 = require("./version");
require("../css/widget.css");
const worker_js_1 = __importDefault(require("file-loader!../lib/worker.js"));
class DagVisualizeModel extends base_1.DOMWidgetModel {
    defaults() {
        return Object.assign({}, super.defaults(), { _model_name: DagVisualizeModel.model_name, _model_module: DagVisualizeModel.model_module, _model_module_version: DagVisualizeModel.model_module_version, _view_name: DagVisualizeModel.view_name, _view_module: DagVisualizeModel.view_module, _view_module_version: DagVisualizeModel.view_module_version, value: '' });
    }
}
DagVisualizeModel.serializers = Object.assign({}, base_1.DOMWidgetModel.serializers);
DagVisualizeModel.model_name = 'DagVisualizeModel';
DagVisualizeModel.model_module = version_1.MODULE_NAME;
DagVisualizeModel.model_module_version = version_1.MODULE_VERSION;
DagVisualizeModel.view_name = 'DagVisualizeView'; // Set to null if no view
DagVisualizeModel.view_module = version_1.MODULE_NAME; // Set to null if no view
DagVisualizeModel.view_module_version = version_1.MODULE_VERSION;
exports.DagVisualizeModel = DagVisualizeModel;
class DagVisualizeView extends base_1.DOMWidgetView {
    render() {
        this.el.classList.add('tiledb-widget');
        this.createSVG();
        this.value_changed();
        /**
         * Debounce rendering function so it won't rerender too fast
         */
        const debouncedOnChange = debounce_1.default(this.value_changed.bind(this), 1500);
        this.model.on('change:value', debouncedOnChange, this);
    }
    value_changed() {
        this.data = JSON.parse(this.model.get('value'));
        /**
         * Reset html and build new graph
         */
        this.createDag();
    }
    calculateBounds(positions) {
        if (typeof this.bounds === 'undefined') {
            const xNums = Object.keys(positions).map((pos) => positions[pos][0]);
            const yNums = Object.keys(positions).map((pos) => positions[pos][1]);
            const padding = 30;
            const verticalPadding = 60;
            const maxHorizontalCoordinate = Math.max(...xNums);
            let maxVerticalCoordinate = Math.max(...yNums);
            // We don't want the ratio of width / height to be too disproportionate
            const constrainRatio = .25;
            maxVerticalCoordinate = Math.max(maxHorizontalCoordinate * constrainRatio, maxVerticalCoordinate + verticalPadding);
            this.bounds = [maxHorizontalCoordinate + padding, maxVerticalCoordinate];
        }
        return this.bounds;
    }
    createSVG() {
        this.wrapper = d3.select(this.el).append('svg').append('g');
        this.svg = d3.select(this.el).select('svg');
        this.createControls();
        this.createTooltip();
    }
    zoom() {
        if (this.initialized) {
            return;
        }
        const [width, height] = this.bounds;
        const svg = this.svg;
        const zoom = d3.zoom().translateExtent([[0, 0], [width, height]]).on('zoom', () => {
            this.wrapper.attr('transform', d3.event.transform);
        });
        svg.call(zoom).on('wheel.zoom', null);
        function zoomHandler() {
            d3.event.preventDefault();
            const direction = (this.id === 'zoom_in') ? .2 : -.2;
            /**
             * In SVG 1.1 <svg> elements did not support transform attributes. In SVG 2 it is proposed that they should.
             * Chrome and Firefox implement this part of the SVG 2 specification, Safari does not yet do so and IE11 never will.
             * That's why we apply transform to the "g" element instead of the "svg"
             */
            svg.transition().duration(300).call(zoom.scaleBy, 1 + direction);
        }
        function resetHandler() {
            svg.transition().duration(300).call(zoom.transform, d3.zoomIdentity);
        }
        setTimeout(() => {
            d3.select(this.el).selectAll('.zoomControl').on('click', zoomHandler);
            d3.select(this.el).selectAll('.resetControl').on('click', resetHandler);
        }, 0);
        this.initialized = true;
    }
    createTooltip() {
        this.tooltip = d3.select(this.el).append('div')
            .attr('class', 'tooltip')
            .style("opacity", 0);
    }
    /**
     * Method to calculate vertical offset from the hidden "fake" root node
     * @param descendants The hierarchy points
     * @param circleSize The size of the circle
     * @param fauxRootNode The name of the fake root node
     */
    calculateYOffset(descendants, circleSize, fauxRootNode) {
        /**
         * If we have already calculated the offset just return it.
         * If not calculate the offset
         */
        if (typeof this.verticalOffset === 'undefined') {
            const originalRoot = descendants.find((node) => Boolean(node.parent && node.parent.id === fauxRootNode));
            /**
             * Calculate the offset of the original root node,
             * since the first faux root node that we added,
             * we hide it and create an empty space to the top.
             */
            const yOffset = originalRoot ? originalRoot.y : 0;
            this.verticalOffset = circleSize - yOffset / 2;
        }
        return this.verticalOffset;
    }
    getRootNodes(nodes, edges) {
        const hasNoParent = (node) => edges.every(([, parent]) => node !== parent);
        return nodes.filter(hasNoParent);
    }
    createDag() {
        const { nodes, edges, node_details, positions } = this.data;
        const bounds = this.calculateBounds(positions);
        /**
         * Sometimes during updates we are getting different/weird positions object
         * So we save and re-use the first positions object we are getting
         */
        this.positions = this.positions || positions;
        this.svg.attr("viewBox", "0 0 " + bounds[0] + " " + bounds[1]);
        this.zoom();
        const numberOfNodes = nodes.length;
        const biggestSide = Math.max(...bounds) - 20; // Remove padding
        const circleSize = Math.min((biggestSide / numberOfNodes), 30);
        const links = edges.map(([parent, child]) => ({
            source: parent,
            target: child,
        }));
        const padding = 20;
        const nodeDetails = Object.keys(node_details).map((node, i) => ({
            index: i,
            status: node_details[node].status,
            id: node,
            fx: this.positions[node][0],
            fy: this.positions[node][1] + padding,
        }));
        const worker = new Worker(worker_js_1.default);
        worker.postMessage({
            nodes: nodeDetails,
            links
        });
        worker.onmessage = (event) => {
            if (event.data.type !== 'end') {
                return;
            }
            /**
             * Remove previous contents
             */
            this.wrapper.selectAll("*").remove();
            const { nodes, links } = event.data;
            this.wrapper.append("g")
                .selectAll("path")
                .data(links)
                .enter().append("path")
                .attr('d', (d) => {
                return `M${d.source.x},${d.source.y} C ${d.source.x},${(d.source.y + d.target.y) / 2} ${d.target.x},${(d.source.y + d.target.y) / 2} ${d.target.x},${d.target.y}`;
            })
                .attr('class', (d) => `path-${d.target.status}`);
            this.wrapper.append("g")
                .selectAll("circle")
                .data(nodes)
                .enter().append("circle")
                .attr("cx", (d) => d.x)
                .attr("cy", (d) => d.y)
                .attr("r", circleSize)
                .attr('class', (d) => d.status)
                .on('mouseover', (d) => {
                this.tooltip.transition()
                    .duration(200)
                    .style('opacity', .9);
                this.tooltip.html(`<p>${d.id}: ${d.status}</p>`)
                    .style('left', `${d3.event.clientX + 10}px`)
                    .style('top', `${d3.event.clientY + 10}px`);
            }).on('mouseout', () => {
                this.tooltip.transition()
                    .duration(500)
                    .style('opacity', 0);
            });
        };
    }
    createControls() {
        const zoomInButton = document.createElement('button');
        const zoomOutButton = document.createElement('button');
        const resetButton = document.createElement('button');
        const className = 'zoomControl';
        zoomInButton.id = 'zoom_in';
        zoomOutButton.id = 'zoom_out';
        resetButton.className = 'resetControl';
        zoomInButton.className = className;
        zoomOutButton.className = className;
        this.el.append(zoomInButton);
        this.el.append(zoomOutButton);
        this.el.append(resetButton);
    }
}
exports.DagVisualizeView = DagVisualizeView;
//# sourceMappingURL=widget.js.map