#!/usr/bin/env python3

from typing import Dict, List, Any
import json


class TDDCorePrompt:
    """Core TDD principles and Red-Green-Refactor cycle enforcement rules"""

    @staticmethod
    def get_tdd_principles() -> str:
        return """
# TDD CORE PRINCIPLES

## Red-Green-Refactor Cycle Enforcement

**THE FUNDAMENTAL RULE**: Only ONE new test per operation. Multiple new tests = TDD violation.

### RED PHASE - Write failing test
1. **Red Phase**: Write ONE failing test that describes desired behavior
   - Test should fail for the right reason (not due to syntax errors)
   - Test should clearly specify expected behavior
   - Focus on WHAT the code should do, not HOW

### GREEN PHASE - Minimal implementation  
2. **Green Phase**: Write MINIMAL code to make the test pass
   - Write the simplest possible implementation
   - Don't anticipate future requirements
   - Avoid over-engineering or premature optimization
   - Implementation should be just enough to satisfy the failing test

### REFACTOR PHASE - Improve design
3. **Refactor Phase**: Improve code structure while keeping tests green
   - Improve design without changing behavior
   - Extract methods, rename variables, remove duplication
   - All tests must remain green during refactoring

## CRITICAL TDD VALIDATION RULES

### New Test Detection
- **Multiple new tests in one operation = VIOLATION**
- **Character-by-character comparison** of old vs new content required
- Count only TRULY NEW tests, not modifications to existing tests
- Test modifications (changing assertions, adding setup) = ALLOWED
- Adding entirely new test functions/methods = COUNTED as new tests

### Implementation Scope Validation  
- **Over-implementation beyond test requirements = VIOLATION**
- Implementation must be justified by test output/failures
- If test says "function not defined" → create empty function only
- If test says "wrong return value" → return correct value only
- Don't implement features not required by current failing tests

### Test vs Implementation File Classification
- **Test files**: Contains test functions, imports test frameworks
- **Implementation files**: Contains business logic, actual functionality  
- **Test file patterns**: `*.test.js`, `*.spec.py`, `test_*.py`, `*_test.go`
- **Writing tests without implementation changes** = ALLOWED
- **Writing implementation without corresponding tests** = VIOLATION
"""


class EditAnalysisPrompt:
    """Analysis prompt for Edit operations - ported from TDD Guard's edit-analysis.ts"""

    @staticmethod
    def get_analysis_prompt(old_content: str, new_content: str, file_path: str) -> str:
        return f"""
# EDIT OPERATION ANALYSIS

**OPERATION**: Edit
**FILE**: {file_path}
**TASK**: Analyze the differences between old and new content to detect TDD violations

## CRITICAL ANALYSIS REQUIREMENTS

### New Test Count Detection
**CRITICAL**: A test is only "new" if it doesn't exist in the old content.
- Compare old content vs new content character by character
- Look for completely new test functions/methods that didn't exist before
- **One new test = ALLOWED**
- **Two or more new tests = VIOLATION** 

### Test Function Patterns
Identify test functions using these patterns:
- Python: `def test_*`, `def should_*`, methods in TestCase classes
- JavaScript: `test(`, `it(`, `describe(`, `expect(`
- Go: `func Test*`, `func Benchmark*`
- Java: `@Test` annotations

### Content Analysis
OLD_CONTENT:
```
{old_content}
```

NEW_CONTENT:  
```
{new_content}
```

### Analysis Steps
1. **Identify test vs implementation file** based on file path and content
2. **Extract test functions** from both old and new content
3. **Count truly new tests** (not modifications of existing tests)
4. **Assess implementation scope** - is it minimal and test-driven?
5. **Check for over-implementation** beyond current test requirements

### TDD Compliance Check
- Is this adding multiple new tests in one operation?
- Is the implementation minimal and focused on making tests pass?
- Does the change follow Red-Green-Refactor principles?
- Are there any premature optimizations or over-engineering?

**VALIDATION DECISION**: Analyze and determine if this Edit operation violates TDD principles.
"""


class WriteAnalysisPrompt:
    """Analysis prompt for Write operations - ported from TDD Guard's write-analysis.ts"""

    @staticmethod
    def get_analysis_prompt(file_path: str, content: str) -> str:
        is_test_file = any(
            pattern in file_path.lower()
            for pattern in ["test", "spec", "_test.", ".test.", "tests/"]
        )

        file_type = "TEST FILE" if is_test_file else "IMPLEMENTATION FILE"

        return f"""
# WRITE OPERATION ANALYSIS

**OPERATION**: Write (creating new file)
**FILE**: {file_path}
**FILE_TYPE**: {file_type}
**TASK**: Validate new file creation against TDD principles

## FILE CONTENT
```
{content}
```

## TDD VALIDATION FOR NEW FILE CREATION

### Test File Creation Rules
If this is a **TEST FILE**:
- ✅ **ALLOWED**: Writing test files with multiple tests
- ✅ **ALLOWED**: Setting up test infrastructure, fixtures, helpers
- ✅ **ALLOWED**: Comprehensive test coverage for new features
- ⚠️  **CHECK**: Ensure tests are meaningful and not just placeholders

### Implementation File Creation Rules  
If this is an **IMPLEMENTATION FILE**:
- ⚠️  **CRITICAL CHECK**: Is there a corresponding failing test?
- ❌ **VIOLATION**: Creating implementation without tests (Test-Last approach)
- ✅ **ALLOWED**: Creating minimal implementation to make existing tests pass
- ❌ **VIOLATION**: Over-implementing beyond current test requirements

### Content Analysis Requirements
1. **File Type Classification**: Test vs Implementation
2. **Purpose Assessment**: What is this file trying to achieve?
3. **Test Coverage**: Are there corresponding tests for implementation files?
4. **Implementation Scope**: Is implementation minimal and test-driven?
5. **TDD Phase**: Which phase of Red-Green-Refactor is this?

### Specific Validation Checks
- **For Test Files**: Count number of test functions being created
- **For Implementation Files**: Verify corresponding test existence
- **For Both**: Check for over-engineering or premature optimization
- **TDD Compliance**: Does this follow proper Red-Green-Refactor flow?

### Test Output Integration
If test results are available, validate:
- Does implementation address specific test failures?
- Is implementation minimal enough to just make tests pass?
- Are there any features implemented beyond test requirements?

**VALIDATION DECISION**: Determine if this Write operation maintains TDD discipline.
"""


class MultiEditAnalysisPrompt:
    """Analysis prompt for MultiEdit operations - ported from TDD Guard's multi-edit-analysis.ts"""

    @staticmethod
    def get_analysis_prompt(edits: List[Dict[str, Any]]) -> str:
        edits_summary = []
        for i, edit in enumerate(edits, 1):
            file_path = edit.get("file_path", f"Edit {i}")
            old_content = (
                edit.get("old_string", "")[:200] + "..."
                if len(edit.get("old_string", "")) > 200
                else edit.get("old_string", "")
            )
            new_content = (
                edit.get("new_string", "")[:200] + "..."
                if len(edit.get("new_string", "")) > 200
                else edit.get("new_string", "")
            )

            edits_summary.append(
                f"""
**EDIT {i}**: {file_path}
OLD: {old_content}
NEW: {new_content}
"""
            )

        return f"""
# MULTI-EDIT OPERATION ANALYSIS

**OPERATION**: MultiEdit (sequential edits to file)
**TASK**: Analyze cumulative changes across all edits for TDD violations

## ALL EDITS SUMMARY
{chr(10).join(edits_summary)}

## CRITICAL MULTI-EDIT TDD VALIDATION

### Cumulative New Test Counting
**MOST IMPORTANT**: Count total new tests across ALL edits combined:
- Analyze each edit individually for new test functions
- **Track cumulative new test count** across all edits
- **One total new test across all edits = ALLOWED**
- **Two or more total new tests across all edits = VIOLATION**

### Sequential Edit Analysis Steps
1. **Process edits in order** as they would be applied
2. **Build cumulative content state** after each edit
3. **Identify new test additions** in each edit step
4. **Track running total** of new tests being added
5. **Assess overall implementation scope** across all changes

### Multi-Edit Specific Violations
- **Incremental over-implementation**: Building too much functionality step by step
- **Test batching**: Adding multiple tests across different edits
- **Scope creep**: Expanding implementation beyond test requirements
- **Feature sprawl**: Implementing multiple features in one operation

### Context Analysis
- **File type classification** for the target file
- **Current TDD phase**: Are we in Red, Green, or Refactor?
- **Change coherence**: Do all edits work toward single test goal?
- **Implementation minimalism**: Is each edit necessary for test passage?

### Validation Decision Framework
1. **Count total new tests** across all edits
2. **Assess implementation scope** relative to test requirements  
3. **Check for over-engineering** or premature optimization
4. **Verify TDD discipline** throughout the edit sequence

**CRITICAL RULE**: If cumulative new tests > 1, this is a TDD violation requiring immediate blocking.

**VALIDATION DECISION**: Analyze all edits collectively and determine TDD compliance.
"""


class TDDContextFormatter:
    """Format test results and context for TDD validation"""

    @staticmethod
    def format_test_output(test_results: Dict[str, Any]) -> str:
        if not test_results:
            return "\n## TEST OUTPUT\nNo recent test results available."

        return f"""
## TEST OUTPUT CONTEXT

### Test Execution Results
**Timestamp**: {test_results.get('timestamp', 'Unknown')}
**Status**: {test_results.get('status', 'Unknown')}

### Test Failures (guide implementation)
{json.dumps(test_results.get('failures', []), indent=2)}

### Test Passes (avoid over-implementation)  
{json.dumps(test_results.get('passes', []), indent=2)}

### Error Messages (inform minimal fixes)
{json.dumps(test_results.get('errors', []), indent=2)}

### TDD GUIDANCE FROM TEST OUTPUT
- **Red Phase**: Use failures to guide what needs to be implemented
- **Green Phase**: Implement minimal code to make failing tests pass
- **Avoid**: Implementing functionality not required by current failures
- **Focus**: Address specific error messages and assertions
"""

    @staticmethod
    def format_tdd_context(context: Dict[str, Any]) -> str:
        """Format complete TDD context including test results, todos, and modifications"""

        test_section = TDDContextFormatter.format_test_output(
            context.get("test_results") or {}
        )

        todos_section = ""
        if context.get("todos"):
            todos_section = f"""
## TODO CONTEXT
Current todos that may influence TDD decisions:
{json.dumps(context.get('todos'), indent=2)}
"""

        modifications_section = ""
        if context.get("recent_modifications"):
            modifications_section = f"""
## RECENT MODIFICATIONS
File changes that provide context for current operation:
{json.dumps(context.get('recent_modifications'), indent=2)}
"""

        return f"""
# TDD VALIDATION CONTEXT

{test_section}

{todos_section}

{modifications_section}

## TDD STATE ASSESSMENT
- **Has valid test data**: {context.get('has_valid_test_data', False)}
- **Current phase likely**: {"Red" if (context.get('test_results') or {}).get('failures') else "Green/Refactor"}
- **Context richness**: {"High" if context.get('test_results') else "Low"}
"""
