"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Application = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const aws_kinesisanalytics_1 = require("aws-cdk-lib/aws-kinesisanalytics");
const logs = require("aws-cdk-lib/aws-logs");
const core = require("aws-cdk-lib");
const environment_properties_1 = require("./private/environment-properties");
const flink_application_configuration_1 = require("./private/flink-application-configuration");
const validation_1 = require("./private/validation");
/**
 * Implements the functionality shared between CDK created and imported
 * IApplications.
 */
class ApplicationBase extends core.Resource {
    /** Implement the convenience `IApplication.addToPrincipalPolicy` method. */
    addToRolePolicy(policyStatement) {
        if (this.role) {
            this.role.addToPrincipalPolicy(policyStatement);
            return true;
        }
        return false;
    }
    get connections() {
        if (!this._connections) {
            throw new Error('This Application isn\'t associated with a VPC. Provide a "vpc" prop when creating the Application or "securityGroups" when importing it.');
        }
        return this._connections;
    }
    /**
     * Return a CloudWatch metric associated with this Flink application.
     *
     * @param metricName The name of the metric
     * @param props Customization properties
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/KinesisAnalytics',
            metricName,
            dimensionsMap: { Application: this.applicationName },
            ...props,
        }).attachTo(this);
    }
    /**
     * The number of Kinesis Processing Units that are used to run your stream
     * processing application. The average number of KPUs used each hour
     * determines the billing for your application.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricKpus(props) {
        return this.metric('KPUs', { statistic: 'Average', ...props });
    }
    /**
     * The time elapsed during an outage for failing/recovering jobs.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricDowntime(props) {
        return this.metric('downtime', { statistic: 'Average', ...props });
    }
    /**
     * The time that the job has been running without interruption.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricUptime(props) {
        return this.metric('uptime', { statistic: 'Average', ...props });
    }
    /**
     * The total number of times this job has fully restarted since it was
     * submitted. This metric does not measure fine-grained restarts.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - sum over 5 minutes
     */
    metricFullRestarts(props) {
        return this.metric('fullRestarts', { statistic: 'Sum', ...props });
    }
    /**
     * The number of times checkpointing has failed.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - sum over 5 minutes
     */
    metricNumberOfFailedCheckpoints(props) {
        return this.metric('numberOfFailedCheckpoints', { statistic: 'Sum', ...props });
    }
    /**
     * The time it took to complete the last checkpoint.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default - maximum over 5 minutes
     */
    metricLastCheckpointDuration(props) {
        return this.metric('lastCheckpointDuration', { statistic: 'Maximum', ...props });
    }
    /**
     * The total size of the last checkpoint.
     *
     * Units: Bytes
     *
     * Reporting Level: Application
     *
     * @default - maximum over 5 minutes
     */
    metricLastCheckpointSize(props) {
        return this.metric('lastCheckpointSize', { statistic: 'Maximum', ...props });
    }
    /**
     * The overall percentage of CPU utilization across task managers. For
     * example, if there are five task managers, Kinesis Data Analytics publishes
     * five samples of this metric per reporting interval.
     *
     * Units: Percentage
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricCpuUtilization(props) {
        return this.metric('cpuUtilization', { statistic: 'Average', ...props });
    }
    /**
     * Overall heap memory utilization across task managers. For example, if there
     * are five task managers, Kinesis Data Analytics publishes five samples of
     * this metric per reporting interval.
     *
     * Units: Percentage
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricHeapMemoryUtilization(props) {
        return this.metric('heapMemoryUtilization', { statistic: 'Average', ...props });
    }
    /**
     * The total time spent performing old garbage collection operations.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default - sum over 5 minutes
     */
    metricOldGenerationGCTime(props) {
        return this.metric('oldGenerationGCTime', { statistic: 'Sum', ...props });
    }
    /**
     * The total number of old garbage collection operations that have occurred
     * across all task managers.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - sum over 5 minutes
     */
    metricOldGenerationGCCount(props) {
        return this.metric('oldGenerationGCCount', { statistic: 'Sum', ...props });
    }
    /**
     * The total number of live threads used by the application.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricThreadsCount(props) {
        return this.metric('threadsCount', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator, or task has
     * received.
     *
     * Units: Count
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricNumRecordsIn(props) {
        return this.metric('numRecordsIn', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator or task has received
     * per second.
     *
     * Units: Count/Second
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricNumRecordsInPerSecond(props) {
        return this.metric('numRecordsInPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator or task has emitted.
     *
     * Units: Count
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricNumRecordsOut(props) {
        return this.metric('numRecordsOut', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator or task has emitted
     * per second.
     *
     * Units: Count/Second
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricNumRecordsOutPerSecond(props) {
        return this.metric('numRecordsOutPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The number of records this operator or task has dropped due to arriving
     * late.
     *
     * Units: Count
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - sum over 5 minutes
     */
    metricNumLateRecordsDropped(props) {
        return this.metric('numLateRecordsDropped', { statistic: 'Sum', ...props });
    }
    /**
     * The last watermark this application/operator/task/thread has received.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - maximum over 5 minutes
     */
    metricCurrentInputWatermark(props) {
        return this.metric('currentInputWatermark', { statistic: 'Maximum', ...props });
    }
    /**
     * The last watermark this application/operator/task/thread has received.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - maximum over 5 minutes
     */
    metricCurrentOutputWatermark(props) {
        return this.metric('currentOutputWatermark', { statistic: 'Maximum', ...props });
    }
    /**
     * The amount of managed memory currently used.
     *
     * Units: Bytes
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricManagedMemoryUsed(props) {
        return this.metric('managedMemoryUsed', { statistic: 'Average', ...props });
    }
    /**
     * The total amount of managed memory.
     *
     * Units: Bytes
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricManagedMemoryTotal(props) {
        return this.metric('managedMemoryTotal', { statistic: 'Average', ...props });
    }
    /**
     * Derived from managedMemoryUsed/managedMemoryTotal.
     *
     * Units: Percentage
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricManagedMemoryUtilization(props) {
        return this.metric('managedMemoryUtilization', { statistic: 'Average', ...props });
    }
    /**
     * The time (in milliseconds) this task or operator is idle (has no data to
     * process) per second. Idle time excludes back pressured time, so if the task
     * is back pressured it is not idle.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricIdleTimeMsPerSecond(props) {
        return this.metric('idleTimeMsPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The time (in milliseconds) this task or operator is back pressured per
     * second.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricBackPressuredTimeMsPerSecond(props) {
        return this.metric('backPressuredTimeMsPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The time (in milliseconds) this task or operator is busy (neither idle nor
     * back pressured) per second. Can be NaN, if the value could not be
     * calculated.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricBusyTimePerMsPerSecond(props) {
        return this.metric('busyTimePerMsPerSecond', { statistic: 'Average', ...props });
    }
}
/**
 * An imported Flink application.
 */
class Import extends ApplicationBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        // Imported applications have no associated role or grantPrincipal
        this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
        this.role = undefined;
        this.applicationArn = attrs.applicationArn;
        const applicationName = core.Stack.of(scope).splitArn(attrs.applicationArn, core.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        if (!applicationName) {
            throw new Error(`applicationArn for fromApplicationArn (${attrs.applicationArn}) must include resource name`);
        }
        this.applicationName = applicationName;
        const securityGroups = attrs.securityGroups ?? [];
        if (securityGroups.length > 0) {
            this._connections = new ec2.Connections({ securityGroups });
        }
    }
}
/**
 * The L2 construct for Flink Kinesis Data Applications.
 *
 * @resource AWS::KinesisAnalyticsV2::Application
 *
 */
class Application extends ApplicationBase {
    constructor(scope, id, props) {
        super(scope, id, { physicalName: props.applicationName });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_kinesisanalytics_flink_alpha_ApplicationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Application);
            }
            throw error;
        }
        validation_1.validateFlinkApplicationProps(props);
        this.role = props.role ?? new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('kinesisanalytics.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        // Permit metric publishing to CloudWatch
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['cloudwatch:PutMetricData'],
            resources: ['*'],
        }));
        const code = props.code.bind(this);
        code.bucket.grantRead(this);
        let vpcConfigurations;
        if (props.vpc) {
            const securityGroups = props.securityGroups ?? [
                new ec2.SecurityGroup(this, 'SecurityGroup', {
                    vpc: props.vpc,
                }),
            ];
            this._connections = new ec2.Connections({ securityGroups });
            const subnetSelection = props.vpcSubnets ?? {
                subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS,
            };
            vpcConfigurations = [{
                    securityGroupIds: securityGroups.map(sg => sg.securityGroupId),
                    subnetIds: props.vpc.selectSubnets(subnetSelection).subnetIds,
                }];
        }
        const resource = new aws_kinesisanalytics_1.CfnApplicationV2(this, 'Resource', {
            applicationName: props.applicationName,
            runtimeEnvironment: props.runtime.value,
            serviceExecutionRole: this.role.roleArn,
            applicationConfiguration: {
                ...code.applicationCodeConfigurationProperty,
                environmentProperties: environment_properties_1.environmentProperties(props.propertyGroups),
                flinkApplicationConfiguration: flink_application_configuration_1.flinkApplicationConfiguration({
                    checkpointingEnabled: props.checkpointingEnabled,
                    checkpointInterval: props.checkpointInterval,
                    minPauseBetweenCheckpoints: props.minPauseBetweenCheckpoints,
                    logLevel: props.logLevel,
                    metricsLevel: props.metricsLevel,
                    autoScalingEnabled: props.autoScalingEnabled,
                    parallelism: props.parallelism,
                    parallelismPerKpu: props.parallelismPerKpu,
                }),
                applicationSnapshotConfiguration: {
                    snapshotsEnabled: props.snapshotsEnabled ?? true,
                },
                vpcConfigurations,
            },
        });
        resource.node.addDependency(this.role);
        const logGroup = props.logGroup ?? new logs.LogGroup(this, 'LogGroup');
        const logStream = new logs.LogStream(this, 'LogStream', { logGroup });
        /* Permit logging */
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:DescribeLogGroups'],
            resources: [
                core.Stack.of(this).formatArn({
                    service: 'logs',
                    resource: 'log-group',
                    arnFormat: core.ArnFormat.COLON_RESOURCE_NAME,
                    resourceName: '*',
                }),
            ],
        }));
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:DescribeLogStreams'],
            resources: [logGroup.logGroupArn],
        }));
        const logStreamArn = `arn:${core.Aws.PARTITION}:logs:${core.Aws.REGION}:${core.Aws.ACCOUNT_ID}:log-group:${logGroup.logGroupName}:log-stream:${logStream.logStreamName}`;
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:PutLogEvents'],
            resources: [logStreamArn],
        }));
        new aws_kinesisanalytics_1.CfnApplicationCloudWatchLoggingOptionV2(this, 'LoggingOption', {
            applicationName: resource.ref,
            cloudWatchLoggingOption: {
                logStreamArn,
            },
        });
        // Permissions required for VPC usage per:
        // https://docs.aws.amazon.com/kinesisanalytics/latest/java/vpc-permissions.html
        if (props.vpc) {
            this.role.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: [
                    'ec2:DescribeVpcs',
                    'ec2:DescribeSubnets',
                    'ec2:DescribeSecurityGroups',
                    'ec2:DescribeDhcpOptions',
                    'ec2:CreateNetworkInterface',
                    'ec2:CreateNetworkInterfacePermission',
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:DeleteNetworkInterface',
                ],
                resources: ['*'],
            }));
        }
        this.applicationName = this.getResourceNameAttribute(resource.ref);
        this.applicationArn = this.getResourceArnAttribute(core.Stack.of(this).formatArn(applicationArnComponents(resource.ref)), applicationArnComponents(this.physicalName));
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core.RemovalPolicy.DESTROY,
        });
    }
    /**
     * Import an existing Flink application defined outside of CDK code by
     * applicationName.
     */
    static fromApplicationName(scope, id, applicationName) {
        const applicationArn = core.Stack.of(scope).formatArn(applicationArnComponents(applicationName));
        return new Import(scope, id, { applicationArn });
    }
    /**
     * Import an existing application defined outside of CDK code by
     * applicationArn.
     */
    static fromApplicationArn(scope, id, applicationArn) {
        return new Import(scope, id, { applicationArn });
    }
    /**
     * Import an existing application defined outside of CDK code.
     */
    static fromApplicationAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_kinesisanalytics_flink_alpha_ApplicationAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromApplicationAttributes);
            }
            throw error;
        }
        return new Import(scope, id, {
            applicationArn: attrs.applicationArn,
            securityGroups: attrs.securityGroups,
        });
    }
}
exports.Application = Application;
_a = JSII_RTTI_SYMBOL_1;
Application[_a] = { fqn: "@aws-cdk/aws-kinesisanalytics-flink-alpha.Application", version: "2.69.0-alpha.0" };
function applicationArnComponents(resourceName) {
    return {
        service: 'kinesisanalytics',
        resource: 'application',
        resourceName,
    };
}
//# sourceMappingURL=data:application/json;base64,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