from __future__ import annotations

import datetime

import sqlglot
from sqlglot import exp


def create_stage(
    expression: exp.Expression, current_database: str | None = None, current_schema: str | None = None
) -> exp.Expression:
    """Transform CREATE STAGE to an INSERT statement for the fake stages table."""
    if not (
        isinstance(expression, exp.Create)
        and (kind := expression.args.get("kind"))
        and isinstance(kind, str)
        and kind.upper() == "STAGE"
        and (table := expression.find(exp.Table))
    ):
        return expression

    catalog = table.catalog or current_database
    schema = table.db or current_schema
    ident = table.this
    if isinstance(ident, exp.Placeholder):
        stage_name = "?"
    elif isinstance(ident, exp.Identifier):
        stage_name = ident.this if ident.quoted else ident.this.upper()
    else:
        raise ValueError(f"Invalid identifier type {ident.__class__.__name__} for stage name")
    now = datetime.datetime.now(datetime.timezone.utc).isoformat()

    is_temp = False
    url = ""
    properties = expression.args.get("properties") or []
    for prop in properties:
        if isinstance(prop, exp.TemporaryProperty):
            is_temp = True
        elif (
            isinstance(prop, exp.Property)
            and isinstance(prop.this, exp.Var)
            and isinstance(prop.this.this, str)
            and prop.this.this.upper() == "URL"
        ):
            value = prop.args.get("value")
            if isinstance(value, exp.Literal):
                url = value.this

    # Determine cloud provider based on url
    cloud = "AWS" if url.startswith("s3://") else None

    stage_type = ("EXTERNAL" if url else "INTERNAL") + (" TEMPORARY" if is_temp else "")
    stage_name_value = stage_name if stage_name == "?" else repr(stage_name)

    insert_sql = f"""
        INSERT INTO _fs_global._fs_information_schema._fs_stages
        (created_on, name, database_name, schema_name, url, has_credentials, has_encryption_key, owner,
        comment, region, type, cloud, notification_channel, storage_integration, endpoint, owner_role_type,
        directory_enabled)
        VALUES (
            '{now}', {stage_name_value}, '{catalog}', '{schema}', '{url}', 'N', 'N', 'SYSADMIN',
            '', NULL, '{stage_type}', {f"'{cloud}'" if cloud else "NULL"}, NULL, NULL, NULL, 'ROLE',
            'N'
        )
        """
    transformed = sqlglot.parse_one(insert_sql, read="duckdb")
    transformed.args["stage_name"] = stage_name
    return transformed
