#! /usr/bin/env python

##############################################################################
##  pyvolve: Python platform for simulating evolutionary sequences.
##
##  Written by Stephanie J. Spielman (stephanie.spielman@gmail.com) 
##############################################################################

''' Suite of unit tests for matrix_builder module.'''

import unittest
import numpy as np
from Bio import Seq
from Bio.Alphabet import generic_dna
from pyvolve import matrix_builder
from pyvolve import Genetics
MOLECULES = Genetics()

ZERO=1e-8
DECIMAL=8


class matrixBuilder_baseClass_tests(unittest.TestCase):
    ''' 
        Set of unittests for simple base class functions in matrixBuilder.
        Functions tested here include _is_TI, _is_syn, _get_nucleotide_diff. 
        All other functions require full model specification, so they are tested elsewhere.
        Note: state_freqs specification is required. To test this, all frequencies must be unique.
    '''
    
    def setUp(self):
        codonFreqs = [0.01617666, 0.00291771, 0.02664918, 0.02999061, 0.00717921, 0.00700012, 0.01435559, 0.0231568, 0.02403056, 0.00737008, 0.03185765, 0.0193576, 0.03277142, 0.02141258, 0.0127537, 0.00298803, 0.0256333, 0.02312437, 0.01861465, 0.01586447, 0.00373147, 0.02662654, 0.00082524, 0.00048916, 0.01191673, 0.00512658, 0.00050502, 0.01688169, 0.01843001, 0.00215437, 0.02659356, 0.02377742, 0.01169375, 0.00097256, 0.02937344, 0.00268204, 0.01414414, 0.02781933, 0.00070877, 0.02370841, 0.02984617, 0.01828081, 0.01002825, 0.00870788, 0.00728006, 0.02179328, 0.00379049, 0.01978996, 0.00443774, 0.01201798, 0.02030269, 0.01238501, 0.01279963, 0.02094385, 0.02810987, 0.00918507, 0.02880549, 0.0029311, 0.0237658, 0.03194712, 0.06148723]
        params = {'state_freqs': codonFreqs}
        self.baseObject = matrix_builder.MatrixBuilder("nucleotide", params)
        
    def test_matrixBuilder_baseClass_is_TI(self):    
        ''' Test that transitions can be properly identified. '''
                
        self.assertTrue( self.baseObject._is_TI('A', 'G'), msg = "matrixBuilder._is_TI() does not think A -> G is a transition.")
        self.assertTrue( self.baseObject._is_TI('G', 'A'), msg = "matrixBuilder._is_TI() does not think G -> A is a transition.")
        self.assertTrue( self.baseObject._is_TI('C', 'T'), msg = "matrixBuilder._is_TI() does not think C -> T is a transition.")
        self.assertTrue( self.baseObject._is_TI('T', 'C'), msg = "matrixBuilder._is_TI() does not think C -> T is a transition.")
        self.assertFalse( self.baseObject._is_TI('A', 'C'), msg = "matrixBuilder._is_TI() mistakenly thinks A -> C is a transition.")
        self.assertFalse( self.baseObject._is_TI('C', 'A'), msg = "matrixBuilder._is_TI() mistakenly thinks C -> A is a transition.")
        self.assertFalse( self.baseObject._is_TI('A', 'T'), msg = "matrixBuilder._is_TI() mistakenly thinks A -> T is a transition.")
        self.assertFalse( self.baseObject._is_TI('T', 'A'), msg = "matrixBuilder._is_TI() mistakenly thinks T -> A is a transition.")
        self.assertFalse( self.baseObject._is_TI('G', 'C'), msg = "matrixBuilder._is_TI() mistakenly thinks G -> C is a transition.")
        self.assertFalse( self.baseObject._is_TI('C', 'G'), msg = "matrixBuilder._is_TI() mistakenly thinks C -> G is a transition.")
        self.assertFalse( self.baseObject._is_TI('G', 'T'), msg = "matrixBuilder._is_TI() mistakenly thinks G -> T is a transition.")
        self.assertFalse( self.baseObject._is_TI('T', 'G'), msg = "matrixBuilder._is_TI() mistakenly thinks T -> G is a transition.")



    def test_Matrix_baseClass_is_syn(self):    
        ''' Test that synonymous vs nonsynymous changes can be properly identified. 
            Assumes that biopython is not broken. This is (theoretically...) a very safe assumption.
        '''
        for source in range(61):
            for target in range(61):
                sourceCodon = MOLECULES.codons[source]
                targetCodon = MOLECULES.codons[target]
                source_aa = str( Seq.Seq(sourceCodon, generic_dna).translate() )
                target_aa = str( Seq.Seq(targetCodon, generic_dna).translate() )
                if source_aa == target_aa:
                    self.assertTrue( self.baseObject._is_syn(source, target), msg = ("matrixBuilder._is_syn() does not think", source, " -> ", target, " is synonymous.") )
                else:
                    self.assertFalse( self.baseObject._is_syn(source, target), msg = ("matrixBuilder._is_syn() mistakenly thinks", source, " -> ", target, " is synonymous.") )


    def test_matrixBuilder_baseClass_get_nucleotide_diff(self):
        ''' Test that nucleotide differences between codons can be identified properly. '''
        
        self.assertEqual( self.baseObject._get_nucleotide_diff( MOLECULES.codons.index('AAA'), MOLECULES.codons.index('AAA')), '', msg = "matrixBuilder._get_nucleotide_diff can't do same codon." )
        self.assertEqual( self.baseObject._get_nucleotide_diff( MOLECULES.codons.index('AAA'), MOLECULES.codons.index('CAA')), 'AC',  msg = "matrixBuilder._get_nucleotide_diff can't do one difference, muliple=False" )
        self.assertEqual( self.baseObject._get_nucleotide_diff( MOLECULES.codons.index('AAA'), MOLECULES.codons.index('ATT')), 'ATAT', msg = "matrixBuilder._get_nucleotide_diff can't do two differences, multiple=True." )
        self.assertEqual( self.baseObject._get_nucleotide_diff( MOLECULES.codons.index('AAA'), MOLECULES.codons.index('TTT')), 'ATATAT', msg = "matrixBuilder._get_nucleotide_diff can't do fully distinct, multiple=True." )







class matrixBuilder_MechCodon_Matrix_scaling_tests(unittest.TestCase):
    ''' 
        Test that scaling is being done properly for mechanistic codon models.
    '''
    
    def setUp(self):
        mu = {'AC':1.,  'CA':1.,  'AG':1.,  'GA':1.,  'AT':1.,  'TA':1.,  'CG':1.,  'GC':1.,  'CT':1.,  'TC':1.,  'GT':1.,  'TG':1.}
        freqs = np.repeat(1./61, 61)
        self.params = { "mu": mu, "state_freqs": freqs}
    

    def test_matrixBuilder_GY(self):    
        ''' Test proper matrix construction, *average* scaling for homogenous GY model.'''
        self.params["beta"] = 0.5
        self.params["alpha"] = 1.
        truematrix = np.loadtxt("tests/matrixFiles/scaled_matrix_homo_gy_w0.5.txt")
        testmatrix = matrix_builder.MechCodon_Matrix("gy", self.params)()
        
        np.testing.assert_array_almost_equal(truematrix, testmatrix, decimal = DECIMAL, err_msg = "Matrix improperly constructed for homogeneous GY codon model.")


    def test_matrixBuilder_MG(self):    
        ''' Test proper matrix construction, *average* scaling for homogenous MG model.'''
        self.params["beta"] = 0.5
        self.params["alpha"] = 1.
        self.params["nuc_freqs"] = [ 0.24043716,  0.26229508,  0.25136612,  0.24590164]
        truematrix = np.loadtxt("tests/matrixFiles/scaled_matrix_homo_mg_w0.5.txt")
        testmatrix = matrix_builder.MechCodon_Matrix("mg", self.params)()  
        np.testing.assert_array_almost_equal(truematrix, testmatrix, decimal = DECIMAL, err_msg = "Matrix improperly constructed for homogeneous MG codon model.")

    def test_matrixBuilder_GY_neutral(self):    
        ''' Test proper matrix construction, *average* scaling for homogenous GY model.'''
        neutral_scaled_matrix = np.array([[-1.822279725094110514e+00, 8.591433487230409893e-02, 1.723904108205525088e-01, 1.068965060825150665e-01, 1.942570456513297295e-01, 0.0, 0.0, 0.0, 4.356111989754437586e-01, 0.0, 0.0, 0.0, 1.986457676236213499e-01, 0.0, 0.0, 0.0, 1.326092327943829419e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 4.959552282739611018e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[1.607414823658994407e-01, -1.094906537046313355e+00, 1.034342464923315025e-01, 1.781608434708584487e-01, 0.0, 1.619275999492563178e-01, 0.0, 0.0, 0.0, 1.221631518664325483e-01, 0.0, 0.0, 0.0, 3.703405230113154117e-02, 0.0, 0.0, 0.0, 1.577449085485805313e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 8.590393095610364405e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 8.779632109571933185e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[2.679024706098324105e-01, 8.591433487230409893e-02, -1.247011389805276371e+00, 1.068965060825150665e-01, 0.0, 0.0, 1.219915864856168652e-01, 0.0, 0.0, 0.0, 1.584168781225658851e-01, 0.0, 0.0, 0.0, 3.329394740736328628e-02, 0.0, 0.0, 0.0, 2.329892164349513239e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.396064497901276014e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[1.607414823658994407e-01, 1.431905581205068501e-01, 1.034342464923315025e-01, -1.333634197125096454e+00, 0.0, 0.0, 0.0, 9.134633331474643902e-02, 0.0, 0.0, 0.0, 3.979264911947038574e-01, 0.0, 0.0, 0.0, 3.882532436424540601e-02, 0.0, 0.0, 0.0, 2.096900300156012420e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 8.107513334907724689e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.074045979079843588e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[1.607414823658994407e-01, 0.0, 0.0, 0.0, -1.634863580897852886e+00, 1.079517332995042073e-01, 2.033193108093614421e-01, 6.089755554316429037e-02, 1.742444795901775256e-01, 0.0, 0.0, 0.0, 4.966144190590533469e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.648464467058495878e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.843161956498621579e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.302937599102454991e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 8.591433487230409893e-02, 0.0, 0.0, 1.295046971008864956e-01, -1.326123839640216628e+00, 8.132772432374457683e-02, 1.522438888579107086e-01, 0.0, 4.886526074657302626e-02, 0.0, 0.0, 0.0, 9.258513075282887028e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.415201511375597243e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 4.730361396706732058e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.211265121777361370e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 1.034342464923315025e-01, 0.0, 3.237617427522162528e-01, 1.079517332995042073e-01, -1.324285349945221846e+00, 6.089755554316429037e-02, 0.0, 0.0, 6.336675124902635403e-02, 0.0, 0.0, 0.0, 8.323486851840822265e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 5.557460846452685205e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.327286510424601884e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.933351925835840590e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 1.068965060825150665e-01, 1.295046971008864956e-01, 2.698793332487605112e-01, 8.132772432374457683e-02, -1.139155672545320463e+00, 0.0, 0.0, 0.0, 1.591705964778815374e-01, 0.0, 0.0, 0.0, 9.706331091061352889e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.850005332145608361e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 6.888502319045020084e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 4.142794799590789750e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[4.018537059147486157e-01, 0.0, 0.0, 0.0, 1.942570456513297295e-01, 0.0, 0.0, 0.0, -1.345461544041241186e+00, 4.886526074657302626e-02, 1.056112520817105715e-01, 1.591705964778815374e-01, 1.986457676236213499e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 8.670995169544923331e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.503479638499273796e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 2.147858371807602751e-01, 0.0, 0.0, 0.0, 1.619275999492563178e-01, 0.0, 0.0, 1.742444795901775256e-01, -1.579877812111093682e+00, 6.336675124902635403e-02, 2.652843274631359050e-01, 0.0, 3.703405230113154117e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.416131640776839007e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.193019588056200675e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.023196414943017885e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 2.585856162308287631e-01, 0.0, 0.0, 0.0, 1.219915864856168652e-01, 0.0, 2.904074659836291539e-01, 4.886526074657302626e-02, -1.330356000292072904e+00, 1.591705964778815374e-01, 0.0, 0.0, 3.329394740736328628e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.034280716449936383e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.997969745453664681e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.481648076982010689e-02, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 2.672412652062877148e-01, 0.0, 0.0, 0.0, 9.134633331474643902e-02, 1.742444795901775256e-01, 8.144210124428837017e-02, 6.336675124902635403e-02, -1.060565262452305202e+00, 0.0, 0.0, 0.0, 3.882532436424540601e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.243933669161152722e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.127096883889864704e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.069959521784316775e-01, 0.0, 0.0, 0.0, 0.0],[1.607414823658994407e-01, 0.0, 0.0, 0.0, 4.856426141283243791e-01, 0.0, 0.0, 0.0, 1.742444795901775256e-01, 0.0, 0.0, 0.0, -1.750716574680476700e+00, 2.468936820075436078e-02, 8.323486851840822265e-02, 2.588354957616360516e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.629766529215850857e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 5.500897490899355757e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 8.321381028922847367e-02, 0.0, 0.0, 0.0],[0.0, 8.591433487230409893e-02, 0.0, 0.0, 0.0, 4.048189998731407391e-01, 0.0, 0.0, 0.0, 4.886526074657302626e-02, 0.0, 0.0, 1.324305117490808814e-01, -1.162043711793115497e+00, 3.329394740736328628e-02, 6.470887394040901464e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.255806897417720058e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.648300282581185161e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.601065204354004462e-03, 0.0, 0.0],[0.0, 0.0, 1.034342464923315025e-01, 0.0, 0.0, 0.0, 3.049789662140421909e-01, 0.0, 0.0, 0.0, 6.336675124902635403e-02, 0.0, 4.966144190590533469e-01, 3.703405230113154117e-02, -1.567430145784482143e+00, 3.882532436424540601e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.109029518978794510e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.079824301539664189e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.042910040528060633e-01, 0.0],[0.0, 0.0, 0.0, 1.068965060825150665e-01, 0.0, 0.0, 0.0, 2.283658332868661045e-01, 0.0, 0.0, 0.0, 1.591705964778815374e-01, 1.324305117490808814e-01, 6.172342050188590890e-02, 3.329394740736328628e-02, -1.261484620099122100e+00, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 6.525195044767972963e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.352279977322475402e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.391238564136020872e-01],[1.607414823658994407e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, -1.619497489414689184e+00, 1.577449085485805313e-01, 3.883153607249188455e-01, 2.096900300156012420e-01, 1.648464467058495878e-02, 0.0, 0.0, 0.0, 3.251623188579346735e-01, 0.0, 0.0, 0.0, 1.629766529215850857e-01, 0.0, 0.0, 0.0, 1.983820913095844518e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 8.591433487230409893e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.326092327943829419e-01, -1.775982293656254818e+00, 2.329892164349513239e-01, 3.494833833593353756e-01, 0.0, 1.415201511375597243e-01, 0.0, 0.0, 0.0, 3.540329101942097378e-01, 0.0, 0.0, 0.0, 2.255806897417720058e-01, 0.0, 0.0, 0.0, 3.436157238244145345e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.194908027392983296e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 1.034342464923315025e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.210153879906382735e-01, 1.577449085485805313e-01, -1.342059981994334850e+00, 2.096900300156012420e-01, 0.0, 0.0, 5.557460846452685205e-02, 0.0, 0.0, 0.0, 3.878552686687261608e-01, 0.0, 0.0, 0.0, 1.109029518978794510e-01, 0.0, 0.0, 0.0, 9.584257991605105165e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 1.068965060825150665e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.326092327943829419e-01, 2.629081809143009041e-01, 2.329892164349513239e-01, -1.597580585288270294e+00, 0.0, 0.0, 0.0, 1.850005332145608361e-01, 0.0, 0.0, 0.0, 3.109834172902881666e-01, 0.0, 0.0, 0.0, 6.525195044767972963e-02, 0.0, 0.0, 0.0, 3.243005333963089321e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.685114947699608900e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 1.942570456513297295e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.326092327943829419e-01, 0.0, 0.0, 0.0, -1.614138520004032795e+00, 9.434676742503980695e-02, 9.262434744087809368e-02, 1.233336888097072287e-01, 1.300649275431738638e-01, 0.0, 0.0, 0.0, 4.074416323039626864e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.137264782599448631e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.257343997756137477e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 1.619275999492563178e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.577449085485805313e-01, 0.0, 0.0, 1.098976311372330469e-02, -1.873641857356903140e+00, 3.704973897635123470e-02, 3.083342220242680787e-01, 0.0, 1.416131640776839007e-01, 0.0, 0.0, 0.0, 5.639517243544300840e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.892144558682692934e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.028162804443403355e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.219915864856168652e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.329892164349513239e-01, 0.0, 2.747440778430826000e-02, 9.434676742503980695e-02, -1.648964596027756579e+00, 1.233336888097072287e-01, 0.0, 0.0, 1.551421074674904366e-01, 0.0, 0.0, 0.0, 2.772573797446986066e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.330914604169840809e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 4.833379814589601753e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 9.134633331474643902e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.096900300156012420e-01, 1.098976311372330469e-02, 2.358669185625995035e-01, 3.704973897635123470e-02, -1.003589906284286126e+00, 0.0, 0.0, 0.0, 1.243933669161152722e-01, 0.0, 0.0, 0.0, 1.631298761191992963e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.755400927618007964e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.035698699897697472e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.161629863934516699e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.315230819859574241e-01, 0.0, 0.0, 0.0, 1.648464467058495878e-02, 0.0, 0.0, 0.0, -1.123194417953233604e+00, 9.440877605178926713e-02, 2.585701791124840887e-01, 8.292891127741017221e-02, 1.629766529215850857e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 6.013918553997096156e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 4.886526074657302626e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.943622713714513561e-01, 0.0, 0.0, 0.0, 1.415201511375597243e-01, 0.0, 0.0, 8.670995169544923331e-02, -1.841308561789326781e+00, 1.034280716449936383e-01, 2.073222781935254166e-01, 0.0, 2.255806897417720058e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.277207835222480270e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 5.057991037357544295e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 4.224450083268423140e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 5.824730410873782960e-01, 0.0, 0.0, 0.0, 5.557460846452685205e-02, 0.0, 2.167748792386231249e-01, 9.440877605178926713e-02, -1.342267660592987966e+00, 8.292891127741017221e-02, 0.0, 0.0, 1.109029518978794510e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.199187898181465622e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.704120192455026550e-02, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.591705964778815374e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 5.242250750390030634e-01, 0.0, 0.0, 0.0, 1.850005332145608361e-01, 8.670995169544923331e-02, 2.360219401294731401e-01, 1.034280716449936383e-01, -1.672381874450714845e+00, 0.0, 0.0, 0.0, 6.525195044767972963e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 4.508387535559459092e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.674898804460791868e-01, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.986457676236213499e-01, 0.0, 0.0, 0.0, 1.326092327943829419e-01, 0.0, 0.0, 0.0, 4.121161167646239520e-02, 0.0, 0.0, 0.0, 1.300649275431738638e-01, 0.0, 0.0, 0.0, -1.239983803045095812e+00, 1.503871264945146891e-01, 1.848382531631324321e-01, 4.350130029845315077e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.200358996359742525e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.386896838153807709e-01, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.703405230113154117e-02, 0.0, 0.0, 0.0, 1.577449085485805313e-01, 0.0, 0.0, 0.0, 3.538003778438992830e-01, 0.0, 0.0, 0.0, 1.416131640776839007e-01, 0.0, 0.0, 1.086511019477233858e-01, -1.051466831044536754e+00, 7.393530126525296731e-02, 1.087532507461328735e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 6.593201130324739812e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 4.002663010885011696e-03, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.329394740736328628e-02, 0.0, 0.0, 0.0, 2.329892164349513239e-01, 0.0, 0.0, 0.0, 1.389365211613171613e-01, 0.0, 0.0, 0.0, 1.551421074674904366e-01, 0.0, 2.716277548693084576e-01, 1.503871264945146891e-01, -1.449555952949662219e+00, 4.350130029845315077e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 8.319297206158657310e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.404850067546768000e-01, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.882532436424540601e-02, 0.0, 0.0, 0.0, 2.096900300156012420e-01, 0.0, 0.0, 0.0, 4.625013330364020625e-01, 0.0, 0.0, 0.0, 1.243933669161152722e-01, 1.086511019477233858e-01, 3.759678162362867226e-01, 7.393530126525296731e-02, -1.875865113908531390e+00, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.340911990928989883e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.478096410340052458e-01],[4.018537059147486157e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.326092327943829419e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, -1.245102539370468886e+00, 3.436157238244145345e-02, 1.597376331934184379e-01, 3.243005333963089321e-02, 1.137264782599448631e-01, 0.0, 0.0, 0.0, 1.503479638499273796e-01, 0.0, 0.0, 0.0, 2.200358996359742525e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 2.147858371807602751e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.577449085485805313e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.983820913095844518e-01, -1.383050252927217283e+00, 9.584257991605105165e-02, 5.405008889938483357e-02, 0.0, 1.892144558682692934e-01, 0.0, 0.0, 0.0, 3.193019588056200675e-01, 0.0, 0.0, 0.0, 6.593201130324739812e-02, 0.0, 0.0, 8.779632109571933185e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 2.585856162308287631e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.329892164349513239e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.306368188493074012e-01, 3.436157238244145345e-02, -1.405084684261096895e+00, 3.243005333963089321e-02, 0.0, 0.0, 1.330914604169840809e-01, 0.0, 0.0, 0.0, 2.997969745453664681e-01, 0.0, 0.0, 0.0, 8.319297206158657310e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 2.672412652062877148e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.096900300156012420e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.983820913095844518e-01, 5.726928730406909834e-02, 9.584257991605105165e-02, -1.210184748417643341e+00, 0.0, 0.0, 0.0, 2.755400927618007964e-02, 0.0, 0.0, 0.0, 1.127096883889864704e-01, 0.0, 0.0, 0.0, 1.340911990928989883e-01, 0.0, 1.074045979079843588e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 4.856426141283243791e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.648464467058495878e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.983820913095844518e-01, 0.0, 0.0, 0.0, -1.807363455439918942e+00, 1.261429705788461864e-01, 2.218191006949734589e-01, 1.836933951745338411e-02, 6.013918553997096156e-02, 0.0, 0.0, 0.0, 5.500897490899355757e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 1.302937599102454991e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 4.048189998731407391e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.415201511375597243e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.436157238244145345e-02, 0.0, 0.0, 7.581765217329657081e-02, -1.204846688596163728e+00, 8.872764027798937803e-02, 4.592334879363346029e-02, 0.0, 1.277207835222480270e-01, 0.0, 0.0, 0.0, 1.648300282581185161e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 1.211265121777361370e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.049789662140421909e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 5.557460846452685205e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 9.584257991605105165e-02, 0.0, 1.895441304332414478e-01, 1.261429705788461864e-01, -1.311689007679858188e+00, 1.836933951745338411e-02, 0.0, 0.0, 1.199187898181465622e-01, 0.0, 0.0, 0.0, 2.079824301539664189e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 1.933351925835840590e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.283658332868661045e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.850005332145608361e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.243005333963089321e-02, 7.581765217329657081e-02, 3.153574264471155075e-01, 8.872764027798937803e-02, -1.347438959823209270e+00, 0.0, 0.0, 0.0, 4.508387535559459092e-02, 0.0, 0.0, 0.0, 3.352279977322475402e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 4.142794799590789750e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 4.356111989754437586e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.300649275431738638e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 4.959552282739611018e-01, 0.0, 0.0, 0.0, 1.137264782599448631e-01, 0.0, 0.0, 0.0, -1.710461488303970823e+00, 8.514718901483202262e-02, 1.998646496969109509e-01, 3.005591690372972496e-02, 2.200358996359742525e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.221631518664325483e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.416131640776839007e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 8.590393095610364405e-02, 0.0, 0.0, 0.0, 1.892144558682692934e-01, 0.0, 0.0, 4.009279035998063873e-02, -1.002324798064107680e+00, 7.994585987876437483e-02, 7.513979225932432282e-02, 0.0, 6.593201130324739812e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.023196414943017885e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.584168781225658851e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.551421074674904366e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.396064497901276014e-01, 0.0, 0.0, 0.0, 1.330914604169840809e-01, 0.0, 1.002319758999516003e-01, 8.514718901483202262e-02, -9.997014304470879242e-01, 3.005591690372972496e-02, 0.0, 0.0, 8.319297206158657310e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.481648076982010689e-02, 0.0, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.979264911947038574e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.243933669161152722e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 8.107513334907724689e-02, 0.0, 0.0, 0.0, 2.755400927618007964e-02, 4.009279035998063873e-02, 2.128679725370800635e-01, 7.994585987876437483e-02, -1.204942774783231974e+00, 0.0, 0.0, 0.0, 1.340911990928989883e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.069959521784316775e-01, 0.0, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 4.966144190590533469e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.629766529215850857e-01, 0.0, 0.0, 0.0, 1.983820913095844518e-01, 0.0, 0.0, 0.0, 2.843161956498621579e-01, 0.0, 0.0, 0.0, 6.013918553997096156e-02, 0.0, 0.0, 0.0, -1.557646115136026133e+00, 4.395467420216493670e-02, 1.386549534359776126e-01, 8.939413272859933479e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 8.321381028922847367e-02, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 9.258513075282887028e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.255806897417720058e-01, 0.0, 0.0, 0.0, 3.436157238244145345e-02, 0.0, 0.0, 0.0, 4.730361396706732058e-01, 0.0, 0.0, 0.0, 1.277207835222480270e-01, 0.0, 0.0, 1.466905997573161591e-01, -1.380523294227523401e+00, 5.546198137439104642e-02, 2.234853318214983509e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.601065204354004462e-03, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 8.323486851840822265e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.109029518978794510e-01, 0.0, 0.0, 0.0, 9.584257991605105165e-02, 0.0, 0.0, 0.0, 3.327286510424601884e-01, 0.0, 0.0, 0.0, 1.199187898181465622e-01, 0.0, 3.667264993932904393e-01, 4.395467420216493670e-02, -1.446994151569805931e+00, 8.939413272859933479e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.042910040528060633e-01, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 9.706331091061352889e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 6.525195044767972963e-02, 0.0, 0.0, 0.0, 3.243005333963089321e-02, 0.0, 0.0, 0.0, 6.888502319045020084e-02, 0.0, 0.0, 0.0, 4.508387535559459092e-02, 1.466905997573161591e-01, 1.098866855054123487e-01, 5.546198137439104642e-02, -7.598773362946905641e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.391238564136020872e-01],[0.0, 8.591433487230409893e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.943622713714513561e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.436157238244145345e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, -1.322172522924015547e+00, 1.790076631799739360e-01, 0.0, 1.211265121777361370e-01, 0.0, 0.0, 5.057991037357544295e-01, 0.0, 0.0, 0.0, 1.601065204354004462e-03, 0.0, 0.0],[0.0, 0.0, 0.0, 1.068965060825150665e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 5.242250750390030634e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.243005333963089321e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.463272018261989049e-01, -1.257920521142937176e+00, 0.0, 0.0, 0.0, 4.142794799590789750e-02, 0.0, 0.0, 2.674898804460791868e-01, 0.0, 0.0, 0.0, 1.391238564136020872e-01],[0.0, 0.0, 0.0, 0.0, 1.942570456513297295e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 4.121161167646239520e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.137264782599448631e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, -9.878246223992109831e-01, 8.075100811849075799e-02, 3.222253209726401169e-01, 2.761863199727193283e-02, 0.0, 0.0, 0.0, 2.080345257230711842e-01, 0.0, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 1.619275999492563178e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.538003778438992830e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.892144558682692934e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 8.779632109571933185e-02, 0.0, 8.686250660683032809e-02, -1.283860274251397149e+00, 1.288901283890560578e-01, 6.904657999317982686e-02, 2.023196414943017885e-01, 0.0, 0.0, 0.0, 4.002663010885011696e-03, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.219915864856168652e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.389365211613171613e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.330914604169840809e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.171562665170757855e-01, 8.075100811849075799e-02, -1.245089465598591794e+00, 2.761863199727193283e-02, 0.0, 1.481648076982010689e-02, 0.0, 0.0, 0.0, 5.107275101320151167e-01, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 9.134633331474643902e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 4.625013330364020625e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.755400927618007964e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.074045979079843588e-01, 8.686250660683032809e-02, 2.018775202962268811e-01, 1.288901283890560578e-01, -1.561242022039863064e+00, 0.0, 0.0, 1.069959521784316775e-01, 0.0, 0.0, 0.0, 3.478096410340052458e-01],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 4.886526074657302626e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.540329101942097378e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.277207835222480270e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.194908027392983296e-01, 0.0, 0.0, 1.211265121777361370e-01, 0.0, 0.0, -1.065980402318292164e+00, 1.481648076982010689e-02, 1.783265869640528190e-01, 0.0, 1.601065204354004462e-03, 0.0, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 6.336675124902635403e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.878552686687261608e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.199187898181465622e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.933351925835840590e-01, 0.0, 2.023196414943017885e-01, -1.278082600045022943e+00, 1.069959521784316775e-01, 0.0, 0.0, 2.042910040528060633e-01, 0.0],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.591705964778815374e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.109834172902881666e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 4.508387535559459092e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 2.685114947699608900e-01, 0.0, 0.0, 0.0, 4.142794799590789750e-02, 3.371994024905030085e-01, 1.481648076982010689e-02, -1.316317071563558372e+00, 0.0, 0.0, 0.0, 1.391238564136020872e-01],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.986457676236213499e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.716277548693084576e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.200358996359742525e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 3.257343997756137477e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, -1.497253750277150575e+00, 1.601065204354004462e-03, 3.404850067546768000e-01, 1.391238564136020872e-01],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.703405230113154117e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 5.639517243544300840e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 6.593201130324739812e-02, 0.0, 0.0, 8.779632109571933185e-02, 0.0, 0.0, 3.028162804443403355e-01, 0.0, 0.0, 2.023196414943017885e-01, 0.0, 0.0, 8.321381028922847367e-02, -1.779227939357875155e+00, 2.042910040528060633e-01, 2.318730940226701731e-01],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.329394740736328628e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.848382531631324321e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 8.319297206158657310e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 4.833379814589601753e-01, 0.0, 0.0, 1.481648076982010689e-02, 0.0, 1.386896838153807709e-01, 1.601065204354004462e-03, -1.078894240294199447e+00, 1.391238564136020872e-01],[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.882532436424540601e-02, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.631298761191992963e-01, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.340911990928989883e-01, 0.0, 1.074045979079843588e-01, 0.0, 0.0, 0.0, 1.035698699897697472e-01, 0.0, 0.0, 1.069959521784316775e-01, 8.321381028922847367e-02, 2.668442007256674464e-03, 2.042910040528060633e-01, -9.441900760018206995e-01]])
        frequencies = np.array([2.250664063425129868e-02, 1.202952114066036665e-02, 1.448261755965576762e-02, 1.496739492534766673e-02, 2.719941021318677751e-02, 2.267271800149749453e-02, 1.708097223580436891e-02, 1.279009666273728946e-02, 2.439729823886594567e-02, 6.841997765198047123e-03, 8.872462027414937738e-03, 2.228668892272884713e-02, 2.781390864147154779e-02, 5.185420055255510717e-03, 4.661739449978819644e-03, 5.436229715644896529e-03, 1.856762985731999147e-02, 2.208706748457185329e-02, 3.262259678569254251e-02, 2.936030003383686277e-02, 2.308140799271927047e-03, 1.981531548220747188e-02, 7.781424699400913379e-03, 2.590333532402653779e-02, 1.821138227822618333e-02, 1.982833893319783242e-02, 2.172262945829318836e-02, 1.741726382796821362e-02, 2.281960390740665756e-02, 3.158527247177905090e-02, 1.552836795396355524e-02, 9.136423142265719458e-03, 2.777698931015492104e-02, 4.811225763608420990e-03, 1.341965043422847954e-02, 4.540779054178468131e-03, 1.592371140990907902e-02, 2.649335876683427127e-02, 1.863515022384919229e-02, 3.858046943975261631e-03, 8.420545941610861965e-03, 1.788316079925655194e-02, 1.679074416887368568e-02, 6.312537162734333830e-03, 3.080890412885516072e-02, 9.231643648263307064e-03, 1.164848238255800084e-02, 1.877513125908538902e-02, 1.229303238234843131e-02, 1.503853673614052172e-02, 1.824342284283565069e-02, 1.695984658558126468e-02, 2.707033453584478666e-02, 5.800642895876114563e-03, 2.832831573617999171e-02, 2.074568451419109717e-03, 1.498131913153471356e-02, 1.165140010170608899e-02, 2.241773477264153347e-04, 2.860434124005755557e-02, 1.947979198564608883e-02])
        neutral_params = {'neutral_scaling':True, 'state_freqs': frequencies, 'alpha':1.0, 'beta':1.5, 'mu': {'AC': 1., 'AG': 2.5, 'AT': 1., 'CG': 1., 'CT': 2.5, 'GT': 1.}}
        testmatrix = matrix_builder.MechCodon_Matrix("gy", neutral_params)()
        np.testing.assert_array_almost_equal(neutral_scaled_matrix, testmatrix, decimal = DECIMAL, err_msg = "Matrix improperly constructed for homogeneous GY codon model with **neutral** scaling.")




class matrixBuilder_MechCodon_Matrix_tests(unittest.TestCase):
    ''' 
        Set of unittests for the matrix_builder.MechCodon_Matrix subclass of matrixBuilder.
        Functions tested here include getTargetFreq, _calc_syn_prob, _calc_nonsyn_prob, _calc_instantaneous_prob.
    '''
    
    def setUp(self):
        
        ################### DO NOT CHANGE ANY OF THESE EVER. #######################
        self.codonFreqs = np.array( [0.01617666, 0.00291771, 0.02664918, 0.02999061, 0.00717921, 0.00700012, 0.01435559, 0.0231568, 0.02403056, 0.00737008, 0.03185765, 0.0193576, 0.03277142, 0.02141258, 0.0127537, 0.00298803, 0.0256333, 0.02312437, 0.01861465, 0.01586447, 0.00373147, 0.02662654, 0.00082524, 0.00048916, 0.01191673, 0.00512658, 0.00050502, 0.01688169, 0.01843001, 0.00215437, 0.02659356, 0.02377742, 0.01169375, 0.00097256, 0.02937344, 0.00268204, 0.01414414, 0.02781933, 0.00070877, 0.02370841, 0.02984617, 0.01828081, 0.01002825, 0.00870788, 0.00728006, 0.02179328, 0.00379049, 0.01978996, 0.00443774, 0.01201798, 0.02030269, 0.01238501, 0.01279963, 0.02094385, 0.02810987, 0.00918507, 0.02880549, 0.0029311, 0.0237658, 0.03194712, 0.06148723] )                  
        muCodonParams = {'AG': 4.0, 'CT': 2.0, 'AC': 1.75, 'AT': 1.5, 'CG': 1.56, 'GT': 4.65}
        self.params_gy = { 'alpha':1.83, 'beta':5.7, 'mu': muCodonParams, 'state_freqs':self.codonFreqs}
        self.params_mg = { 'alpha':1.83, 'beta':5.7, 'mu': muCodonParams, 'state_freqs':self.codonFreqs, 'nuc_freqs': [ 0.24206108,  0.22325576,  0.23687205,  0.29781111]}
        ############################################################################
        
   
    def test_MechCodon_Matrix_MG_calc_prob(self):
        ''' Test (non)synonymous probability calculation for MG. '''
        
        codonMatrix = matrix_builder.MechCodon_Matrix( "mg", self.params_mg )

        # GCA -> GCT, synonymous
        correctProb1 =  0.29781111 * 1.5 * 1.83
        self.assertTrue( abs(codonMatrix._calc_prob(39, "T", "AT", codonMatrix.params["alpha"]) - correctProb1) < ZERO, msg = "matrix_builder.MechCodon_MatrixBuiler._calc_prob can't do syn MG.")
        # TTA -> ATA, nonsyn. Rounding is sadly a bit funky here, so 1e-7. 
        correctProb2 =  0.24206108 * 1.5 * 5.7
        self.assertTrue( abs(codonMatrix._calc_prob(12, "A", "AT", codonMatrix.params["beta"]) - correctProb2) < 1e-7, msg = "matrix_builder.MechCodon_MatrixBuiler._calc_prob can't do nonsyn MG.")
 

    
    

    def test_MechCodon_Matrix_GY_calc_prob(self):
        ''' Test (non)synonymous probability calculation for GY. '''
        
        codonMatrix = matrix_builder.MechCodon_Matrix( "gy", self.params_gy )

        # GCA -> GCT, synonymous
        correctProb1 =  0.02370841 * 1.5 * 1.83
        self.assertTrue( abs(codonMatrix._calc_prob(39, "T", "AT", codonMatrix.params["alpha"]) - correctProb1) < ZERO, msg = "matrix_builder.MechCodon_MatrixBuiler._calc_prob can't do syn GY.")
        # TTA -> ATA, nonsynonymous
        correctProb2 =  0.03277142 * 1.5 * 5.7
        self.assertTrue( abs(codonMatrix._calc_prob(12, "A", "AT", codonMatrix.params["beta"]) - correctProb2) < ZERO, msg = "matrix_builder.MechCodon_MatrixBuiler._calc_prob can't do nonsyn GY.")
 


    def test_MechCodon_Matrix_GY_calc_instantaneous_prob(self):    
        ''' Test that substitution probabilities are properly calculated. Only test GY, since MG will work if calc_prob works.
            Conduct tests for - no change, two changes, three changes, synonymous, nonsynonymous.
        '''
        codonMatrix = matrix_builder.MechCodon_Matrix( "gy", self.params_gy )
        
        # Test no change, two changes, three changes. All should be 0
        self.assertTrue( codonMatrix._calc_instantaneous_prob(7, 7, codonMatrix.params) < ZERO, msg = "matrix_builder.MechCodon_Matrix._calc_instantaneous_prob doesn't return 0 for same codon substitution.")
        self.assertTrue( codonMatrix._calc_instantaneous_prob(7, 8, codonMatrix.params) < ZERO, msg = "matrix_builder.MechCodon_Matrix._calc_instantaneous_prob doesn't return 0 for two nucleotide changes.")
        self.assertTrue( codonMatrix._calc_instantaneous_prob(7, 24, codonMatrix.params) < ZERO, msg = "matrix_builder.MechCodon_Matrix._calc_instantaneous_prob doesn't return 0 for three nucleotide changes.")
        
        # Synonymous. GAG -> GAA
        correctProbSyn = 0.01169375 * 1.83 * 4.0
        self.assertTrue( abs(codonMatrix._calc_instantaneous_prob(34, 32, codonMatrix.params) - correctProbSyn) < ZERO, msg = "matrix_builder.MechCodon_Matrix._calc_instantaneous_prob wrong for GAG -> GAA (synonymous) when GY.")

        # Nonsynonymous. TCG -> ACG
        correctProbNonsyn = 0.01435559 * 5.7 * 1.5
        self.assertTrue( abs(codonMatrix._calc_instantaneous_prob(52, 6, codonMatrix.params) - correctProbNonsyn) < ZERO, msg = "matrix_builder.MechCodon_Matrix._calc_instantaneous_prob wrong for TCG -> ACG (nonsynonymous) when GY.")




class matrixBuilder_ECM_Matrix_tests(unittest.TestCase):
    ''' 
        Set of unittests for the ECM_Matrix subclass of matrixBuilder.
        Functions tested here include _is_syn, getCodonFreq, _calc_syn_prob, _calc_nonsyn_prob, _calc_instantaneous_prob.
    '''
    
    def setUp(self):
        
        ################### DO NOT CHANGE ANY OF THESE EVER. #######################
        # Do not rely on misc for codons in case something happens to it!
        self.codonFreqs = np.array( [0.01617666, 0.00291771, 0.02664918, 0.02999061, 0.00717921, 0.00700012, 0.01435559, 0.0231568, 0.02403056, 0.00737008, 0.03185765, 0.0193576, 0.03277142, 0.02141258, 0.0127537, 0.00298803, 0.0256333, 0.02312437, 0.01861465, 0.01586447, 0.00373147, 0.02662654, 0.00082524, 0.00048916, 0.01191673, 0.00512658, 0.00050502, 0.01688169, 0.01843001, 0.00215437, 0.02659356, 0.02377742, 0.01169375, 0.00097256, 0.02937344, 0.00268204, 0.01414414, 0.02781933, 0.00070877, 0.02370841, 0.02984617, 0.01828081, 0.01002825, 0.00870788, 0.00728006, 0.02179328, 0.00379049, 0.01978996, 0.00443774, 0.01201798, 0.02030269, 0.01238501, 0.01279963, 0.02094385, 0.02810987, 0.00918507, 0.02880549, 0.0029311, 0.0237658, 0.03194712, 0.06148723] )                                
        self.ecmunrest_matrix = np.array([[0.000000000000000000e+00, 4.139570000000000194e-01, 2.931524000000000019e+00, 2.075153999999999943e+00, 1.523250999999999911e+00, 8.947599999999999998e-02, 1.995889999999999886e-01, 8.781630000000000269e-01, 5.815293999999999741e+00, 3.342240000000000211e-01, 1.868193999999999910e+00, 1.293860000000000010e+00, 6.673970000000000180e-01, 2.301999999999999880e-02, 2.753550000000000164e-01, 2.215360000000000107e-01, 3.215816999999999926e+00, 1.434789999999999954e-01, 2.853840000000000265e-01, 8.992620000000000058e-01, 1.042456999999999967e+00, 1.239400000000000050e-02, 4.099999999999999912e-04, 2.471949999999999981e-01, 3.065510000000000179e+00, 2.963580000000000103e-01, 1.103200000000000014e-02, 1.769878999999999980e+00, 4.336200000000000054e-01, 2.142999999999999940e-03, 3.499999999999999693e-05, 6.092199999999999699e-02, 2.258321000000000023e+00, 5.316799999999999998e-02, 1.592480000000000007e-01, 6.048590000000000355e-01, 9.236440000000000206e-01, 5.499999999999999681e-03, 4.820000000000000066e-04, 2.162439999999999918e-01, 7.139609999999999568e-01, 2.863000000000000093e-03, 1.157999999999999958e-03, 4.873000000000000248e-02, 7.082039999999999447e-01, 1.930000000000000075e-03, 1.930000000000000021e-04, 2.014799999999999924e-01, 4.294200000000000100e-02, 4.671359999999999957e-01, 1.496628000000000069e+00, 2.090900000000000050e-02, 4.589999999999999896e-04, 3.409909999999999886e-01, 4.316000000000000392e-03, 5.100500000000000173e-02, 9.951899999999999635e-02, 6.126599999999999824e-01, 5.995999999999999594e-03, 4.963000000000000397e-03, 1.482680000000000109e-01],[4.139570000000000194e-01, 0.000000000000000000e+00, 1.038682000000000105e+00, 7.634676999999999936e+00, 2.599090000000000011e-01, 1.346384999999999943e+00, 8.221330000000000027e-01, 2.960659999999999958e-01, 2.738550000000000151e-01, 4.127466000000000079e+00, 6.004150000000000320e-01, 5.866850000000000120e-01, 2.083700000000000135e-02, 1.754910000000000081e-01, 2.191700000000000037e-01, 2.540500000000000050e-02, 3.774469999999999770e-01, 2.203539000000000136e+00, 9.851450000000000484e-01, 4.609699999999999909e-01, 7.416100000000000469e-02, 5.320920000000000094e-01, 9.541800000000000281e-02, 9.590000000000000000e-04, 1.663600000000000148e-02, 8.631910000000000416e-01, 6.521200000000000607e-02, 3.180000000000000101e-03, 1.454000000000000084e-02, 1.878799999999999915e-01, 3.491600000000000259e-02, 3.299999999999999982e-04, 1.935439999999999938e-01, 2.729646999999999935e+00, 3.727190000000000225e-01, 3.716849999999999876e-01, 4.143499999999999961e-02, 5.659680000000000266e-01, 6.850100000000000633e-02, 1.329000000000000103e-03, 2.803999999999999826e-01, 1.110726000000000102e+00, 2.938839999999999786e-01, 5.904999999999999659e-03, 2.361999999999999950e-03, 2.154919999999999891e-01, 1.358200000000000032e-02, 2.950000000000000148e-04, 4.878140000000000254e-01, 4.658600000000000241e-02, 1.878259999999999930e-01, 1.118208000000000091e+00, 7.258360000000000367e-01, 3.389999999999999784e-03, 3.254219999999999891e-01, 5.390799999999999759e-02, 9.319999999999999942e-04, 2.840000000000000076e-03, 1.436139999999999917e-01, 6.069900000000000295e-02, 9.299999999999999727e-05],[2.931524000000000019e+00, 1.038682000000000105e+00, 0.000000000000000000e+00, 5.246469999999999745e-01, 2.561740000000000128e-01, 6.233659999999999757e-01, 1.144638999999999962e+00, 2.910090000000000177e-01, 1.197613999999999956e+00, 9.288760000000000350e-01, 7.316622999999999877e+00, 3.890040000000000164e-01, 5.626699999999999757e-02, 1.164049999999999946e-01, 4.975930000000000075e-01, 4.143999999999999767e-02, 4.613829999999999876e-01, 5.714690000000000047e-01, 2.285051999999999861e+00, 3.245250000000000079e-01, 1.200200000000000052e-02, 3.036130000000000217e-01, 5.615229999999999944e-01, 2.208160000000000123e-01, 3.451900000000000107e-02, 2.721042999999999878e+00, 3.832199999999999829e+00, 5.608309999999999684e-01, 6.790000000000000245e-04, 2.238400000000000112e-02, 1.330320000000000114e-01, 3.936200000000000115e-02, 3.088509999999999867e-01, 4.266840000000000077e-01, 1.563846000000000069e+00, 1.880969999999999864e-01, 4.381999999999999611e-03, 3.739480000000000026e-01, 7.591320000000000290e-01, 2.686620000000000119e-01, 1.760000000000000063e-03, 1.762239999999999918e-01, 7.439999999999999947e-01, 2.164900000000000158e-02, 5.240000000000000517e-04, 5.614500000000000046e-02, 3.118850000000000233e-01, 7.213799999999999379e-02, 1.408639999999999892e-01, 7.286700000000000121e-02, 4.438000000000000105e-03, 2.516499999999999848e-01, 7.686070000000000402e-01, 1.564679999999999960e-01, 3.479399999999999854e-02, 1.052260000000000001e-01, 1.148899999999999921e-02, 6.200000000000000270e-05, 4.360900000000000193e-02, 3.337800000000000211e-01, 1.312199999999999998e-02],[2.075153999999999943e+00, 7.634676999999999936e+00, 5.246469999999999745e-01, 0.000000000000000000e+00, 1.243753000000000108e+00, 6.755400000000000293e-02, 2.243970000000000131e-01, 2.057449000000000083e+00, 9.368400000000000061e-01, 6.437889999999999446e-01, 4.543470000000000009e-01, 4.971506999999999898e+00, 2.562829999999999830e-01, 1.435099999999999917e-02, 2.089239999999999986e-01, 2.393879999999999897e-01, 1.759732000000000074e+00, 1.885870000000000046e-01, 2.068510000000000071e-01, 2.691225999999999896e+00, 9.610399999999999499e-02, 1.920000000000000049e-04, 2.704999999999999939e-03, 7.873400000000000398e-01, 9.981000000000000288e-03, 6.799999999999999946e-05, 9.810000000000001011e-04, 6.822540000000000271e-01, 1.956799999999999859e-02, 0.000000000000000000e+00, 1.399999999999999877e-04, 2.593799999999999994e-01, 9.755819999999999492e-01, 1.670409999999999950e-01, 1.495779999999999887e-01, 3.529593000000000202e+00, 8.462000000000000077e-02, 4.500000000000000283e-05, 3.771000000000000046e-03, 6.431599999999999540e-01, 9.098610000000000309e-01, 1.121999999999999950e-03, 1.949639999999999984e-01, 1.322233999999999909e+00, 8.843999999999999431e-03, 0.000000000000000000e+00, 6.279999999999999775e-04, 3.185950000000000171e-01, 4.276200000000000140e-02, 7.478700000000000347e-01, 8.720839999999999703e-01, 9.290000000000000297e-04, 4.070599999999999913e-02, 1.773101999999999956e+00, 1.350000000000000017e-04, 7.485899999999999499e-02, 3.600380000000000247e-01, 2.716279999999999806e-01, 9.399999999999999448e-05, 7.787800000000000278e-02, 2.635669999999999957e-01],[1.523250999999999911e+00, 2.599090000000000011e-01, 2.561740000000000128e-01, 1.243753000000000108e+00, 0.000000000000000000e+00, 8.373639000000000721e+00, 9.459274999999999878e+00, 1.915858000000000061e+00, 1.316696000000000089e+00, 4.410839999999999761e-01, 5.119679999999999787e-01, 2.397415000000000074e+00, 2.260423999999999989e+00, 4.890200000000000102e-02, 8.537789999999999546e-01, 2.960139999999999993e-01, 1.660884999999999945e+00, 2.191999999999999851e-02, 1.301999999999999989e-03, 8.237300000000000177e-02, 1.018005000000000049e+00, 2.622999999999999984e-02, 1.500100000000000044e-02, 3.685379999999999767e-01, 3.605749999999999789e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.890000000000000053e-04, 9.445599999999999552e-01, 1.190000000000000058e-04, 6.999999999999999387e-05, 8.570000000000000144e-04, 1.164865999999999957e+00, 3.354000000000000210e-03, 8.379999999999999862e-04, 9.298800000000000121e-02, 2.735831000000000124e+00, 1.966499999999999845e-02, 4.984800000000000342e-02, 3.241620000000000057e-01, 4.667700000000000182e-01, 0.000000000000000000e+00, 6.099999999999999737e-04, 5.339999999999999695e-04, 3.879354999999999887e+00, 2.514999999999999875e-03, 1.143300000000000045e-02, 5.625000000000000139e-02, 5.336300000000000071e-02, 2.080410000000000037e-01, 5.709933000000000369e+00, 4.097210000000000019e-01, 3.498459999999999903e-01, 2.018482999999999805e+00, 6.122699999999999670e-02, 6.042100000000000248e-02, 4.157750000000000057e-01, 8.715680000000000094e-01, 1.053899999999999976e-02, 1.113399999999999980e-02, 7.994800000000000517e-02],[8.947599999999999998e-02, 1.346384999999999943e+00, 6.233659999999999757e-01, 6.755400000000000293e-02, 8.373639000000000721e+00, 0.000000000000000000e+00, 8.470046999999999215e+00, 3.991582999999999881e+00, 2.169799999999999854e-02, 4.699374999999999858e+00, 1.407009999999999927e-01, 2.758650000000000269e-01, 1.670100000000000057e-02, 1.066284999999999927e+00, 4.337590000000000057e-01, 3.457299999999999957e-02, 6.651899999999999480e-02, 9.693869999999999987e-01, 6.174640000000000128e-01, 1.124999999999999915e-03, 2.713799999999999893e-02, 8.709670000000000467e-01, 5.270939999999999515e-01, 4.412499999999999756e-02, 2.079000000000000119e-03, 1.481721000000000066e+00, 1.777400000000000160e-02, 1.059999999999999961e-03, 6.790000000000000245e-04, 8.974900000000000100e-01, 7.365700000000000025e-02, 3.959999999999999762e-04, 1.096639999999999976e-01, 1.013424000000000103e+00, 3.601960000000000162e-01, 5.590000000000000351e-04, 7.734099999999999309e-02, 2.159327999999999914e+00, 7.288909999999999556e-01, 1.201210000000000055e-01, 1.119999999999999983e-04, 5.189030000000000031e-01, 4.470000000000000232e-04, 1.119999999999999983e-04, 1.956999999999999972e-03, 2.862216000000000093e+00, 8.964300000000000046e-02, 3.020000000000000115e-03, 3.935449999999999782e-01, 1.160000000000000007e-04, 1.501069999999999904e-01, 3.829579999999999984e+00, 2.395833000000000101e+00, 1.969829999999999914e-01, 8.060709999999999820e-01, 3.782200000000000145e-02, 7.759999999999999971e-04, 6.200000000000000270e-05, 3.715410000000000101e-01, 2.990000000000000036e-03, 9.299999999999999727e-05],[1.995889999999999886e-01, 8.221330000000000027e-01, 1.144638999999999962e+00, 2.243970000000000131e-01, 9.459274999999999878e+00, 8.470046999999999215e+00, 0.000000000000000000e+00, 2.116657000000000011e+00, 8.790499999999999703e-02, 2.871847999999999956e+00, 7.654950000000000365e-01, 5.635660000000000114e-01, 1.544409999999999950e-01, 3.117590000000000083e-01, 1.376727000000000034e+00, 9.067400000000000460e-02, 2.854000000000000200e-03, 1.030800000000000050e-01, 1.766961000000000004e+00, 7.693800000000000638e-02, 8.347499999999999365e-02, 4.352429999999999910e-01, 9.622950000000000115e-01, 7.971200000000000507e-02, 2.910999999999999959e-03, 1.424300000000000045e-02, 2.985421000000000102e+00, 4.592999999999999861e-03, 4.079999999999999968e-04, 5.950000000000000427e-04, 7.332020000000000204e-01, 2.640000000000000202e-04, 3.464000000000000065e-03, 9.520699999999999996e-02, 1.447392000000000012e+00, 4.259500000000000092e-02, 3.185740000000000238e-01, 4.985040000000000027e-01, 2.866324999999999790e+00, 1.085010000000000002e-01, 1.869999999999999917e-04, 6.459000000000000331e-03, 7.164409999999999945e-01, 3.739999999999999835e-04, 2.527999999999999865e-03, 1.029799999999999986e-02, 3.092499999999999805e+00, 4.867999999999999931e-03, 1.777569999999999983e-01, 4.211899999999999672e-02, 5.165939999999999976e-01, 1.562591000000000063e+00, 4.919173999999999936e+00, 3.611639999999999850e-01, 1.732980000000000076e-01, 3.626809999999999756e-01, 5.387399999999999828e-02, 6.200000000000000270e-05, 3.648900000000000060e-02, 5.868179999999999508e-01, 8.031999999999999210e-03],[8.781630000000000269e-01, 2.960659999999999958e-01, 2.910090000000000177e-01, 2.057449000000000083e+00, 1.915858000000000061e+00, 3.991582999999999881e+00, 2.116657000000000011e+00, 0.000000000000000000e+00, 4.329180000000000250e-01, 6.228679999999999772e-01, 3.096559999999999868e-01, 4.943438999999999695e+00, 3.518819999999999726e-01, 7.587800000000000100e-02, 4.198310000000000097e-01, 1.266653999999999947e+00, 3.744190000000000018e-01, 2.719999999999999979e-04, 2.908799999999999927e-02, 8.987939999999999818e-01, 3.455449999999999910e-01, 5.380499999999999866e-02, 1.147600000000000009e-02, 1.050362999999999936e+00, 1.039999999999999909e-03, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.446009999999999907e-01, 4.348000000000000302e-03, 6.000000000000000152e-05, 6.999999999999999387e-05, 8.312150000000000372e-01, 2.514230000000000076e-01, 1.579999999999999916e-04, 3.358099999999999974e-02, 9.289690000000000447e-01, 3.817960000000000242e-01, 3.855900000000000299e-02, 2.386200000000000127e-02, 2.359361999999999959e+00, 2.448999999999999788e-03, 0.000000000000000000e+00, 5.529999999999999976e-04, 3.803739999999999899e-01, 9.008000000000000382e-03, 1.026999999999999918e-03, 3.635000000000000210e-03, 2.898791000000000118e+00, 5.990000000000000316e-04, 5.999320000000000208e-01, 2.297807000000000155e+00, 4.873170000000000002e-01, 1.997480000000000089e-01, 4.693944000000000116e+00, 1.482999999999999943e-03, 3.593899999999999872e-02, 1.089585000000000026e+00, 1.247299999999999971e-02, 1.410000000000000120e-04, 1.565199999999999925e-02, 4.972929999999999851e-01],[5.815293999999999741e+00, 2.738550000000000151e-01, 1.197613999999999956e+00, 9.368400000000000061e-01, 1.316696000000000089e+00, 2.169799999999999854e-02, 8.790499999999999703e-02, 4.329180000000000250e-01, 0.000000000000000000e+00, 4.039130000000000220e-01, 3.916186999999999863e+00, 1.644621000000000111e+00, 7.800080000000000346e-01, 1.233099999999999995e-02, 3.057139999999999858e-01, 1.160390000000000033e-01, 1.242140000000000022e+00, 2.554400000000000073e-02, 2.889999999999999774e-04, 4.111150000000000082e-01, 4.935080000000000022e-01, 3.840000000000000097e-04, 0.000000000000000000e+00, 1.151100000000000040e-02, 1.949764000000000053e+00, 1.113671000000000078e+00, 2.418858999999999870e+00, 7.450523999999999702e+00, 4.094320000000000181e-01, 6.000000000000000152e-05, 0.000000000000000000e+00, 4.747000000000000351e-03, 6.681730000000000169e-01, 7.179999999999999967e-04, 4.099999999999999912e-04, 3.047400000000000109e-02, 4.617290000000000005e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.385199999999999973e-02, 9.384390000000000231e-01, 8.210000000000000067e-04, 6.956700000000000383e-02, 3.765700000000000297e-02, 5.606419999999999737e-01, 2.049999999999999956e-04, 1.029999999999999964e-04, 3.211599999999999872e-02, 6.090899999999999787e-02, 4.243210000000000037e-01, 1.094735999999999931e+00, 2.043999999999999810e-03, 1.150000000000000035e-04, 4.789000000000000201e-02, 3.088300000000000073e-02, 1.452450000000000130e-01, 1.443879999999999886e-01, 4.237739999999999840e-01, 1.260000000000000009e-02, 1.717999999999999909e-03, 1.405160000000000020e-01],[3.342240000000000211e-01, 4.127466000000000079e+00, 9.288760000000000350e-01, 6.437889999999999446e-01, 4.410839999999999761e-01, 4.699374999999999858e+00, 2.871847999999999956e+00, 6.228679999999999772e-01, 4.039130000000000220e-01, 0.000000000000000000e+00, 1.286990000000000078e+00, 8.579805999999999599e+00, 2.153899999999999912e-02, 2.385329999999999950e-01, 3.045809999999999906e-01, 3.114000000000000101e-02, 2.327009999999999912e-01, 1.282521999999999940e+00, 1.418382999999999949e+00, 5.060000000000000261e-03, 5.194999999999999965e-03, 1.337628999999999957e+00, 1.046809999999999963e-01, 1.247000000000000062e-03, 3.545449999999999990e-01, 1.320875000000000021e+00, 1.360141000000000044e+00, 3.879409999999999803e-01, 2.717999999999999930e-03, 3.005129999999999746e-01, 1.729999999999999941e-02, 3.299999999999999982e-04, 3.083719999999999795e-01, 2.005333999999999950e+00, 8.226430000000000131e-01, 1.248100000000000077e-02, 6.335300000000000653e-02, 2.214735000000000120e+00, 2.002049999999999941e-01, 5.328000000000000271e-03, 2.987380000000000035e-01, 2.392605999999999788e+00, 1.149845999999999924e+00, 3.260350000000000192e-01, 6.570000000000000319e-04, 3.812869999999999870e-01, 4.159999999999999636e-03, 2.190000000000000106e-04, 3.468899999999999761e-01, 1.160000000000000007e-04, 1.049876999999999949e+00, 8.929759999999999920e-01, 9.818281000000000702e+00, 9.476199999999999901e-02, 1.372357000000000049e+00, 1.671380000000000088e-01, 2.480989999999999862e-01, 4.319999999999999838e-04, 1.667059999999999931e-01, 5.790000000000000008e-03, 9.299999999999999727e-05],[1.868193999999999910e+00, 6.004150000000000320e-01, 7.316622999999999877e+00, 4.543470000000000009e-01, 5.119679999999999787e-01, 1.407009999999999927e-01, 7.654950000000000365e-01, 3.096559999999999868e-01, 3.916186999999999863e+00, 1.286990000000000078e+00, 0.000000000000000000e+00, 1.477695999999999898e+00, 3.174660000000000259e-01, 6.236900000000000083e-02, 9.058100000000000041e-01, 9.354200000000000015e-02, 3.785999999999999869e-03, 1.690220000000000056e-01, 1.148387000000000047e+00, 1.409430000000000127e-01, 9.225000000000000561e-03, 3.324400000000000271e-02, 2.738760000000000083e-01, 6.043199999999999961e-02, 5.287418999999999869e+00, 4.087042000000000286e+00, 8.531553000000000608e+00, 6.798629000000000033e+00, 9.510000000000000224e-04, 9.520000000000000467e-04, 4.032000000000000167e-02, 9.560000000000000789e-03, 4.420000000000000318e-03, 8.985999999999999197e-03, 1.195789999999999909e+00, 1.576299999999999923e-02, 4.420000000000000318e-03, 1.768000000000000040e-03, 1.878319999999999990e-01, 4.537399999999999767e-02, 1.655870000000000120e-01, 1.222700000000000002e-02, 1.558783999999999947e+00, 4.743700000000000000e-02, 2.062000000000000031e-03, 2.062000000000000031e-03, 3.143790000000000195e-01, 5.170899999999999802e-02, 2.695100000000000273e-01, 2.004990000000000105e-01, 9.261000000000000135e-03, 2.871500000000000094e-02, 5.143919999999999604e-01, 6.399399999999999533e-02, 7.053299999999999847e-02, 4.967059999999999809e-01, 1.221860000000000029e-01, 2.469999999999999865e-04, 5.677100000000000202e-02, 5.068409999999999860e-01, 8.330999999999999517e-02],[1.293860000000000010e+00, 5.866850000000000120e-01, 3.890040000000000164e-01, 4.971506999999999898e+00, 2.397415000000000074e+00, 2.758650000000000269e-01, 5.635660000000000114e-01, 4.943438999999999695e+00, 1.644621000000000111e+00, 8.579805999999999599e+00, 1.477695999999999898e+00, 0.000000000000000000e+00, 2.698619999999999908e-01, 1.872700000000000059e-02, 3.063750000000000084e-01, 3.591829999999999745e-01, 1.177718000000000043e+00, 4.530000000000000064e-04, 8.326100000000000168e-02, 1.293228000000000044e+00, 2.104800000000000074e-02, 2.880000000000000073e-04, 1.066000000000000107e-03, 1.035015000000000018e+00, 5.308810000000000473e-01, 1.814599999999999894e-02, 2.500600000000000045e-02, 7.723489999999999522e-01, 9.511999999999999622e-03, 6.000000000000000152e-05, 1.050000000000000044e-04, 3.228070000000000106e-01, 8.325919999999999987e-01, 3.879999999999999985e-04, 1.290979999999999905e-01, 1.702817000000000025e+00, 1.263819999999999943e-01, 4.850000000000000253e-04, 6.401999999999999705e-03, 2.323090999999999795e+00, 1.179052999999999907e+00, 5.402499999999999664e-02, 6.849680000000000213e-01, 2.382451000000000096e+00, 6.304999999999999841e-03, 9.699999999999999963e-05, 1.358000000000000049e-03, 5.014560000000000128e-01, 1.800000000000000113e-04, 5.064699999999999758e-01, 5.964323000000000263e+00, 8.363600000000000201e-02, 6.815750000000000419e-01, 2.169045000000000112e+00, 1.817940000000000111e-01, 1.390460000000000029e-01, 2.007767999999999997e+00, 1.092899999999999948e-02, 4.699999999999999724e-05, 9.289000000000000382e-03, 2.255540000000000045e-01],[6.673970000000000180e-01, 2.083700000000000135e-02, 5.626699999999999757e-02, 2.562829999999999830e-01, 2.260423999999999989e+00, 1.670100000000000057e-02, 1.544409999999999950e-01, 3.518819999999999726e-01, 7.800080000000000346e-01, 2.153899999999999912e-02, 3.174660000000000259e-01, 2.698619999999999908e-01, 0.000000000000000000e+00, 5.784671999999999592e+00, 2.230691000000000201e+00, 6.415611000000000175e+00, 3.900870000000000171e-01, 2.719999999999999979e-04, 4.800000000000000122e-05, 4.010899999999999882e-02, 4.805209999999999759e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.014000000000000165e-03, 1.218550000000000050e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.400610000000000355e+00, 1.361479999999999912e-01, 4.888200000000000184e-02, 1.260239000000000109e+00, 3.614029999999999743e-01, 0.000000000000000000e+00, 7.799999999999999859e-05, 1.170999999999999949e-03, 9.839259999999999673e-01, 0.000000000000000000e+00, 7.799999999999999859e-05, 1.794999999999999938e-03, 1.943189999999999917e-01, 0.000000000000000000e+00, 7.799999999999999859e-05, 1.559999999999999972e-04, 1.271061999999999914e+00, 1.542070000000000107e-01, 9.111930000000000307e-01, 2.577577999999999925e+00, 2.545400000000000093e-02, 5.730129999999999946e-01, 4.034349999999999881e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.240000000000000062e-04, 1.213799999999999948e-02, 1.700409999999999977e-01, 6.955500000000000571e-02, 5.159074999999999633e+00, 6.496799999999999797e-02, 1.065536999999999956e+00, 1.075186999999999893e+00],[2.301999999999999880e-02, 1.754910000000000081e-01, 1.164049999999999946e-01, 1.435099999999999917e-02, 4.890200000000000102e-02, 1.066284999999999927e+00, 3.117590000000000083e-01, 7.587800000000000100e-02, 1.233099999999999995e-02, 2.385329999999999950e-01, 6.236900000000000083e-02, 1.872700000000000059e-02, 5.784671999999999592e+00, 0.000000000000000000e+00, 1.428293000000000035e+00, 3.609309999999999796e+00, 1.054300000000000029e-02, 1.278990000000000127e-01, 1.058850000000000069e-01, 4.690000000000000158e-04, 4.479999999999999933e-04, 2.702740000000000142e-01, 2.172299999999999925e-02, 9.600000000000000243e-05, 1.247999999999999977e-03, 2.388389999999999957e-01, 3.063999999999999883e-03, 1.179999999999999951e-04, 2.030170000000000030e-01, 3.765696999999999850e+00, 2.469249000000000027e+00, 3.901899999999999813e-01, 9.173999999999999863e-03, 3.244699999999999668e-02, 3.198399999999999854e-02, 4.199999999999999768e-05, 5.890000000000000053e-04, 5.363620000000000054e-01, 4.629000000000000302e-03, 2.530000000000000239e-04, 8.399999999999999535e-05, 5.542500000000000204e-02, 2.950000000000000148e-04, 0.000000000000000000e+00, 3.137789999999999746e-01, 7.487815999999999583e+00, 3.487341999999999942e+00, 5.759210000000000163e-01, 2.559750000000000081e-01, 1.160000000000000007e-04, 9.600000000000000243e-05, 1.159749999999999948e-01, 3.440000000000000132e-04, 0.000000000000000000e+00, 3.096430000000000016e-01, 1.368489999999999984e-01, 3.110000000000000243e-04, 4.211199999999999666e-02, 1.006045000000000078e+00, 4.780190000000000272e-01, 5.603799999999999754e-02],[2.753550000000000164e-01, 2.191700000000000037e-01, 4.975930000000000075e-01, 2.089239999999999986e-01, 8.537789999999999546e-01, 4.337590000000000057e-01, 1.376727000000000034e+00, 4.198310000000000097e-01, 3.057139999999999858e-01, 3.045809999999999906e-01, 9.058100000000000041e-01, 3.063750000000000084e-01, 2.230691000000000201e+00, 1.428293000000000035e+00, 0.000000000000000000e+00, 1.155097999999999958e+00, 4.436169999999999836e-01, 1.634059999999999957e-01, 8.349760000000000515e-01, 3.235879999999999868e-01, 4.200600000000000167e-02, 8.455000000000000709e-03, 1.934589999999999921e-01, 8.163099999999999523e-02, 1.559600000000000049e-02, 2.978599999999999998e-02, 4.086799999999999877e-01, 2.873600000000000113e-02, 1.635280999999999985e+00, 1.434579000000000049e+00, 2.236556999999999906e+00, 1.220020000000000104e+00, 3.399400000000000338e-02, 1.137900000000000023e-02, 2.624650000000000039e-01, 5.906499999999999945e-02, 5.178960000000000230e-01, 1.692639999999999978e-01, 8.373019999999999907e-01, 4.511339999999999795e-01, 3.772400000000000059e-02, 1.860300000000000148e-02, 4.101989999999999803e-01, 2.747899999999999995e-02, 5.690760000000000263e-01, 3.504729999999999790e-01, 1.399672999999999945e+00, 4.301240000000000063e-01, 3.003540000000000099e-01, 2.943680000000000185e-01, 2.460940000000000072e-01, 4.228200000000000014e-02, 5.270049999999999457e-01, 9.349100000000000465e-02, 3.246129999999999849e-01, 3.216419999999999835e-01, 2.980909999999999949e-01, 1.418466000000000005e+00, 8.006020000000000358e-01, 3.062806999999999835e+00, 6.793700000000000294e-01],[2.215360000000000107e-01, 2.540500000000000050e-02, 4.143999999999999767e-02, 2.393879999999999897e-01, 2.960139999999999993e-01, 3.457299999999999957e-02, 9.067400000000000460e-02, 1.266653999999999947e+00, 1.160390000000000033e-01, 3.114000000000000101e-02, 9.354200000000000015e-02, 3.591829999999999745e-01, 6.415611000000000175e+00, 3.609309999999999796e+00, 1.155097999999999958e+00, 0.000000000000000000e+00, 7.892599999999999616e-02, 0.000000000000000000e+00, 3.859000000000000017e-03, 1.933279999999999998e-01, 2.596999999999999916e-03, 9.600000000000000243e-05, 2.460000000000000164e-04, 2.928549999999999764e-01, 4.159999999999999744e-04, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.643400000000000194e-02, 1.328581000000000012e+00, 1.840100000000000069e-01, 8.930799999999999850e-02, 3.330792999999999893e+00, 3.861200000000000049e-02, 0.000000000000000000e+00, 3.311999999999999840e-03, 6.696599999999999775e-02, 2.262000000000000122e-03, 4.000000000000000327e-05, 8.100000000000000375e-05, 5.363320000000000309e-01, 2.829999999999999942e-04, 2.829999999999999942e-04, 8.100000000000000375e-05, 6.999500000000000166e-02, 2.581653999999999893e+00, 4.586219999999999741e-01, 9.533530000000000060e-01, 6.546162999999999954e+00, 1.800000000000000113e-04, 3.527310000000000167e-01, 3.860000000000000041e-04, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.724030000000000007e-01, 4.049999999999999781e-04, 7.831200000000000661e-02, 4.314560000000000062e-01, 2.130663000000000196e+00, 5.995600000000000235e-02, 1.292934999999999945e+00, 1.119410999999999934e+00],[3.215816999999999926e+00, 3.774469999999999770e-01, 4.613829999999999876e-01, 1.759732000000000074e+00, 1.660884999999999945e+00, 6.651899999999999480e-02, 2.854000000000000200e-03, 3.744190000000000018e-01, 1.242140000000000022e+00, 2.327009999999999912e-01, 3.785999999999999869e-03, 1.177718000000000043e+00, 3.900870000000000171e-01, 1.054300000000000029e-02, 4.436169999999999836e-01, 7.892599999999999616e-02, 0.000000000000000000e+00, 7.519040000000000168e-01, 7.923045000000000115e+00, 2.473438999999999943e+00, 1.052040000000000086e+00, 7.484599999999999587e-02, 6.336600000000000565e-02, 4.351090000000000235e-01, 4.725839999999999819e+00, 1.489399999999999891e-01, 2.999780000000000224e-01, 1.164261999999999908e+00, 1.205737999999999976e+00, 7.179399999999999671e-02, 3.337199999999999889e-02, 5.290449999999999875e-01, 3.081614000000000075e+00, 1.213889999999999969e-01, 3.433140000000000081e-01, 7.067820000000000213e-01, 1.274118000000000084e+00, 1.881400000000000086e-02, 1.165000000000000020e-03, 2.729500000000000259e-01, 5.514930000000000110e-01, 1.619299999999999906e-02, 3.029000000000000008e-03, 4.281200000000000283e-02, 6.069449999999999568e-01, 6.756999999999999985e-03, 4.659999999999999971e-04, 1.508039999999999936e-01, 7.923600000000000088e-02, 4.862810000000000188e-01, 2.066955000000000098e+00, 8.038299999999999612e-02, 2.178999999999999947e-03, 5.083079999999999821e-01, 3.223199999999999676e-02, 5.516299999999999676e-02, 1.631740000000000135e-01, 5.764759999999999884e-01, 1.662800000000000042e-02, 7.507999999999999917e-03, 1.263199999999999878e-01],[1.434789999999999954e-01, 2.203539000000000136e+00, 5.714690000000000047e-01, 1.885870000000000046e-01, 2.191999999999999851e-02, 9.693869999999999987e-01, 1.030800000000000050e-01, 2.719999999999999979e-04, 2.554400000000000073e-02, 1.282521999999999940e+00, 1.690220000000000056e-01, 4.530000000000000064e-04, 2.719999999999999979e-04, 1.278990000000000127e-01, 1.634059999999999957e-01, 0.000000000000000000e+00, 7.519040000000000168e-01, 0.000000000000000000e+00, 2.503267999999999827e+00, 8.685700999999999894e+00, 8.983499999999999819e-02, 7.303060000000000107e-01, 3.641289999999999805e-01, 6.158299999999999885e-02, 5.269040000000000390e-01, 2.478358000000000061e+00, 2.208429999999999893e+00, 2.772649999999999837e-01, 5.992700000000000110e-02, 6.176939999999999653e-01, 3.300359999999999960e-01, 5.861399999999999944e-02, 1.060689999999999966e-01, 9.757270000000000110e-01, 5.660340000000000371e-01, 4.673900000000000277e-02, 4.982000000000000317e-03, 6.960159999999999680e-01, 1.900370000000000115e-01, 9.100000000000000287e-05, 4.982000000000000317e-03, 5.667590000000000128e-01, 5.371399999999999786e-02, 9.100000000000000287e-05, 2.808000000000000166e-03, 4.910340000000000260e-01, 4.411199999999999843e-02, 9.100000000000000287e-05, 3.193995999999999835e+00, 2.657660000000000022e-01, 1.447000000000000022e-02, 1.144692000000000043e+00, 1.145510000000000000e-01, 5.929999999999999941e-04, 4.830760000000000054e-01, 3.692730000000000179e-01, 1.550000000000000000e-04, 1.230000000000000082e-04, 7.617709999999999759e-01, 2.162999999999999992e-03, 4.670000000000000214e-04],[2.853840000000000265e-01, 9.851450000000000484e-01, 2.285051999999999861e+00, 2.068510000000000071e-01, 1.301999999999999989e-03, 6.174640000000000128e-01, 1.766961000000000004e+00, 2.908799999999999927e-02, 2.889999999999999774e-04, 1.418382999999999949e+00, 1.148387000000000047e+00, 8.326100000000000168e-02, 4.800000000000000122e-05, 1.058850000000000069e-01, 8.349760000000000515e-01, 3.859000000000000017e-03, 7.923045000000000115e+00, 2.503267999999999827e+00, 0.000000000000000000e+00, 1.106179000000000023e+00, 1.843269999999999909e-01, 5.295940000000000092e-01, 7.157160000000000188e-01, 1.074340000000000017e-01, 6.173199999999999799e-01, 2.029914000000000218e+00, 4.718199000000000254e+00, 3.408109999999999751e-01, 1.054490000000000011e-01, 4.226710000000000189e-01, 7.037949999999999484e-01, 7.911899999999999489e-02, 5.365670000000000162e-01, 9.283190000000000053e-01, 3.036767000000000216e+00, 1.308729999999999893e-01, 2.219000000000000052e-03, 6.349270000000000191e-01, 1.544626000000000055e+00, 4.090699999999999892e-02, 2.889999999999999774e-04, 3.362769999999999926e-01, 6.342029999999999612e-01, 3.376999999999999794e-03, 1.450000000000000009e-04, 6.932000000000000661e-02, 4.754959999999999742e-01, 9.599999999999999159e-03, 3.323960000000000248e-01, 9.206999999999999906e-02, 2.122000000000000188e-03, 4.233819999999999806e-01, 1.519211000000000089e+00, 4.924599999999999811e-02, 1.055969999999999964e-01, 2.474899999999999878e-01, 5.791000000000000314e-02, 0.000000000000000000e+00, 1.072179999999999939e-01, 2.507480000000000264e-01, 9.759999999999999579e-03],[8.992620000000000058e-01, 4.609699999999999909e-01, 3.245250000000000079e-01, 2.691225999999999896e+00, 8.237300000000000177e-02, 1.124999999999999915e-03, 7.693800000000000638e-02, 8.987939999999999818e-01, 4.111150000000000082e-01, 5.060000000000000261e-03, 1.409430000000000127e-01, 1.293228000000000044e+00, 4.010899999999999882e-02, 4.690000000000000158e-04, 3.235879999999999868e-01, 1.933279999999999998e-01, 2.473438999999999943e+00, 8.685700999999999894e+00, 1.106179000000000023e+00, 0.000000000000000000e+00, 2.862520000000000064e-01, 9.127599999999999603e-02, 5.402099999999999957e-02, 7.752539999999999987e-01, 1.353860000000000063e+00, 1.700879999999999892e-01, 3.163719999999999866e-01, 2.460976000000000052e+00, 4.386479999999999824e-01, 5.732799999999999702e-02, 4.474199999999999705e-02, 6.648659999999999570e-01, 4.524420000000000108e-01, 1.009280000000000038e-01, 1.592169999999999974e-01, 7.467920000000000114e-01, 3.926500000000000129e-02, 5.619999999999999996e-04, 8.012400000000000078e-02, 4.205800000000000094e-01, 9.755400000000000182e-02, 8.429999999999999993e-04, 8.405999999999999583e-02, 3.881560000000000010e-01, 1.958599999999999924e-02, 3.750000000000000078e-04, 3.223700000000000176e-02, 3.382069999999999799e-01, 3.405409999999999826e-01, 3.088480999999999810e+00, 1.281110000000000027e-01, 2.137000000000000011e-03, 7.373000000000000387e-02, 1.253945000000000087e+00, 1.482999999999999943e-03, 1.950729999999999964e-01, 6.340649999999999897e-01, 2.655100000000000168e-02, 1.312000000000000015e-03, 4.027499999999999830e-02, 6.386960000000000415e-01],[1.042456999999999967e+00, 7.416100000000000469e-02, 1.200200000000000052e-02, 9.610399999999999499e-02, 1.018005000000000049e+00, 2.713799999999999893e-02, 8.347499999999999365e-02, 3.455449999999999910e-01, 4.935080000000000022e-01, 5.194999999999999965e-03, 9.225000000000000561e-03, 2.104800000000000074e-02, 4.805209999999999759e-01, 4.479999999999999933e-04, 4.200600000000000167e-02, 2.596999999999999916e-03, 1.052040000000000086e+00, 8.983499999999999819e-02, 1.843269999999999909e-01, 2.862520000000000064e-01, 0.000000000000000000e+00, 5.426732999999999585e+00, 5.127582000000000306e+00, 9.227009999999999934e-01, 6.266030000000000211e-01, 1.701999999999999954e-03, 1.030000000000000013e-02, 6.341299999999999715e-02, 1.073790000000000022e+00, 1.660899999999999876e-02, 4.091700000000000198e-02, 6.566900000000000515e-02, 8.908220000000000027e-01, 3.671999999999999917e-03, 1.791000000000000057e-03, 1.478730000000000044e-01, 1.236456999999999917e+00, 1.307700000000000008e-02, 1.782399999999999957e-02, 3.469780000000000086e-01, 5.562040000000000317e-01, 9.000000000000000567e-05, 1.253999999999999906e-03, 2.507999999999999813e-03, 7.321109999999999562e-01, 5.370000000000000424e-04, 7.169999999999999724e-04, 9.225000000000000561e-03, 2.587300000000000016e-02, 3.214000000000000190e-02, 3.121656000000000208e+00, 1.387420000000000042e-01, 1.515890000000000015e-01, 4.875419999999999754e-01, 1.025000000000000043e-02, 7.030799999999999550e-02, 2.795000000000000175e-03, 2.610689999999999955e-01, 6.557999999999999594e-03, 5.534999999999999989e-03, 2.521699999999999983e-02],[1.239400000000000050e-02, 5.320920000000000094e-01, 3.036130000000000217e-01, 1.920000000000000049e-04, 2.622999999999999984e-02, 8.709670000000000467e-01, 4.352429999999999910e-01, 5.380499999999999866e-02, 3.840000000000000097e-04, 1.337628999999999957e+00, 3.324400000000000271e-02, 2.880000000000000073e-04, 0.000000000000000000e+00, 2.702740000000000142e-01, 8.455000000000000709e-03, 9.600000000000000243e-05, 7.484599999999999587e-02, 7.303060000000000107e-01, 5.295940000000000092e-01, 9.127599999999999603e-02, 5.426732999999999585e+00, 0.000000000000000000e+00, 3.453780000000000072e+00, 3.496083000000000052e+00, 1.540169999999999872e-01, 5.637990000000000501e-01, 8.971010000000000373e-01, 1.242319999999999952e-01, 4.008699999999999763e-02, 6.487690000000000401e-01, 1.497709999999999875e-01, 2.657100000000000087e-02, 1.051119999999999971e-01, 9.761750000000000149e-01, 2.942940000000000000e-01, 7.690000000000000438e-04, 6.975399999999999656e-02, 1.394892000000000021e+00, 4.175650000000000195e-01, 7.302100000000000257e-02, 3.840000000000000097e-04, 6.369160000000000377e-01, 1.441000000000000007e-03, 2.880000000000000073e-04, 1.920000000000000049e-04, 7.600900000000000434e-01, 1.536999999999999955e-03, 1.920000000000000049e-04, 3.104160000000000252e-01, 1.160000000000000007e-04, 7.235199999999999965e-02, 2.487136000000000013e+00, 4.325200000000000156e-01, 1.012040000000000023e-01, 2.157789999999999986e-01, 3.256400000000000267e-02, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.856349999999999723e-01, 3.820000000000000153e-04, 1.399999999999999877e-04],[4.099999999999999912e-04, 9.541800000000000281e-02, 5.615229999999999944e-01, 2.704999999999999939e-03, 1.500100000000000044e-02, 5.270939999999999515e-01, 9.622950000000000115e-01, 1.147600000000000009e-02, 0.000000000000000000e+00, 1.046809999999999963e-01, 2.738760000000000083e-01, 1.066000000000000107e-03, 0.000000000000000000e+00, 2.172299999999999925e-02, 1.934589999999999921e-01, 2.460000000000000164e-04, 6.336600000000000565e-02, 3.641289999999999805e-01, 7.157160000000000188e-01, 5.402099999999999957e-02, 5.127582000000000306e+00, 3.453780000000000072e+00, 0.000000000000000000e+00, 2.918259999999999743e-01, 9.107300000000000118e-02, 7.983460000000000001e-01, 1.164525000000000032e+00, 4.467600000000000043e-02, 1.644199999999999828e-02, 1.182290000000000008e-01, 4.951760000000000050e-01, 7.978000000000000716e-03, 5.409999999999999878e-03, 6.442350000000000021e-01, 9.903300000000000436e-01, 3.155999999999999778e-02, 8.738400000000000334e-02, 9.711480000000000112e-01, 1.950083000000000011e+00, 1.352599999999999983e-02, 1.640000000000000019e-04, 3.957709999999999839e-01, 7.030920000000000503e-01, 1.312000000000000015e-03, 8.200000000000000095e-05, 3.164200000000000346e-02, 6.024279999999999635e-01, 2.460000000000000164e-04, 3.737200000000000244e-02, 4.177000000000000157e-03, 6.492399999999999560e-02, 3.710629999999999762e-01, 2.075225999999999793e+00, 2.144400000000000125e-02, 4.584999999999999666e-03, 2.157880000000000076e-01, 1.550000000000000000e-04, 0.000000000000000000e+00, 7.916099999999999526e-02, 1.129990000000000022e-01, 4.062999999999999771e-03],[2.471949999999999981e-01, 9.590000000000000000e-04, 2.208160000000000123e-01, 7.873400000000000398e-01, 3.685379999999999767e-01, 4.412499999999999756e-02, 7.971200000000000507e-02, 1.050362999999999936e+00, 1.151100000000000040e-02, 1.247000000000000062e-03, 6.043199999999999961e-02, 1.035015000000000018e+00, 2.014000000000000165e-03, 9.600000000000000243e-05, 8.163099999999999523e-02, 2.928549999999999764e-01, 4.351090000000000235e-01, 6.158299999999999885e-02, 1.074340000000000017e-01, 7.752539999999999987e-01, 9.227009999999999934e-01, 3.496083000000000052e+00, 2.918259999999999743e-01, 0.000000000000000000e+00, 3.528069999999999817e-01, 2.397999999999999958e-03, 9.879999999999999893e-03, 5.004330000000000167e-01, 1.814109999999999889e-01, 3.625400000000000150e-02, 1.438799999999999975e-02, 8.226430000000000131e-01, 1.847489999999999966e-01, 9.600000000000000243e-05, 3.443700000000000233e-02, 6.617760000000000309e-01, 3.359240000000000004e-01, 2.206199999999999828e-02, 1.410100000000000069e-02, 1.272426000000000057e+00, 3.549000000000000071e-03, 9.600000000000000243e-05, 1.918000000000000000e-03, 4.188020000000000076e-01, 1.150999999999999896e-03, 1.920000000000000049e-04, 3.840000000000000097e-04, 7.083949999999999969e-01, 4.789999999999999879e-04, 3.378789999999999849e-01, 5.444740000000000135e-01, 1.963580000000000048e-01, 7.877599999999999880e-02, 2.846677000000000124e+00, 2.700000000000000035e-04, 6.183299999999999907e-02, 2.397150000000000114e-01, 1.358000000000000049e-03, 1.410000000000000120e-04, 3.498999999999999940e-03, 3.243679999999999897e-01],[3.065510000000000179e+00, 1.663600000000000148e-02, 3.451900000000000107e-02, 9.981000000000000288e-03, 3.605749999999999789e-01, 2.079000000000000119e-03, 2.910999999999999959e-03, 1.039999999999999909e-03, 1.949764000000000053e+00, 3.545449999999999990e-01, 5.287418999999999869e+00, 5.308810000000000473e-01, 1.218550000000000050e-01, 1.247999999999999977e-03, 1.559600000000000049e-02, 4.159999999999999744e-04, 4.725839999999999819e+00, 5.269040000000000390e-01, 6.173199999999999799e-01, 1.353860000000000063e+00, 6.266030000000000211e-01, 1.540169999999999872e-01, 9.107300000000000118e-02, 3.528069999999999817e-01, 0.000000000000000000e+00, 2.677656999999999954e+00, 5.746310000000000029e-01, 9.595442999999999500e+00, 9.181979999999999587e-01, 1.123949999999999949e-01, 4.196899999999999936e-02, 4.246070000000000122e-01, 4.402170000000000250e-01, 9.981000000000000288e-03, 1.039999999999999909e-03, 7.069999999999999896e-03, 2.865469999999999962e-01, 4.159999999999999744e-04, 6.239999999999999887e-04, 1.871000000000000050e-03, 2.472449999999999926e-01, 8.942000000000000295e-03, 1.434809999999999974e-01, 4.200500000000000067e-02, 2.551470000000000127e-01, 1.871000000000000050e-03, 8.319999999999999488e-04, 3.742999999999999799e-03, 5.168600000000000277e-02, 4.391699999999999771e-02, 5.570150000000000379e-01, 4.088999999999999753e-03, 3.210999999999999879e-03, 6.102999999999999918e-03, 1.661499999999999921e-01, 1.465009999999999923e-01, 2.327279999999999904e-01, 1.366469999999999907e-01, 1.601999999999999952e-02, 1.526999999999999929e-03, 3.012099999999999847e-02],[2.963580000000000103e-01, 8.631910000000000416e-01, 2.721042999999999878e+00, 6.799999999999999946e-05, 0.000000000000000000e+00, 1.481721000000000066e+00, 1.424300000000000045e-02, 0.000000000000000000e+00, 1.113671000000000078e+00, 1.320875000000000021e+00, 4.087042000000000286e+00, 1.814599999999999894e-02, 0.000000000000000000e+00, 2.388389999999999957e-01, 2.978599999999999998e-02, 0.000000000000000000e+00, 1.489399999999999891e-01, 2.478358000000000061e+00, 2.029914000000000218e+00, 1.700879999999999892e-01, 1.701999999999999954e-03, 5.637990000000000501e-01, 7.983460000000000001e-01, 2.397999999999999958e-03, 2.677656999999999954e+00, 0.000000000000000000e+00, 5.563093000000000288e+00, 7.244097000000000008e+00, 5.435999999999999860e-03, 5.448279999999999790e-01, 5.639560000000000128e-01, 1.602200000000000152e-02, 2.567799999999999944e-02, 9.155050000000000132e-01, 5.286920000000000508e-01, 0.000000000000000000e+00, 6.799999999999999946e-05, 1.610247999999999902e+00, 8.374399999999999900e-02, 0.000000000000000000e+00, 7.529999999999999801e-04, 6.760490000000000110e-01, 9.716500000000000137e-02, 4.656000000000000416e-03, 0.000000000000000000e+00, 6.485229999999999606e-01, 3.286999999999999991e-03, 0.000000000000000000e+00, 6.045409999999999950e-01, 1.160000000000000007e-04, 0.000000000000000000e+00, 6.858119999999999772e-01, 3.210600000000000259e-02, 0.000000000000000000e+00, 5.169270000000000254e-01, 1.764759999999999940e-01, 1.288599999999999988e-02, 0.000000000000000000e+00, 2.008060000000000123e-01, 6.399999999999999710e-05, 0.000000000000000000e+00],[1.103200000000000014e-02, 6.521200000000000607e-02, 3.832199999999999829e+00, 9.810000000000001011e-04, 0.000000000000000000e+00, 1.777400000000000160e-02, 2.985421000000000102e+00, 0.000000000000000000e+00, 2.418858999999999870e+00, 1.360141000000000044e+00, 8.531553000000000608e+00, 2.500600000000000045e-02, 0.000000000000000000e+00, 3.063999999999999883e-03, 4.086799999999999877e-01, 0.000000000000000000e+00, 2.999780000000000224e-01, 2.208429999999999893e+00, 4.718199000000000254e+00, 3.163719999999999866e-01, 1.030000000000000013e-02, 8.971010000000000373e-01, 1.164525000000000032e+00, 9.879999999999999893e-03, 5.746310000000000029e-01, 5.563093000000000288e+00, 0.000000000000000000e+00, 2.868484000000000034e+00, 4.090200000000000086e-02, 6.404950000000000365e-01, 7.943660000000000165e-01, 3.942799999999999777e-02, 8.580000000000000387e-04, 1.505269999999999941e-01, 2.171984000000000137e+00, 3.680000000000000003e-04, 1.230000000000000082e-04, 2.047099999999999961e-02, 3.409177999999999820e+00, 0.000000000000000000e+00, 1.041899999999999944e-02, 7.037280000000000202e-01, 5.908330000000000526e-01, 1.176800000000000068e-02, 0.000000000000000000e+00, 5.026000000000000086e-03, 7.013989999999999947e-01, 0.000000000000000000e+00, 8.193100000000000382e-02, 3.713000000000000154e-03, 0.000000000000000000e+00, 4.832000000000000357e-03, 1.903570999999999902e+00, 8.500000000000000611e-05, 1.831430000000000002e-01, 1.135909999999999975e+00, 2.390899999999999970e-02, 0.000000000000000000e+00, 2.735500000000000084e-02, 1.276960000000000039e-01, 9.340000000000000429e-04],[1.769878999999999980e+00, 3.180000000000000101e-03, 5.608309999999999684e-01, 6.822540000000000271e-01, 5.890000000000000053e-04, 1.059999999999999961e-03, 4.592999999999999861e-03, 3.446009999999999907e-01, 7.450523999999999702e+00, 3.879409999999999803e-01, 6.798629000000000033e+00, 7.723489999999999522e-01, 0.000000000000000000e+00, 1.179999999999999951e-04, 2.873600000000000113e-02, 7.643400000000000194e-02, 1.164261999999999908e+00, 2.772649999999999837e-01, 3.408109999999999751e-01, 2.460976000000000052e+00, 6.341299999999999715e-02, 1.242319999999999952e-01, 4.467600000000000043e-02, 5.004330000000000167e-01, 9.595442999999999500e+00, 7.244097000000000008e+00, 2.868484000000000034e+00, 0.000000000000000000e+00, 1.343940000000000135e-01, 6.875799999999999967e-02, 1.501899999999999936e-02, 6.638770000000000504e-01, 3.497800000000000215e-02, 2.359999999999999902e-04, 7.891000000000000444e-03, 1.625250000000000028e-01, 5.890000000000000053e-04, 2.359999999999999902e-04, 1.531000000000000026e-03, 3.242269999999999874e-01, 1.531000000000000069e-02, 2.143499999999999919e-02, 4.322199999999999653e-02, 2.411969999999999947e-01, 3.530000000000000152e-04, 5.890000000000000053e-04, 1.412999999999999977e-03, 3.368280000000000163e-01, 1.376999999999999969e-03, 5.343750000000000444e-01, 5.789999999999999791e-04, 1.208000000000000089e-03, 1.719999999999999958e-03, 6.606220000000000425e-01, 7.781499999999999528e-02, 1.969600000000000170e-02, 7.269079999999999986e-01, 7.410000000000000137e-04, 9.399999999999999448e-05, 3.053999999999999857e-03, 1.438319999999999876e-01],[4.336200000000000054e-01, 1.454000000000000084e-02, 6.790000000000000245e-04, 1.956799999999999859e-02, 9.445599999999999552e-01, 6.790000000000000245e-04, 4.079999999999999968e-04, 4.348000000000000302e-03, 4.094320000000000181e-01, 2.717999999999999930e-03, 9.510000000000000224e-04, 9.511999999999999622e-03, 4.400610000000000355e+00, 2.030170000000000030e-01, 1.635280999999999985e+00, 1.328581000000000012e+00, 1.205737999999999976e+00, 5.992700000000000110e-02, 1.054490000000000011e-01, 4.386479999999999824e-01, 1.073790000000000022e+00, 4.008699999999999763e-02, 1.644199999999999828e-02, 1.814109999999999889e-01, 9.181979999999999587e-01, 5.435999999999999860e-03, 4.090200000000000086e-02, 1.343940000000000135e-01, 0.000000000000000000e+00, 8.808275000000000077e+00, 8.319767000000000579e+00, 1.437257000000000007e+00, 3.134950000000000236e-01, 1.358999999999999965e-03, 1.359999999999999989e-04, 1.236600000000000026e-02, 7.411349999999999882e-01, 1.359999999999999989e-04, 0.000000000000000000e+00, 4.076999999999999895e-03, 1.657839999999999869e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.719999999999999979e-04, 2.047159000000000173e+00, 4.375600000000000322e-02, 1.731219999999999981e-01, 5.551040000000000418e-01, 1.197800000000000080e-02, 1.665000000000000160e-02, 4.960459999999999869e-01, 1.859999999999999945e-04, 1.150000000000000035e-04, 6.780000000000000002e-04, 2.157799999999999996e-02, 1.710599999999999968e-02, 2.064900000000000069e-02, 7.219895000000000174e+00, 1.985580000000000123e-01, 6.560965999999999632e+00, 1.036473999999999895e+00],[2.142999999999999940e-03, 1.878799999999999915e-01, 2.238400000000000112e-02, 0.000000000000000000e+00, 1.190000000000000058e-04, 8.974900000000000100e-01, 5.950000000000000427e-04, 6.000000000000000152e-05, 6.000000000000000152e-05, 3.005129999999999746e-01, 9.520000000000000467e-04, 6.000000000000000152e-05, 1.361479999999999912e-01, 3.765696999999999850e+00, 1.434579000000000049e+00, 1.840100000000000069e-01, 7.179399999999999671e-02, 6.176939999999999653e-01, 4.226710000000000189e-01, 5.732799999999999702e-02, 1.660899999999999876e-02, 6.487690000000000401e-01, 1.182290000000000008e-01, 3.625400000000000150e-02, 1.123949999999999949e-01, 5.448279999999999790e-01, 6.404950000000000365e-01, 6.875799999999999967e-02, 8.808275000000000077e+00, 0.000000000000000000e+00, 5.484087999999999852e+00, 5.874441000000000024e+00, 2.678999999999999958e-03, 1.352549999999999863e-01, 1.006100000000000050e-02, 0.000000000000000000e+00, 3.570000000000000040e-04, 1.191513999999999962e+00, 8.930000000000000221e-04, 4.169999999999999987e-04, 6.000000000000000152e-05, 2.405049999999999966e-01, 1.190000000000000058e-04, 6.000000000000000152e-05, 3.173000000000000126e-02, 2.720152999999999821e+00, 4.774390000000000023e-01, 7.286600000000000021e-02, 6.220890000000000031e-01, 1.160000000000000007e-04, 9.600000000000000243e-05, 3.428129999999999789e-01, 3.440000000000000132e-04, 8.500000000000000611e-05, 6.741759999999999975e-01, 1.137009999999999965e-01, 4.659999999999999971e-04, 1.373823000000000016e+00, 2.306109999999999882e+00, 5.651602999999999710e+00, 8.368399999999999450e-02],[3.499999999999999693e-05, 3.491600000000000259e-02, 1.330320000000000114e-01, 1.399999999999999877e-04, 6.999999999999999387e-05, 7.365700000000000025e-02, 7.332020000000000204e-01, 6.999999999999999387e-05, 0.000000000000000000e+00, 1.729999999999999941e-02, 4.032000000000000167e-02, 1.050000000000000044e-04, 4.888200000000000184e-02, 2.469249000000000027e+00, 2.236556999999999906e+00, 8.930799999999999850e-02, 3.337199999999999889e-02, 3.300359999999999960e-01, 7.037949999999999484e-01, 4.474199999999999705e-02, 4.091700000000000198e-02, 1.497709999999999875e-01, 4.951760000000000050e-01, 1.438799999999999975e-02, 4.196899999999999936e-02, 5.639560000000000128e-01, 7.943660000000000165e-01, 1.501899999999999936e-02, 8.319767000000000579e+00, 5.484087999999999852e+00, 0.000000000000000000e+00, 4.802017000000000202e+00, 1.399999999999999877e-04, 1.568599999999999856e-02, 1.763000000000000123e-01, 4.910000000000000085e-04, 3.510000000000000208e-04, 1.037659999999999971e-01, 1.524024000000000045e+00, 7.720000000000000083e-04, 6.999999999999999387e-05, 4.726799999999999752e-02, 1.850380000000000080e-01, 1.749999999999999982e-04, 2.052899999999999864e-02, 1.085169999999999968e+00, 2.262490000000000112e+00, 1.158000000000000002e-02, 3.088589999999999947e-01, 4.814999999999999836e-03, 0.000000000000000000e+00, 1.021999999999999905e-03, 2.669429999999999858e-01, 0.000000000000000000e+00, 2.659480000000000177e-01, 5.048660000000000370e-01, 2.638999999999999853e-03, 1.045943000000000067e+00, 1.341143999999999892e+00, 2.455337000000000103e+00, 7.355000000000000426e-02],[6.092199999999999699e-02, 3.299999999999999982e-04, 3.936200000000000115e-02, 2.593799999999999994e-01, 8.570000000000000144e-04, 3.959999999999999762e-04, 2.640000000000000202e-04, 8.312150000000000372e-01, 4.747000000000000351e-03, 3.299999999999999982e-04, 9.560000000000000789e-03, 3.228070000000000106e-01, 1.260239000000000109e+00, 3.901899999999999813e-01, 1.220020000000000104e+00, 3.330792999999999893e+00, 5.290449999999999875e-01, 5.861399999999999944e-02, 7.911899999999999489e-02, 6.648659999999999570e-01, 6.566900000000000515e-02, 2.657100000000000087e-02, 7.978000000000000716e-03, 8.226430000000000131e-01, 4.246070000000000122e-01, 1.602200000000000152e-02, 3.942799999999999777e-02, 6.638770000000000504e-01, 1.437257000000000007e+00, 5.874441000000000024e+00, 4.802017000000000202e+00, 0.000000000000000000e+00, 3.955999999999999794e-02, 6.600000000000000506e-05, 4.219999999999999793e-03, 1.294919999999999960e-01, 2.373999999999999808e-03, 1.320000000000000101e-04, 6.600000000000000506e-05, 8.797409999999999952e-01, 3.299999999999999982e-04, 0.000000000000000000e+00, 6.600000000000000506e-05, 1.697770000000000112e-01, 5.095950000000000202e-01, 1.053609999999999963e-01, 9.184399999999999509e-02, 1.897149999999999892e+00, 2.990000000000000036e-04, 4.854689999999999839e-01, 2.889999999999999774e-04, 1.859999999999999945e-04, 0.000000000000000000e+00, 2.779289999999999816e-01, 6.740000000000000114e-04, 1.012200000000000079e-02, 5.432399999999999451e-01, 2.035722999999999949e+00, 2.076919999999999877e-01, 1.161510999999999960e+00, 2.016256999999999966e+00],[2.258321000000000023e+00, 1.935439999999999938e-01, 3.088509999999999867e-01, 9.755819999999999492e-01, 1.164865999999999957e+00, 1.096639999999999976e-01, 3.464000000000000065e-03, 2.514230000000000076e-01, 6.681730000000000169e-01, 3.083719999999999795e-01, 4.420000000000000318e-03, 8.325919999999999987e-01, 3.614029999999999743e-01, 9.173999999999999863e-03, 3.399400000000000338e-02, 3.861200000000000049e-02, 3.081614000000000075e+00, 1.060689999999999966e-01, 5.365670000000000162e-01, 4.524420000000000108e-01, 8.908220000000000027e-01, 1.051119999999999971e-01, 5.409999999999999878e-03, 1.847489999999999966e-01, 4.402170000000000250e-01, 2.567799999999999944e-02, 8.580000000000000387e-04, 3.497800000000000215e-02, 3.134950000000000236e-01, 2.678999999999999958e-03, 1.399999999999999877e-04, 3.955999999999999794e-02, 0.000000000000000000e+00, 8.929030000000000022e-01, 2.984713999999999867e+00, 2.738551999999999875e+00, 9.054879999999999596e-01, 1.405460000000000043e-01, 2.275270000000000070e-01, 5.023549999999999960e-01, 6.391250000000000542e-01, 3.058500000000000107e-02, 1.369370000000000032e-01, 8.914800000000000502e-02, 4.692810000000000037e-01, 4.256499999999999867e-02, 5.502499999999999752e-02, 2.764989999999999948e-01, 4.372100000000000292e-02, 2.571220000000000172e-01, 1.265487000000000029e+00, 8.568000000000000616e-02, 2.178999999999999947e-03, 3.304810000000000247e-01, 3.641000000000000139e-03, 9.965999999999999165e-03, 2.887800000000000089e-02, 2.873729999999999896e-01, 6.464000000000000128e-03, 5.472000000000000301e-03, 8.494500000000000661e-02],[5.316799999999999998e-02, 2.729646999999999935e+00, 4.266840000000000077e-01, 1.670409999999999950e-01, 3.354000000000000210e-03, 1.013424000000000103e+00, 9.520699999999999996e-02, 1.579999999999999916e-04, 7.179999999999999967e-04, 2.005333999999999950e+00, 8.985999999999999197e-03, 3.879999999999999985e-04, 0.000000000000000000e+00, 3.244699999999999668e-02, 1.137900000000000023e-02, 0.000000000000000000e+00, 1.213889999999999969e-01, 9.757270000000000110e-01, 9.283190000000000053e-01, 1.009280000000000038e-01, 3.671999999999999917e-03, 9.761750000000000149e-01, 6.442350000000000021e-01, 9.600000000000000243e-05, 9.981000000000000288e-03, 9.155050000000000132e-01, 1.505269999999999941e-01, 2.359999999999999902e-04, 1.358999999999999965e-03, 1.352549999999999863e-01, 1.568599999999999856e-02, 6.600000000000000506e-05, 8.929030000000000022e-01, 0.000000000000000000e+00, 3.195698000000000150e+00, 4.214693999999999718e+00, 2.791199999999999931e-02, 8.025160000000000071e-01, 7.025940000000000518e-01, 1.861700000000000160e-02, 6.743100000000000482e-02, 1.228004000000000095e+00, 4.035210000000000186e-01, 3.511999999999999844e-02, 6.775999999999999905e-03, 1.190620000000000012e-01, 4.127999999999999725e-02, 4.100999999999999611e-03, 1.847040000000000071e-01, 1.003700000000000078e-02, 7.427999999999999707e-03, 1.023141999999999996e+00, 6.701080000000000370e-01, 2.539999999999999940e-04, 7.161199999999999510e-02, 6.638399999999999856e-02, 0.000000000000000000e+00, 6.200000000000000270e-05, 5.536599999999999855e-02, 6.807999999999999816e-03, 0.000000000000000000e+00],[1.592480000000000007e-01, 3.727190000000000225e-01, 1.563846000000000069e+00, 1.495779999999999887e-01, 8.379999999999999862e-04, 3.601960000000000162e-01, 1.447392000000000012e+00, 3.358099999999999974e-02, 4.099999999999999912e-04, 8.226430000000000131e-01, 1.195789999999999909e+00, 1.290979999999999905e-01, 7.799999999999999859e-05, 3.198399999999999854e-02, 2.624650000000000039e-01, 3.311999999999999840e-03, 3.433140000000000081e-01, 5.660340000000000371e-01, 3.036767000000000216e+00, 1.592169999999999974e-01, 1.791000000000000057e-03, 2.942940000000000000e-01, 9.903300000000000436e-01, 3.443700000000000233e-02, 1.039999999999999909e-03, 5.286920000000000508e-01, 2.171984000000000137e+00, 7.891000000000000444e-03, 1.359999999999999989e-04, 1.006100000000000050e-02, 1.763000000000000123e-01, 4.219999999999999793e-03, 2.984713999999999867e+00, 3.195698000000000150e+00, 0.000000000000000000e+00, 1.091223999999999972e+00, 1.576049999999999951e-01, 9.385860000000000314e-01, 1.251589000000000063e+00, 1.062570000000000042e-01, 5.374799999999999717e-02, 2.397249999999999936e-01, 9.681459999999999511e-01, 3.022299999999999986e-02, 5.301700000000000162e-02, 2.450189999999999868e-01, 3.623279999999999834e-01, 4.984199999999999742e-02, 1.364319999999999977e-01, 4.281499999999999889e-02, 1.061000000000000094e-03, 2.834320000000000173e-01, 1.029128000000000043e+00, 4.238000000000000100e-02, 1.321600000000000032e-02, 1.376340000000000063e-01, 1.443900000000000045e-02, 0.000000000000000000e+00, 3.442800000000000027e-02, 1.599550000000000138e-01, 1.097399999999999938e-02],[6.048590000000000355e-01, 3.716849999999999876e-01, 1.880969999999999864e-01, 3.529593000000000202e+00, 9.298800000000000121e-02, 5.590000000000000351e-04, 4.259500000000000092e-02, 9.289690000000000447e-01, 3.047400000000000109e-02, 1.248100000000000077e-02, 1.576299999999999923e-02, 1.702817000000000025e+00, 1.170999999999999949e-03, 4.199999999999999768e-05, 5.906499999999999945e-02, 6.696599999999999775e-02, 7.067820000000000213e-01, 4.673900000000000277e-02, 1.308729999999999893e-01, 7.467920000000000114e-01, 1.478730000000000044e-01, 7.690000000000000438e-04, 3.155999999999999778e-02, 6.617760000000000309e-01, 7.069999999999999896e-03, 0.000000000000000000e+00, 3.680000000000000003e-04, 1.625250000000000028e-01, 1.236600000000000026e-02, 0.000000000000000000e+00, 4.910000000000000085e-04, 1.294919999999999960e-01, 2.738551999999999875e+00, 4.214693999999999718e+00, 1.091223999999999972e+00, 0.000000000000000000e+00, 1.890900000000000081e-01, 1.514000000000000068e-02, 4.319800000000000029e-02, 3.965210000000000123e-01, 3.750970000000000137e-01, 3.746100000000000124e-02, 2.357470000000000121e-01, 6.292429999999999968e-01, 7.838100000000000622e-02, 7.112000000000000266e-03, 1.149099999999999948e-02, 1.534179999999999988e-01, 1.545200000000000046e-02, 2.237629999999999897e-01, 3.484479999999999800e-01, 3.716999999999999818e-03, 1.175329999999999986e-01, 1.120532000000000084e+00, 1.350000000000000017e-04, 2.824899999999999980e-02, 9.998500000000000443e-02, 4.211199999999999666e-02, 0.000000000000000000e+00, 3.855700000000000099e-02, 6.439699999999999591e-02],[9.236440000000000206e-01, 4.143499999999999961e-02, 4.381999999999999611e-03, 8.462000000000000077e-02, 2.735831000000000124e+00, 7.734099999999999309e-02, 3.185740000000000238e-01, 3.817960000000000242e-01, 4.617290000000000005e-01, 6.335300000000000653e-02, 4.420000000000000318e-03, 1.263819999999999943e-01, 9.839259999999999673e-01, 5.890000000000000053e-04, 5.178960000000000230e-01, 2.262000000000000122e-03, 1.274118000000000084e+00, 4.982000000000000317e-03, 2.219000000000000052e-03, 3.926500000000000129e-02, 1.236456999999999917e+00, 6.975399999999999656e-02, 8.738400000000000334e-02, 3.359240000000000004e-01, 2.865469999999999962e-01, 6.799999999999999946e-05, 1.230000000000000082e-04, 5.890000000000000053e-04, 7.411349999999999882e-01, 3.570000000000000040e-04, 3.510000000000000208e-04, 2.373999999999999808e-03, 9.054879999999999596e-01, 2.791199999999999931e-02, 1.576049999999999951e-01, 1.890900000000000081e-01, 0.000000000000000000e+00, 6.746559999999999668e+00, 1.561123999999999956e+00, 4.400553000000000381e+00, 1.693997999999999893e+00, 2.797300000000000134e-02, 6.720770000000000355e-01, 5.850710000000000077e-01, 3.415722000000000147e+00, 9.240500000000000103e-02, 4.157179999999999764e-01, 7.187189999999999968e-01, 3.473699999999999705e-02, 1.768289999999999862e-01, 4.380678999999999768e+00, 5.584999999999999964e-01, 5.056770000000000431e-01, 1.931349999999999900e+00, 2.973709999999999964e-01, 3.146600000000000091e-02, 8.065539999999999932e-01, 1.041312000000000015e+00, 1.653500000000000109e-02, 1.940599999999999964e-02, 1.099750000000000033e-01],[5.499999999999999681e-03, 5.659680000000000266e-01, 3.739480000000000026e-01, 4.500000000000000283e-05, 1.966499999999999845e-02, 2.159327999999999914e+00, 4.985040000000000027e-01, 3.855900000000000299e-02, 0.000000000000000000e+00, 2.214735000000000120e+00, 1.768000000000000040e-03, 4.850000000000000253e-04, 0.000000000000000000e+00, 5.363620000000000054e-01, 1.692639999999999978e-01, 4.000000000000000327e-05, 1.881400000000000086e-02, 6.960159999999999680e-01, 6.349270000000000191e-01, 5.619999999999999996e-04, 1.307700000000000008e-02, 1.394892000000000021e+00, 9.711480000000000112e-01, 2.206199999999999828e-02, 4.159999999999999744e-04, 1.610247999999999902e+00, 2.047099999999999961e-02, 2.359999999999999902e-04, 1.359999999999999989e-04, 1.191513999999999962e+00, 1.037659999999999971e-01, 1.320000000000000101e-04, 1.405460000000000043e-01, 8.025160000000000071e-01, 9.385860000000000314e-01, 1.514000000000000068e-02, 6.746559999999999668e+00, 0.000000000000000000e+00, 9.373136999999999830e+00, 9.612709000000000614e+00, 3.602199999999999847e-02, 1.797671000000000019e+00, 4.965519999999999934e-01, 8.692300000000000026e-02, 2.856900000000000064e-02, 2.620833000000000190e+00, 6.825790000000000468e-01, 4.658299999999999941e-02, 3.807880000000000154e-01, 5.800000000000000034e-05, 8.421800000000000119e-02, 3.452570000000000139e+00, 1.978447999999999984e+00, 1.119679999999999981e-01, 1.514097000000000026e+00, 4.417800000000000199e-02, 4.659999999999999971e-04, 0.000000000000000000e+00, 5.122029999999999639e-01, 1.654000000000000088e-03, 9.299999999999999727e-05],[4.820000000000000066e-04, 6.850100000000000633e-02, 7.591320000000000290e-01, 3.771000000000000046e-03, 4.984800000000000342e-02, 7.288909999999999556e-01, 2.866324999999999790e+00, 2.386200000000000127e-02, 0.000000000000000000e+00, 2.002049999999999941e-01, 1.878319999999999990e-01, 6.401999999999999705e-03, 7.799999999999999859e-05, 4.629000000000000302e-03, 8.373019999999999907e-01, 8.100000000000000375e-05, 1.165000000000000020e-03, 1.900370000000000115e-01, 1.544626000000000055e+00, 8.012400000000000078e-02, 1.782399999999999957e-02, 4.175650000000000195e-01, 1.950083000000000011e+00, 1.410100000000000069e-02, 6.239999999999999887e-04, 8.374399999999999900e-02, 3.409177999999999820e+00, 1.531000000000000026e-03, 0.000000000000000000e+00, 8.930000000000000221e-04, 1.524024000000000045e+00, 6.600000000000000506e-05, 2.275270000000000070e-01, 7.025940000000000518e-01, 1.251589000000000063e+00, 4.319800000000000029e-02, 1.561123999999999956e+00, 9.373136999999999830e+00, 0.000000000000000000e+00, 5.140068000000000303e+00, 4.658799999999999747e-02, 1.398079000000000072e+00, 1.526140999999999970e+00, 8.355200000000000127e-02, 8.186100000000000321e-02, 9.683509999999999618e-01, 2.211488000000000120e+00, 5.404899999999999982e-02, 1.061870000000000036e-01, 2.703500000000000000e-02, 1.505890000000000006e-01, 1.262618000000000018e+00, 4.658653000000000155e+00, 1.033229999999999982e-01, 5.861110000000000486e-01, 4.460149999999999948e-01, 2.856699999999999864e-02, 0.000000000000000000e+00, 7.485100000000000087e-02, 2.209079999999999933e-01, 5.884000000000000341e-03],[2.162439999999999918e-01, 1.329000000000000103e-03, 2.686620000000000119e-01, 6.431599999999999540e-01, 3.241620000000000057e-01, 1.201210000000000055e-01, 1.085010000000000002e-01, 2.359361999999999959e+00, 1.385199999999999973e-02, 5.328000000000000271e-03, 4.537399999999999767e-02, 2.323090999999999795e+00, 1.794999999999999938e-03, 2.530000000000000239e-04, 4.511339999999999795e-01, 5.363320000000000309e-01, 2.729500000000000259e-01, 9.100000000000000287e-05, 4.090699999999999892e-02, 4.205800000000000094e-01, 3.469780000000000086e-01, 7.302100000000000257e-02, 1.352599999999999983e-02, 1.272426000000000057e+00, 1.871000000000000050e-03, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.242269999999999874e-01, 4.076999999999999895e-03, 4.169999999999999987e-04, 7.720000000000000083e-04, 8.797409999999999952e-01, 5.023549999999999960e-01, 1.861700000000000160e-02, 1.062570000000000042e-01, 3.965210000000000123e-01, 4.400553000000000381e+00, 9.612709000000000614e+00, 5.140068000000000303e+00, 0.000000000000000000e+00, 6.585410000000000430e-01, 2.221400000000000111e-02, 2.389070000000000082e-01, 1.572807999999999984e+00, 6.625780000000000003e-01, 1.319289999999999907e-01, 1.522149999999999892e-01, 2.600427999999999962e+00, 4.189999999999999931e-04, 3.703089999999999993e-01, 1.825877999999999890e+00, 7.717709999999999848e-01, 2.505450000000000177e-01, 3.994416999999999884e+00, 1.173300000000000037e-02, 3.586000000000000298e-02, 1.899864999999999915e+00, 8.892000000000000598e-03, 2.339999999999999958e-04, 3.563000000000000195e-02, 4.835630000000000206e-01],[7.139609999999999568e-01, 2.803999999999999826e-01, 1.760000000000000063e-03, 9.098610000000000309e-01, 4.667700000000000182e-01, 1.119999999999999983e-04, 1.869999999999999917e-04, 2.448999999999999788e-03, 9.384390000000000231e-01, 2.987380000000000035e-01, 1.655870000000000120e-01, 1.179052999999999907e+00, 1.943189999999999917e-01, 8.399999999999999535e-05, 3.772400000000000059e-02, 2.829999999999999942e-04, 5.514930000000000110e-01, 4.982000000000000317e-03, 2.889999999999999774e-04, 9.755400000000000182e-02, 5.562040000000000317e-01, 3.840000000000000097e-04, 1.640000000000000019e-04, 3.549000000000000071e-03, 2.472449999999999926e-01, 7.529999999999999801e-04, 1.041899999999999944e-02, 1.531000000000000069e-02, 1.657839999999999869e-01, 6.000000000000000152e-05, 6.999999999999999387e-05, 3.299999999999999982e-04, 6.391250000000000542e-01, 6.743100000000000482e-02, 5.374799999999999717e-02, 3.750970000000000137e-01, 1.693997999999999893e+00, 3.602199999999999847e-02, 4.658799999999999747e-02, 6.585410000000000430e-01, 0.000000000000000000e+00, 5.214688999999999908e+00, 5.640859999999999985e+00, 1.171294999999999975e+00, 3.246959999999999846e-01, 9.773000000000000437e-03, 1.683899999999999980e-02, 8.033700000000000563e-02, 2.982599999999999835e-02, 8.232299999999999340e-02, 1.720919000000000088e+00, 1.951000000000000043e-03, 1.719999999999999958e-03, 1.212099999999999983e-02, 6.149700000000000305e-02, 7.368199999999999750e-02, 1.976410000000000111e-01, 2.011120000000000130e-01, 1.831500000000000142e-02, 1.082000000000000062e-03, 1.139909999999999951e-01],[2.863000000000000093e-03, 1.110726000000000102e+00, 1.762239999999999918e-01, 1.121999999999999950e-03, 0.000000000000000000e+00, 5.189030000000000031e-01, 6.459000000000000331e-03, 0.000000000000000000e+00, 8.210000000000000067e-04, 2.392605999999999788e+00, 1.222700000000000002e-02, 5.402499999999999664e-02, 0.000000000000000000e+00, 5.542500000000000204e-02, 1.860300000000000148e-02, 2.829999999999999942e-04, 1.619299999999999906e-02, 5.667590000000000128e-01, 3.362769999999999926e-01, 8.429999999999999993e-04, 9.000000000000000567e-05, 6.369160000000000377e-01, 3.957709999999999839e-01, 9.600000000000000243e-05, 8.942000000000000295e-03, 6.760490000000000110e-01, 7.037280000000000202e-01, 2.143499999999999919e-02, 0.000000000000000000e+00, 2.405049999999999966e-01, 4.726799999999999752e-02, 0.000000000000000000e+00, 3.058500000000000107e-02, 1.228004000000000095e+00, 2.397249999999999936e-01, 3.746100000000000124e-02, 2.797300000000000134e-02, 1.797671000000000019e+00, 1.398079000000000072e+00, 2.221400000000000111e-02, 5.214688999999999908e+00, 0.000000000000000000e+00, 8.675226999999999578e+00, 3.935950000000000060e+00, 1.557000000000000007e-03, 2.014769999999999894e-01, 5.104800000000000310e-02, 2.050000000000000173e-03, 1.273880000000000012e-01, 5.800000000000000034e-05, 2.889999999999999774e-04, 8.800320000000000364e-01, 3.560380000000000211e-01, 8.500000000000000611e-05, 3.093739999999999823e-01, 1.053049999999999958e-01, 7.608000000000000179e-03, 0.000000000000000000e+00, 1.728889999999999871e-01, 1.910000000000000077e-04, 0.000000000000000000e+00],[1.157999999999999958e-03, 2.938839999999999786e-01, 7.439999999999999947e-01, 1.949639999999999984e-01, 6.099999999999999737e-04, 4.470000000000000232e-04, 7.164409999999999945e-01, 5.529999999999999976e-04, 6.956700000000000383e-02, 1.149845999999999924e+00, 1.558783999999999947e+00, 6.849680000000000213e-01, 7.799999999999999859e-05, 2.950000000000000148e-04, 4.101989999999999803e-01, 8.100000000000000375e-05, 3.029000000000000008e-03, 5.371399999999999786e-02, 6.342029999999999612e-01, 8.405999999999999583e-02, 1.253999999999999906e-03, 1.441000000000000007e-03, 7.030920000000000503e-01, 1.918000000000000000e-03, 1.434809999999999974e-01, 9.716500000000000137e-02, 5.908330000000000526e-01, 4.322199999999999653e-02, 0.000000000000000000e+00, 1.190000000000000058e-04, 1.850380000000000080e-01, 6.600000000000000506e-05, 1.369370000000000032e-01, 4.035210000000000186e-01, 9.681459999999999511e-01, 2.357470000000000121e-01, 6.720770000000000355e-01, 4.965519999999999934e-01, 1.526140999999999970e+00, 2.389070000000000082e-01, 5.640859999999999985e+00, 8.675226999999999578e+00, 0.000000000000000000e+00, 3.981616999999999962e+00, 7.453600000000000503e-02, 6.422700000000000631e-02, 2.762760000000000216e-01, 3.217699999999999727e-02, 1.431400000000000033e-02, 7.948000000000000204e-03, 3.376000000000000095e-03, 2.974000000000000081e-03, 2.163174999999999848e+00, 2.797000000000000007e-03, 1.839520000000000044e-01, 3.816709999999999825e-01, 1.058840000000000059e-01, 0.000000000000000000e+00, 3.203899999999999804e-02, 1.758609999999999896e-01, 1.877300000000000149e-02],[4.873000000000000248e-02, 5.904999999999999659e-03, 2.164900000000000158e-02, 1.322233999999999909e+00, 5.339999999999999695e-04, 1.119999999999999983e-04, 3.739999999999999835e-04, 3.803739999999999899e-01, 3.765700000000000297e-02, 3.260350000000000192e-01, 4.743700000000000000e-02, 2.382451000000000096e+00, 1.559999999999999972e-04, 0.000000000000000000e+00, 2.747899999999999995e-02, 6.999500000000000166e-02, 4.281200000000000283e-02, 9.100000000000000287e-05, 3.376999999999999794e-03, 3.881560000000000010e-01, 2.507999999999999813e-03, 2.880000000000000073e-04, 1.312000000000000015e-03, 4.188020000000000076e-01, 4.200500000000000067e-02, 4.656000000000000416e-03, 1.176800000000000068e-02, 2.411969999999999947e-01, 2.719999999999999979e-04, 6.000000000000000152e-05, 1.749999999999999982e-04, 1.697770000000000112e-01, 8.914800000000000502e-02, 3.511999999999999844e-02, 3.022299999999999986e-02, 6.292429999999999968e-01, 5.850710000000000077e-01, 8.692300000000000026e-02, 8.355200000000000127e-02, 1.572807999999999984e+00, 1.171294999999999975e+00, 3.935950000000000060e+00, 3.981616999999999962e+00, 0.000000000000000000e+00, 7.252100000000000213e-02, 1.677599999999999925e-02, 2.488299999999999887e-02, 1.641430000000000111e-01, 1.200000000000000030e-04, 1.309400000000000008e-01, 1.167299999999999935e-02, 1.951000000000000043e-03, 7.108999999999999868e-03, 9.761849999999999694e-01, 4.504400000000000071e-02, 3.931300000000000072e-02, 4.208990000000000231e-01, 7.410000000000000137e-04, 1.410000000000000120e-04, 3.880999999999999901e-03, 1.646589999999999998e-01],[7.082039999999999447e-01, 2.361999999999999950e-03, 5.240000000000000517e-04, 8.843999999999999431e-03, 3.879354999999999887e+00, 1.956999999999999972e-03, 2.527999999999999865e-03, 9.008000000000000382e-03, 5.606419999999999737e-01, 6.570000000000000319e-04, 2.062000000000000031e-03, 6.304999999999999841e-03, 1.271061999999999914e+00, 3.137789999999999746e-01, 5.690760000000000263e-01, 2.581653999999999893e+00, 6.069449999999999568e-01, 2.808000000000000166e-03, 1.450000000000000009e-04, 1.958599999999999924e-02, 7.321109999999999562e-01, 1.920000000000000049e-04, 8.200000000000000095e-05, 1.150999999999999896e-03, 2.551470000000000127e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.530000000000000152e-04, 2.047159000000000173e+00, 3.173000000000000126e-02, 2.052899999999999864e-02, 5.095950000000000202e-01, 4.692810000000000037e-01, 6.775999999999999905e-03, 5.301700000000000162e-02, 7.838100000000000622e-02, 3.415722000000000147e+00, 2.856900000000000064e-02, 8.186100000000000321e-02, 6.625780000000000003e-01, 3.246959999999999846e-01, 1.557000000000000007e-03, 7.453600000000000503e-02, 7.252100000000000213e-02, 0.000000000000000000e+00, 5.637508999999999659e+00, 3.945647000000000126e+00, 5.313949000000000034e+00, 6.042999999999999761e-02, 2.361790000000000000e-01, 8.137199999999999989e-01, 2.790000000000000054e-04, 1.150000000000000035e-04, 1.356000000000000000e-03, 8.698599999999999388e-02, 5.861600000000000144e-02, 7.542330000000000423e-01, 2.299542999999999893e+00, 4.740300000000000069e-02, 7.624250000000000194e-01, 5.655660000000000132e-01],[1.930000000000000075e-03, 2.154919999999999891e-01, 5.614500000000000046e-02, 0.000000000000000000e+00, 2.514999999999999875e-03, 2.862216000000000093e+00, 1.029799999999999986e-02, 1.026999999999999918e-03, 2.049999999999999956e-04, 3.812869999999999870e-01, 2.062000000000000031e-03, 9.699999999999999963e-05, 1.542070000000000107e-01, 7.487815999999999583e+00, 3.504729999999999790e-01, 4.586219999999999741e-01, 6.756999999999999985e-03, 4.910340000000000260e-01, 6.932000000000000661e-02, 3.750000000000000078e-04, 5.370000000000000424e-04, 7.600900000000000434e-01, 3.164200000000000346e-02, 1.920000000000000049e-04, 1.871000000000000050e-03, 6.485229999999999606e-01, 5.026000000000000086e-03, 5.890000000000000053e-04, 4.375600000000000322e-02, 2.720152999999999821e+00, 1.085169999999999968e+00, 1.053609999999999963e-01, 4.256499999999999867e-02, 1.190620000000000012e-01, 2.450189999999999868e-01, 7.112000000000000266e-03, 9.240500000000000103e-02, 2.620833000000000190e+00, 9.683509999999999618e-01, 1.319289999999999907e-01, 9.773000000000000437e-03, 2.014769999999999894e-01, 6.422700000000000631e-02, 1.677599999999999925e-02, 5.637508999999999659e+00, 0.000000000000000000e+00, 8.744445000000000690e+00, 9.569170000000000176e-01, 5.828010000000000135e-01, 1.160000000000000007e-04, 9.600000000000000243e-05, 3.968969999999999998e-01, 1.260999999999999968e-03, 8.500000000000000611e-05, 1.273908000000000040e+00, 9.204400000000000082e-02, 1.087000000000000076e-03, 5.989999999999999665e-03, 9.461359999999999770e-01, 1.408669999999999922e-01, 2.633800000000000030e-02],[1.930000000000000021e-04, 1.358200000000000032e-02, 3.118850000000000233e-01, 6.279999999999999775e-04, 1.143300000000000045e-02, 8.964300000000000046e-02, 3.092499999999999805e+00, 3.635000000000000210e-03, 1.029999999999999964e-04, 4.159999999999999636e-03, 3.143790000000000195e-01, 1.358000000000000049e-03, 9.111930000000000307e-01, 3.487341999999999942e+00, 1.399672999999999945e+00, 9.533530000000000060e-01, 4.659999999999999971e-04, 4.411199999999999843e-02, 4.754959999999999742e-01, 3.223700000000000176e-02, 7.169999999999999724e-04, 1.536999999999999955e-03, 6.024279999999999635e-01, 3.840000000000000097e-04, 8.319999999999999488e-04, 3.286999999999999991e-03, 7.013989999999999947e-01, 1.412999999999999977e-03, 1.731219999999999981e-01, 4.774390000000000023e-01, 2.262490000000000112e+00, 9.184399999999999509e-02, 5.502499999999999752e-02, 4.127999999999999725e-02, 3.623279999999999834e-01, 1.149099999999999948e-02, 4.157179999999999764e-01, 6.825790000000000468e-01, 2.211488000000000120e+00, 1.522149999999999892e-01, 1.683899999999999980e-02, 5.104800000000000310e-02, 2.762760000000000216e-01, 2.488299999999999887e-02, 3.945647000000000126e+00, 8.744445000000000690e+00, 0.000000000000000000e+00, 8.832620999999999611e+00, 1.163089999999999957e-01, 9.629999999999999671e-03, 7.720000000000000083e-04, 1.394000000000000057e-03, 4.518989999999999951e-01, 2.539999999999999940e-04, 5.302780000000000271e-01, 2.770719999999999850e-01, 1.268440000000000123e-01, 2.469899999999999873e-02, 3.710720000000000129e-01, 1.108802000000000065e+00, 6.892700000000000216e-02],[2.014799999999999924e-01, 2.950000000000000148e-04, 7.213799999999999379e-02, 3.185950000000000171e-01, 5.625000000000000139e-02, 3.020000000000000115e-03, 4.867999999999999931e-03, 2.898791000000000118e+00, 3.211599999999999872e-02, 2.190000000000000106e-04, 5.170899999999999802e-02, 5.014560000000000128e-01, 2.577577999999999925e+00, 5.759210000000000163e-01, 4.301240000000000063e-01, 6.546162999999999954e+00, 1.508039999999999936e-01, 9.100000000000000287e-05, 9.599999999999999159e-03, 3.382069999999999799e-01, 9.225000000000000561e-03, 1.920000000000000049e-04, 2.460000000000000164e-04, 7.083949999999999969e-01, 3.742999999999999799e-03, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.368280000000000163e-01, 5.551040000000000418e-01, 7.286600000000000021e-02, 1.158000000000000002e-02, 1.897149999999999892e+00, 2.764989999999999948e-01, 4.100999999999999611e-03, 4.984199999999999742e-02, 1.534179999999999988e-01, 7.187189999999999968e-01, 4.658299999999999941e-02, 5.404899999999999982e-02, 2.600427999999999962e+00, 8.033700000000000563e-02, 2.050000000000000173e-03, 3.217699999999999727e-02, 1.641430000000000111e-01, 5.313949000000000034e+00, 9.569170000000000176e-01, 8.832620999999999611e+00, 0.000000000000000000e+00, 3.589999999999999984e-04, 5.715050000000000407e-01, 2.219000000000000052e-03, 1.859999999999999945e-04, 1.150000000000000035e-04, 5.639950000000000241e-01, 4.316000000000000392e-03, 3.876299999999999885e-02, 1.598824000000000023e+00, 7.481959999999999722e-01, 6.590500000000000524e-02, 5.296189999999999509e-01, 8.639700000000000157e-01],[4.294200000000000100e-02, 4.878140000000000254e-01, 1.408639999999999892e-01, 4.276200000000000140e-02, 5.336300000000000071e-02, 3.935449999999999782e-01, 1.777569999999999983e-01, 5.990000000000000316e-04, 6.090899999999999787e-02, 3.468899999999999761e-01, 2.695100000000000273e-01, 1.800000000000000113e-04, 2.545400000000000093e-02, 2.559750000000000081e-01, 3.003540000000000099e-01, 1.800000000000000113e-04, 7.923600000000000088e-02, 3.193995999999999835e+00, 3.323960000000000248e-01, 3.405409999999999826e-01, 2.587300000000000016e-02, 3.104160000000000252e-01, 3.737200000000000244e-02, 4.789999999999999879e-04, 5.168600000000000277e-02, 6.045409999999999950e-01, 8.193100000000000382e-02, 1.376999999999999969e-03, 1.197800000000000080e-02, 6.220890000000000031e-01, 3.088589999999999947e-01, 2.990000000000000036e-04, 4.372100000000000292e-02, 1.847040000000000071e-01, 1.364319999999999977e-01, 1.545200000000000046e-02, 3.473699999999999705e-02, 3.807880000000000154e-01, 1.061870000000000036e-01, 4.189999999999999931e-04, 2.982599999999999835e-02, 1.273880000000000012e-01, 1.431400000000000033e-02, 1.200000000000000030e-04, 6.042999999999999761e-02, 5.828010000000000135e-01, 1.163089999999999957e-01, 3.589999999999999984e-04, 0.000000000000000000e+00, 4.177764999999999951e+00, 7.080899999999999694e-02, 5.241160000000000263e-01, 2.139669999999999905e-01, 6.958799999999999708e-02, 6.284380000000000521e-01, 1.421996000000000038e+00, 5.485999999999999904e-02, 3.352900000000000325e-02, 4.483501000000000403e+00, 1.778329999999999911e-01, 4.810420000000000251e-01],[4.671359999999999957e-01, 4.658600000000000241e-02, 7.286700000000000121e-02, 7.478700000000000347e-01, 2.080410000000000037e-01, 1.160000000000000007e-04, 4.211899999999999672e-02, 5.999320000000000208e-01, 4.243210000000000037e-01, 1.160000000000000007e-04, 2.004990000000000105e-01, 5.064699999999999758e-01, 5.730129999999999946e-01, 1.160000000000000007e-04, 2.943680000000000185e-01, 3.527310000000000167e-01, 4.862810000000000188e-01, 2.657660000000000022e-01, 9.206999999999999906e-02, 3.088480999999999810e+00, 3.214000000000000190e-02, 1.160000000000000007e-04, 4.177000000000000157e-03, 3.378789999999999849e-01, 4.391699999999999771e-02, 1.160000000000000007e-04, 3.713000000000000154e-03, 5.343750000000000444e-01, 1.665000000000000160e-02, 1.160000000000000007e-04, 4.814999999999999836e-03, 4.854689999999999839e-01, 2.571220000000000172e-01, 1.003700000000000078e-02, 4.281499999999999889e-02, 2.237629999999999897e-01, 1.768289999999999862e-01, 5.800000000000000034e-05, 2.703500000000000000e-02, 3.703089999999999993e-01, 8.232299999999999340e-02, 5.800000000000000034e-05, 7.948000000000000204e-03, 1.309400000000000008e-01, 2.361790000000000000e-01, 1.160000000000000007e-04, 9.629999999999999671e-03, 5.715050000000000407e-01, 4.177764999999999951e+00, 0.000000000000000000e+00, 3.417910000000000115e-01, 4.711499999999999716e-02, 5.813600000000000018e-02, 6.881690000000000307e-01, 6.834199999999999997e-02, 1.140050999999999926e+00, 8.229990000000000361e-01, 7.985820000000000141e-01, 5.033969999999999834e-01, 3.372789999999999955e-01, 4.317980999999999625e+00],[1.496628000000000069e+00, 1.878259999999999930e-01, 4.438000000000000105e-03, 8.720839999999999703e-01, 5.709933000000000369e+00, 1.501069999999999904e-01, 5.165939999999999976e-01, 2.297807000000000155e+00, 1.094735999999999931e+00, 1.049876999999999949e+00, 9.261000000000000135e-03, 5.964323000000000263e+00, 4.034349999999999881e-01, 9.600000000000000243e-05, 2.460940000000000072e-01, 3.860000000000000041e-04, 2.066955000000000098e+00, 1.447000000000000022e-02, 2.122000000000000188e-03, 1.281110000000000027e-01, 3.121656000000000208e+00, 7.235199999999999965e-02, 6.492399999999999560e-02, 5.444740000000000135e-01, 5.570150000000000379e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.789999999999999791e-04, 4.960459999999999869e-01, 9.600000000000000243e-05, 0.000000000000000000e+00, 2.889999999999999774e-04, 1.265487000000000029e+00, 7.427999999999999707e-03, 1.061000000000000094e-03, 3.484479999999999800e-01, 4.380678999999999768e+00, 8.421800000000000119e-02, 1.505890000000000006e-01, 1.825877999999999890e+00, 1.720919000000000088e+00, 2.889999999999999774e-04, 3.376000000000000095e-03, 1.167299999999999935e-02, 8.137199999999999989e-01, 9.600000000000000243e-05, 7.720000000000000083e-04, 2.219000000000000052e-03, 7.080899999999999694e-02, 3.417910000000000115e-01, 0.000000000000000000e+00, 5.982573000000000363e+00, 7.424222000000000321e+00, 5.921778999999999904e+00, 1.219370000000000037e-01, 1.958330000000000071e-01, 7.519799999999999818e-01, 1.184812999999999894e+00, 5.667699999999999821e-02, 6.118869999999999587e-01, 2.780900000000000039e-01],[2.090900000000000050e-02, 1.118208000000000091e+00, 2.516499999999999848e-01, 9.290000000000000297e-04, 4.097210000000000019e-01, 3.829579999999999984e+00, 1.562591000000000063e+00, 4.873170000000000002e-01, 2.043999999999999810e-03, 8.929759999999999920e-01, 2.871500000000000094e-02, 8.363600000000000201e-02, 0.000000000000000000e+00, 1.159749999999999948e-01, 4.228200000000000014e-02, 0.000000000000000000e+00, 8.038299999999999612e-02, 1.144692000000000043e+00, 4.233819999999999806e-01, 2.137000000000000011e-03, 1.387420000000000042e-01, 2.487136000000000013e+00, 3.710629999999999762e-01, 1.963580000000000048e-01, 4.088999999999999753e-03, 6.858119999999999772e-01, 4.832000000000000357e-03, 1.208000000000000089e-03, 1.859999999999999945e-04, 3.428129999999999789e-01, 1.021999999999999905e-03, 1.859999999999999945e-04, 8.568000000000000616e-02, 1.023141999999999996e+00, 2.834320000000000173e-01, 3.716999999999999818e-03, 5.584999999999999964e-01, 3.452570000000000139e+00, 1.262618000000000018e+00, 7.717709999999999848e-01, 1.951000000000000043e-03, 8.800320000000000364e-01, 2.974000000000000081e-03, 1.951000000000000043e-03, 2.790000000000000054e-04, 3.968969999999999998e-01, 1.394000000000000057e-03, 1.859999999999999945e-04, 5.241160000000000263e-01, 4.711499999999999716e-02, 5.982573000000000363e+00, 0.000000000000000000e+00, 5.359077000000000091e+00, 3.655089999999999950e+00, 1.323765000000000081e+00, 8.131199999999999539e-02, 3.434630000000000183e-01, 1.160899999999999953e-02, 6.560040000000000315e-01, 1.588729999999999865e-01, 3.007400000000000004e-02],[4.589999999999999896e-04, 7.258360000000000367e-01, 7.686070000000000402e-01, 4.070599999999999913e-02, 3.498459999999999903e-01, 2.395833000000000101e+00, 4.919173999999999936e+00, 1.997480000000000089e-01, 1.150000000000000035e-04, 9.818281000000000702e+00, 5.143919999999999604e-01, 6.815750000000000419e-01, 0.000000000000000000e+00, 3.440000000000000132e-04, 5.270049999999999457e-01, 0.000000000000000000e+00, 2.178999999999999947e-03, 1.145510000000000000e-01, 1.519211000000000089e+00, 7.373000000000000387e-02, 1.515890000000000015e-01, 4.325200000000000156e-01, 2.075225999999999793e+00, 7.877599999999999880e-02, 3.210999999999999879e-03, 3.210600000000000259e-02, 1.903570999999999902e+00, 1.719999999999999958e-03, 1.150000000000000035e-04, 3.440000000000000132e-04, 2.669429999999999858e-01, 0.000000000000000000e+00, 2.178999999999999947e-03, 6.701080000000000370e-01, 1.029128000000000043e+00, 1.175329999999999986e-01, 5.056770000000000431e-01, 1.978447999999999984e+00, 4.658653000000000155e+00, 2.505450000000000177e-01, 1.719999999999999958e-03, 3.560380000000000211e-01, 2.163174999999999848e+00, 7.108999999999999868e-03, 1.150000000000000035e-04, 1.260999999999999968e-03, 4.518989999999999951e-01, 1.150000000000000035e-04, 2.139669999999999905e-01, 5.813600000000000018e-02, 7.424222000000000321e+00, 5.359077000000000091e+00, 0.000000000000000000e+00, 1.510964999999999892e+00, 4.931789999999999785e-01, 4.100460000000000216e-01, 1.434469999999999912e-01, 1.018799999999999914e-02, 1.982770000000000088e-01, 6.940910000000000135e-01, 3.413700000000000068e-02],[3.409909999999999886e-01, 3.389999999999999784e-03, 1.564679999999999960e-01, 1.773101999999999956e+00, 2.018482999999999805e+00, 1.969829999999999914e-01, 3.611639999999999850e-01, 4.693944000000000116e+00, 4.789000000000000201e-02, 9.476199999999999901e-02, 6.399399999999999533e-02, 2.169045000000000112e+00, 4.240000000000000062e-04, 0.000000000000000000e+00, 9.349100000000000465e-02, 1.724030000000000007e-01, 5.083079999999999821e-01, 5.929999999999999941e-04, 4.924599999999999811e-02, 1.253945000000000087e+00, 4.875419999999999754e-01, 1.012040000000000023e-01, 2.144400000000000125e-02, 2.846677000000000124e+00, 6.102999999999999918e-03, 0.000000000000000000e+00, 8.500000000000000611e-05, 6.606220000000000425e-01, 6.780000000000000002e-04, 8.500000000000000611e-05, 0.000000000000000000e+00, 2.779289999999999816e-01, 3.304810000000000247e-01, 2.539999999999999940e-04, 4.238000000000000100e-02, 1.120532000000000084e+00, 1.931349999999999900e+00, 1.119679999999999981e-01, 1.033229999999999982e-01, 3.994416999999999884e+00, 1.212099999999999983e-02, 8.500000000000000611e-05, 2.797000000000000007e-03, 9.761849999999999694e-01, 1.356000000000000000e-03, 8.500000000000000611e-05, 2.539999999999999940e-04, 5.639950000000000241e-01, 6.958799999999999708e-02, 6.881690000000000307e-01, 5.921778999999999904e+00, 3.655089999999999950e+00, 1.510964999999999892e+00, 0.000000000000000000e+00, 1.382440000000000058e-01, 6.975800000000000056e-02, 1.811752999999999947e+00, 2.497070000000000123e-01, 5.260200000000000292e-02, 2.749900000000000122e-01, 7.739350000000000396e-01],[4.316000000000000392e-03, 3.254219999999999891e-01, 3.479399999999999854e-02, 1.350000000000000017e-04, 6.122699999999999670e-02, 8.060709999999999820e-01, 1.732980000000000076e-01, 1.482999999999999943e-03, 3.088300000000000073e-02, 1.372357000000000049e+00, 7.053299999999999847e-02, 1.817940000000000111e-01, 1.213799999999999948e-02, 3.096430000000000016e-01, 3.246129999999999849e-01, 4.049999999999999781e-04, 3.223199999999999676e-02, 4.830760000000000054e-01, 1.055969999999999964e-01, 1.482999999999999943e-03, 1.025000000000000043e-02, 2.157789999999999986e-01, 4.584999999999999666e-03, 2.700000000000000035e-04, 1.661499999999999921e-01, 5.169270000000000254e-01, 1.831430000000000002e-01, 7.781499999999999528e-02, 2.157799999999999996e-02, 6.741759999999999975e-01, 2.659480000000000177e-01, 6.740000000000000114e-04, 3.641000000000000139e-03, 7.161199999999999510e-02, 1.321600000000000032e-02, 1.350000000000000017e-04, 2.973709999999999964e-01, 1.514097000000000026e+00, 5.861110000000000486e-01, 1.173300000000000037e-02, 6.149700000000000305e-02, 3.093739999999999823e-01, 1.839520000000000044e-01, 4.504400000000000071e-02, 8.698599999999999388e-02, 1.273908000000000040e+00, 5.302780000000000271e-01, 4.316000000000000392e-03, 6.284380000000000521e-01, 6.834199999999999997e-02, 1.219370000000000037e-01, 1.323765000000000081e+00, 4.931789999999999785e-01, 1.382440000000000058e-01, 0.000000000000000000e+00, 2.101149999999999962e-01, 6.838377999999999624e+00, 4.001199999999999896e-02, 5.616200000000000081e-01, 2.406320000000000126e-01, 4.595099999999999879e-02],[5.100500000000000173e-02, 5.390799999999999759e-02, 1.052260000000000001e-01, 7.485899999999999499e-02, 6.042100000000000248e-02, 3.782200000000000145e-02, 3.626809999999999756e-01, 3.593899999999999872e-02, 1.452450000000000130e-01, 1.671380000000000088e-01, 4.967059999999999809e-01, 1.390460000000000029e-01, 1.700409999999999977e-01, 1.368489999999999984e-01, 3.216419999999999835e-01, 7.831200000000000661e-02, 5.516299999999999676e-02, 3.692730000000000179e-01, 2.474899999999999878e-01, 1.950729999999999964e-01, 7.030799999999999550e-02, 3.256400000000000267e-02, 2.157880000000000076e-01, 6.183299999999999907e-02, 1.465009999999999923e-01, 1.764759999999999940e-01, 1.135909999999999975e+00, 1.969600000000000170e-02, 1.710599999999999968e-02, 1.137009999999999965e-01, 5.048660000000000370e-01, 1.012200000000000079e-02, 9.965999999999999165e-03, 6.638399999999999856e-02, 1.376340000000000063e-01, 2.824899999999999980e-02, 3.146600000000000091e-02, 4.417800000000000199e-02, 4.460149999999999948e-01, 3.586000000000000298e-02, 7.368199999999999750e-02, 1.053049999999999958e-01, 3.816709999999999825e-01, 3.931300000000000072e-02, 5.861600000000000144e-02, 9.204400000000000082e-02, 2.770719999999999850e-01, 3.876299999999999885e-02, 1.421996000000000038e+00, 1.140050999999999926e+00, 1.958330000000000071e-01, 8.131199999999999539e-02, 4.100460000000000216e-01, 6.975800000000000056e-02, 2.101149999999999962e-01, 0.000000000000000000e+00, 2.645560000000000134e-01, 2.710719999999999796e-01, 1.183337000000000083e+00, 6.329470000000000374e-01, 7.868709999999999871e-01],[9.951899999999999635e-02, 9.319999999999999942e-04, 1.148899999999999921e-02, 3.600380000000000247e-01, 4.157750000000000057e-01, 7.759999999999999971e-04, 5.387399999999999828e-02, 1.089585000000000026e+00, 1.443879999999999886e-01, 2.480989999999999862e-01, 1.221860000000000029e-01, 2.007767999999999997e+00, 6.955500000000000571e-02, 3.110000000000000243e-04, 2.980909999999999949e-01, 4.314560000000000062e-01, 1.631740000000000135e-01, 1.550000000000000000e-04, 5.791000000000000314e-02, 6.340649999999999897e-01, 2.795000000000000175e-03, 0.000000000000000000e+00, 1.550000000000000000e-04, 2.397150000000000114e-01, 2.327279999999999904e-01, 1.288599999999999988e-02, 2.390899999999999970e-02, 7.269079999999999986e-01, 2.064900000000000069e-02, 4.659999999999999971e-04, 2.638999999999999853e-03, 5.432399999999999451e-01, 2.887800000000000089e-02, 0.000000000000000000e+00, 1.443900000000000045e-02, 9.998500000000000443e-02, 8.065539999999999932e-01, 4.659999999999999971e-04, 2.856699999999999864e-02, 1.899864999999999915e+00, 1.976410000000000111e-01, 7.608000000000000179e-03, 1.058840000000000059e-01, 4.208990000000000231e-01, 7.542330000000000423e-01, 1.087000000000000076e-03, 1.268440000000000123e-01, 1.598824000000000023e+00, 5.485999999999999904e-02, 8.229990000000000361e-01, 7.519799999999999818e-01, 3.434630000000000183e-01, 1.434469999999999912e-01, 1.811752999999999947e+00, 6.838377999999999624e+00, 2.645560000000000134e-01, 0.000000000000000000e+00, 6.455710000000000059e-01, 7.691199999999999426e-02, 3.959420000000000162e-01, 7.335869999999999891e-01],[6.126599999999999824e-01, 2.840000000000000076e-03, 6.200000000000000270e-05, 2.716279999999999806e-01, 8.715680000000000094e-01, 6.200000000000000270e-05, 6.200000000000000270e-05, 1.247299999999999971e-02, 4.237739999999999840e-01, 4.319999999999999838e-04, 2.469999999999999865e-04, 1.092899999999999948e-02, 5.159074999999999633e+00, 4.211199999999999666e-02, 1.418466000000000005e+00, 2.130663000000000196e+00, 5.764759999999999884e-01, 1.230000000000000082e-04, 0.000000000000000000e+00, 2.655100000000000168e-02, 2.610689999999999955e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.358000000000000049e-03, 1.366469999999999907e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.410000000000000137e-04, 7.219895000000000174e+00, 1.373823000000000016e+00, 1.045943000000000067e+00, 2.035722999999999949e+00, 2.873729999999999896e-01, 6.200000000000000270e-05, 0.000000000000000000e+00, 4.211199999999999666e-02, 1.041312000000000015e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 8.892000000000000598e-03, 2.011120000000000130e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.410000000000000137e-04, 2.299542999999999893e+00, 5.989999999999999665e-03, 2.469899999999999873e-02, 7.481959999999999722e-01, 3.352900000000000325e-02, 7.985820000000000141e-01, 1.184812999999999894e+00, 1.160899999999999953e-02, 1.018799999999999914e-02, 2.497070000000000123e-01, 4.001199999999999896e-02, 2.710719999999999796e-01, 6.455710000000000059e-01, 0.000000000000000000e+00, 1.518579999999999930e-01, 8.541945999999999373e+00, 2.395821999999999896e+00],[5.995999999999999594e-03, 1.436139999999999917e-01, 4.360900000000000193e-02, 9.399999999999999448e-05, 1.053899999999999976e-02, 3.715410000000000101e-01, 3.648900000000000060e-02, 1.410000000000000120e-04, 1.260000000000000009e-02, 1.667059999999999931e-01, 5.677100000000000202e-02, 4.699999999999999724e-05, 6.496799999999999797e-02, 1.006045000000000078e+00, 8.006020000000000358e-01, 5.995600000000000235e-02, 1.662800000000000042e-02, 7.617709999999999759e-01, 1.072179999999999939e-01, 1.312000000000000015e-03, 6.557999999999999594e-03, 2.856349999999999723e-01, 7.916099999999999526e-02, 1.410000000000000120e-04, 1.601999999999999952e-02, 2.008060000000000123e-01, 2.735500000000000084e-02, 9.399999999999999448e-05, 1.985580000000000123e-01, 2.306109999999999882e+00, 1.341143999999999892e+00, 2.076919999999999877e-01, 6.464000000000000128e-03, 5.536599999999999855e-02, 3.442800000000000027e-02, 0.000000000000000000e+00, 1.653500000000000109e-02, 5.122029999999999639e-01, 7.485100000000000087e-02, 2.339999999999999958e-04, 1.831500000000000142e-02, 1.728889999999999871e-01, 3.203899999999999804e-02, 1.410000000000000120e-04, 4.740300000000000069e-02, 9.461359999999999770e-01, 3.710720000000000129e-01, 6.590500000000000524e-02, 4.483501000000000403e+00, 5.033969999999999834e-01, 5.667699999999999821e-02, 6.560040000000000315e-01, 1.982770000000000088e-01, 5.260200000000000292e-02, 5.616200000000000081e-01, 1.183337000000000083e+00, 7.691199999999999426e-02, 1.518579999999999930e-01, 0.000000000000000000e+00, 6.755370000000000541e-01, 6.011530999999999736e+00],[4.963000000000000397e-03, 6.069900000000000295e-02, 3.337800000000000211e-01, 7.787800000000000278e-02, 1.113399999999999980e-02, 2.990000000000000036e-03, 5.868179999999999508e-01, 1.565199999999999925e-02, 1.717999999999999909e-03, 5.790000000000000008e-03, 5.068409999999999860e-01, 9.289000000000000382e-03, 1.065536999999999956e+00, 4.780190000000000272e-01, 3.062806999999999835e+00, 1.292934999999999945e+00, 7.507999999999999917e-03, 2.162999999999999992e-03, 2.507480000000000264e-01, 4.027499999999999830e-02, 5.534999999999999989e-03, 3.820000000000000153e-04, 1.129990000000000022e-01, 3.498999999999999940e-03, 1.526999999999999929e-03, 6.399999999999999710e-05, 1.276960000000000039e-01, 3.053999999999999857e-03, 6.560965999999999632e+00, 5.651602999999999710e+00, 2.455337000000000103e+00, 1.161510999999999960e+00, 5.472000000000000301e-03, 6.807999999999999816e-03, 1.599550000000000138e-01, 3.855700000000000099e-02, 1.940599999999999964e-02, 1.654000000000000088e-03, 2.209079999999999933e-01, 3.563000000000000195e-02, 1.082000000000000062e-03, 1.910000000000000077e-04, 1.758609999999999896e-01, 3.880999999999999901e-03, 7.624250000000000194e-01, 1.408669999999999922e-01, 1.108802000000000065e+00, 5.296189999999999509e-01, 1.778329999999999911e-01, 3.372789999999999955e-01, 6.118869999999999587e-01, 1.588729999999999865e-01, 6.940910000000000135e-01, 2.749900000000000122e-01, 2.406320000000000126e-01, 6.329470000000000374e-01, 3.959420000000000162e-01, 8.541945999999999373e+00, 6.755370000000000541e-01, 0.000000000000000000e+00, 1.204355999999999982e+00],[1.482680000000000109e-01, 9.299999999999999727e-05, 1.312199999999999998e-02, 2.635669999999999957e-01, 7.994800000000000517e-02, 9.299999999999999727e-05, 8.031999999999999210e-03, 4.972929999999999851e-01, 1.405160000000000020e-01, 9.299999999999999727e-05, 8.330999999999999517e-02, 2.255540000000000045e-01, 1.075186999999999893e+00, 5.603799999999999754e-02, 6.793700000000000294e-01, 1.119410999999999934e+00, 1.263199999999999878e-01, 4.670000000000000214e-04, 9.759999999999999579e-03, 6.386960000000000415e-01, 2.521699999999999983e-02, 1.399999999999999877e-04, 4.062999999999999771e-03, 3.243679999999999897e-01, 3.012099999999999847e-02, 0.000000000000000000e+00, 9.340000000000000429e-04, 1.438319999999999876e-01, 1.036473999999999895e+00, 8.368399999999999450e-02, 7.355000000000000426e-02, 2.016256999999999966e+00, 8.494500000000000661e-02, 0.000000000000000000e+00, 1.097399999999999938e-02, 6.439699999999999591e-02, 1.099750000000000033e-01, 9.299999999999999727e-05, 5.884000000000000341e-03, 4.835630000000000206e-01, 1.139909999999999951e-01, 0.000000000000000000e+00, 1.877300000000000149e-02, 1.646589999999999998e-01, 5.655660000000000132e-01, 2.633800000000000030e-02, 6.892700000000000216e-02, 8.639700000000000157e-01, 4.810420000000000251e-01, 4.317980999999999625e+00, 2.780900000000000039e-01, 3.007400000000000004e-02, 3.413700000000000068e-02, 7.739350000000000396e-01, 4.595099999999999879e-02, 7.868709999999999871e-01, 7.335869999999999891e-01, 2.395821999999999896e+00, 6.011530999999999736e+00, 1.204355999999999982e+00, 0.000000000000000000e+00]])
        self.ecmrest_matrix = np.array([[0.000000000000000000e+00, 3.662670000000000092e-01, 7.628770000000000273e-01, 4.929483000000000281e+00, 3.919360000000000177e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 8.423761999999999972e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.647619999999999862e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.404435999999999574e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.811398000000000064e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[3.662670000000000092e-01, 0.000000000000000000e+00, 9.828930000000000167e-01, 3.863647999999999971e+00, 0.000000000000000000e+00, 2.134739999999999860e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.784767999999999688e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.926909999999999790e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.105602000000000196e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.686074000000000073e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.283610000000000084e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[7.628770000000000273e-01, 9.828930000000000167e-01, 0.000000000000000000e+00, 1.010212000000000110e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.869856999999999658e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.000358000000000080e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 9.750739999999999963e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.715692000000000217e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.090641000000000194e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[4.929483000000000281e+00, 3.863647999999999971e+00, 1.010212000000000110e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.715476000000000001e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.623529000000000444e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.010539999999999994e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.004762000000000377e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.091654000000000124e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.245404999999999873e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[3.919360000000000177e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.967654999999999710e+00, 2.678802000000000127e+00, 8.307876000000000261e+00, 7.540550000000000308e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.962349999999999817e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 9.255750000000000366e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.064068999999999932e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.033932000000000073e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 2.134739999999999860e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.967654999999999710e+00, 0.000000000000000000e+00, 8.414968999999999255e+00, 6.035739999999999661e+00, 0.000000000000000000e+00, 1.337943000000000104e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.231468000000000007e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.434549999999999992e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.312254999999999949e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.793594999999999828e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 5.869856999999999658e+00, 0.000000000000000000e+00, 2.678802000000000127e+00, 8.414968999999999255e+00, 0.000000000000000000e+00, 9.578981999999999886e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.577271000000000090e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 8.155284999999999229e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.238450000000000051e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.461548999999999765e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.103903999999999996e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.715476000000000001e+00, 8.307876000000000261e+00, 6.035739999999999661e+00, 9.578981999999999886e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 9.399321999999999733e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.801563999999999943e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.578975999999999935e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.107629000000000197e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 8.229100000000000747e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[8.423761999999999972e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.540550000000000308e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.208999999999999936e-01, 6.292704999999999771e+00, 1.792245000000000088e+00, 1.903175000000000061e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.372237999999999847e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.432409999999999961e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 7.784767999999999688e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.337943000000000104e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.208999999999999936e-01, 0.000000000000000000e+00, 6.011969999999999814e+00, 6.637667999999999680e+00, 0.000000000000000000e+00, 4.377789999999999737e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.033629999999999605e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.285542999999999658e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.047654000000000085e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 1.000358000000000080e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.577271000000000090e+00, 0.000000000000000000e+00, 6.292704999999999771e+00, 6.011969999999999814e+00, 0.000000000000000000e+00, 3.324580999999999786e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.231662000000000035e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.296291000000000082e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.388513999999999804e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 8.250819999999999821e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.623529000000000444e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 9.399321999999999733e+00, 1.792245000000000088e+00, 6.637667999999999680e+00, 3.324580999999999786e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.781180000000000208e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.834665999999999908e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.907799999999999718e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.719489000000000267e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[2.647619999999999862e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.962349999999999817e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.903175000000000061e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.623935999999999602e+00, 2.082127999999999979e+00, 8.298899999999999721e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.134458999999999662e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.294298000000000393e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.165879999999999972e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 2.926909999999999790e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.231468000000000007e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.377789999999999737e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.623935999999999602e+00, 0.000000000000000000e+00, 1.303951000000000082e+00, 1.407279999999999920e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.381840999999999653e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.373369999999999869e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.860430000000000472e-01, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 9.750739999999999963e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 8.155284999999999229e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.231662000000000035e+00, 0.000000000000000000e+00, 2.082127999999999979e+00, 1.303951000000000082e+00, 0.000000000000000000e+00, 1.281784000000000034e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.570123000000000157e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.078444000000000180e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.911095999999999684e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.010539999999999994e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.801563999999999943e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.781180000000000208e-01, 8.298899999999999721e+00, 1.407279999999999920e-01, 1.281784000000000034e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.301224999999999632e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.096281000000000283e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.682028999999999996e+00],[6.404435999999999574e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.918903999999999943e+00, 4.959151000000000309e+00, 3.440380000000000216e+00, 5.381867999999999874e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.415757000000000154e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.419057999999999709e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.485369000000000383e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 4.105602000000000196e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.918903999999999943e+00, 0.000000000000000000e+00, 5.369462999999999653e+00, 8.086159999999999570e+00, 0.000000000000000000e+00, 3.774544000000000121e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.016899000000000441e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.230751000000000150e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.006826999999999916e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.087800999999999796e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 2.715692000000000217e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.959151000000000309e+00, 5.369462999999999653e+00, 0.000000000000000000e+00, 1.086327000000000043e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.794388000000000094e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 9.920977000000000601e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.812540000000000040e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.686782000000000004e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.004762000000000377e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.440380000000000216e+00, 8.086159999999999570e+00, 1.086327000000000043e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.021681999999999979e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.019725999999999910e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.583115999999999968e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.837059999999999915e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.444964000000000137e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 9.255750000000000366e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.381867999999999874e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.053827000000000069e+00, 1.295212999999999948e+00, 5.480963000000000029e+00, 1.693662000000000001e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.393680000000000030e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.923972000000000016e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.312810999999999950e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.434549999999999992e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.774544000000000121e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.053827000000000069e+00, 0.000000000000000000e+00, 8.557938999999999297e+00, 7.461627000000000010e+00, 0.000000000000000000e+00, 1.699302000000000090e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.389734999999999943e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.120188999999999879e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.704600999999999811e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.238450000000000051e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.794388000000000094e+00, 0.000000000000000000e+00, 1.295212999999999948e+00, 8.557938999999999297e+00, 0.000000000000000000e+00, 8.407090999999999426e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.144296000000000202e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.776159999999999850e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.034856000000000442e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.303479999999999972e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.578975999999999935e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.021681999999999979e+00, 5.480963000000000029e+00, 7.461627000000000010e+00, 8.407090999999999426e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.341649999999999898e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.263838000000000017e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.159184999999999910e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.769355000000000011e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.372237999999999847e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.415757000000000154e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.693662000000000001e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.930265999999999593e+00, 3.919751999999999903e+00, 9.459951000000000221e+00, 3.023938999999999933e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 9.104099999999999693e-02, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.033629999999999605e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.016899000000000441e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.699302000000000090e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.930265999999999593e+00, 0.000000000000000000e+00, 9.483729999999999549e+00, 4.603856999999999644e+00, 0.000000000000000000e+00, 2.250770000000000159e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.252469999999999972e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.632945000000000091e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.296291000000000082e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 9.920977000000000601e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.144296000000000202e+00, 0.000000000000000000e+00, 3.919751999999999903e+00, 9.483729999999999549e+00, 0.000000000000000000e+00, 8.047499999999999654e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.462944999999999940e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.803738000000000063e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.026085999999999832e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.834665999999999908e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.019725999999999910e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.341649999999999898e-01, 9.459951000000000221e+00, 4.603856999999999644e+00, 8.047499999999999654e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.795649999999999524e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.114999999999999908e-02, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.617091000000000056e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.134458999999999662e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.419057999999999709e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.393680000000000030e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.023938999999999933e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.811205000000000176e+00, 5.650115999999999694e+00, 6.803875999999999813e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.737489000000000061e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.752638000000000140e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.381840999999999653e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.230751000000000150e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.389734999999999943e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.250770000000000159e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.811205000000000176e+00, 0.000000000000000000e+00, 2.078564000000000078e+00, 8.905483999999999511e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.914971999999999674e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.266373000000000193e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.570123000000000157e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.812540000000000040e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.776159999999999850e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.462944999999999940e+00, 0.000000000000000000e+00, 5.650115999999999694e+00, 2.078564000000000078e+00, 0.000000000000000000e+00, 8.432339000000000695e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.164804999999999868e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 8.772180000000000533e-01, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.301224999999999632e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.583115999999999968e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.263838000000000017e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.795649999999999524e-01, 6.803875999999999813e+00, 8.905483999999999511e+00, 8.432339000000000695e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.473622999999999905e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.090751000000000026e+00],[2.811398000000000064e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.485369000000000383e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.500879999999999992e-01, 1.535700000000000121e-01, 2.710609999999999964e+00, 2.774445000000000050e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 8.739999999999999991e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.277860999999999914e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 3.686074000000000073e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.006826999999999916e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.500879999999999992e-01, 0.000000000000000000e+00, 3.779053000000000218e+00, 6.058989999999999654e-01, 0.000000000000000000e+00, 1.866565000000000030e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.561628999999999934e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.085220000000000129e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.947020000000000195e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 2.090641000000000194e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.686782000000000004e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.535700000000000121e-01, 3.779053000000000218e+00, 0.000000000000000000e+00, 1.296294000000000057e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 9.441919000000000395e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.393809999999999993e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.761050000000000004e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.091654000000000124e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.837059999999999915e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.710609999999999964e+00, 6.058989999999999654e-01, 1.296294000000000057e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.930260000000000309e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.042623999999999995e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.529149999999999787e-01, 0.000000000000000000e+00, 3.675530000000000186e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.064068999999999932e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.923972000000000016e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.774445000000000050e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.090423000000000364e+00, 9.529140999999999195e+00, 1.910225000000000062e+00, 2.985092999999999996e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.291147999999999740e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 9.880382000000000886e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.312254999999999949e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.120188999999999879e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.866565000000000030e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.090423000000000364e+00, 0.000000000000000000e+00, 7.153469999999999551e-01, 6.516319000000000194e+00, 0.000000000000000000e+00, 2.303487000000000062e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.308415000000000106e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.675837999999999717e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.461548999999999765e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.034856000000000442e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 9.441919000000000395e+00, 0.000000000000000000e+00, 9.529140999999999195e+00, 7.153469999999999551e-01, 0.000000000000000000e+00, 5.512585999999999764e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.644970999999999961e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.166553999999999647e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.863158000000000092e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.107629000000000197e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.159184999999999910e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.930260000000000309e-01, 1.910225000000000062e+00, 6.516319000000000194e+00, 5.512585999999999764e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.541379000000000055e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.682091999999999921e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.540052000000000199e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.432409999999999961e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 9.104099999999999693e-02, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 8.739999999999999991e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.985092999999999996e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.120953000000000088e+00, 9.084270999999999319e+00, 5.181000000000000050e-01, 7.024110000000000076e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.285542999999999658e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.252469999999999972e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.561628999999999934e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.303487000000000062e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.120953000000000088e+00, 0.000000000000000000e+00, 8.064826000000000050e+00, 9.488519999999999399e+00, 0.000000000000000000e+00, 5.427170000000000050e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.528509999999999813e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.388513999999999804e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.803738000000000063e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.393809999999999993e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.644970999999999961e+00, 0.000000000000000000e+00, 9.084270999999999319e+00, 8.064826000000000050e+00, 0.000000000000000000e+00, 3.246935999999999822e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.025010000000000199e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 8.108560000000000212e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.907799999999999718e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.114999999999999908e-02, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.042623999999999995e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.541379000000000055e+00, 5.181000000000000050e-01, 9.488519999999999399e+00, 3.246935999999999822e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.033849999999999714e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.104727000000000015e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.294298000000000393e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.737489000000000061e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.277860999999999914e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.291147999999999740e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.024110000000000076e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.531461000000000183e+00, 1.898141000000000078e+00, 6.045078000000000173e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.362719999999999931e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.373369999999999869e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.914971999999999674e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.085220000000000129e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.308415000000000106e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.427170000000000050e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.531461000000000183e+00, 0.000000000000000000e+00, 1.657664000000000026e+00, 6.669954999999999856e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.923391999999999991e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.078444000000000180e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.164804999999999868e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.761050000000000004e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.166553999999999647e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.025010000000000199e-01, 0.000000000000000000e+00, 1.898141000000000078e+00, 1.657664000000000026e+00, 0.000000000000000000e+00, 8.901166999999999163e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.022101000000000148e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.096281000000000283e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.473622999999999905e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.529149999999999787e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.682091999999999921e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.033849999999999714e-01, 6.045078000000000173e+00, 6.669954999999999856e+00, 8.901166999999999163e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.261813000000000073e+00],[0.000000000000000000e+00, 2.283610000000000084e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.087800999999999796e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.947020000000000195e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 9.942818000000000822e+00, 0.000000000000000000e+00, 5.615471999999999575e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.442779999999999896e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.177929999999999922e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.245404999999999873e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.444964000000000137e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.675530000000000186e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 9.942818000000000822e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.206053999999999959e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.820840000000000458e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.610894000000000048e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.033932000000000073e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.312810999999999950e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 9.880382000000000886e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.681751000000000218e+00, 2.991861000000000104e+00, 4.407955000000000290e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.952332000000000178e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.793594999999999828e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.704600999999999811e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.675837999999999717e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.615471999999999575e+00, 0.000000000000000000e+00, 3.681751000000000218e+00, 0.000000000000000000e+00, 5.097020999999999802e+00, 3.889101999999999837e+00, 9.339206000000000785e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.913570999999999911e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.103903999999999996e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.303479999999999972e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.863158000000000092e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.991861000000000104e+00, 5.097020999999999802e+00, 0.000000000000000000e+00, 7.057443000000000133e+00, 0.000000000000000000e+00, 4.662001000000000062e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 8.126913999999999305e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 8.229100000000000747e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.769355000000000011e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.540052000000000199e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.206053999999999959e+00, 4.407955000000000290e+00, 3.889101999999999837e+00, 7.057443000000000133e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.191480999999999568e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.658050999999999942e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.047654000000000085e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.632945000000000091e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.528509999999999813e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.442779999999999896e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 9.339206000000000785e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.326799999999999979e-02, 4.777963999999999878e+00, 0.000000000000000000e+00, 1.558523000000000103e+00, 0.000000000000000000e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 8.250819999999999821e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.026085999999999832e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 8.108560000000000212e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.662001000000000062e+00, 0.000000000000000000e+00, 7.326799999999999979e-02, 0.000000000000000000e+00, 6.771770000000000289e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.369644000000000084e+00, 0.000000000000000000e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 4.719489000000000267e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.617091000000000056e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.104727000000000015e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.820840000000000458e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 6.191480999999999568e+00, 4.777963999999999878e+00, 6.771770000000000289e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.347363999999999784e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.165879999999999972e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.752638000000000140e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.362719999999999931e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.952332000000000178e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.089599999999999944e-02, 4.748755000000000059e+00, 1.315609999999999946e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.860430000000000472e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.266373000000000193e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.923391999999999991e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.177929999999999922e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.913570999999999911e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.558523000000000103e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.089599999999999944e-02, 0.000000000000000000e+00, 4.756287999999999627e+00, 1.192023999999999972e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 7.911095999999999684e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 8.772180000000000533e-01, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.022101000000000148e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 8.126913999999999305e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 5.369644000000000084e+00, 0.000000000000000000e+00, 4.748755000000000059e+00, 4.756287999999999627e+00, 0.000000000000000000e+00, 5.427075999999999567e+00],[0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.682028999999999996e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.090751000000000026e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 2.261813000000000073e+00, 0.000000000000000000e+00, 6.610894000000000048e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 1.658050999999999942e+00, 0.000000000000000000e+00, 0.000000000000000000e+00, 3.347363999999999784e+00, 1.315609999999999946e+00, 1.192023999999999972e+00, 5.427075999999999567e+00, 0.000000000000000000e+00]])
        self.params = { 'alpha': 0.5, 'beta': 2.1, 'k_ti': 3.5, 'k_tv': 0.75 }
        ############################################################################



    def test_ECM_Matrix_initEmpiricalMatrix(self):
        ''' Tests that class initialization properly imported the empirical replacement matrix. '''
        
        self.params["state_freqs"] = self.codonFreqs

        codonMatrix = matrix_builder.ECM_Matrix( "ecmrest", self.params )
        np.testing.assert_array_almost_equal(codonMatrix.emp_matrix, self.ecmrest_matrix, decimal = DECIMAL, err_msg = "ECM_Matrix.initEmpiricalMatrix doesn't return restricted empirical matrix properly.")

        
        codonMatrix = matrix_builder.ECM_Matrix( "ecmunrest", self.params )
        np.testing.assert_array_almost_equal(codonMatrix.emp_matrix, self.ecmunrest_matrix, decimal = DECIMAL, err_msg = "ECM_Matrix.initEmpiricalMatrix doesn't return unrestricted empirical matrix properly.")
        




    def test_ECM_Matrix_set_kappa_param(self):
        ''' Tests _set_kappa_param function. '''
        
        codonMatrix = matrix_builder.ECM_Matrix( "ecmrest", self.params )
        
        self.assertEqual( codonMatrix._set_kappa_param('CT'), 3.5, msg = ("ECM_Matrix._set_kappa_param() doesn't work for single transition."))
        self.assertEqual( codonMatrix._set_kappa_param('CG'), 0.75, msg = ("ECM_Matrix._set_kappa_param() doesn't work for single transversion."))
        self.assertEqual( codonMatrix._set_kappa_param('AGGA'), 3.5**2, msg = ("ECM_Matrix._set_kappa_param() doesn't work for two transitions."))
        self.assertEqual( codonMatrix._set_kappa_param('TGAT'), 0.75**2, msg = ("ECM_Matrix._set_kappa_param() doesn't work for two transversions."))
        self.assertEqual( codonMatrix._set_kappa_param('AGTG'), 3.5*0.75, msg = ("ECM_Matrix._set_kappa_param() doesn't work for single TI, single TV."))
        self.assertEqual( codonMatrix._set_kappa_param('AGTCCA'), 3.5**2 * 0.75, msg = ("ECM_Matrix._set_kappa_param() doesn't work for 2 TI's, 1 TV."))
        self.assertEqual( codonMatrix._set_kappa_param('GTTACT'), 0.75**2 * 3.5, msg = ("ECM_Matrix._set_kappa_param() doesn't work for 1 TI, 2 TV's."))
        self.assertEqual( codonMatrix._set_kappa_param('AGTCCT'), 3.5**3, msg = ("ECM_Matrix._set_kappa_param() doesn't work for 3 TI's."))
        self.assertEqual( codonMatrix._set_kappa_param('ATGCCA'), 0.75**3, msg = ("ECM_Matrix._set_kappa_param() doesn't work for 3 TV's."))


    def test_ECM_Matrix_calc_instantaneous_prob_restricted(self):
        ''' Tests _calc_instantaneous_prob for restricted ECM matrix. '''
        
        self.params["state_freqs"] = self.codonFreqs
        codonMatrix = matrix_builder.ECM_Matrix( "ecmrest", self.params )

        self.assertEqual( codonMatrix._calc_instantaneous_prob(0, 0, codonMatrix.params), 0., msg = ("ECM_Matrix._calc_instantaneous_prob() doesn't work for zero changes, restricted matrix."))
        self.assertEqual( codonMatrix._calc_instantaneous_prob(0, 5, codonMatrix.params), 0., msg = ("ECM_Matrix._calc_instantaneous_prob() doesn't work for two changes, restricted matrix."))
        self.assertEqual( codonMatrix._calc_instantaneous_prob(0, 45, codonMatrix.params), 0., msg = ("ECM_Matrix._calc_instantaneous_prob() doesn't work for three changes, restricted matrix."))

        # AAA -> AAG. synonymous, single TI
        correct = 0.02664918 * 0.762877 * 3.5 * 0.5
        self.assertTrue( abs(codonMatrix._calc_instantaneous_prob(0, 2, codonMatrix.params) - correct) < ZERO, msg = ("ECM_Matrix._calc_instantaneous_prob() doesn't work for synonymous single TI change, restricted matrix."))
        
        # AAA -> AAT. nonsynonymous, single TV
        correct = 0.02999061 * 4.929483 * 0.75 * 2.1
        self.assertTrue( abs(codonMatrix._calc_instantaneous_prob(0, 3, codonMatrix.params) - correct) < ZERO, msg = ("ECM_Matrix._calc_instantaneous_prob() doesn't work for nonsynonymous single TV change, restricted matrix."))
        


    def test_ECM_Matrix_calc_instantaneous_prob_unestricted(self):
        ''' Tests _calc_instantaneous_prob for unrestricted ECM matrix. '''
        
        self.params["state_freqs"] = self.codonFreqs
        codonMatrix = matrix_builder.ECM_Matrix( "ecmunrest", self.params )

        # no change
        self.assertEqual( codonMatrix._calc_instantaneous_prob(0, 0, codonMatrix.params), 0., msg = ("ECM_Matrix._calc_instantaneous_prob() doesn't work for zero changes, unrestricted matrix."))

        # AAA -> AAG. synonymous, single TI
        correct = 0.02664918 * 2.931524 * 3.5 * 0.5
        self.assertTrue( abs(codonMatrix._calc_instantaneous_prob(0, 2, codonMatrix.params) - correct) < ZERO, msg = ("ECM_Matrix._calc_instantaneous_prob() doesn't work for synonymous single TI change, unrestricted matrix."))
        
        # AAA -> AAT. nonsynonymous, single TV
        correct = 0.02999061 * 2.07515399 * 0.75 * 2.1
        self.assertTrue( abs(codonMatrix._calc_instantaneous_prob(0, 3, codonMatrix.params) - correct) < ZERO, msg = ("ECM_Matrix._calc_instantaneous_prob() doesn't work for nonsynonymous single TV change, unrestricted matrix."))
        
        # AAA -> AGG. nonsynonymous, two TI
        correct = 0.03185765 * 1.868194 * 3.5 * 3.5 * 2.1
        self.assertTrue( abs(codonMatrix._calc_instantaneous_prob(0, 10, codonMatrix.params) - correct) < ZERO, msg = ("ECM_Matrix._calc_instantaneous_prob() doesn't work for nonsynonymous 2 TI change, unrestricted matrix."))
        
        # AAA -> ACC. nonsynonymous, two TV
        correct = 0.00700012 * 0.089476 * 0.75 * 0.75 * 2.1
        self.assertTrue( abs(codonMatrix._calc_instantaneous_prob(0, 5, codonMatrix.params) - correct) < ZERO, msg = ("ECM_Matrix._calc_instantaneous_prob() doesn't work for nonsynonymous 2 TV change, unrestricted matrix."))
        
        # AAA -> ACG. nonsynonymous, 1 TI, 1 TV
        correct = 0.01435559 * 0.199589 * 0.75 * 3.5 * 2.1
        self.assertTrue( abs(codonMatrix._calc_instantaneous_prob(0, 6, codonMatrix.params) - correct) < ZERO, msg = ("ECM_Matrix._calc_instantaneous_prob() doesn't work for nonsynonymous 2 TV change, unrestricted matrix."))
        
        # AAA -> TCG. nonsynonymous, 1 TI, 2 TV
        correct = 0.01279963 * 0.000459 * 0.75 * 0.75 * 3.5 * 2.1
        self.assertTrue( abs(codonMatrix._calc_instantaneous_prob(0, 52, codonMatrix.params) - correct) < ZERO, msg = ("ECM_Matrix._calc_instantaneous_prob() doesn't work for nonsynonymous 2 TV change, unrestricted matrix."))








class matrixBuilder_AminoAcid_Matrix_tests(unittest.TestCase):
    ''' 
        Set of unittests for the matrix_builder.AminoAcid_Matrix subclass of matrixBuilder, which deals with empirical amino acid models.
        Functions tested here include initEmpiricalMatrix, _calc_instantaneous_prob.
        We are going to just test everything with the LG matrix.
    '''
    
    def setUp(self):
        ################### DO NOT CHANGE ANY OF THESE EVER. #######################
        self.aminoFreqs = [0.03467759, 0.02767874, 0.05246165, 0.03385361, 0.07774578, 0.06417143, 0.01284807, 0.09590641, 0.04063569, 0.04324441, 0.05862815, 0.0198158, 0.07119442, 0.08725553, 0.0105108, 0.05877746, 0.03735058, 0.04630849, 0.05067367, 0.0762617]
        self.lg_mat = np.array([[0.0, 2.489084, 0.395144, 1.038545, 0.253701, 2.06604, 0.358858, 0.14983, 0.536518, 0.395337, 1.124035, 0.276818, 1.177651, 0.969894, 0.425093, 4.727182, 2.139501, 2.54787, 0.180717, 0.218959], [2.489084, 0.0, 0.062556, 0.003499, 1.105251, 0.569265, 0.640543, 0.320627, 0.013266, 0.594007, 0.89368, 0.528768, 0.075382, 0.084808, 0.534551, 2.784478, 1.14348, 1.959291, 0.670128, 1.165532], [0.395144, 0.062556, 0.0, 5.24387, 0.017416, 0.844926, 0.927114, 0.01069, 0.282959, 0.015076, 0.025548, 5.076149, 0.394456, 0.523386, 0.123954, 1.240275, 0.42586, 0.037967, 0.02989, 0.135107], [1.038545, 0.003499, 5.24387, 0.0, 0.018811, 0.348847, 0.423881, 0.044265, 1.807177, 0.069673, 0.173735, 0.541712, 0.419409, 4.128591, 0.36397, 0.611973, 0.604545, 0.245034, 0.077852, 0.120037], [0.253701, 1.105251, 0.017416, 0.018811, 0.0, 0.089586, 0.682139, 1.112727, 0.023918, 2.592692, 1.798853, 0.089525, 0.094464, 0.035855, 0.052722, 0.361819, 0.165001, 0.654683, 2.457121, 7.803902], [2.06604, 0.569265, 0.844926, 0.348847, 0.089586, 0.0, 0.311484, 0.008705, 0.296636, 0.044261, 0.139538, 1.437645, 0.196961, 0.267959, 0.390192, 1.73999, 0.129836, 0.076701, 0.268491, 0.054679], [0.358858, 0.640543, 0.927114, 0.423881, 0.682139, 0.311484, 0.0, 0.108882, 0.697264, 0.366317, 0.442472, 4.509238, 0.508851, 4.813505, 2.426601, 0.990012, 0.584262, 0.119013, 0.597054, 5.306834], [0.14983, 0.320627, 0.01069, 0.044265, 1.112727, 0.008705, 0.108882, 0.0, 0.159069, 4.145067, 4.273607, 0.191503, 0.078281, 0.072854, 0.126991, 0.064105, 1.033739, 10.649107, 0.11166, 0.232523], [0.536518, 0.013266, 0.282959, 1.807177, 0.023918, 0.296636, 0.697264, 0.159069, 0.0, 0.1375, 0.656604, 2.145078, 0.390322, 3.234294, 6.326067, 0.748683, 1.136863, 0.185202, 0.049906, 0.131932], [0.395337, 0.594007, 0.015076, 0.069673, 2.592692, 0.044261, 0.366317, 4.145067, 0.1375, 0.0, 6.312358, 0.068427, 0.24906, 0.582457, 0.301848, 0.182287, 0.302936, 1.702745, 0.619632, 0.299648], [1.124035, 0.89368, 0.025548, 0.173735, 1.798853, 0.139538, 0.442472, 4.273607, 0.656604, 6.312358, 0.0, 0.371004, 0.099849, 1.672569, 0.484133, 0.34696, 2.020366, 1.898718, 0.696175, 0.481306], [0.276818, 0.528768, 5.076149, 0.541712, 0.089525, 1.437645, 4.509238, 0.191503, 2.145078, 0.068427, 0.371004, 0.0, 0.161787, 1.695752, 0.751878, 4.008358, 2.000679, 0.083688, 0.045376, 0.612025], [1.177651, 0.075382, 0.394456, 0.419409, 0.094464, 0.196961, 0.508851, 0.078281, 0.390322, 0.24906, 0.099849, 0.161787, 0.0, 0.624294, 0.332533, 1.338132, 0.571468, 0.296501, 0.095131, 0.089613], [0.969894, 0.084808, 0.523386, 4.128591, 0.035855, 0.267959, 4.813505, 0.072854, 3.234294, 0.582457, 1.672569, 1.695752, 0.624294, 0.0, 2.807908, 1.223828, 1.080136, 0.210332, 0.236199, 0.257336], [0.425093, 0.534551, 0.123954, 0.36397, 0.052722, 0.390192, 2.426601, 0.126991, 6.326067, 0.301848, 0.484133, 0.751878, 0.332533, 2.807908, 0.0, 0.858151, 0.578987, 0.170887, 0.593607, 0.31444], [4.727182, 2.784478, 1.240275, 0.611973, 0.361819, 1.73999, 0.990012, 0.064105, 0.748683, 0.182287, 0.34696, 4.008358, 1.338132, 1.223828, 0.858151, 0.0, 6.472279, 0.098369, 0.248862, 0.400547], [2.139501, 1.14348, 0.42586, 0.604545, 0.165001, 0.129836, 0.584262, 1.033739, 1.136863, 0.302936, 2.020366, 2.000679, 0.571468, 1.080136, 0.578987, 6.472279, 0.0, 2.188158, 0.140825, 0.245841], [2.54787, 1.959291, 0.037967, 0.245034, 0.654683, 0.076701, 0.119013, 10.649107, 0.185202, 1.702745, 1.898718, 0.083688, 0.296501, 0.210332, 0.170887, 0.098369, 2.188158, 0.0, 0.18951, 0.249313], [0.180717, 0.670128, 0.02989, 0.077852, 2.457121, 0.268491, 0.597054, 0.11166, 0.049906, 0.619632, 0.696175, 0.045376, 0.095131, 0.236199, 0.593607, 0.248862, 0.140825, 0.18951, 0.0, 3.151815], [0.218959, 1.165532, 0.135107, 0.120037, 7.803902, 0.054679, 5.306834, 0.232523, 0.131932, 0.299648, 0.481306, 0.612025, 0.089613, 0.257336, 0.31444, 0.400547, 0.245841, 0.249313, 3.151815, 0.0]])
        
        self.params={'state_freqs':self.aminoFreqs}
        ############################################################################

    
   
    def test_AminoAcid_Matrix_initEmpiricalMatrix(self):
        ''' Tests that class initialization properly imported the empirical replacement matrix. '''
        aaMatrix = matrix_builder.AminoAcid_Matrix("lg", self.params)
        np.testing.assert_array_almost_equal(aaMatrix.emp_matrix, self.lg_mat, decimal = DECIMAL, err_msg = "matrix_builder.AminoAcid_Matrix.initEmpiricalMatrix doesn't return empirical matrix properly.")

        
    def test_AminoAcid_Matrix_calc_instantaneous_prob(self):
        ''' Test _calc_instantaneous_prob function for amino acid class. '''
        aaMatrix = matrix_builder.AminoAcid_Matrix("lg", self.params)
        correctProb = 3.499e-03 * 0.03385361
        self.assertEqual(aaMatrix._calc_instantaneous_prob(1, 3, aaMatrix.params), correctProb, msg = "matrix_builder.AminoAcid_Matrix._calc_instantaneous_prob fail.")











class matrixBuilder_Nucleotide_Matrix_tests(unittest.TestCase):
    ''' 
        Set of unittests for the matrix_builder.Nucleotide_Matrix subclass of matrixBuilder.
        Functions tested here include getNucleotideFreq, _calc_instantaneous_prob.
    '''
    
    def setUp(self):
        ################### DO NOT CHANGE ANY OF THESE EVER. #######################
        muParams = {'AG':0.1, 'CT':0.125, 'AC': 0.08, 'AT':0.05, 'CG':0.125, 'GT':0.13}
        params = {'state_freqs': [0.34, 0.21, 0.27, 0.18], 'mu': muParams}
        self.nucMatrix = matrix_builder.Nucleotide_Matrix("nucleotide", params)
        ############################################################################

   
    def test_Nucleotide_Matrix_calc_instantaneous_prob(self):
        ''' Test function to retrieve instantaneous substitution probability between nucleotides. Just test a few. '''
        correctAT = 0.18 * 0.05
        self.assertEqual(self.nucMatrix._calc_instantaneous_prob(0, 3, self.nucMatrix.params), correctAT, msg = "nucleotideMatrix._calc_instantaneous_prob doesn't properly work for A->T.")
        correctTA = 0.34 * 0.05
        self.assertEqual(self.nucMatrix._calc_instantaneous_prob(3, 0, self.nucMatrix.params), correctTA, msg = "nucleotideMatrix._calc_instantaneous_prob doesn't properly work for T->A.")
        correctCA = 0.34 * 0.08
        self.assertEqual(self.nucMatrix._calc_instantaneous_prob(1, 0, self.nucMatrix.params), correctCA, msg = "nucleotideMatrix._calc_instantaneous_prob doesn't properly work for C->A.")








class matrixBuilder_MutSel_nuc_tests(unittest.TestCase):
    ''' 
        Set of unittests for the matrix_builder.MutSel_Matrix subclass of matrixBuilder, when nucleotides are the unit of evolution.
        Most functionality covered in the matrix_builder.mutSel_codon tests, so this suite is more limited.
    '''
    def setUp(self):
        ################### DO NOT CHANGE ANY OF THESE EVER. #######################
        nucFreqs = np.array( [0.25, 0.19, 0.25, 0.31] )
        self.muParams = {'AG':0.125, 'GA':0.125, 'CT':0.125, 'TC':0.125, 'AC': 0.13, 'CA':0.12, 'AT':0.125, 'TA':0.125, 'CG':0.125, 'GC':0.125, 'GT':0.13, 'TG':0.12}
        params = {'state_freqs': nucFreqs, 'mu': self.muParams, 'calc_by_freqs':True}
        self.mutSelMatrix = matrix_builder.MutSel_Matrix("mutsel", params )
        ############################################################################


    def test_MutSel_Matrix_nuc_setup(self):    
        self.assertTrue(self.mutSelMatrix._size == 4, msg= "size not correctly assigned as 4 for nuc mutsel model.")
        self.assertTrue(self.mutSelMatrix._code == MOLECULES.nucleotides, msg = "nucleotides not correctly assigned as code for nuc mutsel model.")

        
    def test_MutSel_Matrix_nuc_calc_substitution_prob(self):    
        self.assertTrue( abs(self.mutSelMatrix._calc_instantaneous_prob(0, 1, self.mutSelMatrix.params) - 0.108317257) < ZERO, msg = "matrix_builder.MutSel_Matrix._calc_substitution_prob fails when target and source have different frequencies. NUCLEOTIDES.")


    def test_MutSel_Matrix_nuc_obtain_scaling_factor(self):
        '''
            Test that proper scaling factor is obtained
        '''
        truefactor = -0.374866623971
        testfactor = self.mutSelMatrix._obtain_scaling_factor()
        self.assertTrue( abs(truefactor - testfactor) <= ZERO, msg = "Could not compute proper scaling factor for nucleotide mutsel matrix made with state frequencies.")






class matrixBuilder_mutSel_codon_statefreqs_tests(unittest.TestCase):
    ''' 
        Set of unittests for the matrix_builder.MutSel_Matrix subclass of matrixBuilder, when codons are the unit of evolution.
        Specifically, this class tests matrixBuilder when state_freqs are provided for mutsel construction.
    '''
    def setUp(self):
        ################### DO NOT CHANGE ANY OF THESE EVER. #######################
        codonFreqs = np.array( [0, 0.04028377, 0.02664918, 0, 0.00717921, 0.00700012, 0.0231568, 0.0231568, 0.02403056, 0.00737008, 0.03185765, 0.0193576, 0.03277142, 0.02141258, 0.0127537, 0.00298803, 0.0256333, 0.02312437, 0.01861465, 0.01586447, 0.00373147, 0.02662654, 0.00082524, 0.00048916, 0.01191673, 0.00512658, 0.00050502, 0.01688169, 0.01843001, 0.00215437, 0.02659356, 0.02377742, 0.01169375, 0.00097256, 0.02937344, 0.00268204, 0.01414414, 0.02781933, 0.00070877, 0.02370841, 0.02984617, 0.01828081, 0.01002825, 0.00870788, 0.00728006, 0.02179328, 0.00379049, 0.01978996, 0.00443774, 0.01201798, 0.02030269, 0.01238501, 0.01279963, 0.02094385, 0.02810987, 0.00918507, 0.02880549, 0.0029311, 0.0237658, 0.03194712, 0.06148723] )
        self.muParams = {'AG':0.125, 'GA':0.125, 'CT':0.125, 'TC':0.125, 'AC': 0.125, 'CA':0.125, 'AT':0.125, 'TA':0.125, 'CG':0.125, 'GC':0.125, 'GT':0.13, 'TG':0.12}
        params = {'state_freqs': codonFreqs, 'mu': self.muParams, 'calc_by_freqs':True}
        self.mutSelMatrix = matrix_builder.MutSel_Matrix( "mutsel", params )
        ############################################################################

    def test_MutSel_Matrix_codon_setup(self):    
        self.assertTrue(self.mutSelMatrix._size == 61, msg= "size not correctly assigned as 61 for codon mutsel model, state_freqs approach.")
        self.assertTrue(self.mutSelMatrix._code == MOLECULES.codons, msg = "codons not correctly assigned as code for codon mutsel model, state_freqs approach.")


    def test_MutSel_Matrix_codon_obtain_scaling_factor(self):
        '''
            Test that proper scaling factor is obtained
        '''
        truefactor = -1.07768081412
        testfactor = self.mutSelMatrix._obtain_scaling_factor()
        self.assertTrue( abs(truefactor - testfactor) <= ZERO, msg = "Could not compute proper scaling factor for codon mutsel matrix made with state frequencies.")


    def test_MutSel_Matrix_codon_calc_instantaneous_prob(self):    
        ''' Test function _calc_instantaneous_prob for mutation-selection model subclass.
            Test for one or both have freq 0, have equal freq, have no changes, have multiple changes, and finally, have different freq.
        '''

        # Target and/or source have 0 or equal frequency should return 0.
        self.assertTrue( abs(self.mutSelMatrix._calc_instantaneous_prob(0, 3, self.mutSelMatrix.params) - 0.) < ZERO, msg = "matrix_builder.MutSel_Matrix._calc_instantaneous_prob wrong when both codons have 0 frequency, state_freqs approach.")
        self.assertTrue( abs(self.mutSelMatrix._calc_instantaneous_prob(0, 4, self.mutSelMatrix.params) - 0.) < ZERO, msg = "matrix_builder.MutSel_Matrix._calc_instantaneous_prob wrong when source codon only has 0 frequency, state_freqs approach.")
        self.assertTrue( abs(self.mutSelMatrix._calc_instantaneous_prob(7, 3, self.mutSelMatrix.params) - 0.) < ZERO, msg = "matrix_builder.MutSel_Matrix._calc_instantaneous_prob wrong when target codon only has 0 frequency, state_freqs approach.")
        
        # Too few or too many changes
        self.assertTrue( abs(self.mutSelMatrix._calc_instantaneous_prob(6, 6, self.mutSelMatrix.params) - 0.) < ZERO, msg = "matrix_builder.MutSel_Matrix._calc_instantaneous_prob wrong when source and target are the same, state_freqs approach.")
        self.assertTrue( abs(self.mutSelMatrix._calc_instantaneous_prob(6, 53, self.mutSelMatrix.params) - 0.) < ZERO, msg = "matrix_builder.MutSel_Matrix._calc_instantaneous_prob wrong when two changes between codon, state_freqs approachs.")
        self.assertTrue( abs(self.mutSelMatrix._calc_instantaneous_prob(0, 60, self.mutSelMatrix.params) - 0.) < ZERO, msg = "matrix_builder.MutSel_Matrix._calc_instantaneous_prob wrong when three changes between codons, state_freqs approach.")
        
        # Different frequencies.
        self.assertTrue( abs(self.mutSelMatrix._calc_instantaneous_prob(56, 55, self.mutSelMatrix.params) - 0.0673146677) < ZERO, msg = "matrix_builder.MutSel_Matrix._calc_instantaneous_prob wrong when codons have different frequency, state_freqs approach.")



class matrixBuilder_MutSel_codon_fitness_tests(unittest.TestCase):
    ''' 
        Set of unittests for the matrix_builder.MutSel_Matrix subclass of MatrixBuilder, when codons are the unit of evolution.
    '''
    def setUp(self):
        ################### DO NOT CHANGE ANY OF THESE EVER. #######################
        self.codon_fitness = np.array( [0, 0.04028377, 0.02664918, 0, 0.00717921, 0.00700012, 0.0231568, 0.0231568, 0.02403056, 0.00737008, 0.03185765, 0.0193576, 0.03277142, 0.02141258, 0.0127537, 0.00298803, 0.0256333, 0.02312437, 0.01861465, 0.01586447, 0.00373147, 0.02662654, 0.00082524, 0.00048916, 0.01191673, 0.00512658, 0.00050502, 0.01688169, 0.01843001, 0.00215437, 0.02659356, 0.02377742, 0.01169375, 0.00097256, 0.02937344, 0.00268204, 0.01414414, 0.02781933, 0.00070877, 0.02370841, 0.02984617, 0.01828081, 0.01002825, 0.00870788, 0.00728006, 0.02179328, 0.00379049, 0.01978996, 0.00443774, 0.01201798, 0.02030269, 0.01238501, 0.01279963, 0.02094385, 0.02810987, 0.00918507, 0.02880549, 0.0029311, 0.0237658, 0.03194712, 0.06148723] )
        self.muParams = {'AG':0.1, 'GA':0.1, 'CT':0.1, 'TC':0.1, 'AC': 0.1, 'CA':0.1, 'AT':0.1, 'TA':0.1, 'CG':0.1, 'GC':0.1, 'GT':0.1, 'TG':0.1} # symmetric mutation means eq freqs are same as fitness values
        self.params = {'fitness': self.codon_fitness, 'mu': self.muParams}
        self.mutSelMatrix = matrix_builder.MutSel_Matrix( "mutsel", {'fitness': self.codon_fitness, 'mu': self.muParams, 'calc_by_freqs':False} )

        ############################################################################


    def test_MutSel_Matrix_codon_calc_instantaneous_prob(self):    
        ''' 
            Test function _calc_instantaneous_prob for mutation-selection model subclass, with fitness input.
        '''
        
        
        # Equal fitness.
        self.assertTrue( abs(self.mutSelMatrix._calc_instantaneous_prob(6, 7, self.mutSelMatrix.params) - 0.1) < ZERO, msg = "matrix_builder.MutSel_Matrix._calc_instantaneous_prob wrong when codons have same fitness.")

        # Different fitness.
        self.assertTrue( abs(self.mutSelMatrix._calc_instantaneous_prob(0, 1, self.mutSelMatrix.params) - 0.1020277113) < ZERO, msg = "matrix_builder.MutSel_Matrix._calc_instantaneous_prob wrong when codons have different fitness.")




# 
# 
# def run_matrix_builder_test():
#     run_tests = unittest.TextTestRunner()
#     
#     print "Testing assemble_matrix function of matrixBuilder for codon model"
#     test_suite_call = unittest.TestLoader().loadTestsFromTestCase(matrixBuilder_call_tests)
#     run_tests.run(test_suite_call)
#     
#     print "Testing sanity check functions in matrixBuilder"
#     test_suite_call = unittest.TestLoader().loadTestsFromTestCase(matrixBuilder_sanity)
#     run_tests.run(test_suite_call)
# 
#     print "Testing the functions in the base class matrixBuilder"
#     test_suite_baseMatrix = unittest.TestLoader().loadTestsFromTestCase(matrixBuilder_baseClass_tests)
#     run_tests.run(test_suite_baseMatrix)
#     
#     print "Testing matrix_builder.MechCodon_Matrix, a subclass of the parent matrixBuilder"
#     test_suite_mechCodonMatrix = unittest.TestLoader().loadTestsFromTestCase(matrixBuilder_matrix_builder.MechCodon_Matrix_tests)
#     run_tests.run(test_suite_mechCodonMatrix)
# 
#     print "Testing aminoAcids_Matrix, a subclass of the parent matrixBuilder"
#     test_suite_aminoAcidMatrix = unittest.TestLoader().loadTestsFromTestCase(matrixBuilder_matrix_builder.AminoAcid_Matrix_tests)
#     run_tests.run(test_suite_aminoAcidMatrix)
# 
#     print "Testing ECM_Matrix, a subclass of the parent matrixBuilder"
#     test_suite_empCodonMatrix = unittest.TestLoader().loadTestsFromTestCase(matrixBuilder_ECM_Matrix_tests)
#     run_tests.run(test_suite_empCodonMatrix)
#  
#     print "Testing matrix_builder.Nucleotide_Matrix, a subclass of the parent matrixBuilder"
#     test_suite_nucleotideMatrix = unittest.TestLoader().loadTestsFromTestCase(matrixBuilder_matrix_builder.Nucleotide_Matrix_tests)
#     run_tests.run(test_suite_nucleotideMatrix)
# 
#     print "Testing matrix_builder.MutSel_Matrix, a subclass of the parent matrixBuilder, when unit of evolution is codons."
#     test_suite_mutSelMatrix_codon = unittest.TestLoader().loadTestsFromTestCase(matrixBuilder_matrix_builder.mutSel_codon_Matrix_tests)
#     run_tests.run(test_suite_mutSelMatrix_codon)
# 
#     print "Testing matrix_builder.MutSel_Matrix, a subclass of the parent matrixBuilder, when unit of evolution is nucleotides."
#     test_suite_mutSelMatrix_nuc = unittest.TestLoader().loadTestsFromTestCase(matrixBuilder_matrix_builder.mutSel_nuc_Matrix_tests)
#     run_tests.run(test_suite_mutSelMatrix_nuc) 
#     
#     
