"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Annotations = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const constructs_1 = require("constructs");
const DEPRECATIONS_SYMBOL = Symbol.for('@aws-cdk/core.deprecations');
/**
 * Includes API for attaching annotations such as warning messages to constructs.
 *
 * @stability stable
 */
class Annotations {
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * Returns the annotations API for a construct scope.
     *
     * @param scope The scope.
     * @stability stable
     */
    static of(scope) {
        return new Annotations(scope);
    }
    /**
     * Adds a warning metadata entry to this construct.
     *
     * The CLI will display the warning when an app is synthesized, or fail if run
     * in --strict mode.
     *
     * @param message The warning message.
     * @stability stable
     */
    addWarning(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.WARN, message);
    }
    /**
     * Adds an info metadata entry to this construct.
     *
     * The CLI will display the info message when apps are synthesized.
     *
     * @param message The info message.
     * @stability stable
     */
    addInfo(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.INFO, message);
    }
    /**
     * Adds an { "error": <message> } metadata entry to this construct.
     *
     * The toolkit will fail synthesis when errors are reported.
     *
     * @param message The error message.
     * @stability stable
     */
    addError(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.ERROR, message);
    }
    /**
     * Adds a deprecation warning for a specific API.
     *
     * Deprecations will be added only once per construct as a warning and will be
     * deduplicated based on the `api`.
     *
     * If the environment variable `CDK_BLOCK_DEPRECATIONS` is set, this method
     * will throw an error instead with the deprecation message.
     *
     * @param api The API being deprecated in the format `module.Class.property` (e.g. `@aws-cdk/core.Construct.node`).
     * @param message The deprecation message to display, with information about alternatives.
     * @stability stable
     */
    addDeprecation(api, message) {
        const text = `The API ${api} is deprecated: ${message}. This API will be removed in the next major release`;
        // throw if CDK_BLOCK_DEPRECATIONS is set
        if (process.env.CDK_BLOCK_DEPRECATIONS) {
            throw new Error(`${constructs_1.Node.of(this.scope).path}: ${text}`);
        }
        // de-dup based on api key
        const set = this.deprecationsReported;
        if (set.has(api)) {
            return;
        }
        this.addWarning(text);
        set.add(api);
    }
    /**
     * Adds a message metadata entry to the construct node, to be displayed by the CDK CLI.
     * @param level The message level
     * @param message The message itself
     */
    addMessage(level, message) {
        constructs_1.Node.of(this.scope).addMetadata(level, message);
    }
    /**
     * Returns the set of deprecations reported on this construct.
     */
    get deprecationsReported() {
        let set = this.scope[DEPRECATIONS_SYMBOL];
        if (!set) {
            set = new Set();
            Object.defineProperty(this.scope, DEPRECATIONS_SYMBOL, { value: set });
        }
        return set;
    }
}
exports.Annotations = Annotations;
//# sourceMappingURL=data:application/json;base64,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