"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerVolumeConsistency = exports.BundlingDockerImage = void 0;
const child_process_1 = require("child_process");
const crypto = require("crypto");
const fs_1 = require("./fs");
/**
 * A Docker image used for asset bundling.
 *
 * @stability stable
 */
class BundlingDockerImage {
    /** @param image The Docker image */
    constructor(image, _imageHash) {
        this.image = image;
        this._imageHash = _imageHash;
    }
    /**
     * Reference an image on DockerHub or another online registry.
     *
     * @param image the image name.
     * @stability stable
     */
    static fromRegistry(image) {
        return new BundlingDockerImage(image);
    }
    /**
     * Reference an image that's built directly from sources on disk.
     *
     * @param path The path to the directory containing the Docker file.
     * @param options Docker build options.
     * @stability stable
     */
    static fromAsset(path, options = {}) {
        const buildArgs = options.buildArgs || {};
        // Image tag derived from path and build options
        const tagHash = crypto.createHash('sha256').update(JSON.stringify({
            path,
            ...options,
        })).digest('hex');
        const tag = `cdk-${tagHash}`;
        const dockerArgs = [
            'build', '-t', tag,
            ...(options.file ? ['-f', options.file] : []),
            ...flatten(Object.entries(buildArgs).map(([k, v]) => ['--build-arg', `${k}=${v}`])),
            path,
        ];
        dockerExec(dockerArgs);
        // Fingerprints the directory containing the Dockerfile we're building and
        // differentiates the fingerprint based on build arguments. We do this so
        // we can provide a stable image hash. Otherwise, the image ID will be
        // different every time the Docker layer cache is cleared, due primarily to
        // timestamps.
        const hash = fs_1.FileSystem.fingerprint(path, { extraHash: JSON.stringify(options) });
        return new BundlingDockerImage(tag, hash);
    }
    /**
     * Provides a stable representation of this image for JSON serialization.
     *
     * @returns The overridden image name if set or image hash name in that order
     * @stability stable
     */
    toJSON() {
        var _a;
        return (_a = this._imageHash) !== null && _a !== void 0 ? _a : this.image;
    }
    /**
     * Runs a Docker image.
     *
     * @stability stable
     */
    run(options = {}) {
        var _a, _b;
        const volumes = options.volumes || [];
        const environment = options.environment || {};
        const entrypoint = ((_a = options.entrypoint) === null || _a === void 0 ? void 0 : _a[0]) || null;
        const command = [
            ...((_b = options.entrypoint) === null || _b === void 0 ? void 0 : _b[1]) ? [...options.entrypoint.slice(1)]
                : [],
            ...options.command
                ? [...options.command]
                : [],
        ];
        const dockerArgs = [
            'run', '--rm',
            ...options.user
                ? ['-u', options.user]
                : [],
            ...flatten(volumes.map(v => { var _a; return ['-v', `${v.hostPath}:${v.containerPath}:${(_a = v.consistency) !== null && _a !== void 0 ? _a : DockerVolumeConsistency.DELEGATED}`]; })),
            ...flatten(Object.entries(environment).map(([k, v]) => ['--env', `${k}=${v}`])),
            ...options.workingDirectory
                ? ['-w', options.workingDirectory]
                : [],
            ...entrypoint
                ? ['--entrypoint', entrypoint]
                : [],
            this.image,
            ...command,
        ];
        dockerExec(dockerArgs);
    }
    /**
     * Copies a file or directory out of the Docker image to the local filesystem.
     *
     * @stability stable
     */
    cp(imagePath, outputPath) {
        const { stdout } = dockerExec(['create', this.image]);
        const match = stdout.toString().match(/([0-9a-f]{16,})/);
        if (!match) {
            throw new Error('Failed to extract container ID from Docker create output');
        }
        const containerId = match[1];
        const containerPath = `${containerId}:${imagePath}`;
        try {
            dockerExec(['cp', containerPath, outputPath]);
        }
        catch (err) {
            throw new Error(`Failed to copy files from ${containerPath} to ${outputPath}: ${err}`);
        }
        finally {
            dockerExec(['rm', '-v', containerId]);
        }
    }
}
exports.BundlingDockerImage = BundlingDockerImage;
/**
 * Supported Docker volume consistency types.
 *
 * Only valid on macOS due to the way file storage works on Mac
 *
 * @stability stable
 */
var DockerVolumeConsistency;
(function (DockerVolumeConsistency) {
    DockerVolumeConsistency["CONSISTENT"] = "consistent";
    DockerVolumeConsistency["DELEGATED"] = "delegated";
    DockerVolumeConsistency["CACHED"] = "cached";
})(DockerVolumeConsistency = exports.DockerVolumeConsistency || (exports.DockerVolumeConsistency = {}));
function flatten(x) {
    return Array.prototype.concat([], ...x);
}
function dockerExec(args, options) {
    var _a, _b, _c;
    const prog = (_a = process.env.CDK_DOCKER) !== null && _a !== void 0 ? _a : 'docker';
    const proc = child_process_1.spawnSync(prog, args, options !== null && options !== void 0 ? options : {
        stdio: [
            'ignore',
            process.stderr,
            'inherit',
        ],
    });
    if (proc.error) {
        throw proc.error;
    }
    if (proc.status !== 0) {
        if (proc.stdout || proc.stderr) {
            throw new Error(`[Status ${proc.status}] stdout: ${(_b = proc.stdout) === null || _b === void 0 ? void 0 : _b.toString().trim()}\n\n\nstderr: ${(_c = proc.stderr) === null || _c === void 0 ? void 0 : _c.toString().trim()}`);
        }
        throw new Error(`${prog} exited with status ${proc.status}`);
    }
    return proc;
}
//# sourceMappingURL=data:application/json;base64,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