"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RemoveTag = exports.Tags = exports.Tag = void 0;
// import * as cxapi from '@aws-cdk/cx-api';
const annotations_1 = require("./annotations");
const aspect_1 = require("./aspect");
const tag_manager_1 = require("./tag-manager");
/**
 * The common functionality for Tag and Remove Tag Aspects
 */
class TagBase {
    constructor(key, props = {}) {
        this.key = key;
        this.props = props;
    }
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(construct) {
        if (tag_manager_1.TagManager.isTaggable(construct)) {
            this.applyTag(construct);
        }
    }
}
/**
 * The Tag Aspect will handle adding a tag to this node and cascading tags to children.
 *
 * @stability stable
 */
class Tag extends TagBase {
    /**
     * @stability stable
     */
    constructor(key, value, props = {}) {
        super(key, props);
        this.defaultPriority = 100;
        if (value === undefined) {
            throw new Error('Tag must have a value');
        }
        this.value = value;
    }
    /**
     * (deprecated) DEPRECATED: add tags to the node of a construct and all its the taggable children.
     *
     * @deprecated use `Tags.of(scope).add()`
     */
    static add(scope, key, value, props = {}) {
        annotations_1.Annotations.of(scope).addDeprecation('@aws-cdk/core.Tag.add(scope,k,v)', 'Use "Tags.of(scope).add(k,v)" instead');
        Tags.of(scope).add(key, value, props);
    }
    /**
     * (deprecated) DEPRECATED: remove tags to the node of a construct and all its the taggable children.
     *
     * @deprecated use `Tags.of(scope).remove()`
     */
    static remove(scope, key, props = {}) {
        annotations_1.Annotations.of(scope).addDeprecation('@aws-cdk/core.Tag.remove(scope,k,v)', 'Use "Tags.of(scope).remove(k,v)" instead');
        Tags.of(scope).remove(key, props);
    }
    /**
     * @stability stable
     */
    applyTag(resource) {
        var _a;
        if (resource.tags.applyTagAspectHere(this.props.includeResourceTypes, this.props.excludeResourceTypes)) {
            resource.tags.setTag(this.key, this.value, (_a = this.props.priority) !== null && _a !== void 0 ? _a : this.defaultPriority, this.props.applyToLaunchedInstances !== false);
        }
    }
}
exports.Tag = Tag;
/**
 * Manages AWS tags for all resources within a construct scope.
 *
 * @stability stable
 */
class Tags {
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * Returns the tags API for this scope.
     *
     * @param scope The scope.
     * @stability stable
     */
    static of(scope) {
        return new Tags(scope);
    }
    /**
     * add tags to the node of a construct and all its the taggable children.
     *
     * @stability stable
     */
    add(key, value, props = {}) {
        aspect_1.Aspects.of(this.scope).add(new Tag(key, value, props));
    }
    /**
     * remove tags to the node of a construct and all its the taggable children.
     *
     * @stability stable
     */
    remove(key, props = {}) {
        aspect_1.Aspects.of(this.scope).add(new RemoveTag(key, props));
    }
}
exports.Tags = Tags;
/**
 * The RemoveTag Aspect will handle removing tags from this node and children.
 *
 * @stability stable
 */
class RemoveTag extends TagBase {
    /**
     * @stability stable
     */
    constructor(key, props = {}) {
        super(key, props);
        this.defaultPriority = 200;
    }
    /**
     * @stability stable
     */
    applyTag(resource) {
        var _a;
        if (resource.tags.applyTagAspectHere(this.props.includeResourceTypes, this.props.excludeResourceTypes)) {
            resource.tags.removeTag(this.key, (_a = this.props.priority) !== null && _a !== void 0 ? _a : this.defaultPriority);
        }
    }
}
exports.RemoveTag = RemoveTag;
//# sourceMappingURL=data:application/json;base64,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