from typing import Sequence, Union, Any

from collections import defaultdict

import numpy as np

def hex2color(color):
  b = (color & 0xff) / 255.0
  g = ((color >> 8) & 0xff) / 255.0
  r = ((color >> 16) & 0xff) / 255.0
  return (r,g,b)

# created with help of https://coolors.co/001021-ffa400-009ffd-eaf6ff-9e7b9b
COLORS = [
  (1.0, 0.6431372549019608, 0.0), # alice blue (white)
  (0.0, 0.6235294117647059, 0.9921568627450981), # orange
  (0.9176470588235294, 0.9647058823529412, 1.0), # celestial blue
  (0.8941176470588236, 0.5843137254901961, 0.6196078431372549), # salmon
  (0.9411764705882353, 0.2196078431372549, 0.4196078431372549), # cerise
  (0.6, 0.8666666666666667, 0.7843137254901961), # tiffany blue (green)
  (0.8117647058823529, 0.6941176470588235, 0.7176470588235294), # tearose red
  (0.8980392156862745, 0.3254901960784314, 0.5058823529411764), # blush
  (0.7647058823529411, 0.7647058823529411, 0.9019607843137255),# periwinkle
]

BBOX_COLORS = [
  0xE9D758, # yellow
  0x06D6A0, # emerald
  0x00ABE7, # picton blue
  0xF58549, # orange (crayola)
  0xEDC9FF, # mauve
]
BBOX_COLORS = [ hex2color(c) for c in BBOX_COLORS ]

RADII_KEYWORDS = ('radius', 'radii', 'r')
CROSS_SECTION_KEYWORDS = ('cross_section', 'x')
COMPONENT_KEYWORDS = ('component', 'components', 'c')

def toiter(obj, is_iter=False):
  if isinstance(obj, (str, dict, np.ndarray)):
    if is_iter:
      return [ obj ], False
    return [ obj ]

  try:
    iter(obj)
    if is_iter:
      return obj, True
    return obj 
  except TypeError:
    if is_iter:
      return [ obj ], False
    return [ obj ]

def is_mesh(obj):
  """zmesh meshes are fine, but not covered by cloudvolume.Mesh"""
  return hasattr(obj, "vertices") and hasattr(obj, "faces")

def is_skeleton(obj):
  return hasattr(obj, "vertices") and hasattr(obj, "edges")

def is_bbox(obj):
  return hasattr(obj, "minpt") and hasattr(obj, "maxpt")

def is_point_cloud(obj):
  return isinstance(obj, np.ndarray) and obj.ndim == 2

# vtk code written with help of ChatGPT and DeepSeek
def objects(
  objects:Sequence[Any],
  skeleton_color_by='r',
):
  """
  Produce a 3D co-visualization of meshes and skeletons together.
  """
  import vtk
  objects = toiter(objects)

  if skeleton_color_by in RADII_KEYWORDS:
    skeleton_color_by = 'r'
  elif skeleton_color_by in COMPONENT_KEYWORDS:
    skeleton_color_by = 'c'
  elif skeleton_color_by in CROSS_SECTION_KEYWORDS:
    skeleton_color_by = 'x'
  else:
    skeleton_color_by = 'c'

  pairs = defaultdict(list)

  for obj in objects:
    if hasattr(obj, "id"):
      pairs[obj.id].append(obj)
    elif hasattr(obj, "segid"):
      pairs[obj.segid].append(obj)
    elif hasattr(obj, "label"):
      pairs[obj.label].append(obj)
    else:
      pairs[None].append(obj)

  meshes = []
  skels = []
  ptcs = []
  bboxes = []

  for obj in objects:
    if is_mesh(obj):
      meshes.append(obj)
    elif is_skeleton(obj):
      skels.append(obj)
    elif is_point_cloud(obj):
      ptcs.append(obj)
    elif is_bbox(obj):
      bboxes.append(obj)
    else:
      raise ValueError(f"Unable to determine object display type: {obj}")

  lut = None
  if skeleton_color_by in ('r', 'x'):
    min_val = np.inf
    max_val = -np.inf
    prop = "radii" if skeleton_color_by == 'r' else 'cross_sectional_area'
    for skel in skels:
      min_val = min(getattr(skel, prop).min(), min_val)
      max_val = max(getattr(skel, prop).max(), max_val)
    lut = create_color_map(min_val, max_val)

  actors = []

  for obj in pairs[None]:
    if is_mesh(obj):
      actors.append(
        create_vtk_mesh(obj, opacity=1.0)
      )
    elif is_skeleton(obj):
      skels.append(obj)
      actors.extend(
        create_vtk_skeleton(obj, color_by=skeleton_color_by, lut=lut)
      )
    elif is_point_cloud(obj):
      actors.append(
        create_vtk_point_cloud(obj)
      )

  pairs.pop(None)

  segids = []

  for i, (label, objs) in enumerate(pairs.items()):
    segids.append(label)
    mesh_opacity = 1.0
    for obj in objs:
      if is_skeleton(obj):
        mesh_opacity = 0.2
        break

    for obj in objs:
      if is_mesh(obj):
        actors.append(
          create_vtk_mesh(obj, opacity=mesh_opacity)
        )
      elif is_skeleton(obj):
        skels.append(obj)
        actors.extend(
          create_vtk_skeleton(obj, color_by=skeleton_color_by, lut=lut)
        )

  segids.sort()

  for i, bbx in enumerate(bboxes):
    color = BBOX_COLORS[i % len(BBOX_COLORS)]
    actors.append(
      create_vtk_bbox(bbx, color)
    )

  if skeleton_color_by in ('r', 'x') and len(skels) > 0:
    scalar_bar = vtk.vtkScalarBarActor()
    scalar_bar.SetLookupTable(lut)
    scalar_bar.SetTitle("Radius" if skeleton_color_by == 'r' else 'Cross Sectional Area')
    scalar_bar.SetWidth(0.1)
    scalar_bar.SetHeight(0.3)

    actors.append(scalar_bar)

  display_actors(segids, actors)

def display_actors(segids, actors):
  if len(actors) == 0:
    return

  try:
    import vtk
  except ImportError:
    print("The visualize viewer requires the OpenGL based vtk. Try: pip install vtk --upgrade")
    return

  renderer = vtk.vtkRenderer()
  renderer.SetBackground(0.0, 0.06274509803921569, 0.12941176470588237) # rich black

  render_window = vtk.vtkRenderWindow()
  render_window.AddRenderer(renderer)
  render_window.SetSize(2000, 1500)

  interactor = vtk.vtkRenderWindowInteractor()
  style = vtk.vtkInteractorStyleTrackballCamera()
  interactor.SetInteractorStyle(style)

  interactor.SetRenderWindow(render_window)

  for actor in actors:
    renderer.AddActor(actor)

  window_name = f"Mesh & Skeleton Viewer"
  if segids:
    segids = [ str(x) for x in segids ]
    window_name += f" ({','.join(segids)})"
  
  render_window.SetWindowName(window_name)
  
  render_window.Render()
  interactor.Start()

def create_vtk_skeleton(skel, color_by, lut):
  import vtk.util.numpy_support

  actors = []

  if hasattr(skel, "components") and color_by == 'c':
    skels = skel.components()
  else:
    skels = [ skel ]

  for i, skel in enumerate(skels):
    points = vtk.vtkPoints()
    points.SetData(vtk.util.numpy_support.numpy_to_vtk(skel.vertices))

    lines = vtk.vtkCellArray()

    for edge in skel.edges:
      line = vtk.vtkLine()
      line.GetPointIds().SetId(0, edge[0])
      line.GetPointIds().SetId(1, edge[1])
      lines.InsertNextCell(line)
  
    polyline = vtk.vtkPolyData()
    polyline.SetPoints(points)
    polyline.SetLines(lines)

    mapper = vtk.vtkPolyDataMapper()
    mapper.SetInputData(polyline)

    if color_by == 'r' and skel.radii is not None:
      radii = vtk.util.numpy_support.numpy_to_vtk(skel.radii)
      radii.SetName("Radii")
      polyline.GetPointData().SetScalars(radii)
      mapper.SetScalarVisibility(1)
      mapper.SetScalarModeToUsePointData()
      mapper.SetLookupTable(lut)
      mapper.UseLookupTableScalarRangeOn()
    elif color_by == 'x' and skel.cross_sectional_area is not None:
      xsa = vtk.util.numpy_support.numpy_to_vtk(skel.cross_sectional_area)
      xsa.SetName("Cross Sectional Area")
      polyline.GetPointData().SetScalars(xsa)
      mapper.SetScalarVisibility(1)
      mapper.SetScalarModeToUsePointData()
      mapper.SetLookupTable(lut)
      mapper.UseLookupTableScalarRangeOn()

    actor = vtk.vtkActor()
    actor.SetMapper(mapper)
    actor.GetProperty().SetLineWidth(2)

    if color_by == 'c':
      color = COLORS[i % len(COLORS)]
      actor.GetProperty().SetColor(*color)
    actors.append(actor)

  return actors

def create_vtk_mesh(mesh, opacity=1.0):
  import vtk
  from vtk.util.numpy_support import numpy_to_vtk, numpy_to_vtkIdTypeArray

  vertices = mesh.vertices
  faces = mesh.faces

  vtk_points = vtk.vtkPoints()
  vtk_points.SetData(numpy_to_vtk(vertices))
  
  polydata = vtk.vtkPolyData()
  polydata.SetPoints(vtk_points)
  
  vtk_faces = vtk.vtkCellArray()
  vtk_faces.SetCells(
    faces.shape[0], 
    numpy_to_vtkIdTypeArray(
      np.hstack([np.full((faces.shape[0], 1), 3), faces]).flatten()
    )
  )

  polydata.SetPolys(vtk_faces)
  
  mapper = vtk.vtkPolyDataMapper()
  mapper.SetInputData(polydata)

  actor = vtk.vtkActor()
  actor.SetMapper(mapper)

  actor.GetProperty().SetOpacity(opacity)

  return actor

def create_vtk_bbox(bbox, color=None):
  import vtk
  vtk_points = vtk.vtkPoints()
  
  minpt = bbox.minpt.clone()
  maxpt = bbox.maxpt.clone()

  corners = [
    minpt,
    [ maxpt[0], minpt[1], minpt[2] ],
    [ minpt[0], maxpt[1], minpt[2] ],
    [ minpt[0], minpt[1], maxpt[2] ],
    [ maxpt[0], maxpt[1], minpt[2] ],
    [ maxpt[0], minpt[1], maxpt[2] ],
    [ minpt[0], maxpt[1], maxpt[2] ],
    maxpt,
  ]

  for pt in corners:
    vtk_points.InsertNextPoint(np.asarray(pt))

  edges = [
    [0,1], [0,2], [0,3],
    [7,6], [7,5], [7,4],
    [6,2], [6,3],
    [3,5], [5,1],
    [2,4], [4,1],
  ]

  lines = vtk.vtkCellArray()

  for edge in edges:
    line = vtk.vtkLine()
    line.GetPointIds().SetId(0, edge[0])
    line.GetPointIds().SetId(1, edge[1])
    lines.InsertNextCell(line)

  poly_data = vtk.vtkPolyData()
  poly_data.SetPoints(vtk_points)
  poly_data.SetLines(lines)

  mapper = vtk.vtkPolyDataMapper()
  mapper.SetInputData(poly_data)

  if color is None:
    color = BBOX_COLORS[0]

  actor = vtk.vtkActor()
  actor.SetMapper(mapper)
  actor.GetProperty().SetColor(*color)
  actor.GetProperty().SetLineWidth(2)

  return actor

def create_vtk_point_cloud(ptc):
  import vtk
  from vtk.util.numpy_support import numpy_to_vtk

  colors = vtk.vtkNamedColors()
  bkg = map(lambda x: x / 255.0, [40, 40, 40, 255])
  colors.SetColor("BkgColor", *bkg)

  points = vtk.vtkPoints()
  vtk_points = numpy_to_vtk(ptc, deep=True)
  points.SetData(vtk_points)

  verts = vtk.vtkCellArray()
  n_points = ptc.shape[0]
  connectivity = np.vstack((np.ones(n_points, dtype=np.int64), np.arange(n_points, dtype=np.int64))).T.flatten()
  verts.SetCells(n_points, numpy_to_vtk(connectivity, deep=True, array_type=vtk.VTK_ID_TYPE))

  polydata = vtk.vtkPolyData()
  polydata.SetPoints(points)
  polydata.SetVerts(verts)

  mapper = vtk.vtkPolyDataMapper()
  mapper.SetInputData(polydata)

  cylinderActor = vtk.vtkActor()
  cylinderActor.SetMapper(mapper)
  cylinderActor.GetProperty().SetColor(colors.GetColor3d("Mint"))

  return cylinderActor

def create_color_map(min_val, max_val):
  import vtk
  import vtk.util.numpy_support

  lut = vtk.vtkLookupTable()
  lut.SetScaleToLinear()
  lut.SetRange(min_val, max_val)
  
  lut.SetNumberOfColors(256)
  cmap = fast_256_color_map()
  for i, (r,g,b) in enumerate(cmap):
    lut.SetTableValue(i, r, g, b, 1.0)
  lut.Build()

  return lut

def fast_256_color_map():
  """a percepturally uniform color map from https://www.kennethmoreland.com/color-advice/"""
  return [
    [0.05639932216773370, 0.056399092153948,  0.4700000908789250],
    [0.06443194593704830, 0.06748122089429930,  0.47746792297101200],
    [0.07190389960400100, 0.07804712602017210,  0.48495121359058800],
    [0.07892416836694120, 0.0882359810389429, 0.49244983979264200],
    [0.08557038629717870, 0.0981410827903004, 0.4999636829875800],
    [0.09189999818606120, 0.10782804652340500,  0.5074926286471110],
    [0.09795680732284220, 0.11734474538657700,  0.5150365660347640],
    [0.1037750394124900,  0.12672713622677700,  0.5225953879587200],
    [0.10938198175602300, 0.1360028675714410, 0.5301689905448800],
    [0.11479976219466100, 0.14519361537463600,  0.5377572730282970],
    [0.12004658617215000, 0.15431665134039100,  0.5453601375613080],
    [0.12513762006916400, 0.16338592867055300,  0.5529774890368360],
    [0.13008563650194000, 0.1724128535393210, 0.5606092349255300],
    [0.13490149517433900, 0.18140684568347000,  0.5682552851254840],
    [0.139594507486971, 0.19037575377632900,  0.5759155518234640],
    [0.14417271730081800, 0.1993261685199890, 0.5835899493666130],
    [0.14864312012942800, 0.20826366224775400,  0.5912783941437520],
    [0.15301183638613800, 0.21719297478227600,  0.5989808044754440],
    [0.15728424984920000, 0.22611815936358200,  0.606697100512088],
    [0.16146511945012600, 0.23504269848578200,  0.6144272041393700],
    [0.16555867035801100, 0.2439695967637210, 0.6221710388904570],
    [0.1695686688206180,  0.2529014560596130, 0.6299285298643790],
    [0.1734984841350040,  0.2618405367621220, 0.6376996036501020],
    [0.1773511403266610,  0.2707888081502450, 0.6454841882558260],
    [0.18112935952977000, 0.27974799007582100,  0.6532822130430910],
    [0.18483559862296200, 0.2887195876839070, 0.6610936086653070],
    [0.18847208034391000, 0.2977049205068440, 0.6689183070103790],
    [0.19204081985356200, 0.30670514697906400,  0.6767562411470750],
    [0.19554364752633700, 0.31572128520009800,  0.6846073452748770],
    [0.19898222859159200, 0.32475423060468700,  0.6924715546770410],
    [0.20235808013339100, 0.3338047710684280, 0.700348805676614],
    [0.2056725858623350,  0.3428735998755730, 0.7082390355951980],
    [0.20892700899908400, 0.35196132689557100,  0.716142182714247],
    [0.21212250354990800, 0.3610684882515540, 0.7240581862387160],
    [0.21526012420691000, 0.3701955547135140, 0.7319869862628920],
    [0.2183408350668930,  0.379342939008371,  0.7399285237382380],
    [0.2213655173314000,  0.3885110022064910, 0.7478827404431220],
    [0.22433497612471500, 0.39770005931773100,  0.7558495789542830],
    [0.2272499465453700,  0.4069103842084510, 0.7638289826199100],
    [0.23011109904924500, 0.41614221393323800,  0.7718208955342460],
    [0.2329190442477430,  0.42539575256053200,  0.7798252625135710],
    [0.23567433719240800, 0.43467117455924300,  0.7878420290735100],
    [0.23837748120717100, 0.4439686278034620, 0.7958711414075460],
    [0.24102893132086000, 0.45328823624400400,  0.8039125463666710],
    [0.24413123396108300, 0.46242758064956300,  0.8110912093541010],
    [0.248710826714151, 0.4709583584046360, 0.8155627713888570],
    [0.25319805741062100, 0.4795061765161260, 0.8200336923321040],
    [0.2575967867866220,  0.4880711150864920, 0.8245039683127460],
    [0.2619105369566000,  0.4966532423251760, 0.8289735958201660],
    [0.26614252917141400, 0.5052526155531000, 0.833442571690173],
    [0.27029571635856100, 0.5138692821156820, 0.8379108930914740],
    [0.27437281129328600, 0.5225032802137980, 0.8423785575126760],
    [0.2783763110913720,  0.5311546396609890, 0.8468455627497800],
    [0.2823085185893620,  0.5398233825743000, 0.8513119068941470],
    [0.28617156107822600, 0.548509524005271,  0.8557775883209310],
    [0.2899674067765170,  0.5572130725169170, 0.8602426056779430],
    [0.2936978793644270,  0.5659340307118660, 0.8647069578749350],
    [0.2973646708476800,  0.5746723957162740, 0.869170644073296],
    [0.30096935297736400, 0.5834281596236580, 0.8736336636761250],
    [0.30451338741658500, 0.5922013099023430, 0.8780960163186830],
    [0.3079981348158110,  0.6009918297698320, 0.8825577018592040],
    [0.31142486293469400, 0.6097996985370830, 0.887018720370038],
    [0.3147947539281480,  0.6186248919253750, 0.891479072129139],
    [0.31810891089760600, 0.6274673823581590, 0.8959387576118560],
    [0.32136836379441900, 0.6363271392300790, 0.9003977774830370],
    [0.3245740747503920,  0.6452041291551360, 0.9048561325894160],
    [0.3277269429004550,  0.6540983161957360, 0.9093138239522960],
    [0.33082780875392,  0.6630096620742710, 0.9137708527604810],
    [0.33387745816350500, 0.6719381263686550, 0.9182272203634850],
    [0.33687662593511000, 0.6808836666931530, 0.9226829282649830],
    [0.3398259991159490,  0.6898462388657010, 0.9271379781165010],
    [0.34272621999410500, 0.6988257970628030, 0.9315923717113360],
    [0.3455778888385990,  0.7078222939630030, 0.9360461109787020],
    [0.3483815664056250,  0.7168356808798380, 0.9404991979780850],
    [0.35113777623364100, 0.7258659078850920, 0.9449516348938050],
    [0.3538470067474430,  0.7349129239231140, 0.9494034240297740],
    [0.35650971318903900, 0.7439766769168810, 0.9538545678044440],
    [0.3686486159195670,  0.7496719743724780, 0.9535369353880290],
    [0.3815959970274270,  0.754937681291743,  0.9525909356580740],
    [0.39413356233626700, 0.76021468885306, 0.9516377182615300],
    [0.4063024735081890,  0.7655028952016580, 0.9506772210374790],
    [0.41813776877433000, 0.770802200400734,  0.9497093812939490],
    [0.4296695519408270,  0.7761125063850730, 0.9487341357987290],
    [0.4409239016605540,  0.7814337169159090, 0.9477514207699870],
    [0.4519235771899750,  0.7867657375369960, 0.946761171866697],
    [0.4626885735064480,  0.7921084755318630, 0.9457633241788700],
    [0.47323656318688600, 0.7974618398822040, 0.9447578122175740],
    [0.48358325196848400, 0.8028257412273940, 0.9437445699047480],
    [0.49374266767646100, 0.8082000918250790, 0.942723530562804],
    [0.5037273971200840,  0.8135848055128320, 0.9416946269040060],
    [0.513548781929538, 0.8189797976708280, 0.9406577910196200],
    [0.5232170816783550,  0.824384985185519,  0.9396129543688380],
    [0.5327416107075670,  0.8298002864142790, 0.9385600477674610],
    [0.5421308536351070,  0.8352256211509950, 0.9374990013763350],
    [0.5513925634576400,  0.8406609105925690, 0.9364297446895420],
    [0.5605338453348350,  0.8461060773063230, 0.9353522065223280],
    [0.5695612285196900,  0.8515610451982570, 0.9342663149987700],
    [0.5784807284139460,  0.8570257394821650, 0.933171997539168],
    [0.5872979003496580,  0.8625000866495560, 0.9320691808471650],
    [0.5960178864007430,  0.8679840144403790, 0.9309577908965700],
    [0.6046454562929240,  0.8734774518145240, 0.9298377529178950],
    [0.6131850432926930,  0.8789803289240680, 0.928708991384588],
    [0.6216407758051290,  0.8844925770862560, 0.9275714299989510],
    [0.6300165052886300,  0.8900141287571940, 0.9264249916777440],
    [0.6383158309955970,  0.8955449175062280, 0.9252695985374530],
    [0.6465421219672680,  0.9010848779910000, 0.9241051718792330],
    [0.6546985366444450,  0.906633945933147,  0.9229316321734860],
    [0.6627880404010160,  0.9121920580946440, 0.9217488990441000],
    [0.6708134212616970,  0.9177591522547540, 0.9205568912523090],
    [0.6787773040275250,  0.9233351671875820, 0.9193555266801760],
    [0.6866821630009090,  0.9289200426402110, 0.9181447223136930],
    [0.6999242376409360,  0.9306804811857100, 0.9110229813056300],
    [0.7141064137882080,  0.9315263367224490, 0.9024833660761210],
    [0.7279336925459710,  0.9323724836736860, 0.893935359958582],
    [0.7414298894957880,  0.9332188724027480, 0.8853784248335960],
    [0.7546163852159120,  0.9340654535099640, 0.8768119990173190],
    [0.7675124589516790,  0.9349121778316000, 0.8682354959686940],
    [0.7801355659032910,  0.9357589964387720, 0.859648302907607],
    [0.7925015693285680,  0.9366058606363210, 0.8510497793365110],
    [0.8046249361255780,  0.9374527219616660, 0.8424392554572570],
    [0.8165189026653990,  0.9382995321836300, 0.8338160304740420],
    [0.8281956162125860,  0.9391462433012380, 0.8251793707724440],
    [0.8396662561766550,  0.9399928075424910, 0.8165285079634750],
    [0.850941138594273, 0.9408391773631160, 0.8078626367803620],
    [0.8620298065857840,  0.9416853054452900, 0.7991809128144840],
    [0.8729411090153830,  0.9425311446963390, 0.7904824500753620],
    [0.883683269177892, 0.9433766482474210, 0.7817663183579190],
    [0.8942639450117820,  0.9442217694521810, 0.7730315403983100],
    [0.9013393453951880,  0.9421644543953600, 0.7628809831324590],
    [0.9049372073519950,  0.9372074689690950, 0.7513347455660150],
    [0.9084188627047560,  0.9322537256219690, 0.7397952846250470],
    [0.9117869647290890,  0.9273031938576770, 0.7282619301457690],
    [0.9150440635332550,  0.9223558422042180, 0.7167339628433540],
    [0.9181926120887530,  0.917411638185727,  0.705210610241299],
    [0.9212349718263310,  0.9124705482933760, 0.693691042189503],
    [0.9241734178355450,  0.9075325379553220, 0.6821743659208000],
    [0.9270101437020870,  0.9025975715056520, 0.6706596205884530],
    [0.9297472660136880,  0.8976656121522960, 0.6591457712185590],
    [0.9323868285623560,  0.892736621943863,  0.6476317020013140],
    [0.9349308062680260,  0.8878105617353510, 0.6361162088333360],
    [0.9373811088462700,  0.8828873911527020, 0.6245979910093220],
    [0.9397395842406090,  0.8779670685561430, 0.6130756419448370],
    [0.9420080218380370,  0.8730495510022660, 0.6015476387924040],
    [0.9441881554846690,  0.8681347942047980, 0.5900123307896370],
    [0.9462816663168980,  0.8632227524940100, 0.578467926150012],
    [0.9482901854220710,  0.8583133787747000, 0.5669124772730020],
    [0.9502152963414720,  0.853406624482705,  0.5553438640092350],
    [0.9520585374272720,  0.848502439539863,  0.5437597746664450],
    [0.9538214040641280,  0.8436007723073770, 0.5321576843809450],
    [0.9555053507652130,  0.8387015695375000, 0.5205348304043150],
    [0.9571069887237200,  0.8337997270407560, 0.5089002361768480],
    [0.9566218429723000,  0.8267655822198440, 0.5023498930554820],
    [0.956100029721289, 0.8197315111349760, 0.49580838283005900],
    [0.95554182649875,  0.8126973482555960, 0.4892756459719220],
    [0.9549475061895150,  0.8056629217034790, 0.4827516177573550],
    [0.954317337160106, 0.7986280529645120, 0.47623622792067600],
    [0.9536515833800670,  0.7915925565844130, 0.46972940028094900],
    [0.9529505045398370,  0.784556239847339,  0.4632310523399570],
    [0.952214356165343, 0.7775189024361980, 0.456741094848844],
    [0.9514433897294170,  0.7704803360734190, 0.45025943134055500],
    [0.9506378527602000,  0.7634403241408240, 0.44378595762488100],
    [0.949797988946638, 0.7563986412771410, 0.43732056124260000],
    [0.9489240382412170,  0.749355052951553,  0.43086312087479200],
    [0.9480162369600320,  0.7423093150115770, 0.4244135057029740],
    [0.9470748178803200,  0.7352611732034020, 0.41797157471521400],
    [0.9461000103355570,  0.728210362662648,  0.4115371759528210],
    [0.9450920403082330,  0.721156607373351,  0.405110145691556],
    [0.9440511305203960,  0.7140996195927660, 0.3986903075506200],
    [0.9429775005220720,  0.7070390992393840, 0.3922774715218330],
    [0.9418713667776480,  0.699974733241297,  0.3858714329104830],
    [0.9407329427503150,  0.6929061948418100, 0.37947197117826600],
    [0.9395624389846550,  0.6858331428588960, 0.3730788486775170],
    [0.9383600631874550,  0.6787552208947670, 0.36669180926449300],
    [0.9371260203068370,  0.6716720564914810, 0.3603105767779150],
    [0.9358605126097800,  0.6645832602281190, 0.35393485336703300],
    [0.9345637397581070,  0.6574884247546080, 0.347564317651384],
    [0.9332358988830150,  0.6503871237567960, 0.3411986226918730],
    [0.9318771846582290,  0.6432789108468100, 0.3348373937498950],
    [0.9304877893718340,  0.6361633183721530, 0.3284802258078330],
    [0.92906790299687,  0.629039856136278,  0.3221266808202770],
    [0.9276177132607420,  0.6219080100226250, 0.31577628466064500],
    [0.9244587776148210,  0.6146190347403950, 0.31117238599971300],
    [0.9211661252736100,  0.607317318420597,  0.3067008710491440],
    [0.917858929793371, 0.6000131994170060, 0.30223874550682800],
    [0.9145371930891840,  0.5927062777200020, 0.29778605983046500],
    [0.9112009168084140,  0.5853961322067050, 0.2933428649432670],
    [0.9078501023330210,  0.5780823192771120, 0.28890921223404900],
    [0.904484750781961, 0.5707643713807800, 0.2844851535568000],
    [0.9011048630136880,  0.5634417954233610, 0.28007074122970300],
    [0.897710439628758, 0.5561140710411120, 0.2756660280335360],
    [0.8943014809725430,  0.548780648730045,  0.27127106720940600],
    [0.8908779871380700,  0.5414409478148400, 0.266885912455737],
    [0.8874399579689790,  0.5340943542407850, 0.2625106179244610],
    [0.883987393062613, 0.5267402181699660, 0.2581452382163140],
    [0.8805202917732620,  0.5193778513605250, 0.25378982837517700],
    [0.8770386532155370,  0.5120065243051010, 0.24944444388134500],
    [0.8735424762679210,  0.5046254631014190, 0.24510914064364600],
    [0.8700317595764770,  0.4972338460243790, 0.24078397499029600],
    [0.8665065015587390,  0.48983079976480400,  0.23646900365836300],
    [0.8629667004077870,  0.4824153952951660, 0.23216428378172300],
    [0.859412354096527, 0.4749866433169580, 0.22786987287736000],
    [0.8558434603821780,  0.46754348923778500,  0.22358582882985800],
    [0.8522600168109810,  0.4600848076185490, 0.21931220987391900],
    [0.8486620207231470,  0.45260939602205400,  0.215049074574716],
    [0.8450494692580500,  0.4451159681836960, 0.21079648180589200],
    [0.8414223593596880,  0.43760314641227500,  0.20655449072497300],
    [0.8377806877824180,  0.4300694531140440, 0.20232316074595900],
    [0.8341244510969860,  0.42251330131525400,  0.19810255150883200],
    [0.8304536456968720,  0.41493298403715300,  0.1938927228456910],
    [0.826768267804951, 0.40732666235176800,  0.18969373474319000],
    [0.8230683134805160,  0.39969235191591600,  0.18550564730095000],
    [0.8193537786266550,  0.3920279077434000, 0.1813285206855440],
    [0.8156246589980200,  0.38433100692968000,  0.17716241507966200],
    [0.8118809502090020,  0.376599128987434,  0.17300739062597500],
    [0.808122647742342, 0.36882953338249600,  0.16886350736522200],
    [0.8043497469581880,  0.3610192337743990, 0.1647308251679620],
    [0.8005622431036400,  0.3531649683594050, 0.16060940365938100],
    [0.7953588442835880,  0.34676259298048700,  0.1591933503708630],
    [0.7899130354331120,  0.34060776010612600,  0.1582418165218540],
    [0.7844696388302480,  0.3344417791797090, 0.15728531127875700],
    [0.7790286437022240,  0.32826360174348600,  0.1563238293793060],
    [0.7735900390106230,  0.32207209971854000,  0.1553573653058290],
    [0.7681538134456000,  0.3158660576010760, 0.15438591328215300],
    [0.7627199554199750,  0.30964416368653900,  0.15340946727040200],
    [0.7572884530631830,  0.3034050001721940, 0.15242802096770300],
    [0.751859294215082, 0.29714703196117000,  0.1514415678027760],
    [0.7464324664196130,  0.2908685939573000, 0.15045010093241900],
    [0.7410079569183130,  0.2845678765989660, 0.149453613237877],
    [0.7355857526436780,  0.2782429093293660, 0.14845209732110700],
    [0.7301658402123660,  0.2718915416379210, 0.1474455455009100],
    [0.7247482059182420,  0.26551142122923100,  0.1464339498089580],
    [0.7193328357252660,  0.25909996877794600,  0.14541730198568100],
    [0.7139197152602080,  0.2526543486040680, 0.14439559347604500],
    [0.708508829805194, 0.24617143444564800,  0.14336881542518300],
    [0.703100164290088, 0.23964776930397100,  0.14233695867390000],
    [0.6976937032846880,  0.23307951807533300,  0.14130001375404500],
    [0.6922894309907460,  0.22646241134319800,  0.1402579708837310],
    [0.6868873312338100,  0.21979167825645400,  0.13921081996241700],
    [0.6814873874548690,  0.21306196582343200,  0.13815855056584500],
    [0.6760895827018230,  0.20626724114969000,  0.13710115194081500],
    [0.6706938996207490,  0.19940067205610700,  0.13603861299980800],
    [0.6653003204469770,  0.19245448000868300,  0.13497092231545300],
    [0.6599088269959670,  0.1854197571855820, 0.13389806811482100],
    [0.6545194006539880,  0.17828623651495400,  0.13282003827355600],
    [0.649132022368588, 0.17104199919226000,  0.13173682030982600],
    [0.643746672638862, 0.16367309781549700,  0.13064840137810500],
    [0.6383633315055160,  0.15616306369442900,  0.1295547682627610],
    [0.632981978540714, 0.14849225213406000,  0.12845590737146700],
    [0.6276025928377240,  0.1406369561627570, 0.12735180472841400],
    [0.6222251530003430,  0.13256818116951300,  0.12624244596732600],
    [0.616849637132119, 0.12424990884503300,  0.12512781632427900],
    [0.6114760228253570,  0.11563656645301500,  0.12400790063030800],
    [0.6061042871499130,  0.10666921103551600,  0.12288268330380800],
    [0.6007344066417850,  0.09726953748587850,  0.12175214834271000],
    [0.5953663572914870,  0.08732998809377950,  0.12061627931644300],
    [0.5900001145322250,  0.07669636770019070,  0.11947505935767000],
  ]

