"""
URLconf for login-protected views of the cms package. These urls are processed by
:mod:`~integreat_cms.core.middleware.access_control_middleware.AccessControlMiddleware`.
Views which should not have login protection go into :mod:`~integreat_cms.cms.urls.public`.
"""

from __future__ import annotations

from typing import TYPE_CHECKING

from django.urls import include, path

from ..forms import (
    EventTranslationForm,
    LanguageForm,
    LanguageTreeNodeForm,
    OfferTemplateForm,
    PageTranslationForm,
    POITranslationForm,
    RegionForm,
)
from ..models import (
    Event,
    Language,
    OfferTemplate,
    Page,
    POI,
    POICategory,
    Role,
)
from ..views import (
    analytics,
    bulk_action_views,
    chat,
    contacts,
    dashboard,
    delete_views,
    events,
    external_calendars,
    feedback,
    form_views,
    imprint,
    language_tree,
    linkcheck,
    list_views,
    media,
    organizations,
    pages,
    poi_categories,
    pois,
    push_notifications,
    region_condition,
    regions,
    release_notes,
    roles,
    settings,
    statistics,
    translations,
    users,
    utils,
)

if TYPE_CHECKING:
    from django.urls.resolvers import URLPattern

#: The media library ajax url patterns are reused twice (for the admin media library and the region media library)
media_ajax_urlpatterns: list[URLPattern] = [
    path(
        "media/",
        include(
            [
                path(
                    "directory/",
                    include(
                        [
                            path(
                                "path/",
                                media.get_directory_path_ajax,
                                name="mediacenter_directory_path",
                            ),
                            path(
                                "content/",
                                media.get_directory_content_ajax,
                                name="mediacenter_get_directory_content",
                            ),
                            path(
                                "create/",
                                media.create_directory_ajax,
                                name="mediacenter_create_directory",
                            ),
                            path(
                                "update/",
                                media.edit_directory_ajax,
                                name="mediacenter_edit_directory",
                            ),
                            path(
                                "delete/",
                                media.delete_directory_ajax,
                                name="mediacenter_delete_directory",
                            ),
                        ]
                    ),
                ),
                path(
                    "file/",
                    include(
                        [
                            path(
                                "upload/",
                                media.upload_file_ajax,
                                name="mediacenter_upload_file",
                            ),
                            path(
                                "edit/",
                                media.edit_file_ajax,
                                name="mediacenter_edit_file",
                            ),
                            path(
                                "move/",
                                media.move_file_ajax,
                                name="mediacenter_move_file",
                            ),
                            path(
                                "delete/",
                                media.delete_file_ajax,
                                name="mediacenter_delete_file",
                            ),
                            path(
                                "replace/",
                                media.replace_file_ajax,
                                name="mediacenter_replace_file",
                            ),
                            path(
                                "get-usages/",
                                media.get_file_usages_ajax,
                                name="mediacenter_get_file_usages",
                            ),
                        ]
                    ),
                ),
                path(
                    "search/",
                    media.get_query_search_results_ajax,
                    name="mediacenter_get_search_result",
                ),
                path(
                    "filter/unused/",
                    media.get_unused_media_files_ajax,
                    name="mediacenter_filter_unused_media_files",
                ),
            ]
        ),
    ),
]

#: The user setting url patterns are reused twice (for the staff area and the region area)
user_settings_urlpatterns: list[URLPattern] = [
    path(
        "user-settings/",
        include(
            [
                path("", settings.UserSettingsView.as_view(), name="user_settings"),
                path(
                    "totp/",
                    include(
                        [
                            path(
                                "",
                                settings.TOTPRegisterView.as_view(),
                                name="register_totp",
                            ),
                        ]
                    ),
                ),
                path(
                    "totp/",
                    include(
                        [
                            path(
                                "delete/",
                                settings.TOTPDeleteView.as_view(),
                                name="delete_totp",
                            ),
                        ]
                    ),
                ),
                path(
                    "mfa/",
                    include(
                        [
                            path(
                                "authenticate/",
                                settings.AuthenticateModifyMfaView.as_view(),
                                name="authenticate_modify_mfa",
                            ),
                            path(
                                "get-challenge/",
                                settings.GetMfaChallengeView.as_view(),
                                name="get_mfa_challenge",
                            ),
                            path(
                                "register/",
                                settings.RegisterUserFidoKeyView.as_view(),
                                name="register_new_fido_key",
                            ),
                            path(
                                "delete/<int:key_id>/",
                                settings.DeleteUserFidoKeyView.as_view(),
                                name="delete_fido_key",
                            ),
                        ]
                    ),
                ),
            ]
        ),
    ),
]

#: The page order table url patterns are reused twice (for root pages and for subpages)
page_order_table_urlpatterns: list[URLPattern] = [
    path(
        "",
        pages.get_page_order_table_ajax,
        name="get_page_order_table_ajax",
    ),
    path(
        "page-<int:page_id>/",
        pages.get_page_order_table_ajax,
        name="get_page_order_table_ajax",
    ),
]

#: The url patterns of this module (see :doc:`django:topics/http/urls`)
urlpatterns: list[URLPattern] = [
    path(
        "admin-dashboard/",
        dashboard.AdminDashboardView.as_view(),
        name="admin_dashboard",
    ),
    path(
        "region-condition/",
        include(
            [
                path(
                    "",
                    region_condition.RegionConditionView.as_view(),
                    name="region_condition",
                ),
                path(
                    "export/<str:file_format>/",
                    region_condition.export_region_conditions,
                    name="export_region_conditions",
                ),
            ]
        ),
    ),
    path(
        "linkcheck/",
        include(
            [
                path(
                    "",
                    linkcheck.LinkcheckRedirectView.as_view(),
                    name="linkcheck_landing",
                ),
                path(
                    "<slug:url_filter>/",
                    include(
                        [
                            path(
                                "",
                                linkcheck.LinkcheckListView.as_view(),
                                name="linkcheck",
                            ),
                            path(
                                "<int:url_id>/",
                                linkcheck.LinkcheckListView.as_view(),
                                name="edit_url",
                            ),
                        ]
                    ),
                ),
                path(
                    "search_and_replace_link",
                    linkcheck.LinkReplaceView.as_view(),
                    name="search_and_replace_link",
                ),
            ]
        ),
    ),
    path(
        "regions/",
        include(
            [
                path("", regions.RegionListView.as_view(), name="regions"),
                path(
                    "new/",
                    form_views.CustomCreateView.as_view(form_class=RegionForm),
                    name="new_region",
                ),
                path(
                    "<slug>/",
                    include(
                        [
                            path(
                                "edit/",
                                regions.RegionUpdateView.as_view(),
                                name="edit_region",
                            ),
                            path(
                                "delete/", regions.delete_region, name="delete_region"
                            ),
                        ]
                    ),
                ),
            ]
        ),
    ),
    path("media-library/", media.AdminMediaListView.as_view(), name="media_admin"),
    path(
        "languages/",
        include(
            [
                path(
                    "",
                    list_views.ModelListView.as_view(
                        queryset=Language.objects.all().prefetch_related(
                            "language_tree_nodes"
                        )
                    ),
                    name="languages",
                ),
                path(
                    "new/",
                    form_views.CustomCreateView.as_view(form_class=LanguageForm),
                    name="new_language",
                ),
                path(
                    "<slug:slug>/",
                    include(
                        [
                            path(
                                "edit/",
                                form_views.CustomUpdateView.as_view(
                                    form_class=LanguageForm
                                ),
                                name="edit_language",
                            ),
                            path(
                                "delete/",
                                delete_views.CustomDeleteView.as_view(
                                    model=Language,
                                ),
                                name="delete_language",
                            ),
                        ]
                    ),
                ),
            ]
        ),
    ),
    path(
        "users/",
        include(
            [
                path("", users.UserListView.as_view(), name="users"),
                path("new/", users.UserFormView.as_view(), name="new_user"),
                path(
                    "<int:user_id>/",
                    include(
                        [
                            path(
                                "edit/",
                                users.UserFormView.as_view(),
                                name="edit_user",
                            ),
                            path("delete/", users.delete_user, name="delete_user"),
                            path(
                                "resend-activation-link",
                                users.resend_activation_link,
                                name="resend_activation_link",
                            ),
                        ]
                    ),
                ),
            ]
        ),
    ),
    path(
        "location-categories/",
        include(
            [
                path(
                    "",
                    list_views.ModelListView.as_view(
                        model=POICategory,
                        extra_context={"languages": Language.objects.all()},
                    ),
                    name="poicategories",
                ),
                path(
                    "new/",
                    poi_categories.POICategoryCreateView.as_view(),
                    name="new_poicategory",
                ),
                path(
                    "<int:pk>/",
                    include(
                        [
                            path(
                                "edit/",
                                poi_categories.POICategoryUpdateView.as_view(),
                                name="edit_poicategory",
                            ),
                            path(
                                "delete/",
                                poi_categories.POICategoryDeleteView.as_view(),
                                name="delete_poicategory",
                            ),
                        ]
                    ),
                ),
            ]
        ),
    ),
    path(
        "roles/",
        include(
            [
                path("", list_views.ModelListView.as_view(model=Role), name="roles"),
                path("new/", roles.RoleFormView.as_view(), name="new_role"),
                path(
                    "<int:role_id>/",
                    include(
                        [
                            path(
                                "edit/",
                                roles.RoleFormView.as_view(),
                                name="edit_role",
                            ),
                            path(
                                "delete/",
                                roles.RoleFormView.as_view(),
                                name="delete_role",
                            ),
                        ]
                    ),
                ),
            ]
        ),
    ),
    path(
        "feedback/",
        include(
            [
                path(
                    "",
                    feedback.AdminFeedbackListView.as_view(),
                    name="admin_feedback",
                ),
                path(
                    "archived/",
                    feedback.AdminFeedbackListView.as_view(archived=True),
                    name="admin_feedback_archived",
                ),
                path(
                    "mark-read/",
                    feedback.mark_admin_feedback_as_read,
                    name="mark_admin_feedback_as_read",
                ),
                path(
                    "mark-unread/",
                    feedback.mark_admin_feedback_as_unread,
                    name="mark_admin_feedback_as_unread",
                ),
                path(
                    "archive/",
                    feedback.archive_admin_feedback,
                    name="archive_admin_feedback",
                ),
                path(
                    "restore/",
                    feedback.restore_admin_feedback,
                    name="restore_admin_feedback",
                ),
                path(
                    "delete/",
                    feedback.delete_admin_feedback,
                    name="delete_admin_feedback",
                ),
            ]
        ),
    ),
    path(
        "offer-templates/",
        include(
            [
                path(
                    "",
                    list_views.ModelListView.as_view(
                        queryset=OfferTemplate.objects.all().prefetch_related("regions")
                    ),
                    name="offertemplates",
                ),
                path(
                    "new/",
                    form_views.CustomCreateView.as_view(form_class=OfferTemplateForm),
                    name="new_offertemplate",
                ),
                path(
                    "<slug:slug>/",
                    include(
                        [
                            path(
                                "edit/",
                                form_views.CustomUpdateView.as_view(
                                    form_class=OfferTemplateForm
                                ),
                                name="edit_offertemplate",
                            ),
                            path(
                                "delete/",
                                delete_views.CustomDeleteView.as_view(
                                    model=OfferTemplate, protect_manytomany="regions"
                                ),
                                name="delete_offertemplate",
                            ),
                        ]
                    ),
                ),
            ]
        ),
    ),
    path("", include(user_settings_urlpatterns)),
    path(
        "ajax/",
        include(
            [
                path("", include(media_ajax_urlpatterns)),
                path(
                    "chat/",
                    include(
                        [
                            path(
                                "send-message/",
                                chat.send_chat_message,
                                name="send_chat_message",
                            ),
                            path(
                                "delete-message/<int:message_id>/",
                                chat.delete_chat_message,
                                name="delete_chat_message",
                            ),
                        ]
                    ),
                ),
                path(
                    "search/",
                    utils.search_content_ajax,
                    name="search_content_ajax",
                ),
            ]
        ),
    ),
    path(
        "release-notes/",
        release_notes.ReleaseNotesView.as_view(),
        name="release_notes",
    ),
    path(
        "<region_slug>/",
        include(
            [
                path("", dashboard.DashboardView.as_view(), name="dashboard"),
                path(
                    "ajax/",
                    include(
                        [
                            path(
                                "dashboard/",
                                include(
                                    [
                                        path(
                                            "broken-links/",
                                            dashboard.DashboardView.get_broken_links_context,
                                            name="get_broken_links_ajax",
                                        )
                                    ]
                                ),
                            ),
                            path("", include(media_ajax_urlpatterns)),
                            path(
                                "<slug:language_slug>/",
                                include(
                                    [
                                        path(
                                            "<int:page_id>/",
                                            include(
                                                [
                                                    path(
                                                        "cancel-translation-process/",
                                                        pages.cancel_translation_process_ajax,
                                                        name="cancel_translation_process_ajax",
                                                    ),
                                                    path(
                                                        "preview/",
                                                        pages.preview_page_ajax,
                                                        name="preview_page_ajax",
                                                    ),
                                                    path(
                                                        "content/",
                                                        pages.get_page_content_ajax,
                                                        name="get_page_content_ajax",
                                                    ),
                                                ]
                                            ),
                                        ),
                                        path(
                                            "search/",
                                            utils.search_content_ajax,
                                            name="search_content_ajax",
                                        ),
                                        path(
                                            "<model_type>/slugify/",
                                            utils.slugify_ajax,
                                            name="slugify_ajax",
                                        ),
                                        path(
                                            "<model_type>/word_count/",
                                            utils.build_json_for_machine_translation,
                                            name="word_count",
                                        ),
                                    ]
                                ),
                            ),
                            path(
                                "render/",
                                include(
                                    [
                                        path(
                                            "<slug:language_slug>/",
                                            include(
                                                [
                                                    path(
                                                        "mirrored-page-field/",
                                                        pages.render_mirrored_page_field,
                                                        name="render_mirrored_page_field",
                                                    ),
                                                    path(
                                                        "partial-page-tree/",
                                                        pages.render_partial_page_tree_views,
                                                        name="get_page_tree_ajax",
                                                    ),
                                                ]
                                            ),
                                        ),
                                        path(
                                            "page-order-table/",
                                            include(
                                                [
                                                    path(
                                                        "",
                                                        include(
                                                            page_order_table_urlpatterns
                                                        ),
                                                    ),
                                                    path(
                                                        "parent-<int:parent_id>/",
                                                        include(
                                                            page_order_table_urlpatterns
                                                        ),
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                            ),
                            path(
                                "chat/",
                                include(
                                    [
                                        path(
                                            "send-message/",
                                            chat.send_chat_message,
                                            name="send_chat_message",
                                        ),
                                        path(
                                            "delete-message/<int:message_id>/",
                                            chat.delete_chat_message,
                                            name="delete_chat_message",
                                        ),
                                    ]
                                ),
                            ),
                            path(
                                "get_hix_score/",
                                utils.get_hix_score,
                                name="get_hix_score",
                            ),
                            path(
                                "statistics/",
                                include(
                                    [
                                        path(
                                            "total-views/",
                                            statistics.get_total_visits_ajax,
                                            name="statistics_total_visits",
                                        ),
                                        path(
                                            "update-chart/",
                                            statistics.get_visits_per_language_ajax,
                                            name="statistics_visits_per_language",
                                        ),
                                    ]
                                ),
                            ),
                            path(
                                "page-permission/",
                                include(
                                    [
                                        path(
                                            "grant/",
                                            pages.grant_page_permission_ajax,
                                            name="grant_page_permission_ajax",
                                        ),
                                        path(
                                            "revoke/",
                                            pages.revoke_page_permission_ajax,
                                            name="revoke_page_permission_ajax",
                                        ),
                                    ]
                                ),
                            ),
                            path(
                                "content-edit-lock/",
                                include(
                                    [
                                        path(
                                            "heartbeat/",
                                            utils.content_edit_lock_heartbeat,
                                            name="content_edit_lock_heartbeat",
                                        ),
                                        path(
                                            "release/",
                                            utils.content_edit_lock_release,
                                            name="content_edit_lock_release",
                                        ),
                                    ]
                                ),
                            ),
                            path(
                                "search-poi/",
                                events.search_poi_ajax,
                                name="search_poi_ajax",
                            ),
                            path(
                                "search/",
                                utils.search_content_ajax,
                                name="search_content_ajax",
                            ),
                            path(
                                "dismiss-tutorial/<slug:slug>/",
                                settings.DismissTutorial.as_view(),
                                name="dismiss_tutorial",
                            ),
                            path(
                                "locations/auto-complete-address/",
                                pois.auto_complete_address,
                                name="auto_complete_poi_address",
                            ),
                            path(
                                "locations/get-address-from-coordinates/",
                                pois.get_address_from_coordinates,
                                name="get_address_from_coordinates",
                            ),
                        ]
                    ),
                ),
                path(
                    "analytics/",
                    include(
                        [
                            path(
                                "statistics/",
                                statistics.AnalyticsView.as_view(),
                                name="statistics",
                            ),
                            path(
                                "translation-coverage/",
                                analytics.TranslationCoverageView.as_view(),
                                name="translation_coverage",
                            ),
                            path(
                                "linkcheck/",
                                include(
                                    [
                                        path(
                                            "",
                                            linkcheck.LinkcheckRedirectView.as_view(),
                                            name="linkcheck_landing",
                                        ),
                                        path(
                                            "stats",
                                            linkcheck.LinkcheckStatsView.as_view(),
                                            name="linkcheck_stats",
                                        ),
                                        path(
                                            "<slug:url_filter>/",
                                            include(
                                                [
                                                    path(
                                                        "",
                                                        linkcheck.LinkcheckListView.as_view(),
                                                        name="linkcheck",
                                                    ),
                                                    path(
                                                        "<int:url_id>/",
                                                        linkcheck.LinkcheckListView.as_view(),
                                                        name="edit_url",
                                                    ),
                                                ]
                                            ),
                                        ),
                                        path(
                                            "search_and_replace_link",
                                            linkcheck.LinkReplaceView.as_view(),
                                            name="search_and_replace_link",
                                        ),
                                    ]
                                ),
                            ),
                        ]
                    ),
                ),
                path(
                    "translations/",
                    include(
                        [
                            path(
                                "manage/",
                                translations.TranslationsManagementView.as_view(),
                                name="translations_management",
                            ),
                        ],
                    ),
                ),
                path(
                    "external-calendars/",
                    include(
                        [
                            path(
                                "",
                                external_calendars.ExternalCalendarList.as_view(),
                                name="external_calendar_list",
                            ),
                            path(
                                "new/",
                                external_calendars.ExternalCalendarFormView.as_view(),
                                name="new_external_calendar",
                            ),
                            path(
                                "<int:calendar_id>/",
                                include(
                                    [
                                        path(
                                            "edit/",
                                            external_calendars.ExternalCalendarFormView.as_view(),
                                            name="edit_external_calendar",
                                        ),
                                        path(
                                            "delete/",
                                            external_calendars.delete_external_calendar,
                                            name="delete_external_calendar",
                                        ),
                                    ]
                                ),
                            ),
                        ]
                    ),
                ),
                path(
                    "pages/",
                    include(
                        [
                            path("", pages.PageTreeView.as_view(), name="pages"),
                            path(
                                "<slug:language_slug>/",
                                include(
                                    [
                                        path(
                                            "",
                                            pages.PageTreeView.as_view(),
                                            name="pages",
                                        ),
                                        path(
                                            "new/",
                                            pages.PageFormView.as_view(),
                                            name="new_page",
                                        ),
                                        path(
                                            "archived/",
                                            pages.PageTreeView.as_view(archived=True),
                                            name="archived_pages",
                                        ),
                                        path(
                                            "xliff/",
                                            include(
                                                [
                                                    path(
                                                        "download/",
                                                        pages.ExportXliffView.as_view(
                                                            prefetch_translations=True
                                                        ),
                                                        name="download_xliff",
                                                    ),
                                                    path(
                                                        "only-public/",
                                                        pages.ExportXliffView.as_view(
                                                            only_public=True,
                                                            prefetch_public_translations=True,
                                                        ),
                                                        name="download_xliff_only_public",
                                                    ),
                                                    path(
                                                        "multiple-languages/",
                                                        pages.ExportMultiLanguageXliffView.as_view(),
                                                        name="download_xliff_multiple_languages",
                                                    ),
                                                    path(
                                                        "upload/",
                                                        pages.upload_xliff,
                                                        name="upload_xliff",
                                                    ),
                                                    path(
                                                        "import/<uuid:xliff_dir>/",
                                                        pages.PageXliffImportView.as_view(),
                                                        name="import_xliff",
                                                    ),
                                                ],
                                            ),
                                        ),
                                        path(
                                            "export/",
                                            pages.GeneratePdfView.as_view(),
                                            name="export_pdf",
                                        ),
                                        path(
                                            "bulk-archive/",
                                            bulk_action_views.BulkArchiveView.as_view(
                                                model=Page,
                                            ),
                                            name="bulk_archive_pages",
                                        ),
                                        path(
                                            "bulk-restore/",
                                            bulk_action_views.BulkRestoreView.as_view(
                                                model=Page,
                                            ),
                                            name="bulk_restore_pages",
                                        ),
                                        path(
                                            "machine-translate/",
                                            bulk_action_views.BulkMachineTranslationView.as_view(
                                                model=Page,
                                                form=PageTranslationForm,
                                            ),
                                            name="machine_translation_pages",
                                        ),
                                        path(
                                            "bulk-publish/",
                                            bulk_action_views.BulkPublishingView.as_view(
                                                model=Page,
                                            ),
                                            name="publish_multiple_pages",
                                        ),
                                        path(
                                            "bulk-draft/",
                                            bulk_action_views.BulkDraftingView.as_view(
                                                model=Page,
                                            ),
                                            name="draft_multiple_pages",
                                        ),
                                        path(
                                            "bulk-cancel-translation-process/",
                                            pages.CancelTranslationProcess.as_view(
                                                model=Page,
                                            ),
                                            name="cancel_translation_process",
                                        ),
                                        path(
                                            "<int:page_id>/",
                                            include(
                                                [
                                                    path(
                                                        "edit/",
                                                        include(
                                                            [
                                                                path(
                                                                    "",
                                                                    pages.PageFormView.as_view(),
                                                                    name="edit_page",
                                                                ),
                                                                path(
                                                                    "side-by-side/",
                                                                    pages.PageSideBySideView.as_view(),
                                                                    name="sbs_edit_page",
                                                                ),
                                                            ],
                                                        ),
                                                    ),
                                                    path(
                                                        "versions/",
                                                        include(
                                                            [
                                                                path(
                                                                    "",
                                                                    pages.PageVersionView.as_view(),
                                                                    name="page_versions",
                                                                ),
                                                                path(
                                                                    "<int:selected_version>/",
                                                                    pages.PageVersionView.as_view(),
                                                                    name="page_versions",
                                                                ),
                                                            ],
                                                        ),
                                                    ),
                                                    path(
                                                        "archive/",
                                                        pages.archive_page,
                                                        name="archive_page",
                                                    ),
                                                    path(
                                                        "restore/",
                                                        pages.restore_page,
                                                        name="restore_page",
                                                    ),
                                                    path(
                                                        "refresh/",
                                                        pages.refresh_date,
                                                        name="refresh_page",
                                                    ),
                                                    path(
                                                        "delete/",
                                                        pages.delete_page,
                                                        name="delete_page",
                                                    ),
                                                    # warning: the move url is also hardcoded in src/integreat_cms/static/js/tree_drag_and_drop.js
                                                    path(
                                                        "move/<int:target_id>/<slug:position>/",
                                                        pages.move_page,
                                                        name="move_page",
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                            ),
                        ]
                    ),
                ),
                path(
                    "imprint/",
                    include(
                        [
                            path(
                                "",
                                imprint.ImprintFormView.as_view(),
                                name="edit_imprint",
                            ),
                            path(
                                "<slug:language_slug>/",
                                include(
                                    [
                                        path(
                                            "edit/",
                                            include(
                                                [
                                                    path(
                                                        "",
                                                        imprint.ImprintFormView.as_view(),
                                                        name="edit_imprint",
                                                    ),
                                                    path(
                                                        "side-by-side/",
                                                        imprint.ImprintSideBySideView.as_view(),
                                                        name="sbs_edit_imprint",
                                                    ),
                                                ],
                                            ),
                                        ),
                                        path(
                                            "versions/",
                                            include(
                                                [
                                                    path(
                                                        "",
                                                        imprint.ImprintVersionView.as_view(),
                                                        name="imprint_versions",
                                                    ),
                                                    path(
                                                        "<int:selected_version>",
                                                        imprint.ImprintVersionView.as_view(),
                                                        name="imprint_versions",
                                                    ),
                                                ],
                                            ),
                                        ),
                                    ]
                                ),
                            ),
                            path(
                                "delete/",
                                imprint.delete_imprint,
                                name="delete_imprint",
                            ),
                        ]
                    ),
                ),
                path(
                    "organizations/",
                    include(
                        [
                            path(
                                "",
                                organizations.OrganizationListView.as_view(),
                                name="organizations",
                            ),
                            path(
                                "new/",
                                organizations.OrganizationFormView.as_view(),
                                name="new_organization",
                            ),
                            path(
                                "archived/",
                                organizations.OrganizationListView.as_view(
                                    archived=True
                                ),
                                name="archived_organizations",
                            ),
                            path(
                                "bulk-archive/",
                                organizations.ArchiveBulkAction.as_view(),
                                name="bulk_archive_organization",
                            ),
                            path(
                                "bulk-restore/",
                                organizations.RestoreBulkAction.as_view(),
                                name="bulk_restore_organization",
                            ),
                            path(
                                "bulk-delete/",
                                organizations.DeleteBulkAction.as_view(),
                                name="bulk_delete_organization",
                            ),
                            path(
                                "<int:organization_id>/",
                                include(
                                    [
                                        path(
                                            "edit/",
                                            organizations.OrganizationFormView.as_view(),
                                            name="edit_organization",
                                        ),
                                        path(
                                            "delete/",
                                            organizations.delete,
                                            name="delete_organization",
                                        ),
                                        path(
                                            "archive/",
                                            organizations.archive,
                                            name="archive_organization",
                                        ),
                                        path(
                                            "restore/",
                                            organizations.restore,
                                            name="restore_organization",
                                        ),
                                    ]
                                ),
                            ),
                        ]
                    ),
                ),
                path(
                    "events/",
                    include(
                        [
                            path("", events.EventListView.as_view(), name="events"),
                            path(
                                "<slug:language_slug>/",
                                include(
                                    [
                                        path(
                                            "",
                                            events.EventListView.as_view(),
                                            name="events",
                                        ),
                                        path(
                                            "archived/",
                                            events.EventListView.as_view(archived=True),
                                            name="events_archived",
                                        ),
                                        path(
                                            "new/",
                                            events.EventFormView.as_view(),
                                            name="new_event",
                                        ),
                                        path(
                                            "machine-translate/",
                                            bulk_action_views.BulkMachineTranslationView.as_view(
                                                model=Event, form=EventTranslationForm
                                            ),
                                            name="machine_translation_events",
                                        ),
                                        path(
                                            "bulk-archive/",
                                            bulk_action_views.BulkArchiveView.as_view(
                                                model=Event
                                            ),
                                            name="bulk_archive_events",
                                        ),
                                        path(
                                            "bulk-restore/",
                                            bulk_action_views.BulkRestoreView.as_view(
                                                model=Event
                                            ),
                                            name="bulk_restore_events",
                                        ),
                                        path(
                                            "bulk-publish/",
                                            bulk_action_views.BulkPublishingView.as_view(
                                                model=Event,
                                            ),
                                            name="publish_multiple_events",
                                        ),
                                        path(
                                            "bulk-draft/",
                                            bulk_action_views.BulkDraftingView.as_view(
                                                model=Event,
                                            ),
                                            name="draft_multiple_events",
                                        ),
                                        path(
                                            "<int:event_id>/",
                                            include(
                                                [
                                                    path(
                                                        "edit/",
                                                        events.EventFormView.as_view(),
                                                        name="edit_event",
                                                    ),
                                                    path(
                                                        "copy/",
                                                        events.copy,
                                                        name="copy_event",
                                                    ),
                                                    path(
                                                        "archive/",
                                                        events.archive,
                                                        name="archive_event",
                                                    ),
                                                    path(
                                                        "restore/",
                                                        events.restore,
                                                        name="restore_event",
                                                    ),
                                                    path(
                                                        "delete/",
                                                        events.delete,
                                                        name="delete_event",
                                                    ),
                                                    path(
                                                        "versions/",
                                                        include(
                                                            [
                                                                path(
                                                                    "",
                                                                    events.EventVersionView.as_view(),
                                                                    name="event_versions",
                                                                ),
                                                                path(
                                                                    "<int:selected_version>/",
                                                                    events.EventVersionView.as_view(),
                                                                    name="event_versions",
                                                                ),
                                                            ],
                                                        ),
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                            ),
                        ]
                    ),
                ),
                path(
                    "pois/",
                    include(
                        [
                            path("", pois.POIListView.as_view(), name="pois"),
                            path(
                                "<slug:language_slug>/",
                                include(
                                    [
                                        path(
                                            "",
                                            pois.POIListView.as_view(),
                                            name="pois",
                                        ),
                                        path(
                                            "archived/",
                                            pois.POIListView.as_view(archived=True),
                                            name="archived_pois",
                                        ),
                                        path(
                                            "new/",
                                            pois.POIFormView.as_view(),
                                            name="new_poi",
                                        ),
                                        path(
                                            "machine-translate/",
                                            bulk_action_views.BulkMachineTranslationView.as_view(
                                                model=POI, form=POITranslationForm
                                            ),
                                            name="machine_translation_pois",
                                        ),
                                        path(
                                            "bulk-archive/",
                                            bulk_action_views.BulkArchiveView.as_view(
                                                model=POI
                                            ),
                                            name="bulk_archive_pois",
                                        ),
                                        path(
                                            "bulk-restore/",
                                            bulk_action_views.BulkRestoreView.as_view(
                                                model=POI
                                            ),
                                            name="bulk_restore_pois",
                                        ),
                                        path(
                                            "bulk-publish/",
                                            bulk_action_views.BulkPublishingView.as_view(
                                                model=POI,
                                            ),
                                            name="publish_multiple_pois",
                                        ),
                                        path(
                                            "bulk-draft/",
                                            bulk_action_views.BulkDraftingView.as_view(
                                                model=POI,
                                            ),
                                            name="draft_multiple_pois",
                                        ),
                                        path(
                                            "show-poi-form-ajax/<str:poi_title>/",
                                            pois.POIFormAjaxView.as_view(),
                                            name="show_poi_form_ajax",
                                        ),
                                        path(
                                            "create-poi-ajax/",
                                            pois.POIFormAjaxView.as_view(),
                                            name="create_poi_ajax",
                                        ),
                                        path(
                                            "<int:poi_id>/",
                                            include(
                                                [
                                                    path(
                                                        "view/",
                                                        pois.view_poi,
                                                        name="view_poi",
                                                    ),
                                                    path(
                                                        "edit/",
                                                        pois.POIFormView.as_view(),
                                                        name="edit_poi",
                                                    ),
                                                    path(
                                                        "archive/",
                                                        pois.archive_poi,
                                                        name="archive_poi",
                                                    ),
                                                    path(
                                                        "restore/",
                                                        pois.restore_poi,
                                                        name="restore_poi",
                                                    ),
                                                    path(
                                                        "delete/",
                                                        pois.delete_poi,
                                                        name="delete_poi",
                                                    ),
                                                    path(
                                                        "versions/",
                                                        include(
                                                            [
                                                                path(
                                                                    "",
                                                                    pois.POIVersionView.as_view(),
                                                                    name="poi_versions",
                                                                ),
                                                                path(
                                                                    "<int:selected_version>/",
                                                                    pois.POIVersionView.as_view(),
                                                                    name="poi_versions",
                                                                ),
                                                            ],
                                                        ),
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                            ),
                        ]
                    ),
                ),
                path(
                    "contact/",
                    include(
                        [
                            path(
                                "",
                                contacts.ContactListView.as_view(),
                                name="contacts",
                            ),
                            path(
                                "archived/",
                                contacts.ContactListView.as_view(archived=True),
                                name="archived_contacts",
                            ),
                            path(
                                "new/",
                                contacts.ContactFormView.as_view(),
                                name="new_contact",
                            ),
                            path(
                                "bulk-archive/",
                                contacts.ArchiveContactBulkAction.as_view(),
                                name="bulk_archive_contacts",
                            ),
                            path(
                                "bulk-restore/",
                                contacts.RestoreContactBulkAction.as_view(),
                                name="bulk_restore_contacts",
                            ),
                            path(
                                "bulk-delete/",
                                contacts.DeleteContactBulkAction.as_view(),
                                name="bulk_delete_contacts",
                            ),
                            path(
                                "<int:contact_id>/",
                                include(
                                    [
                                        path(
                                            "edit/",
                                            contacts.ContactFormView.as_view(),
                                            name="edit_contact",
                                        ),
                                        path(
                                            "copy/",
                                            contacts.copy_contact,
                                            name="copy_contact",
                                        ),
                                        path(
                                            "archive/",
                                            contacts.archive_contact,
                                            name="archive_contact",
                                        ),
                                        path(
                                            "restore/",
                                            contacts.restore_contact,
                                            name="restore_contact",
                                        ),
                                        path(
                                            "delete/",
                                            contacts.delete_contact,
                                            name="delete_contact",
                                        ),
                                    ]
                                ),
                            ),
                        ]
                    ),
                ),
                path(
                    "feedback/",
                    include(
                        [
                            path(
                                "",
                                feedback.RegionFeedbackListView.as_view(),
                                name="region_feedback",
                            ),
                            path(
                                "archived/",
                                feedback.RegionFeedbackListView.as_view(archived=True),
                                name="region_feedback_archived",
                            ),
                            path(
                                "mark-read/",
                                feedback.mark_region_feedback_as_read,
                                name="mark_region_feedback_as_read",
                            ),
                            path(
                                "mark-unread/",
                                feedback.mark_region_feedback_as_unread,
                                name="mark_region_feedback_as_unread",
                            ),
                            path(
                                "archive/",
                                feedback.archive_region_feedback,
                                name="archive_region_feedback",
                            ),
                            path(
                                "restore/",
                                feedback.restore_region_feedback,
                                name="restore_region_feedback",
                            ),
                            path(
                                "delete/",
                                feedback.delete_region_feedback,
                                name="delete_region_feedback",
                            ),
                            path(
                                "export/<str:file_format>/",
                                feedback.export_region_feedback,
                                name="export_region_feedback",
                            ),
                        ]
                    ),
                ),
                path(
                    "push-notifications/",
                    include(
                        [
                            path(
                                "",
                                push_notifications.PushNotificationListView.as_view(),
                                name="push_notifications",
                            ),
                            path(
                                "<slug:language_slug>/",
                                include(
                                    [
                                        path(
                                            "",
                                            push_notifications.PushNotificationListView.as_view(),
                                            name="push_notifications",
                                        ),
                                        path(
                                            "templates/",
                                            push_notifications.PushNotificationListView.as_view(
                                                templates=True
                                            ),
                                            name="push_notifications_templates",
                                        ),
                                        path(
                                            "new/",
                                            push_notifications.PushNotificationFormView.as_view(),
                                            name="new_push_notification",
                                        ),
                                        path(
                                            "<int:push_notification_id>/",
                                            include(
                                                [
                                                    path(
                                                        "edit/",
                                                        push_notifications.PushNotificationFormView.as_view(),
                                                        name="edit_push_notification",
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                            ),
                        ]
                    ),
                ),
                path(
                    "language-tree/",
                    include(
                        [
                            path(
                                "",
                                language_tree.LanguageTreeView.as_view(),
                                name="languagetreenodes",
                            ),
                            path(
                                "new/",
                                language_tree.LanguageTreeNodeCreateView.as_view(),
                                name="new_languagetreenode",
                            ),
                            path(
                                "bulk-make-visible/",
                                language_tree.BulkMakeVisibleView.as_view(),
                                name="bulk_make_languagetreenodes_visible",
                            ),
                            path(
                                "bulk-hide/",
                                language_tree.BulkHideView.as_view(),
                                name="bulk_hide_languagetreenodes",
                            ),
                            path(
                                "bulk-activate/",
                                language_tree.BulkActivateView.as_view(),
                                name="bulk_activate_languagetreenodes",
                            ),
                            path(
                                "bulk-disable/",
                                language_tree.BulkDisableView.as_view(),
                                name="bulk_disable_languagetreenodes",
                            ),
                            path(
                                "<int:pk>/",
                                include(
                                    [
                                        path(
                                            "edit/",
                                            language_tree.LanguageTreeNodeUpdateView.as_view(
                                                form_class=LanguageTreeNodeForm
                                            ),
                                            name="edit_languagetreenode",
                                        ),
                                        path(
                                            "delete/",
                                            language_tree.delete_language_tree_node,
                                            name="delete_languagetreenode",
                                        ),
                                        # warning: the move url is also hardcoded in src/integreat_cms/static/js/tree_drag_and_drop.js
                                        path(
                                            "move/<int:target_id>/<slug:target_position>/",
                                            language_tree.move_language_tree_node,
                                            name="move_languagetreenode",
                                        ),
                                    ]
                                ),
                            ),
                        ]
                    ),
                ),
                path("media-library/", media.MediaListView.as_view(), name="media"),
                path(
                    "users/",
                    include(
                        [
                            path(
                                "",
                                users.RegionUserListView.as_view(),
                                name="region_users",
                            ),
                            path(
                                "new/",
                                users.RegionUserFormView.as_view(),
                                name="new_region_user",
                            ),
                            path(
                                "<int:user_id>/",
                                include(
                                    [
                                        path(
                                            "edit/",
                                            users.RegionUserFormView.as_view(),
                                            name="edit_region_user",
                                        ),
                                        path(
                                            "delete/",
                                            users.delete_region_user,
                                            name="delete_region_user",
                                        ),
                                        path(
                                            "resend-activation-link/",
                                            users.resend_activation_link_region,
                                            name="resend_activation_link_region",
                                        ),
                                    ]
                                ),
                            ),
                        ]
                    ),
                ),
                path("", include(user_settings_urlpatterns)),
                path(
                    "release-notes/",
                    release_notes.ReleaseNotesView.as_view(),
                    name="release_notes",
                ),
            ]
        ),
    ),
]
