import json
import os
import subprocess
import sys
from pathlib import Path
from subprocess import SubprocessError
from typing import Callable, Optional, Union

import ipywidgets as widgets  # type: ignore
import nbformat
import typer
from IPython.display import display
from nbformat.notebooknode import NotebookNode  # type: ignore
from pydantic import BaseModel, Field, ValidationError

from garden_ai import GardenConstants

NOTEBOOK_DISPLAY_METADATA_CELL = (
    '"""\n'
    "This cell is auto-generated by Garden.\n"
    "You can use this widget to edit your notebooks metadata. \n"
    "That way, the next time you run this notebook, Garden will start it with the same environment.\n"
    "Any changes made to your notebook's metadata using the widget will be saved when the notebook is saved.\n\n"
    "Notebook metadata fields:\n"
    "   - Global Garden DOI: If specified, `garden-ai notebook publish` will add all entrypoints in this notebook to the Garden with the given DOI.\n"
    "     If you want to specify a different Garden DOI for individual entrypoints, you can provide that entrypoint's\n"
    "     'entrypoint' decorator with the optional 'garden_doi' argument. Providing the decorator with a DOI\n"
    "     will override the Global DOI for that specific entrypoint.\n"
    "   - Base image name: The name of the garden base image you want to start this notebook with.\n"
    "     To see a list of the available Garden base images, see the dropdown menu under 'Base Image' below or \n"
    "     use 'garden-ai notebook list-premade-images'\n"
    "   - Requirements: Any additional requirements that should be installed in this notebook's container.\n"
    "     After making changes to your notebook's requirements, the widget will show a 'Install new requirements' button\n"
    "     that installs the new requirements to the container and updates your local requirements file if one was provided.\n"
    '"""\n\n'
    "from garden_ai.notebook_metadata import display_metadata_widget\n"
    "display_metadata_widget()"
)

METADATA_CELL_TAG = "garden_display_metadata_cell"


class RequirementsData(BaseModel):
    file_format: str = "pip"
    contents: Union[dict, list] = Field(default_factory=list)


class NotebookMetadata(BaseModel):
    global_notebook_doi: Optional[str]
    notebook_image_name: Optional[str]
    notebook_image_uri: Optional[str]
    notebook_requirements: RequirementsData = Field(default_factory=RequirementsData)


def add_notebook_metadata(
    notebook_path: Path,
):
    """
    Adds metadata editor widget cell to top of the notebook if missing
    Adds empty `garden_metadata` dict to the notebook's metadata if missing
    """
    ntbk = _read_notebook(notebook_path)

    # If metadata widget cell does not exist, add to top of notebook
    if not _has_metadata_cell_tag(ntbk):
        new_cell = nbformat.v4.new_code_cell(NOTEBOOK_DISPLAY_METADATA_CELL)
        del new_cell["id"]
        new_cell["metadata"] = {
            "tags": [METADATA_CELL_TAG],
        }
        ntbk.cells.insert(0, new_cell)

    # Add empty garden_metadata dict to notebooks metadata if missing
    if "garden_metadata" not in ntbk["metadata"]:
        ntbk["metadata"]["garden_metadata"] = {}
    # If some of the fields are set, dont want to write over them
    for field in list(NotebookMetadata.model_fields):
        if field not in ntbk["metadata"]["garden_metadata"]:
            ntbk["metadata"]["garden_metadata"][field] = None

    # Write updated notebook data to file
    nbformat.write(ntbk, notebook_path, version=nbformat.NO_CONVERT)


def _has_metadata_cell_tag(ntbk: nbformat.NotebookNode, tag=METADATA_CELL_TAG) -> bool:
    """Return True if cell with 'garden_display_metadata_cell' tag is found ntbk, otherwise False"""
    found_cell = False
    for cell in ntbk.cells:
        cell_tags = cell.get("metadata", {}).get("tags", [])
        if tag in cell_tags:
            found_cell = True
    return found_cell


def get_notebook_metadata(notebook_path: Path) -> NotebookMetadata:
    ntbk = _read_notebook(notebook_path)

    # Return empty NotebookMetadata if was unable to find cell source
    if "garden_metadata" not in ntbk["metadata"]:
        typer.echo("Unable to find garden metadata.")
        return NotebookMetadata(
            global_notebook_doi=None,
            notebook_image_name=None,
            notebook_image_uri=None,
        )

    try:
        return NotebookMetadata.model_validate(ntbk["metadata"]["garden_metadata"])
    except ValidationError:
        # Return empty NotebookMetadata if was unable parse the saved garden_metadata
        typer.echo("Unable to parse garden metadata cell.")
        return NotebookMetadata(
            global_notebook_doi=None,
            notebook_image_name=None,
            notebook_image_uri=None,
        )


def set_notebook_metadata(
    notebook_path: Path,
    notebook_global_doi: Optional[str],
    base_image_name: Optional[str],
    base_image_uri: str,
    requirements_data: Optional[RequirementsData],
):
    ntbk = _read_notebook(notebook_path)

    ntbk["metadata"]["garden_metadata"] = NotebookMetadata(
        global_notebook_doi=notebook_global_doi,
        notebook_image_name=base_image_name,
        notebook_image_uri=base_image_uri,
        notebook_requirements=requirements_data,
    ).model_dump()

    # Write updated notebook data to file
    nbformat.write(ntbk, notebook_path, version=nbformat.NO_CONVERT)


def read_requirements_data(requirements_path: Path) -> Optional[RequirementsData]:
    # For txt requirements files, contents is list of lines, format is pip
    if requirements_path.suffix in {".txt"}:
        file_format = "pip"
        with open(requirements_path, "r") as req_file:
            # read lines into list and strip any newlines
            file_contents = [line.replace("\n", "") for line in req_file.readlines()]
            req_file.close()
        return RequirementsData(file_format=file_format, contents=file_contents)
    else:
        typer.echo("Invalid requirements file format.")
        return None


def save_requirements_data(
    requirements_dir_path: Path, requirements_data: RequirementsData
) -> Optional[Path]:
    # Save requirements_data to requirements file in either pip or conda format
    # Returns path to new requirements file or None if was unable to write.
    file_format = requirements_data.file_format
    contents = requirements_data.contents

    if file_format == "pip":
        # requirements file is txt
        requirements_path = requirements_dir_path / "requirements.txt"
        with open(requirements_path, "w") as req_file:
            # contents is list of requirements
            file_contents = ""
            for line in contents:
                file_contents += f"{line}\n"
            req_file.write(file_contents)
        return requirements_path
    else:
        typer.echo(
            f"Invalid format for requirements data, must be either pip or conda, got {file_format}. Ignoring requirements."
        )
        return None


def make_html_popup_widget(message, alert_type="alert", background_color="#1cb4eb"):
    # create html widget popup box that has info message with a close button on it
    html_content = f"""
    <div class="{alert_type}">
        <span class="closebtn" onclick="this.parentElement.style.display='none';">&times;</span>
        {message}
    </div>
    """

    css_content = f"""
    <style>
    .alert {{
        padding: 20px;
        background-color: {background_color};
        color: white;
        margin-bottom: 15px;
        border-radius: 5px;
        position: relative;
        width: 95%;
    }}
    .closebtn {{
        margin-left: 15px;
        color: white;
        font-weight: bold;
        float: right;
        font-size: 22px;
        line-height: 20px;
        cursor: pointer;
        transition: 0.3s;
    }}
    .closebtn:hover {{
        color: black;
    }}
    </style>
    """

    return widgets.HTML(f"{css_content}{html_content}")


def _build_doi_widget(nb_meta: NotebookMetadata) -> widgets.Textarea:
    return widgets.Textarea(
        value=nb_meta.global_notebook_doi,
        placeholder="Global Garden DOI",
        continuous_update=False,
        disabled=False,
    )


def _build_base_image_widget(nb_meta: NotebookMetadata) -> widgets.Dropdown:
    return widgets.Dropdown(
        options=GardenConstants.PREMADE_IMAGES.keys(),
        value=nb_meta.notebook_image_name,
        disabled=False,
    )


def _build_reqs_widget(nb_meta: NotebookMetadata) -> widgets.Textarea:
    reqs_string = ""
    if nb_meta.notebook_requirements.file_format == "pip":
        reqs_string = "\n".join([req for req in nb_meta.notebook_requirements.contents])

    return widgets.Textarea(
        value=reqs_string,
        placeholder="Requirements",
        layout=widgets.Layout(width="100%", height="80px"),
        continuous_update=False,
        disabled=False,
    )


def _build_update_reqs_widget(description: str) -> widgets.Button:
    return widgets.Button(
        description=description,
        style=widgets.ButtonStyle(button_color="lightgreen", font_weight="bold"),
        layout=widgets.Layout(width="100%", height="50px"),
    )


def _build_accordion_widget(
    children: list[widgets.Widget], titles: list[str]
) -> widgets.Accordion:
    return widgets.Accordion(
        children=children,
        titles=titles,
    )


def _build_metadata_widget(children: list[widgets.Widget]) -> widgets.VBox:
    return widgets.VBox(
        children=children,
    )


def _build_metadata_widgets(nb_meta: NotebookMetadata) -> list[widgets.Widget]:
    doi_widget = _build_doi_widget(nb_meta)
    base_image_widget = _build_base_image_widget(nb_meta)
    reqs_widget = _build_reqs_widget(nb_meta)
    update_reqs_widget = _build_update_reqs_widget(
        description="Install new requirements"
    )
    accordion_widget = _build_accordion_widget(
        children=[doi_widget, base_image_widget, reqs_widget],
        titles=["Global DOI", "Base Image", "Requirements"],
    )
    metadata_widget = _build_metadata_widget(children=[accordion_widget])
    return [
        doi_widget,
        base_image_widget,
        reqs_widget,
        update_reqs_widget,
        accordion_widget,
        metadata_widget,
    ]


def doi_observer(
    nb_meta: NotebookMetadata,
    metadata_widget: widgets.VBox,
    output: widgets.Output,
) -> Callable:
    def _doi_observer(change):
        with output:
            nb_meta.global_notebook_doi = change.new.strip()
            # save empty field as None
            if nb_meta.global_notebook_doi == "":
                nb_meta.global_notebook_doi = None
            _save_metadata_as_json(nb_meta)

            msg = (
                "You have made changes to your notebook's metadata. "
                "Make sure to save your notebook to also save the new metadata."
            )
            info_widget = make_html_popup_widget(msg)

            # add info widget to display
            metadata_widget.children = [info_widget] + [
                value
                for value in list(metadata_widget.children)
                if not isinstance(value, widgets.HTML)
            ]

    return _doi_observer


def base_image_observer(
    nb_meta: NotebookMetadata,
    metadata_widget: widgets.VBox,
    output: widgets.Output,
) -> Callable:
    def _base_image_observer(change):
        with output:
            nb_meta.notebook_image_name = change.new
            nb_meta.notebook_image_uri = GardenConstants.PREMADE_IMAGES[change.new]
            _save_metadata_as_json(nb_meta)

            msg = (
                "You have made changes to your notebook's metadata. "
                "Make sure to save your notebook to also save the new metadata.<br>"
                "The new Garden base image you have selected will used next time you start this notebook."
            )
            info_widget = make_html_popup_widget(msg)

            # add info widget to display
            metadata_widget.children = [info_widget] + [
                value
                for value in list(metadata_widget.children)
                if not isinstance(value, widgets.HTML)
            ]

    return _base_image_observer


def reqs_observer(
    nb_meta: NotebookMetadata,
    metadata_widget: widgets.VBox,
    update_reqs_widget: widgets.Button,
    output: widgets.Output,
) -> Callable:
    def _reqs_observer(change):
        with output:
            nb_meta.notebook_requirements.contents = change.new.split("\n")
            if "" in nb_meta.notebook_requirements.contents:
                nb_meta.notebook_requirements.contents.remove("")
            _save_metadata_as_json(nb_meta)
            # if changes to requirements have been made, show the
            # 'Install new requirements' button
            if update_reqs_widget not in metadata_widget.children:
                update_reqs_widget.disabled = False
                metadata_widget.children = list(metadata_widget.children) + [
                    update_reqs_widget
                ]

            msg = (
                "You have made changes to your notebook's metadata. "
                "Make sure to save your notebook to also save the new metadata.<br>"
                "Use 'Install new requirements' to install the new requirements to the container and "
                "update your requirements file if one was provided."
            )
            info_widget = make_html_popup_widget(msg)

            # add info widget to display
            metadata_widget.children = [info_widget] + [
                value
                for value in list(metadata_widget.children)
                if not isinstance(value, widgets.HTML)
            ]

    return _reqs_observer


def update_reqs_observer(
    nb_meta: NotebookMetadata,
    metadata_widget: widgets.VBox,
    update_reqs_widget: widgets.Button,
    output: widgets.Output,
) -> Callable:
    from rich.status import Status

    from garden_ai.app.console import console

    def _update_reqs_observer(button):
        with output:
            # disable button so user cant start multiple installs
            button.disabled = True

            # save changes to requirements file
            # REQUIREMENTS_PATH env var set to the containers requirements file path
            # in 'start_container_with_notebook' only if user provided requirements.
            reqs_path = os.environ.get("REQUIREMENTS_PATH", None)
            if reqs_path is not None:
                # container requirements file is bound to users local requirements file
                save_requirements_data(
                    Path(reqs_path).parent, nb_meta.notebook_requirements
                )

            # pip install new requirements
            status = Status(
                "[bold green] Installing new requirements...", console=console
            )
            with status:
                for req in nb_meta.notebook_requirements.contents:
                    status.update(f"[bold green] Installing requirement: {req}")
                    try:
                        subprocess.check_call(
                            [sys.executable, "-m", "pip", "install", req]
                        )
                    except SubprocessError as e:
                        # if new req failed to install, display popup error msg, enable install button and return
                        print(e)
                        msg = f"Failed to install {req}"
                        info_widget = make_html_popup_widget(
                            msg, background_color="#fc0f03"
                        )
                        metadata_widget.children = [info_widget] + [
                            value
                            for value in list(metadata_widget.children)
                            if not isinstance(value, widgets.HTML)
                        ]
                        button.disabled = False
                        return

            # remove update button from metadata_widget
            new_children = list(metadata_widget.children)
            new_children.remove(update_reqs_widget)
            metadata_widget.children = new_children

            # re-enable button
            button.disabled = False

            msg = "Successfully installed the new requirements."
            info_widget = make_html_popup_widget(msg, background_color="#0ff702")
            metadata_widget.children = [info_widget] + [
                value
                for value in list(metadata_widget.children)
                if not isinstance(value, widgets.HTML)
            ]

    return _update_reqs_observer


def display_metadata_widget():
    """
    Displays the metadata editor widget
    When one of the widgets fields is changed, pickles and saves the updated NotebookMetadata.
    When the notebook is saved, the post_save_hook in custom_jupyter_config will
    go and look for the pickled NotebookMetadata and save it to the notebooks metadata.
    When any requirements are changed, will display the button 'Install new requirements',
    that installs the new requirements to the container and updates the users requirements
    file if one was provided.
    The widget does not support conda requirements since we are
    planning on removing support for them.
    """
    # NOTEBOOK_PATH env var set in start_container_with_notebook
    notebook_path = Path(os.environ["NOTEBOOK_PATH"])
    nb_meta = get_notebook_metadata(notebook_path)

    output = widgets.Output()

    # Build the widgets
    (
        doi_widget,
        base_image_widget,
        reqs_widget,
        update_reqs_widget,
        accordion_widget,
        metadata_widget,
    ) = _build_metadata_widgets(nb_meta)

    # Setup event handlers
    doi_widget.observe(doi_observer(nb_meta, metadata_widget, output), "value")
    base_image_widget.observe(
        base_image_observer(nb_meta, metadata_widget, output),
        "value",
    )
    reqs_widget.observe(
        reqs_observer(nb_meta, metadata_widget, update_reqs_widget, output),
        "value",
    )
    update_reqs_widget.on_click(
        update_reqs_observer(nb_meta, metadata_widget, update_reqs_widget, output)
    )

    # Display the metadata_widget to kick things off
    # other widgets will display based on user interaction
    display(metadata_widget, output)


def _save_metadata_as_json(nb_meta: NotebookMetadata):
    working_dir = Path(os.environ.get("NOTEBOOK_PATH")).parent  # type: ignore
    with open(working_dir / "notebook_metadata.json", "w") as file:
        json.dump(nb_meta.model_dump(), file)


def _read_notebook(notebook_path: Path) -> NotebookNode:
    # Read notebook contents with nbformat
    try:
        ntbk = nbformat.read(notebook_path, as_version=4)
        return ntbk
    except Exception as e:
        typer.echo(f"Unable to parse notebook: {notebook_path}, {e}")
        raise typer.Exit(1) from e
