"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto = require("crypto");
const stack_1 = require("../stack");
const token_1 = require("../token");
function generatePhysicalName(resource) {
    const stack = stack_1.Stack.of(resource);
    const stackPart = new PrefixNamePart(stack.stackName, 25);
    const idPart = new SuffixNamePart(resource.node.uniqueId, 24);
    const region = stack.region;
    if (token_1.Token.isUnresolved(region) || !region) {
        throw new Error(`Cannot generate a physical name for ${resource.node.path}, because the region is un-resolved or missing`);
    }
    const account = stack.account;
    if (token_1.Token.isUnresolved(account) || !account) {
        throw new Error(`Cannot generate a physical name for ${resource.node.path}, because the account is un-resolved or missing`);
    }
    const parts = [stackPart, idPart]
        .map(part => part.generate());
    const hashLength = 12;
    const sha256 = crypto.createHash('sha256')
        .update(stackPart.bareStr)
        .update(idPart.bareStr)
        .update(region)
        .update(account);
    const hash = sha256.digest('hex').slice(0, hashLength);
    const ret = [...parts, hash].join('');
    return ret.toLowerCase();
}
exports.generatePhysicalName = generatePhysicalName;
class NamePart {
    constructor(bareStr) {
        this.bareStr = bareStr;
    }
}
class PrefixNamePart extends NamePart {
    constructor(bareStr, prefixLength) {
        super(bareStr);
        this.prefixLength = prefixLength;
    }
    generate() {
        return this.bareStr.slice(0, this.prefixLength);
    }
}
class SuffixNamePart extends NamePart {
    constructor(str, suffixLength) {
        super(str);
        this.suffixLength = suffixLength;
    }
    generate() {
        const strLen = this.bareStr.length;
        const startIndex = Math.max(strLen - this.suffixLength, 0);
        return this.bareStr.slice(startIndex, strLen);
    }
}
//# sourceMappingURL=data:application/json;base64,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