"use strict";
const lib_1 = require("../lib");
const intrinsic_1 = require("../lib/private/intrinsic");
const util_1 = require("./util");
class StackWithPostProcessor extends lib_1.Stack {
    // ...
    _toCloudFormation() {
        const template = super._toCloudFormation();
        // manipulate template (e.g. rename "Key" to "key")
        template.Resources.myResource.Properties.Environment.key =
            template.Resources.myResource.Properties.Environment.Key;
        delete template.Resources.myResource.Properties.Environment.Key;
        return template;
    }
}
module.exports = {
    'a stack can be serialized into a CloudFormation template, initially it\'s empty'(test) {
        const stack = new lib_1.Stack();
        test.deepEqual(util_1.toCloudFormation(stack), {});
        test.done();
    },
    'stack objects have some template-level propeties, such as Description, Version, Transform'(test) {
        const stack = new lib_1.Stack();
        stack.templateOptions.templateFormatVersion = 'MyTemplateVersion';
        stack.templateOptions.description = 'This is my description';
        stack.templateOptions.transform = 'SAMy';
        test.deepEqual(util_1.toCloudFormation(stack), {
            Description: 'This is my description',
            AWSTemplateFormatVersion: 'MyTemplateVersion',
            Transform: 'SAMy'
        });
        test.done();
    },
    'Stack.isStack indicates that a construct is a stack'(test) {
        const stack = new lib_1.Stack();
        const c = new lib_1.Construct(stack, 'Construct');
        test.ok(lib_1.Stack.isStack(stack));
        test.ok(!lib_1.Stack.isStack(c));
        test.done();
    },
    'stack.id is not included in the logical identities of resources within it'(test) {
        const stack = new lib_1.Stack(undefined, 'MyStack');
        new lib_1.CfnResource(stack, 'MyResource', { type: 'MyResourceType' });
        test.deepEqual(util_1.toCloudFormation(stack), { Resources: { MyResource: { Type: 'MyResourceType' } } });
        test.done();
    },
    'stack.templateOptions can be used to set template-level options'(test) {
        const stack = new lib_1.Stack();
        stack.templateOptions.description = 'StackDescription';
        stack.templateOptions.templateFormatVersion = 'TemplateVersion';
        stack.templateOptions.transform = 'DeprecatedField';
        stack.templateOptions.transforms = ['Transform'];
        stack.templateOptions.metadata = {
            MetadataKey: 'MetadataValue'
        };
        test.deepEqual(util_1.toCloudFormation(stack), {
            Description: 'StackDescription',
            Transform: ['DeprecatedField', 'Transform'],
            AWSTemplateFormatVersion: 'TemplateVersion',
            Metadata: { MetadataKey: 'MetadataValue' }
        });
        test.done();
    },
    // This approach will only apply to TypeScript code, but at least it's a temporary
    // workaround for people running into issues caused by SDK-3003.
    // We should come up with a proper solution that involved jsii callbacks (when they exist)
    // so this can be implemented by jsii languages as well.
    'Overriding `Stack._toCloudFormation` allows arbitrary post-processing of the generated template during synthesis'(test) {
        const stack = new StackWithPostProcessor();
        new lib_1.CfnResource(stack, 'myResource', {
            type: 'AWS::MyResource',
            properties: {
                MyProp1: 'hello',
                MyProp2: 'howdy',
                Environment: {
                    Key: 'value'
                }
            }
        });
        test.deepEqual(stack._toCloudFormation(), { Resources: { myResource: { Type: 'AWS::MyResource',
                    Properties: { MyProp1: 'hello',
                        MyProp2: 'howdy',
                        Environment: { key: 'value' } } } } });
        test.done();
    },
    'Stack.getByPath can be used to find any CloudFormation element (Parameter, Output, etc)'(test) {
        const stack = new lib_1.Stack();
        const p = new lib_1.CfnParameter(stack, 'MyParam', { type: 'String' });
        const o = new lib_1.CfnOutput(stack, 'MyOutput', { value: 'boom' });
        const c = new lib_1.CfnCondition(stack, 'MyCondition');
        test.equal(stack.node.findChild(p.node.path), p);
        test.equal(stack.node.findChild(o.node.path), o);
        test.equal(stack.node.findChild(c.node.path), c);
        test.done();
    },
    'Stack names can have hyphens in them'(test) {
        const root = new lib_1.App();
        new lib_1.Stack(root, 'Hello-World');
        // Did not throw
        test.done();
    },
    'Include should support non-hash top-level template elements like "Description"'(test) {
        const stack = new lib_1.Stack();
        const template = {
            Description: 'hello, world'
        };
        new lib_1.CfnInclude(stack, 'Include', { template });
        const output = util_1.toCloudFormation(stack);
        test.equal(typeof output.Description, 'string');
        test.done();
    },
    'Pseudo values attached to one stack can be referenced in another stack'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        // THEN
        const assembly = app.synth();
        const template1 = assembly.getStack(stack1.stackName).template;
        const template2 = assembly.getStack(stack2.stackName).template;
        test.deepEqual(template1, {
            Outputs: {
                ExportsOutputRefAWSAccountIdAD568057: {
                    Value: { Ref: 'AWS::AccountId' },
                    Export: { Name: 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }
                }
            }
        });
        test.deepEqual(template2, {
            Parameters: {
                SomeParameter: {
                    Type: 'String',
                    Default: { 'Fn::ImportValue': 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }
                }
            }
        });
        test.done();
    },
    'Cross-stack references are detected in resource properties'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const resource1 = new lib_1.CfnResource(stack1, 'Resource', { type: 'BLA' });
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another resource
        new lib_1.CfnResource(stack2, 'SomeResource', { type: 'AWS::Some::Resource', properties: {
                someProperty: new intrinsic_1.Intrinsic(resource1.ref),
            } });
        // THEN
        const assembly = app.synth();
        const template2 = assembly.getStack(stack2.stackName).template;
        test.deepEqual(template2, {
            Resources: {
                SomeResource: {
                    Type: 'AWS::Some::Resource',
                    Properties: {
                        someProperty: { 'Fn::ImportValue': 'Stack1:ExportsOutputRefResource1D5D905A' }
                    }
                }
            }
        });
        test.done();
    },
    'cross-stack references in lazy tokens work'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: lib_1.Lazy.stringValue({ produce: () => account1 }) });
        const assembly = app.synth();
        const template1 = assembly.getStack(stack1.stackName).template;
        const template2 = assembly.getStack(stack2.stackName).template;
        // THEN
        test.deepEqual(template1, {
            Outputs: {
                ExportsOutputRefAWSAccountIdAD568057: {
                    Value: { Ref: 'AWS::AccountId' },
                    Export: { Name: 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }
                }
            }
        });
        test.deepEqual(template2, {
            Parameters: {
                SomeParameter: {
                    Type: 'String',
                    Default: { 'Fn::ImportValue': 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }
                }
            }
        });
        test.done();
    },
    'Cross-stack use of Region and account returns nonscoped intrinsic because the two stacks must be in the same region anyway'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnOutput(stack2, 'DemOutput', { value: stack1.region });
        new lib_1.CfnOutput(stack2, 'DemAccount', { value: stack1.account });
        // THEN
        const assembly = app.synth();
        const template2 = assembly.getStack(stack2.stackName).template;
        test.deepEqual(template2, {
            Outputs: {
                DemOutput: {
                    Value: { Ref: 'AWS::Region' },
                },
                DemAccount: {
                    Value: { Ref: 'AWS::AccountId' },
                }
            }
        });
        test.done();
    },
    'cross-stack references in strings work'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: `TheAccountIs${account1}` });
        const assembly = app.synth();
        const template2 = assembly.getStack(stack2.stackName).template;
        // THEN
        test.deepEqual(template2, {
            Parameters: {
                SomeParameter: {
                    Type: 'String',
                    Default: { 'Fn::Join': ['', ['TheAccountIs', { 'Fn::ImportValue': 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }]] }
                }
            }
        });
        test.done();
    },
    'cannot create cyclic reference between stacks'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        const account2 = new lib_1.ScopedAws(stack2).accountId;
        // WHEN
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        new lib_1.CfnParameter(stack1, 'SomeParameter', { type: 'String', default: account2 });
        test.throws(() => {
            lib_1.ConstructNode.prepare(app.node);
            // tslint:disable-next-line:max-line-length
        }, "'Stack2' depends on 'Stack1' (Stack2/SomeParameter -> Stack1.AWS::AccountId). Adding this dependency (Stack1/SomeParameter -> Stack2.AWS::AccountId) would create a cyclic reference.");
        test.done();
    },
    'stacks know about their dependencies'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        lib_1.ConstructNode.prepare(app.node);
        // THEN
        test.deepEqual(stack2.dependencies.map(s => s.node.id), ['Stack1']);
        test.done();
    },
    'cannot create references to stacks in other regions/accounts'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1', { env: { account: '123456789012', region: 'es-norst-1' } });
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2', { env: { account: '123456789012', region: 'es-norst-2' } });
        // WHEN
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        test.throws(() => {
            lib_1.ConstructNode.prepare(app.node);
        }, /Can only reference cross stacks in the same region and account/);
        test.done();
    },
    'stack with region supplied via props returns literal value'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'Stack1', { env: { account: '123456789012', region: 'es-norst-1' } });
        // THEN
        test.equal(stack.resolve(stack.region), 'es-norst-1');
        test.done();
    },
    'overrideLogicalId(id) can be used to override the logical ID of a resource'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const bonjour = new lib_1.CfnResource(stack, 'BonjourResource', { type: 'Resource::Type' });
        // { Ref } and { GetAtt }
        new lib_1.CfnResource(stack, 'RefToBonjour', { type: 'Other::Resource', properties: {
                RefToBonjour: bonjour.ref,
                GetAttBonjour: bonjour.getAtt('TheAtt').toString()
            } });
        bonjour.overrideLogicalId('BOOM');
        // THEN
        test.deepEqual(util_1.toCloudFormation(stack), { Resources: { BOOM: { Type: 'Resource::Type' },
                RefToBonjour: { Type: 'Other::Resource',
                    Properties: { RefToBonjour: { Ref: 'BOOM' },
                        GetAttBonjour: { 'Fn::GetAtt': ['BOOM', 'TheAtt'] } } } } });
        test.done();
    },
    'Stack name can be overridden via properties'(test) {
        // WHEN
        const stack = new lib_1.Stack(undefined, 'Stack', { stackName: 'otherName' });
        // THEN
        test.deepEqual(stack.stackName, 'otherName');
        test.done();
    },
    'Stack name is inherited from App name if available'(test) {
        // WHEN
        const root = new lib_1.App();
        const app = new lib_1.Construct(root, 'Prod');
        const stack = new lib_1.Stack(app, 'Stack');
        // THEN
        test.deepEqual(stack.stackName, 'ProdStackD5279B22');
        test.done();
    },
    'stack construct id does not go through stack name validation if there is an explicit stack name'(test) {
        // GIVEN
        const app = new lib_1.App();
        // WHEN
        const stack = new lib_1.Stack(app, 'invalid as : stack name, but thats fine', {
            stackName: 'valid-stack-name'
        });
        // THEN
        const session = app.synth();
        test.deepEqual(stack.stackName, 'valid-stack-name');
        test.ok(session.tryGetArtifact('valid-stack-name'));
        test.done();
    },
    'stack validation is performed on explicit stack name'(test) {
        // GIVEN
        const app = new lib_1.App();
        // THEN
        test.throws(() => new lib_1.Stack(app, 'boom', { stackName: 'invalid:stack:name' }), /Stack name must match the regular expression/);
        test.done();
    },
    'Stack.of(stack) returns the correct stack'(test) {
        const stack = new lib_1.Stack();
        test.same(lib_1.Stack.of(stack), stack);
        const parent = new lib_1.Construct(stack, 'Parent');
        const construct = new lib_1.Construct(parent, 'Construct');
        test.same(lib_1.Stack.of(construct), stack);
        test.done();
    },
    'Stack.of() throws when there is no parent Stack'(test) {
        const root = new lib_1.Construct(undefined, 'Root');
        const construct = new lib_1.Construct(root, 'Construct');
        test.throws(() => lib_1.Stack.of(construct), /No stack could be identified for the construct at path/);
        test.done();
    },
    'stack.availabilityZones falls back to Fn::GetAZ[0],[2] if region is not specified'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'MyStack');
        // WHEN
        const azs = stack.availabilityZones;
        // THEN
        test.deepEqual(stack.resolve(azs), [
            { "Fn::Select": [0, { "Fn::GetAZs": "" }] },
            { "Fn::Select": [1, { "Fn::GetAZs": "" }] }
        ]);
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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