"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// tslint:disable-next-line:no-var-requires
const crypto = require("crypto");
const encoding_1 = require("./encoding");
/**
 * Resources with this ID are hidden from humans
 *
 * They do not appear in the human-readable part of the logical ID,
 * but they are included in the hash calculation.
 */
const HIDDEN_FROM_HUMAN_ID = 'Resource';
/**
 * Resources with this ID are complete hidden from the logical ID calculation.
 */
const HIDDEN_ID = 'Default';
const PATH_SEP = '/';
const HASH_LEN = 8;
const MAX_HUMAN_LEN = 240; // max ID len is 255
/**
 * Calculates a unique ID for a set of textual components.
 *
 * This is done by calculating a hash on the full path and using it as a suffix
 * of a length-limited "human" rendition of the path components.
 *
 * @param components The path components
 * @returns a unique alpha-numeric identifier with a maximum length of 255
 */
function makeUniqueId(components) {
    components = components.filter(x => x !== HIDDEN_ID);
    if (components.length === 0) {
        throw new Error('Unable to calculate a unique id for an empty set of components');
    }
    // Lazy require in order to break a module dependency cycle
    const unresolvedTokens = components.filter(c => encoding_1.unresolved(c));
    if (unresolvedTokens.length > 0) {
        throw new Error(`ID components may not include unresolved tokens: ${unresolvedTokens.join(',')}`);
    }
    // top-level resources will simply use the `name` as-is in order to support
    // transparent migration of cloudformation templates to the CDK without the
    // need to rename all resources.
    if (components.length === 1) {
        return removeNonAlphanumeric(components[0]);
    }
    const hash = pathHash(components);
    const human = removeDupes(components)
        .filter(x => x !== HIDDEN_FROM_HUMAN_ID)
        .map(removeNonAlphanumeric)
        .join('')
        .slice(0, MAX_HUMAN_LEN);
    return human + hash;
}
exports.makeUniqueId = makeUniqueId;
/**
 * Take a hash of the given path.
 *
 * The hash is limited in size.
 */
function pathHash(path) {
    const md5 = crypto.createHash('md5').update(path.join(PATH_SEP)).digest("hex");
    return md5.slice(0, HASH_LEN).toUpperCase();
}
/**
 * Removes all non-alphanumeric characters in a string.
 */
function removeNonAlphanumeric(s) {
    return s.replace(/[^A-Za-z0-9]/g, '');
}
/**
 * Remove duplicate "terms" from the path list
 *
 * If the previous path component name ends with this component name, skip the
 * current component.
 */
function removeDupes(path) {
    const ret = new Array();
    for (const component of path) {
        if (ret.length === 0 || !ret[ret.length - 1].endsWith(component)) {
            ret.push(component);
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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