"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn_dynamic_reference_1 = require("./cfn-dynamic-reference");
const intrinsic_1 = require("./private/intrinsic");
/**
 * Work with secret values in the CDK
 *
 * Secret values in the CDK (such as those retrieved from SecretsManager) are
 * represented as regular strings, just like other values that are only
 * available at deployment time.
 *
 * To help you avoid accidental mistakes which would lead to you putting your
 * secret values directly into a CloudFormation template, constructs that take
 * secret values will not allow you to pass in a literal secret value. They do
 * so by calling `Secret.assertSafeSecret()`.
 *
 * You can escape the check by calling `Secret.plainText()`, but doing
 * so is highly discouraged.
 */
class SecretValue extends intrinsic_1.Intrinsic {
    /**
     * Construct a literal secret value for use with secret-aware constructs
     *
     * *Do not use this method for any secrets that you care about.*
     *
     * The only reasonable use case for using this method is when you are testing.
     */
    static plainText(secret) {
        return new SecretValue(secret);
    }
    /**
     * Creates a `SecretValue` with a value which is dynamically loaded from AWS Secrets Manager.
     * @param secretId The ID or ARN of the secret
     * @param options Options
     */
    static secretsManager(secretId, options = {}) {
        if (!secretId) {
            throw new Error(`secretId cannot be empty`);
        }
        const parts = [
            secretId,
            'SecretString',
            options.jsonField || '',
            options.versionStage || '',
            options.versionId || ''
        ];
        const dyref = new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SECRETS_MANAGER, parts.join(':'));
        return this.cfnDynamicReference(dyref);
    }
    /**
     * Use a secret value stored from a Systems Manager (SSM) parameter.
     *
     * @param parameterName The name of the parameter in the Systems Manager
     * Parameter Store. The parameter name is case-sensitive.
     *
     * @param version An integer that specifies the version of the parameter to
     * use. You must specify the exact version. You cannot currently specify that
     * AWS CloudFormation use the latest version of a parameter.
     */
    static ssmSecure(parameterName, version) {
        const parts = [parameterName, version];
        return this.cfnDynamicReference(new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SSM_SECURE, parts.join(':')));
    }
    /**
     * Obtain the secret value through a CloudFormation dynamic reference.
     *
     * If possible, use `SecretValue.ssmSecure` or `SecretValue.secretsManager` directly.
     *
     * @param ref The dynamic reference to use.
     */
    static cfnDynamicReference(ref) {
        return new SecretValue(ref);
    }
    /**
     * Obtain the secret value through a CloudFormation parameter.
     *
     * Generally, this is not a recommended approach. AWS Secrets Manager is the
     * recommended way to reference secrets.
     *
     * @param param The CloudFormation parameter to use.
     */
    static cfnParameter(param) {
        if (!param.noEcho) {
            throw new Error(`CloudFormation parameter must be configured with "NoEcho"`);
        }
        return new SecretValue(param.value);
    }
}
exports.SecretValue = SecretValue;
//# sourceMappingURL=data:application/json;base64,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