"use strict";
const nodeunit = require("nodeunit");
const lib_1 = require("../../lib");
const physical_name_generator_1 = require("../../lib/private/physical-name-generator");
class TestResource extends lib_1.Resource {
}
module.exports = nodeunit.testCase({
    'generates correct physical names'(test) {
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'TestStack', { env: { account: '012345678912', region: 'bermuda-triangle-1' } });
        const testResourceA = new TestResource(stack, 'A');
        const testResourceB = new TestResource(testResourceA, 'B');
        test.equal(physical_name_generator_1.generatePhysicalName(testResourceA), 'teststackteststackaa164c141d59b37c1b663');
        test.equal(physical_name_generator_1.generatePhysicalName(testResourceB), 'teststackteststackab27595cd34d8188283a1f');
        test.done();
    },
    'generates different names in different accounts'(test) {
        const appA = new lib_1.App();
        const stackA = new lib_1.Stack(appA, 'TestStack', { env: { account: '012345678912', region: 'bermuda-triangle-1' } });
        const resourceA = new TestResource(stackA, 'Resource');
        const appB = new lib_1.App();
        const stackB = new lib_1.Stack(appB, 'TestStack', { env: { account: '012345678913', region: 'bermuda-triangle-1' } });
        const resourceB = new TestResource(stackB, 'Resource');
        test.notEqual(physical_name_generator_1.generatePhysicalName(resourceA), physical_name_generator_1.generatePhysicalName(resourceB));
        test.done();
    },
    'generates different names in different regions'(test) {
        const appA = new lib_1.App();
        const stackA = new lib_1.Stack(appA, 'TestStack', { env: { account: '012345678912', region: 'bermuda-triangle-1' } });
        const resourceA = new TestResource(stackA, 'Resource');
        const appB = new lib_1.App();
        const stackB = new lib_1.Stack(appB, 'TestStack', { env: { account: '012345678912', region: 'bermuda-triangle-2' } });
        const resourceB = new TestResource(stackB, 'Resource');
        test.notEqual(physical_name_generator_1.generatePhysicalName(resourceA), physical_name_generator_1.generatePhysicalName(resourceB));
        test.done();
    },
    'fails when the region is an unresolved token'(test) {
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'TestStack', { env: { account: '012345678912', region: lib_1.Aws.REGION } });
        const testResource = new TestResource(stack, 'A');
        test.throws(() => physical_name_generator_1.generatePhysicalName(testResource), /Cannot generate a physical name for TestStack\/A, because the region is un-resolved or missing/);
        test.done();
    },
    'fails when the region is not provided'(test) {
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'TestStack', { env: { account: '012345678912' } });
        const testResource = new TestResource(stack, 'A');
        test.throws(() => physical_name_generator_1.generatePhysicalName(testResource), /Cannot generate a physical name for TestStack\/A, because the region is un-resolved or missing/);
        test.done();
    },
    'fails when the account is an unresolved token'(test) {
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'TestStack', { env: { account: lib_1.Aws.ACCOUNT_ID, region: 'bermuda-triangle-1' } });
        const testResource = new TestResource(stack, 'A');
        test.throws(() => physical_name_generator_1.generatePhysicalName(testResource), /Cannot generate a physical name for TestStack\/A, because the account is un-resolved or missing/);
        test.done();
    },
    'fails when the account is not provided'(test) {
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'TestStack', { env: { region: 'bermuda-triangle-1' } });
        const testResource = new TestResource(stack, 'A');
        test.throws(() => physical_name_generator_1.generatePhysicalName(testResource), /Cannot generate a physical name for TestStack\/A, because the account is un-resolved or missing/);
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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